/* hilite_compromise.pov
 * Persistence of Vision Ray Tracer scene description file
 *
 * This program is an attempt to find reasonable compromise specular highlight
 * values for glossy non-metallic objects, since POV-Ray doesn't currently
 * support Fresnel highlights.
 *
 * The number to the upper left of each test object is the index of refraction
 * of the left half.  The number to the upper right is the constant reflection
 * value of the right half, and is intended to be used as a 'specular albedo'
 * value in POV-Ray 3.7.
 *
 * If you are using POV-Ray 3.6.2 or earlier, you will need to multiply the
 * upper right number by the following formula to use as a 'specular' value:
 *
 *   (1 / roughness + 2) / (4 * (2 - pow (2, -0.5 / roughness)))
 *
 * Copyright © 2014 Richard Callwood III.  Some rights reserved.
 * This work is licensed under the GNU General Public License 3.0.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See http://www.gnu.org/licenses/gpl-3.0.html for the legalese.
 *
 * Vers  Date         Comments
 * ----  ----         --------
 * 1.0   03-nov-2014  Created.
 */
#version 3.7;

global_settings { assumed_gamma 1 charset utf8 }

background { rgb <0.5, 0.7, 1.0> }

#declare NROWS = 2;
#declare NCOLS = 6;
#declare SPACE = 0.3;
#declare ROOM = 0.6;
#declare START = 1.35;
#declare STEP = 0.1;
#declare S_FONT = "arial" // a Unicode font

#declare WIDTH = SPACE + NCOLS * (2 + SPACE);
#declare HEIGHT = SPACE + NROWS * (2 + ROOM);

#declare hSuggest = floor (image_width * HEIGHT / WIDTH + 0.5);
#if (floor (image_height + 0.5) != hSuggest)
  #warning concat
  ( "Suggested image dimensions: +W", str (image_width, 0, 0),
    " +H", str (hSuggest, 0, 0)
  )
#end

camera
{ orthographic
  location <WIDTH/2, HEIGHT/2, -10>
  right WIDTH * x
  up HEIGHT * y
}

#macro Test (IofR)
 // A linear formula obviously doesn't work for all IORs, but it's good enough
 // for most common materials:
  #local Fudge = IofR * 0.142 - 0.163;
  union
  { cylinder
    { -x, 0, 1
      pigment { rgb 0 }
      finish { reflection { 0 1 fresnel } conserve_energy }
      interior { ior IofR }
    }
    cylinder
    { x, 0, 1
      pigment { rgb 0 }
      finish { reflection { Fudge } }
    }
    text
    { ttf S_FONT concat (str (IofR, 0, 2), " \u2192 ", str (Fudge, 0, 4)) 1, 0
      pigment { red 1 }
      finish { ambient 0.5 }
      scale 0.3
      translate <-1, 1.1, 0>
    }
    no_reflection
  }
#end

#for (Row, 0, NROWS-1)
  #for (Col, 0, NCOLS-1)
    object
    { Test (START + Row * NCOLS * STEP + Col * STEP)
      translate
      < SPACE + 1 + Col * (2 + SPACE),
        SPACE + 1 + (NROWS - 1 - Row) * (2 + ROOM),
        0
      >
    }
  #end
#end
