// %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// %                                                               %
// %   IsoWood Include File version 0.3                            %
// %                                                               %
// %   isowood3.inc - macros for wood objects                      %
// %                                                               %
// %   for POV-Ray version 3.5, some features require MegaPOV 1.x  %
// %                                                               %
// %   written August 2000 - August 2003                           %
// %   by Christoph Hormann <chris_hormann@gmx.de>                 %
// %   newest version can be found at                              %
// %   http://www.tu-bs.de/~y0013390/iso_wood.html                 %
// %                                                               %
// %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
// %                                                               %
// %   Feel free to use this file for your povray work, you are    %
// %   also allowed to modify and redistribute it as long as you   %
// %   leave these comments intact and highlight any changes you   %
// %   make.                                                       %
// %                                                               %
// %   I encourage you to contact me if you have ideas for         %
// %   improvements                                                %
// %                                                               %
// %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

//------------------------------------------------------------------

#include "functions.inc"
#include "shapes.inc"


//--------------------------------------------
//===== This is the global quality parameter
//===== used by most of the macros:
//=====
//=====   0: plain color shape
//=====   1: textured shape
//=====   2: isosurface
//--------------------------------------------

#ifndef (IW_Quality)
  #declare IW_Quality = 2;
#end

//--------------------------------------------------------------------
//===== These are general parameters for all isosurfaces used.
//===== You can change them here or in another pov file if you like
//--------------------------------------------------------------------

#ifndef (IW_ISOaccuracy)
  #declare IW_ISOaccuracy = 0.001;
#end
#ifndef (IW_ISOmaxgrad)
  #declare IW_ISOmaxgrad = 1.1;
#end

#ifndef (IW_ISOmax_trace)
  #declare IW_ISOmax_trace=0;
#end

//--------------------------------------------
//===== This is the texture used for
//===== IW_Quality 0
//--------------------------------------------

#ifndef (IW_Plain_Tex)
  #declare IW_Plain_Tex =
    texture { pigment { color rgb <1,0,0> } }
#end

//--------------------------------------------
//===== IsoWood random seed
//--------------------------------------------

#ifndef (IW_Rand_Seed)
  #declare IW_Rand_Seed = seed(0);
#end


#macro IW_Rand()
  rand(IW_Rand_Seed)*10000
#end

//--------------------------------------------
//===== dummy identifiers
//--------------------------------------------

#declare No_Texture = texture { }
#declare No_Pigment = pigment { color rgb 0 }
#declare No_Normal = normal { bumps 0 }
#declare No_Finish = finish { }
#declare ID_No_Warp = 0;

//--------------------------------------------
//===== macro identifiers
//--------------------------------------------

#declare ID_IW_Plank_Round = 1;
#declare ID_IW_Plank_RoundW = 11;
#declare ID_IW_Plank_RoundT = 12;
#declare ID_IW_Plank_Round_Cut = 2;
#declare ID_IW_Plank_Round_CutE = 20;
#declare ID_IW_Plank_Round_CutW = 21;
#declare ID_IW_Plank_Round_CutT = 22;
#declare ID_IW_Plank_Chamfered = 3;
#declare ID_IW_Plank_ChamferedE = 30;
#declare ID_IW_Plank_ChamferedW = 31;
#declare ID_IW_Plank_ChamferedT = 32;

//==========================================================================================
//===== Warp macros ========================================================================
//==========================================================================================

#macro IW_Warp_01 (Seed, xSize, ySize, zSize)

  #local SZ=(xSize+ySize)*0.5*0.1;
  #local Density=0.14;

  warp {
    black_hole <xSize*0.5, 0, 0>, SZ
    falloff 1.5
    strength 2
    repeat <min(xSize, SZ/Density),min(ySize, SZ/Density), SZ/Density>
    turbulence <0.0, 0.35, 0.5>
    inverse
  }
#end //-------------------------------------------------------------------------------------

#macro IW_Warp_02 (Seed, xSize, ySize, zSize)

  #local SZ=(xSize+ySize)*0.5*0.1;
  #local Density=0.1;

  warp {
    black_hole <xSize*0.5, 0, 0>, SZ
    falloff 2.5
    strength 2
    repeat <min(xSize, SZ/Density),min(ySize, SZ/Density), SZ/Density>
    turbulence <0.02, 0.75, 0.75>
    inverse
  }

  #local SZ=(xSize+ySize)*0.5*0.15;
  #local Density=0.1;

  warp {
    black_hole <xSize*0.5, 0, 0>, SZ
    falloff 2.1
    strength 3.7
    repeat <min(xSize, SZ/Density),min(ySize, 0.7*SZ/Density), 0.8*SZ/Density>
    turbulence <0.0,0.4, 0.4>
    inverse
  }

#end //-------------------------------------------------------------------------------------


#macro IW_Warp_03 (Seed, xSize, ySize, zSize)

  #version unofficial MegaPov 1.0;

  warp {
    displace {
      bozo
      color_map { [0 rgb 0][0.5 rgb 0.01][1 rgb 0.08] }
      turbulence 0.2
      translate 3*x
      scale 0.3
      type 1
    }
  }
#end //-------------------------------------------------------------------------------------


#macro IW_Warp_04 (Seed, xSize, ySize, zSize)

  #version unofficial MegaPov 1.0;

  warp {
    displace {
      bozo
      color_map { [0 rgb 0][0.5 rgb 0.01][1 rgb 0.1] }
      turbulence 0.04
      scale 0.4
      type 1
    }
  }

  #local Cnt=0;
  #local rd=seed(Seed);

  #while (Cnt < 100)

    #local PsY=rand(rd)*ySize;
    #local PsZ=rand(rd)*zSize;

    #if (eval_pigment (
      pigment {
      bozo
      color_map { [0.6 rgb 0.0][0.9 rgb 0.7] }
      turbulence 0.04
      scale 0.4 } , <xSize*0.5, PsY, PsZ>).red > 0.2)

    warp {
      black_hole <xSize*0.5, PsY, PsZ>, 0.1
      falloff 2.1
      strength 4.7
      inverse
    }

    #local Cnt=Cnt+1;

    #end

  #end

  #local Cnt=0;
  #local rd=seed(Seed+5);

  #while (Cnt < 100)

    #local PsY=rand(rd)*ySize;
    #local PsZ=rand(rd)*8;

    #if (eval_pigment (
      pigment {
      bozo
      color_map { [0.6 rgb 0.0][0.9 rgb 0.7] }
      turbulence 0.04
      scale 0.4 } , <-xSize*0.5, PsY, PsZ>).red > 0.2)

    warp {
      black_hole <-xSize*0.5, PsY, PsZ>, 0.1
      falloff 2.1
      strength 4.7
      inverse
    }

    #local Cnt=Cnt+1;

    #end


  #end

#end //-------------------------------------------------------------------------------------


#macro IW_Warp_05 (Seed, xSize, ySize, zSize)

  #local rd=seed(Seed);

  #local SZ=(xSize+ySize)*0.5*0.28;
  #local Density=0.06;

  warp {
    black_hole <xSize*0.5, 0.1, 0.7*rand(rd)*(SZ/Density)>, SZ
    falloff 2.1
    strength 4.0
    repeat <0,min(ySize, 0.5*SZ/Density), SZ/Density>
    turbulence <0.0,0.6, 0.6>
    inverse
  }

  warp {
    black_hole <-xSize*0.5, 0.1, 0.7*rand(rd)*(SZ/Density)>, SZ
    falloff 2.1
    strength 4.0
    repeat <0,min(ySize, 0.5*SZ/Density), SZ/Density>
    turbulence <0.0,0.5, 0.5>
    inverse
  }

  #local SZ=(xSize+ySize)*0.5*0.25;
  #local Density=0.08;

  warp {
    black_hole <xSize*0.5, 0.3, 0.7*rand(rd)*(SZ/Density)>, SZ
    falloff 2.5
    strength 2.7
    repeat <0,min(ySize, 0.5*SZ/Density), SZ/Density>
    turbulence <0.02, 0.75, 0.75>
    inverse
  }

  warp {
    black_hole <-xSize*0.5, 0.3, 0.7*rand(rd)*(SZ/Density)>, SZ
    falloff 2.5
    strength 2.7
    repeat <0,min(ySize, 0.5*SZ/Density), SZ/Density>
    turbulence <0.02, 0.75, 0.75>
    inverse
  }


#end //-------------------------------------------------------------------------------------


#macro IW_Select_Warp (WarpNbr, Seed, xSize, ySize, zSize)

  #switch (WarpNbr)
    #case (1)
      IW_Warp_01 (Seed, xSize, ySize, zSize)
      #break
    #case (2)
      IW_Warp_02 (Seed, xSize, ySize, zSize)
      #break
    #case (3)
      IW_Warp_03 (Seed, xSize, ySize, zSize)
      #break
    #case (4)
      IW_Warp_04 (Seed, xSize, ySize, zSize)
      #break
    #case (5)
      IW_Warp_05 (Seed, xSize, ySize, zSize)
      #break
    #case (6)
      IW_Warp_06 (Seed, xSize, ySize, zSize)
      #break
    #case (7)
      IW_Warp_07 (Seed, xSize, ySize, zSize)
      #break
    #case (8)
      IW_Warp_08 (Seed, xSize, ySize, zSize)
      #break
    #case (9)
      IW_Warp_09 (Seed, xSize, ySize, zSize)
  #end

#end //-------------------------------------------------------------------------------------


//------------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_Plank_Round - rounded box =======================================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------
#macro IW_Plank_Round (Seed, xSize, ySize, zSize, Rotate, Translate, Round, Strength, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}


object {

#switch (IW_Quality)
  #case (1)
    Round_Box_Merge(
      <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
      < xSize*0.5, ySize*0.5, zSize*0.5>, Round
    )
    texture {
      T_WoodA
      normal {
        function { fn_Wood(x,y,z).grey*Strength } 6.0
        accuracy IW_ISOaccuracy
      }
    }
    #break
  #case (2)
    #local fnShape=function{ f_rounded_box(x, y, z, Round,xSize*0.5,ySize*0.5,zSize*0.5)}

    isosurface {
      function { fnShape(x,y,z)+fn_Wood(x,y,z).grey*Strength }
      contained_by {
        box{ <-xSize*0.5-Strength*1.2,-ySize*0.5-Strength*1.2,-zSize*0.5-Strength*1.2>,
             < xSize*0.5+Strength*1.2, ySize*0.5+Strength*1.2, zSize*0.5+Strength*1.2> }
      }

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { IW_Plain_Tex }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}

#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_Plank_RoundW - rounded box with warp ============================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------
#macro IW_Plank_RoundW (Seed, xSize, ySize, zSize, Rotate, Translate,
                        Round, Strength, Pigm, Fin, Norm, WarpNbr)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y

      IW_Select_Warp (WarpNbr, Seed, xSize, ySize, zSize)
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

     IW_Select_Warp (WarpNbr, Seed, xSize, ySize, zSize)
   }
   normal { Norm }
   finish { Fin }
}


object {

#switch (IW_Quality)
  #case (1)
    Round_Box_Merge(
      <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
      < xSize*0.5, ySize*0.5, zSize*0.5>, Round
    )
    texture {
      T_WoodA
      normal {
        function { fn_Wood(x,y,z).grey*Strength } 6.0
        accuracy IW_ISOaccuracy
      }
    }
    #break
  #case (2)
    #local fnShape=function{ f_rounded_box(x, y, z, Round,xSize*0.5,ySize*0.5,zSize*0.5)}

    isosurface {
      function { fnShape(x,y,z)+fn_Wood(x,y,z).grey*Strength }
      contained_by {
        box{ <-xSize*0.5-Strength*1.2,-ySize*0.5-Strength*1.2,-zSize*0.5-Strength*1.2>,
             < xSize*0.5+Strength*1.2, ySize*0.5+Strength*1.2, zSize*0.5+Strength*1.2> }
      }

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { IW_Plain_Tex }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}

#end  //------------------------------------------------------------------------------------



//==========================================================================================
//===== IW_Plank_RoundT - rounded box with separate texture ================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------
#macro IW_Plank_RoundT (Seed, xSize, ySize, zSize, Rotate, Translate,
                        Round, Strength, Pigm, Text)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y
}


object {

#switch (IW_Quality)
  #case (1)
    Round_Box_Merge(
      <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
      < xSize*0.5, ySize*0.5, zSize*0.5>, Round
    )
    texture {
      T_WoodA
      normal {
        function { fn_Wood(x,y,z).grey*Strength } 6.0
        accuracy IW_ISOaccuracy
      }
    }
    #break
  #case (2)
    #local fnShape=function{ f_rounded_box(x, y, z, Round,xSize*0.5,ySize*0.5,zSize*0.5)}

    isosurface {
      function { fnShape(x,y,z)+fn_Wood(x,y,z).grey*Strength }
      contained_by{
        box{ <-xSize*0.5-Strength*1.2,-ySize*0.5-Strength*1.2,-zSize*0.5-Strength*1.2>,
             < xSize*0.5+Strength*1.2, ySize*0.5+Strength*1.2, zSize*0.5+Strength*1.2> }
      }

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { IW_Plain_Tex }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}

#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_Plank_Round_Cut - rounded box cut at the front/back =============================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Plank_Round_Cut (Seed, xSize, ySize, zSize, Rotate, Translate,
                           Round, Strength, Strength2, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (IW_Quality)
  #case (1)
    merge {
      box{ <-xSize*0.5+Round,-ySize*0.5,-zSize*0.5>,
           < xSize*0.5-Round, ySize*0.5, zSize*0.5> }
      box{ <-xSize*0.5+Round,-ySize*0.5+Round,-zSize*0.5>,
           <-xSize*0.5,ySize*0.5-Round, zSize*0.5> }
      box{ < xSize*0.5-Round,-ySize*0.5+Round,-zSize*0.5>,
           < xSize*0.5,ySize*0.5-Round, zSize*0.5> }
      cylinder { <-xSize*0.5+Round,-ySize*0.5+Round,-zSize*0.5>,
                 <-xSize*0.5+Round,-ySize*0.5+Round, zSize*0.5>, Round }
      cylinder { < xSize*0.5-Round,-ySize*0.5+Round,-zSize*0.5>,
                 < xSize*0.5-Round,-ySize*0.5+Round, zSize*0.5>, Round }
      cylinder { <-xSize*0.5+Round, ySize*0.5-Round,-zSize*0.5>,
                 <-xSize*0.5+Round, ySize*0.5-Round, zSize*0.5>, Round }
      cylinder { < xSize*0.5-Round, ySize*0.5-Round,-zSize*0.5>,
                 < xSize*0.5-Round, ySize*0.5-Round, zSize*0.5>, Round }
    }
    texture {
      T_WoodA
      normal {
        function { fn_Wood(x,y,z).grey*Strength } 6.0
        accuracy IW_ISOaccuracy
      }
    }
    #break
  #case (2)
    #local fnShape =function{ f_rounded_box(x, y, z, Round,xSize*0.5,ySize*0.5,zSize*0.5+Round)}
    #local fnShape2=function{ (abs(z)-zSize*0.5) }

    isosurface {
      function {
        max(
          fnShape(x,y,z) +fn_Wood(x,y,z).grey*Strength,
          fnShape2(x,y,z)+fn_Wood(x,y,z).grey*Strength2
        )
      }
      contained_by{
        box{ <-xSize*0.5-Strength*1.2,-ySize*0.5-Strength*1.2,-zSize*0.5-Strength2*1.2>,
             < xSize*0.5+Strength*1.2, ySize*0.5+Strength*1.2, zSize*0.5+Strength2*1.2> }
      }

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { IW_Plain_Tex }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//=== IW_Plank_Round_CutE - rounded box cut at the front/back increased roughness to ends ==
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Plank_Round_CutE (Seed, xSize, ySize, zSize, Rotate, Translate,
                            Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      pigment_pattern {
        planar
        rotate 90*x
        scale zSize*0.5
        color_map { [0 rgb 1][1 rgb 0] }
      }

      pigment_map {
        [0.2 Pigm color_map { [0 rgb 0.0][1 rgb Strength/Strength2 ] } ]
        [1.0 Pigm color_map { [0 rgb 0.0][1 rgb 1.0] } ]
      }

      translate <Trsx, Trsy, 0>
      rotate Rotx*x
      rotate Roty*y

      IW_Select_Warp (WarpNbr, Seed, xSize, ySize, zSize)
     }
   }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, 0>
     rotate Rotx*x
     rotate Roty*y

     IW_Select_Warp (WarpNbr, Seed, xSize, ySize, zSize)

   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (IW_Quality)
  #case (1)
    merge {
      box{ <-xSize*0.5+Round,-ySize*0.5,-zSize*0.5>,
           < xSize*0.5-Round, ySize*0.5, zSize*0.5> }
      box{ <-xSize*0.5+Round,-ySize*0.5+Round,-zSize*0.5>,
           <-xSize*0.5,ySize*0.5-Round, zSize*0.5> }
      box{ < xSize*0.5-Round,-ySize*0.5+Round,-zSize*0.5>,
           < xSize*0.5,ySize*0.5-Round, zSize*0.5> }
      cylinder { <-xSize*0.5+Round,-ySize*0.5+Round,-zSize*0.5>,
                 <-xSize*0.5+Round,-ySize*0.5+Round, zSize*0.5>, Round }
      cylinder { < xSize*0.5-Round,-ySize*0.5+Round,-zSize*0.5>,
                 < xSize*0.5-Round,-ySize*0.5+Round, zSize*0.5>, Round }
      cylinder { <-xSize*0.5+Round, ySize*0.5-Round,-zSize*0.5>,
                 <-xSize*0.5+Round, ySize*0.5-Round, zSize*0.5>, Round }
      cylinder { < xSize*0.5-Round, ySize*0.5-Round,-zSize*0.5>,
                 < xSize*0.5-Round, ySize*0.5-Round, zSize*0.5>, Round }
    }
    texture {
      T_WoodA
      normal {
        function { fn_Wood(x,y,z).grey*Strength } 6.0
        accuracy IW_ISOaccuracy
      }
    }
    #break
  #case (2)
    #local fnShape =function{ f_rounded_box(x, y, z, Round,xSize*0.5,ySize*0.5,zSize*0.5+Round)}
    #local fnShape2=function{ (abs(z)-zSize*0.5) }

    isosurface {
      function {
        max(
          fnShape(x,y,z),
          fnShape2(x,y,z)
        ) +fn_Wood(x,y,z).grey*Strength2
      }
      contained_by {
        box {
          <-xSize*0.5-max(Strength,Strength2)*1.2,
           -ySize*0.5-max(Strength,Strength2)*1.2,
           -zSize*0.5-max(Strength,Strength2)*1.2>,
          < xSize*0.5+max(Strength,Strength2)*1.2,
            ySize*0.5+max(Strength,Strength2)*1.2,
            zSize*0.5+max(Strength,Strength2)*1.2>
        }
      }

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { IW_Plain_Tex }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_Plank_Round_CutW - rounded box cut at the front/back with warp ==================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Plank_Round_CutW (Seed, xSize, ySize, zSize, Rotate, Translate,
                            Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y

      IW_Select_Warp (WarpNbr, Seed, xSize, ySize, zSize)
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm

     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

     IW_Select_Warp (WarpNbr, Seed, xSize, ySize, zSize)

   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (IW_Quality)
  #case (1)
    merge {
      box{ <-xSize*0.5+Round,-ySize*0.5,-zSize*0.5>,
           < xSize*0.5-Round, ySize*0.5, zSize*0.5> }
      box{ <-xSize*0.5+Round,-ySize*0.5+Round,-zSize*0.5>,
           <-xSize*0.5,ySize*0.5-Round, zSize*0.5> }
      box{ < xSize*0.5-Round,-ySize*0.5+Round,-zSize*0.5>,
           < xSize*0.5,ySize*0.5-Round, zSize*0.5> }
      cylinder { <-xSize*0.5+Round,-ySize*0.5+Round,-zSize*0.5>,
                 <-xSize*0.5+Round,-ySize*0.5+Round, zSize*0.5>, Round }
      cylinder { < xSize*0.5-Round,-ySize*0.5+Round,-zSize*0.5>,
                 < xSize*0.5-Round,-ySize*0.5+Round, zSize*0.5>, Round }
      cylinder { <-xSize*0.5+Round, ySize*0.5-Round,-zSize*0.5>,
                 <-xSize*0.5+Round, ySize*0.5-Round, zSize*0.5>, Round }
      cylinder { < xSize*0.5-Round, ySize*0.5-Round,-zSize*0.5>,
                 < xSize*0.5-Round, ySize*0.5-Round, zSize*0.5>, Round }
    }
    texture {
      T_WoodA
      normal {
        function { fn_Wood(x,y,z).grey*Strength } 6.0
        accuracy IW_ISOaccuracy
      }
    }
    #break
  #case (2)
    #local fnShape =function{ f_rounded_box(x, y, z, Round,xSize*0.5,ySize*0.5,zSize*0.5+Round)}
    #local fnShape2=function{ (abs(z)-zSize*0.5) }

    isosurface {
      function {
        max(
          fnShape(x,y,z) +fn_Wood(x,y,z).grey*Strength,
          fnShape2(x,y,z)+fn_Wood(x,y,z).grey*Strength2
        )
      }
      contained_by {
        box{ <-xSize*0.5-Strength*1.2,-ySize*0.5-Strength*1.2,-zSize*0.5-Strength2*1.2>,
             < xSize*0.5+Strength*1.2, ySize*0.5+Strength*1.2, zSize*0.5+Strength2*1.2> }
      }

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { IW_Plain_Tex }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_Plank_Round_CutT - rounded box cut at the front/back with separate texture ======
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Plank_Round_CutT (Seed, xSize, ySize, zSize, Rotate, Translate,
                            Round, Strength, Strength2, Pigm, Text)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y
}

object {

#switch (IW_Quality)
  #case (1)
    merge {
      box{ <-xSize*0.5+Round,-ySize*0.5,-zSize*0.5>,
           < xSize*0.5-Round, ySize*0.5, zSize*0.5> }
      box{ <-xSize*0.5+Round,-ySize*0.5+Round,-zSize*0.5>,
           <-xSize*0.5,ySize*0.5-Round, zSize*0.5> }
      box{ < xSize*0.5-Round,-ySize*0.5+Round,-zSize*0.5>,
           < xSize*0.5,ySize*0.5-Round, zSize*0.5> }
      cylinder { <-xSize*0.5+Round,-ySize*0.5+Round,-zSize*0.5>,
                 <-xSize*0.5+Round,-ySize*0.5+Round, zSize*0.5>, Round }
      cylinder { < xSize*0.5-Round,-ySize*0.5+Round,-zSize*0.5>,
                 < xSize*0.5-Round,-ySize*0.5+Round, zSize*0.5>, Round }
      cylinder { <-xSize*0.5+Round, ySize*0.5-Round,-zSize*0.5>,
                 <-xSize*0.5+Round, ySize*0.5-Round, zSize*0.5>, Round }
      cylinder { < xSize*0.5-Round, ySize*0.5-Round,-zSize*0.5>,
                 < xSize*0.5-Round, ySize*0.5-Round, zSize*0.5>, Round }
    }
    texture {
      T_WoodA
      normal {
        function { fn_Wood(x,y,z).grey*Strength } 6.0
        accuracy IW_ISOaccuracy
      }
    }
    #break
  #case (2)
    #local fnShape =function{ f_rounded_box(x, y, z, Round,xSize*0.5,ySize*0.5,zSize*0.5+0.2)}
    #local fnShape2=function{ (abs(z)-zSize*0.5) }


    isosurface{
      function {
        max(
          fnShape(x,y,z) +fn_Wood(x,y,z).grey*Strength,
          fnShape2(x,y,z)+fn_Wood(x,y,z).grey*Strength2
        )
      }
      contained_by{
        box{ <-xSize*0.5-Strength*1.2,-ySize*0.5-Strength*1.2,-zSize*0.5-Strength2*1.2>,
             < xSize*0.5+Strength*1.2, ySize*0.5+Strength*1.2, zSize*0.5+Strength2*1.2> }
      }

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { IW_Plain_Tex }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_Plank_DRound - double rounded box ===============================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------
#macro IW_Plank_DRound (Seed, xSize, ySize, zSize, Rotate, Translate,
                        Round, Round2, Strength, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}


object {

#switch (IW_Quality)
  #case (1)
    merge {
      box {
        <-xSize*0.5+Round2,-ySize*0.5, -zSize*0.5+Round>,
        < xSize*0.5-Round2, ySize*0.5,  zSize*0.5-Round>
      }
      box {
        <-xSize*0.5+Round, -ySize*0.5, -zSize*0.5+Round2>,
        <-xSize*0.5+Round2, ySize*0.5,  zSize*0.5-Round2>
      }
      box {
        < xSize*0.5-Round, -ySize*0.5, -zSize*0.5+Round2>,
        < xSize*0.5-Round2, ySize*0.5,  zSize*0.5-Round2>
      }

      box {
        <-xSize*0.5+Round2,-ySize*0.5+Round, -zSize*0.5>,
        < xSize*0.5-Round2, ySize*0.5-Round,  zSize*0.5>
      }

      box {
        <-xSize*0.5,       -ySize*0.5+Round, -zSize*0.5+Round2>,
        <-xSize*0.5+Round,  ySize*0.5-Round,  zSize*0.5-Round2>
      }
      box {
        < xSize*0.5,       -ySize*0.5+Round, -zSize*0.5+Round2>,
        < xSize*0.5-Round,  ySize*0.5-Round,  zSize*0.5-Round2>
      }

      /*
      box {
        <-xSize*0.5+Round2,-ySize*0.5,  zSize*0.5-Round2>,
        < xSize*0.5-Round2, ySize*0.5,  zSize*0.5-Round>
      }
      box {
        <-xSize*0.5,-ySize*0.5+Round, -zSize*0.5+Round2>,
        < xSize*0.5, ySize*0.5-Round,  zSize*0.5-Round2>
      }
      box {
        <-xSize*0.5+Round,-ySize*0.5, -zSize*0.5+Round2>,
        < xSize*0.5-Round, ySize*0.5,  zSize*0.5-Round2>
      }  */

      cylinder {
        <-xSize*0.5+Round,-ySize*0.5+Round, -zSize*0.5+Round2>,
        <-xSize*0.5+Round,-ySize*0.5+Round,  zSize*0.5-Round2>,
        Round
      }
      cylinder {
        <xSize*0.5-Round,-ySize*0.5+Round, -zSize*0.5+Round2>,
        <xSize*0.5-Round,-ySize*0.5+Round,  zSize*0.5-Round2>,
        Round
      }
      cylinder {
        <-xSize*0.5+Round, ySize*0.5-Round, -zSize*0.5+Round2>,
        <-xSize*0.5+Round, ySize*0.5-Round,  zSize*0.5-Round2>,
        Round
      }
      cylinder {
        <xSize*0.5-Round, ySize*0.5-Round, -zSize*0.5+Round2>,
        <xSize*0.5-Round, ySize*0.5-Round,  zSize*0.5-Round2>,
        Round
      }
      cylinder {
        <-xSize*0.5+Round2,-ySize*0.5+Round, -zSize*0.5+Round>,
        < xSize*0.5-Round2,-ySize*0.5+Round, -zSize*0.5+Round>,
        Round
      }
      cylinder {
        <-xSize*0.5+Round2,-ySize*0.5+Round,  zSize*0.5-Round>,
        < xSize*0.5-Round2,-ySize*0.5+Round,  zSize*0.5-Round>,
        Round
      }
      cylinder {
        <-xSize*0.5+Round2, ySize*0.5-Round, -zSize*0.5+Round>,
        < xSize*0.5-Round2, ySize*0.5-Round, -zSize*0.5+Round>,
        Round
      }
      cylinder {
        <-xSize*0.5+Round2, ySize*0.5-Round,  zSize*0.5-Round>,
        < xSize*0.5-Round2, ySize*0.5-Round,  zSize*0.5-Round>,
        Round
      }
      Round_Cylinder_Merge(
        <-xSize*0.5+Round2,-ySize*0.5,-zSize*0.5+Round2>
        <-xSize*0.5+Round2, ySize*0.5,-zSize*0.5+Round2>
        Round2, Round
      )
      Round_Cylinder_Merge(
        <-xSize*0.5+Round2,-ySize*0.5, zSize*0.5-Round2>
        <-xSize*0.5+Round2, ySize*0.5, zSize*0.5-Round2>
        Round2, Round
      )
      Round_Cylinder_Merge(
        < xSize*0.5-Round2,-ySize*0.5,-zSize*0.5+Round2>
        < xSize*0.5-Round2, ySize*0.5,-zSize*0.5+Round2>
        Round2, Round
      )
      Round_Cylinder_Merge(
        < xSize*0.5-Round2,-ySize*0.5, zSize*0.5-Round2>
        < xSize*0.5-Round2, ySize*0.5, zSize*0.5-Round2>
        Round2, Round
      )
    }
    texture {
      T_WoodA
      normal {
        function { fn_Wood(x,y,z).grey*Strength } 6.0
        accuracy IW_ISOaccuracy
      }
    }
    #break
  #case (2)
    #local fnShape1=function{ f_rounded_box(x, y, z, Round,xSize*0.5-Round2+Round,ySize*0.5,zSize*0.5)}
    #local fnShape2=function{ f_rounded_box(x, y, z, Round,xSize*0.5,ySize*0.5,zSize*0.5-Round2+Round)}
    #local fnCorner=
      function{
        min(
          f_torus(x, y-ySize*0.5+Round, z, Round2-Round, Round),
          min(
            max(
              f_r(x, 0, z)-Round2,
              y-ySize*0.5+Round
            ),
            max(
              f_r(x, 0, z)-Round2+Round,
              y-ySize*0.5
            )
          )
        )
      }

    #local fnShape=
      function{
        min(
          min(
            fnShape1(x,y,z),
            fnShape2(x,y,z)
          ),
          fnCorner(x-xSize*0.5+Round2,y,z-zSize*0.5+Round2)
        )
      }

    isosurface {
      function { fnShape(abs(x),abs(y),abs(z))+fn_Wood(x,y,z).grey*Strength }
      contained_by {
        box{ <-xSize*0.5-Strength*1.2,-ySize*0.5-Strength*1.2,-zSize*0.5-Strength*1.2>,
             < xSize*0.5+Strength*1.2, ySize*0.5+Strength*1.2, zSize*0.5+Strength*1.2> }
      }

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { IW_Plain_Tex }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}

#end  //------------------------------------------------------------------------------------



//==========================================================================================
//===== IW_Plank_Chamfered - chamfered box cut at the front/back ===========================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Plank_Chamfered (Seed, xSize, ySize, zSize, Rotate, Translate,
                           Round, Round2, Strength, Strength2, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (IW_Quality)
  #case (1)
    intersection {
      box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
           < xSize*0.5, ySize*0.5, zSize*0.5> }
      plane {  x+y, -Round translate < xSize*0.5, ySize*0.5, 0> }
      plane {  x-y, -Round translate < xSize*0.5,-ySize*0.5, 0> }
      plane { -x+y, -Round translate <-xSize*0.5, ySize*0.5, 0> }
      plane { -x-y, -Round translate <-xSize*0.5,-ySize*0.5, 0> }

      plane {  x+z, -Round2 translate < xSize*0.5, 0, zSize*0.5> }
      plane {  x-z, -Round2 translate < xSize*0.5, 0,-zSize*0.5> }
      plane { -x+z, -Round2 translate <-xSize*0.5, 0, zSize*0.5> }
      plane { -x-z, -Round2 translate <-xSize*0.5, 0,-zSize*0.5> }

      plane {  y+z, -Round2 translate < 0, ySize*0.5, zSize*0.5> }
      plane {  y-z, -Round2 translate < 0, ySize*0.5,-zSize*0.5> }
      plane { -y+z, -Round2 translate < 0,-ySize*0.5, zSize*0.5> }
      plane { -y-z, -Round2 translate < 0,-ySize*0.5,-zSize*0.5> }
    }
    texture {
      T_WoodA
      normal {
        function { fn_Wood(x,y,z).grey*Strength } 6.0
        accuracy IW_ISOaccuracy
      }
    }
    #break
  #case (2)
    #local fnShape =function{ max((abs(x)-xSize*0.5), (abs(y)-ySize*0.5)) }
    #local fnShape2=function{ (abs(z)-zSize*0.5) }
    #local fnShape3=function{ (abs(x)+abs(y)-(xSize+ySize)*0.5)+Round }
    #local fnShape4=function{ max(
                        ((abs(z)+abs(y)-(zSize+ySize)*0.5)+Round2),
                        ((abs(z)+abs(x)-(zSize+xSize)*0.5)+Round2) ) }


    isosurface{
      function {
        max(
          max(fnShape (x,y,z), fnShape3(x,y,z))+fn_Wood(x,y,z).grey*Strength,
          max(fnShape2(x,y,z), fnShape4(x,y,z))+fn_Wood(x,y,z).grey*Strength2
        )
      }
      contained_by{
        box{
          <-xSize*0.5-max(Strength,Strength2)*1.2,
           -ySize*0.5-max(Strength,Strength2)*1.2,
           -zSize*0.5-max(Strength,Strength2)*1.2>,
          < xSize*0.5+max(Strength,Strength2)*1.2,
            ySize*0.5+max(Strength,Strength2)*1.2,
            zSize*0.5+max(Strength,Strength2)*1.2>
        }
      }

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { IW_Plain_Tex }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------



//==========================================================================================
//== IW_Plank_ChamferedE - chamfered box cut at the front/back increased roughness to ends =
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Plank_ChamferedE (Seed, xSize, ySize, zSize, Rotate, Translate,
                            Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      gradient z
      triangle_wave

      pigment_map {
        [0.2 Pigm color_map { [0 rgb 0.0][1 rgb Strength/Strength2 ] } scale 1/zSize ]
        [1.0 Pigm color_map { [0 rgb 0.0][1 rgb 1.0] } scale 1/zSize ]
      }

      scale zSize

      translate <Trsx, Trsy, 0>
      rotate Rotx*x
      rotate Roty*y

      IW_Select_Warp (WarpNbr, Seed, xSize, ySize, zSize)
     }
   }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

     IW_Select_Warp (WarpNbr, Seed, xSize, ySize, zSize)
   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (IW_Quality)
  #case (1)
    intersection {
      box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
           < xSize*0.5, ySize*0.5, zSize*0.5> }
      plane {  x+y, -Round translate < xSize*0.5, ySize*0.5, 0> }
      plane {  x-y, -Round translate < xSize*0.5,-ySize*0.5, 0> }
      plane { -x+y, -Round translate <-xSize*0.5, ySize*0.5, 0> }
      plane { -x-y, -Round translate <-xSize*0.5,-ySize*0.5, 0> }

      plane {  x+z, -Round2 translate < xSize*0.5, 0, zSize*0.5> }
      plane {  x-z, -Round2 translate < xSize*0.5, 0,-zSize*0.5> }
      plane { -x+z, -Round2 translate <-xSize*0.5, 0, zSize*0.5> }
      plane { -x-z, -Round2 translate <-xSize*0.5, 0,-zSize*0.5> }

      plane {  y+z, -Round2 translate < 0, ySize*0.5, zSize*0.5> }
      plane {  y-z, -Round2 translate < 0, ySize*0.5,-zSize*0.5> }
      plane { -y+z, -Round2 translate < 0,-ySize*0.5, zSize*0.5> }
      plane { -y-z, -Round2 translate < 0,-ySize*0.5,-zSize*0.5> }
    }
    texture {
      T_WoodA
      normal {
        function { fn_Wood(x,y,z).grey*Strength } 6.0
        accuracy IW_ISOaccuracy
      }
    }
    #break
  #case (2)
    #local fnShape =function{ max((abs(x)-xSize*0.5), (abs(y)-ySize*0.5)) }
    #local fnShape2=function{ (abs(z)-zSize*0.5) }
    #local fnShape3=function{ (abs(x)+abs(y)-(xSize+ySize)*0.5)+Round }
    #local fnShape4=function{ max(
                        ((abs(z)+abs(y)-(zSize+ySize)*0.5)+Round2),
                        ((abs(z)+abs(x)-(zSize+xSize)*0.5)+Round2) ) }


    isosurface{
      function {
        max(
          max(fnShape (x,y,z), fnShape3(x,y,z)),
          max(fnShape2(x,y,z), fnShape4(x,y,z))
        )+fn_Wood(x,y,z).grey*Strength2
      }
      contained_by{
        box{ <-xSize*0.5-Strength2*1.2,-ySize*0.5-Strength2*1.2,-zSize*0.5-Strength2*1.2>,
             < xSize*0.5+Strength2*1.2, ySize*0.5+Strength2*1.2, zSize*0.5+Strength2*1.2> }
      }

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { IW_Plain_Tex }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------




//==========================================================================================
//===== IW_Plank_ChamferedW - chamfered box cut at the front/back with warp ================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Plank_ChamferedW (Seed, xSize, ySize, zSize, Rotate, Translate,
                            Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y

      IW_Select_Warp (WarpNbr, Seed, xSize, ySize, zSize)
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

     IW_Select_Warp (WarpNbr, Seed, xSize, ySize, zSize)
   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (IW_Quality)
  #case (1)
    intersection {
      box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
           < xSize*0.5, ySize*0.5, zSize*0.5> }
      plane {  x+y, -Round translate < xSize*0.5, ySize*0.5, 0> }
      plane {  x-y, -Round translate < xSize*0.5,-ySize*0.5, 0> }
      plane { -x+y, -Round translate <-xSize*0.5, ySize*0.5, 0> }
      plane { -x-y, -Round translate <-xSize*0.5,-ySize*0.5, 0> }

      plane {  x+z, -Round2 translate < xSize*0.5, 0, zSize*0.5> }
      plane {  x-z, -Round2 translate < xSize*0.5, 0,-zSize*0.5> }
      plane { -x+z, -Round2 translate <-xSize*0.5, 0, zSize*0.5> }
      plane { -x-z, -Round2 translate <-xSize*0.5, 0,-zSize*0.5> }

      plane {  y+z, -Round2 translate < 0, ySize*0.5, zSize*0.5> }
      plane {  y-z, -Round2 translate < 0, ySize*0.5,-zSize*0.5> }
      plane { -y+z, -Round2 translate < 0,-ySize*0.5, zSize*0.5> }
      plane { -y-z, -Round2 translate < 0,-ySize*0.5,-zSize*0.5> }
    }
    texture {
      T_WoodA
      normal {
        function { fn_Wood(x,y,z).grey*Strength } 6.0
        accuracy IW_ISOaccuracy
      }
    }
    #break
  #case (2)
    #local fnShape =function{ max((abs(x)-xSize*0.5), (abs(y)-ySize*0.5)) }
    #local fnShape2=function{ (abs(z)-zSize*0.5) }
    #local fnShape3=function{ (abs(x)+abs(y)-(xSize+ySize)*0.5)+Round }
    #local fnShape4=function{ max(
                        ((abs(z)+abs(y)-(zSize+ySize)*0.5)+Round2),
                        ((abs(z)+abs(x)-(zSize+xSize)*0.5)+Round2) ) }


    isosurface{
      function {
        max(
          max(fnShape (x,y,z), fnShape3(x,y,z))+fn_Wood(x,y,z).grey*Strength,
          max(fnShape2(x,y,z), fnShape4(x,y,z))+fn_Wood(x,y,z).grey*Strength2
        )
      }
      contained_by{
        box{
          <-xSize*0.5-max(Strength,Strength2)*1.2,
           -ySize*0.5-max(Strength,Strength2)*1.2,
           -zSize*0.5-max(Strength,Strength2)*1.2>,
          < xSize*0.5+max(Strength,Strength2)*1.2,
            ySize*0.5+max(Strength,Strength2)*1.2,
            zSize*0.5+max(Strength,Strength2)*1.2>
        }
      }

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { IW_Plain_Tex }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------



//==========================================================================================
//===== IW_Plank_ChamferedT - chamfered box cut at the front/back with separate texture ====
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Plank_ChamferedT (Seed, xSize, ySize, zSize, Rotate, Translate,
                            Round, Round2, Strength, Strength2, Pigm, Text)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y
}

object {

#switch (IW_Quality)
  #case (1)
    intersection {
      box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
           < xSize*0.5, ySize*0.5, zSize*0.5> }
      plane {  x+y, -Round translate < xSize*0.5, ySize*0.5, 0> }
      plane {  x-y, -Round translate < xSize*0.5,-ySize*0.5, 0> }
      plane { -x+y, -Round translate <-xSize*0.5, ySize*0.5, 0> }
      plane { -x-y, -Round translate <-xSize*0.5,-ySize*0.5, 0> }

      plane {  x+z, -Round2 translate < xSize*0.5, 0, zSize*0.5> }
      plane {  x-z, -Round2 translate < xSize*0.5, 0,-zSize*0.5> }
      plane { -x+z, -Round2 translate <-xSize*0.5, 0, zSize*0.5> }
      plane { -x-z, -Round2 translate <-xSize*0.5, 0,-zSize*0.5> }

      plane {  y+z, -Round2 translate < 0, ySize*0.5, zSize*0.5> }
      plane {  y-z, -Round2 translate < 0, ySize*0.5,-zSize*0.5> }
      plane { -y+z, -Round2 translate < 0,-ySize*0.5, zSize*0.5> }
      plane { -y-z, -Round2 translate < 0,-ySize*0.5,-zSize*0.5> }
    }
    texture {
      T_WoodA
      normal {
        function { fn_Wood(x,y,z).grey*Strength } 6.0
        accuracy IW_ISOaccuracy
      }
    }
    #break
  #case (2)
    #local fnShape =function{ max((abs(x)-xSize*0.5), (abs(y)-ySize*0.5)) }
    #local fnShape2=function{ (abs(z)-zSize*0.5) }
    #local fnShape3=function{ (abs(x)+abs(y)-(xSize+ySize)*0.5)+Round }
    #local fnShape4=function{ max(
                        ((abs(z)+abs(y)-(zSize+ySize)*0.5)+Round2),
                        ((abs(z)+abs(x)-(zSize+xSize)*0.5)+Round2) ) }


    isosurface{
      function {
        max(
          max(fnShape (x,y,z), fnShape3(x,y,z))+fn_Wood(x,y,z).grey*Strength,
          max(fnShape2(x,y,z), fnShape4(x,y,z))+fn_Wood(x,y,z).grey*Strength2
        )
      }
      contained_by{
        box{
          <-xSize*0.5-max(Strength,Strength2)*1.2,
           -ySize*0.5-max(Strength,Strength2)*1.2,
           -zSize*0.5-max(Strength,Strength2)*1.2>,
          < xSize*0.5+max(Strength,Strength2)*1.2,
            ySize*0.5+max(Strength,Strength2)*1.2,
            zSize*0.5+max(Strength,Strength2)*1.2>
        }
      }

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      texture {
        T_WoodA
      }
    }
    #break
#else  //0
  box{ <-xSize*0.5,-ySize*0.5,-zSize*0.5>,
       < xSize*0.5, ySize*0.5, zSize*0.5>
    texture { IW_Plain_Tex }
  }
#end

  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------



//==========================================================================================
//===== IW_Cylinder - cylindrical wood =====================================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Cylinder (Seed, Radius, Length, Rotate, Translate, Strength, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}


object {

#switch (IW_Quality)
  #case (1)
    cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius
      texture {
        T_WoodA
        normal {
          function { fn_Wood(x,y,z).grey*Strength } 6.0
          accuracy IW_ISOaccuracy
        }
      }
    }
    #break
  #case (2)

    #local fnShape =function{ sqrt(y*y + x*x) - Radius }
    #local fnShape2=function{ (abs(z)-Length*0.5) }

    isosurface{
      function {max(fnShape(x,y,z), fnShape2(x,y,z))+fn_Wood(x,y,z).grey*Strength}

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      contained_by{
         box{ < -Radius-Strength*1.2,-Radius-Strength*1.2, -Length*0.5-Strength*1.2>,
              <  Radius+Strength*1.2, Radius+Strength*1.2, Length*0.5+Strength*1.2> }
      }
      texture {
         T_WoodA
      }
    }
    #break
#else  //0
  cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius
    texture { IW_Plain_Tex }
  }
#end

  translate <0, 0, Length*0.5>
}




#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_CylinderT - cylindrical wood with separate texture  =============================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_CylinderT (Seed, Radius, Length, Rotate, Translate, Strength, Pigm, Text)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y
}


object {

#switch (IW_Quality)
  #case (1)
    cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius
      texture {
        T_WoodA
        normal {
          function { fn_Wood(x,y,z).grey*Strength } 6.0
          accuracy IW_ISOaccuracy
        }
      }
    }
    #break
  #case (2)

    #local fnShape =function{ sqrt(y*y + x*x) - Radius }
    #local fnShape2=function{ (abs(z)-Length*0.5) }

    isosurface{
      function {max(fnShape(x,y,z), fnShape2(x,y,z))+fn_Wood(x,y,z).grey*Strength}

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      contained_by{
         box{ < -Radius-Strength*1.2,-Radius-Strength*1.2, -Length*0.5-Strength*1.2>,
              <  Radius+Strength*1.2, Radius+Strength*1.2, Length*0.5+Strength*1.2> }
      }
      texture {
         T_WoodA
      }
    }
    #break
#else  //0
  cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius
    texture { IW_Plain_Tex }
  }
#end

  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_Cylinder_Chamfered - cylindrical wood with cut & chamfered ends =================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Cylinder_Chamfered (Seed, Radius, Length, Rotate, Translate, Round,
                              Strength, Strength2, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (IW_Quality)
  #case (1)
    intersection {
      cylinder { <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
      cone { < 0, 0,  Length*0.5+Radius-Round>, 0,
             < 0, 0, -Length*0.5-Round>, Length+Radius }
      cone { < 0, 0, -Length*0.5-Radius+Round>, 0,
             < 0, 0,  Length*0.5+Round>, Length+Radius }
      texture {
        T_WoodA
        normal {
          function { fn_Wood(x,y,z).grey*Strength } 6.0
          accuracy IW_ISOaccuracy
        }
      }
    }
    #break
  #case (2)

    #local fnShape =function{ sqrt(y*y + x*x) - Radius }
    #local fnShape2=function{ (abs(z)-Length*0.5) }
    #local fnShape3=function{ sqrt( x*x+y*y )+(abs(z)-(Length*0.5+Radius-Round))   }

    isosurface{
      function {
        max(
          fnShape(x,y,z)+fn_Wood(x,y,z).grey*Strength,
          max(fnShape2(x,y,z), fnShape3(x,y,z))+fn_Wood(x,y,z).grey*Strength2
        )
      }
      contained_by{
         box{ < -Radius-Strength*1.2,-Radius-Strength*1.2, -Length*0.5-Strength2*1.2>,
              <  Radius+Strength*1.2, Radius+Strength*1.2, Length*0.5+Strength2*1.2> }
      }

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      texture {
         T_WoodA
      }
    }

    #break
#else  //0
  cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius
    texture { IW_Plain_Tex }
  }
#end

  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------



//==========================================================================================
// IW_Cylinder_ChamferedT - cylindrical wood with cut & chamfered ends with separate texture
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Cylinder_ChamferedT (Seed, Radius, Length, Rotate, Translate, Round,
                     Strength, Strength2, Pigm, Text)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y
}

object {

#switch (IW_Quality)
  #case (1)
    intersection {
      cylinder { <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
      cone { < 0, 0,  Length*0.5+Radius-Round>, 0,
             < 0, 0, -Length*0.5-Round>, Length+Radius }
      cone { < 0, 0, -Length*0.5-Radius+Round>, 0,
             < 0, 0,  Length*0.5+Round>, Length+Radius }
      texture {
        T_WoodA
        normal {
          function { fn_Wood(x,y,z).grey*Strength } 6.0
          accuracy IW_ISOaccuracy
        }
      }
    }
    #break
  #case (2)

    #local fnShape =function{ sqrt(y*y + x*x) - Radius }
    #local fnShape2=function{ (abs(z)-Length*0.5) }
    #local fnShape3=function{ sqrt( x*x+y*y )+(abs(z)-(Length*0.5+Radius-Round))   }

    isosurface{
      function {
        max(
          fnShape(x,y,z)+fn_Wood(x,y,z).grey*Strength,
          max(fnShape2(x,y,z), fnShape3(x,y,z))+fn_Wood(x,y,z).grey*Strength2
        )
      }
      contained_by{
         box{ < -Radius-Strength*1.2,-Radius-Strength*1.2, -Length*0.5-Strength2*1.2>,
              <  Radius+Strength*1.2, Radius+Strength*1.2, Length*0.5+Strength2*1.2> }
      }

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      texture {
         T_WoodA
      }
    }

    #break
#else  //0
  cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius
    texture { IW_Plain_Tex }
  }
#end

  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------



//==========================================================================================
//===== IW_Cylinder_Half - half cylindrical wood ===========================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Cylinder_Half (Seed, Radius, Length, Rotate, Translate, Strength, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (IW_Quality)
  #case (1)
    intersection {
      cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
      plane { y, 0 }
      texture {
        T_WoodA
        normal {
          function { fn_Wood(x,y,z).grey*Strength } 6.0
          accuracy IW_ISOaccuracy
        }
      }
    }
    #break
  #case (2)

    #local fnShape =function{ sqrt(y*y + x*x) - Radius }
    #local fnShape2=function{ (abs(z)-Length*0.5) }
    #local fnHalf  =function{ -y }

    isosurface{
      function {
        max(
          max(fnShape(x,y,z), fnShape2(x,y,z)),
          fnHalf(x,y,z)
        )+fn_Wood(x,y,z).grey*Strength}

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      contained_by{
         box{ < -Radius-Strength*1.2,-Radius-Strength*1.2, -Length*0.5-Strength*1.2>,
              <  Radius+Strength*1.2, Radius+Strength*1.2, Length*0.5+Strength*1.2> }
      }
      texture {
         T_WoodA
      }
    }

    #break
#else  //0
  intersection {
    cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
    plane { y, 0 }
    texture { IW_Plain_Tex }
  }
#end

  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_Cylinder_HalfT - half cylindrical wood with separate texture ====================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Cylinder_HalfT (Seed, Radius, Length, Rotate, Translate, Strength, Pigm, Text)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y
}

object {

#switch (IW_Quality)
  #case (1)
    intersection {
      cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
      plane { y, 0 }
      texture {
        T_WoodA
        normal {
          function { fn_Wood(x,y,z).grey*Strength } 6.0
          accuracy IW_ISOaccuracy
        }
      }
    }
    #break
  #case (2)

    #local fnShape =function{ sqrt(y*y + x*x) - Radius }
    #local fnShape2=function{ (abs(z)-Length*0.5) }
    #local fnHalf  =function{ -y }

    isosurface{
      function {
        max(
          max(fnShape(x,y,z), fnShape2(x,y,z)),
          fnHalf(x,y,z)
        )+fn_Wood(x,y,z).grey*Strength}

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      contained_by{
         box{ < -Radius-Strength*1.2,-Radius-Strength*1.2, -Length*0.5-Strength*1.2>,
              <  Radius+Strength*1.2, Radius+Strength*1.2, Length*0.5+Strength*1.2> }
      }
      texture {
         T_WoodA
      }
    }

    #break
#else  //0
  intersection {
    cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
    plane { y, 0 }
    texture { IW_Plain_Tex }
  }
#end

  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------


//=========================================================================================
//===== IW_Cylinder_Half_Chamfered - half cylindrical wood with cut & chamfered ends ======
//=========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Cylinder_Half_Chamfered (Seed, Radius, Length, Rotate, Translate, Round,
                                   Strength, Strength2, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}

object {

#switch (IW_Quality)
  #case (1)
    intersection {
      cylinder { <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
      cone { < 0, 0,  Length*0.5+Radius-Round>, 0,
             < 0, 0, -Length*0.5-Round>, Length+Radius }
      cone { < 0, 0, -Length*0.5-Radius+Round>, 0,
             < 0, 0,  Length*0.5+Round>, Length+Radius }
      plane { y, 0 }
      texture {
        T_WoodA
        normal {
          function { fn_Wood(x,y,z).grey*Strength } 6.0
          accuracy IW_ISOaccuracy
        }
      }
    }
    #break
  #case (2)

    #local fnShape =function{ sqrt(y*y + x*x) - Radius }
    #local fnShape2=function{ (abs(z)-Length*0.5) }
    #local fnShape3=function{ sqrt( x*x+y*y )+(abs(z)-(Length*0.5+Radius-Round))   }
    #local fnHalf  =function{ -y }


    isosurface{
      function {
        max(
          fnShape(x,y,z)+fn_Wood(x,y,z).grey*Strength,
          max(
            max(fnShape2(x,y,z), fnShape3(x,y,z)),
            fnHalf(x, y, z)
          )+fn_Wood(x,y,z).grey*Strength2
        )
      }

      contained_by{
         box{
           < -Radius-max(Strength,Strength2)*1.2,
             -Radius-max(Strength,Strength2)*1.2,
             -Length*0.5-max(Strength,Strength2)*1.2>,
           <  Radius+max(Strength,Strength2)*1.2,
              Radius+max(Strength,Strength2)*1.2,
              Length*0.5+max(Strength,Strength2)*1.2>
         }
      }

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      texture {
         T_WoodA
      }
    }

    #break
#else  //0
  intersection {
    cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
    plane { y, 0 }
    texture { IW_Plain_Tex }
  }
#end

  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
// IW_Cylinder_Half_ChamferedT -
//        half cyl. wood with cut & chamfered ends with separate texture ===================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Cylinder_Half_ChamferedT (Seed, Radius, Length, Rotate, Translate, Round,
                      Strength, Strength2, Pigm, Text)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y
}

object {

#switch (IW_Quality)
  #case (1)
    intersection {
      cylinder { <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
      cone { < 0, 0,  Length*0.5+Radius-Round>, 0,
             < 0, 0, -Length*0.5-Round>, Length+Radius }
      cone { < 0, 0, -Length*0.5-Radius+Round>, 0,
             < 0, 0,  Length*0.5+Round>, Length+Radius }
      plane { y, 0 }
      texture {
        T_WoodA
        normal {
          function { fn_Wood(x,y,z).grey*Strength } 6.0
          accuracy IW_ISOaccuracy
        }
      }
    }
    #break
  #case (2)

    #local fnShape =function{ sqrt(y*y + x*x) - Radius }
    #local fnShape2=function{ (abs(z)-Length*0.5) }
    #local fnShape3=function{ sqrt( x*x+y*y )+(abs(z)-(Length*0.5+Radius-Round))   }
    #local fnHalf  =function{ -y }


    isosurface{
      function {
        max(
          fnShape(x,y,z)+fn_Wood(x,y,z).grey*Strength,
          max(
            max(fnShape2(x,y,z), fnShape3(x,y,z)),
            fnHalf(x, y, z)
          )+fn_Wood(x,y,z).grey*Strength2
        )
      }

      contained_by{
         box{
           < -Radius-max(Strength,Strength2)*1.2,
             -Radius-max(Strength,Strength2)*1.2,
             -Length*0.5-max(Strength,Strength2)*1.2>,
           <  Radius+max(Strength,Strength2)*1.2,
              Radius+max(Strength,Strength2)*1.2,
              Length*0.5+max(Strength,Strength2)*1.2>
         }
      }

      accuracy IW_ISOaccuracy
      max_gradient IW_ISOmaxgrad

      texture {
         T_WoodA
      }
    }

    #break
#else  //0
  intersection {
    cylinder{ <0,0,-Length*0.5>, < 0, 0, Length*0.5>, Radius }
    plane { y, 0 }
    texture { IW_Plain_Tex }
  }
#end

  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_Cylinder_Post_Round - cylindrical wood post with rounded end ====================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Cylinder_Post_Round (Seed, Radius, Length, Rotate, Translate,
                           Strength, Round, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}


#local fnShape =function{ sqrt(y*y + x*x) - Radius }
#local fnShape2=function{ max(z-Length+Round, fnShape(x,y,z)) }
#local fnShape3=function{ f_torus(x, z-Length+Round, y, Radius-Round, Round) }
#local fnShape4=function{ sqrt(y*y + x*x) - Radius + Round }
#local fnShape5=function{ max(z-Length, fnShape4(x,y,z)) }


isosurface{
  function {
    min(
      fnShape2(x,y,z),
      fnShape3(x,y,z),
      fnShape5(x,y,z)      
    )+fn_Wood(x,y,z).grey*Strength    
  }

  accuracy IW_ISOaccuracy
  max_gradient IW_ISOmaxgrad

  contained_by{
     box{ < -Radius-Strength*1.2,-Radius-Strength*1.2, -Strength*1.2>,
          <  Radius+Strength*1.2, Radius+Strength*1.2, Length+Round+Strength*1.2> }
  }
  texture {
     T_WoodA
  }
  //translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//= IW_Cylinder_Post_RoundT - cylindrical wood post with rounded end with separate texture =
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Cylinder_Post_RoundT (Seed, Radius, Length, Rotate, Translate,
                           Strength, Round, Pigm, Text)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y
}

#local fnShape =function{ sqrt(y*y + x*x) - Radius }
#local fnShape2=function{ max(z-Length+Round, fnShape(x,y,z)) }
#local fnShape3=function{ f_torus(x, z-Length+Round, y, Radius-Round, Round) }
#local fnShape4=function{ sqrt(y*y + x*x) - Radius + Round }
#local fnShape5=function{ max(z-Length, fnShape4(x,y,z)) }


isosurface{
  function {
    min(
      fnShape2(x,y,z),
      fnShape3(x,y,z),
      fnShape5(x,y,z)      
    )+fn_Wood(x,y,z).grey*Strength    
  }

  accuracy IW_ISOaccuracy
  max_gradient IW_ISOmaxgrad

  contained_by{
     box{ < -Radius-Strength*1.2,-Radius-Strength*1.2, -Strength*1.2>,
          <  Radius+Strength*1.2, Radius+Strength*1.2, Length+Round+Strength*1.2> }
  }
  texture {
     T_WoodA
  }
  //translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------

//==========================================================================================
//===== IW_Cylinder_Post_Broken - cylindrical wood post with broken end ====================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Cylinder_Post_Broken (Seed, Radius, Length, fn_Break, Rotate, Translate,
                                Strength, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}


#local fnShape =function{ sqrt(y*y + x*x) - Radius }
#local fnShape2=function{ z-Length+fn_Break(x, y, 0) }

isosurface{
  function {
    max(
      fnShape(x,y,z),
      fnShape2(x,y,z)
    )+fn_Wood(x,y,z).grey*Strength    
  }

  accuracy IW_ISOaccuracy
  max_gradient IW_ISOmaxgrad

  contained_by{
     box{ < -Radius-Strength*1.2,-Radius-Strength*1.2, -Strength*1.2>,
          <  Radius+Strength*1.2, Radius+Strength*1.2, Radius+Length+Strength*1.2> }
  }
  texture {
     T_WoodA
  }
}


#end  //------------------------------------------------------------------------------------

//==========================================================================================
//= IW_Cylinder_Post_BrokenT - cylindrical wood post with broken end with separate texture =
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Cylinder_Post_BrokenT (Seed, Radius, Length, fn_Break, Rotate, Translate,
                                Strength, Pigm, Text)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y
}

#local fnShape =function{ sqrt(y*y + x*x) - Radius }
#local fnShape2=function{ z-Length+fn_Break(x, y, 0) }

isosurface{
  function {
    max(
      fnShape(x,y,z),
      fnShape2(x,y,z)
    )+fn_Wood(x,y,z).grey*Strength    
  }

  accuracy IW_ISOaccuracy
  max_gradient IW_ISOmaxgrad

  contained_by{
     box{ < -Radius-Strength*1.2,-Radius-Strength*1.2, -Strength*1.2>,
          <  Radius+Strength*1.2, Radius+Strength*1.2, Radius+Length+Strength*1.2> }
  }
  texture {
     T_WoodA
  }
}


#end  //------------------------------------------------------------------------------------

//==========================================================================================
//===== IW_Cylinder_Spike4 - cylindrical wood spike - 4 cuts ===============================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Cylinder_Spike4 (Seed, Radius, Length, Rotate, Translate,
                           Strength, Strength2, Angle, Irreg, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}


#local fnShape =function{ sqrt(y*y + x*x) - Radius }
#local fnShape2=function{ (abs(z)-Length*0.5) }

#local Ang=rand(rd)*2*pi;
#local RndA=sin(Ang);
#local RndB=cos(Ang);
#local RndC=rand(rd)*Irreg + 1/max(tan(radians(min(Angle, 89.9))),0.001);
#local fnSpike1=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/2) + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + 1/max(tan(radians(min(Angle, 89.9))),0.001);
#local fnSpike2=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/2)*2 + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + 1/max(tan(radians(min(Angle, 89.9))),0.001);
#local fnSpike3=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/2)*3 + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + 1/max(tan(radians(min(Angle, 89.9))),0.001);
#local fnSpike4=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }


#local fnSpike=
  function{
    max(
      max(fnSpike1(x, y, z), fnSpike2(x, y, z)),
      max(fnSpike3(x, y, z), fnSpike4(x, y, z))
    )
  }


isosurface{
  function {
    max(
      max(fnShape(x,y,z), fnShape2(x,y,z))+fn_Wood(x,y,z).grey*Strength,
      fnSpike(x,y,z)+fn_Wood(x,y,z).grey*Strength2
    )
  }

  accuracy IW_ISOaccuracy
  max_gradient IW_ISOmaxgrad

  contained_by{
     box{ < -Radius-Strength*1.2,-Radius-Strength*1.2, -Length*0.5-Strength2*1.2>,
          <  Radius+Strength*1.2, Radius+Strength*1.2, Length*0.5+Strength2*1.2> }
  }
  texture {
     T_WoodA
  }
  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------



//==========================================================================================
//===== IW_Cylinder_Spike8 - cylindrical wood spike - 8 cuts ===============================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Cylinder_Spike8 (Seed, Radius, Length, Rotate, Translate,
                           Strength, Strength2, Angle, Irreg, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}


#local fnShape =function{ sqrt(y*y + x*x) - Radius }
#local fnShape2=function{ (abs(z)-Length*0.5) }

#local Ang=rand(rd)*2*pi;
#local RndA=sin(Ang);
#local RndB=cos(Ang);
#local RndC=rand(rd)*Irreg + 1/max(tan(radians(min(Angle, 89.9))),0.001);
#local fnSpike1=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/4) + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + 1/max(tan(radians(min(Angle, 89.9))),0.001);
#local fnSpike2=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/4)*2 + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + 1/max(tan(radians(min(Angle, 89.9))),0.001);
#local fnSpike3=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/4)*3 + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + 1/max(tan(radians(min(Angle, 89.9))),0.001);
#local fnSpike4=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/4)*4 + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + 1/max(tan(radians(min(Angle, 89.9))),0.001);
#local fnSpike5=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/4)*5 + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + 1/max(tan(radians(min(Angle, 89.9))),0.001);
#local fnSpike6=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/4)*6 + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + 1/max(tan(radians(min(Angle, 89.9))),0.001);
#local fnSpike7=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }
#local AngA=Ang+(pi/4)*7 + (rand(rd)-0.5)*Irreg;
#local RndA=sin(AngA);
#local RndB=cos(AngA);
#local RndC=rand(rd)*Irreg + 1/max(tan(radians(min(Angle, 89.9))),0.001);
#local fnSpike8=function{ (RndA*x + RndB*y + RndC*(z-Length*0.5)) }

#local fnSpike=
  function{
    max(
      max(
        max(fnSpike1(x, y, z), fnSpike2(x, y, z)),
        max(fnSpike3(x, y, z), fnSpike4(x, y, z))
      ),
      max(
        max(fnSpike5(x, y, z), fnSpike6(x, y, z)),
        max(fnSpike7(x, y, z), fnSpike8(x, y, z))
      )
    )
  }


isosurface{
  function {
    max(
      max(fnShape(x,y,z), fnShape2(x,y,z))+fn_Wood(x,y,z).grey*Strength,
      fnSpike(x,y,z)+fn_Wood(x,y,z).grey*Strength2
    )
  }

  accuracy IW_ISOaccuracy
  max_gradient IW_ISOmaxgrad

  contained_by{
     box{ < -Radius-Strength*1.2,-Radius-Strength*1.2, -Length*0.5-Strength2*1.2>,
          <  Radius+Strength*1.2, Radius+Strength*1.2, Length*0.5+Strength2*1.2> }
  }
  texture {
     T_WoodA
  }
  translate <0, 0, Length*0.5>
}


#end  //------------------------------------------------------------------------------------



//==========================================================================================
//==========================================================================================

#macro IW_Box_Round (Seed, Start, End, Rotate, Translate, Round, Strength, Pigm, Fin, Norm)

  #local StartP = Start+<0,0,0>;
  #local EndP = End+<0,0,0>;
  #local xSize = abs(EndP.x - StartP.x);
  #local ySize = abs(EndP.y - StartP.y);
  #local zSize = abs(EndP.z - StartP.z);

  object {
    IW_Plank_Round (Seed, xSize, ySize, zSize, Rotate, Translate, Round, Strength, Pigm, Fin, Norm)
    translate < min(EndP.x, StartP.x), min(EndP.y, StartP.y), min(EndP.z, StartP.z) >
  }
#end  //------------------------------------------------------------------------------------

#macro IW_Box_Round_Cut (Seed, Start, End, Rotate, Translate,
                         Round, Strength, Strength2, Pigm, Fin, Norm)

  #local StartP = Start+<0,0,0>;
  #local EndP = End+<0,0,0>;
  #local xSize = abs(EndP.x - StartP.x);
  #local ySize = abs(EndP.y - StartP.y);
  #local zSize = abs(EndP.z - StartP.z);

  object {
    IW_Plank_Round_Cut (Seed, xSize, ySize, zSize, Rotate, Translate,
                        Round, Strength, Strength2, Pigm, Fin, Norm)
    translate < min(EndP.x, StartP.x), min(EndP.y, StartP.y), min(EndP.z, StartP.z) >
  }
#end  //------------------------------------------------------------------------------------

#macro IW_Box_Chamfered (Seed, Start, End, Rotate, Translate,
                         Round, Round2, Strength, Strength2, Pigm, Fin, Norm)

  #local StartP = Start+<0,0,0>;
  #local EndP = End+<0,0,0>;
  #local xSize = abs(EndP.x - StartP.x);
  #local ySize = abs(EndP.y - StartP.y);
  #local zSize = abs(EndP.z - StartP.z);

  object {
    IW_Plank_Chamfered (Seed, xSize, ySize, zSize, Rotate, Translate,
                        Round, Round2, Strength, Strength2, Pigm, Fin, Norm)
    translate < min(EndP.x, StartP.x), min(EndP.y, StartP.y), min(EndP.z, StartP.z) >
  }
#end  //------------------------------------------------------------------------------------


//==========================================================================================
//==========================================================================================


#macro IW_Box_Round_CutE (Seed, Start, End, Rotate, Translate,
                          Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)

  #local StartP = Start+<0,0,0>;
  #local EndP = End+<0,0,0>;
  #local xSize = abs(EndP.x - StartP.x);
  #local ySize = abs(EndP.y - StartP.y);
  #local zSize = abs(EndP.z - StartP.z);

  object {
    IW_Plank_Round_CutE (Seed, xSize, ySize, zSize, Rotate, Translate,
                         Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
    translate < min(EndP.x, StartP.x), min(EndP.y, StartP.y), min(EndP.z, StartP.z) >
  }
#end  //------------------------------------------------------------------------------------

#macro IW_Box_ChamferedE (Seed, Start, End, Rotate, Translate,
                            Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)

  #local StartP = Start+<0,0,0>;
  #local EndP = End+<0,0,0>;
  #local xSize = abs(EndP.x - StartP.x);
  #local ySize = abs(EndP.y - StartP.y);
  #local zSize = abs(EndP.z - StartP.z);

  object {
    IW_Plank_ChamferedE (Seed, xSize, ySize, zSize, Rotate, Translate,
                         Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
    translate < min(EndP.x, StartP.x), min(EndP.y, StartP.y), min(EndP.z, StartP.z) >
  }
#end  //------------------------------------------------------------------------------------


//==========================================================================================
//==========================================================================================

#macro IW_Box_RoundW (Seed, Start, End, Rotate, Translate,
                      Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)

  #local StartP = Start+<0,0,0>;
  #local EndP = End+<0,0,0>;
  #local xSize = abs(EndP.x - StartP.x);
  #local ySize = abs(EndP.y - StartP.y);
  #local zSize = abs(EndP.z - StartP.z);

  object {
    IW_Plank_RoundW (Seed, xSize, ySize, zSize, Rotate, Translate,
                   Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
    translate < min(EndP.x, StartP.x), min(EndP.y, StartP.y), min(EndP.z, StartP.z) >
  }
#end  //------------------------------------------------------------------------------------


#macro IW_Box_Round_CutW (Seed, Start, End, Rotate, Translate,
                          Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)

  #local StartP = Start+<0,0,0>;
  #local EndP = End+<0,0,0>;
  #local xSize = abs(EndP.x - StartP.x);
  #local ySize = abs(EndP.y - StartP.y);
  #local zSize = abs(EndP.z - StartP.z);

  object {
    IW_Plank_Round_CutW (Seed, xSize, ySize, zSize, Rotate, Translate,
                         Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
    translate < min(EndP.x, StartP.x), min(EndP.y, StartP.y), min(EndP.z, StartP.z) >
  }
#end  //------------------------------------------------------------------------------------

#macro IW_Box_ChamferedW (Seed, Start, End, Rotate, Translate,
                          Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)

  #local StartP = Start+<0,0,0>;
  #local EndP = End+<0,0,0>;
  #local xSize = abs(EndP.x - StartP.x);
  #local ySize = abs(EndP.y - StartP.y);
  #local zSize = abs(EndP.z - StartP.z);

  object {
    IW_Plank_ChamferedW (Seed, xSize, ySize, zSize, Rotate, Translate,
                         Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
    translate < min(EndP.x, StartP.x), min(EndP.y, StartP.y), min(EndP.z, StartP.z) >
  }
#end  //------------------------------------------------------------------------------------


//==========================================================================================
//==========================================================================================


#macro IW_Box_RoundT (Seed, Start, End, Rotate, Translate, Round, Strength, Pigm, Text)

  #local StartP = Start+<0,0,0>;
  #local EndP = End+<0,0,0>;
  #local xSize = abs(EndP.x - StartP.x);
  #local ySize = abs(EndP.y - StartP.y);
  #local zSize = abs(EndP.z - StartP.z);

  object {
    IW_Plank_RoundT (Seed, xSize, ySize, zSize, Rotate, Translate, Round, Strength, Pigm, Text)
    translate < min(EndP.x, StartP.x), min(EndP.y, StartP.y), min(EndP.z, StartP.z) >
  }
#end  //------------------------------------------------------------------------------------

#macro IW_Box_Round_CutT (Seed, Start, End, Rotate, Translate,
                          Round, Strength, Strength2, Pigm, Fin, Norm)

  #local StartP = Start+<0,0,0>;
  #local EndP = End+<0,0,0>;
  #local xSize = abs(EndP.x - StartP.x);
  #local ySize = abs(EndP.y - StartP.y);
  #local zSize = abs(EndP.z - StartP.z);

  object {
    IW_Plank_Round_CutT (Seed, xSize, ySize, zSize, Rotate, Translate,
                         Round, Strength, Strength2, Pigm, Text)
    translate < min(EndP.x, StartP.x), min(EndP.y, StartP.y), min(EndP.z, StartP.z) >
  }
#end  //------------------------------------------------------------------------------------

#macro IW_Box_ChamferedT (Seed, Start, End, Rotate, Translate,
                          Round, Round2, Strength, Strength2, Pigm, Text)

  #local StartP = Start+<0,0,0>;
  #local EndP = End+<0,0,0>;
  #local xSize = abs(EndP.x - StartP.x);
  #local ySize = abs(EndP.y - StartP.y);
  #local zSize = abs(EndP.z - StartP.z);

  object {
    IW_Plank_ChamferedT (Seed, xSize, ySize, zSize, Rotate, Translate,
                         Round, Round2, Strength, Strength2, Pigm, Text)
    translate < min(EndP.x, StartP.x), min(EndP.y, StartP.y), min(EndP.z, StartP.z) >
  }
#end  //------------------------------------------------------------------------------------

//==========================================================================================
//==========================================================================================


//==========================================================================================
//===== IW_Plank_RoundN - rounded box (distorted) ==========================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Plank_RoundN (Seed, xSize, ySize, zSize, Rotate, Translate, Round,
                        Strength, Pigm, Fin, Norm, C_Map, WarpNbr, fn_Disp,
                        fn_Max_Val)

#local rd=seed(Seed);

#local fn_Max=fn_Max_Val+<0,0,0>;

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y

      IW_Select_Warp (WarpNbr, Seed, xSize, ySize, zSize)
    }
  }

#local fnShape =function{ f_rounded_box(x, y, z, Round,xSize*0.5,ySize*0.5,zSize*0.5)}

#local fnShapeT =function{ fnShape (x+fn_Disp(x, y, z).red,y+fn_Disp(x, y, z).green,z+fn_Disp(x, y, z).blue) }
#local fnBumpsT =function{ fn_Wood (x+fn_Disp(x, y, z).red,y+fn_Disp(x, y, z).green,z+fn_Disp(x, y, z).blue).grey }

#local T_WoodA = texture {
   pigment {
     function { fnBumpsT(x, y, z) }
     color_map { C_Map }
   }
   normal { Norm }
   finish { Fin }
}

isosurface{
  function {
    fnShapeT(x,y,z)+fnBumpsT(x,y,z)*Strength
  }
  contained_by{
     box{ <-xSize*0.5-fn_Max.x,-ySize*0.5-fn_Max.y,-zSize*0.5-fn_Max.z>,
          < xSize*0.5+fn_Max.x, ySize*0.5+fn_Max.y, zSize*0.5+fn_Max.z> }
  }

  accuracy IW_ISOaccuracy
  max_gradient IW_ISOmaxgrad

  texture {
    T_WoodA
  }
  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_Plank_Round_CutN - rounded box cut at the front/back (distorted) ================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Plank_Round_CutN (Seed, xSize, ySize, zSize, Rotate, Translate, Round,
                            Strength, Strength2, Pigm, Fin, Norm, C_Map, WarpNbr,
                            fn_Disp, fn_Max_Val)

#local rd=seed(Seed);

#local fn_Max=fn_Max_Val+<0,0,0>;

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y

      IW_Select_Warp (WarpNbr, Seed, xSize, ySize, zSize)
    }
  }

#local fnShape =function{ f_rounded_box(x, y, z, Round,xSize*0.5,ySize*0.5,zSize*0.5+Round)}
#local fnShape2=function{ (abs(z)-zSize*0.5) }

#local fnShapeT =function{ fnShape (x+fn_Disp(x, y, z).red,y+fn_Disp(x, y, z).green,z+fn_Disp(x, y, z).blue) }
#local fnShape2T=function{ fnShape2(x+fn_Disp(x, y, z).red,y+fn_Disp(x, y, z).green,z+fn_Disp(x, y, z).blue) }
#local fnBumpsT =function{ fn_Wood (x+fn_Disp(x, y, z).red,y+fn_Disp(x, y, z).green,z+fn_Disp(x, y, z).blue).grey }

#local T_WoodA = texture {
   pigment {
     function { fnBumpsT(x, y, z) }
     color_map { C_Map }
   }
   normal { Norm }
   finish { Fin }
}

isosurface{
  function {
    max(
      fnShapeT(x,y,z)+fnBumpsT(x,y,z)*Strength,
      fnShape2T(x,y,z)+fnBumpsT(x,y,z)*Strength2
    )
  }
  contained_by{
     box{ <-xSize*0.5-fn_Max.x,-ySize*0.5-fn_Max.y,-zSize*0.5-fn_Max.z>,
          < xSize*0.5+fn_Max.x, ySize*0.5+fn_Max.y, zSize*0.5+fn_Max.z> }
  }

  accuracy IW_ISOaccuracy
  max_gradient IW_ISOmaxgrad

  texture {
    T_WoodA
  }
  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_Plank_ChamferedN - chamfered box cut at the front/back (distorted) ==============
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_Plank_ChamferedN (Seed, xSize, ySize, zSize, Rotate, Translate, Round, Round2,
                            Strength, Strength2, Pigm, Fin, Norm, C_Map, WarpNbr,
                            fn_Disp, fn_Max_Val)

#local rd=seed(Seed);

#local fn_Max=fn_Max_Val+<0,0,0>;

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y

      IW_Select_Warp (WarpNbr, Seed, xSize, ySize, zSize)
    }
  }

#local fnShape =function{ max(abs(x)-xSize*0.5, abs(y)-ySize*0.5) }
#local fnShape2=function{ (abs(z)-zSize*0.5) }
#local fnShape3=function{ (abs(x)+abs(y)-(xSize+ySize)*0.5)+Round }
#local fnShape4=function{ max((abs(z)+abs(y)-(zSize+ySize)*0.5)+Round2,
                              (abs(z)+abs(x)-(zSize+xSize)*0.5)+Round2) }


#local fnShapeT =function{ fnShape (x+fn_Disp(x, y, z).red,y+fn_Disp(x, y, z).green,z+fn_Disp(x, y, z).blue) }
#local fnShape2T=function{ fnShape2(x+fn_Disp(x, y, z).red,y+fn_Disp(x, y, z).green,z+fn_Disp(x, y, z).blue) }
#local fnShape3T=function{ fnShape3(x+fn_Disp(x, y, z).red,y+fn_Disp(x, y, z).green,z+fn_Disp(x, y, z).blue) }
#local fnShape4T=function{ fnShape4(x+fn_Disp(x, y, z).red,y+fn_Disp(x, y, z).green,z+fn_Disp(x, y, z).blue) }
#local fnBumpsT =function{ fn_Wood (x+fn_Disp(x, y, z).red,y+fn_Disp(x, y, z).green,z+fn_Disp(x, y, z).blue).grey }

#local T_WoodA = texture {
   pigment {
     function { fnBumpsT(x,y,z) }
     color_map { C_Map }
   }
   normal { Norm }
   finish { Fin }
}

isosurface{
  function {
    max(
      max(fnShapeT (x,y,z), fnShape3T(x,y,z))+fnBumpsT(x,y,z)*Strength,
      max(fnShape2T(x,y,z), fnShape4T(x,y,z))+fnBumpsT(x,y,z)*Strength2
    )
  }

  contained_by{
     box{ <-xSize*0.5-fn_Max.x,-ySize*0.5-fn_Max.y,-zSize*0.5-fn_Max.z>,
          < xSize*0.5+fn_Max.x, ySize*0.5+fn_Max.y, zSize*0.5+fn_Max.z> }
  }

  accuracy IW_ISOaccuracy
  max_gradient IW_ISOmaxgrad

  texture {
    T_WoodA
  }
  translate <xSize*0.5, ySize*0.5, zSize*0.5>
}


#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_CylinderN - cylindrical wood (distorted) ========================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_CylinderN (Seed, Radius, Length, Rotate, Translate, Strength,
                     Pigm, Fin, Norm, C_Map, fn_Disp, fn_Max_Val)

#local rd=seed(Seed);

#local fn_Max=fn_Max_Val+<0,0,0>;

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local fnShape =function{ sqrt(y*y + x*x) - Radius }
#local fnShape2=function{ (abs(z)-Length*0.5) }

#local fnShapeT =function{ fnShape (x+fn_Disp(x, y, z).red,y+fn_Disp(x, y, z).green,z+fn_Disp(x, y, z).blue) }
#local fnShape2T=function{ fnShape2(x+fn_Disp(x, y, z).red,y+fn_Disp(x, y, z).green,z+fn_Disp(x, y, z).blue) }
#local fnBumpsT =function{ fn_Wood (x+fn_Disp(x, y, z).red,y+fn_Disp(x, y, z).green,z+fn_Disp(x, y, z).blue).grey }

#local T_WoodA = texture {
   pigment {
     function { fnBumpsT(x,y,z) }
     color_map { C_Map }
   }
   normal { Norm }
   finish { Fin }
}

isosurface{
  function {
    max(
      fnShapeT(x,y,z),
      fnShape2T(x,y,z)
    )+fnBumpsT(x,y,z)*Strength
  }

  accuracy IW_ISOaccuracy
  max_gradient IW_ISOmaxgrad

  #if (IW_ISOmax_trace>0)
    max_trace IW_ISOmax_trace
  #end

  contained_by{
     box{ <-Radius-fn_Max.x,-Radius-fn_Max.y,-Length*0.5-fn_Max.z>,
          < Radius+fn_Max.x, Radius+fn_Max.y, Length*0.5+fn_Max.z> }
  }

  texture {
     T_WoodA
  }

  translate <0, 0, Length*0.5>
}



#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_CylinderR - cylindrical wood (variable radius) ==================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------

#macro IW_CylinderR (Seed, Radius, Length, Rotate, Translate, Strength,
                     Pigm, Fin, Norm, fn_Disp, fn_Max_Val)

#local rd=seed(Seed);

#local Trsx = (rand(rd)-0.5)*Translate*Radius;
#local Trsy = (rand(rd)-0.5)*Translate*Radius;
#local Trsz = (rand(rd)-0.5)*Translate*Radius;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}

#local fnShape =function{ sqrt(y*y + x*x) - Radius - fn_Disp(x,y,z) }
#local fnShape2=function{ (abs(z)-Length*0.5) }

isosurface{
  function {
    max(
      fnShape(x,y,z),
      fnShape2(x,y,z)
    )+fn_Wood(x,y,z).grey*Strength
  }

  accuracy IW_ISOaccuracy
  max_gradient IW_ISOmaxgrad

  #if (IW_ISOmax_trace>0)
    max_trace IW_ISOmax_trace
  #end

  contained_by{
     box{ < -Radius-fn_Max_Val,-Radius-fn_Max_Val, -Length*0.5-Strength*1.2>,
          <  Radius+fn_Max_Val, Radius+fn_Max_Val,  Length*0.5+Strength*1.2> }
  }

  texture {
     T_WoodA
  }

  translate <0, 0, Length*0.5>
}



#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_Shape - arbitrary wood shape ====================================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------
#macro IW_Shape (Seed, fn_Shape, Start, End, Rotate, Translate, Strength, Pigm, Fin, Norm)

#local rd=seed(Seed);

#local StartP=Start+<0,0,0>;
#local EndP=End+<0,0,0>;

#local xSize=abs(StartP.x-EndP.x);
#local ySize=abs(StartP.y-EndP.y);
#local zSize=abs(StartP.z-EndP.z);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y
   }
   normal { Norm }
   finish { Fin }
}


isosurface {
  function { fn_Shape(x,y,z)+fn_Wood(x,y,z).grey*Strength }
  contained_by {
    box{ StartP, EndP }
  }

  accuracy IW_ISOaccuracy
  max_gradient IW_ISOmaxgrad

  texture {
    T_WoodA
  }
}

#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_ShapeW - arbitrary wood shape with warp =========================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------
#macro IW_ShapeW (Seed, fn_Shape, Start, End, Rotate, Translate, Strength,
                  Pigm, Fin, Norm, WarpNbr)

#local rd=seed(Seed);

#local StartP=Start+<0,0,0>;
#local EndP=End+<0,0,0>;

#local xSize=abs(Start.x-End.x);
#local ySize=abs(Start.y-End.y);
#local zSize=abs(Start.z-End.z);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y

      IW_Select_Warp (WarpNbr, Seed, xSize, ySize, zSize)
    }
  }

#local T_WoodA = texture {
   pigment {
     Pigm
     translate <Trsx, Trsy, Trsz>
     rotate Rotx*x
     rotate Roty*y

     IW_Select_Warp (WarpNbr, Seed, xSize, ySize, zSize)
   }
   normal { Norm }
   finish { Fin }
}


isosurface {
  function { fn_Shape(x,y,z)+fn_Wood(x,y,z).grey*Strength }
  contained_by {
    box{ StartP, EndP }
  }

  accuracy IW_ISOaccuracy
  max_gradient IW_ISOmaxgrad

  texture {
    T_WoodA
  }
}

#end  //------------------------------------------------------------------------------------



//==========================================================================================
//===== IW_ShapeT - arbitrary wood shape with separate texture =============================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------
#macro IW_ShapeT (Seed, fn_Shape, Start, End, Rotate, Translate, Strength, Pigm, Text)

#local rd=seed(Seed);

#local StartP=Start+<0,0,0>;
#local EndP=End+<0,0,0>;

#local xSize=abs(Start.x-End.x);
#local ySize=abs(Start.y-End.y);
#local zSize=abs(Start.z-End.z);

#local Trsx = (rand(rd)-0.5)*Translate*xSize;
#local Trsy = (rand(rd)-0.5)*Translate*ySize;
#local Trsz = (rand(rd)-0.5)*Translate*zSize;
#local Rotx = (rand(rd)-0.5)*Rotate;
#local Roty = (rand(rd)-0.5)*Rotate;

#local fn_Wood =
  function {
    pigment {
      Pigm
      color_map {
        [ 0.000  color rgb 0.0 ]
        [ 1.000  color rgb 1.0 ]
      }
      translate <Trsx, Trsy, Trsz>
      rotate Rotx*x
      rotate Roty*y
    }
  }

#local T_WoodA = texture {
   Text
   translate <Trsx, Trsy, Trsz>
   rotate Rotx*x
   rotate Roty*y
}


isosurface {
  function { fn_Shape(x,y,z)+fn_Wood(x,y,z).grey*Strength }
  contained_by {
    box{ StartP, EndP }
  }

  accuracy IW_ISOaccuracy
  max_gradient IW_ISOmaxgrad

  texture {
    T_WoodA
  }
}

#end  //------------------------------------------------------------------------------------



//==========================================================================================
//===== IW_Parquet_A - Parquet of IW_Plank_X - pattern A ===================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------


#macro IW_Parquet_A (Seed, Start, End, xSize, ySize, Height, Rotate, Translate,
                     Round, Round2, Strength, Strength2, Pigm, Fin, Norm, Text, WarpNbr, Typ)

  #if (ySize > xSize)
    #local buf = xSize;
    #local xSize = ySize;
    #local ySize = buf;
  #end

  #if (mod(xSize, ySize)> 0)
    #error "xSize and ySize in Wood_Parquet_A do not fit"
  #else
    #debug "building parquet ...\n\n"

    #local Pl_Cnt=0;
    #local rd=seed(Seed);

    #local xASize = abs(End.x - Start.x);
    #local yASize = abs(End.y - Start.y);

    intersection {     //-------------------------------
     union {
      #local Acnt=0;
      #while ((Acnt < yASize+ySize) & (Acnt < xASize+ySize))

        //########################################################

        #local Ycnt=0;

        #while (Ycnt+Acnt < yASize+xSize)

          #local Xcnt=0;

          #while (Xcnt+Acnt < xASize+ySize)
            object { //===========================================
              #switch (IW_Quality)
                #case (1)
                #case (2)
                  #switch (Typ)
                    #case (1)
                      IW_Plank_Round (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                      Round, Strength, Pigm, Fin, Norm)
                      #break
                    #case (11)
                      IW_Plank_RoundW (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                       Round, Strength, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (12)
                      IW_Plank_RoundT (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                       Round, Strength, Pigm, Text)
                      #break
                    #case (2)
                      IW_Plank_Round_Cut (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                          Round, Strength, Strength2, Pigm, Fin, Norm)
                      #break
                    #case (20)
                      IW_Plank_Round_CutE (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                           Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (21)
                      IW_Plank_Round_CutW (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                           Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (22)
                      IW_Plank_Round_CutT (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                           Round, Strength, Strength2, Pigm, Text)
                      #break
                    #case (3)
                      IW_Plank_Chamfered (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                          Round, Round2, Strength, Strength2, Pigm, Fin, Norm)
                      #break
                    #case (30)
                      IW_Plank_ChamferedE (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                           Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (31)
                      IW_Plank_ChamferedW (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                           Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (32)
                      IW_Plank_ChamferedT (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                           Round, Round2, Strength, Strength2, Pigm, Text)
                  #end
                #break
                #else  //0
                  box { <0,0,0>, <Height, ySize, xSize> texture { pigment { color rgb <0,0,1> } }}
              #end


              rotate -90*y rotate 90*z
              translate < min(End.x, Start.x) + Acnt + Xcnt, min(End.y, Start.y) + Acnt + Ycnt, -Height >
            } //==================================================

            #local Pl_Cnt=Pl_Cnt + 1;
            #debug concat(str(Pl_Cnt,0,0),"\r")

            #local Xcnt = Xcnt + xSize*2;
          #end
          #local Ycnt = Ycnt + xSize*2;
        #end

        //########################################################
        #local Ycnt=0;

        #while (Ycnt+Acnt < yASize+ySize)

          #local Xcnt=0;

          #while (Xcnt+Acnt < xASize+xSize)

            object { //===========================================
              #switch (IW_Quality)
                #case (1)
                #case (2)
                  #switch (Typ)
                    #case (1)
                      IW_Plank_Round (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                      Round, Strength, Pigm, Fin, Norm)
                      #break
                    #case (11)
                      IW_Plank_RoundW (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                       Round, Strength, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (12)
                      IW_Plank_RoundT (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                       Round, Strength, Pigm, Text)
                      #break
                    #case (2)
                      IW_Plank_Round_Cut (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                          Round, Strength, Strength2, Pigm, Fin, Norm)
                      #break
                    #case (20)
                      IW_Plank_Round_CutE (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                           Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (21)
                      IW_Plank_Round_CutW (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                           Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (22)
                      IW_Plank_Round_CutT (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                           Round, Strength, Strength2, Pigm, Text)
                      #break
                    #case (3)
                      IW_Plank_Chamfered (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                          Round, Round2, Strength, Strength2, Pigm, Fin, Norm)
                      #break
                    #case (30)
                      IW_Plank_ChamferedE (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                           Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (31)
                      IW_Plank_ChamferedW (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                           Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                      #break
                    #case (32)
                      IW_Plank_ChamferedT (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                           Round, Round2, Strength, Strength2, Pigm, Text)
                  #end
                #break
                #else  //0
                  box { <0,0,0>, <Height, ySize, xSize> texture { pigment { color rgb <1,0,0> } }}
              #end

              rotate -90*y
              translate < min(End.x, Start.x) + Acnt + Xcnt, min(End.y, Start.y) + Acnt + Ycnt, -Height >
            } //==================================================

            #local Pl_Cnt=Pl_Cnt + 1;
            #debug concat(str(Pl_Cnt,0,0),"\r")

            #local Xcnt = Xcnt + xSize*2;
          #end
          #local Ycnt = Ycnt + xSize*2;
        #end
        #local Acnt = Acnt + ySize;
      #end

      //########################################################

     }
     box { <Start.x, Start.y, Height>, <End.x, End.y, -Height*2>
       texture { IW_Plain_Tex }
     }
    }

    #debug concat("parquet created (",str(Pl_Cnt,0,0) ," planks)\n")
  #end

#end  //------------------------------------------------------------------------------------


//==========================================================================================
//===== IW_Parquet_B - Parquet of IW_Plank_X - pattern B ===================================
//==========================================================================================

//------------------------------------------------------------------------------------------
//------------------------------------------------------------------------------------------


#macro IW_Parquet_B (Seed, Start, End, xSize, ySize, Height, Rotate, Translate,
                     Round, Round2, Strength, Strength2, Pigm, Fin, Norm, Text, WarpNbr, Typ)

  #if (ySize > xSize)
    #local buf = xSize;
    #local xSize = ySize;
    #local ySize = buf;
  #end

  #if (mod(xSize, ySize)> 0)
    #error "xSize and ySize in Wood_Parquet_B do not fit"
  #else
    #debug "building parquet ...\n\n"

    #local Pl_Cnt=0;
    #local rd=seed(Seed);

    #local xASize = abs(End.x - Start.x);
    #local yASize = abs(End.y - Start.y);

    intersection {     //-------------------------------
      union {
        #local Ycnt=0;
        #while (Ycnt < yASize)

          #local Xcnt=0;
          #while (Xcnt < xASize)

            #local Acnt=0;
            #while (Acnt < div(xSize, ySize))

              #if (mod(div(Xcnt,xSize)+div(Ycnt,xSize), 2)=0)

                object { //===========================================

                  #switch (IW_Quality)
                    #case (1)
                    #case (2)
                      #switch (Typ)
                        #case (1)
                          IW_Plank_Round (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                          Round, Strength, Pigm, Fin, Norm)
                          #break
                        #case (11)
                          IW_Plank_RoundW (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                           Round, Strength, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (12)
                          IW_Plank_RoundT (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                           Round, Strength, Pigm, Text)
                          #break
                        #case (2)
                          IW_Plank_Round_Cut (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                              Round, Strength, Strength2, Pigm, Fin, Norm)
                          #break
                        #case (20)
                          IW_Plank_Round_CutE (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                               Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (21)
                          IW_Plank_Round_CutW (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                               Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (22)
                          IW_Plank_Round_CutT (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                               Round, Strength, Strength2, Pigm, Text)
                          #break
                        #case (3)
                          IW_Plank_Chamfered (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                              Round, Round2, Strength, Strength2, Pigm, Fin, Norm)
                          #break
                        #case (30)
                          IW_Plank_ChamferedE (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                               Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (31)
                          IW_Plank_ChamferedW (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                               Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (32)
                          IW_Plank_ChamferedT (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                               Round, Round2, Strength, Strength2, Pigm, Text)
                      #end
                    #break
                    #else  //0
                      box { <0,0,0>, <Height, ySize, xSize> texture { pigment { color rgb <0,0,Acnt/5> } }}
                  #end


                  rotate 90*y
                  translate < min(End.x, Start.x) + Xcnt, min(End.y, Start.y) + Ycnt + Acnt*ySize, -Height >
                } //==================================================
              #else
                object { //===========================================

                  #switch (IW_Quality)
                    #case (1)
                    #case (2)
                      #switch (Typ)
                        #case (1)
                          IW_Plank_Round (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                          Round, Strength, Pigm, Fin, Norm)
                          #break
                        #case (11)
                          IW_Plank_RoundW (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                           Round, Strength, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (12)
                          IW_Plank_RoundT (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                           Round, Strength, Pigm, Text)
                          #break
                        #case (2)
                          IW_Plank_Round_Cut (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                              Round, Strength, Strength2, Pigm, Fin, Norm)
                          #break
                        #case (20)
                          IW_Plank_Round_CutE (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                               Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (21)
                          IW_Plank_Round_CutW (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                               Round, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (22)
                          IW_Plank_Round_CutT (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                               Round, Strength, Strength2, Pigm, Text)
                          #break
                        #case (3)
                          IW_Plank_Chamfered (rand(rd)*10000, Height, ySize, xSize, Rotate, Translate,
                                              Round, Round2, Strength, Strength2, Pigm, Fin, Norm)
                          #break
                        #case (30)
                          IW_Plank_ChamferedE (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                               Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (31)
                          IW_Plank_ChamferedW (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                               Round, Round2, Strength, Strength2, Pigm, Fin, Norm, WarpNbr)
                          #break
                        #case (32)
                          IW_Plank_ChamferedT (rand(rd)*1000, Height, ySize, xSize, Rotate, Translate,
                                               Round, Round2, Strength, Strength2, Pigm, Text)
                      #end
                    #break
                    #else  //0
                      box { <0,0,0>, <Height, ySize, xSize> texture { pigment { color rgb <Acnt/5,0,0> } }}
                  #end


                  rotate 90*y rotate 90*z
                  translate < min(End.x, Start.x) + Xcnt + (Acnt+1)*ySize, min(End.y, Start.y) + Ycnt, -Height >
                } //==================================================
              #end

              #local Pl_Cnt=Pl_Cnt + 1;
              #debug concat(str(Pl_Cnt,0,0),"\r")

              #local Acnt = Acnt + 1;
            #end

            #local Xcnt = Xcnt + xSize;
          #end
          #local Ycnt = Ycnt + xSize;
        #end
      }
      box { <Start.x, Start.y, Height>, <End.x, End.y, -Height*2>
        texture { IW_Plain_Tex }
      }
    }



    #debug concat("parquet created (",str(Pl_Cnt,0,0) ," planks)\n")
  #end

#end  //------------------------------------------------------------------------------------

