/* refract-spectral.pov
 * Persistence of Vision Ray Tracer scene description file
 *
 * A study in dispersion and caustics using spectral rendering.
 *
 * This scene requires:
 *  - POV-Ray 3.7 or later
 *  - Lightsys 4c     http://www.ignorancia.org/en/index.php?page=Lightsys
 *  - SpectralRender  http://www.lilysoft.org/CGI/SR/Spectral%20Render.htm
 *
 * This scene is based on the POV-Ray sample file diffract.pov, and modified
 * on 2014 March 02 by Richard Callwood III.  Some rights reserved.
 */
// This work is licensed under the Creative Commons Attribution-ShareAlike 3.0 Unported License.
// To view a copy of this license, visit http://creativecommons.org/licenses/by-sa/3.0/ or send a
// letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.

// +W800 +H600 +A0.1 +AM2 +FE +KI1 +KF36 +KFI38 +KFF73 Declare=Area_Light=1
// +W800 +H600 -A

#version 3.7;

#ifndef (Preview) #declare Preview = off; #end

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
#if (clock_on | Preview)

#ifndef (Area_Light) #declare Area_Light = off; #end
#ifndef (Dispersion) #declare Dispersion = 1; #end // exaggeration factor
#ifndef (Photons) #declare Photons = on; #end
#ifndef (Rad) #declare Rad = on; #end
#ifndef (Specular) #declare Specular = yes; #end

#include "spectral.inc"

#declare Pretrace = 1 / max (image_width, image_height);
global_settings
{ assumed_gamma 1
  max_trace_level 256
  #if (Rad)
    radiosity
    { count 50
      error_bound 0.5
      pretrace_start Pretrace * 64
      pretrace_end Pretrace * 2
      recursion_limit 2
    }
  #end
  #if (Photons) photons { spacing 0.0025 autostop 0 } #end
}

//----------------------------------------------------------------------
// This scene uses a non-standard camera set-up. 
// (See CAMERA in the included documentation for details.) 
// If you are new to POV-Ray, you might want to try a different demo scene.
//----------------------------------------------------------------------
camera {  //  Camera StdCam
  angle 45
  location  <3.50, -15.00, 3.00>
  direction <0.0,     0.0,  1.6542>
  sky       <0.0,     0.0,  1.0>  // Use right handed-system!
  up        <0.0,     0.0,  1.0>  // Where Z is up
  right x*image_width/image_height // keep proportions with any aspect ratio
  look_at   <0.000, 0.000, -2.7500>
}

#declare Intensity = 20;
#declare L_Fade_Distance = 20;
#declare L_Fade_Power = 2;

#declare L_Radius = 2;
// Attempt to match original diffract.pov light intensity
#declare L_Position = <0, 95, 65>;
#declare L_Distance = vlength (L_Position);
#declare Match = 2 / (1 + pow (L_Distance / L_Fade_Distance, L_Fade_Power));
#declare Adjust = Match * (1 + pow (L_Distance / L_Radius, 2)) / 2;
#declare CplusYplusM = 2; // Combined intensity of diffract.pov light sources
#declare Intensity = Intensity * CplusYplusM * Adjust;

#declare Light = E_Daylight (5750);
light_source
{ L_Position
  SpectralEmission (Light) * Intensity
  #if(Area_Light)
    area_light L_Radius * x, L_Radius * z, 9, 9 adaptive 1
    circular orient jitter
  #end
  fade_distance L_Radius
  fade_power 2
  #if (!Specular)
    looks_like
    { sphere
      { 0, L_Radius
        T_Spectral_Emitting (Light, Intensity)
        no_radiosity
      }
    }
  #end
}

sky_sphere
{ pigment
  { //gradient y
    gradient -z
    color_map
    { [0.0 C_Spectral (E_D65) * 0.1]
      [1.0 C_Spectral (E_D65) * 0.3]
    }
  }
}

#default
{ finish
  { ambient SpectralEmission (E_D65) / 6 diffuse 1
    #if (Specular)
      specular albedo 0.05 // Fresnel compromise
      roughness 0.0001
    #end
    #if (!Photons) caustics 1 #end // Yeah, I know.  See FlySpray #319.
  }
}

#declare Filter = IOR_Acryl;
#if (SpectralMode)
  #declare Filter [WavelengthIndex] =
    IOR_Acryl[17] + (IOR_Acryl [WavelengthIndex] - IOR_Acryl[17]) * Dispersion;
#end
#declare Material01 = M_Spectral_Filter (Value_1, Filter, 100)

union {
  cylinder {   <-3,0,0>, <3,0,0>, 0.3 }
  torus { 1.0, 0.25
    rotate z*90
    }
  material { Material01 }
  photons { target reflection on refraction on collect off }
  translate  <0.0, -4.0, -0.5>
}

box { <-1, -1, -1>, <1, 1, 1>
  material { Material01 }
  photons { target reflection on refraction on collect off }

  scale <3.0, 0.5, 0.5>
  translate  -1.75*z
  rotate x*45
  translate  -1.5*y
}

sphere { <0,0,0>,1
  material { Material01 }
  photons { target reflection on refraction on collect off }
  translate <3, 3, -1>
}
sphere { <0,0,0>,1
  material { Material01 }
  photons { target reflection on refraction on collect off }

  translate  <0,3.0, -0.5>
}
sphere { <0,0,0>,1
  material { Material01 }
  photons { target reflection on refraction on collect off }
  translate  <-3.0, 3.0, -1>
}
cone { 0, 1, -2*z, 0
  material { Material01 }
  photons { target reflection on refraction on collect off }
  translate  <-4.0, 0.3, 0>
}
cone { 0, 1, -2*z, 0
  material { Material01 }
  photons { target reflection on refraction on collect off }
  translate  <4.0, 0.3, 0>
}

plane
{ z, -2
  hollow on
  pigment { C_Gray (0.36) }
  finish { specular 0 }
}

//%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
#else
  //Make sure to comment out ALL #declares of FName in SpectralComposer.pov!
  #declare FName = "refract-spectral"
  #include "SpectralComposer.pov"
#end
