/*
	Geodesic.inc
	2013 Samuel Benge
	Distribute freely.
	
	This is a utility for creating geodesic spheres and domes. It aims to be both easy to use, and extensible.
	
	As usual: for maximum speed, DO NOT include this file! Simply copy its contents to your main scene file; you will save a lot of parsing time by doing so,
	especially for more complex models.
	
	The Macros
	
		Geodesic_Sphere(Geo_Type, Geo_Freq)
			Geo_Type: the type of base poyhedron to use (0 = tetrahedron; 1 = octahedron; 2 = icosahedron)
			Geo_Freq: the frequency, or complexity, of the geodesic object (0 to ?; 0 = base polyhedron)
			
			The Geodesic_Sphere() macro is usually called within a union{} block, but may also be called within a mesh{} block if the Geodesic_Triangle() object
			contains only triangles. Other possibilities may exist.
		
		Geodesic_Triangle(P1, P2, P3)
			P1: triangle point #1
			P2: triangle point #2
			P3: triangle point #3
		
			The Geodesic_Triangle() macro is something you design yourself, to create custom triangle objects for your geodesic sphere/dome. It is called internally
			within the Geodesic_Sphere() macro. A default version of this macro, which only creates triangles, is included.
			
			P1, P2, P3 are the three points that you will use whenever contructing your own triangle objects. These points are normalized: they lay on a unit sphere
			centered at <0, 0, 0>.
			
			Other variables available to this macro:
				Geo_T: the number of the current polyhedron triangle that is being subdivided.
				P1_, P2_, P3_: non-normalized versions of P1, P2 & P3. These map to the base polyhedron's flat faces, and not to a sphere.
	
	Created Data
	
		Geodesic_PolyGrad
			
			Geodesic_PolyGrad is a function that is created whenever you call the Geodesic_Sphere() macro. It is a gradient that starts at 0 (black) at each of the
			base-polyhedron's triangle edges, gradating to 1 (white) at the center of each of the base-polyhedron's triangles. It can be mapped to both the base
			polyhedron, or to a sphere.
			
			It is also destroyed whenever you call the Geodesic_Sphere() macro, and subsequently recreated to match the current geodesic type.
	
*/

// default trianbgle object
#macro Geodesic_Triangle(P1, P2, P3)
	triangle{P1, P2, P3}
#end

// the geodesic sphere macro
#macro Geodesic_Sphere(Geo_Type, Geo_Freq)

	#macro Reflect_Plane(V_Plane)
		transform{ Point_At_Trans(V_Plane) inverse }
		warp{repeat y*1e7 flip y}
		Point_At_Trans(V_Plane)
	#end

	#ifdef(Geodesic_PolyGrad) #undef Geodesic_PolyGrad #end

	// base polyhedron to use
	#switch(min(2, max(0, Geo_Type)))
	
		#case(0) // tetrahedron
		
			#local Geodesic_Tris =
		  	array[4]{
		  		array[3]{vnormalize(<-1, 1, -1>), vnormalize(<1, 1, 1>), vnormalize(<1, -1, -1>)}
		  		array[3]{vnormalize(<-1, 1, -1>), vnormalize(<1, 1, 1>), vnormalize(<-1, -1, 1>)}
		  		array[3]{vnormalize(<-1, 1, -1>), vnormalize(<-1, -1, 1>), vnormalize(<1, -1, -1>)}
		  		array[3]{vnormalize(<1, 1, 1>), vnormalize(<-1, -1, 1>), vnormalize(<1, -1, -1>)}
		  	}
		  
		  #local Geo_Mid = vrotate((<-1, 1, -1> + <1, 1, 1> + <1, -1, -1>)/3, -y*45);
		  #local Geo_Ang_ = atan2(Geo_Mid.x, Geo_Mid.y);
		  
		  // corrective rotation
		  #local Geo_P1 = vnormalize(vrotate(1, -y*45));
		  #local Geo_Ang = atan2(Geo_P1.z, Geo_P1.y);
		  #for(Geo_V, 0, dimension_size(Geodesic_Tris, 1)-1)
				#local Geodesic_Tris[Geo_V][0] = vrotate(vrotate(Geodesic_Tris[Geo_V][0], -y*45), <degrees(Geo_Ang), 180, 0>);
	  		#local Geodesic_Tris[Geo_V][1] = vrotate(vrotate(Geodesic_Tris[Geo_V][1], -y*45), <degrees(Geo_Ang), 180, 0>);
	  		#local Geodesic_Tris[Geo_V][2] = vrotate(vrotate(Geodesic_Tris[Geo_V][2], -y*45), <degrees(Geo_Ang), 180, 0>);
	  	#end
	  	
	  	#declare Geodesic_PolyGrad =
		  	function{
			  	pattern{
			  		function{abs(atan2(x, y)/Geo_Ang_/1.000001)}
			  		rotate y*45
			  		Reflect_Plane(<-1, 1, -1>-<1, -1, -1>)
			  		Reflect_Plane(1-<-1, 1, -1>)
			  		Reflect_Plane(<1, -1, -1>-<-1, -1, 1>)
			  		Reflect_Plane(1-<1, -1, -1>)
			  		rotate -y*45
			  		rotate <-degrees(Geo_Ang_), 0, 0>
			  	}
		  	}
		  
		#break // tetrahedron
		
		#case(1) // octahedron
		
			#local Geodesic_Tris =
		  	array[8]{
		  		array[3]{y ,x ,z}
		  		array[3]{y ,-x ,z}
		  		array[3]{-y ,x ,z}
		  		array[3]{-y ,-x ,z}
		  		array[3]{y ,x ,-z}
		  		array[3]{y ,-x ,-z}
		  		array[3]{-y ,x ,-z}
		  		array[3]{-y ,-x ,-z}
		  	}
		  
		  #local Geo_Mid = vrotate((x+y+z)/3, y*45);
		  #local Geo_Ang = atan2(Geo_Mid.y, Geo_Mid.x);
		  
		  #declare Geodesic_PolyGrad =
		  	function{
			  	pattern{
			  		function{abs(atan2(y, -z)/Geo_Ang/1.000001)}
			  		rotate -y*45
			  		Reflect_Plane(-z-y)
			  		Reflect_Plane(-z-y)
			  		Reflect_Plane(x+z)
			  		Reflect_Plane(x)
			  		Reflect_Plane(-z)
			  		Reflect_Plane(y)
			  	}
		  	}
		  
		
		#break // octahedron
		
		#case(2) // icosahedron
		  
		  #local Geo_Phi = (1+sqrt(5))/2;
			#local Geo_DPhi = 1/Geo_Phi;
		  
		  #local Geodesic_Tris =
		  	array[20]{
		  		array[3]{vnormalize(<-Geo_DPhi, 1, 0>), vnormalize(<Geo_DPhi, 1, 0>), vnormalize(<0, Geo_DPhi, -1>)}
		  		array[3]{vnormalize(<-Geo_DPhi, 1, 0>), vnormalize(<Geo_DPhi, 1, 0>), vnormalize(<0, Geo_DPhi, 1>)}
		  		array[3]{vnormalize(<-Geo_DPhi, -1, 0>), vnormalize(<Geo_DPhi, -1, 0>), vnormalize(<0, -Geo_DPhi, -1>)}
		  		array[3]{vnormalize(<-Geo_DPhi, -1, 0>), vnormalize(<Geo_DPhi, -1, 0>), vnormalize(<0, -Geo_DPhi, 1>)}
		  		array[3]{vnormalize(<Geo_DPhi, 1, 0>), vnormalize(<1, 0, Geo_DPhi>), vnormalize(<1, 0, -Geo_DPhi>)}
		  		array[3]{vnormalize(<Geo_DPhi, -1, 0>), vnormalize(<1, 0, Geo_DPhi>), vnormalize(<1, 0, -Geo_DPhi>)}
		  		array[3]{vnormalize(<-Geo_DPhi, 1, 0>), vnormalize(<-1, 0, Geo_DPhi>), vnormalize(<-1, 0, -Geo_DPhi>)}
		  		array[3]{vnormalize(<-Geo_DPhi, -1, 0>), vnormalize(<-1, 0, Geo_DPhi>), vnormalize(<-1, 0, -Geo_DPhi>)}
		  		array[3]{vnormalize(<0, Geo_DPhi, -1>), vnormalize(<1, 0, -Geo_DPhi>), vnormalize(<0, -Geo_DPhi, -1>)}
		  		array[3]{vnormalize(<0, Geo_DPhi, -1>), vnormalize(<-1, 0, -Geo_DPhi>), vnormalize(<0, -Geo_DPhi, -1>)}
		  		array[3]{vnormalize(<0, Geo_DPhi, 1>), vnormalize(<1, 0, Geo_DPhi>), vnormalize(<0, -Geo_DPhi, 1>)}
		  		array[3]{vnormalize(<0, Geo_DPhi, 1>), vnormalize(<-1, 0, Geo_DPhi>), vnormalize(<0, -Geo_DPhi, 1>)}
		  		array[3]{vnormalize(<Geo_DPhi, 1, 0>), vnormalize(<1, 0, -Geo_DPhi>), vnormalize(<0, Geo_DPhi, -1>)}
		  		array[3]{vnormalize(<-Geo_DPhi, 1, 0>), vnormalize(<-1, 0, -Geo_DPhi>), vnormalize(<0, Geo_DPhi, -1>)}
		  		array[3]{vnormalize(<Geo_DPhi, -1, 0>), vnormalize(<1, 0, -Geo_DPhi>), vnormalize(<0, -Geo_DPhi, -1>)}
		  		array[3]{vnormalize(<-Geo_DPhi, -1, 0>), vnormalize(<-1, 0, -Geo_DPhi>), vnormalize(<0, -Geo_DPhi, -1>)}
		  		array[3]{vnormalize(<Geo_DPhi, 1, 0>), vnormalize(<1, 0, Geo_DPhi>), vnormalize(<0, Geo_DPhi, 1>)}
		  		array[3]{vnormalize(<-Geo_DPhi, 1, 0>), vnormalize(<-1, 0, Geo_DPhi>), vnormalize(<0, Geo_DPhi, 1>)}
		  		array[3]{vnormalize(<Geo_DPhi, -1, 0>), vnormalize(<1, 0, Geo_DPhi>), vnormalize(<0, -Geo_DPhi, 1>)}
		  		array[3]{vnormalize(<-Geo_DPhi, -1, 0>), vnormalize(<-1, 0, Geo_DPhi>), vnormalize(<0, -Geo_DPhi, 1>)}
		  	}
		  
		  #local Geo_Mid = (vnormalize(<-Geo_DPhi, 1, 0>) + vnormalize(<Geo_DPhi, 1, 0>) + vnormalize(<0, Geo_DPhi, -1>))/3;
		  #local Geo_Ang_ = atan2(Geo_Mid.z, Geo_Mid.y);
		  
		  // corrective rotation
	  	#local Geo_P1 = vnormalize(<0, Geo_DPhi, -1>);
	  	#local Geo_Ang = atan2(1, Geo_DPhi);
	  	#for(Geo_V, 0, dimension_size(Geodesic_Tris, 1)-1)
	  		#local Geodesic_Tris[Geo_V][0] = vrotate(Geodesic_Tris[Geo_V][0], x*degrees(Geo_Ang));
	  		#local Geodesic_Tris[Geo_V][1] = vrotate(Geodesic_Tris[Geo_V][1], x*degrees(Geo_Ang));
	  		#local Geodesic_Tris[Geo_V][2] = vrotate(Geodesic_Tris[Geo_V][2], x*degrees(Geo_Ang));
	  	#end
	  	
	  	#declare Geodesic_PolyGrad =
		  	function{
			  	pattern{
			  		function{abs(atan2(z, y)/Geo_Ang_/1.000001)}
			  		Reflect_Plane(<-Geo_DPhi, 1, 0> - <0, Geo_DPhi, -1>)
			  		Reflect_Plane(x)
			  		Reflect_Plane(-z)
			  		Reflect_Plane(<-Geo_DPhi, 1, 0> - <1, 0, -Geo_DPhi>)
			  		Reflect_Plane(<1, 0, -Geo_DPhi> - <0, Geo_DPhi, 1>)
			  		Reflect_Plane(<Geo_DPhi, 1, 0> - <-1, 0, -Geo_DPhi>)
			  		Reflect_Plane(y)
			  		Reflect_Plane(-z)
			  		rotate x*degrees(Geo_Ang)
					}
				}
		  	
		#break // icosahedron
	
	#end // switch Geo_Type
	
	// subdivide/project/draw triangles
	#local Geo_Freq_ = Geo_Freq + 1;
	#local Geo_Freq_ = max(Geo_Freq_, 1);
	#for(Geo_T, 0, dimension_size(Geodesic_Tris, 1)-1)
		#local Geo_P1 = Geodesic_Tris[Geo_T][0];
		#local Geo_P2 = Geodesic_Tris[Geo_T][1];
		#local Geo_P3 = Geodesic_Tris[Geo_T][2];
		#for(Geo_V, 1, Geo_Freq_)
			#local Geo_PA = Geodesic_Lerp(Geo_P1, Geo_P2, Geo_V/Geo_Freq_);
			#local Geo_PB = Geodesic_Lerp(Geo_P1, Geo_P3, Geo_V/Geo_Freq_);
			#local Geo_PA2 = Geodesic_Lerp(Geo_P1, Geo_P2, (Geo_V-1)/Geo_Freq_);
			#local Geo_PB2 = Geodesic_Lerp(Geo_P1, Geo_P3, (Geo_V-1)/Geo_Freq_);
			#if(Geo_V<Geo_Freq_)
				#local Geo_PA3 = Geodesic_Lerp(Geo_P1, Geo_P2, (Geo_V+1)/Geo_Freq_);
				#local Geo_PB3 = Geodesic_Lerp(Geo_P1, Geo_P3, (Geo_V+1)/Geo_Freq_);
			#end
			#for(Geo_U, 0, Geo_V)
				#local Geo_PC = Geodesic_Lerp(Geo_PA, Geo_PB, Geo_U/Geo_V); //
				#if(Geo_U>0)
					// triangle corners
					#if(Geo_V=1)
						#local Geo_TP1 = Geo_P1;
					#else
						#local Geo_TP1 = Geodesic_Lerp(Geo_PA2, Geo_PB2, (Geo_U-1)/(Geo_V-1)); //
					#end
					#local Geo_TP2 = Geodesic_Lerp(Geo_PA, Geo_PB, (Geo_U-1)/Geo_V); //
					#local Geo_TP3 = Geo_PC;
					
					#local P1_ = Geo_TP1;
					#local P2_ = Geo_TP2;
					#local P3_ = Geo_TP3;
					
					// "even" triangles
					Geodesic_Triangle(vnormalize(Geo_TP1), vnormalize(Geo_TP2), vnormalize(Geo_TP3))
					// "odd" triangles
					#if(Geo_V<Geo_Freq_)
						#local TP3b = Geodesic_Lerp(Geo_PA3, Geo_PB3, (Geo_U)/(Geo_V+1)); //
						#local P1_ = TP3b;
						Geodesic_Triangle(vnormalize(TP3b), vnormalize(Geo_TP2), vnormalize(Geo_TP3))
					#end
				#end
			#end // #for Geo_U
		#end // #for Geo_V
	#end // #for Geo_T

#end // Geodesic_Sphere

// linear interpolation macro
#macro Geodesic_Lerp(A, B, C) A*(1-C) + B*C #end

// copied from transforms.inc
#macro Point_At_Trans(YAxis)
   #local Y = vnormalize(YAxis);
   #local X = VPerp_To_Vector(Y);
   #local Z = vcross(X, Y);
   Shear_Trans(X, Y, Z)
#end
#macro Shear_Trans(A, B, C)
   transform {
      matrix < A.x, A.y, A.z,
             B.x, B.y, B.z,
             C.x, C.y, C.z,
             0, 0, 0>
   }
#end
#macro VPerp_To_Vector(v0)
   #if (vlength(v0) = 0)
      #local vN = <0, 0, 0>;
   #else
      #local Dm = min(abs(v0.x), abs(v0.y), abs(v0.z));
      #if (abs(v0.z) = Dm)
         #local vN = vnormalize(vcross(v0, z));
      #else
         #if (abs(v0.y) = Dm)
            #local vN = vnormalize(vcross(v0, y));
         #else
            #local vN = vnormalize(vcross(v0, x));
         #end
      #end
   #end
   vN
#end