// smooth_csg.inc - by Tekno Frannansa
// Based on some half-remembered maths someone posted long-ago on the povray newsgroups, but I forget who...

/*
This file defines macros to imitate POV's primitives & CSG operations

Basic usage example:

SmoothCSG(
	mf_Difference2(
		mf_Merge2(
			mf_Box(-1,1,.02),
			mf_Cylinder(-z*1.2,z*1.2,.8,.1),
			.1
		),
		mf_Cylinder(-z*2, z*2, .3,.1),
		.02
	),
	-1.25, 1.25
)
*/


#version 3.7;

#ifndef (__SMOOTH_CSG_INC__)
#declare __SMOOTH_CSG_INC__=1;


// GLOBALS
// define SCSG_PREVIEW to turn off all smoothing. This still uses isosurfaces so it's not super-fast, but it is useful for debugging.



// CSG OPERATIONS

// Intersect an array of objects.
// This would work really well with an in-line array declaration: mf_Intersection(array[5]{mf_Cylinder(...),mf_Sphere(...),...etc.
// but unfortunately pov doesn't like invoking macros in array initialiser lists, so we need to go via the mf_Intersection2,3,4, etc. macros.
#macro mf_Intersection(a,Smoothing)
	#local n=dimension_size(a,1);
	#if ( Smoothing = 0 | defined(SCSG_PREVIEW) )
		// hard edges, no smoothing
		function {
			#local i=1;#while(i<n)
				max(
			#local i=i+1;#end
			a[0](x,y,z)
			#local i=1;#while(i<n)
				,a[i](x,y,z))
			#local i=i+1;#end
		}
	#else
		// smoothing
		function {
			ln(
				exp(a[0](x,y,z)/Smoothing)
				#local i=1;#while(i<n)
					+exp(a[i](x,y,z)/Smoothing)
				#local i=i+1;#end
			)*Smoothing
		}
	#end
#end

#macro mf_Merge(a,Smoothing)
	#local n=dimension_size(a,1);
	#if ( Smoothing = 0 | defined(SCSG_PREVIEW) )
		// hard edges, no smoothing
		function {
			#local i=1;#while(i<n)
				min(
			#local i=i+1;#end
			a[0](x,y,z)
			#local i=1;#while(i<n)
				,a[i](x,y,z))
			#local i=i+1;#end
		}
	#else
		function {
			-ln(
				exp(-a[0](x,y,z)/Smoothing)
				#local i=1;#while(i<n)
					+exp(-a[i](x,y,z)/Smoothing)
				#local i=i+1;#end
			)*Smoothing
		}
	#end
#end

#macro mf_Difference(a,Smoothing)
	#local n=dimension_size(a,1);
	#if ( Smoothing = 0 | defined(SCSG_PREVIEW) )
		// hard edges, no smoothing
		function {
			#local i=1;#while(i<n)
				max(
			#local i=i+1;#end
			a[0](x,y,z)
			#local i=1;#while(i<n)
				,-a[i](x,y,z))
			#local i=i+1;#end
		}
	#else
		function {
			ln(
				exp(a[0](x,y,z)/Smoothing)
				#local i=1;#while(i<n)
					+exp(-a[i](x,y,z)/Smoothing)
				#local i=i+1;#end
			)*Smoothing
		}
	#end
#end


#macro mf_Intersection2(A,B,              Smoothing) mf_Intersection(array[2]{A,B              },Smoothing) #end
#macro mf_Intersection3(A,B,C,            Smoothing) mf_Intersection(array[3]{A,B,C            },Smoothing) #end
#macro mf_Intersection4(A,B,C,D,          Smoothing) mf_Intersection(array[4]{A,B,C,D          },Smoothing) #end
#macro mf_Intersection5(A,B,C,D,E,        Smoothing) mf_Intersection(array[5]{A,B,C,D,E        },Smoothing) #end
#macro mf_Intersection6(A,B,C,D,E,F,      Smoothing) mf_Intersection(array[6]{A,B,C,D,E,F      },Smoothing) #end
#macro mf_Intersection7(A,B,C,D,E,F,G,    Smoothing) mf_Intersection(array[7]{A,B,C,D,E,F,G    },Smoothing) #end
#macro mf_Intersection8(A,B,C,D,E,F,G,H,  Smoothing) mf_Intersection(array[8]{A,B,C,D,E,F,G,H  },Smoothing) #end
#macro mf_Intersection9(A,B,C,D,E,F,G,H,I,Smoothing) mf_Intersection(array[9]{A,B,C,D,E,F,G,H,I},Smoothing) #end

#macro mf_Merge2(A,B,              Smoothing) mf_Merge(array[2]{A,B              },Smoothing) #end
#macro mf_Merge3(A,B,C,            Smoothing) mf_Merge(array[3]{A,B,C            },Smoothing) #end
#macro mf_Merge4(A,B,C,D,          Smoothing) mf_Merge(array[4]{A,B,C,D          },Smoothing) #end
#macro mf_Merge5(A,B,C,D,E,        Smoothing) mf_Merge(array[5]{A,B,C,D,E        },Smoothing) #end
#macro mf_Merge6(A,B,C,D,E,F,      Smoothing) mf_Merge(array[6]{A,B,C,D,E,F      },Smoothing) #end
#macro mf_Merge7(A,B,C,D,E,F,G,    Smoothing) mf_Merge(array[7]{A,B,C,D,E,F,G    },Smoothing) #end
#macro mf_Merge8(A,B,C,D,E,F,G,H,  Smoothing) mf_Merge(array[8]{A,B,C,D,E,F,G,H  },Smoothing) #end
#macro mf_Merge9(A,B,C,D,E,F,G,H,I,Smoothing) mf_Merge(array[9]{A,B,C,D,E,F,G,H,I},Smoothing) #end

#macro mf_Difference2(A,B,              Smoothing) mf_Difference(array[2]{A,B              },Smoothing) #end
#macro mf_Difference3(A,B,C,            Smoothing) mf_Difference(array[3]{A,B,C            },Smoothing) #end
#macro mf_Difference4(A,B,C,D,          Smoothing) mf_Difference(array[4]{A,B,C,D          },Smoothing) #end
#macro mf_Difference5(A,B,C,D,E,        Smoothing) mf_Difference(array[5]{A,B,C,D,E        },Smoothing) #end
#macro mf_Difference6(A,B,C,D,E,F,      Smoothing) mf_Difference(array[6]{A,B,C,D,E,F      },Smoothing) #end
#macro mf_Difference7(A,B,C,D,E,F,G,    Smoothing) mf_Difference(array[7]{A,B,C,D,E,F,G    },Smoothing) #end
#macro mf_Difference8(A,B,C,D,E,F,G,H,  Smoothing) mf_Difference(array[8]{A,B,C,D,E,F,G,H  },Smoothing) #end
#macro mf_Difference9(A,B,C,D,E,F,G,H,I,Smoothing) mf_Difference(array[9]{A,B,C,D,E,F,G,H,I},Smoothing) #end


// PRIMITIVES			

#declare f_Sphere = function(x,y,z, X,Y,Z, R) { sqrt(pow(x-X,2)+pow(y-Y,2)+pow(z-Z,2))-R }

#macro mf_Sphere(Centre, Radius)
	#local C = Centre*<1,1,1>;
	#local CX = C.x; #local CY=C.y; #local CZ=C.z;
	function { f_Sphere(x,y,z, CX,CY,CZ, Radius) }
#end

#declare f_Plane = function(x,y,z, X,Y,Z, D) { X*x+Y*y+Z*z-D }

#macro mf_Plane(Normal, Distance)
	#local Normal = vnormalize(Normal);
	#local NX = Normal.x; #local NY=Normal.y; #local NZ=Normal.z;
	function { f_Plane(x,y,z, NX,NY,NZ, Distance) }
#end

#declare f_Cylinder2 = function(x,y,z, DX,DY,DZ, R) { sqrt(x*x+y*y+z*z-pow(x*DX+y*DY+z*DZ,2))-R } //D must be normalized
#declare f_Cylinder = function(x,y,z, X,Y,Z, DX,DY,DZ, R) { f_Cylinder2(x-X,y-Y,z-Z, DX,DY,DZ, R) }

#macro mf_Cylinder(P0,P1,R,Smoothing)
	#local P0 = P0*<1,1,1>;
	#local P1 = P1*<1,1,1>;
	#local X=P0.x;#local Y=P0.y;#local Z=P0.z;
	#local D=vnormalize(P1-P0);
	#local DX=D.x;#local DY=D.y;#local DZ=D.z;
	mf_Intersection3( function { f_Cylinder(x,y,z, X,Y,Z, DX,DY,DZ, R) }, mf_Plane(-D,vdot(-D,P0)), mf_Plane(D,vdot(D,P1)), Smoothing)
#end

#declare f_CylinderLoci2 = function(PA,PB,BA,BPA,APB) { select( BPA, sqrt(PA), select( APB, sqrt(PB), sqrt(PA-BPA*BPA/BA) ) ) };
#declare f_CylinderLoci =
	function(x,y,z, AX,AY,AZ, BX,BY,BZ, R) {
		f_CylinderLoci2(
			(x-AX)*(x-AX)+(y-AY)*(y-AY)+(z-AZ)*(z-AZ),
			(x-BX)*(x-BX)+(y-BY)*(y-BY)+(z-BZ)*(z-BZ),
			(AX-BX)*(AX-BX)+(AY-BY)*(AY-BY)+(AZ-BZ)*(AZ-BZ),
			(BX-AX)*(x-AX)+(BY-AY)*(y-AY)+(BZ-AZ)*(z-AZ),
			(AX-BX)*(x-BX)+(AY-BY)*(y-BY)+(AZ-BZ)*(z-BZ)
		)
		-R
	};

// round-capped cylinder, based on measuring true distance to the line A,B
// maybe quicker to just do 2 planes selecting between spheres and infinite cylinder
#macro mf_CylinderLoci(A,B,R)
	#local A = A*<1,1,1>;
	#local B = B*<1,1,1>;
	#local AX = A.x; #local AY=A.y; #local AZ=A.z;
	#local BX = B.x; #local BY=B.y; #local BZ=B.z;
	function { f_CylinderLoci(x,y,z, AX,AY,AZ, BX,BY,BZ, R) }
#end


// the fastest type of cylinder for CSG
#macro mf_CylinderInfinite(P0,P1,R)
	#local P0 = P0*<1,1,1>;
	#local P1 = P1*<1,1,1>;
	#local X=P0.x;#local Y=P0.y;#local Z=P0.z;
	#local D=vnormalize(P1-P0);
	#local DX=D.x;#local DY=D.y;#local DZ=D.z;
	function { f_Cylinder(x,y,z, X,Y,Z, DX,DY,DZ, R) }
#end


#macro mf_Cone(P0,R0,P1,R1,Smoothing)
	#local P0 = P0*<1,1,1>;
	#local P1 = P1*<1,1,1>;
	#local X=P0.x;#local Y=P0.y;#local Z=P0.z;
	#local PD = P1-P0;
	#local D=vnormalize(PD);
	#local DX=D.x;#local DY=D.y;#local DZ=D.z;

	// compute the diagonal in UV space (R,H space).
	#local H = vlength(PD);
	#local DR = R1-R0;
	#local M = vnormalize(<H,-DR>);
	#local DDU = M.x; #local DDV = M.y;
	#local M0 = R0*M.x; // dot a point <R0,0> on the diagonal with the perp-vector of the diagonal.
	
	// diagonal edge
	#local f0 = function(u,v) { u*DDU + v*DDV - M0 }

	// smooth triangle / trapezoid
	#if ( Smoothing = 0 | defined(SCSG_PREVIEW) )
		#local f1 = function(u,v) { max( max(f0(u,v), f0(-u,v)), max(-v,v-H) ) }
	#else
		#local f1 = function(u,v) { ln( exp(f0(u,v)/Smoothing) + exp(f0(-u,v)/Smoothing) + exp(-v/Smoothing) + exp((v-H)/Smoothing) )*Smoothing }
	#end
	
	// compute u&v vectors to convert 3D space into 2D with cone axis along v.
	#local f2 = function(x,y,z,d) { f1( sqrt(x*x+y*y+z*z-d*d), d ) }
	#local f3 = function(x,y,z) { f2(x,y,z,DX*x+DY*y+DZ*z) }
	
	// precision is better if we do everything relative to one of the points
	function { f3(x-X,y-Y,z-Z) }
#end


#declare f_Torus = function( x,y,z, R0,R1 ){ sqrt(y*y+pow(sqrt(x*x+z*z)-R0,2))-R1 }

// more useful torus: about z axis, with specified centre
#macro mf_Torus2(C,R0,R1)
	#local C = C*<1,1,1>;
	#local CX = C.x; #local CY=C.y; #local CZ=C.z;
	function { f_Torus(x-CX,z-CZ,y-CY, R0,R1) }
#end


#macro mf_Box(A,B,Smoothing)
	#local A = A*<1,1,1>;
	#local B = B*<1,1,1>;
	#local M = (A+B)/2;
	#local D = VAbs((A-B)/2);
	#local MX=M.x;#local MY=M.y;#local MZ=M.z;
	#local DX=D.x;#local DY=D.y;#local DZ=D.z;

	#if ( Smoothing = 0 | defined(SCSG_PREVIEW) )
		#local f_BoxInternalIntersection = function(a,b,c,d,e,f) { max(max(max(a,b),max(c,d)),max(e,f)) };
	#else
		#local f_BoxInternalIntersection = function(a,b,c,d,e,f) { ln(exp(a/Smoothing)+exp(b/Smoothing)+exp(c/Smoothing)+exp(d/Smoothing)+exp(e/Smoothing)+exp(f/Smoothing))*Smoothing };
	#end

	function { f_BoxInternalIntersection( x-MX-DX, y-MY-DY, z-MZ-DZ, MX-x-DX, MY-y-DY, MZ-z-DZ ) }
#end


//MODIFIERS

// mf_Transform
// Note that anything other than rotate & translate will cause distortions to the smoothing.
// Smoothing relies on functions giving the world-space distance from the surface. Scaling & shearing distort that.
#macro mf_Transform( f, Trans )
	#local f_Trans = function { transform { Trans inverse } }
	function { f( f_Trans(x,y,z).x, f_Trans(x,y,z).y, f_Trans(x,y,z).z ) }
#end

#macro mf_Inverse(A) function { -A(x,y,z) } #end

#macro mf_Hollow(A) function { abs(A(x,y,z)) } #end // useful for making creases, very bad for making objects

#macro mf_Offset(A,b) function { A(x,y,z)+b } #end


// WRAPPERS
#macro SmoothCSG( f, BoundsMin, BoundsMax )
	isosurface {
		function { f(x,y,z) }
		max_gradient 2
		contained_by { box { BoundsMin, BoundsMax } }
		accuracy 1/10000 // fix some shadow artefacts
	}
#end


#end //#ifndef __SMOOTH_CSG_INC__



/* Currently unsupported object types:
	blob, height_field, julia, lathe*, ovus, prism*, sphere_sweep*, superellipsoid, sor*, text*
	bicubic_patch*, disc*, mesh*, mesh2*, polygon*, triangle*, smooth_triangle*
	poly, cubic, quartic, quadric
	parametric*
	
	(* probably impossible)
*/
