//==============================================================================
// Persistence of Vision Ray Tracer Scene File
//------------------------------------------------------------------------------
// File         : MultiPassBlobsComposite.pov
// POV Version  : 3.7
// Description  : Example of "post process" image compositing directly in 
//                POV-Ray. Render *without* antialiasing to preserve sharpness 
//                of source images.
// Date         : 03/08/2011
// Author       : Robert W. McGregor 
//                   Rob@McGregorFineArt.com
//                   www.McGregorFineArt.com
// Command-Line : +w1024 +h768 +fn16 +kff6 -a  
//------------------------------------------------------------------------------ 
// A video demonstrating the rendering (at 11x speed) and compositing (at 
// regular speed) of this scene is online at: http://vimeo.com/20987441  
//==============================================================================

#version 3.7;

#include "functions.inc"
#include "rwm_composite.inc"

global_settings { assumed_gamma 1 }
                                          
// Set this to false if you've already tweaked the contrast on the depth map
// in GIMP, Photoshop, etc., but AutoContrast_Grayscale() should be good enough
#declare Use_AutoContrast_DepthhMap = true;

//------------------------------------------------------------------------------ 
// Camera
//------------------------------------------------------------------------------ 
camera {  
   orthographic
   location <0, 0, -1>
   look_at <0, 0, 0>
   right  x*image_width/image_height
}                     

//------------------------------------------------------------------------------ 
// setup the various render passes as pigments
//------------------------------------------------------------------------------ 

#local pgmt_beauty = pigment { 
   image_map { png "MultiPassBlobs1" map_type 0 once } 
}

#local pgmt_ao = pigment { 
   image_map { png "MultiPassBlobs2" map_type 0 once } 
} 

#local pgmt_depth = pigment { 
   image_map { png "MultiPassBlobs3" 
      // must set to gamma 1 for proper eval_pigment results
      #if (Use_AutoContrast_DepthhMap) gamma 1 #end  
      map_type 0 once 
   } 
} 

#local pgmt_refl = pigment { 
   image_map { png "MultiPassBlobs4" map_type 0 once } 
} 

#local pgmt_sslt = pigment { 
   image_map { png "MultiPassBlobs5" map_type 0 once } 
} 

#local pgmt_direct = pigment { 
   image_map { png "MultiPassBlobs6" map_type 0 once } 
} 

// the composite image doesn't exist until after the first frame
#if (frame_number > 1)
   #local pgmt_comp = pigment { 
      image_map { png concat("MultiPassBlobsComposite", str(frame_number-1, 0, 0)) map_type 0 once } 
   }
#end

//------------------------------------------------------------------------------ 
// For each render pass type, build the composite up one layer at a time
//------------------------------------------------------------------------------ 

#switch (frame_number)
   #case (1) 
      // multiply ao pass onto beauty pass, giving more definition to forms
      #declare p = pigment { Pigment_Multiply(pgmt_beauty, pgmt_ao, 0.5) } 
      #break
   
   #case (2)                           
      // screen reflection pass onto beauty+ao comp
      #declare p = pigment { Pigment_Screen(pgmt_comp, pgmt_refl, 0.5) }  
      #break
   
   #case (3)
      // screen subsurface pass onto beauty+ao+refl comp
      #declare p = pigment { Pigment_Screen(pgmt_comp, pgmt_sslt, 1) }  
      #break
   
   #case (4)                                                            
      // In this case, I wanted more than 100% subsurface screened on, 
      // so do it again to brighten the SSLT effect a bit more 
      #declare p = pigment { Pigment_Screen(pgmt_comp, pgmt_sslt, 0.5) }  
      #break
   
   #case (5)   
      #if (Use_AutoContrast_DepthhMap)
         // auto-contast for the grayscale depth map image
         #declare pgmt_depth = pigment { AutoContrast_Grayscale(pgmt_depth) }  
      #end
      
      // screen depth pass onto beauty+ao+refl+sslt comp for some atmospheric depth
      #declare p = pigment { Pigment_Screen(pgmt_comp, pgmt_depth, 1) }  
      #break

   #case (6)
      // (This pass isn't really needed for this scene, but as an example here it is anyway)
      // For this shot the lighting is already nearing the blowout point, so I 
      // multiply the direct lighting pass onto the beauty+ao+refl+sslt+depth comp 
      // to mute the non-direct lighted areas just a bit, increaing overall contrast. 
      // In other shots screen mode can be used to brighten the direct lighting instead.
      #declare p = pigment { Pigment_Multiply(pgmt_comp, pgmt_direct, 0.05) }  
      #break
#end
                                                       
// project the composite image onto a plane
plane { z, 0 
   hollow
   pigment { p translate -0.5 scale x*image_width/image_height } 
   finish { ambient 3 diffuse 0 }
}
