//==============================================================================
// Persistence of Vision Ray Tracer Scene File
//------------------------------------------------------------------------------
// File         : MultiPassBlobs.pov
// POV Version  : 3.7
// Description  : Demonstrates creation of multiple render passes in POV-Ray
// Date         : 03/07/2011
// Author       : Robert W. McGregor, Rob@McGregorFineArt.com 
// Command-Line : +w1024 +h768 +fn16 +kff6  
//------------------------------------------------------------------------------ 
// A video demonstrating the rendering (at 11x speed) and compositing (at 
// regular speed) of this scene is online at: http://vimeo.com/20987441  
//------------------------------------------------------------------------------ 

#version 3.7;

#include "shapes.inc"
#include "rwm_multipass.inc"

default { 
   finish { ambient 0 specular 0 diffuse 0.8 reflection 0 } 
   pigment { rgb 0 } 
}
background { rgb 0 } 

//------------------------------------------------------------------------------ 
// Scene/Render Configuration 
//------------------------------------------------------------------------------ 

// Render pass types to determine which textures to use for the various passes
// (these are also auto-declared in rwm_multipass.inc). The Pigment_Wrapper() 
// and Finish_Wrapper() macros in rwm_multipass.inc use these to auto-redefine 
// textures for the various render passes

/*
#declare Quick_Pass             = 0;
#declare Beauty_Pass            = 1;
#declare Ambient_Occlusion_Pass = 2;
#declare Depth_Pass             = 3;
#declare Reflection_Pass        = 4;
#declare Subsurface_Pass        = 5;
#declare Direct_Lighting_Pass   = 6;
*/

// The animation frame number determines the render pass type for the current render pass
#declare Current_Pass = frame_number; 

// Scene switches        
#declare Use_Image_Maps      = 0;    // 0=procedural texture; 1=use image and bump maps for the ground texture 
#declare Use_Environment_Map = 1;    // 0=pigmented "sky dome"; 1=use EXR environment map for realistic lighting 
#declare Use_Soft_Shadows    = 1;    // 0=point light; 1=area light  
#declare Use_Focal_Blur      = 1;    // 0=no focal blur; 1=use focal blur 
#declare Blur_Samples        = 7*10; // number of blur samples to use when focal blur is on

//------------------------------------------------------------------------------ 
// global_settings 
//------------------------------------------------------------------------------ 

global_settings { 
   assumed_gamma 1
   mm_per_unit 3 
   subsurface { samples 64, 32 }
   
   // Use radiosity for only beauty and AO passes 
   #if ((Current_Pass=Beauty_Pass)|(Current_Pass=Ambient_Occlusion_Pass))
      radiosity { 
         #if (Current_Pass=Beauty_Pass) 
            // Beauty_Pass
            brightness      1  
            pretrace_start  64/image_width
            pretrace_end    8/image_width
            recursion_limit 3
            count           200
            error_bound     0.15         
         #else   
            // Ambient_Occlusion_Pass
            brightness      1 
            recursion_limit 1  // keep at 1 bounce for deeper AO shadows
            pretrace_start  8/image_width
            pretrace_end    4/image_width
            count           1000
            error_bound     0.06         
         #end 
         nearest_count 10
         normal off 
         always_sample off
      }
   #end 
}

//------------------------------------------------------------------------------ 
// Camera
//------------------------------------------------------------------------------ 

#declare Cam_Loc = <0, 32, -80>;
#declare Cam_At  = <0, -1, 0>;

camera {
   location Cam_Loc                                
   look_at  Cam_At
   right x*image_width/image_height 
   angle 50 
   #if (Use_Focal_Blur)
      aperture 2        
      blur_samples Blur_Samples
      focal_point <10, 0, -2>
      confidence 0.9
      variance 1/200
   #end
}                     

//------------------------------------------------------------------------------ 
// Parallel area light
//------------------------------------------------------------------------------ 

// no lights for AO and depth passes
#if ((Current_Pass!=Ambient_Occlusion_Pass)&(Current_Pass!=Depth_Pass))
   #declare Light_Pos = <2000, 500, 1000>;
   light_source { Light_Pos rgb 8
      parallel
      point_at <-20, -3.56, 20>
      fade_power 3 
      fade_distance VDist(<0,0,0>, Light_Pos)/2
      
      // No area light on the subsurface pass, as the render time gets MUCH longer
      // with no appreciable benefit
      #if ((Use_Soft_Shadows)&(Current_Pass!=Subsurface_Pass)) 
         area_light <150, 0, 0>, <0, 0, 150>, 6, 6
         adaptive 2
         jitter
         orient circular
      #end
   }
   #if (Current_Pass=Beauty_Pass)
      // some fill lighting for the beauty pass
      light_source { <0, -15.5, -50> rgb 0.3 shadowless } 
      light_source { <-100, 0, -50> rgb 0.3 shadowless } 
   #end
#end

//------------------------------------------------------------------------------ 
// Environment / Sky
//------------------------------------------------------------------------------ 

// AO pass uses a white "sky dome"
#if (Current_Pass=Ambient_Occlusion_Pass)
   plane { z, 1000 hollow no_shadow 
      pigment {rgb 1} 
      finish { emission 0.025 ambient 0 diffuse 0 } 
   }
   background { rgb 0.015 }

#else
   // other passes use either EXR or pigmented "sky dome"
   sphere { 0, 100000 hollow inverse
      #if (Use_Environment_Map)
         pigment { image_map { exr "Landscape2_Small_Blurred" interpolate 2 map_type 1 }}
         rotate y*140
      #else
         pigment {
            function { min(1, max(0, y)) }
            poly_wave 0.6
            color_map {
               [0 srgb <110,150,180>/255]
               [1 srgb <175,175,175>/255]
            }
            scale 0.125
         }
      #end
      finish { ambient 1.25 emission 1.25  diffuse 0}
   }
#end

// Depth_Pass is just fog
#if (Current_Pass=Depth_Pass) 
   fog {
     fog_type  1
     // Adjust the distance until you have a pretty good dark to light 
     // contrast. The goal is to have a black to white contrast, so this  
     // pass can be auto-contrasted in GIMP or Photoshop. Or just use the 
     // default DepthMap_AutoContrast() macro from rwm_composite.inc in 
     // your compositing scene file (it does a pretty good job with no user 
     // intervention).
     distance  2000   
     color rgb 1
   }
#end

//------------------------------------------------------------------------------ 
// the old overall brightness/contrast manipulation trick

#switch (Current_Pass)
   #case (Ambient_Occlusion_Pass) 
      sphere{Cam_Loc, 1 hollow pigment{ rgb 1 transmit 4 } }
      #break
   
   #case (Depth_Pass)     
      sphere{Cam_Loc, 1 hollow pigment{ rgb 1 transmit 1 } }
      #break

   #else
      sphere{Cam_Loc, 1 hollow pigment{ rgb 1 transmit 1.25 } } 
#end  

//------------------------------------------------------------------------------ 
// Floor
//------------------------------------------------------------------------------ 

#declare P1 = pigment { 
   average
   pigment_map {
      [1 
         #if (Use_Image_Maps) 
            // not interpolating pigment/bump images makes them more crisp in an 
            // antialiased/focal blur shot, otherwise they're antialiased twice
            image_map { png "rock22_tile" interpolate 0 } 
         #else 
            agate
            color_map { 
            [0.00 srgb<150, 145, 140>/255]
            [0.02 srgb<200, 200, 200>/255]
            [0.98 srgb<179, 178, 177>/255]
            [1.00 srgb<220, 220, 220>/255]
            }
            turbulence 0.1
            scale 15
         #end
      ]
      [2 
         bozo 
         pigment_map {
            [1
               granite turbulence 0.1 scale 5 
               color_map { [0 rgb 0.35][1 rgb 0.8] } 
            ]
            [1 
               cells turbulence 0.5 scale 3
               color_map { [0 srgb<169, 168, 167>/255][1 srgb<135, 134, 133>/255] }
            ]
         }
         turbulence 0.2 
         scale 5
      ]
   }
   translate -0.5 
}

#declare N1 =  normal {
   #if (Use_Image_Maps) 
      bump_map { png "rock22_tile_bump" interpolate 0 bump_size -0.5 } 
      translate -0.5 
   #else
      agate 
      normal_map {
         [0 agate 0.1]   
         [1 bozo 0.05 turbulence 0.5]
      } 
      turbulence 0.1
      scale 15
   #end
}

#declare T_Floor = texture { 
   #if (Use_Image_Maps)  
      // use an image pattern for as a reflection mask (for added realism). 
      // Interpolation is important for image pattern masks
      image_pattern { png "rock22_tile_bump" interpolate 2 map_type 0 }
      translate -0.5
      rotate x*90
      scale 120
      turbulence 0.1 
   #else
      // or, use a pigment pattern for as a reflection mask (for added realism)
      pigment_pattern {
         agate
         color_map { 
            [0.00 srgb<150, 145, 140>/255]
            [0.02 srgb<200, 200, 200>/255]
            [0.98 srgb<170, 170, 170>/255]
            [1.00 srgb<220, 220, 220>/255]
         }
         turbulence 0.1
         scale 15
      }
   #end
   texture_map {
      [0 
         // The Pigment_Wrapper() and Finish_Wrapper() macros auto-adapt textures for the various render passes
         Pigment_Wrapper(pigment { P1 })
         normal { N1 }
         Finish_Wrapper(finish { reflection 0 }, false )
      ]
      [1 
         Pigment_Wrapper(pigment { P1 })
         normal { N1 }
         Finish_Wrapper(finish { reflection { 0, 0.2 fresnel on } conserve_energy }, false )
      ]
   }
}

plane { y, -31.11/2 
   texture { T_Floor }
   rotate y*35
   interior { ior 1.54 } // IOR needed for Fresnel reflections
}

//------------------------------------------------------------------------------ 
// Test object
//------------------------------------------------------------------------------ 

// This is just the "blob with negative strength" from the menu
// insert -> special shapes -> blob and fractal  
#declare Blob = blob {
   threshold 0.5
   cylinder { <-7, 0, 0>, <7, 0, 0>, 4, 2 }
   cylinder { <0, 0, -7>, <0, 0, 7>, 4, 2 }
   sphere { <0, 0.5, 0>, 4.15, -5 }
   scale <2, 6, 2>
   rotate y*45
   rotate <90, -30, 0>
} 
#declare Blob = Center_Object(Blob, x+y+z)

//------------------------------------------------------------------------------ 
// Texture

// Subsurface texture with RC3-style subsurface coefficients for the finish
#declare P_SSLT_Marble = pigment { rgb <0.83, 0.79, 0.75>  }
#declare F_SSLT_Marble = finish { 
   subsurface { <2.19, 2.62, 3.00>, <0.0021, 0.0041, 0.0071> }
   diffuse 0
}  

#declare T_Thing1 = 
   texture {           
      Pigment_Wrapper(pigment { P_SSLT_Marble transmit 0 })
      Finish_Wrapper(F_SSLT_Marble, true)
   }
#declare T_Thing2 = 
   texture { 
      Pigment_Wrapper(
         pigment { 
            bozo 
            #if (Current_Pass=Subsurface_Pass)
               // For the subsurface pass, soften the filtered colors in the color map below 
               #local TRANSMIT_AMT = 0.3;  
               #local FILTER_AMT   = 1;
               #local INTENSITY    = 1.1;
            #else
               #local TRANSMIT_AMT = 0; 
               #local FILTER_AMT   = 0.4;
               #local INTENSITY    = 0.45;
            #end
            color_map { 
               [0.00 srgb <202, 185, 127>/255*INTENSITY filter FILTER_AMT transmit TRANSMIT_AMT]
               [0.25 srgb <247, 235, 179>/255*INTENSITY filter FILTER_AMT transmit TRANSMIT_AMT] 
               [0.75 srgb <160, 135, 79 >/255*INTENSITY filter FILTER_AMT transmit TRANSMIT_AMT]
               [1.00 srgb <193, 169, 112>/255*INTENSITY filter FILTER_AMT transmit TRANSMIT_AMT] 
            } 
            turbulence 0.3
            lambda 2
            octaves 3
            omega 2
            scale 3.5 
         }
      )
      normal { agate -0.1 scale 3 }
      Finish_Wrapper(finish { reflection { 0, 0.65 fresnel on } conserve_energy }, false)
   }

// Layer the textures
#declare T_Thing = 
texture { T_Thing1 }   // subsurface
texture { T_Thing2 }   // reflection

union { 
   object { Blob 
      translate <10, 0, -2>
      texture { T_Thing }
   }
   object { Blob 
      rotate y*45 
      translate <-25, 0, 50>
      texture { T_Thing }
   }
   // IOR needed for Fresnel reflections and proper subsurface evaluation
   interior { ior 1.54 } 
}
