//=============================================================================
// Persistence of Vision Ray Tracer Scene File
//-----------------------------------------------------------------------------
// File        : rwm_composite.inc
// POV Version : 3.7
// Description : Image compositing macros for multipass render post-processing
//               diectly in POV-Ray
// Date        : 10/14/2008
// Author      : Robert W. McGregor
//                  Rob@McGregorFineArt.com
//                  www.McGregorFineArt.com
//-----------------------------------------------------------------------------
// Macros: 
//    Pigment_Attenuate()
//    Pigment_Multiply()
//    Pigment_Screen()
//    AutoContrast_Grayscale() 
//    Debug_Vector() 
//=============================================================================

#ifndef(RWM_COMPOSITE_INC_TEMP)
#declare RWM_COMPOSITE_INC_TEMP = version;

#include "rwm_multipass.inc"

//-----------------------------------------------------------------------------
// Macro  : Pigment_Attenuate()  
//-----------------------------------------------------------------------------
// Params : [p1]      pigment to attenuate
//          [amt]     attenuation amount (0..1)  
//          [rgb_clr] attenuation color vector  
//-----------------------------------------------------------------------------

#macro Pigment_Attenuate(p1, amt, rgb_clr)
   
   #local p2 = pigment { rgb rgb_clr }   
      
   // return the new attenuated pigment
   #local pgmt_Attenuated = pigment {
      average
      pigment_map {
         [amt   p1]
         [1-amt p2]
      }
   }   
   pgmt_Attenuated
#end  

//-----------------------------------------------------------------------------
// Macro  : Pigment_Multiply()  
//-----------------------------------------------------------------------------
// Params : [pgmt_Orig]  Original pigment
//          [pgmt_Blend] Pigment to blend in       
//          [blendAmt]   amount of pgmt_Blend multiplied into pgmt_Orig (0..1)
//-----------------------------------------------------------------------------
// Formula: Result_Color = Top_Color * Bottom_Color / 255
//-----------------------------------------------------------------------------

#macro Pigment_Multiply(pgmt_Orig, pgmt_Blend, blendAmt)
   
   #local pgmt_new = pigment { 
      Pigment_Attenuate(pgmt_Blend, blendAmt, 1) 
   }
   
   #local fp1 = function { pigment { pgmt_Orig } }
   #local fp2 = function { pigment { pgmt_new } }
      
   // multiply pigments
   #local RedChannel = pigment {
      function { fp1(x,y,z).red * fp2(x,y,z).red }
      color_map { [0 rgb 0][1 rgb <1,0,0>] }
   }
   
   #local GreenChannel = pigment {
      function { fp1(x,y,z).green * fp2(x,y,z).green }
      color_map { [0 rgb 0][1 rgb <0,1,0>] }
   }
   
   #local BlueChannel = pigment {
      function { fp1(x,y,z).blue * fp2(x,y,z).blue }
      color_map { [0 rgb 0][1 rgb <0,0,1>] }
   }

   // return this new blended pigment
   #local pgmt_Blended = pigment { 
      average
      pigment_map {
         [1 RedChannel]
         [1 GreenChannel]
         [1 BlueChannel]
      }
   }   
   
   pgmt_Blended
#end

//-----------------------------------------------------------------------------
// Macro  : Pigment_Screen()  
//-----------------------------------------------------------------------------
// Params : [pgmt_Orig]  Original pigment
//          [pgmt_Blend] Pigment to blend in       
//          [blendAmt]   amount of pgmt_Blend screened onto pgmt_Orig (0..1)
//-----------------------------------------------------------------------------
// Formula: Result_Color = 255-[((255-Top_Color)*(255-Bottom_Color))/255]
//-----------------------------------------------------------------------------
// Screen blend mode is like the opposite of multiply. The result is a 
// brighter picture. The values of the pixels in the two layers are inverted, 
// multiplied, and then inverted again. 
//-----------------------------------------------------------------------------

#macro Pigment_Screen(pgmt_Orig, pgmt_Blend, blendAmt)
   
   #local pgmt_new = pigment { 
      Pigment_Attenuate(pgmt_Blend, blendAmt, 0) 
   }
   
   #local fp1 = function { pigment { pgmt_Orig } }
   #local fp2 = function { pigment { pgmt_new } }
      
   // screen blend pigments 
   #local RedChannel = pigment {
      function { 1-(((1-fp1(x,y,z).red)*(1-fp2(x,y,z).red))) }
      color_map { [0 rgb 0][1 rgb <1,0,0>] }
   }
   
   #local GreenChannel = pigment {
      function { 1-(((1-fp1(x,y,z).green)*(1-fp2(x,y,z).green))) }
      color_map { [0 rgb 0][1 rgb <0,1,0>] }
   }
   
   #local BlueChannel = pigment {
      function { 1-(((1-fp1(x,y,z).blue)*(1-fp2(x,y,z).blue))) }
      color_map { [0 rgb 0][1 rgb <0,0,1>] }
   }

   // "return" this new blended pigment
   #local pgmt_Blend = pigment { 
      average
      pigment_map {
         [1 RedChannel]
         [1 GreenChannel]
         [1 BlueChannel]
      }
   }   
   
   pgmt_Blend
#end

//-----------------------------------------------------------------------------
// Macro : AutoContrast_Grayscale()  
//-----------------------------------------------------------------------------
// Params: 
//    [pgmt_Depth] grayscale source pigment (image_map with gamma 1 flag set)
//-----------------------------------------------------------------------------
#macro AutoContrast_Grayscale(pgmt_grayscale)

   #local fn = function { pigment { pgmt_grayscale } }
   
   #local sW = 1 / image_width;   // width step per pixel
   #local sH = 1 / image_height;  // height step per pixel
   
   // Find the brightest and darkest pixels in the pigment
   #local imgMin  = <1000, 1000, 1000>; 
   #local imgMax  = <-1000, -1000, -1000>; 
   #local nPixels = image_width*image_height;
   #local px      = 0;
   
   #debug "Calculating AutoContrast_Grayscale pigment...\n"
  
   #local yy = 0;
   #while (yy < 1)
      #local xx = 0;
      #while (xx < 1)
         #local xPixMid = xx + sW / 2; // x center of current pixel
         #local yPixMid = yy + sH / 2; // y center of current pixel
         
         #local clr = fn(xPixMid, yPixMid, 0); // color vector of current pixel      
         #if (clr.gray < imgMin.gray) #local imgMin = clr; #end
         #if (clr.gray > imgMax.gray) #local imgMax = clr; #end
         #local xx = xx + sW;
         
         // progress feedback
         #local px = px + 1;
         #if (mod(px, int(nPixels/10))=0) 
            #debug concat(str(px/nPixels*100,0,0)"%\n")
         #end
      #end
      #local yy = yy + sH;
   #end
   
   Debug_Vector("imgMin Grayscale", imgMin)
   Debug_Vector("imgMax Grayscale", imgMax)
   
   #local fMin = imgMin.gray;
   #local fMax = 1/imgMax.gray;
   
   // Interpolate to stretch the mix/max contrast range from 0-1;
   // the pow(2.2) is a bit of a hack, but gives results quite close to 
   // Photoshop's auto-contrast (good enough for a depth map)
   #local P_AutoContrast = pigment { 
      function { pow((fn(x,y,z).gray-fMin)*fMax, 2.2) } 
   }
   P_AutoContrast
#end

//-----------------------------------------------------------------------------
// Macro : Debug_Vector()  
//-----------------------------------------------------------------------------
#macro Debug_Vector(name, vec)
   #debug concat(
      "##### ", name, " = <",  
      str(vec.x,1,2), ", ",  str(vec.y,1,2), ", ",  str(vec.z,1,2), ">\n "
   ) 
#end

#end  // #ifndef