/*
 *  Proximity Text.pov
 *
 *    Test code for the Proximity Macros
 */


// -- Include Files -----------------------------------------------------------

#include "DF3ProximityPattern.inc"



// -- Scene Settings ----------------------------------------------------------

/* DF3 Proximity Options */
#declare use_df3_proximity_pattern = true;
#declare generate_df3 = true;  // Generate pattern (turn off after 1st render)
#declare interpolation_mode = 2;  // 0, 1 or 2. Use 0 for debugging only
#declare mesh_mode = false;  // Set to true for use with mesh objects (slower)

/* Material */
#declare add_noise = true;  // Adds random noise to the proximity pattern
#declare bronze = true;  // true for bronze, false for "false colour" effect

/* Lighting */
#declare use_area_light = false;  // area_light or point light
#declare use_radiosity = true;  // overrides use_area_light
#declare radiosity_quality = 2;  // use values from 1 to 5

/* Focal Blur */
#declare use_focal_blur = false;  // On or off



// -- The object to apply the DF3 proximity pattern to ------------------------

#declare test_object = 
	julia_fractal {
		<0.1,0.8,-0.1,0.01>
		quaternion 
		sqr
		max_iteration 8
		precision 1000
		scale 60 translate y*60 rotate y*45
	}

// #declare test_object = sphere { 0, 1 scale 60 translate y*60 }



// -- Proximity Pattern -------------------------------------------------------

/* Parameters */
#declare resolution = <30,30,30>;
#declare num_samples = 100;
#declare sample_radius_fraction = 2;  // Size in "voxels" of sample radius

/* Only need to generate the DF3 Pattern the very first time */
#if( generate_df3 = true )
	Proximity_SetMeshMode( mesh_mode )
		// Proximity_SetMeshMode( true ) works with any mesh, but runs slower.

	Proximity_SetBailoutValue( 30 )
		// Make this bigger (closer to "num_samples") if you get spots.
		// Make this value lower for slightly faster parse times ("wronger
        // answers faster").
		// The bailout can be turned off with Proximity_SetBailouOff() (and
        // is also off by default if Proximity_SetBailoutValue() is never
        // called).
		
	Proximity_GenerateMap(test_object, resolution, num_samples, sample_radius_fraction, "test.df3")
#end

/* Load the DF3 Proximity Pattern */
Proximity_SetInterpolation( interpolation_mode )
#declare df3_pattern = Proximity_LoadMap(test_object, resolution, "test.df3")



// -- Global Settings ---------------------------------------------------------

global_settings {
    max_trace_level 2
}



// -- Lighting ----------------------------------------------------------------

#if( use_radiosity )
	#include "rad_def.inc"
	#default {
	   finish {
	      ambient 0
	   }
	}
	global_settings {
	       radiosity {
			pretrace_start 0.08
			pretrace_end 0.005
			count 20 * radiosity_quality
			nearest_count 5
			error_bound 1
			recursion_limit 2
			low_error_factor 0.5
			gray_threshold 0.0
			minimum_reuse 0.015
			brightness 1
			adc_bailout 0.01/2
	        }
	}
#else
	light_source {
		< 0, 1000, 200>
		rgb 1
	#if ( use_area_light = true )
		area_light < 2000,0,0 >,< 0,2000,0 >, 5,5
		adaptive 3
		jitter
	#end
	}
#end

#macro LightBox( Width, Height, Distance, Latitude, Longitude, Rotation, Colour )
	light_group {
		box {
			<-Width/2, -Height/2, 0>, < Width/2, Height/2, 0.01>
			translate -z*Distance
			rotate x* Longitude
			rotate y*Latitude
			material { 
				texture { 
					pigment { colour rgb 1 } 
					finish { ambient 1 }
				}
			}
		}

		#local dd = cos( Latitude ) * (Distance/2);
		#local xx = sin( Longitude ) * dd;
		#local yy = sin( Latitude ) * (Distance/2);
		#local zz = cos( Latitude ) * dd;

		light_source {
			< -xx, -yy, -zz >
			rgb Colour
		}		

	}
#end

LightBox( 6000, 4000, 2000, 180, 70, 0, rgb<1,1,1>*3 )
LightBox( 1000, 1000, 1200, -90, 10, 0, rgb<0,0,1>*1 )
LightBox( 1500, 600, 1000, 30, 10, 0, rgb<1,1,0>*1 )



// -- Materials ---------------------------------------------------------------

#declare shiny_brass =
	texture {
		pigment { rgb <0.8,0.5,0>*0.9 }
		finish { brilliance 2 reflection 0.33 metallic conserve_energy }
	};

#declare dull_brass =
	texture {
		pigment {
			bumps scale 5
			color_map {
				[ 0, rgb <0.8,0.5,0> * 0.6 ]
				[ 1, rgb <0.8,0.5,0> * 0.5 ]
			}
		}
		finish { reflection 0.01 metallic conserve_energy }
	};

#declare corroded_brass =
	texture {
		pigment { rgb <0.49,0.81,0.77> }
		normal { bumps scale 0.5 }
		finish { reflection 0 }
	};



// -- Create the object and materials -----------------------------------------

object {
        test_object

#if( use_df3_proximity_pattern )
	texture {
		pigment_pattern {
			average
			pigment_map {
				[ 3 df3_pattern ]
#if( add_noise = true )
				[ 0.2 slope { y altitude <0,1,0> } color_map { [0 rgb 1] [1 rgb 0] } scale 120 ]
				[ 0.4 bozo color_map { [0 rgb 0] [1 rgb 1] } scale 4 ]
				[ 0.3 bozo color_map { [0 rgb 0] [1 rgb 1] } scale 1 ]
				[ 0.2 bozo color_map { [0 rgb 0] [1 rgb 1] } scale 0.33 ]
				[ 0.1 bozo color_map { [0 rgb 0] [1 rgb 1] } scale 0.1 ]
#end
			}
		}
		texture_map {
#if( bronze = true )
			[ 0.31 shiny_brass ]
			[ 0.43 dull_brass ]
			[ 0.56 dull_brass ]
			[ 0.64 corroded_brass]
#else
			[ 0.0 pigment { rgb <0,169,224>/255 } ]
			[ 0.2 pigment { rgb <50,52,144>/255 } ]
			[ 0.4 pigment { rgb <234,22,136>/255 } ]
			[ 0.6 pigment { rgb <235,46,46>/255 } ]
			[ 0.8 pigment { rgb <253,233,45>/255 } ]
			[ 1.0 pigment { rgb <0,158,84>/255 } ]
#end
		}
	}
#else
	pigment { rgb 1 }
#end
}



// -- The Rest of the Scene ---------------------------------------------------

camera {
    location  <0.0, 200.0, -500.0>
    look_at   <0.0, 50,  0.0>
    right     x*image_width/image_height
    angle 30
#if( use_focal_blur )
    focal_point < 0.0, 50,  0.0 >
    aperture 6
    blur_samples 10
#end
}

sphere {
	0, 20
	translate y*20
	translate x*-70
	translate z*-50
	texture { finish { reflection 1 } }
}

plane {
    y, -1.0
    pigment { checker rgb 1 rgb 0.8 scale 100 }
}



