/*
 *	DF3 Proximity Pattern
 *
 *		Version 0.91
 *			-	Fixes where the centers of each voxel are when interpolating
 *				the DF3 data.
 *			-	Changes the API to require the resolution to be passed 
 *				to the Proximity_LoadMap() function.
 *			-	Adds a tunable bailout parameter for the voxel sampling
 *
 *		Version 0.9
 *			-	Initial Release
 *    
 */


// -- Include Files -----------------------------------------------------------

#include "arrays.inc"



// -- Internals Parameters and APIs -------------------------------------------

#declare Proximity_Interpolation = 2;
#declare Proximity_MeshMode = false;
#declare Proximity_BailoutValue = -1;

#macro Proximity_SetInterpolation( n )
	#declare Proximity_Interpolation = n;
#end

#macro Proximity_SetMeshMode( b )
	#declare Proximity_Interpolation = b;
#end

#macro Proximity_SetBailoutValue( n )
	#declare Proximity_BailoutValue = n;
#end

#macro Proximity_SetBailouOff()
	#declare Proximity_BailoutValue = -1;
#end



// -- Internal Macros ---------------------------------------------------------

#macro inside_m( obj, pos )
	#local current = pos;
	#local dir = <1,1,1>;
	#local done = 0;
	#local num = 0;
	#while ( done != 1 )
		#local norm = <0,0,0>;
		#local inter = trace( obj, current, dir, norm );
		#if ( vlength( norm ) != 0 )
			#local num = num + 1;
			#local current = inter + dir*0.001;
		#else
			#local done = 1;
		#end
	#end
	(mod( num, 2 ) = 1)
#end


#macro halton( index, base )
	#local out = 0.0;
	#local fraction = 1.0 / base;
	#local i = index;
	#while( i > 0 )
		#local remainder = mod( i, base );
		#local out = out + (fraction * remainder);
		#local i = int(i / base);
		#local fraction = fraction / base;
	#end

	out
#end


#macro halton3D( n )
	#local baseX = 2;
	#local baseY = 3;
	#local baseZ = 5;

	< halton( n, baseX ), halton( n, baseY ), halton( n, baseZ ) >
#end



// -- Public API --------------------------------------------------------------

#macro Proximity_GenerateMap( obj, map_size, num_samples, fraction, filename )
    // Step 1. Setup the sampling array (halton seq)
    #local halton_array = array[num_samples];
    #local i = 0;
    #local j = 0;
    #while( i < num_samples )
        #local randomPoint = halton3D( j ) - 0.5;
        #if( vlength( randomPoint ) < 0.5 )
            #declare halton_array[ i ] = randomPoint;
            #local i = i + 1;
        #end
        #local j = j + 1;
    #end

    // Step 2. Create the prox array
    #local prox_array = array[map_size.x][map_size.y][map_size.z];

    // Step 3. Determine the extents of the object
    #local minX = min_extent(obj).x;
    #local maxX = max_extent(obj).x;
    #local minY = min_extent(obj).y;
    #local maxY = max_extent(obj).y;
    #local minZ = min_extent(obj).z;
    #local maxZ = max_extent(obj).z;
    
    #local extentXSize = maxX - minX;
    #local extentYSize = maxY - minY;
    #local extentZSize = maxZ - minZ;
    
    #local numXSamples = map_size.x;
    #local numYSamples = map_size.y;
    #local numZSamples = map_size.z;

    #local adjMinX = minX - ( extentXSize / ( numXSamples - 2 ) );
    #local adjMaxX = maxX + ( extentXSize / ( numXSamples - 2 ) );
    #local adjMinY = minY - ( extentXSize / ( numYSamples - 2 ) );
    #local adjMaxY = maxY + ( extentXSize / ( numYSamples - 2 ) );
    #local adjMinZ = minZ - ( extentXSize / (  numZSamples - 2 ) );
    #local adjMaxZ = maxZ + ( extentXSize / (  numZSamples - 2 ) );

    #local sliceXSize = adjMaxX - adjMinX;
    #local sliceYSize = adjMaxY - adjMinY;
    #local sliceZSize = adjMaxZ - adjMinZ;

    #local stepXSize = sliceXSize / (numXSamples);
    #local stepYSize = sliceYSize / (numYSamples);
    #local stepZSize = sliceZSize / (numZSamples);

    // Step 4. Calculate the proximity data
    #local i = 0;
    #while( i < numXSamples )
        #local j = 0;
        #while( j < numYSamples )
            #local k = 0;
            #while( k < numZSamples )
                #local xpos = adjMinX + (stepXSize / 2) + (i * stepXSize);
                #local ypos = adjMinY + (stepYSize / 2) + (j * stepYSize);
                #local zpos = adjMinZ + (stepZSize / 2) + (k * stepZSize);
                #local numInside = 0;
                #local n = 0;
                #while( n < num_samples )
                	#if( Proximity_MeshMode = true )
	                    #if( inside_m ( obj, <xpos, ypos, zpos> + halton_array [ n ] * stepXSize * fraction ) )
	                        #local numInside = numInside + 1;
	                    #end
                    #else
	                    #if( inside ( obj, <xpos, ypos, zpos> + halton_array [ n ] * stepXSize * fraction ) )
	                        #local numInside = numInside + 1;
	                    #end
			#end
                    #local n = n + 1;
                    #if( n = Proximity_BailoutValue & (numInside = 0 | numInside = Proximity_BailoutValue) )
                        #if( numInside = Proximity_BailoutValue )
                            #local numInside  = num_samples;
                        #end
                        #local n = num_samples;
                    #end
                #end
                #local prox_array[i][j][k] = numInside / num_samples;
                #local k = k + 1;
            #end
            #local j = j + 1;
        #end
        #local i = i + 1;
    #end

    // Write the DF3 file
    ARRAYS_WriteDF3( prox_array, filename, 8 )
#end



#macro Proximity_DebugMapPlacement( obj, resolution )
    #local minX = min_extent(obj).x;
    #local maxX = max_extent(obj).x;
    #local minY = min_extent(obj).y;
    #local maxY = max_extent(obj).y;
    #local minZ = min_extent(obj).z;
    #local maxZ = max_extent(obj).z;
    
    #local extentXSize = maxX - minX;
    #local extentYSize = maxY - minY;
    #local extentZSize = maxZ - minZ;
    
    #local adjMinX = minX - ( extentXSize / ( resolution.x - 2 ) );
    #local adjMaxX = maxX + ( extentXSize / ( resolution.x - 2 ) );
    #local adjMinY = minY - ( extentYSize / ( resolution.y - 2 ) );
    #local adjMaxY = maxY + ( extentYSize / ( resolution.y - 2 ) );
    #local adjMinZ = minZ - ( extentZSize / ( resolution.z - 2 ) );
    #local adjMaxZ = maxZ + ( extentZSize / ( resolution.z - 2 ) );

    #local sliceXSize = adjMaxX - adjMinX;
    #local sliceYSize = adjMaxY - adjMinY;
    #local sliceZSize = adjMaxZ - adjMinZ;

    #local numXSamples = resolution.x;
    #local numYSamples = resolution.y;
    #local numZSamples = resolution.z;

    #local stepXSize = sliceXSize / (numXSamples);
    #local stepYSize = sliceYSize / (numYSamples);
    #local stepZSize = sliceZSize / (numZSamples);

	box { 
		0, 1
		scale < adjMaxX-adjMinX, adjMaxY-adjMinY, adjMaxY-adjMinY >
		translate < adjMinX, adjMinY, adjMinZ >
		#if( Proximity_Interpolation != 0 )
			translate < stepXSize, stepYSize, stepZSize > / (resolution * 2 )
		#end
		pigment { rgbt <0,1,0,0.8> }
	}
	
#end

#macro Proximity_LoadMap( obj, resolution, filename )
    #local minX = min_extent(obj).x;
    #local maxX = max_extent(obj).x;
    #local minY = min_extent(obj).y;
    #local maxY = max_extent(obj).y;
    #local minZ = min_extent(obj).z;
    #local maxZ = max_extent(obj).z;
    
    #local extentXSize = maxX - minX;
    #local extentYSize = maxY - minY;
    #local extentZSize = maxZ - minZ;
    
    #local adjMinX = minX - ( extentXSize / ( resolution.x - 2 ) );
    #local adjMaxX = maxX + ( extentXSize / ( resolution.x - 2 ) );
    #local adjMinY = minY - ( extentYSize / ( resolution.y - 2 ) );
    #local adjMaxY = maxY + ( extentYSize / ( resolution.y - 2 ) );
    #local adjMinZ = minZ - ( extentZSize / ( resolution.z - 2 ) );
    #local adjMaxZ = maxZ + ( extentZSize / ( resolution.z - 2 ) );

    #local sliceXSize = adjMaxX - adjMinX;
    #local sliceYSize = adjMaxY - adjMinY;
    #local sliceZSize = adjMaxZ - adjMinZ;

    #local numXSamples = resolution.x;
    #local numYSamples = resolution.y;
    #local numZSamples = resolution.z;

    #local stepXSize = sliceXSize / (numXSamples);
    #local stepYSize = sliceYSize / (numYSamples);
    #local stepZSize = sliceZSize / (numZSamples);

	#declare temp = pigment {
		density_file df3 filename 
		interpolate Proximity_Interpolation
		scale < adjMaxX-adjMinX, adjMaxY-adjMinY, adjMaxZ-adjMinZ >
		translate < adjMinX, adjMinY, adjMinZ >
		#if( Proximity_Interpolation != 0 )
			translate < stepXSize, stepYSize, stepZSize > / 2
                // Adjust for interpolation
		#end
	}
	
    temp
#end

