//  Persistence of Vision Ray Tracer version 3.6 Include File
//  File ruler.inc
//  Author: Randall Sawyer
//  Last updated: 2006.11.28
//  Description: Macro which creates a ruler object.  Parameters specify
//  the dimensions and attributes of the ruler including its orientation.
//  The edge of the ruler at the tick marked '0' will be at < 0, 0, 0 >.

#ifndef(RULER_INC_TEMP)
#declare RULER_INC_TEMP = version;
#version 3.6;

#ifdef(View_POV_Include_Stack)
    #debug "including ruler.inc\n"
#end


// Usage:
//  ruler( len[float], wid[float], dir[float[0, 1, 2, 3]], font_name[string], font_scale[float], tick_interval[float[integer]], tick_thickness[float], number_interval[float[integer]] )

//  len: The ruler will actually be len+1 long with the first and last ticks being 0.5 in from the end.
//  wid: The ruler's width.  If the numbers and ticks are too crowded, then increase this parameter.
//  dir: 0 = North = horizontal with ticks on top
//       1 = East  = vertical with ticks to the right
//       2 = South = horizontal with ticks below
//       3 = West  = vertical with ticks to the left
//  font: The font used for the numbers
//  font_scale: Change this as needed, it affects the width and height of the numbers, but not the depth.
//  tick_interval: This value defines the frequencey of the ticks.
//  tick_thickness: Increase this value to see the ticks better; decrease it to make them smaller.  This parameter also determines the depth of the numbers.
//  number_interval: Numbers always begin with '0', followed by multiples of number_interval.  Also, ticks neighboring numbers are lengthened.

#macro ruler( len, wid, dir, font_name, font_scale, tick_interval, tick_thickness, number_interval )

//  Direction fail-safe:
    #local d = floor( min( max( dir, 0 ), 4) );

difference {

// body
    union {
        prism {
            linear_spline
            linear_sweep
            -0.5, len+0.5, 6
            < 0, 0 >, < wid/3, -wid/12 >, < wid, -wid/12 >, < wid, tick_thickness >, < 0, tick_thickness >, < 0, 0 >
            hollow
            pigment { color rgb < 1, 1, 0.5 > }
        }
        prism {
            linear_spline
            linear_sweep
            -0.495, len+0.495, 6
            < 0.01*wid, 0 >, < wid/3, -0.99*wid/12 >, < 0.99*wid, -0.99*wid/12 >, < 0.99*wid, 0.9*tick_thickness >, < 0.01*wid, 0.9*tick_thickness >, < 0.01*wid, 0 >
            pigment { color rgb 0 }
        }

    //  Orient the body:
        #if ( mod(d, 3) )
            scale < -1, 1, 1 >
        #end//if
        #if ( 1 - mod( d, 2 ) )
            rotate -90*z
        #end//if
    }//body

// ticks
    union {
        #local n = floor( len/tick_interval );
        #local r = floor( number_interval/tick_interval );
        #local i = 0;
        #while ( i <= n )
            cylinder {
                < -0.1*wid, i*tick_interval, 0 >, < 0.125*wid, i*tick_interval, 0 >, tick_thickness/2
                #if ( !mod( i, r ) )
                    scale < 2, 1, 1 >
                #end//if
            }
            sphere {
                < 0.125*wid, i*tick_interval, 0 >, tick_thickness/2
                #if ( !mod( i, r ) )
                    translate 0.125*wid*x
                #end//if
            }
            #local i = i + 1;
        #end//while
        rotate 14.0362*y

    //  Orient the ticks
        #if ( mod(d, 3) )
            scale < -1, 1, 1 >
        #end//if
        #if ( 1 - mod( d, 2 ) )
            rotate -90*z
        #end//if
    }//ticks

// numbers
    union {
        #local i = 0;
        #while ( i <= len )
            text {
                ttf font_name str(i, 0, 0)
                tick_thickness, 0
                scale < font_scale, font_scale, 1 >
                translate i * ( mod( d, 2 ) ? y : x )
            }
            #local i = i + number_interval;
        #end//while

    //  Direction-dependent number translations:
        #switch ( d )
            #case (0)
                translate -wid*0.875*y
            #break
            #case (1)
                translate -wid*0.875*x
            #break
            #case (2)
                translate wid*11/24*y
            #break
            #case (3)
                translate wid*11/24*x
            #break
        #end//switch

        translate -(tick_thickness/2 + wid/12) * z

    }//numbers

}//difference

#end//ruler


#version RULER_INC_TEMP;
#end//ruler.inc
