// Persistence of Vision Ray Tracer Scene Description File
// File: boulder.pov
// Vers: 3.6
// Desc: Mesh based boulder with texturing
// Date: 06/12/06
// Auth: Abe Madey  bullfrog AT taconic DOT net 
// 
// My objectives here were to: 
// 1. Get a decent looking mesh based boulder.  
// I decided to use the rock maker macro by John VanSickle (http://enphilistor.50megs.com/index.htm).
// A mesh can provide some nice straight edges which help give a rock a "sharp" look (something that is
// a little more difficult with isosurfaces. On the other hand, meshes come with their own chalanges,
// most notably the "shadow line" artifact. I tried mitigating this with the use of area lights with
// only moderate success.
// So, to state the obvious, the "makerock.inc" and "filerock.inc" macros are required. They are
// with this scene (with permission), but are also available from http://enphilistor.50megs.com/inc.htm
// along with instructions.
//
// 2. Fill in the details with a decent perturbed normal texture 
// The wrinkles pattern is an implementation of the classic 1/f summed perlin noise and works well 
// for texturing rough surfaces. There are, however, two things which can significantly affect how the
// wrinkles pattern (and some others too) renders. The first is the difference between using 
//      wrinkles
// or using 
//      pigment_pattern{wrinkles}
// in the normal statement. The latter is, to my eyes, more defined and is employed here. The second aspect
// is the use of the accuracy modifier (which only applies to non "traditional" POV normal patterns). It 
// basically determines the spacing between points for taking the local derivative (and hence getting 
// the gradient or slope). The default is 0.02. Changing this can have a dramatic effect on the
// appearance of the normal. In this scene it is set to .001 in order to get a crisper appearance. Note
// that anti-aliasing can knock out some of the finer detail.
//
// 3. Put some lichens on the rock using procedural textures (actually an after thought). 
// This is mainly an exercise in procedural texturing.      

#version 3.6; 
#include "functions.inc"

//***********SCENE VARIABLES****************    
#declare Radiosity=off; //turning this off will invoke fill lights 
#declare Sun_alt=30;    //key light source altitude
#declare Sun_azm=-90;   //key light source horizontal angle (from +z)  

//makerock.inc parameters
#declare Seed=15;      
#declare Smooth=1;
#declare Splits=5;
#declare Size=.6;
#declare Rough=-.6;
#declare Omega=.40;
#declare Rounding=1; 
#declare FileName="bouldermesh.inc" //file name used if writing to a file

//other controls   
#declare WriteToFile=no; //this is to invoke "filemesh.inc" to write the mesh to an inc file.
#declare ReadFromFile=no;//this is to invoke reading the previously written mesh file. 
#declare RockTrans=transform{rotate -y*55}                 


//******************************************

//Radiosity settings are not necessarily tuned as well as they could be. 
global_settings {
  #if (Radiosity)
    radiosity {
      pretrace_start 0.08          
      pretrace_end   0.04          
      count 100                    
      nearest_count 5              
      error_bound 1.8              
      recursion_limit 3            
      low_error_factor .5          
      gray_threshold 0.25          
      minimum_reuse 0.015          
      brightness 1.2               

      adc_bailout 0.01/2
      normal on           
    }
  #end
}
 
//make sure default ambient values dont mess with radiosity
#default {finish{ambient 0 }} 

// ----------------------------------------

camera {
  location  <0.0, 1.1, -4.0>
  direction 2.5*z
  right     x*image_width/image_height
  look_at   <-0.1, 0.2,  0.0>
}

sky_sphere {
  pigment {
    gradient y
    color_map {
      [0.0 rgb <0.718,0.820,0.906>]
      [0.7 rgb <0.141,0.255,0.635>]
    }
  }
} 
//key light
light_source {
  z*10e6          
  color rgb <0.9608,0.9294,0.6392>*2.0    
  parallel
  area_light x*2e6, y*2e6, 5,5 circular orient adaptive 1
  rotate <-Sun_alt, Sun_azm, 0>
}
#if(!Radiosity)
// camera fill 
light_source {
  <0, 0, 0> 
  color rgb .3 
  translate <0.0, 0.5, -4.0>
}
light_source { //blue fill light opposing key light
  <0, 0, 0>           
  color rgb <0.5490,0.6353,0.9490> 
  translate z*10e6
  rotate <-Sun_alt, Sun_azm+180, 0>
  shadowless
   
} 
#end


plane {
  y, -.4
  pigment { color rgb .75 } 
}


 
  

//basic rock texture
#declare RockTex=
texture{
 pigment{
  slope y  
  cubic_wave
  color_map{
   [0  color rgb .25 transmit 0]
   [1 color rgb .6   transmit 0]
  }
 }
 normal{
  pigment_pattern{wrinkles scale .35 poly_wave 3}
  //wrinkles  scale .1 poly_wave 2
  bump_size 1.8
  accuracy .001
 }
 finish{brilliance 1} 
} 



#declare f_absnoise3d=function{pow(abs(2*f_noise3d(x,y,z)-1),.3)}
#macro Turb() scale 2 warp{turbulence 1 lambda 3 octaves 3} scale 1/2 scale .05 #end
#declare Spots=function{pattern{crackle form x}}

#macro LichenPig(Ltcol,Medcol,Dkcol,Coverage)
  pigment_pattern{
    function{Spots(x,y,z)} 
    color_map{[0 rgb 1][Coverage rgb 0]} 
    translate 10         
    scale 2 warp{turbulence .3 lambda 2.5 octaves 4} scale 1/2 
    scale 1.5
  } 
  pigment_map{
    [0.00 
      function{f_noise3d(x,y,z)}
      Turb() 
      color_map{
        [0.00 rgb Ltcol transmit 1]
        [1.00 rgb Dkcol transmit 1]
      }
    ]
    [0.01 
      function{f_noise3d(x,y,z)} 
      Turb()
     color_map{
        [0.00 rgb Ltcol transmit 0]
        [1.00 rgb Dkcol transmit 0]
      }
    ]
    [0.15 
      function{f_absnoise3d(x,y,z)}
      Turb() 
      color_map{
        [0.05 rgb Medcol transmit 1]
        [0.05 rgb Medcol transmit 0]
        [1.00 rgb Dkcol  transmit 0]
      }
    ]
  }
  scale .25
 
#end 

#declare LichenLt= <0.6431,0.6863,0.5686>;
#declare LichenMed=<0.5451,0.5843,0.4863>;
#declare LichenDk= <0.3569,0.4235,0.3137>;

#declare LichenTex=
texture{
  pigment{
  planar
  pigment_map{
    #local i=0;
    #while (i<100)
    [i/100 LichenPig(LichenLt,LichenMed,LichenDk,.6*(i/100))]
    #local i=i+1;
    #end
  }
  }
  normal{ 
    pigment_pattern{ 
      planar
      pigment_map{
        #local i=0;
        #while (i<100)
        [i/100
          average
          pigment_map{ 
            [1 LichenPig(.5,1,0,.6*(i/100))]
            [1 pigment_pattern{wrinkles scale .25 poly_wave 2}]
          }
        ]
        #local i=i+1;
        #end
      }
    } 
    accuracy .005 
    bump_size .25
  }
  finish{brilliance 2}
}

#if(WriteToFile)
  #include "filerock.inc"
#end

#if(ReadFromFile)
  #declare Rock=object{#include FileName }
#else   
  #declare Rock=object{#include "makerock.inc" }
#end


object{
 Rock 
 double_illuminate
 texture{RockTex}
 texture{LichenTex}
 transform{RockTrans}
}

