// worlds.inc
/*
 * Copyright (C) 2006 Terry Ron Vantreese
 * All rights reserved.
 *
 */


/*

Problem arises when applying extreme values in POV-Ray; many objects disappear if the floating-point error is too 
high.  Numbers related to distances and sizes are stated in units of kilometers, with the "unitscale" factor.  If 
objects begin to disappear, the unitscale factor may be used to bring objects back into view.  However, setting the 
unitscale to extreme in either direction causes artifacts.

Unitscale originally represented "kilometers per unit" but in practice it was a source of confusion.  Now it
represents "units per kilometer".

*/

#include "rad_def.inc"
#include "functions.inc"


#declare epsilon=1/256;
#declare unitscale=0.001;	// units per kilometer

#declare albedo_magnitude=1;
#declare e=exp(1);


// Gamma setting really affect the brilliance power.

#declare land_brilliance=1;	// was 0.75, was previously 0.5
#declare water_brilliance=1;	// was 0.75
#declare cloud_brilliance=1;	// was 0.75
#declare atmos_brilliance=1;	// was 0.875


#declare surface_accuracy=1/1024;



#declare jpeg_type = 100;
#declare png_type = 200;
#declare gif_type = 300;
#macro map_image(image_type, image_name)
	#switch (image_type)
	#case (jpeg_type)
		jpeg image_name
		#break
	#case (png_type)
		png image_name
		#break
	#case (gif_type)
		gif image_name
		#break
	#end
#end

//#declare _half_power=pow(2, 1/2);
//#declare _half_power=2;		// Last used 01-Jun-2006
//#declare _half_power=2.71828182846;
#declare _half_power=exp(1);
//#declare _half_power=2;
//#declare _hh_gamma=1/2.2;
#declare _hh_gamma=1/exp(1);
#declare _hh_antigamma=1/_hh_gamma;
#macro HalfHeight(lr, aa, hh, ah)
	#local _land_radius=(lr*unitscale);
	#local _atm_altitude=(aa*unitscale);
	#local _half_height=(hh*unitscale);
	#local _atm_radius=((lr+aa)*unitscale);
	#local _atm_height=ah*ah;
	min(1, pow(pow(_half_power, (_land_radius-_atm_radius*sqrt(x*x+y*y+z*z))/(_half_height))/_atm_height, _hh_gamma))
#end

#declare sh_gamma=1/exp(1);
#macro ScaleHeight(_base_scale_radius, _atm_scale_radius)
//	min(1,pow(exp(_base_scale_radius-_atm_scale_radius*sqrt(x*x+y*y+z*z)), sh_gamma))
	min(1,exp(_base_scale_radius-_atm_scale_radius*sqrt(x*x+y*y+z*z)))
#end


#macro albedo(_albvalue)
	diffuse (albedo_magnitude*_albvalue)
#end


#macro TransformWorld(_distance, _hour, _year, _heading, _pitch, _bank, _tilt, _tiltshift)
	rotate <0, 90-(15*_hour)-360*(_year-_tiltshift), 0>
	rotate <_tilt, 360*(_year-_tiltshift), 0>
	translate <0, 0, (_distance*unitscale)>
	rotate <_pitch, (_heading-180), _bank>
#end



#ifndef ( PixelRatio )
#declare PixelRatio=1;
#end


#macro trvCam(_zoom, _lookat, _viewfrom)
	camera
	{
		direction <0, 0, _zoom>
		up y
		right PixelRatio*x*image_width/(image_height)
		look_at _lookat*unitscale
		location _viewfrom*unitscale
	}
#end

#macro TerragenViewFrom(_zoom, _fromobject, _shift_x, _shift_y, _shift_z)
	camera
	{
		direction <0, 0, _zoom>
		up y*image_height/image_width
		right x*PixelRatio
		look_at <_shift_x*unitscale, _shift_y*unitscale, _shift_z*unitscale>
		location <0, 0, -(_fromobject*unitscale)>
	}
#end
#macro OmniViewFrom(_fromobject, _shift_x, _shift_y, _shift_z)
	camera
	{
		omnimax
//		direction <0, 0, _zoom>
		up y
		right pixel_ratio*x*image_width/image_height
		look_at <_shift_x*unitscale, _shift_y*unitscale, _shift_z*unitscale>
		location <0, 0, -(_fromobject*unitscale)>
	}
#end





#macro SunLight(_distance, _sun_radius, _color, _magnitude, _heading, _pitch, _bank)
	#local _sun_diameter=(2*_sun_radius*unitscale);
	light_source {
		<0, 0, -(_distance*unitscale)> _color*_magnitude
		fade_power 0.5
		fade_distance _sun_diameter*sqrt(2)
	//	area_light <_sun_diameter, 0, 0> <0, _sun_diameter, 0> 17, 17
	//	area_light <_sun_diameter, 0, 0> <0, _sun_diameter, 0> 3, 3
	//	circular
		rotate <_pitch, (_heading-180), _bank>
	}
	sphere {<0, 0, -(_distance*unitscale)>, (_sun_radius*unitscale)
		texture {
			pigment {color _color}
			finish {
				ambient 2*_magnitude
			}
		}
		no_shadow
		rotate <_pitch, (_heading-180), _bank>
	}
#end


#macro Water(_radius, _albedo, _r, _g, _b, _f, _zenith, _tangent)
	sphere
	{
		<0, 0, 0>, (_radius*unitscale)
		hollow
		texture {
			pigment { color rgbf <_r, _g, _b, _f> }
		//	pigment { color rgbf <1, 1, 1, 1> }
			finish {
				ambient 0
				albedo(_albedo)
				specular 1/8
				roughness 1/16
				brilliance water_brilliance
				reflection {
					_zenith, _tangent
				//	fresnel
					metallic
				}
				conserve_energy
			//	metallic
			}
			normal { bumps bump_size 0.00075 scale <4, 1, 1>*(0.25*unitscale) }
		}
		interior {
			ior 1.33
			media {
				method 3
			//	samples 20 intervals 5
				absorption color rgb <(1-_r), (1-_g), (1-_b)>*0.005
			}
		}
	}
#end

#macro Land(_radius, _albedo, _landmapimagetype, _landmap)
	sphere
	{
		<0, 0, 0>, (_radius*unitscale)
		texture
		{
			pigment
			{
				image_map
				{
					map_image(_landmapimagetype, _landmap)
					map_type 1
				}
			}
			finish
			{
				ambient 0
				brilliance land_brilliance
				albedo(_albedo)
			}
		}
	}
#end


#macro LandMask(_radius)
	sphere
	{
		<0, 0, 0>, (_radius*unitscale)
		texture
		{
			pigment
			{
				image_map
				{
					map_image(png_type, "../mask-map.png")
					map_type 1
				}
			}
			finish
			{
				ambient 0
				brilliance land_brilliance
				albedo(1)
			}
		}
	}
#end



#macro LandSurface(_radius, _hirelief, _lorelief, _albedo, _landmapimagetype, _landmap, _landbumpimagetype, _landbump)
	#local _planetradius=_radius*unitscale;
	#local _planethirelief=_hirelief*unitscale;
	#local _planetlorelief=_lorelief*unitscale;
	#local _planetdelta=_planethirelief-_planetlorelief;
	#local fn_PlanetBump=function{
		pigment{
			image_map{
				map_image(_landbumpimagetype, _landbump)
				map_type 1
				//once
				interpolate 2
			}
		}
	}
	isosurface
	{
		function {
			f_sphere(x, y, z, (_planetradius+_planetlorelief))-(fn_PlanetBump(x, y, z).red*_planetdelta)
		}
		contained_by {
			sphere{
				0,(_planetradius+_planethirelief)
			}
		}
		accuracy surface_accuracy*unitscale
		texture
		{
			pigment
			{
				image_map
				{
					map_image(_landmapimagetype, _landmap)
					map_type 1
				}
			}
			finish
			{
				ambient 0
				brilliance land_brilliance
				albedo(_albedo)
			}
		}
	}
#end


#macro LandRSurface(_radius, _hirelief, _lorelief, _albedo, _landmapimagetype, _landmap, _landbumpimagetype, _landbump)
	#local _planetradius=_radius*unitscale;
	#local _planethirelief=_hirelief*unitscale;
	#local _planetlorelief=_lorelief*unitscale;
	#local _planetdelta=_planethirelief-_planetlorelief;
	#local fn_PlanetBump=function{
		pigment{
			image_map{
				map_image(_landbumpimagetype, _landbump)
				map_type 1
				//once
				interpolate 2
			}
		}
	}
	isosurface
	{
		function {
			f_sphere(x, y, z, (_planetradius+_planetlorelief))-((1-fn_PlanetBump(x, y, z).red)*_planetdelta)
		}
		contained_by {
			sphere{
				0,(_planetradius+_planethirelief)
			}
		}
		accuracy surface_accuracy*unitscale
		texture
		{
			pigment
			{
				image_map
				{
					map_image(_landmapimagetype, _landmap)
					map_type 1
				}
			}
			finish
			{
				ambient 0
				brilliance land_brilliance
				albedo(_albedo)
			}
		}
	}
#end



#macro LandBump(_radius, _albedo, _landmapimagetype, _landmap, _landbumpimagetype, _landbump)
	sphere
	{
		<0, 0, 0>, (_radius*unitscale)
		texture
		{
			pigment
			{
				image_map
				{
					map_image(_landmapimagetype, _landmap)
					map_type 1
				}
			}
			normal
			{
				bump_map
				{
					map_image(_landbumpimagetype, _landbump)
					map_type 1
				}
				bump_size 2
			}
			finish
			{
				ambient 0
				brilliance land_brilliance
				albedo(_albedo)
			}
		}
	}
#end

#macro LandRBump(_radius, _albedo, _landmapimagetype, _landmap, _landbumpimagetype, _landbump)
	sphere
	{
		<0, 0, 0>, (_radius*unitscale)
		texture
		{
			pigment
			{
				image_map
				{
					map_image(_landmapimagetype, _landmap)
					map_type 1
				}
			}
			normal
			{
				bump_map
				{
					map_image(_landbumpimagetype, _landbump)
					map_type 1
				}
				bump_size -2
			}
			finish
			{
				ambient 0
				brilliance land_brilliance
				albedo(_albedo)
			}
		}
	}
#end

#macro GasPlanet(_radius, _albedo, _imagetype, _cloudimage)
	sphere
	{
		<0, 0, 0>, (_radius*unitscale)
		texture
		{
			pigment
			{
				image_map
				{
					map_image(_imagetype, _cloudimage)
					map_type 1
				}
			}
			finish
			{
				ambient 0
				//specular 0.125
				brilliance cloud_brilliance
				albedo(_albedo)
			}
		}
	}
#end

#macro Clouds(_radius, _altitude, _albedo, _imagetype, _cloudimage, _lo_color, _hi_color)
	#local _cloud_radius=(_radius+_altitude)*unitscale;
	sphere
	{
		<0, 0, 0>, _cloud_radius
		hollow on
		texture
		{
			pigment
			{
			/*	image_map
				{
					map_image(_imagetype, _cloudimage)
					map_type 1
				}*/
				image_pattern
				{
					map_image(_imagetype, _cloudimage)
					map_type 1
					interpolate 4
				//	use_alpha
				}
				color_map
				{
					[0.0	color _lo_color]
					[1.0	color _hi_color]
				}
			}
			finish
			{
				ambient 0
			//	specular 0.125
				brilliance cloud_brilliance
				albedo(_albedo)
			}
		}
	}
#end


#macro CloudSphere(_radius, _minalt, _maxalt, _r, _g, _b)
	#local _color=rgb<_r, _g, _b>;
	difference {
		sphere {
			<0, 0, 0>, (_radius+_maxalt)*unitscale
		}
		sphere {
			<0, 0, 0>, (_radius+_minalt)*unitscale
		}
		hollow on
		texture {pigment {color rgbt 1}}
		interior {
			media {
				method 3 aa_threshold 0.1 aa_level 3
				samples 4 intervals 1
				scattering {
					2, color _color/unitscale
					extinction 1
				}
				density {
					wrinkles
					scale <5, 2, 2>*500*unitscale
					warp {
						turbulence 5*unitscale
					}
					color_map {
						[0	color _color*pow(2,0)]
						[1/8	color _color*pow(2, -1)]
						[2/8	color _color*pow(2, -2)]
						[3/8	color _color*pow(2, -3)]
						[4/8	color _color*pow(2, -4)]
						[5/8	color _color*pow(2, -5)]
						[6/8	color _color*pow(2, -6)]
						[7/8	color _color*pow(2, -7)]
						[1	color _color*0]
					}
				}
			}
		}
	}
#end



// Under HalfHeight function regular power:
// Optimal atmosphere altitude for density is 4 * half height
// Optimal atmosphere altitude for altitude is 10 * half height

// Under HalfHeight function sqrt() power:
// Optimal for density is 10 * half height
// Optimal for altitude has not been determined, but object continues to lose density with increased altitude.

// Atmosphere() to be decommissioned -- too expensive to maintain and render.
#declare Atmosphere_Option=5;
#macro Atmosphere(_land_radius, _hh_value, _r, _g, _b, _atm_mag, _decay_magnitude, _fade_distance)
	#switch (Atmosphere_Option)
	#case (1)	// HalfHeight() using power of 2
		#local _atm_height=4;
		#local _atm_lo=pow(16, _hh_gamma);
		#break
	#case (2)	// HalfHeight() using sqrt(power of 2)
		#local _atm_height=10;
		#local _atm_lo=pow(1024, _hh_gamma);
		#break
	#case (3)	// sphere with density_map
		#local _atm_height=12;
		#local _atm_lo=pow(4096, _hh_gamma);
		#break
	#case (4)	// HalfHeight() using power of e, suggesting to User to reduce extinction amount.
		#local _atm_height=16;
		#local _atm_lo=pow(65536, _hh_gamma);
		#break
	#case (5)	// Expensive, use only in abundant resources.
		#local _atm_height=32;
		#local _atm_lo=pow(65536*65536, _hh_gamma);
		#break
	#end
//	#local _half_height=_hh_value/2;
//	#local _half_height=_hh_value*_hh_gamma;
	#local _half_height=_hh_value;
	#local _atm_magnitude=_atm_mag;

//	#local _atm_altitude=_atm_height*_half_height*_hh_antigamma;
	#local _atm_altitude=_atm_height*_half_height;

	#local _atm_radius=_land_radius+_atm_altitude;
	#local _color=rgb<_r, _g, _b>;
	#local _color_map=_color;

	#local _atm_hi_color=_color_map*_atm_height;
//	#local _atm_lo_color=_color_map/_atm_lo;
	#local _atm_lo_color=_color_map*0;

//	difference {
		sphere {
			<0, 0, 0>, _atm_radius*unitscale
		}
//		sphere {
//			<0, 0, 0>, (_land_radius-_atm_altitude)*unitscale
//		}
		hollow on
		//no_shadow
		material {
			texture {
				pigment {
					rgbt 1
				}
				finish
				{
					ambient 0
					brilliance atmos_brilliance
					diffuse 0
				}
			}
			interior {
			//	ior 1
				#if (_fade_distance != 0)
					fade_distance _half_height*unitscale*_fade_distance
					fade_power _decay_magnitude
					fade_color <_r, _g, _b>
				#end
				media {	// Info from POV-Ray manual: Density*sqrt(3)/vlength(Scale)
					method 3
					absorption _color*(_atm_magnitude/unitscale)*_decay_magnitude
					scattering {
						4, _color*(_atm_magnitude/unitscale)
					//	eccentricity 0.125	// available only on scattering type 5.
						extinction _decay_magnitude
					}
					density {
						function { HalfHeight(_land_radius, _atm_altitude, _half_height, _atm_height) }
						density_map {
							[0	_atm_lo_color]
							[1	_atm_hi_color]
						}

						scale (_atm_radius*unitscale)
					}
				}
			}
		}
//	}
#end


#macro AtmSphere(_land_radius, _scale_height, _r, _g, _b, _atm_magnitude, _decay_magnitude)
	#local _atm_height=32;
	#local _atm_altitude=_atm_height*_scale_height;
	#local _atm_radius=_land_radius+_atm_altitude;
	#local _color=rgb<_r, _g, _b>;

	#local _atm_hi_color=_color;
	#local _atm_lo_color=_color*0;

	#local _base_scale_radius=(_land_radius/_scale_height);
	#local _atm_scale_radius=(_land_radius/_scale_height+_atm_height);

//	difference {
		sphere {
			<0, 0, 0>, _atm_radius*unitscale
		}
//		sphere {
//			<0, 0, 0>, (_land_radius-_atm_altitude)*unitscale
//		}
		hollow on
		//no_shadow
		material {
			texture {
				pigment {
					rgbt 1
				}
				finish
				{
					ambient 0
					brilliance atmos_brilliance
					diffuse 0
				}
			}
			interior {
				media {	// Info from POV-Ray manual: Density*sqrt(3)/vlength(Scale)
					method 3
					absorption _color*(_atm_magnitude/unitscale)*_decay_magnitude
					scattering {
						4, _color*(_atm_magnitude/unitscale)
					//	eccentricity 0.125	// available only on scattering type 5.
						extinction 1
					}
					density {
						function { ScaleHeight(_base_scale_radius, _atm_scale_radius) }
						density_map {
							[0	_atm_lo_color]
							[1	_atm_hi_color]
						}

						scale (_atm_radius*unitscale)
					}
				}
			}
		}
//	}
#end


#macro Atm2Sphere(_land_radius, _scale_height, _r, _g, _b, _atm_magnitude, _decay_magnitude, _scale_height2, _r2, _g2, _b2, _atm_magnitude2, _decay_magnitude2)
	#local _atm_height=32;

	#local _atm_altitude=_atm_height*_scale_height;
	#local _atm_radius=_land_radius+_atm_altitude;
	#local _color=rgb<_r, _g, _b>;

	#local _atm_hi_color=_color;
	#local _atm_lo_color=_color*0;

	#local _base_scale_radius=(_land_radius/_scale_height);
	#local _atm_scale_radius=(_land_radius/_scale_height+_atm_height);

	#local _atm_altitude2=_atm_height*_scale_height2;
	#local _atm_radius2=_land_radius+_atm_altitude2;
	#local _color2=rgb<_r2, _g2, _b2>;

	#local _atm_hi_color2=_color2;
	#local _atm_lo_color2=_color2*0;

	#local _base_scale_radius2=(_land_radius/_scale_height2);
	#local _atm_scale_radius2=(_land_radius/_scale_height2+_atm_height);

//	difference {
		sphere {
			<0, 0, 0>, _atm_radius*unitscale
		}
//		sphere {
//			<0, 0, 0>, (_land_radius-_atm_altitude)*unitscale
//		}
		hollow on
		//no_shadow
		material {
			texture {
				pigment {
					rgbt 1
				}
				finish
				{
					ambient 0
					brilliance atmos_brilliance
					diffuse 0
				}
			}
			interior {	// Info from POV-Ray manual: Density*sqrt(3)/vlength(Scale)
				media {
					method 3
					absorption _color*(_atm_magnitude/unitscale)*_decay_magnitude
					scattering {
						4, _color*(_atm_magnitude/unitscale)
					//	eccentricity 0.125	// available only on scattering type 5.
						extinction 1
					}
					density {
						function { ScaleHeight(_base_scale_radius, _atm_scale_radius) }
					//	function { pow(ScaleHeight(_base_scale_radius, _atm_scale_radius), sh_gamma) }
						density_map {
							[0	_atm_lo_color]
							[1	_atm_hi_color]
						}

						scale (_atm_radius*unitscale)
					}
				}
				media {
					method 3
					absorption _color2*(_atm_magnitude2/unitscale)*_decay_magnitude2
					scattering {
						2, _color2*(_atm_magnitude2/unitscale)
					//	eccentricity 0.125	// available only on scattering type 5.
						extinction 1
					}
					density {
						function { ScaleHeight(_base_scale_radius2, _atm_scale_radius2) }
						density_map {
							[0	_atm_lo_color2]
							[1	_atm_hi_color2]
						}

						scale (_atm_radius2*unitscale)
					}
				}
			}
		}
//	}
#end




// Stars

// Crackle method

#macro Stars_Crackle(_heading, _pitch, _bank)
sky_sphere {
	//create impression of multiple stars of different sizes.
	pigment {
		pigment_pattern {
			crackle
			form <1,0,0>
			pigment_map {
				[0
					crackle
					solid
					colour_map {
						[0	rgb 1]
						[1	rgb 0]
					}
				]
				[1 rgb 0]
			}
		}

		//poly_wave 10
		#local _starcolor=rgb 5;
		#local _powerfig=sqrt(2);
		colour_map {
			[0	rgb 0]
			[1/12	_starcolor*pow(_powerfig, -1024)]
			[2/12	_starcolor*pow(_powerfig, -512)]
			[3/12	_starcolor*pow(_powerfig, -256)]
			[4/12	_starcolor*pow(_powerfig, -128)]
			[5/12	_starcolor*pow(_powerfig, -64)]
			[6/12	_starcolor*pow(_powerfig, -32)]
			[7/12	_starcolor*pow(_powerfig, -16)]
			[8/12	_starcolor*pow(_powerfig, -8)]
			[9/12	_starcolor*pow(_powerfig, -4)]
			[10/12	_starcolor*pow(_powerfig, -2)]
			[11/12	_starcolor*pow(_powerfig, -1)]
			[1	_starcolor*pow(_powerfig, 0)]
		}

		//scale 1/sqrt(image_width*image_width + image_height*image_height)
		scale 1/1024
	}
	TransformWorld(0, 0, 0, _heading, _pitch, _bank, 0, 0)
}
#end


#macro Stars_Crackle_Color(_heading, _pitch, _bank)
sky_sphere {
	//create impression of multiple stars of different sizes.
	pigment {
		pigment_pattern {
			crackle
			form <1,0,0>
			pigment_map {
				[0
					crackle
					solid
					colour_map {
						[0	rgb 1]
						[1	rgb 0]
					}
				]
				[1 rgb 0]
			}
		}

		//poly_wave 10
		#local _starcolor=rgb 5;
		#local _powerfig=sqrt(2);
		colour_map {
			[0	rgb 0]
			[1/12	rgb <1, 0, 0>*8*pow(_powerfig, -1024)]
			[2/12	rgb <1, .25, 0>*8*pow(_powerfig, -512)]
			[3/12	rgb <1, .5, 0>*8*pow(_powerfig, -256)]
			[4/12	rgb <1, .75, 0>*8*pow(_powerfig, -128)]
			[5/12	rgb <1, 1, 0>*8*pow(_powerfig, -64)]
			[6/12	rgb <.75, 1, 0>*8*pow(_powerfig, -32)]
			[7/12	rgb <.5, 1, 0>*8*pow(_powerfig, -16)]
			[8/12	rgb <.25, 1, .33>*8*pow(_powerfig, -8)]
			[9/12	rgb <0, 1, .67>*8*pow(_powerfig, -4)]
			[10/12	rgb <0, 0.75, 1>*8*pow(_powerfig, -2)]
			[11/12	rgb <0, .5, 1>*8*pow(_powerfig, -1)]
			[1	rgb <0, 0, 1>*8*pow(_powerfig, 0)]
		}

		//scale 1/sqrt(image_width*image_width + image_height*image_height)
		scale 1/1024
	}
	TransformWorld(0, 0, 0, _heading, _pitch, _bank, 0, 0)
}
#end



// Spotted PolyWave method (had limited examination.)

#macro Stars_PolyWave(_heading, _pitch, _bank)
sky_sphere {
	pigment {
		//stars
		spotted
		scale .004*320/(sqrt(image_width*image_width + image_height*image_height)*.7) //keep it the right size relative to the image pixels.
		poly_wave 38
		colour_map {
			[0 rgb 0]
			[1 rgb 30]
		}
	}

	TransformWorld(0, 0, 0, _heading, _pitch, _bank, 0, 0)
}
#end



#macro RingsMap(_map, _outer, _inner, _nscale, _divisions, _ringalbedo)
	#local ABdiff=_outer-_inner;
	#local Xring=_outer/_nscale;
	#local Malpha=_nscale*_inner/_outer;
	#local Step=ABdiff*_nscale/(_divisions*_outer);

	disc
	{
		<0, 0, 0> y, Malpha+(_map)*Step, Malpha+(_map-1)*Step
		pigment
		{
			onion
			colour_map
			{
				Planet_RingMap(_map)
			}
		}
		finish
		{
			ambient 0
			brilliance 1
			albedo(_ringalbedo)
		}
		scale Xring*unitscale
	}
	hollow on
	double_illuminate
#end



#macro PlanetRings(_ringimage, _outer, _inner, _nscale, _divisions, _ringalbedo, _distance, _hour, _year, _heading, _pitch, _bank, _tilt, _tiltshift)
	#include _ringimage
	#local ABdiff=_outer-_inner;
	#local Xring=_outer/_nscale;
	#local Malpha=_nscale*_inner/_outer;
	#local Step=ABdiff*_nscale/(_divisions*_outer);
	#local i=1;
	#while (i<=_divisions)
		object {
			disc
			{
				
				<0, 0, 0> y, Malpha+(i)*Step, Malpha+(i-1)*Step
				pigment
				{
					onion
					colour_map
					{
						Planet_RingMap(i)
					}
				}
				finish
				{
					ambient 0
					brilliance 1
					albedo(_ringalbedo)
				}
				scale Xring*unitscale
			}
			hollow on
			double_illuminate
			TransformWorld(_distance, _hour, _year, _heading, _pitch, _bank, _tilt, _tiltshift)
		}
		#local i=i+1;
	#end
#end

#macro ExpensivePlanetRings(_ringimage, _outer, _inner, _divisions, _ringalbedo, _distance, _hour, _year, _heading, _pitch, _bank, _tilt, _tiltshift)
	#include _ringimage
	#local Alpha=_inner;
	#local Beta=_outer;
	#local ABdiff=Beta-Alpha;
	#local ABdivisions=_divisions;
	#local N=2;
	#local M=1;
	#local Xring=Beta/N;
	#local Malpha=N/Beta*Alpha;
	#local i=0;
	#local Step=ABdiff*N/(ABdivisions*Beta);
	#while (i<ABdivisions)
	object {
		disc
		{
			<0, 0, 0> y, Malpha+(i)*Step, Malpha+(i-1)*Step
			pigment
			{
			//	onion
				Planet_Rings(i)
			}
			finish
			{
				ambient 0
				brilliance 1
				albedo(_ringalbedo)
			}
			scale Xring*unitscale
		}
		hollow on
		double_illuminate
		TransformWorld(_distance, _hour, _year, _heading, _pitch, _bank, _tilt, _tiltshift)
	}
	#local i=i+1;
	#end
#end
