#include "math.inc"

/*
SmoothTransition is intended to drive animation motions that start, pick up speed, then coast to a stop. It takes three arguments: The clock value at which to start the animation, the time to end the animation, and the current clock value. It returns a value from zero to one, inclusive.

An example call might look like this:

sphere {
	<0, 0, 0> 1
	pigment {rgb 1}

	translate SmoothTransition(0, 10, clock) * 10 * y
}

This will start moving at clock = 0, and stop at clock = 10. It will move from <0, 0, 0> to <0, 10, 0>.

You can also use it in rotate and scale statements, or do more complicated math with it. For example, maybe we want to pan a camera past an object, while keeping the camera continuously pointed at that object:

camera {
	look_at <5, 0, 0>
	position <0, 0, -10 + SmoothTransition(0, 10, clock) * 20>
}

This moves the camera from <0, 0, -10> to <0, 0, 10> while keeping it focused at <5, 0, 0>.

BTW, SmoothTransition is especially impressive looking when used for moving the camera around. :-)
*/

/*
How it works:

Imagine a plot of position versus time for an object. If it remains still, the plot line is horizontal. If it moves at constant speed, the line is a straight diagonal.

We want the object to speed up and then slow down. This line would start horizontal, then curve upwards, then level out.

For the acceleration, we can use the plot y = x^2. For the deceleration, we transform this plot so that the focus is at the time and position we want to stop at, and negate it, so it curves down. Then we can transform both curves so that they touch at one point. By combining these two, we have a line that curves up, then levels out. From this, we have our animation function.
*/

#macro SmoothTransition(Start, End, Current)
	#if (Start > End)
		#error "SmoothTransition--Start must be less than than End"
	#end
	#switch (Current)
		#range (Start, (End - Start) / 2 + Start) // Accelerating
			(pow((Current - Start) / (End - Start), 2) * 2) // Aaah! Scary math!
		#break
		#range ((End - Start) / 2 + Start, End) // Decelerating
			(1 - pow((Current - End) / (End - Start), 2) * 2)
		#break
		#else
			#if (Current <= Start) // Not moving yet
				0
			#else // Done moving
				1
			#end
	#end
#end

/*
This macro is used the same way as SmoothTransition. It moves things at a contant rate, instead of speeding up and slowing down.
*/
#macro ConstantTransition(Start, End, Current)
	#if (Current < Start)
		0
	#else
		#if (Current < End)
			((Current - Start) / (End - Start))
		#else
			1
		#end
	#end
#end

/*
This macro creates a motion like ConstantTransition, but which repeats back and forth between 0 and 1.

The motion repeats every _Period_ ticks.
*/

#macro RepeatingConstantTransition(Period, Current)
	#local Ramp = clamp(Current, 0, 2 * Period) / Period;
	#if (Ramp > 1)
		(2 - Ramp)
	#else
		(Ramp)
	#end
#end
