// rainbow.inc v1.0 (2005) by Florian Siegmund (florian.siegmund@gmx.at)
//
// ---
//
// explanation of identifiers:
//
// rb_type:
//      #1: rainbow with pov-ray's rainbow feature (renders very fast)
//      #2: rainbow with pov-ray's media feature (renders slower but is more accurate and shows up pigment_projected_onto_rb if declared)
//
// rb_light_location:
//      location of the light source which is responsible for the rainbow (light source is imaginary, not really put into the scene!)
//
// rb_light_c:
//      color of the rainbow's light source
//
// rb_area_light_radius:
//      radius of the imaginary light source in order to get a smoother rainbow (the area light looks like the surface of a sphere)
//
// rb_area_light_angle:
//      angle between the camera and the rainbow's area light source (is computed automatically if rb_area_light_radius is declared)
//      
// rb_area_light_samples:
//      imaginary area light samples (higher values result in longer parsing times but do not change rendering speed)
//
// rb_brightness:
//      brightness of the rainbow
//
// rb_distance:
//      distance of the rainbow which is calculated like the distance in pov-ray's fog feature
//
// rb_c_map_samples:
//      number of color map entries in the rainbow pigment
//
// rb_spectrum_samples:
//      number of samples in the wavelength spectrum used to compute the rainbow, which are then averaged in the rainbow pigment
//
// rb_intervals:
//      number of intervals (each of them consisting of 256 samples) used to simplify the rainbow pigment (in order to render faster)
//
// rb_media_intervals:
//      number of intervals in the rainbow media (only used with rainbow type #2)
//
// secondary_rb:
//      determines whether a secondary rainbow is computed or not
//
// rb_sphere_radius:
//      radius of the sphere that contains the rainbow media (does not need to be declared in normal cases)
//
// pigment_projected_onto_rb:
//      pigment that is used to determine in which areas the rainbow will be visible
//
// camera location:
//      location of the scene camera
//
// ---
//
// the code is written in a way so that the rainbow can also be computed if none of the identifiers above are declared,
// but you should at least declare rb_light_location and camera_location in order to get a correct result,
// otherwise there will be a warning in the message window.
//
// ---
//
// syntax:
//
// #declare rb_type = INTEGER;                // INTEGER = [1 | 2]   // default: 1
// #declare rb_light_c = VECTOR;              // VECTOR dim = 3      // default: <1, 1, 1>
// #declare rb_area_light_radius = FLOAT;     // FLOAT = ]-oo; oo[   // will be converted to the absolute value of FLOAT
// #declare rb_area_light_angle = FLOAT;      // FLOAT = ]-oo; oo[   // default: 0.2 (only used if rb_area_light_radius is not declared)
// #declare rb_area_light_samples = INTEGER;  // INTEGER = [1; oo[   // default: 20
// #declare rb_brightness = FLOAT;            // FLOAT = ]-oo; oo[   // default: 1
// #declare rb_distance = FLOAT;              // FLOAT = ]0; oo[     // default: 1
// #declare rb_c_map_samples = INTEGER;       // INTEGER = [2; 256]  // default: 60
// #declare rb_spectrum_samples = INTEGER;    // INTEGER = [1; 128]  // default: 20
// #declare rb_intervals = INTEGER;           // INTEGER = [1; oo[   // if rb_type = 1: default: 10  // if rb_type = 2: default: 2
// #declare rb_media_intervals = INTEGER;     // INTEGER = [2; oo[   // default: 2
// #declare secondary_rb = BOOL;              //                     // default: no
// #declare rb_sphere_radius = FLOAT;         // FLOAT = ]0; oo[     // default: 1e6
// #declare rb_light_location = VECTOR;       // VECTOR dim = 3
// #declare camera_location = VECTOR;         // VECTOR dim = 3
// #declare pigment_projected_onto_rb = pigment {...}                // only used if rb_type = 2
//
// #include "rainbow.inc"

#version 3.6;

#include "wavl_rgb.inc"

// default values

#ifndef (rb_type) #declare rb_type = 1; #end
#ifndef (rb_light_c) #declare rb_light_c = <1, 1, 1>; #end
#ifndef (rb_area_light_radius)
    #ifndef (rb_area_light_angle) #declare rb_area_light_angle = 0.2; #end
#end
#ifndef (rb_area_light_samples) #declare rb_area_light_samples = 20; #end
#ifndef (rb_brightness) #declare rb_brightness = 1; #end
#ifndef (rb_distance) #declare rb_distance = 1; #end
#ifndef (rb_c_map_samples) #declare rb_c_map_samples = 60; #end
#ifndef (rb_spectrum_samples) #declare rb_spectrum_samples = 20; #end
#ifndef (rb_intervals)
    #if (rb_type = 1)
        #declare rb_intervals = 10;
    #else
        #declare rb_intervals = 2;
    #end
#end
#ifndef (rb_media_intervals) #declare rb_media_intervals = 2; #end
#ifndef (secondary_rb) #declare secondary_rb = no; #end
#ifndef (rb_sphere_radius) #declare rb_sphere_radius = 1e6; #end

// necessary pre-calculations

#ifdef (rb_area_light_radius)
    #declare rb_area_light_radius = abs (rb_area_light_radius);
#end

#ifdef (rb_area_light_radius)
    #declare rb_area_angle = 2*asin (rb_area_light_radius/vlength (camera_location-rb_light_location));
#else
    #declare rb_area_angle = rb_area_light_angle;
#end

#if (secondary_rb)
    #declare rb_bness = 2*rb_brightness;
#else
    #declare rb_bness = rb_brightness;
#end

// warnings and errors

#ifdef (rb_area_light_radius)
    #if (vlength (camera_location-rb_light_location) < rb_area_light_radius)
        #warning "Camera is inside the rainbow area light source.\n"
    #end
#end

#if (rb_area_light_samples < 1)
    #warning "rb_area_light_samples has to be higher than or equal to 1. rb_area_light_samples is set to default value (20).\n"
    #declare rb_area_light_samples = 20;
#end

#if ((rb_type != 1) & (rb_type != 2))
    #warning "rb_type has to be 1 or 2. rb_type is set to 1."
    #declare rb_type = 1;
#end

#if (rb_distance <= 0)
    #warning "rb_distance has to be higher than zero. rb_distance is set to 1.\n"
    #declare rb_distance = 1;
#end

#ifndef (rb_light_location)
    #warning "rb_light_location has to be declared. rb_light_location is set to <0, 0, -1>.\n"
    #declare rb_light_location = <0, 0, -1>;
#end

#ifndef (camera_location)
    #if ((rb_light_location.x = 0) & (rb_light_location.y = 0) & (rb_light_location.z = 0))
        #warning "camera_location has to be declared. camera_location is set to <0, 0, 1>.\n"
        #declare camera_location = <0, 0, 1>;
    #else
        #warning "camera_location has to be declared. camera_location is set to <0, 0, 0>."
        #declare camera_location = <0, 0, 0>;
    #end
#end

#if ((camera_location.x = rb_light_location.x) & (camera_location.y = rb_light_location.y) & (camera_location.z = rb_light_location.z))
    #warning "camera_location and rb_light_location are the same. rb_light_location is set to camera_location + <-1, 0, 0>.\n"
    #declare rb_light_location = camera_location+<0, 0, -1>;
#end

#if ((rb_c_map_samples < 2) | (rb_c_map_samples > 256))
    #warning "rb_c_map_samples has to be in the interval [2; 256]. rb_c_map_samples is set to default value (60).\n"
    #declare rb_c_map_samples = 60;
#end

#if ((rb_spectrum_samples < 1) | (rb_spectrum_samples > 128))
    #warning "rb_spectrum_samples has to be in the interval [1; 128]. rb_spectrum_samples is set to default value (20).\n"
    #declare rb_spectrum_samples = 20;
#end

#if (rb_intervals <= 0)
    #if (rb_type = 1)
        #warning "rb_intervals has to be higher than zero. rb_intervals is set to default value for rainbow type #1 (10).\n"
        #declare rb_intervals = 10;
    #else
        #warning "rb_intervals has to be higher than zero. rb_intervals is set to default value for rainbow type #2 (2).\n"
        #declare rb_intervals = 2;
    #end
#end
    
#if (rb_sphere_radius <= 0)
    #warning "rb_sphere_radius has to be higher than zero. rb_sphere_radius is set to default value (1e6).\n"
    #declare rb_sphere_radius = 1e6;
#end

#if (rb_media_intervals < 2)
    #warning "rb_media_intervals has to be higher than or equal to 2. rb_media_intervals is set to default value (2).\n"
    #declare rb_media_intervals = 2;
#end

// main rainbow calculation macro

#macro calculate_rb_color ()
    #declare alpha_i = asin (ray_dist)/pi*180;
    #declare alpha_t = asin (sin (alpha_i/180*pi)/ior_w)/pi*180;
    #declare refl_pp = -sin ((alpha_i-alpha_t)/180*pi)/sin ((alpha_i+alpha_t)/180*pi);
    #declare refl_pl = tan ((alpha_i-alpha_t)/180*pi)/tan ((alpha_i+alpha_t)/180*pi);
    #declare c_e_1 = 1-(pow (refl_pp, 2)+pow (refl_pl, 2))/2;
    #declare refl_pp1 = -sin ((alpha_t-alpha_i)/180*pi)/sin ((alpha_t+alpha_i)/180*pi);
    #declare refl_pl1 = tan ((alpha_t-alpha_i)/180*pi)/tan ((alpha_t+alpha_i)/180*pi);
    #declare c_e_2 = c_e_1*(pow (refl_pp1, 2)+pow (refl_pl1, 2))/2;

    #declare conserve_e = c_e_2*(1-(pow (refl_pp1, 2)+pow (refl_pl1, 2))/2);
    #declare spread_e = atan ((1/pi)/abs (2/cos (asin (ray_dist))-4/(ior_w*cos (asin (ray_dist/ior_w)))))/pi*2;
    #declare spread_angle = abs (2*asin (ray_dist)-4*asin (ray_dist/ior_w))/pi*180;

    #declare c_e_3 = c_e_2*(pow (refl_pp1, 2)+pow (refl_pl1, 2))/2;

    #declare conserve_e2 = c_e_3*(1-(pow (refl_pp1, 2)+pow (refl_pl1, 2))/2);
    #declare spread_e2 = atan ((1/pi)/abs (2/cos (asin (ray_dist))-6/(ior_w*cos (asin (ray_dist/ior_w)))))/pi*2;
    #declare spread_angle2 = (pi+2*asin (ray_dist)-6*asin (ray_dist/ior_w))/pi*180;
#end

// calculate imaginary rainbow samples (with wavelength for infrared light)

#macro calculate_imaginary_sample ()
    #declare light_wl = 700;
    #declare ior_w = 1.3295+(1-(light_wl-380)/320)*0.0112;
    #declare max_spread_angle = 0;
    #declare c_m_s_c = 1;
    #declare break_c_map = no;
    #while ((c_m_s_c <= rb_c_map_samples) & (break_c_map = no))
        #declare ray_dist = c_m_s_c/(rb_c_map_samples);

        calculate_rb_color ()
        #declare max_spread_angle = max (spread_angle, max_spread_angle);

        #if (spread_angle < max_spread_angle)
            #declare break_c_map = yes;;
        #end

        #declare c_m_s_c = c_m_s_c+1;
    #end
    #declare latest_c_map_entry = max_spread_angle/180;
#end

#macro calculate_imaginary_sample2 ()
    #declare light_wl = 700;
    #declare ior_w = 1.3295+(1-(light_wl-380)/320)*0.0112;
    #declare min_spread_angle = 180;
    #declare c_m_s_c = 1;
    #declare break_c_map = no;
    #while ((c_m_s_c <= rb_c_map_samples) & (break_c_map = no))
        #declare ray_dist = c_m_s_c/(rb_c_map_samples);

        calculate_rb_color ()
        #declare min_spread_angle = min (spread_angle2, min_spread_angle);

        #if (spread_angle2 > min_spread_angle)
            #declare break_c_map = yes;;
        #end

        #declare c_m_s_c = c_m_s_c+1;
    #end
    #declare latest_c_map_entry2 = 1-min_spread_angle/180;
#end

// rainbow pigment

#declare rb_pigment =
    pigment {
        average
        pigment_map {
            calculate_imaginary_sample ()
            calculate_imaginary_sample2 ()

            #declare s_s_c = 1;
            #while (s_s_c <= rb_spectrum_samples)
                #declare light_wl = 700-320*(s_s_c/(rb_spectrum_samples+1)); // [380; 700]
                #declare ior_w = 1.3295+(1-(light_wl-380)/320)*0.0112; // [1.3407; 1.3295]
                // primary rainbow
                [
                    average
                    pigment_map {
                        [
                            function {acos (-y/sqrt (x*x+y*y+z*z))/pi}
                            color_map {
                                #declare max_spread_angle = 0;
                                #declare c_m_s_c = 1;
                                #declare break_c_map = no;
                                #while ((c_m_s_c <= rb_c_map_samples) & (break_c_map = no))
                                    #declare ray_dist = c_m_s_c/rb_c_map_samples;

                                    calculate_rb_color ()
                                    #declare max_spread_angle = max (spread_angle, max_spread_angle);

                                    #if (spread_angle < max_spread_angle)
                                        #declare break_c_map = yes;
                                    #else
                                        [spread_angle/180
                                            #declare current_rb_color = convert_wl_to_rgb (light_wl)*conserve_e*spread_e/0.050783;
                                            color rgb
                                                <current_rb_color.x*rb_light_c.x,
                                                current_rb_color.y*rb_light_c.y,
                                                current_rb_color.z*rb_light_c.z>*rb_bness
                                        ]
                                    #end

                                    #declare c_m_s_c = c_m_s_c+1;
                                #end
                                [latest_c_map_entry  color rgb 0]
                                [1  color rgb 0]
                            }
                        ]
                        [
                            function {acos (-y/sqrt (x*x+y*y+z*z))/pi}
                            color_map {
                                #declare max_spread_angle = 0;
                                #declare c_m_s_c = 0;
                                #declare break_c_map = no;
                                #while ((c_m_s_c <= rb_c_map_samples) & (break_c_map = no))
                                    #declare ray_dist = 1-(c_m_s_c/rb_c_map_samples);

                                    calculate_rb_color ()
                                    #declare max_spread_angle = max (spread_angle, max_spread_angle);

                                    #if (spread_angle < max_spread_angle)
                                        #declare break_c_map = yes;
                                    #else
                                        [spread_angle/180
                                            #declare current_rb_color = convert_wl_to_rgb (light_wl)*conserve_e*spread_e/0.050783;
                                            color rgb
                                                <current_rb_color.x*rb_light_c.x,
                                                current_rb_color.y*rb_light_c.y,
                                                current_rb_color.z*rb_light_c.z>*rb_bness
                                        ]
                                    #end

                                    #declare c_m_s_c = c_m_s_c+1;
                                #end
                                [latest_c_map_entry  color rgb 0]
                                [1  color rgb 0]
                                #declare latest_c_map_entry = max_spread_angle/180;
                            }
                        ]
                    }
                ]
                // secondary rainbow
                #if (secondary_rb)
                [
                    average
                    pigment_map {
                        [
                            function {acos (-y/sqrt (x*x+y*y+z*z))/pi}
                            color_map {
                                #declare min_spread_angle = 180;
                                #declare c_m_s_c = 1;
                                #declare break_c_map = no;
                                #while ((c_m_s_c <= rb_c_map_samples) & (break_c_map = no))
                                    #declare ray_dist = c_m_s_c/rb_c_map_samples;

                                    calculate_rb_color ()
                                    #declare min_spread_angle = min (spread_angle2, min_spread_angle);

                                    #if (spread_angle2 > min_spread_angle)
                                        #declare break_c_map = yes;
                                    #else
                                        [1-spread_angle2/180
                                            #declare current_rb_color = convert_wl_to_rgb (light_wl)*conserve_e2*spread_e2/0.050783;
                                            color rgb
                                                <current_rb_color.x*rb_light_c.x,
                                                current_rb_color.y*rb_light_c.y,
                                                current_rb_color.z*rb_light_c.z>*rb_bness
                                        ]
                                    #end

                                    #declare c_m_s_c = c_m_s_c+1;
                                #end
                                [latest_c_map_entry2  color rgb 0]
                                [1  color rgb 0]
                            }
                            rotate 180*x
                        ]
                        [
                            function {acos (-y/sqrt (x*x+y*y+z*z))/pi}
                            color_map {
                                #declare min_spread_angle = 180;
                                #declare c_m_s_c = 0;
                                #declare break_c_map = no;
                                #while ((c_m_s_c <= rb_c_map_samples) & (break_c_map = no))
                                    #declare ray_dist = 1-c_m_s_c/rb_c_map_samples;

                                    calculate_rb_color ()
                                    #declare min_spread_angle = min (spread_angle2, min_spread_angle);

                                    #if (spread_angle2 > min_spread_angle)
                                        #declare break_c_map = yes;
                                    #else
                                        [1-spread_angle2/180
                                            #declare current_rb_color = convert_wl_to_rgb (light_wl)*conserve_e2*spread_e2/0.050783;
                                            color rgb
                                                <current_rb_color.x*rb_light_c.x,
                                                current_rb_color.y*rb_light_c.y,
                                                current_rb_color.z*rb_light_c.z>*rb_bness
                                        ]
                                    #end

                                    #declare c_m_s_c = c_m_s_c+1;
                                #end
                                [latest_c_map_entry2  color rgb 0]
                                [1  color rgb 0]
                                #declare latest_c_map_entry2 = 1-min_spread_angle/180;
                            }
                            rotate 180*x
                        ]
                    }
                ]
                #end
                #declare s_s_c = s_s_c+1;
            #end
        }
    }

// macros for evaluating colors at certain points in the rainbow pigment

#macro eval_pigment(pigm, vec)
    #local fn = function {pigment {pigm}}
    #local result = (fn (vec.x, vec.y, vec.z));
    result
#end

#macro mean (A)
   #local N = dimension_size (A, 1);
   #local C = 0;
   #local V = 0;
   #while (C < N)
      #local V = V+A [C];
      #local C = C+1;
   #end
   (V/N)
#end

// create rainbows

#if (rb_type = 1) // rainbow type with pov-ray's rainbow feature
// renders very fast but is not as as accurate as type #2,
// and it does not show up pigment_projected_onto_rb (because that's not possible with this rainbow type)

#declare rb_i_c = 0;
#while (rb_i_c < rb_intervals)
    rainbow {
        direction camera_location-rb_light_location
        distance rb_distance
        angle 45/rb_intervals+90*rb_i_c/rb_intervals
        width 90/rb_intervals
        color_map {
            #declare rb_c_m_c = 0;
            #while (rb_c_m_c < 255)
                #if (rb_area_angle = 0)
                    // calculate rainbow sample per interval
                    #declare eval_rb_color =
                        eval_pigment (
                            rb_pigment,
                            sin (rb_c_m_c/254*(pi/2)/rb_intervals+(pi/2)*rb_i_c/rb_intervals)*x+
                            cos (pi-rb_c_m_c/254*(pi/2)/rb_intervals-(pi/2)*rb_i_c/rb_intervals)*y
                        );
                #else
                    // calculate rainbow area light samples per interval
                    #declare eval_rb_color_arr = array [rb_area_light_samples]
                    #declare a_l_s_c = 1;
                    #while (a_l_s_c <= rb_area_light_samples)
                        #declare eval_rb_color_arr [a_l_s_c-1] =
                            eval_pigment (
                                rb_pigment,
                                sin (rb_c_m_c/254*(pi/2)/rb_intervals+(pi/2)*rb_i_c/rb_intervals+
                                    rb_area_angle*asin (a_l_s_c/(rb_area_light_samples+1)*2-1)/pi
                                )*x+
                                cos (pi-rb_c_m_c/254*(pi/2)/rb_intervals-(pi/2)*rb_i_c/rb_intervals+
                                    rb_area_angle*asin (a_l_s_c/(rb_area_light_samples+1)*2-1)/pi
                                )*y
                            );
                        #declare a_l_s_c = a_l_s_c+1;
                    #end
                    #declare eval_rb_color = mean (eval_rb_color_arr);
                #end
                #declare rb_color_avrg = (eval_rb_color.x+eval_rb_color.y+eval_rb_color.z)/3;
                #if (rb_color_avrg = 0)
                    #declare rb_color_rgbt = <rb_light_c.x, 0, 0, 1>;
                #else
                    #declare rb_color_rgbt =
                        <(eval_rb_color/rb_color_avrg).x,
                        (eval_rb_color/rb_color_avrg).y,
                        (eval_rb_color/rb_color_avrg).z,
                        1-rb_color_avrg>;
                #end
                [rb_c_m_c/254  color rgbt rb_color_rgbt]
                #declare rb_c_m_c = rb_c_m_c+1;
            #end
        }
    }
    #declare rb_i_c = rb_i_c+1;
#end

rainbow {
    direction rb_light_location-camera_location
    distance rb_distance
    angle 45
    width 90
    color_map {
        #declare rb_c_m_c = 255;
        #while (rb_c_m_c > 0)
            #if (rb_area_angle = 0)
                // calculate rainbow sample
                #declare eval_rb_color =
                    eval_pigment (rb_pigment, sin (pi/2+rb_c_m_c/254*(pi/2))*x+cos (pi/2-rb_c_m_c/254*(pi/2))*y);
            #else
                // calculate rainbow area light samples
                #declare eval_rb_color_arr = array [rb_area_light_samples]
                #declare a_l_s_c = 1;
                #while (a_l_s_c <= rb_area_light_samples)
                    #declare eval_rb_color_arr [a_l_s_c-1] =
                        eval_pigment (
                            rb_pigment,
                            sin (pi/2+rb_c_m_c/254*(pi/2)+
                                rb_area_angle*asin (a_l_s_c/(rb_area_light_samples+1)*2-1)/pi
                            )*x+
                            cos (pi/2-rb_c_m_c/254*(pi/2)+
                                rb_area_angle*asin (a_l_s_c/(rb_area_light_samples+1)*2-1)/pi
                            )*y
                        );
                    #declare a_l_s_c = a_l_s_c+1;
                #end
                #declare eval_rb_color = mean (eval_rb_color_arr);
            #end
            #declare rb_color_avrg = (eval_rb_color.x+eval_rb_color.y+eval_rb_color.z)/3;
            #if (rb_color_avrg = 0)
                #declare rb_color_rgbt = <rb_light_c.x, 0, 0, 1>;
            #else
                #declare rb_color_rgbt =
                    <(eval_rb_color/rb_color_avrg).x,
                    (eval_rb_color/rb_color_avrg).y,
                    (eval_rb_color/rb_color_avrg).z,
                    1-rb_color_avrg>;
            #end
            [(1-rb_c_m_c/254)  color rgbt rb_color_rgbt]
            #declare rb_c_m_c = rb_c_m_c-1;
        #end
    }
}

#else // (if rainbow type is 2) rainbow type with pov-ray's media feature
// renders slower but is more accurate,
// and it shows up pigment_projected_onto_rb (if declared)

// simplified rainbow densities in order to render faster (instead of using the calculated rainbow pigment directly as a density)

#declare simplified_rb_density1_arr = array [rb_intervals]
#declare simplified_rb_density2_arr = array [rb_intervals]

#declare rb_i_c = 0;
#while (rb_i_c < rb_intervals)
    #declare simplified_rb_density1_arr [rb_i_c] = // medium rainbow angle is 41.76
        density {
            function {acos (-y/sqrt (x*x+y*y+z*z))/pi}
            color_map {
                [0  color rgb 1]
                [sin ((rb_i_c/rb_intervals)*pi/2)*(41.76/180)  color rgb 1]
                #declare rb_c_m_c = 0;
                #while (rb_c_m_c < 251)
                    #declare c_m_s_loc = (rb_i_c/rb_intervals)+(rb_c_m_c/250)*(1/rb_intervals);
                    #if (rb_area_angle = 0)
                        // calculate rainbow density sample #1
                        #declare eval_rb_color =
                            eval_pigment (
                                rb_pigment,
                                sin (sin (c_m_s_loc*pi/2)*(41.76/180)*pi)*x+cos (pi-sin (c_m_s_loc*pi/2)*(41.76/180)*pi)*y
                            );
                    #else
                        // calculate rainbow area light density sample #1
                        #declare eval_rb_color_arr = array [rb_area_light_samples]
                        #declare a_l_s_c = 1;
                        #while (a_l_s_c <= rb_area_light_samples)
                            #declare eval_rb_color_arr [a_l_s_c-1] =
                                eval_pigment (
                                    rb_pigment,
                                    sin (sin (c_m_s_loc*pi/2)*(41.76/180)*pi+
                                        rb_area_angle*asin (a_l_s_c/(rb_area_light_samples+1)*2-1)/pi
                                    )*x+
                                    cos (pi-sin (c_m_s_loc*pi/2)*(41.76/180)*pi+
                                        rb_area_angle*asin (a_l_s_c/(rb_area_light_samples+1)*2-1)/pi
                                    )*y
                                );
                            #declare a_l_s_c = a_l_s_c+1;
                        #end
                    #declare eval_rb_color = mean (eval_rb_color_arr);
                    #end
                    [sin (c_m_s_loc*pi/2)*(41.76/180)  color rgb <eval_rb_color.x, eval_rb_color.y, eval_rb_color.z>]
                    #declare rb_c_m_c = rb_c_m_c+1;
                #end
                [sin (c_m_s_loc*pi/2)*(41.76/180)  color rgb 1]
                [1  color rgb 1]
            }
        }
    #declare rb_i_c = rb_i_c+1;
#end

#declare rb_i_c = 0;
#while (rb_i_c < rb_intervals)
    #declare simplified_rb_density2_arr [rb_i_c] =
        density {
            function {acos (-y/sqrt (x*x+y*y+z*z))/pi}
            color_map {
                [0  color rgb 1]
                [(41.76/180)+(1-sin (pi/2+(rb_i_c/rb_intervals)*pi/2))*((180-41.76)/180)  color rgb 1]
                #declare rb_c_m_c = 0;
                #while (rb_c_m_c < 251)
                    #declare c_m_s_loc = (rb_i_c/rb_intervals)+(rb_c_m_c/250)*(1/rb_intervals);
                    #if (rb_area_angle = 0)
                        // calculate rainbow density sample #2
                        #declare eval_rb_color =
                            eval_pigment (
                                rb_pigment,
                                sin (((41.76/180)+(1-sin (pi/2+c_m_s_loc*pi/2))*((180-41.76)/180))*pi)*x+
                                cos (pi-((41.76/180)+(1-sin (pi/2+c_m_s_loc*pi/2))*((180-41.76)/180))*pi)*y
                            );
                    #else
                        // calculate rainbow area light density sample #2
                        #declare eval_rb_color_arr = array [rb_area_light_samples]
                        #declare a_l_s_c = 1;
                        #while (a_l_s_c <= rb_area_light_samples)
                            #declare eval_rb_color_arr [a_l_s_c-1] =
                                eval_pigment (
                                    rb_pigment,
                                    sin (((41.76/180)+(1-sin (pi/2+c_m_s_loc*pi/2))*((180-41.76)/180))*pi+
                                        rb_area_angle*asin (a_l_s_c/(rb_area_light_samples+1)*2-1)/pi
                                    )*x+
                                    cos (pi-((41.76/180)+(1-sin (pi/2+c_m_s_loc*pi/2))*((180-41.76)/180))*pi+
                                        rb_area_angle*asin (a_l_s_c/(rb_area_light_samples+1)*2-1)/pi
                                    )*y
                                );
                            #declare a_l_s_c = a_l_s_c+1;
                        #end
                    #declare eval_rb_color = mean (eval_rb_color_arr);
                    #end
                    [(41.76/180)+(1-sin (pi/2+c_m_s_loc*pi/2))*((180-41.76)/180)
                        color rgb <eval_rb_color.x, eval_rb_color.y, eval_rb_color.z>
                    ]
                    #declare rb_c_m_c = rb_c_m_c+1;
                #end
                [(41.76/180)+(1-sin (pi/2+c_m_s_loc*pi/2))*((180-41.76)/180)  color rgb 1]
                [1  color rgb 1]
            }
        }
    #declare rb_i_c = rb_i_c+1;
#end

// reorientation rainbow macros

#macro VPerp_To_Vector (v0)
   #if (vlength(v0) = 0)
      #local vN = <0, 0, 0>;
   #else
      #local Dm = min (abs (v0.x), abs (v0.y), abs (v0.z));
      #if (abs (v0.z) = Dm)
         #local vN = vnormalize (vcross (v0, z));
      #else
         #if (abs (v0.y) = Dm)
            #local vN = vnormalize (vcross (v0, y));
         #else
            #local vN = vnormalize (vcross (v0, x));
         #end
      #end
   #end
   vN
#end

#macro orient_rb_shear (A, B, C)
   transform {
      matrix < A.x, A.y, A.z,
             B.x, B.y, B.z,
             C.x, C.y, C.z,
             0, 0, 0>
   }
#end

#macro orient_rb (YAxis)
   #local Y = vnormalize (YAxis);
   #local X = VPerp_To_Vector (Y);
   #local Z = vcross (X, Y);
   orient_rb_shear (X, Y, Z)
#end

// rainbow density correction ("fog-like")

#declare density_correction_pattern =
    density {
        function {(1-exp (-sqrt (x*x+y*y+z*z)/rb_distance))/sqrt (x*x+y*y+z*z)}
        color_map {
            [0  color rgb 0]
            [1  color rgb 0.085]
        }
    }

// rainbow sphere

union {
    sphere {0, rb_sphere_radius}
    sphere {0, rb_sphere_radius+0.00001*rb_sphere_radius}
    pigment {color rgbt 1}
    hollow
    interior {
        media {
            emission 1
            intervals rb_media_intervals
            #ifdef (pigment_projected_onto_rb)
                #declare p_p_o_rb_func = function {pigment {pigment_projected_onto_rb}}
                density {
                    function {p_p_o_rb_func (x/sqrt (x*x+y*y+z*z), y/sqrt (x*x+y*y+z*z), z/sqrt (x*x+y*y+z*z)).red}
                    color_map {
                        [0  color rgb y+z]
                        [1  color rgb 1]
                    }
                }
                density {
                    function {p_p_o_rb_func (x/sqrt (x*x+y*y+z*z), y/sqrt (x*x+y*y+z*z), z/sqrt (x*x+y*y+z*z)).green}
                    color_map {
                        [0  color rgb x+z]
                        [1  color rgb 1]
                    }
                }
                density {
                    function {p_p_o_rb_func (x/sqrt (x*x+y*y+z*z), y/sqrt (x*x+y*y+z*z), z/sqrt (x*x+y*y+z*z)).blue}
                    color_map {
                        [0  color rgb x+y]
                        [1  color rgb 1]
                    }
                }
            #end
            #declare rb_i_c = 0;
            #while (rb_i_c < rb_intervals)
                density {simplified_rb_density1_arr [rb_i_c]}
                density {simplified_rb_density2_arr [rb_i_c]}
                #declare rb_i_c = rb_i_c+1;
            #end
            density {density_correction_pattern}
            #declare rb_camera_hole =
                function {
                    pigment {
                        object {
                            sphere {0, 0.00001}
                            color rgb 1
                            color rgb 0
                        }
                    }
                }
            density {
                function {rb_camera_hole (x, y, z).gray}
                color_map {
                    [0  color rgb 0]
                    [1  color rgb 1]
                }
            }
        }
    }
    orient_rb (rb_light_location-camera_location)
    translate camera_location-(1e-10)*vnormalize (rb_light_location-camera_location)
}

#end // end of putting rainbow type #1 or #2 into the scene
