//  File: simballs.inc
//  Description: Povray rigid ball dynamics simulator
//  Last updated: 2003.10.29
//  Created by: Alain Ducharme
//
//  Some of the limitations:
//  - Bug: when balls get stuck, angular momentum not handled properly
//  - Center of mass is fixed at the center of the balls
//  - Perfect spheres, no surface deformation/elasticity
//  - No molecular level dynamics ;-)
//
//  If you expand/fix/enhance, please let me know: Alain_Ducharme@hotmail.com
//

#ifndef(SIMBALL_INC_TEMP)
#declare SIMBALL_INC_TEMP = version;
#version 3.5;

#include "tdmedres.inc"  // Default: SetTraceRes(1)

#declare SimTime = 1;         // Default simulation time in seconds
#declare SimStep = 1/200;     // Default simSeconds between simulation steps
#declare FrameStep = clock_delta*SimTime; // SimSeconds between frames
#declare G=-9.81;             // Reference gravitational acceleration (where Pov units = meters)

#macro SetNumBalls(TB)
  #declare NumBalls = TB;
  #declare Br = array[NumBalls] // Ball Radius
  #declare Bm = array[NumBalls] // Ball Mass
  #declare Bp = array[NumBalls] // Ball Position
  #declare Bv = array[NumBalls] // Ball Velocity
  #declare Bw = array[NumBalls] // Ball Angular velocity in circles
  #declare Bo = array[NumBalls] // Ball Orientation, quaternion, must be normalized at all times
  #declare Bg = array[NumBalls] // Ball gravity  (if > 0, fake lighter than air)
  #declare Bc = array[NumBalls] // Ball Coefficient of restitution (bounciness): 0 (none) to 1 (perfectly elastic)
  #declare Bu = array[NumBalls] // Ball Coefficient of friction (slip): 0 (no friction) to 1 (very high friction)
#end

#macro LoadBalls()
  // Load previous frame's simball data from file
  #if (clock > 0)
    #fopen SBFile "sb.dat" read
    #read (SBFile,Bif)  // Balls in file
    #local I = 0;
    #while (I < Bif)
      #read (SBFile,Br[I],Bm[I],Bg[I],Bc[I],Bu[I],Bp[I],Bv[I],Bw[I],RA)
      #declare Bo[I] = RA; // #read can't handle initialized 4D vector (must be initialized due to array)
      #undef RA
      #local I = I + 1;
    #end
    #fclose SBFile
  #end
#end

#macro SetCN(Collnorm)
  #declare CY = Collnorm;
  #local S = min(abs(CY.x), abs(CY.y), abs(CY.z));
  #if (S = abs(CY.y))
    #declare CX = vnormalize(<-CY.z,0,CY.x>);
  #else
    #if (S = abs(CY.x))
      #declare CX = vnormalize(<0,-CY.z,CY.y>);
    #else
      #declare CX = vnormalize(<-CY.y,CY.x,0>);
    #end
  #end
  #declare CZ = vcross(CX, CY);
#end

#macro VYToCN(Vec)
  <vdot(Vec, CX),vdot(Vec, CY),vdot(Vec, CZ)>
#end

#macro VYFromCN(Vec)
  #local OV = Vec.x*CX + Vec.y*CY + Vec.z*CZ;
  <vdot(OV, x),vdot(OV, y),vdot(OV, z)>
#end

#macro IC(V,W,CR,CF)
  #local CF = min(sqrt(CF*Bu[I])*abs(V.y),1);
  #local WzVx = (W.z+V.x)/2*CF;
  #local WxVz = (W.x-V.z)/2*CF;
  #declare W = <W.x-WxVz,W.y,W.z-WzVx>;
  #local CR = sqrt(CR*Bc[I]);
  #declare V = <V.x-WzVx,-(V.y*CR),V.z+WxVz>;
#end

#macro ImmoYColl(CR,CF,P)
  // Collision with immovable object with a normal of y
  #if (vdot(Bv[I],y) < 0)
    #declare Bp[I] = P;
    IC(Bv[I],Bw[I],CR,CF)
  #end
#end

// Could make a MovColl by adding a velocity vector for the moving object...

#macro ImmoColl(Collnorm,CR,CF,P)
  #if (vdot(Bv[I],Collnorm) < 0)  //...or comment this condition for quick and dirty (may cause stickyness)
    #declare Bp[I] = P;
    SetCN(Collnorm)
    #declare V = VYToCN(Bv[I]);
    #declare W = VYToCN(Bw[I]);
    IC(V,W,CR,CF)
    #declare Bv[I] = VYFromCN(V);
    #declare Bw[I] = VYFromCN(W);
  #end  
#end

#macro SetTraceRes(R)
  // R = 0 to 2 for now
  #switch (R)
    #case (0)
      #include "tdlowres.inc"
      #break
    #case (1)
      #include "tdmedres.inc"
      #break
    #case (2)
      #include "tdhighres.inc"
      #break
  #end
#end

#macro TraceColl(Obj,CR,CF)
  #local TP = 0;
  #while (TP < NumTD)
    #local TN = <0,0,0>;
    #local RT = trace(Obj, Bp[I], Around[TP], TN);
    #if (vlength(TN)!=0 )
      #if (vlength(Bp[I]-RT) < Br[I])
        ImmoColl(TN,CR,CF,RT - Br[I]*Around[TP])
      #end
    #end
    #declare TP=TP+1;
  #end
#end  

#macro BTraceColl(Obj,CR,CF)
  #local COmin = min_extent(Obj)-Br[I];
  #local COmax = max_extent(Obj)+Br[I];
  #if ((Bp[I].x > COmin.x & Bp[I].y > COmin.y & Bp[I].z > COmin.z) & (Bp[I].x < COmax.x & Bp[I].y < COmax.y & Bp[I].z < COmax.z))
    TraceColl(Obj,CR,CF)
  #end
#end  

#macro RNTraceColl(Obj,CR,CF)
// Special case for Povray objects with reversed normals
  #local COmin = min_extent(Obj)-Br[I];
  #local COmax = max_extent(Obj)+Br[I];
  #if ((Bp[I].x > COmin.x & Bp[I].y > COmin.y & Bp[I].z > COmin.z) & (Bp[I].x < COmax.x & Bp[I].y < COmax.y & Bp[I].z < COmax.z))
    #local TP = 0;
    #while (TP < NumTD)
      #local TN = <0,0,0>;
      #local RT = trace(Obj, Bp[I], Around[TP], TN);
      #if (vlength(TN)!=0 )
        #if (vlength(Bp[I]-RT) < Br[I])
          ImmoColl(-TN,CR,CF,RT - Br[I]*Around[TP])
        #end
      #end
      #declare TP=TP+1;
    #end
  #end
#end

#macro TraceText(Obj,CR,CF,Z)
// Special case for Povray text object bug: side normals are outside in
// Z is normal vector representing text object's local Z axis in world coordinates (direction it's facing)
  #local COmin = min_extent(Obj)-Br[I];
  #local COmax = max_extent(Obj)+Br[I];
  #if ((Bp[I].x > COmin.x & Bp[I].y > COmin.y & Bp[I].z > COmin.z) & (Bp[I].x < COmax.x & Bp[I].y < COmax.y & Bp[I].z < COmax.z))
    #local TP = 0;
    #while (TP < NumTD)
      #local TN = <0,0,0>;
      #local RT = trace(Obj, Bp[I], Around[TP], TN);
      #if (vlength(TN)!=0 )
        #if (vlength(Bp[I]-RT) < Br[I])
          #if (abs(vdot(Z,TN)) < 0.99999)
            #declare TN = -TN;
          #end
          ImmoColl(TN,CR,CF,RT - Br[I]*Around[TP])
        #end
      #end
      #declare TP=TP+1;
    #end
  #end
#end

#macro VelPos()
  // Update velocity and position
  #declare Bv[I] = <Bv[I].x, Bv[I].y+(Bg[I]*SimStep), Bv[I].z>;
  #declare Bp[I] = Bp[I] + Bv[I]*SimStep;
#end  

#macro B2B()
  // Collisions between balls
  #local J = 0;
  #while (J < NumBalls)
    #if (J != I)
      #local VDiff = Bp[I]-Bp[J];
      #local Dist = vlength(VDiff);
      #if (Dist < Br[I]+Br[J]) // Ball-Ball collision

        #if (vdot(Bv[I],Bv[J]) < 0 | vlength(Bv[I]) >= vlength(Bv[J])) // Vel compare

          #local VDiff = vnormalize(VDiff);
          SetCN(VDiff)

          #declare Bp[I] = Bp[J] + (Br[I]+Br[J]) * VDiff;

          #local IV = VYToCN(Bv[I]);
          #local JV = VYToCN(Bv[J]);
          #local IW = VYToCN(Bw[I]);
          #local JW = VYToCN(Bw[J]);

          #local CR = sqrt(Bc[I]*Bc[J]);  // Coefficient of restitution
          #if ( Bm[I] = Bm[J])
            #local IVy = JV.y*CR;
            #local JVy = IV.y*CR;
          #else  
            #local IJM = Bm[I]+Bm[J];
            #local IVy = (IV.y*(Bm[I]-Bm[J])/IJM+JV.y*(2*Bm[J])/IJM)*CR;
            #local JVy = (IV.y*(2*Bm[I])/IJM+JV.y*(Bm[J]-Bm[I])/IJM)*CR;
          #end  

          #local CF = min(sqrt(Bu[I]*Bu[J])*abs(IV.y-JV.y),1); // Coefficient of friction
          #local WzVx = ((IV.x+IW.z)-(JV.x-JW.z))/4*CF;
          #local WxVz = ((IV.z+IW.x)-(JV.z-JW.x))/4*CF;

          #local IV = <(IV.x-WzVx),IVy,(IV.z-WxVz)>;
          #local JV = <(JV.x+WzVx),JVy,(JV.z+WxVz)>;
          #local IW = <(IW.x+WxVz),IW.y,(IW.z-WzVx)>;
          #local JW = <(JW.x-WxVz),JW.y,(JW.z-WzVx)>;

          #declare Bw[I] = VYFromCN(IW);
          #declare Bw[J] = VYFromCN(JW);

          #declare Bv[I] = VYFromCN(IV);
          #declare Bv[J] = VYFromCN(JV);

        #end  // Vel compare
      #end // Ball-Ball collision
    #end // if (J != I)
    #local J = J + 1;
  #end // while (J < NumBalls)
#end

// Need higher precision than default
#macro VstrC(V)
    concat("<",vstr(3,V,",",0,16),">,")
#end
#macro V4strC(V)
    concat("<",vstr(4,V,",",0,16),">,")
#end

#macro OpenSBFile()
  #fopen SBFile "sb.dat" write
  #write (SBFile,NumBalls,",\n")
#end

#macro WriteBall()
  #write (SBFile,Br[I],",",Bm[I],",",Bg[I],",",Bc[I],",",Bu[I],",",VstrC(Bp[I]),VstrC(Bv[I]),VstrC(Bw[I]),V4strC(Bo[I]),"\n")
#end

#macro CloseSBFile()
  #fclose SBFile
#end

#version SIMBALL_INC_TEMP;
#end //simball.inc
