// Persistence of Vision Ray Tracer Scene Description File
// File: SBBasic.pov
// Vers: 3.5
// Desc: SimBall example file, mix of collisions
// Date: 2003.10.29
// Auth: Alain Ducharme
//
// Command line for animation:
// +kff300

#version 3.5;

#include "quaternions.inc"
#include "simballs.inc"

global_settings {
  assumed_gamma 1.0
}

default { texture { pigment { color rgb 1 } finish { ambient 0.1 specular 0 roughness 1 } } }

camera {
  location  <0, 2, -6.5>
  angle     53
  right     x*image_width/image_height
  look_at   <0.0, 1.1, 0.0>
}

light_source { <-1,2,-2.25> color rgb <1, 1, 1> fade_distance 3 fade_power 2 
}

#declare S1 = seed(42);

#declare SimTime = 10;          // Simulation time in seconds
#declare SimStep = 1/200;       // Seconds between simulation steps
#declare FrameStep = clock_delta*SimTime;

#if (clock < 0.70)
  SetNumBalls(3+3*int(clock*(1/0.70)*3))
#else
  SetNumBalls(12)
#end  

#declare Room = <6/2,3,5/2>;  // Room dimensions (xz/2 for convenience)
#declare Wall = 0.1;          // Wall thickness

// Initialize Balls
#declare I = 0;
#while (I < NumBalls)
  #declare RM = 0.1+0.2*rand(S1);  // Radius and Mass
  #declare Bg[I] = G;         // Gravity
  #declare Br[I] = RM;        // Radius
  #declare Bm[I] = RM;        // Mass
  #switch (mod(I,3)) 
    #case (0)
      #declare Bp[I] = <-Room.x-RM*2,1+RM,0>; // Position
      #declare Bv[I] = <1+2*rand(S1),0,0>;    // Velocity
      #declare Bw[I] = <0,0,-1>;  // Angular Velocity in circles
    #break
    #case (1)
      #declare Bp[I] = <0,Room.y+RM*2,0>; // Position
      #declare Bv[I] = <0,0,0>;   // Velocity
      #declare Bw[I] = <0,5,0>;   // Angular Velocity in circles
    #break
    #case (2)
      #declare Bp[I] = <Room.x+RM*2,1.45-RM,0>; // Position
      #declare Bv[I] = <-4-2*rand(S1),0,0>;     // Velocity
      #declare Bw[I] = <0,0,3>;   // Angular Velocity in circles
    #break
  #end
  #declare Bo[I] = <0,0,0,1>; // Orientation: quaternion, must be normalized
  #declare Bc[I] = 0.8;       // Coefficient of restitution (bounciness)
  #declare Bu[I] = 0.8;       // Coefficient of friction (traction)
  #declare I = I + 1;
#end

LoadBalls()

// Declare & Draw collision objects
#declare CanStick = sor{
	8,
	<0.538286,-0.013457>,
	<0.538286,-0.013457>,
	<0.269143,0.255686>,
	<0.134571,0.524828>,
	<0.067286,0.592114>,
	<0.134571,0.726686>,
	<0.040371,1.002557>,
	<0.000000,1.063114>
  scale <1,0.5,1>
}
CanStick

box { <-Room.x,-Wall,-Room.z>, <Room.x,0,Room.z> }                  // Floor
difference {
  union {
    box { <-Room.x,Room.y,-Room.z>, <Room.x,Room.y+Wall,Room.z> }       // Ceiling
    cylinder { <0,Room.y+1,0>, <0,Room.y-0.05,0>, 0.5 }
  }
  cylinder { <0,Room.y+1,0>, <0,Room.y-0.1,0>, 0.45 }
}  
//box { <-Room.x,0,-Room.z>, <Room.x,Room.y,-Room.z-Wall> no_image }  // Front
box { <-Room.x,0,Room.z>, <Room.x,Room.y,Room.z+Wall> }             // Back
difference {
  union {
    box { <-Room.x,0,-Room.z>, <-Room.x-Wall,Room.y,Room.z> }           // Left
    cylinder { <-Room.x-1,1.2,0>, <-Room.x+0.05,1.2,0>, 0.5 }
  }  
  cylinder { <-Room.x-1,1.2,0>, <-Room.x+0.1,1.2,0>, 0.45 }
}  
difference {
  union {
  box { <Room.x,0,-Room.z>, <Room.x+Wall,Room.y,Room.z> }             // Right
  cylinder { <Room.x+1,1.2,0>, <Room.x-0.05,1.2,0>, 0.5 }
  }  
  cylinder { <Room.x+1,1.2,0>, <Room.x-0.1,1.2,0>, 0.45 }
}  

// Raised floor with spherical hole
#declare TF = 0.2;
#declare CSp = <0,4.95,0>;
#declare CSr = 5;
difference {
  box { <-Room.x,0,-Room.z>, <Room.x,TF,Room.z> } // Raised Floor
  sphere { CSp, CSr }                             // Spherical Hole
}  
#declare CSl = CSr*cos(asin((CSp.y-TF)/CSr));

// Ramp
difference {
  box { <-Room.x-1,TF,-Room.z>, <0,1,Room.z> }
  plane { -x-y, Room.x/3}
}
#declare RO = Room.x/3/sin(pi/4); // Ramp Offset
#declare RN = vnormalize(x+y);    // Ramp Normal

// Sim Loop
#declare Time = 0;
#while (Time < FrameStep)
  #declare I = 0;
  #while (I < NumBalls)
    VelPos()
    B2B()

    // Start Collisions with Immovable objects

    #if (Bp[I].y-Br[I] < TF)  // Top Floor
      // Not handled: sphere cut-out outer edge
      #if (vlength(Bp[I]-<0,TF,0>) < CSl )  // Sphere cut-out
        #if (vlength(Bp[I]-CSp) > CSr-Br[I])
          ImmoColl(vnormalize(CSp-Bp[I]),0.8,0.5,Bp[I]) //(CSr-Br[I])*vnormalize(Bp[I]-CSp)+CSp)
        #end
      #else
        ImmoYColl(0.8,0.5,<Bp[I].x,TF+Br[I],Bp[I].z>)
      #end
      #if (Bp[I].y < Br[I]) // Floor
        ImmoYColl(0.8,0.5,<Bp[I].x,Br[I],Bp[I].z>)
      #end
    #else
      #if (Bp[I].y > (Room.y-Br[I])) // Ceiling
        #declare Bv[I] = -Bv[I];
        ImmoYColl(0.8,0.5,<Bp[I].x,Room.y-Br[I],Bp[I].z>)
        #declare Bv[I] = -Bv[I];
      #end
    #end

    #if (Bp[I].x > Room.x-Br[I]) // Right wall
      ImmoColl(-x,0.8,0.5,<Room.x-Br[I],Bp[I].y,Bp[I].z>)
    #end

    #if (Bp[I].x-Br[I] < -RO-TF) // Bottom of ramp
      #if (Bp[I].x-Br[I] < -RO-1)  // Top of ramp
      // Not handled: edge at top of ramp
        #if (Bp[I].y-Br[I] < 1) // Floor at Ramp top
          ImmoYColl(0.8,0.5,<Bp[I].x,1+Br[I],Bp[I].z>)
        #end
        #if (Bp[I].x < -Room.x+Br[I]) // Left wall
          ImmoColl(x,0.8,0.5,<-Room.x+Br[I],Bp[I].y,Bp[I].z>)
        #end
      #else
        #if (vdot(RN,Bp[I]) < -Room.x/3+Br[I]) // Ramp
          ImmoColl(RN,0.8,0.5,Bp[I])
        #end
      #end
    #end  

    #if (Bp[I].z > Room.z-Br[I]) // Front
      ImmoColl(-z,0.8,0.5,<Bp[I].x,Bp[I].y,Room.z-Br[I]>)
    #end

    #if (Bp[I].z < -Room.z+Br[I]) // Back
      ImmoColl(z,0.8,0.5,<Bp[I].x,Bp[I].y,-Room.z+Br[I]>)
    #end

    BTraceColl(CanStick,0.8,0.5)

    // Increment ball's orientation
//  QInc(Bo[I],Bw[I]/Br[I],SimStep)

    #declare I = I + 1;
  #end // #while (I < NumBalls)
  #declare Time = Time + SimStep;
#end // #while (Time < FrameStep)
// End of Sim Loop

// Save ball data and draw spheres
OpenSBFile()
#declare I = 0;
#while (I < NumBalls)

  // Increment ball's orientation
  // This should be in the Sim loop but in most cases it will look ok here
  // It saves on computations here; use it in Sim loop instead for more precision
  QInc(Bo[I],Bw[I]/Br[I],FrameStep)
                   
  // Draw sphere
  sphere { <0 0 0> Br[I]
    texture {
      pigment { uv_mapping checker color rgb <1, 1, 1>, color rgb <1,0,0> scale 0.15 }
      finish { specular 0.9 roughness 0.02 }
    }
//    scale Br[I]   // Could set radius to 1 above and scale here depending on how you want to handle texture
    // Set orientation
    QMatrix(Bo[I])
    translate Bp[I]
  }

  // Write ball data
  WriteBall()

  #declare I = I + 1;
#end
CloseSBFile()
