// Persistence of Vision Ray Tracer Scene Description File
// File: sbmb.pov
// Vers: 3.5
// Desc: SimBalls example file, Marbles in a Bowl
// Date: 2003.10.29
// Auth: Alain Ducharme
//
// Command line for animation:
// +kff180

// Units: Decimeters

#version 3.5;

#include "colors.inc"
#include "simballs.inc"
#include "quaternions.inc"

#declare Glass=off;
#declare Photons=off;

global_settings {
  assumed_gamma 1.0
  max_trace_level 5
  #if (Photons)
    photons {
      spacing 0.02
    }
  #end
}

default { texture { pigment { color rgb 1 } finish { ambient 0.1 specular 0 roughness 1 } } }

camera {
  location  <0.0, 1.2, -2.5>
  angle     53
  right     x*image_width/image_height
  look_at   <0.0, 0.5,  0.0>
}

light_source {
  <0,2,0>
  color rgb <1,1,1>
  fade_distance 2
  fade_power 3
  photons {
    refraction on
    reflection on
  }
}

#declare S1 = seed(42);

#declare SimTime = 3;             // Simulation time in seconds
#declare SimStep = 1/200;         // Seconds between simulation steps
#declare FrameStep = clock_delta*SimTime;

#if (clock < 0.70)
  SetNumBalls(1+int(clock*(1/0.70)*6))
#else
  SetNumBalls(6)
#end  

// Set initial ball parameters
#declare I = 0;
#while (I < NumBalls)
  #declare Bg[I] = -98.1;           // Gravity
  #declare RM = 0.1+0.15*rand(S1);  // Radius & Mass
  #declare Br[I] = RM;              // Radius
  #declare Bm[I] = RM;              // Mass
  #declare Bp[I] = <-1.75,0.5,0>;   // Initial Position  
  #declare VxWz = 4+5*rand(S1);
  #declare VzWx = -1+2*rand(S1); 
  #declare Bv[I] = <VxWz,10,VzWx>;  // Initial Velocity
  #declare Bw[I] = <VzWx,0,-VxWz>;  // Initial Angular Velocity in circles
  #declare Bo[I] = <0,0,0,1>;       // Orientation: quaternion, must be normalized
  #declare Bc[I] = 0.9;             // Coefficient of restitution (bounciness)
  #declare Bu[I] = 0.5;             // Coefficient of friction (slipperiness)
  #declare I = I + 1;
#end

LoadBalls()

// Declare collision objects
#declare Bowl = lathe{
	cubic_spline
	20,
	<0.000000,0.082805>,
	<0.000000,0.109516>,
	<0.080134,0.109516>,
	<0.267112,0.109516>,
	<0.534224,0.162938>,
	<0.721202,0.323205>,
	<0.801336,0.536895>,
	<0.828047,0.697162>,
	<0.881469,0.777295>,
	<0.988314,0.777295>,
	<0.988314,0.750584>,
	<0.908180,0.750584>,
	<0.854758,0.697162>,
	<0.828047,0.536895>,
	<0.747913,0.296494>,
	<0.534224,0.109516>,
	<0.267112,0.056093>,
	<0.454090,0.002671>,
	<0.000000,0.002671>,
	<0.000000,-0.024040>
  #if (Glass)
    material {
      texture {
        pigment {rgbt 1}
        finish {
          ambient 0.0
          diffuse 0.05
          specular 0.6
          roughness 0.0005
          reflection {
            0.1, 1.0
            fresnel on
          }
          conserve_energy
        }
      }
      interior {
        ior 1.5
        fade_power 1001
        fade_distance 0.9
        fade_color <0.5,0.8,0.6>
      }
    }
    photons {
      target 1.0
      refraction on
      reflection on
    }
  #end
  scale <1,0.75,1>
}

// Draw collision object(s)
Bowl
plane { y,0 finish { ambient 0 } }

// Sim Loop
#declare Time = 0;
#while (Time < FrameStep)
  #declare I = 0;
  #while (I < NumBalls)
    VelPos()  // Update ball velocity & position
    B2B()     // Detect & process ball to ball collisions

    // Collisions with Immovable objects

    // Lathe has normals outside in, so reverse normal trace collision
    RNTraceColl(Bowl,0.8,0.5)

    #if (Bp[I].y < (Br[I])) // Floor
      ImmoYColl(0.8,0.5,<Bp[I].x,Br[I],Bp[I].z>)
    #end

    // Increment ball's orientation
//    QInc(Bo[I],Bw[I]/Br[I],SimStep)

    #declare I = I + 1;
  #end // #while (I < NumBalls)
  #declare Time = Time + SimStep;
#end // #while (Time < FrameStep)
// End of Sim Loop

#declare Colors = array[10] {
  Blue,Orange,Green,OrangeRed,SlateBlue,Red,DarkPurple,NewMidnightBlue,Yellow,LimeGreen
}

// Save ball data and draw spheres
OpenSBFile()
#declare I = 0;
#while (I < NumBalls)

  // Increment ball's orientation
  // This should be in the Sim loop but in most cases it will look ok here
  // It saves on computations here; use it in Sim loop (with SimStep) for more exactness
  QInc(Bo[I],Bw[I]/Br[I],FrameStep)

  // Draw sphere
  sphere { <0 0 0> Br[I]
    #if (Glass)
      material {
        texture {
          pigment { color rgbt 1 }
          finish {
            ambient 0
            diffuse 0
            reflection { 0.1, 1.0 fresnel on }
            specular 0.5
            roughness 0.05
            conserve_energy
          }
          normal {
            agate 0.1
          }
        }
        interior {
          ior 1.5
          fade_power 1001
          fade_distance 0.2
          fade_color Colors[mod(I,10)]
        }
      }
      photons {
        target 1.0
        refraction on
        reflection on
      }
    #else
      texture {
        pigment { uv_mapping checker color rgb <1, 1, 1>, color Colors[mod(I,10)] scale 0.15 }
        finish { specular 0.9 roughness 0.02 }
      }
    #end
    // Set orientation
    QMatrix(Bo[I])
    translate Bp[I]
  }

  // Write ball data
  WriteBall()

  #declare I = I + 1;
#end
CloseSBFile()
