// Persistence of Vision Ray Tracer Include File
// File: param.inc
// Vers: 3.5
// Desc: An include file to generate a mesh2 of smooth_triangles 
//       with uv_vectors from a set of parametric functions. 
// Date: 2001-11-03
// Auth: Ingo Janssen

/*======
Parametric(Urange, Vrange, Iter_U, Iter_V, FileName): Builds a parametric surface 
           out of three given functions (Fx, Fy, Fz). These functions have to be
           declared before the macro is invoked.
Urange   : A 2-D vector that gives the boundaries of u.
Vrange   : A 2-D vector that gives the boundaries of v. These are the range
           within whitch the surface is calculated. It also is the range to
           whitch the uv-coordinates are applied. The area of the texture from
           the x-y-plane that is mapped to the object is defined by the
           rectangle <u min,v min>, <u max,v max>.
Iter_U   : Sets the resolution of the mesh in the u range.
Iter_V   : Sets the resolution of the mesh in the v range.
FileName : The name of the file to whitch the mesh will be written. If is an
           empty string (""), no file will be written. If a file name is given,
           the macro will first check if it already exists. If that is so, it
           will expect a mesh2 with the name "Surface" and try to parse the
           existing file.
Use :
   
         #include "param.inc"

Set up a set of parametric functions, Fx, Fy and Fz. This can be done using
the function(){} statement or using #macro().

A unit sphere may look like this:

         #declare Fx = function(U,V){sin(U)*cos(V)}
         #declare Fy = function(U,V){cos(U)}
         #declare Fz = function(U,V){sin(U)*sin(V)}
         
or using macros like this:

         #macro Fx(U,V) (sin(U)*cos(V)) #end
         #macro Fy(U,V) (cos(U))        #end
         #macro Fz(U,V) (sin(U)*sin(V)) #end 

Using functions is much faster. Using macros makes it possible to use float- and
vector functions that are not supported in function(){}, for example trace().

Now the macro can be used in an object:
         object {
            parametric (<FromU(0), pi>, <0, 2*pi> 20, 40 ""}
            pigment{..}
         }

Note the use of the macro FromU() to start the u-range with. This means that the
range is specified as 0 < u <= pi. The minimum will get close to, but not reach
the value. The three other macros for this are ToU(), FromV() and ToV(). They
have to be used when the equations can not be solved for the boundaries.

The resolution of the mesh is set with Iter_U and Iter_V, the amount of steps by
which the u and v ranges are divided. The total amount of triangles calculated
will be Iter_U*Iter_V*2.

If you want to save the resulting mesh to a file, enter a filename in the macro.

Trouble shooting:

** Warning: Normal vector in mesh2 cannot be zero - changing it to <1,0,0>
** Parse Error: Normal vector in mesh smooth_triangle cannot be zero.
-- Check if the u and v boundaries are specified the right way? Should all
values be included, or should FromU() ToV() etc. be used?

** Parse Error: Expected 'numeric expression', undeclared identifier '...' found
instead.
-- Actually the same error as above, but it occurred while reading a mesh from
file, written while using erroneous uv boundaries. Delete the file.

** Parse Error: No matching } in 'smooth_triangle', camera found instead
--This happens while reading an unfinished mesh. Was parsing stopped while
writing the mesh? Delete the file.
*/

#version 3.5;
#include "makemesh.inc"

#declare EPS=(1e-12);
#declare EPSNorm=(1e-14);

#declare __FU=0; #declare __TU=0;
#declare __FV=0; #declare __TV=0;
#macro FromU(N)#local N=(N+EPS); #declare __FU=1; (N) #end
#macro ToU(N)  #local N=(N-EPS); #declare __TU=1; (N) #end
#macro FromV(N)#local N=(N+EPS); #declare __FV=1; (N) #end
#macro ToV(N)  #local N=(N-EPS); #declare __TV=1; (N) #end

#macro CalcNormal(P)                 //calculate the normal of a vertex,
   #local Un=U+(iU/10);              //based on two nearby points.
   #local Vn=V+(iV/10);
   #if(__TU&Un>=Umax)
      #local Un=U+EPSNorm;
   #end
   #if(__TV&Vn>=Vmax)         
      #local Vn=V+EPSNorm;
   #end
   #local N2=<(Fx(Un,V)),(Fy(Un,V)),(Fz(Un,V))>;
   #local N3=<(Fx(U,Vn)),(Fy(U,Vn)),(Fz(U,Vn))>;
   #local A=(N2-P);
   #local B=(N3-P);      
   #declare Norm=vnormalize(vcross(A,B));   
   (Norm)
#end

#macro Parametric(Urange, Vrange, Iter_U, Iter_V, FileName)
   #if(strlen(FileName)>0)
      #if(file_exists(FileName))
         #debug concat("\n Parsing mesh2 from file: ", FileName, "\n")
         #local Build=0;
         #include FileName
         object{Surface}
      #else
         #local Build=1;
      #end
   #else
      #local Build=1;
   #end
   #if(Build)
      #local Umin=Urange.u;
      #local Umax=Urange.v;
      #local Vmin=Vrange.u;
      #local Vmax=Vrange.v;
      #declare iU=(Umax-Umin)/Iter_U;
      #declare iV=(Vmax-Vmin)/Iter_V;
      #local NumVertices=(Iter_U+1)*(Iter_V+1);
      #local NumFaces=Iter_U*Iter_V*2;
      #debug concat("\n Calculating ",str(NumVertices,0,0)," vertices for ", str(NumFaces,0,0)," triangles\n\n")
      #debug "\n"
      #local VecArr=array[NumVertices] 
      #local NormArr=array[NumVertices] 
      #local UVArr=array[NumVertices]
      #local Count=0;   
      #local I=0;  
      #while (I<Iter_V+1)
         #local V=RangeMM(I,0,Iter_V,Vmin,Vmax);
         #local J=0;
         #while (J<Iter_U+1)
            #local U=RangeMM(J,0,Iter_U,Umin,Umax);
            #local VecArr[Count]=<(Fx(U,V)),(Fy(U,V)),(Fz(U,V))>;  
            #local NormArr[Count]=CalcNormal(VecArr[Count]);  
            #local UVArr[Count]=<U,V>;
            #local Count=Count+1;
            #local J=J+1;            
         #end
         #debug concat("\r Done ", str(Count,0,0)," vertices : ",str(100*Count/NumVertices,0,2)," %")
         #local I=I+1;
      #end
      BuildWriteMesh2(VecArr, NormArr, UVArr, Iter_U, Iter_V, FileName)
   #end
   #declare __FU=0;   #declare __TU=0;
   #declare __FV=0;   #declare __TV=0;
#end   

