// This work is licensed under the Creative Commons Attribution-ShareAlike 3.0 Unported License.
// To view a copy of this license, visit http://creativecommons.org/licenses/by-sa/3.0/ or send a
// letter to Creative Commons, 444 Castro Street, Suite 900, Mountain View, California, 94041, USA.

//    Persistence of Vision Ray Tracer version 3.7 Include File
//    File: granites.inc
//    Last updated: 2021.4.10
//    Description: Predefined granite textures for Persistence of Vision Raytracer version 2.2
//                 by Daniel Mecklenburg Jr 
//                 Redefined and made consistent for version 3.7+ by Thomas de Groot (2021)

#ifndef(Granites_Inc_Temp)
#declare Granites_Inc_Temp = version;
#version 3.7;

/*=========================================================================================================
    Content & Notes:
    ===============
    Compared to the original code from 1996, these granite patterns have been slightly, and sometimes
    more drastically modified in order to resemble more closely to the real world examples found on
    the internet. Original code has been commented out in those cases.
    
    For each granite three versions are provided. You choose them with the SCS switch (see below) 
    which determines in which color space the granite is to be rendered.
    
    SCS=0 renders the granite in linear color space, essentially the original code.
    SCS=1 just transposes linear color space (rgb) to standard color space (srgb).
    SCS=2 Converts linear color space to standard color space using conversion macros.
    
    Mohogany granite - polished surface             :  MohoganyPol_1 & MohoganyPol_2  : MohoganyPol.pov
    Mohogany granite - frosted surface              :  MohoganyFro_1 & MohoganyFro_2  : MohoganyFro.pov
    Impala granite - polished surface               :  ImpalaPol_1 & ImpalaPol_2      : ImpalaPol.pov
    Impala granite - frosted surface                :  ImpalaFro_1 & ImpalaFro_2      : ImpalaFro.pov
    India Black granite - polished surface          :  IndiaBPol_1 & IndiaBPol_2      : IndiaBPol.pov
    India Black granite - frosted surface           :  IndiaBFro_1 & IndiaBFro_2      : IndiaBFro.pov
    North American Pink granite - polished surface  :  NAPPol_1 & NAPPol_2            : NAPPol.pov
    North American Pink granite - frosted surface   :  NAPFro                         : NAPFro.pov
    Canadian Pink granite - polished surface        :  CanPPol_1 & CanPPol_2          : CanPPol.pov
    Canadian Pink granite - frosted surface         :  CanPFro                        : CanPFro.pov
    Southern Gray granite - polished surface        :  SGrayPol_1 & SGrayPol_2        : SGrayPol.pov
    Southern Gray granite - frosted surface         :  SGrayFro                       : SGrayFro.pov
    Medium Barre Gray granite - polished surface    :  MGrayPol_1 & MGrayPol_2        : MGrayPol.pov
    Medium Barre Gray granite - frosted surface     :  MGrayFro                       : MGrayFro.pov
    Saint-Andr green granite - polished surface    :  StAndGPol_1 & StAndGPol_2      : StAndGPol.pov
    Saint-Andr green granite - frosted surface     :  StAndGFro                      : StAndGFro.pov

    Note for correct use:
    The granite patterns have been scaled in such a way that, when applied to a unit-sized POV-Ray object, 
    they correspond most closely to the real world examples from which they have been modelled. Evidently,
    the user is free to deviate from this standard use.
    
=========================================================================================================*/
//Standard Color Space (sRGB) conversion:
#ifndef (SCS) #declare SCS = 2;  #end //0=linear color space; 1=sRGB; 2=sRGB transformation
#ifndef (SatBoost) #declare SatBoost = 0.01; #end
#ifndef (BrightBoost) #declare BrightBoost = -0.01; #end  //0 = no brightness change; >0 = brighter; <0 = darker

//transformation functions by Ive (do NOT change!):
//from scRGB (linear color space) -> sRGB:
#declare sRGB_Gamma = 
function(C) {
  select(C-0.0031308, C*12.92 , 1.055*pow(C,1/2.4)-0.055)
}

#macro scRGB_to_sRGB(Color)
/*rgb*/ <sRGB_Gamma(Color.red),
         sRGB_Gamma(Color.green),
         sRGB_Gamma(Color.blue)>
#end

//---------------------------------------------------------------------------
//Colour saturation/brightness variation code by Clipka.
//[povray.advanced-users 15-07-2016: Reverse engineering pigments and textures by BaldEagle]:

#macro ColourSat(SatBoost, BrightBoost, Raw)
#declare SB        = SatBoost; // Saturation boost
#declare VB        = BrightBoost; // Brightness ("volume") boost
#declare RawColour = srgb Raw;
#declare SatColour = RawColour-SB;
#declare MyColour  = SatColour*(1+VB)*RawColour.gray/SatColour.gray;
MyColour
#end

//Saturation and Brightness values for Clipka's saturation macro:
#ifndef (SatBoost) #declare SatBoost = 0.01; #end
#ifndef (BrightBoost) #declare BrightBoost = -0.1; #end   //0 = no brightness change; >0 = brighter; <0 = darker

/*===============================================================================================*/

// Mohogany granite - polished surface
#switch (SCS)
#case (0)
  #declare MohoganyPol_1 =
  texture {
    pigment {
      granite
      //turbulence 0.4
      color_map {
        [0.00 rgb <0.000, 0.000, 0.000>]
        [0.25 rgb <0.059, 0.059, 0.059>]
        [0.25 rgb <0.086, 0.027, 0.059>]
        [0.35 rgb <0.118, 0.118, 0.078>]
        [0.35 rgb <0.200, 0.137, 0.110>]
        [1.00 rgb <0.769, 0.329, 0.298>]
      }
      scale 0.1 //0.8
    }
    finish {
      diffuse 0.6
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

  #declare MohoganyPol_2 =
  texture {MohoganyPol_1}
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 rgbt <0.200, 0.200, 0.200, 0.149>]
        [0.25 rgbt <0.200, 0.200, 0.200, 0.949>]
        [0.25 rgbt <1.000, 1.000, 1.000, 1.000>]
        [1.00 rgbt <1.000, 1.000, 1.000, 1.000>]
      }
      scale 0.3
      translate <20.0, 20.0, 33.0>
      rotate <0.3, 0.2, 0.0>
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

#break
#case (1)
  #declare MohoganyPol_1 =
  texture {
    pigment {
      granite
      //turbulence 0.4
      color_map {
        [0.00 srgb <0.000, 0.000, 0.000>]
        [0.25 srgb <0.059, 0.059, 0.059>]
        [0.25 srgb <0.086, 0.027, 0.059>]
        [0.35 srgb <0.118, 0.118, 0.078>]
        [0.35 srgb <0.200, 0.137, 0.110>]
        [1.00 srgb <0.769, 0.329, 0.298>]
      }
      scale 0.1 //0.8
    }
    finish {
      diffuse 0.6
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

  #declare MohoganyPol_2 =
  texture {MohoganyPol_1}
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 srgbt <0.200, 0.200, 0.200, 0.149>]
        [0.25 srgbt <0.200, 0.200, 0.200, 0.949>]
        [0.25 srgbt <1.000, 1.000, 1.000, 1.000>]
        [1.00 srgbt <1.000, 1.000, 1.000, 1.000>]
      }
      scale 0.3
      translate <20.0, 20.0, 33.0>
      rotate <0.3, 0.2, 0.0>
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

#break
#case (2)
  #declare T1_MohoganyPol =
  texture {
    pigment {
      granite
      //turbulence 0.4
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.000, 0.000, 0.000>))]
        [0.25 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.059, 0.059, 0.059>))]
        [0.25 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.086, 0.027, 0.059>))]
        [0.35 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.118, 0.118, 0.078>))]
        [0.35 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.200, 0.137, 0.110>))]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.769, 0.329, 0.298>))]
      }
      scale 0.1 //0.8
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  #declare T2_MohoganyPol =
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.200, 0.200, 0.200>)) transmit 0.149]
        [0.25 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.200, 0.200, 0.200>)) transmit 0.949]
        [0.25 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
      }
      scale 0.3
      translate <20.0, 20.0, 33.0>
      rotate <0.3, 0.2, 0.0>
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

  #declare MohoganyPol_1 = texture {T1_MohoganyPol}

  #declare MohoganyPol_2 =
  texture {T1_MohoganyPol}
  texture {T2_MohoganyPol}

#end

//-------------------------------------------------------
// Mohogany granite - frosted surface
#switch (SCS)
#case (0)
  #declare MohoganyFro_1 =
  texture {
    pigment {
      granite
      //turbulence 0.4
      color_map {
        [0.00 color rgb<0.000, 0.000, 0.000>]
        [0.25 color rgb<0.059, 0.059, 0.059>]
        [0.25 color rgb<0.086, 0.027, 0.059>]
        [0.35 color rgb<0.118, 0.118, 0.078>]
        [0.35 color rgb<0.200, 0.137, 0.110>]
        [1.00 color rgb<0.769, 0.329, 0.298>]
      }
      scale 0.1 //0.8
    }
    normal {
      granite 0.25
      scale 0.1
    }
    finish {
      diffuse 0.6
    }
  }
  
  #declare MohoganyFro_2 =
  texture {MohoganyFro_1}
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 color rgbt<0.200, 0.200, 0.200, 0.149>]
        [0.25 color rgbt<0.200, 0.200, 0.200, 0.949>]
        [0.25 color rgbt<1.000, 1.000, 1.000, 1.000>]
        [1.00 color rgbt<1.000, 1.000, 1.000, 1.000>]
      }
      scale 0.3
      translate <20.0, 20.0, 33.0>
      rotate <0.3, 0.2, 0.0>
    }
    normal {
      granite 0.25
      turbulence 0.2
      scale 0.3
    }
    finish {
      diffuse 0.6
    }
  }
  
#break
#case (1)
  #declare MohoganyFro_1 =
  texture {
    pigment {
      granite
      //turbulence 0.4
      color_map {
        [0.00 srgb <0.000, 0.000, 0.000>]
        [0.25 srgb <0.059, 0.059, 0.059>]
        [0.25 srgb <0.086, 0.027, 0.059>]
        [0.35 srgb <0.118, 0.118, 0.078>]
        [0.35 srgb <0.200, 0.137, 0.110>]
        [1.00 srgb <0.769, 0.329, 0.298>]
      }
      scale 0.1 //0.8
    }
    normal {
      granite 0.25
      scale 0.1
    }
    finish {
      diffuse 0.6
    }
  }
  
  #declare MohoganyFro_2 =
  texture {MohoganyFro_1}
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 color srgbt<0.200, 0.200, 0.200, 0.149>]
        [0.25 color srgbt<0.200, 0.200, 0.200, 0.949>]
        [0.25 color srgbt<1.000, 1.000, 1.000, 1.000>]
        [1.00 color srgbt<1.000, 1.000, 1.000, 1.000>]
      }
      scale 0.3
      translate <20.0, 20.0, 33.0>
      rotate <0.3, 0.2, 0.0>
    }
    normal {
      granite 0.25
      turbulence 0.2
      scale 0.3
    }
    finish {
      diffuse 0.6
    }
  }
  
#break
#case (2)
  #declare T1_MohoganyFro =
  texture {
    pigment {
      granite
      //turbulence 0.4
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.000, 0.000, 0.000>))]
        [0.25 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.059, 0.059, 0.059>))]
        [0.25 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.086, 0.027, 0.059>))]
        [0.35 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.118, 0.118, 0.078>))]
        [0.35 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.200, 0.137, 0.110>))]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.769, 0.329, 0.298>))]
      }
      scale 0.1 //0.8
    }
    normal {
      granite 0.25
      scale 0.1
    }
    finish {
      diffuse 0.6
    }
  }

  #declare T2_MohoganyFro =
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.200, 0.200, 0.200>)) transmit 0.149]
        [0.25 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.200, 0.200, 0.200>)) transmit 0.949]
        [0.25 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
      }
      scale 0.3
      translate <20.0, 20.0, 33.0>
      rotate <0.3, 0.2, 0.0>
    }
    normal {
      granite 0.25
      turbulence 0.2
      scale 0.3
    }
    finish {
      diffuse 0.6
    }
  }
  
  #declare MohoganyFro_1 = texture {T1_MohoganyFro}
  
  #declare MohoganyFro_2 =
  texture {T1_MohoganyFro}
  texture {T2_MohoganyFro}
  
#end
 
//-------------------------------------------------------
// Impala granite - polished surface
#switch (SCS)
#case (0)
  #declare ImpalaPol_1 =
  texture {
    pigment {
      granite
      //turbulence 0.7
      color_map {
        [0.00 rgb <0.000, 0.000, 0.000>]
        [0.45 rgb <0.078, 0.039, 0.067>]
        [0.45 rgb <0.286, 0.329, 0.357>]
        [1.00 rgb <0.157, 0.118, 0.129>]
      }
      scale 0.05
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare ImpalaPol_2 =
  texture {ImpalaPol_1}
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 rgbt <0.659, 0.659, 0.698, 0.149>]
        [0.25 rgbt <0.498, 0.498, 0.549, 0.949>]
        [0.25 rgbt <1.000, 1.000, 1.000, 1.000>]
        [1.00 rgbt <1.000, 1.000, 1.000, 1.000>]
      }
      scale 0.3
      translate <33.0, 20.0, 0.0>
      rotate <0.3, 0.2, 0.3>
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

#break
#case (1)
  #declare ImpalaPol_1 =
  texture {
    pigment {
      granite
      //turbulence 0.7
      color_map {
        [0.00 srgb <0.000, 0.000, 0.000>]
        [0.45 srgb <0.078, 0.039, 0.067>]
        [0.45 srgb <0.286, 0.329, 0.357>]
        [1.00 srgb <0.157, 0.118, 0.129>]
      }
      scale 0.05
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare ImpalaPol_2 =
  texture {ImpalaPol_1}
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 srgbt <0.659, 0.659, 0.698, 0.149>]
        [0.25 srgbt <0.498, 0.498, 0.549, 0.949>]
        [0.25 srgbt <1.000, 1.000, 1.000, 1.000>]
        [1.00 srgbt <1.000, 1.000, 1.000, 1.000>]
      }
      scale 0.3
      translate <33.0, 20.0, 0.0>
      rotate <0.3, 0.2, 0.3>
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

#break
#case (2)
  #declare T1_ImpalaPol =
  texture {
    pigment {
      granite
      //turbulence 0.7
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.000, 0.000, 0.000>))]
        [0.45 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.078, 0.039, 0.067>))]
        [0.45 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.286, 0.329, 0.357>))]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.157, 0.118, 0.129>))]
      }
      scale 0.05
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare T2_ImpalaPol =
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.659, 0.659, 0.698>)) transmit 0.149]
        [0.25 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.498, 0.498, 0.549>)) transmit 0.949]
        [0.25 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
      }
      scale 0.3
      translate <33.0, 20.0, 0.0>
      rotate <0.3, 0.2, 0.3>
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare ImpalaPol_1 = texture {T1_ImpalaPol}

  #declare ImpalaPol_2 =
  texture {T1_ImpalaPol}
  texture {T2_ImpalaPol}
  
#end

//-------------------------------------------------------
// Impala granite - frosted surface
#switch (SCS)
#case (0)
  #declare ImpalaFro_1 =
  texture {
    pigment {
      granite
      //turbulence 0.7
      color_map {
        [0.00 rgb <0.298, 0.298, 0.298>]
        [0.45 rgb <0.376, 0.337, 0.369>]
        [0.45 rgb <0.486, 0.529, 0.557>]
        [1.00 rgb <0.357, 0.318, 0.329>]
      }
      scale 0.05
    }
    normal {
      granite 0.25
      scale 0.05
    }
    finish {
      diffuse 0.6
    }
  }
  
  #declare ImpalaFro_2 =
  texture {ImpalaFro_1}
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 rgbt <0.659, 0.659, 0.698, 0.149>]
        [0.25 rgbt <0.498, 0.498, 0.549, 0.949>]
        [0.25 rgbt <1.000, 1.000, 1.000, 1.000>]
        [1.00 rgbt <1.000, 1.000, 1.000, 1.000>]
      }
      scale 0.3
      translate <33.0, 20.0, 0.0>
      rotate <0.3, 0.2, 0.3>
    }
    normal {
      granite 0.25
      scale 0.3
    }
    finish {
      diffuse 0.6
    }
  }

#break
#case (1)
  #declare ImpalaFro_1 =
  texture {
    pigment {
      granite
      //turbulence 0.7
      color_map {
        [0.00 srgb <0.298, 0.298, 0.298>]
        [0.45 srgb <0.376, 0.337, 0.369>]
        [0.45 srgb <0.486, 0.529, 0.557>]
        [1.00 srgb <0.357, 0.318, 0.329>]
      }
      scale 0.05
    }
    normal {
      granite 0.25
      scale 0.05
    }
    finish {
      diffuse 0.6
    }
  }
  
  #declare ImpalaFro_2 =
  texture {ImpalaFro_1}
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 srgbt <0.659, 0.659, 0.698, 0.149>]
        [0.25 srgbt <0.498, 0.498, 0.549, 0.949>]
        [0.25 srgbt <1.000, 1.000, 1.000, 1.000>]
        [1.00 srgbt <1.000, 1.000, 1.000, 1.000>]
      }
      scale 0.3
      translate <33.0, 20.0, 0.0>
      rotate <0.3, 0.2, 0.3>
    }
    normal {
      granite 0.25
      scale 0.3
    }
    finish {
      diffuse 0.6
    }
  }

#break
#case (2)
  #declare T1_ImpalaFro =
  texture {
    pigment {
      granite
      //turbulence 0.7
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.298, 0.298, 0.298>))]
        [0.45 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.376, 0.337, 0.369>))]
        [0.45 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.486, 0.529, 0.557>))]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.357, 0.318, 0.329>))]
      }
      scale 0.05
    }
    normal {
      granite 0.25
      scale 0.05
    }
    finish {
      diffuse 0.6
    }
  }
  
  #declare T2_ImpalaFro =
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.659, 0.659, 0.698>)) transmit 0.149]
        [0.25 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.498, 0.498, 0.549>)) transmit 0.949]
        [0.25 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
      }
      scale 0.3
      translate <33.0, 20.0, 0.0>
      rotate <0.3, 0.2, 0.3>
    }
    normal {
      granite 0.25
      scale 0.3
    }
    finish {
      diffuse 0.6
    }
  }
  
  #declare ImpalaFro_1 = texture {T1_ImpalaFro}
  
  #declare ImpalaFro_2 =
  texture {T1_ImpalaFro}
  texture {T2_ImpalaFro}
  
#end

//-------------------------------------------------------
// India Black granite - polished surface
#switch (SCS)
#case (0)
  #declare IndiaBPol_1 =
  texture {
    pigment {
      granite
      //turbulence 0.7
      color_map {
        [0.00 rgb <0.000, 0.000, 0.000>]
        [0.45 rgb <0.008, 0.020, 0.008>]
        [1.00 rgb <0.008, 0.020, 0.008>]
      }
      scale 0.1
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare IndiaBPol_2 =
  texture {IndiaBPol_1}
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 rgbt <0.200, 0.200, 0.200, 0.149>]
        [0.25 rgbt <0.200, 0.200, 0.200, 0.949>]
        [0.25 rgbt <1.000, 1.000, 1.000, 1.000>]
        [1.00 rgbt <1.000, 1.000, 1.000, 1.000>]
      }
      scale 0.3
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

#break
#case (1)
  #declare IndiaBPol_1 =
  texture {
    pigment {
      granite
      //turbulence 0.7
      color_map {
        [0.00 srgb <0.000, 0.000, 0.000>]
        [0.45 srgb <0.008, 0.020, 0.008>]
        [1.00 srgb <0.008, 0.020, 0.008>]
      }
      scale 0.1
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare IndiaBPol_2 =
  texture {IndiaBPol_1}
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 srgbt <0.200, 0.200, 0.200, 0.149>]
        [0.25 srgbt <0.200, 0.200, 0.200, 0.949>]
        [0.25 srgbt <1.000, 1.000, 1.000, 1.000>]
        [1.00 srgbt <1.000, 1.000, 1.000, 1.000>]
      }
      scale 0.3
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

#break
#case (2)
  #declare T1_IndiaBPol =
  texture {
    pigment {
      granite
      //turbulence 0.7
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.000, 0.000, 0.000>))]
        [0.45 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.008, 0.020, 0.008>))]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.008, 0.020, 0.008>))]
      }
      scale 0.1
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare T2_IndiaBPol =
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.200, 0.200, 0.200>)) transmit 0.149]
        [0.25 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.200, 0.200, 0.200>)) transmit 0.949]
        [0.25 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
      }
      scale 0.3
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare IndiaBPol_1 = texture {T1_IndiaBPol}
  
  #declare IndiaBPol_2 =
  texture {T1_IndiaBPol}
  texture {T2_IndiaBPol}
  
#end

//-------------------------------------------------------
// India Black granite - frosted surface
#switch (SCS)
#case (0)
  #declare IndiaBFro_1 =
  texture {
    pigment {
      granite
      //turbulence 0.7
      color_map {
        [0.00 color rgb<0.298, 0.298, 0.298>]
        [0.45 color rgb<0.298, 0.318, 0.318>]
        [0.45 color rgb<0.286, 0.329, 0.357>]
        [1.00 color rgb<0.298, 0.318, 0.318>]
      }
      scale 0.1//0.5
    }
    normal {
      granite 0.25
      scale 0.1
    }
    finish {
      diffuse 0.6
    }
  }

  #declare IndiaBFro_2 =
  texture {IndiaBFro_1}
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 rgbt <0.200, 0.200, 0.200, 0.149>]
        [0.25 rgbt <0.200, 0.200, 0.200, 0.949>]
        [0.25 rgbt <1.000, 1.000, 1.000, 1.000>]
        [1.00 rgbt <1.000, 1.000, 1.000, 1.000>]
      }
      scale 0.3
    }
    normal {
      granite 0.25
      scale 0.3
    }
    finish {
      diffuse 0.6
    }
  }
  
#break
#case (1)
  #declare IndiaBFro_1 =
  texture {
    pigment {
      granite
      //turbulence 0.7
      color_map {
        [0.00 srgb <0.298, 0.298, 0.298>]
        [0.45 srgb <0.298, 0.318, 0.318>]
        [0.45 srgb <0.286, 0.329, 0.357>]
        [1.00 srgb <0.298, 0.318, 0.318>]
      }
      scale 0.1//0.5
    }
    normal {
      granite 0.25
      scale 0.1
    }
    finish {
      diffuse 0.6
    }
  }

  #declare IndiaBFro_2 =
  texture {IndiaBFro_1}
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 srgbt <0.200, 0.200, 0.200, 0.149>]
        [0.25 srgbt <0.200, 0.200, 0.200, 0.949>]
        [0.25 srgbt <1.000, 1.000, 1.000, 1.000>]
        [1.00 srgbt <1.000, 1.000, 1.000, 1.000>]
      }
      scale 0.3
    }
    normal {
      granite 0.25
      scale 0.3
    }
    finish {
      diffuse 0.6
    }
  }

#break
#case (2)
  #declare T1_IndiaBFro =
  texture {
    pigment {
      granite
      //turbulence 0.7
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.298, 0.298, 0.298>))]
        [0.45 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.298, 0.318, 0.318>))]
        [0.45 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.286, 0.329, 0.357>))]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.298, 0.318, 0.318>))]
      }
      scale 0.1//0.5
    }
    normal {
      granite 0.25
      scale 0.1
    }
    finish {
      diffuse 0.6
    }
  }

  #declare T2_IndiaBFro =
  texture {
    pigment {
      granite
      turbulence 0.2
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.200, 0.200, 0.200>)) transmit 0.149]
        [0.25 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.200, 0.200, 0.200>)) transmit 0.949]
        [0.25 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
      }
      scale 0.3
    }
    normal {
      granite 0.25
      scale 0.3
    }
    finish {
      diffuse 0.6
    }
  }
  
  #declare IndiaBFro_1 = texture {T1_IndiaBFro}
  
  #declare IndiaBFro_2 =
  texture {T1_IndiaBFro}
  texture {T2_IndiaBFro}

#end

//-------------------------------------------------------
// North American Pink granite - polished surface
#switch (SCS)
#case (0)
  #declare NAPPol_1 =
  texture {
    pigment {
      granite
      //turbulence 0.8
      color_map {
        [0.00 rgb <0.380, 0.200, 0.247>]
        [0.50 rgb <0.698, 0.463, 0.337>]
        [0.50 rgb <0.675, 0.506, 0.455>]
        [1.00 rgb <0.922, 0.843, 0.804>]
      }
      scale 0.1//0.8
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare NAPPol_2 =
  texture {NAPPol_1}
  texture {
    pigment {
      granite
      turbulence 0.8
      color_map {
        [0.00 rgbt <1.000, 1.000, 1.000, 1.000>]
        [0.60 rgbt <1.000, 1.000, 1.000, 1.000>]
        [0.60 rgbt <0.098, 0.078, 0.078, 0.498>]
        [1.00 rgb <0.047, 0.039, 0.039>]
      }
      scale 0.3
      translate <20.0, 20.0, 0.0>
      rotate <30.0, 30.0, 30.0>
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
#break
#case (1)
  #declare NAPPol_1 =
  texture {
    pigment {
      granite
      //turbulence 0.8
      color_map {
        [0.00 srgb <0.380, 0.200, 0.247>]
        [0.50 srgb <0.698, 0.463, 0.337>]
        [0.50 srgb <0.675, 0.506, 0.455>]
        [1.00 srgb <0.922, 0.843, 0.804>]
      }
      scale 0.1//0.8
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare NAPPol_2 =
  texture {NAPPol_1}
  texture {
    pigment {
      granite
      turbulence 0.8
      color_map {
        [0.00 srgbt <1.000, 1.000, 1.000, 1.000>]
        [0.60 srgbt <1.000, 1.000, 1.000, 1.000>]
        [0.60 srgbt <0.098, 0.078, 0.078, 0.498>]
        [1.00 srgb <0.047, 0.039, 0.039>]
      }
      scale 0.3
      translate <20.0, 20.0, 0.0>
      rotate <30.0, 30.0, 30.0>
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
#break
#case (2)
  #declare T1_NAPPol =
  texture {
    pigment {
      granite
      //turbulence 0.8
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.380, 0.200, 0.247>))]
        [0.50 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.698, 0.463, 0.337>))]
        [0.50 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.675, 0.506, 0.455>))]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.922, 0.843, 0.804>))]
      }
      scale 0.1//0.8
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare T2_NAPPol =
  texture {
    pigment {
      granite
      turbulence 0.8
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [0.60 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [0.60 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.098, 0.078, 0.078>)) transmit 0.498]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.047, 0.039, 0.039>))]
      }
      scale 0.3
      translate <20.0, 20.0, 0.0>
      rotate <30.0, 30.0, 30.0>
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

  #declare NAPPol_1 = texture {T1_NAPPol}
  
  #declare NAPPol_2 =
  texture {T1_NAPPol}
  texture {T2_NAPPol}

#end

//-------------------------------------------------------
// North American Pink granite - frosted surface
#switch (SCS)
#case (0)
  #declare NAPFro =
  texture {
    pigment {
      granite
      //turbulence 0.8
      color_map {
        [0.00 rgb <0.576, 0.435, 0.482>]
        [0.50 rgb <0.635, 0.506, 0.455>]
        [0.50 rgb <0.675, 0.506, 0.455>]
        [0.72 rgb <0.961, 0.863, 0.843>]
        [0.72 rgb <0.275, 0.275, 0.275>]
        [1.00 rgb <0.196, 0.196, 0.196>]
      }
    scale 0.1//0.5
    }
    normal {
      granite 0.25
      scale 0.1
    }
    finish {
      diffuse 0.6
    }
  }

#break
#case (1)
  #declare NAPFro =
  texture {
    pigment {
      granite
      //turbulence 0.8
      color_map {
        [0.00 srgb <0.576, 0.435, 0.482>]
        [0.50 srgb <0.635, 0.506, 0.455>]
        [0.50 srgb <0.675, 0.506, 0.455>]
        [0.72 srgb <0.961, 0.863, 0.843>]
        [0.72 srgb <0.275, 0.275, 0.275>]
        [1.00 srgb <0.196, 0.196, 0.196>]
      }
    scale 0.1//0.5
    }
    normal {
      granite 0.25
      scale 0.1
    }
    finish {
      diffuse 0.6
    }
  }

#break
#case (2)
  #declare NAPFro =
  texture {
    pigment {
      granite
      //turbulence 0.8
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.576, 0.435, 0.482>))]
        [0.50 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.635, 0.506, 0.455>))]
        [0.50 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.675, 0.506, 0.455>))]
        [0.72 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.961, 0.863, 0.843>))]
        [0.72 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.275, 0.275, 0.275>))]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.196, 0.196, 0.196>))]
      }
    scale 0.1//0.5
    }
    normal {
      granite 0.25
      scale 0.1
    }
    finish {
      diffuse 0.6
    }
  }

#end
  
//-------------------------------------------------------
// Canadian Pink granite - polished surface
#switch (SCS)
#case (0)
  #declare CanPPol_1 =
  texture {
    pigment {
      granite
      //turbulence 0.8
      color_map {
        [0.00 rgb <0.380, 0.200, 0.247>]
        [0.50 rgb <0.761, 0.486, 0.361>]
        [0.50 rgb <0.675, 0.506, 0.455>]
        [1.00 rgb <0.804, 0.588, 0.533>]
      }
      scale 0.1//0.8
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

  #declare CanPPol_2 =
  texture {CanPPol_1}
  texture {
    pigment {
      granite
      //turbulence 0.8
      color_map {
        [0.00 rgbt <1.000, 1.000, 1.000, 1.000>]
        [0.60 rgbt <1.000, 1.000, 1.000, 1.000>]
        [0.60 rgbt <0.098, 0.078, 0.078, 0.498>]
        [1.00 rgb <0.047, 0.039, 0.039>]
      }
      translate <20.0, 20.0, 0.0>
      rotate <30.0, 30.0, 30.0>
      scale 0.3
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

#break
#case (1)
  #declare CanPPol_1 =
  texture {
    pigment {
      granite
      //turbulence 0.8
      color_map {
        [0.00 srgb <0.380, 0.200, 0.247>]
        [0.50 srgb <0.761, 0.486, 0.361>]
        [0.50 srgb <0.675, 0.506, 0.455>]
        [1.00 srgb <0.804, 0.588, 0.533>]
      }
      scale 0.1//0.8
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

  #declare CanPPol_2 =
  texture {CanPPol_1}
  texture {
    pigment {
      granite
      turbulence 0.8
      color_map {
        [0.00 srgbt <1.000, 1.000, 1.000, 1.000>]
        [0.60 srgbt <1.000, 1.000, 1.000, 1.000>]
        [0.60 srgbt <0.098, 0.078, 0.078, 0.498>]
        [1.00 srgb <0.047, 0.039, 0.039>]
      }
      translate <20.0, 20.0, 0.0>
      rotate <30.0, 30.0, 30.0>
      scale 0.3
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

#break
#case (2)
  #declare T1_CanPPol =
  texture {
    pigment {
      granite
      //turbulence 0.8
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.380, 0.200, 0.247>))]
        [0.50 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.761, 0.486, 0.361>))]
        [0.50 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.675, 0.506, 0.455>))]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.804, 0.588, 0.533>))]
      }
      scale 0.1//0.8
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

  #declare T2_CanPPol =
  texture {
    pigment {
      granite
      turbulence 0.8
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [0.60 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [0.60 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.098, 0.078, 0.078>)) transmit 0.498]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.047, 0.039, 0.039>))]
      }
      translate <20.0, 20.0, 0.0>
      rotate <30.0, 30.0, 30.0>
      scale 0.3
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

  #declare CanPPol_1 = texture {T1_CanPPol}
  
  #declare CanPPol_2 =
  texture {T1_CanPPol}
  texture {T2_CanPPol}

#end

//-------------------------------------------------------
// Canadian Pink granite - frosted surface
#switch (SCS)
#case (0)
  #declare CanPFro =
  texture {
    pigment {
      granite
      //turbulence 0.8
      color_map {
        [0.00 rgb <0.576, 0.435, 0.482>]
        [0.50 rgb <0.635, 0.506, 0.455>]
        [0.50 rgb <0.675, 0.506, 0.455>]
        [0.80 rgb <0.961, 0.863, 0.843>]
        [0.80 rgb <0.275, 0.275, 0.275>]
        [1.00 rgb <0.196, 0.196, 0.196>]
      }
      scale 0.1//0.5
    }
    normal {
      granite 0.25
      scale 0.1
    }
    finish {
      diffuse 0.6
    }
  }

#break
#case (1)
  #declare CanPFro =
  texture {
    pigment {
      granite
      //turbulence 0.8
      color_map {
        [0.00 srgb <0.576, 0.435, 0.482>]
        [0.50 srgb <0.635, 0.506, 0.455>]
        [0.50 srgb <0.675, 0.506, 0.455>]
        [0.80 srgb <0.961, 0.863, 0.843>]
        [0.80 srgb <0.275, 0.275, 0.275>]
        [1.00 srgb <0.196, 0.196, 0.196>]
      }
      scale 0.1//0.5
    }
    normal {
      granite 0.25
      scale 0.1
    }
    finish {
      diffuse 0.6
    }
  }
  
#break
#case (2)
  #declare CanPFro =
  texture {
    pigment {
      granite
      //turbulence 0.8
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.576, 0.435, 0.482>))]
        [0.50 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.635, 0.506, 0.455>))]
        [0.50 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.675, 0.506, 0.455>))]
        [0.80 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.961, 0.863, 0.843>))]
        [0.80 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.275, 0.275, 0.275>))]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.196, 0.196, 0.196>))]
      }
      scale 0.1//0.5
    }
    normal {
      granite 0.25
      scale 0.1
    }
    finish {
      diffuse 0.6
    }
  }

#end

//-------------------------------------------------------
// Southern Gray granite - polished surface
#switch (SCS)
#case (0)
  #declare SGrayPol_1 =
  texture {
    pigment {
      granite
      //turbulence 0.9
      color_map {
        [0.00 rgb <0.941, 0.941, 0.941>]
        [0.20 rgb <0.784, 0.784, 0.784>]
        [1.00 rgb <0.471, 0.471, 0.471>]
      }
      scale 0.05//0.8
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare SGrayPol_2 =
  texture {SGrayPol_1}
  texture {
    pigment {
      granite
      turbulence 0.8
      color_map {
        [0.00 rgbt <1.000, 1.000, 1.000, 1.000>]
        [0.50 rgbt <1.000, 1.000, 1.000, 1.000>]
        [0.50 rgbt <0.200, 0.157, 0.157, 0.498>]
        [1.00 rgb <0.000, 0.000, 0.000>]
      }
      translate <33.0, 20.0, 0.0>
      rotate <0.3, 0.2, 0.3>
      scale 0.1//0.2
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
#break
#case (1)
  #declare SGrayPol_1 =
  texture {
    pigment {
      granite
      //turbulence 0.9
      color_map {
        [0.00 srgb <0.941, 0.941, 0.941>]
        [0.20 srgb <0.784, 0.784, 0.784>]
        [1.00 srgb <0.471, 0.471, 0.471>]
      }
      scale 0.05//0.8
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare SGrayPol_2 =
  texture {SGrayPol_1}
  texture {
    pigment {
      granite
      turbulence 0.8
      color_map {
        [0.00 srgbt <1.000, 1.000, 1.000, 1.000>]
        [0.50 srgbt <1.000, 1.000, 1.000, 1.000>]
        [0.50 srgbt <0.200, 0.157, 0.157, 0.498>]
        [1.00 srgb <0.000, 0.000, 0.000>]
      }
      translate <33.0, 20.0, 0.0>
      rotate <0.3, 0.2, 0.3>
      scale 0.1//0.2
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
#break
#case (2)
  #declare T1_SGrayPol =
  texture {
    pigment {
      granite
      //turbulence 0.9
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.941, 0.941, 0.941>))]
        [0.20 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.784, 0.784, 0.784>))]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.471, 0.471, 0.471>))]
      }
      scale 0.05//0.8
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare T2_SGrayPol =
  texture {
    pigment {
      granite
      turbulence 0.8
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [0.50 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [0.50 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.200, 0.157, 0.157>)) transmit 0.498]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.000, 0.000, 0.000>))]
      }
      translate <33.0, 20.0, 0.0>
      rotate <0.3, 0.2, 0.3>
      scale 0.1//0.2
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

  #declare SGrayPol_1 = texture {T1_SGrayPol}
  
  #declare SGrayPol_2 =
  texture {T1_SGrayPol}
  texture {T2_SGrayPol}

#end

//-------------------------------------------------------
// Southern Gray granite - frosted surface
#switch (SCS)
#case (0)
  #declare SGrayFro =
  texture {
    pigment {
      granite
      //turbulence 0.9
      color_map {
        [0.00 rgb <0.980, 0.980, 0.980>]
        [0.60 rgb <0.745, 0.745, 0.745>]
        [0.60 rgb <0.345, 0.345, 0.345>]
        [1.00 rgb <0.549, 0.549, 0.549>]
      }
      scale 0.05//0.25
    }
    normal {
      granite 0.25
      scale 0.05
    }
    finish {
      diffuse 0.6
    }
  }

#break
#case (1)
  #declare SGrayFro =
  texture {
    pigment {
      granite
      //turbulence 0.9
      color_map {
        [0.00 srgb <0.980, 0.980, 0.980>]
        [0.60 srgb <0.745, 0.745, 0.745>]
        [0.60 srgb <0.345, 0.345, 0.345>]
        [1.00 srgb <0.549, 0.549, 0.549>]
      }
      scale 0.05//0.25
    }
    normal {
      granite 0.25
      scale 0.05
    }
    finish {
      diffuse 0.6
    }
  }

#break
#case (2)
  #declare SGrayFro =
  texture {
    pigment {
      granite
      turbulence 0.9
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.980, 0.980, 0.980>))]
        [0.60 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.745, 0.745, 0.745>))]
        [0.60 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.345, 0.345, 0.345>))]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.549, 0.549, 0.549>))]
      }
      scale 0.05//0.25
    }
    normal {
      granite 0.25
      scale 0.05
    }
    finish {
      diffuse 0.6
    }
  }
  
#end
//-------------------------------------------------------
// Medium Barre Gray granite - polished surface
#switch (SCS)
#case (0)
  #declare MGrayPol_1 =
  texture {
    pigment {
      granite//agate
      color_map {
        [0.00 rgb <0.961, 0.961, 0.961>]
        [0.50 rgb <0.863, 0.863, 0.863>]
        [0.50 rgb <0.510, 0.510, 0.510>]
        [1.00 rgb <0.314, 0.314, 0.314>]
      }
      scale 0.075//0.12
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare MGrayPol_2 =
  texture {MGrayPol_1}
  texture {
    pigment {
      granite
      turbulence 0.9
      color_map {
        [0.00 rgbt <1.000, 1.000, 1.000, 1.000>]
        [0.60 rgbt <1.000, 1.000, 1.000, 1.000>]
        [0.60 rgbt <0.200, 0.259, 0.157, 0.600>]
        [1.00 rgb <0.000, 0.000, 0.000>]
      }
      translate <33.0, 20.0, 0.0>
      rotate <0.3, 0.2, 0.3>
      scale 0.3
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

#break
#case (1)
  #declare MGrayPol_1 =
  texture {
    pigment {
      granite//agate
      color_map {
        [0.00 srgb <0.961, 0.961, 0.961>]
        [0.50 srgb <0.863, 0.863, 0.863>]
        [0.50 srgb <0.510, 0.510, 0.510>]
        [1.00 srgb <0.314, 0.314, 0.314>]
      }
      scale 0.075//0.12
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare MGrayPol_2 =
  texture {MGrayPol_1}
  texture {
    pigment {
      granite
      turbulence 0.9
      color_map {
        [0.00 srgbt <1.000, 1.000, 1.000, 1.000>]
        [0.60 srgbt <1.000, 1.000, 1.000, 1.000>]
        [0.60 srgbt <0.200, 0.259, 0.157, 0.600>]
        [1.00 srgb <0.000, 0.000, 0.000>]
      }
      translate <33.0, 20.0, 0.0>
      rotate <0.3, 0.2, 0.3>
      scale 0.3
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }


#break
#case (2)
  #declare T1_MGrayPol =
  texture {
    pigment {
      granite//agate
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.961, 0.961, 0.961>))]
        [0.50 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.863, 0.863, 0.863>))]
        [0.50 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.510, 0.510, 0.510>))]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.314, 0.314, 0.314>))]
      }
      scale 0.075//0.12
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare T2_MGrayPol =
  texture {
    pigment {
      granite
      turbulence 0.9
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [0.60 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [0.60 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.200, 0.259, 0.157>)) transmit 0.600]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.000, 0.000, 0.000>))]
      }
      translate <33.0, 20.0, 0.0>
      rotate <0.3, 0.2, 0.3>
      scale 0.3
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare MGrayPol_1 = texture {T1_MGrayPol}
  
  #declare MGrayPol_2 =
  texture {T1_MGrayPol}
  texture {T2_MGrayPol}

#end

//-------------------------------------------------------
// Medium Barre Gray granite - frosted surface
#switch (SCS)
#case (0)
  #declare MGrayFro =
  texture {
    pigment {
      granite
      //turbulence 0.9
      color_map {
        [0.00 rgb <0.980, 0.980, 0.980>]
        [0.70 rgb <0.863, 0.863, 0.863>]
        [0.70 rgb <0.267, 0.306, 0.267>]
        [1.00 rgb <0.471, 0.529, 0.471>]
      }
      scale 0.1//0.25
    }
    normal {
      granite 0.25
      scale 0.1
    }
    finish {
      diffuse 0.6
    }
  }
  
#break
#case (1)
  #declare MGrayFro =
  texture {
    pigment {
      granite
      //turbulence 0.9
      color_map {
        [0.00 srgb <0.980, 0.980, 0.980>]
        [0.70 srgb <0.863, 0.863, 0.863>]
        [0.70 srgb <0.267, 0.306, 0.267>]
        [1.00 srgb <0.471, 0.529, 0.471>]
      }
      scale 0.1//0.25
    }
    normal {
      granite 0.25
      scale 0.1
    }
    finish {
      diffuse 0.6
    }
  }

#break
#case (2)
  #declare MGrayFro =
  texture {
    pigment {
      granite
      //turbulence 0.9
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.980, 0.980, 0.980>))]
        [0.70 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.863, 0.863, 0.863>))]
        [0.70 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.267, 0.306, 0.267>))]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.471, 0.529, 0.471>))]
      }
      scale 0.1//0.25
    }
    normal {
      granite 0.25
      scale 0.1
    }
    finish {
      diffuse 0.6
    }
  }

#end

//-------------------------------------------------------
// St. Andre Green granite - polished surface
#switch (SCS)
#case (0)
  #declare StAndGPol_1 =
  texture {
    pigment {
      granite//agate
      //agate_turb 0.7
      color_map {
        [0.00 rgb <0.000, 0.000, 0.000>]
        //[0.65 rgb <0.059, 0.027, 0.047>]
        [0.65 rgb <0.059, 0.047, 0.027>]
        [0.65 rgb <0.067, 0.067, 0.047>]
        [0.85 rgb <0.118, 0.118, 0.078>]
        [0.85 rgb <0.149, 0.157, 0.129>]
        [0.94 rgb <0.349, 0.259, 0.200>]
        [1.00 rgb <0.369, 0.259, 0.247>]
      }
      scale 0.1//0.2
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare StAndGPol_2 =
  texture {StAndGPol_1}
  texture {
    pigment {
      granite
      turbulence 0.4
      color_map {
        //[0.00 rgbt <0.157, 0.157, 0.200, 0.149>]
        //[0.45 rgbt <0.000, 0.000, 0.047, 0.749>]
        [0.00 rgbt <0.157, 0.200, 0.157, 0.149>]
        [0.45 rgbt <0.000, 0.047, 0.000, 0.749>]
        [0.45 rgbt <1.000, 1.000, 1.000, 1.000>]
        [0.50 rgbt <1.000, 1.000, 1.000, 1.000>]
        [0.50 rgb <0.000, 0.000, 0.000>]
        [0.63 rgb <0.000, 0.000, 0.000>]
        [0.63 rgbt <1.000, 1.000, 1.000, 1.000>]
        [0.90 rgb <1.000, 1.000, 1.000>]
        [0.90 rgb <0.451, 0.498, 0.451>]
        [0.91 rgb <1.000, 1.000, 1.000>]
        [0.91 rgbt <1.000, 1.000, 1.000, 1.000>]
        [1.00 rgb <1.000, 1.000, 1.000>]
      }
      scale 0.2
      translate <20.0, 20.0, 0.0>
      rotate <0.3, 0.2, 0.3>
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

#break
#case (1)
  #declare StAndGPol_1 =
  texture {
    pigment {
      granite//agate
      //agate_turb 0.7
      color_map {
        [0.00 srgb <0.000, 0.000, 0.000>]
        //[0.65 srgb <0.059, 0.027, 0.047>]
        [0.65 srgb <0.059, 0.047, 0.027>]
        [0.65 srgb <0.067, 0.067, 0.047>]
        [0.85 srgb <0.118, 0.118, 0.078>]
        [0.85 srgb <0.149, 0.157, 0.129>]
        [0.94 srgb <0.349, 0.259, 0.200>]
        [1.00 srgb <0.369, 0.259, 0.247>]
      }
      scale 0.1//0.2
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare StAndGPol_2 =
  texture {StAndGPol_1}
  texture {
    pigment {
      granite
      turbulence 0.4
      color_map {
        //[0.00 srgbt <0.157, 0.157, 0.200, 0.149>]
        //[0.45 srgbt <0.000, 0.000, 0.047, 0.749>]
        [0.00 srgbt <0.157, 0.200, 0.157, 0.149>]
        [0.45 srgbt <0.000, 0.047, 0.000, 0.749>]
        [0.45 srgbt <1.000, 1.000, 1.000, 1.000>]
        [0.50 srgbt <1.000, 1.000, 1.000, 1.000>]
        [0.50 srgb <0.000, 0.000, 0.000>]
        [0.63 srgb <0.000, 0.000, 0.000>]
        [0.63 srgbt <1.000, 1.000, 1.000, 1.000>]
        [0.90 srgb <1.000, 1.000, 1.000>]
        [0.90 srgb <0.451, 0.498, 0.451>]
        [0.91 srgb <1.000, 1.000, 1.000>]
        [0.91 srgbt <1.000, 1.000, 1.000, 1.000>]
        [1.00 srgb <1.000, 1.000, 1.000>]
      }
      scale 0.2
      translate <20.0, 20.0, 0.0>
      rotate <0.3, 0.2, 0.3>
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

#break
#case (2)
  #declare T1_StAndGPol =
  texture {
    pigment {
      granite//agate
      //agate_turb 0.7
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.000, 0.000, 0.000>))]
        //[0.65 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.059, 0.027, 0.047>))]
        [0.65 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.059, 0.047, 0.027>))]
        [0.65 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.067, 0.067, 0.047>))]
        [0.85 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.118, 0.118, 0.078>))]
        [0.85 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.149, 0.157, 0.129>))]
        [0.94 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.349, 0.259, 0.200>))]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.369, 0.259, 0.247>))]
      }
      scale 0.1//0.2
    }
    finish {
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }
  
  #declare T2_StAndGPol =
  texture {
    pigment {
      granite
      turbulence 0.4
      color_map {
        //[0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.157, 0.157, 0.200>)) transmit 0.149]
        //[0.45 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.000, 0.000, 0.047>)) transmit 0.749]
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.157, 0.200, 0.157>)) transmit 0.149]
        [0.45 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.000, 0.047, 0.000>)) transmit 0.749]
        [0.45 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [0.50 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [0.50 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.000, 0.000, 0.000>))]
        [0.63 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.000, 0.000, 0.000>))]
        [0.63 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [0.90 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>))]
        [0.90 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.451, 0.498, 0.451>))]
        [0.91 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>))]
        [0.91 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>)) transmit 1.000]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<1.000, 1.000, 1.000>))]
      }
      scale 0.2
      translate <20.0, 20.0, 0.0>
      rotate <0.3, 0.2, 0.3>
    }
    finish {
      diffuse 0.8
      brilliance 1.5
      specular 0.9
      roughness 0.005
    }
  }

  #declare StAndGPol_1 = texture {T1_StAndGPol}
  
  #declare StAndGPol_2 =
  texture {T1_StAndGPol}
  texture {T2_StAndGPol}

#end

//-------------------------------------------------------
// St. Andre Green granite - frosted surface
#switch (SCS)
#case (0)
  #declare StAndGFro =
  texture {
    pigment {
      granite
      //turbulence 0.7
      color_map {
        [0.00 rgb <0.098, 0.098, 0.098>]
        [0.35 rgb <0.259, 0.259, 0.259>]
        [0.35 rgb <0.329, 0.388, 0.357>]
        [0.54 rgb <0.420, 0.420, 0.376>]
        [0.54 rgb <0.376, 0.337, 0.310>]
        [1.00 rgb <0.518, 0.529, 0.498>]
      }
      scale 0.1
    }
    normal {
      granite
      //turbulence 0.7
      normal_map {
        [0.00 granite 0.1]
        [0.35 granite 0.2]
        [0.35 granite 0.4]
        [0.54 granite 0.6]
        [0.54 granite 0.8]
        [1.00 granite 1.0]
      }
      scale 0.1
    }
    finish {
      diffuse 0.6
      brilliance 1.6
    }
  }

#break
#case (1)
  #declare StAndGFro =
  texture {
    pigment {
      granite
      //turbulence 0.7
      color_map {
        [0.00 srgb <0.098, 0.098, 0.098>]
        [0.35 srgb <0.259, 0.259, 0.259>]
        [0.35 srgb <0.329, 0.388, 0.357>]
        [0.54 srgb <0.420, 0.420, 0.376>]
        [0.54 srgb <0.376, 0.337, 0.310>]
        [1.00 srgb <0.518, 0.529, 0.498>]
      }
      scale 0.1
    }
    normal {
      granite
      //turbulence 0.7
      normal_map {
        [0.00 granite 0.1]
        [0.35 granite 0.2]
        [0.35 granite 0.4]
        [0.54 granite 0.6]
        [0.54 granite 0.8]
        [1.00 granite 1.0]
      }
      scale 0.1
    }
    finish {
      diffuse 0.6
      brilliance 1.6
    }
  }

#break
#case (2)
  #declare StAndGFro =
  texture {
    pigment {
      granite
      //turbulence 0.7
      color_map {
        [0.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.098, 0.098, 0.098>))]
        [0.35 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.259, 0.259, 0.259>))]
        [0.35 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.329, 0.388, 0.357>))]
        [0.54 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.420, 0.420, 0.376>))]
        [0.54 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.376, 0.337, 0.310>))]
        [1.00 ColourSat(SatBoost, BrightBoost, scRGB_to_sRGB(<0.518, 0.529, 0.498>))]
      }
      scale 0.1
    }
    normal {
      granite
      //turbulence 0.7
      normal_map {
        [0.00 granite 0.1]
        [0.35 granite 0.2]
        [0.35 granite 0.4]
        [0.54 granite 0.6]
        [0.54 granite 0.8]
        [1.00 granite 1.0]
      }
      scale 0.1
    }
    finish {
      diffuse 0.6
      brilliance 1.6
    }
  }

#end

#version Granites_Inc_Temp;
#end
