/* srgb.inc version 1.0.1
 * Persistence of Vision Ray Tracer include file
 *
 * Utilities for converting colors in linear space to sRGB, including
 * hexadecimal string conversions.
 *
 * Usage note:  I do not know how relevant this file would be for any
 * assumed_gamma settings other than 1.0.
 *
 * Copyright © 2013 - 2015 Richard Callwood III.  Some rights reserved.
 * This work is licensed under the GNU General Public License 3.0.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See http://www.gnu.org/licenses/gpl-3.0.html for the legalese.
 *
 * Vers.  Date         Comments
 * -----  ----         --------
 *        2013-Dec-04  Created.
 * 1.0    2014-Mar-25  Prepared for public release.
 * 1.0.1  2015-Mar-25  A typo in sRGB_Convert_c() is corrected.
 */
#ifndef (sRGB_Inc_Temp) #declare sRGB_Inc_Temp = version;
#version 3.5;

#declare SRGB__HEX_DIGITS = "0123456789ABCDEF"

/*------------------------------------------------------------------------------
 * Converts a value 0...1 in linear space to sRGB.
 * From IEC 61966-2-1:1999.
 */
#declare sRGB_fn_Convert = function (x)
{ select
  ( 0.0031308 - x,
    1.055 * pow (x, 1/2.4) - 0.055,
    12.92 * x
  )
}

/*------------------------------------------------------------------------------
 * Converts a 3-D vector in linear space to sRGB.
 */
#macro sRGB_Convert_v (V)
  #local srgb_V = color V;
  < sRGB_fn_Convert (srgb_V.red),
    sRGB_fn_Convert (srgb_V.green),
    sRGB_fn_Convert (srgb_V.blue)
  >
#end

/*------------------------------------------------------------------------------
 * Converts a color in linear space to sRGB.
 */
#macro sRGB_Convert_c (C)
  #local srgb_C = color C;
  < sRGB_fn_Convert (srgb_C.red),
    sRGB_fn_Convert (srgb_C.green),
    sRGB_fn_Convert (srgb_C.blue),
    srgb_C.filter,
    srgb_C.transmit
  >
#end

/*------------------------------------------------------------------------------
 * Converts an integer to a hexadecimal string.
 * Arguments:
 *   I     - The number to convert.  If it is not an integer, it will be
 *           rounded.
 *   Field - The minimum length of the converted string.  The result will be
 *           zero-padded if necessary.
 */
#macro sRGB_Hex_integer_s (I, Field)
  #local srgb_Sgn = select (I, -1, 0, 1);
  #local srgb_I = floor (abs (I) + 0.5);
  #local srgb_S = "";
  #while (srgb_I > 0)
    #local srgb_S =
      concat (substr (SRGB__HEX_DIGITS, mod (srgb_I, 16) + 1, 1), srgb_S)
    #local srgb_I = floor (srgb_I / 16);
  #end
  #while (strlen (srgb_S) < Field)
    #local srgb_S = concat ("0", srgb_S)
  #end
  #if (srgb_Sgn < 0)
    concat ("-", srgb_S)
  #else
    srgb_S
  #end
#end

/*------------------------------------------------------------------------------
 * Converts a value 0...1 to a two-digit hexadecimal string, without converting
 * to sRGB.
 * If the value is negative, "--" is returned.
 * If the value is greater than 1.0, "++" is returned.
 */
#macro sRGB_Hex_linear_s (Value)
  #if (Value < 0)
    "--"
  #else #if (Value > 1)
    "++"
  #else
    sRGB_Hex_integer_s (Value * 255, 2)
  #end
  #end
#end

/*------------------------------------------------------------------------------
 * Converts a value 0...1 in linear space to an sRGB hexadecimal string.
 * If the value is negative, "--" is returned.
 * If the value is greater than 1.0, "++" is returned.
 */
#macro sRGB_Hex_convert_s (Value)
  sRGB_Hex_linear_s (sRGB_fn_Convert (Value))
#end

/*------------------------------------------------------------------------------
 * Converts a color to a 3-D hexadecimal string, without converting to sRGB.
 * If a channel value is negative, "--" is returned for that channel.
 * If a channel value is greater than 1.0, "++" is returned for that channel.
 */
#macro sRGB_Hex_straight_color_s (C)
  #local srgb_C = color (C);
  concat
  ( sRGB_Hex_linear_s (srgb_C.red),
    sRGB_Hex_linear_s (srgb_C.green),
    sRGB_Hex_linear_s (srgb_C.blue)
  )
#end

/*------------------------------------------------------------------------------
 * Converts a color in linear space to an sRGB 3-D hexadecimal string.
 * If a channel value is negative, "--" is returned for that channel.
 * If a channel value is greater than 1.0, "++" is returned for that channel.
 */
#macro sRGB_Hex_convert_color_s (C)
  sRGB_Hex_straight_color_s (sRGB_Convert_v (C))
#end

#version sRGB_Inc_Temp;
#end
