//====================================================================================
// Persistence of Vision Ray Tracer Scene Description File
// File: Wind_pattern.pov
// Vers: 3.7
// Desc: Utility for creating a wind pattern effect through rushes.
// Date: September 2014
// Auth: Thomas de Groot
//====================================================================================

#version 3.7;

#include "colors.inc"
#include "rand.inc"
#include "functions.inc"
#include "transforms.inc"
#include "math.inc"

// First you need to create an image map that will be used as basis for a height_field.
// Set Mapping to on to do this. Once you have a satisfactory map, set Mapping to off.

#declare Mapping = on;

//====================================================================================
// (1) Create an image map
//====================================================================================
// Render settings (right-click on a line below):
// +w640 +h480 +a0.1 +am2 +bm2 +wt6
// +w1024 +h1024 +a0.1 +am2 +bm2 +wt6 +ft

//use +ft for a tga file

#if (Mapping)

global_settings {
  assumed_gamma 1.0
  hf_gray_16
  number_of_waves 1
}

camera {
  orthographic
  location <0, 0, -10>
  look_at 0
}

#declare R = seed(19);
#declare Pig =
pigment {
  ripples
  color_map {
    [0.0 srgb <0.01,0.01,0.01>]
    [1.0 srgb <1,1,1>]
  }
  scale 1/50
  warp {
    turbulence <0.1,0.2,0>*5
    //octaves 9
    //lambda 1.5
    //omega 0.8
  }
  scale <1,4,1>
  rotate 35*z
  scale 50
  frequency 0.1
  poly_wave 1.5
  scale 1
}

#declare Dist = 5;

plane { 
  z, 0
  pigment {
    average
    pigment_map {
      [1.5 Pig]
      [RRand(0.5,1,R) Pig scale RRand(0.8, 1.2, R) translate <RRand(-Dist, Dist, R), RRand(-Dist, Dist, R), 0>]
      [RRand(0.5,1,R) Pig scale RRand(0.8, 1.2, R) translate <RRand(-Dist, Dist, R), RRand(-Dist, Dist, R), 0>]
      [RRand(0.5,1,R) Pig scale RRand(0.8, 1.2, R) translate <RRand(-Dist, Dist, R), RRand(-Dist, Dist, R), 0>]
      [RRand(0.5,1,R) Pig scale RRand(0.8, 1.2, R) translate <RRand(-Dist, Dist, R), RRand(-Dist, Dist, R), 0>]
      [RRand(0.5,1,R) Pig scale RRand(0.8, 1.2, R) translate <RRand(-Dist, Dist, R), RRand(-Dist, Dist, R), 0>]
    }
    scale 0.1
  }
  finish {emission 10}
}

//====================================================================================
// (2) Create a wind pattern
//====================================================================================
// Render settings (right-click on a line below):
// +w640 +h640 +a0.3 +am2 +bm2 +wt6
// +w1280 +h1280 +a0.3 +am2 +bm2 +wt6

#else

global_settings {
  assumed_gamma 1.0
  max_trace_level 5			//default [5]
  radiosity {
    pretrace_start 0.08           // start pretrace at this size
    pretrace_end   0.004           // end pretrace at this size
    count 100, 1000                      // higher -> higher quality (1..1600) [35]
    nearest_count 10, 5               // higher -> higher quality (1..10) [5]
    error_bound 1               // higher -> smoother, less accurate [1.8]
    recursion_limit 1             // how much interreflections are calculated (1..5+) [3]
    low_error_factor .3           // reduce error_bound during last pretrace step
    gray_threshold 0.0            // increase for weakening colors (0..1) [0]
    minimum_reuse 0.015           // reuse of old radiosity samples [0.015]
    maximum_reuse 0.1							// new parameter 3.7 [0.2]
    brightness 1                  // brightness of radiosity effects (0..1) [1]

    adc_bailout 0.01/2
    normal off                   // take surface normals into account [off]
    media off                    // take media into account [off]
    always_sample off           // turn sampling in final trace off [on]
    //max_sample 1.0              // maximum brightness of samples
  }
  //noise_generator 2
}

camera {
  location  <0, 100, -400.0>
  direction z*1.0
  right     x*image_width/image_height
  look_at   <0.0, 5.0,  0.0>
  angle 70
}

light_source {
  <0, 0, 0>
  color rgb <1, 1, 1>*2
  translate <-30, 20, -30>*1000
}

sky_sphere {
  pigment {
    gradient y
    color_map {
      [0.0 rgb <0.6,0.7,1.0>]
      [0.7 rgb <0.0,0.1,0.8>]
    }
  }
}

// A simple scene:
// Water:
plane {
  y, 5 
  hollow
  material {
    texture {
      pigment {srgb <0.74118, 0.71765, 0.41961>+<0.4, 0.1, 0> filter 0.4 }
      finish {
        conserve_energy  
        diffuse 0.2  
        ambient 0  
        reflection {0.01, 0.2 fresnel on metallic 0}
      }
      normal {crackle 0.15 scale <0.45,0.25,0.25>*2 turbulence 0.5 } 
    }
    //add interior and media to your taste:
    /*interior {
      ...
    }*/
  }
}

// Bottom:
plane {y, 0 
  pigment {
    bozo
    pigment_map {
      [0 srgb <0.5, 0.5, 0.1>]
      [1 srgb <0.7, 0.4, 0.1>]
    }
    scale 0.1
  }
}

//====================================================================================
// Building of the height_field:

#declare HF_scale = <800, 100, 600>;
#declare HF_res   = 2000;

#declare F_HF_01 =
function {
  pigment {
    image_map {
      // Read the image_map you made previously from your preferred folder:
      tga "Wind_pattern.tga" gamma 1.0
      map_type 0
      interpolate 2
    }
    /*
    warp {repeat x}
    warp {repeat y}
    scale 50
    warp {
      turbulence 0.2 
      octaves 1 //[6]
      lambda 1  //[2]
      omega 0.2 //[0.5]
    }
    scale 1/50
    */
  }
}

#declare hf =
intersection {
  height_field {
    function HF_res, HF_res {F_HF_01(x,y,z).hf}
    smooth
    translate <-0.5, 0, -0.5>
    scale <1, 1, 1>
    scale HF_scale
  }
  box {
    <-0.5, 0.0, -0.5>, <0.5, 1.0, 0.5>
    scale HF_scale * <0.99, 1, 0.99>
  }
  pigment {srgb <0.6, 0.6, 0.6>}
}

//object {hf}

#debug concat("\n  min_extent surface: <",vstr(3, min_extent(hf), ", ", 0,3),">\n")
#debug concat("  max_extent surface: <",vstr(3, max_extent(hf), ", ", 0,3),">\n\n")

//====================================================================================
// An object used for the selection of random locations:

#local Box = box {<-0.5, 100.0, -0.5>, <0.5, 100.1, 0.5> scale <HF_scale.x*0.98, 1, HF_scale.z*0.98>}
#local R = seed(0);

//====================================================================================
// Using the provided reed stalk:

#include "ReedMesh_03_ik_POV_geom.inc"

#macro Reed(Bend)

//random rotation for the whole stalk:
#local Rot = RRand(-180, 180, R);

//construction of the Inverse kinematic stalk:
#local Bend1 = Bend+((2*Bend)/10);
#local Plume =
union {
  object {ReedMesh_03_ik_Reed01_plume_ material{Reed01_plume_} hollow translate -10.40*y rotate Rot*y rotate Bend1*x translate (10.40-10.18)*y }
  object {ReedMesh_03_ik_Reed01_11_  material{Reed01_11_} hollow translate -10.18*y rotate Rot*y}
}

#local Bend2 = Bend+(Bend/10);
#local Top =
union {
  object {Plume rotate Bend2*x translate (10.18-9.86)*y}
  object {ReedMesh_03_ik_Reed01_10_  /*material{Reed01_10_}*/ hollow translate -9.86*y rotate Rot*y}
}

#local Stem9 =
union {
  object {Top rotate Bend*x translate (9.86-9.54)*y}
  object{ReedMesh_03_ik_Reed01_9_  /*material{Reed01_9_}*/ hollow translate -9.54*y rotate Rot*y}
}

#local Bend = Bend-(Bend/20);
#local Stem8 =
union {
  object {Stem9 rotate Bend*x translate (9.54-9.165)*y}
  object{ReedMesh_03_ik_Reed01_8_  /*material{Reed01_8_}*/ hollow translate -9.165*y rotate Rot*y}
}

#local Bend = Bend-(Bend/20);
#local Stem7 =
union {
  object {Stem8 rotate Bend*x translate (9.165-8.79)*y}
  object{ReedMesh_03_ik_Reed01_7_  /*material{Reed01_7_}*/ hollow translate -8.79*y rotate Rot*y}
}

#local Bend = Bend-(Bend/20);
#local Stem6 =
union {
  object {Stem7 rotate Bend*x translate (8.79-7.72)*y}
  object{ReedMesh_03_ik_Reed01_6_  /*material{Reed01_6_}*/ hollow translate -7.72*y rotate Rot*y}
}

#local Bend = Bend-(Bend/20);
#local Stem5 =
union {
  object {Stem6 rotate Bend*x translate (7.72-6.20)*y}
  object{ReedMesh_03_ik_Reed01_5_  /*material{Reed01_5_}*/ hollow translate -6.20*y rotate Rot*y}
}

#local Bend = Bend-(Bend/20);
#local Stem4 =
union {
  object {Stem5 rotate Bend*x translate (6.20-4.65)*y}
  object{ReedMesh_03_ik_Reed01_4_ /* material{Reed01_4_}*/ hollow translate -4.65*y rotate Rot*y}
}

#local Bend = Bend-(Bend/20);
#local Stem3 =
union {
  object {Stem4 rotate Bend*x translate (4.65-3.10)*y}
  object{ReedMesh_03_ik_Reed01_3_  /*material{Reed01_3_}*/ hollow translate -3.10*y rotate Rot*y}
}

#local Bend = Bend-(Bend/20);
#local Stem2 =
union {
  object {Stem3 rotate Bend*x translate (3.10-1.55)*y}
  object{ReedMesh_03_ik_Reed01_2_  /*material{Reed01_2_}*/ hollow translate -1.55*y rotate Rot*y}
}

#local Bend = Bend-(Bend/20);
#declare ReedMesh_03 =
union {
  object {Stem2 rotate Bend*x translate 1.55*y}
  object{ReedMesh_03_ik_Reed01_1_  /*material{Reed01_1_}*/ hollow rotate Rot*y}
  pigment {
    gradient y
    pigment_map {
      [0.2 srgb <0.8, 0.85, 0.5>]
      [0.3 srgb <RRand(0.50,0.55,R), RRand(0.80,0.85,R), RRand(0.2,0.3,R)>]
    }
    scale <1, 10, 1>
  }
}

#end  //macro

//====================================================================================
// Planting the reed stalks:

//wind strength (>0.0 to <=1.0):
#declare WindStrength = 0.3;

#for (N, 0, 15000)
  //random selection of a location:
  #local Norm = <0,0,0>;
  #local Start = VRand_In_Obj(Box,R);
  #local Pos = trace (hf, Start, -y, Norm);
  
  //basic correction:
  #if (Norm.x = 0.0 & Norm.y <= 0.0 & Norm.z = 0.0)
    #local Norm = <0.2, 1.0, 0.2>;
  #end
  
  //#debug concat("  Norm: <",vstr(3, Norm, ", ", 0,3),">\n")
  
  //calculation of angles:
  #local VerAng = VAngleD(<Norm.x, 0, Norm.z>, Norm);
  #local HorAng = VRotationD(<0, 0, 1>, Norm, y);
  #if(HorAng<0) #local HorAng = HorAng+180; #end
  
  //#debug concat("  VertAngle: ",str(VerAng, 5, 3),"\n")
  //#debug concat("  HoriAngle: ",str(HorAng, 5, 3),"\n\n")
  
  //bending and transformation of the reed stalk:
  Reed((90-VerAng)*WindStrength)
  object {ReedMesh_03 scale RRand(0.80, 1.20, R) scale 10 rotate HorAng*y translate <Pos.x, 0, Pos.z>}
#end


#end  //of Mapping switch