//================================================================================
// POV-Ray spectral render materials V0.1
//
// format for all specular definitions: 
// 36 columns for the wavelebgths of 380nm to 730nm in 10nm steps
//
// Ive, September 2012
//
//================================================================================


#ifndef (spectral_materials_inc) #declare spectral_materials_inc = 1;


//================================================================================
// parametric IOR / water, glasses and minerals
//
//================================================================================

// water 20C
// IOR (550nm) = 1.33468  input W^2
#declare ParamIOR_Water = function(x) {
  sqrt( 1 + 5.684027565e-1*x/(x-5.101829712e-3) + 1.726177391e-1*x/(x-1.821153936e-2) + 2.086189578e-2*x/(x-2.620722293e-2) + 1.130748688e-1*x/(x-1.069792721e1) )
}

// optical glass N-BAF10 (Schott)
// IOR (550nm) = 1.67306   input: W^2
#declare ParamIOR_Glass_BAF10 = function(x) {
  sqrt( 1 + 1.5851495*x/(x-0.00926681282)  + 0.143559385*x/(x-0.0424489805) + 1.08521269*x/(x-105.613573) )
}

// optical glass N-BAK1 (Schott)
// IOR (550nm) = 1.57462   input: W^2
#declare ParamIOR_Glass_BAK1 = function(x) {
  sqrt( 1 + 1.12365662*x/(x-0.00644742752) + 0.309276848*x/(x-0.0222284402) + 0.881511957*x/(x-107.297751) )
}

// optical glass N-BK7 (Schott)
// IOR (550nm) = 1.51852   input: W^2
#declare ParamIOR_Glass_BK7 = function(x) {
  sqrt( 1 + 1.03961212*x/(x-0.00600069867) + 0.231792344*x/(x-0.0200179144) + 1.01046945*x/(x-103.560653) )
}       

// flint glass F2 (Schott)
// IOR (550nm) = 1.62366  input: W^2
#declare ParamIOR_FlintGlass_F2 = function(x) {
  sqrt( 1 + 1.34533359*x/(x-0.00997743871) + 0.209073176*x/(x-0.0470450767) + 0.937357162*x/(x-111.886764) )
} 

// flint glass F5 (Schott)
// IOR (550nm) = 1.60679  input: W^2
#declare ParamIOR_FlintGlass_F5 = function(x) {
  sqrt( 1 + 1.3104463 *x/(x-0.00958633048) + 0.19603426 *x/(x-0.0457627627) + 0.96612977*x/(x-115.011883) )
} 

// dense flint glass N-SF2 (Schott)
// IOR (550nm) = 1.65176  input: W^2
#declare ParamIOR_FlintGlass_SF2 = function(x) {
  sqrt( 1 + 1.47343127*x/(x-0.0109019098) + 0.163681849*x/(x-0.0585683687) + 1.36920899*x/(x-127.404933) )
} 

// crown glass K7 (Schott)
// IOR (550nm) = 1.51293   input: W^2
#declare ParamIOR_CrownGlass_K7 = function(x) {
  sqrt( 1 + 1.1273555*x/(x-0.00720341707) + 0.124412303*x/(x-0.0269835916) + 0.827100531*x/(x-100.384588) )
}

// crown glass K10 (Schott)
// IOR (550nm) = 1.50327   input: W^2
#declare ParamIOR_CrownGlass_K10 = function(x) {
  sqrt( 1 + 1.15687082*x/(x-0.00809424251) + 0.0642625444*x/(x-0.0386051284) + 0.872376139*x/(x-104.74773) )
}

// dense crown glass SK5 (Schott)
// IOR (550nm) = 1.59119  input: W^2     
#declare ParamIOR_CrownGlass_SK5 = function(x) {
  sqrt( 1 + 0.991463823*x/(x-0.00522730467) + 0.495982121*x/(x-0.0172733646) + 0.987393925*x/(x-98.3594579) )
}            

// diamond 
// IOR (550nm) = 2.42295  input: W^2
#declare ParamIOR_Diamond = function(x) {
  sqrt( 1 + 4.3356*x/(x-pow(0.1060,2)) + 0.3306*x/(x-pow(0.1750,2)) )
}                       
 
// PMMA acrylic glass
// IOR (550nm) = 1.49363  input: W
#declare ParamIOR_Acryl = function(x) {
  sqrt( 2.399964 - 8.308636e-2*pow(x,2) - 1.919569e-1*pow(x,-2) + 8.720608e-2*pow(x,-4) - 1.666411e-2*pow(x,-6) + 1.169519e-3*pow(x,-8) )
}

//================================================================================

#if (WavelengthIndex >= 0) 

//================================================================================
// SPECTRAL MODE
//================================================================================


//================================================================================
// Diffuse reflectance for the 24 patches of the "classic" GretagMacbeth
// ColorChecker chart.  
// And to be safe: ColorChecker is a Registered Trademark of X-Rite
// and X-Rite is a Trademark and I'm not related with one or the other in any way. 
// But here we go:  1-4 rows  A-F columns
//
//================================================================================

// wavelength nm              380      390      400      410      420      430      440      450      460      470      480      490      500      510      520      530      540      550      560      570      580      590      600      610      620      630      640      650      660      670      680      690      700      710      720      730

#declare D_CC_A1 = array[36] {0.05475, 0.05833, 0.06116, 0.06238, 0.06231, 0.06207, 0.06183, 0.06159, 0.06154, 0.06162, 0.06203, 0.06296, 0.06518, 0.07027, 0.07640, 0.07949, 0.08128, 0.08429, 0.09058, 0.10290, 0.11905, 0.13426, 0.14320, 0.14688, 0.15077, 0.15810, 0.16819, 0.17890, 0.18755, 0.18964, 0.18577, 0.18149, 0.18161, 0.18721, 0.19605, 0.20949}
#declare D_CC_A2 = array[36] {0.05381, 0.05369, 0.05326, 0.05370, 0.05402, 0.05452, 0.05495, 0.05516, 0.05568, 0.05664, 0.05840, 0.06122, 0.06822, 0.08942, 0.12461, 0.15350, 0.17379, 0.19944, 0.24827, 0.33542, 0.44399, 0.53847, 0.58667, 0.59484, 0.59059, 0.58662, 0.58417, 0.58386, 0.58975, 0.60251, 0.62039, 0.63880, 0.65481, 0.66255, 0.66255, 0.66681}
#declare D_CC_A3 = array[36] {0.06624, 0.07864, 0.10159, 0.14554, 0.19951, 0.24440, 0.28250, 0.30936, 0.30759, 0.27781, 0.23087, 0.17754, 0.12971, 0.09428, 0.06954, 0.05399, 0.04582, 0.04167, 0.03944, 0.03831, 0.03775, 0.03775, 0.03797, 0.03852, 0.03925, 0.03987, 0.04080, 0.04232, 0.04418, 0.04547, 0.04584, 0.04642, 0.04837, 0.05217, 0.05731, 0.06498}
#declare D_CC_A4 = array[36] {0.18936, 0.25464, 0.42260, 0.66021, 0.81098, 0.86212, 0.87658, 0.88417, 0.89104, 0.89566, 0.89932, 0.90370, 0.90718, 0.90908, 0.91091, 0.91005, 0.91122, 0.91402, 0.91343, 0.91602, 0.91548, 0.91584, 0.91433, 0.91547, 0.91764, 0.91863, 0.92101, 0.92291, 0.92386, 0.92199, 0.92242, 0.92477, 0.92749, 0.92977, 0.93041, 0.93329}

#declare D_CC_B1 = array[36] {0.11713, 0.14345, 0.17453, 0.19093, 0.19560, 0.19900, 0.20423, 0.21318, 0.22842, 0.25127, 0.28005, 0.30878, 0.32945, 0.33336, 0.31460, 0.28628, 0.27349, 0.27646, 0.27720, 0.28930, 0.33938, 0.42022, 0.48779, 0.52511, 0.54574, 0.56156, 0.57788, 0.59497, 0.61180, 0.62475, 0.63810, 0.65596, 0.67822, 0.69958, 0.71709, 0.73382}
#declare D_CC_B2 = array[36] {0.12236, 0.16448, 0.22850, 0.28608, 0.32730, 0.36108, 0.38757, 0.39963, 0.39157, 0.36243, 0.31612, 0.26024, 0.20858, 0.16831, 0.13768, 0.11656, 0.10425, 0.09637, 0.08980, 0.08551, 0.08372, 0.08396, 0.08432, 0.08411, 0.08386, 0.08517, 0.08977, 0.09785, 0.10912, 0.12346, 0.14269, 0.16930, 0.20465, 0.24395, 0.28719, 0.33249}
#declare D_CC_B3 = array[36] {0.05195, 0.05306, 0.05420, 0.05545, 0.05677, 0.05856, 0.06137, 0.06576, 0.07483, 0.09269, 0.12488, 0.17789, 0.24579, 0.30725, 0.33716, 0.33354, 0.31653, 0.29299, 0.26186, 0.22999, 0.19765, 0.16504, 0.13501, 0.11490, 0.10397, 0.09791, 0.09439, 0.09235, 0.09277, 0.09653, 0.10240, 0.10842, 0.11345, 0.11533, 0.11392, 0.11427}
#declare D_CC_B4 = array[36] {0.17085, 0.23206, 0.36507, 0.50656, 0.56749, 0.58270, 0.58770, 0.59009, 0.59099, 0.58977, 0.58841, 0.58843, 0.58898, 0.58948, 0.59059, 0.59002, 0.58990, 0.59030, 0.58929, 0.59094, 0.59031, 0.58971, 0.58713, 0.58515, 0.58304, 0.57996, 0.57779, 0.57595, 0.57440, 0.57221, 0.57061, 0.56922, 0.56828, 0.56797, 0.56648, 0.56631}

#declare D_CC_C1 = array[36] {0.13036, 0.17707, 0.25101, 0.30625, 0.32392, 0.32993, 0.33283, 0.33097, 0.32342, 0.31134, 0.29823, 0.28533, 0.26943, 0.25037, 0.23144, 0.21426, 0.19942, 0.18451, 0.16938, 0.15729, 0.14911, 0.14482, 0.14186, 0.14057, 0.14067, 0.14109, 0.14257, 0.14654, 0.15184, 0.15351, 0.15009, 0.14395, 0.13639, 0.13235, 0.13496, 0.14673}
#declare D_CC_C2 = array[36] {0.09600, 0.11466, 0.13058, 0.13508, 0.13345, 0.13159, 0.13021, 0.12811, 0.12505, 0.12048, 0.11512, 0.10985, 0.10494, 0.09982, 0.09516, 0.09265, 0.09247, 0.09319, 0.09621, 0.10812, 0.15557, 0.26539, 0.39871, 0.50008, 0.55632, 0.57945, 0.58773, 0.59063, 0.59251, 0.59445, 0.59785, 0.60219, 0.60690, 0.60925, 0.60896, 0.61024}
#declare D_CC_C3 = array[36] {0.04992, 0.04877, 0.04759, 0.04724, 0.04716, 0.04735, 0.04742, 0.04692, 0.04607, 0.04518, 0.04444, 0.04429, 0.04468, 0.04560, 0.04678, 0.04764, 0.04859, 0.05037, 0.05385, 0.05986, 0.07212, 0.10356, 0.17752, 0.31207, 0.46683, 0.58083, 0.64443, 0.67484, 0.69018, 0.69824, 0.70592, 0.71495, 0.72370, 0.73010, 0.73371, 0.73841}
#declare D_CC_C4 = array[36] {0.14421, 0.19246, 0.27184, 0.33081, 0.35042, 0.35692, 0.36123, 0.36326, 0.36297, 0.36081, 0.35874, 0.35811, 0.35846, 0.35919, 0.36041, 0.36046, 0.36056, 0.36083, 0.36043, 0.36185, 0.36175, 0.36133, 0.35932, 0.35753, 0.35543, 0.35241, 0.34990, 0.34764, 0.34547, 0.34272, 0.34017, 0.33760, 0.33531, 0.33383, 0.33180, 0.33054}

#declare D_CC_D1 = array[36] {0.05124, 0.05423, 0.05599, 0.05704, 0.05786, 0.05895, 0.06030, 0.06131, 0.06228, 0.06325, 0.06478, 0.06738, 0.07531, 0.10120, 0.14536, 0.17826, 0.18394, 0.17011, 0.14938, 0.13274, 0.12186, 0.11517, 0.10948, 0.10536, 0.10434, 0.10599, 0.10891, 0.11189, 0.11406, 0.11395, 0.11240, 0.11215, 0.11482, 0.11977, 0.12459, 0.13030}
#declare D_CC_D2 = array[36] {0.09199, 0.11601, 0.14561, 0.16853, 0.17847, 0.17301, 0.15797, 0.13878, 0.11913, 0.10140, 0.08695, 0.07518, 0.06609, 0.06032, 0.05646, 0.05312, 0.05121, 0.05124, 0.05195, 0.05187, 0.05120, 0.05242, 0.05841, 0.07318, 0.09552, 0.11893, 0.14139, 0.16554, 0.19405, 0.22706, 0.26539, 0.30892, 0.35455, 0.39577, 0.43584, 0.47847}
#declare D_CC_D3 = array[36] {0.05798, 0.05442, 0.05216, 0.05198, 0.05263, 0.05398, 0.05608, 0.05942, 0.06659, 0.08068, 0.10688, 0.15204, 0.22507, 0.33553, 0.46239, 0.55873, 0.61573, 0.64973, 0.67222, 0.69387, 0.70995, 0.72319, 0.73144, 0.73904, 0.74620, 0.75180, 0.75816, 0.76394, 0.76869, 0.77098, 0.77551, 0.78240, 0.79018, 0.79619, 0.79930, 0.80366}
#declare D_CC_D4 = array[36] {0.10519, 0.13133, 0.16260, 0.18017, 0.18592, 0.18953, 0.19286, 0.19423, 0.19378, 0.19233, 0.19106, 0.19085, 0.19125, 0.19158, 0.19205, 0.19210, 0.19217, 0.19231, 0.19206, 0.19263, 0.19238, 0.19193, 0.19059, 0.18939, 0.18799, 0.18587, 0.18398, 0.18232, 0.18085, 0.17925, 0.17777, 0.17604, 0.17434, 0.17337, 0.17219, 0.17139}

#declare D_CC_E1 = array[36] {0.14423, 0.19827, 0.29443, 0.37544, 0.40837, 0.42095, 0.42618, 0.42609, 0.41932, 0.40343, 0.37927, 0.34636, 0.31112, 0.28124, 0.25388, 0.22889, 0.21420, 0.20835, 0.20162, 0.19440, 0.19257, 0.20018, 0.21441, 0.22952, 0.24058, 0.25396, 0.27851, 0.31322, 0.34779, 0.36587, 0.36579, 0.35942, 0.35799, 0.36493, 0.37723, 0.39783}
#declare D_CC_E2 = array[36] {0.06103, 0.06125, 0.06192, 0.06291, 0.06397, 0.06593, 0.06921, 0.07473, 0.08549, 0.10506, 0.13867, 0.19209, 0.27073, 0.37611, 0.47578, 0.53122, 0.54916, 0.54571, 0.52807, 0.50446, 0.47052, 0.42764, 0.38125, 0.34680, 0.32744, 0.31771, 0.31247, 0.30994, 0.31441, 0.32741, 0.34523, 0.36255, 0.37622, 0.38054, 0.37767, 0.37941}
#declare D_CC_E3 = array[36] {0.14455, 0.19511, 0.28259, 0.34577, 0.36182, 0.35432, 0.33361, 0.30571, 0.27623, 0.24756, 0.21805, 0.18988, 0.16786, 0.14896, 0.12697, 0.10723, 0.09962, 0.10189, 0.10356, 0.10907, 0.13680, 0.19963, 0.29013, 0.40006, 0.51580, 0.61486, 0.68655, 0.73177, 0.75975, 0.77433, 0.78314, 0.79256, 0.80337, 0.81155, 0.81718, 0.82541}
#declare D_CC_E4 = array[36] {0.06796, 0.07672, 0.08388, 0.08741, 0.08888, 0.09044, 0.09187, 0.09204, 0.09135, 0.09039, 0.08975, 0.08965, 0.08981, 0.08988, 0.08997, 0.08996, 0.09001, 0.09006, 0.08986, 0.08990, 0.08951, 0.08916, 0.08852, 0.08806, 0.08749, 0.08645, 0.08551, 0.08478, 0.08420, 0.08368, 0.08321, 0.08254, 0.08176, 0.08143, 0.08093, 0.08068}

#declare D_CC_F1 = array[36] {0.13627, 0.17946, 0.24689, 0.29682, 0.32028, 0.33708, 0.35550, 0.38119, 0.41913, 0.46596, 0.51048, 0.54581, 0.56719, 0.57426, 0.56908, 0.55068, 0.52351, 0.48843, 0.44521, 0.39987, 0.35043, 0.29939, 0.25243, 0.22096, 0.20431, 0.19579, 0.19088, 0.18823, 0.19072, 0.19942, 0.21159, 0.22310, 0.23164, 0.23332, 0.22941, 0.22935}
#declare D_CC_F2 = array[36] {0.06282, 0.06284, 0.06334, 0.06354, 0.06371, 0.06442, 0.06536, 0.06599, 0.06694, 0.06841, 0.07128, 0.07571, 0.08722, 0.12531, 0.20583, 0.30526, 0.38315, 0.43094, 0.46915, 0.51789, 0.56793, 0.60688, 0.62805, 0.63703, 0.63999, 0.64198, 0.64545, 0.64824, 0.65102, 0.65307, 0.65736, 0.66403, 0.67265, 0.67970, 0.68376, 0.68829}
#declare D_CC_F3 = array[36] {0.10773, 0.14119, 0.19247, 0.23641, 0.26085, 0.28550, 0.31740, 0.35313, 0.39024, 0.42597, 0.44561, 0.44423, 0.42321, 0.38549, 0.33672, 0.28273, 0.23128, 0.18506, 0.14554, 0.11807, 0.10053, 0.08958, 0.08156, 0.07640, 0.07406, 0.07305, 0.07294, 0.07381, 0.07559, 0.07675, 0.07648, 0.07499, 0.07277, 0.07200, 0.07374, 0.07935}
#declare D_CC_F4 = array[36] {0.03102, 0.03199, 0.03228, 0.03256, 0.03275, 0.03282, 0.03282, 0.03262, 0.03248, 0.03240, 0.03233, 0.03231, 0.03228, 0.03215, 0.03209, 0.03198, 0.03196, 0.03198, 0.03192, 0.03195, 0.03182, 0.03179, 0.03173, 0.03185, 0.03193, 0.03190, 0.03191, 0.03194, 0.03202, 0.03212, 0.03222, 0.03223, 0.03224, 0.03233, 0.03237, 0.03250}



//================================================================================
// more diffuse/specular reflectance data should go here ...
//
//================================================================================

// wavelength nm                 380      390      400      410      420      430      440      450      460      470      480      490      500      510      520      530      540      550      560      570      580      590      600      610      620      630      640      650      660      670      680      690      700      710      720      730

// todo... 


//================================================================================
// IOR tables:
//
//================================================================================

// wavelength nm                 380      390      400      410      420      430      440      450      460      470      480      490      500      510      520      530      540      550      560      570      580     590     600     610      620      630      640      650      660      670      680      690      700      710      720      730
                                                                                                                  
#declare IOR_15     = array[36] {1.513,   1.512,   1.511,   1.510,   1.509,   1.508,   1.507,   1.506,   1.506,   1.505,   1.504,   1.504,   1.503,   1.4502,  1.502,   1.501,   1.501,   1.500,   1.500,   1.499,   1.499,  1.498,  1.498,  1.498,   1.497,   1.497,   1.497,   1.496,   1.496,   1.496,   1.496,   1.495,   1.495,   1.495,   1.495,   1.494}


//================================================================================
// IOR functions: water, glasses and minerals
//
//================================================================================

// water (20C)
#declare IOR_Water           = array[36];

// glasses - taken from Schott (a German glass manufacturer) data sheets
// optical glass
#declare IOR_Glass_BAF10     = array[36];
#declare IOR_Glass_BAK1      = array[36];
#declare IOR_Glass_BK7       = array[36];
// flint glass
#declare IOR_FlintGlass_F2   = array[36];
#declare IOR_FlintGlass_F5   = array[36];
#declare IOR_FlintGlass_SF2  = array[36];
// crown glass
#declare IOR_CrownGlass_K7   = array[36];
#declare IOR_CrownGlass_K10  = array[36];
#declare IOR_CrownGlass_SK5  = array[36];

// minerals
#declare IOR_Diamond         = array[36];  

#declare IOR_Acryl           = array[36]; 


// fill the arrays 
#for (I, 0, 35)
  #local W  = 0.38 + I*0.01;
  #local W2 = pow(W,2);      
  
  #declare IOR_Water[I]           = ParamIOR_Water(W2);

  #declare IOR_Glass_BAF10[I]     = ParamIOR_Glass_BAF10(W2);
  #declare IOR_Glass_BAK1[I]      = ParamIOR_Glass_BAK1(W2);
  #declare IOR_Glass_BK7[I]       = ParamIOR_Glass_BK7(W2);
  #declare IOR_FlintGlass_F2[I]   = ParamIOR_FlintGlass_F2(W2);
  #declare IOR_FlintGlass_F5[I]   = ParamIOR_FlintGlass_F5(W2);
  #declare IOR_FlintGlass_SF2[I]  = ParamIOR_FlintGlass_SF2(W2);
  #declare IOR_CrownGlass_K7[I]   = ParamIOR_CrownGlass_K7(W2);
  #declare IOR_CrownGlass_K10[I]  = ParamIOR_CrownGlass_K10(W2);
  #declare IOR_CrownGlass_SK5[I]  = ParamIOR_CrownGlass_SK5(W2);

  #declare IOR_Diamond[I]         = ParamIOR_Diamond(W2);

  #declare IOR_Acryl[I]           = ParamIOR_Acryl(W);
#end 



//================================================================================
// dummy arrays for quick use with Spectral_XXX macros
//
//================================================================================

#declare Value_0 = array[36] {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0}
#declare Value_1 = array[36] {1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1}


//================================================================================
// spectral data set mixer
// example:
// F1 = 1.0 and F2 = 1.0 gives 50:50 of 100%
// F1 = 0.9 and F2 = 0.1 gives 90:10 of 100%
// F1 = 9   and F2 = 1   gives 90:10 of 100%
//
//================================================================================

#macro D_Average (SpecData1, F1, SpecData2, F2)
  #local F = F1 + F2; 
  
  #if ((F <= 0.0) | (F1 < 0) | (F2 < 0))
    error("D_Average: F1 + F2 has to be > 0.0");
  #end  

  #local FF1 = F1/F;
  #local FF2 = F2/F;

  #local Tmp = array[36];

  #for(I, 0, 35)
    #declare Tmp[I] = SpecData1[I]*FF1 + SpecData2[I]*FF2;
  #end
  
  Tmp
#end


#macro D_RGB (R, G, B)
  #local Tmp = array[36];

  #for(I, 0, 35)
    #declare Tmp[I] = D_CC_C3[I]*R*1.1 + D_CC_B3[I]*G*1.7 + D_CC_A3[I]*B*2.0;
  #end
  
  Tmp
#end


//================================================================================
// basic helper macros for the use with spectral data.
//
//================================================================================

#macro C_Spectral (Diffuse)
  rgb Diffuse[WavelengthIndex]
#end   


#macro C_Spectral_Filter (Diffuse, Filter)
  rgb Diffuse[WavelengthIndex] filter Filter[WavelengthIndex]
#end  


#macro C_Spectral_Transmit (Diffuse, Transmit)
  rgb Diffuse[WavelengthIndex] transmit Transmit[WavelengthIndex]
#end  


#macro IOR_Spectral (IOR)
  ior IOR[WavelengthIndex]
#end


#macro FadeColor_Spectral1 (Color)
  fade_color rgb Color[WavelengthIndex]
#end

#macro FadeColor_Spectral2 (Color)
  fade_color rgb pow(Color[WavelengthIndex],2)
#end

#macro FadeColor_Spectral (Color)
  fade_color rgb pow(Color[WavelengthIndex],3)
#end


//================================================================================
// metal helper macros for the use with spectral data.
//
//================================================================================

#macro D_Metal (N,K)
  
  #local Tmp = array[36];

  #for(I, 0, 35)
    #declare Tmp[I] = (pow(N[I]-1,2) + pow(K[I],2)) / (pow(N[I]+1,2) + pow(K[I],2));
  #end
  
  Tmp

#end               


#macro C_Metal (N,K)
  rgb  ( pow(N[WavelengthIndex]-1, 2) + pow(K[WavelengthIndex], 2) ) / 
       ( pow(N[WavelengthIndex]+1, 2) + pow(K[WavelengthIndex], 2) )
#end               


#macro IOR_Metal (N,K)
  ior pow(K[WavelengthIndex],2) / pow(N[WavelengthIndex],2)
#end


#macro EXT_Metal (N,K)
  K[WavelengthIndex]
#end


//================================================================================
//================================================================================

#else

//================================================================================
// PREVIEW MODE!
//
// the following is just for the "preview" mode using precalculated values
//================================================================================


//================================================================================
// GretagMacbeth ColorChecker chart.  
//================================================================================

#declare D_CC_A1 = <0.1724, 0.0840, 0.0574>;
#declare D_CC_A2 = <0.7167, 0.1991, 0.0270>;
#declare D_CC_A3 = <0.0234, 0.0491, 0.2916>;
#declare D_CC_A4 = <0.9165, 0.9155, 0.8706>;

#declare D_CC_B1 = <0.5480, 0.2987, 0.2172>;
#declare D_CC_B2 = <0.0650, 0.1062, 0.3924>;
#declare D_CC_B3 = <0.0653, 0.3017, 0.0644>;
#declare D_CC_B4 = <0.5820, 0.5912, 0.5840>;

#declare D_CC_C1 = <0.1099, 0.1970, 0.3361>;
#declare D_CC_C2 = <0.5417, 0.0884, 0.1204>;
#declare D_CC_C3 = <0.4293, 0.0317, 0.0404>;
#declare D_CC_C4 = <0.3553, 0.3610, 0.3591>;

#declare D_CC_D1 = <0.1040, 0.1504, 0.0523>;
#declare D_CC_D2 = <0.1043, 0.0440, 0.1397>;
#declare D_CC_D3 = <0.8570, 0.5752, 0.0078>;
#declare D_CC_D4 = <0.1876, 0.1924, 0.1917>;

#declare D_CC_E1 = <0.2242, 0.2178, 0.4305>;
#declare D_CC_E2 = <0.3547, 0.5078, 0.0483>;
#declare D_CC_E3 = <0.5030, 0.0887, 0.3055>;
#declare D_CC_E4 = <0.0871, 0.0899, 0.0910>;

#declare D_CC_F1 = <0.1235, 0.5191, 0.4047>;
#declare D_CC_F2 = <0.7804, 0.3540, 0.0216>;
#declare D_CC_F3 = <0.0000, 0.2489, 0.3831>;
#declare D_CC_F4 = <0.0318, 0.0320, 0.0327>;


//================================================================================
// more diffuse/specular reflectance data should go here ...
//
//================================================================================


//================================================================================
// IOR at 550nm
//
//================================================================================

#declare IOR_15      = 1.5;

//================================================================================
// parametric IOR at 550nm
//
//================================================================================

#local W2 = pow(0.55,2);

#declare IOR_Water           = ParamIOR_Water(W2);

#declare IOR_Glass_BAF10     = ParamIOR_Glass_BAF10(W2);
#declare IOR_Glass_BAK1      = ParamIOR_Glass_BAK1(W2);
#declare IOR_Glass_BK7       = ParamIOR_Glass_BK7(W2);
#declare IOR_FlintGlass_F2   = ParamIOR_FlintGlass_F2(W2);
#declare IOR_FlintGlass_F5   = ParamIOR_FlintGlass_F5(W2);
#declare IOR_FlintGlass_SF2  = ParamIOR_FlintGlass_SF2(W2);
#declare IOR_CrownGlass_K7   = ParamIOR_CrownGlass_K7(W2);
#declare IOR_CrownGlass_K10  = ParamIOR_CrownGlass_K10(W2);
#declare IOR_CrownGlass_SK5  = ParamIOR_CrownGlass_SK5(W2);

#declare IOR_Diamond         = ParamIOR_Diamond(W2);

#declare IOR_Acryl           = ParamIOR_Acryl(0.55);


//================================================================================
// dummy values for the use with Spectral_XXX macros
//
//================================================================================

#declare Value_0 = 0;
#declare Value_1 = 1;


//================================================================================
// rgb mixer
//
//================================================================================

#macro D_Average (SpecData1, F1, SpecData2, F2)
  #local F = F1 + F2; 
  
  #if ((F <= 0.0) | (F1 < 0) | (F2 < 0))
    error("D_Average: F1 + F2 has to be > 0.0");
  #end  

  #local FF1 = F1/F;
  #local FF2 = F2/F;

  (SpecData1*FF1 + SpecData2*FF2)
#end


#macro D_RGB (R, G, B)
  (D_CC_C3*R*1.1 + D_CC_B3*G*1.7 + D_CC_A3*B*2.0)
#end


//================================================================================
// basic helper macros for preview - corresponding to those with spectral data.
//
//================================================================================ 

#macro C_Spectral (Diffuse)
  rgb Diffuse
#end


#macro C_Spectral_Filter (Diffuse, Filter)
  rgb Diffuse filter Filter
#end  


#macro C_Spectral_Transmit (Diffuse, Transmit)
  rgb Diffuse transmit Transmit
#end  


#macro IOR_Spectral (IOR)
  ior IOR
#end


#macro FadeColor_Spectral1 (Color) 
  #local C = rgb Color;
  fade_color rgb <C.red, C.green, C.blue>
#end


#macro FadeColor_Spectral2 (Color) 
  #local C = rgb Color;
  fade_color rgb <pow(C.red,2), pow(C.green,2), pow(C.blue,2)>
#end

#macro FadeColor_Spectral (Color) 
  #local C = rgb Color;
  fade_color rgb <pow(C.red,3), pow(C.green,3), pow(C.blue,3)>
#end




//================================================================================
// metal helper macros for preview - corresponding to those with spectral data.
//
//================================================================================

#macro D_Metal (N,K)
  /*
  creating a linear spline "on the fly" as this is what CIE.inc (and the function
  Reflective2RGB() expects. It finally returns a simple rgb<> color.
  */

  #local TS = spline { linear_spline
  #for (I, 0, 35)
    #local W = 380 + I*10;
    W, ( ( pow(N[I]-1, 2) + pow(K[I], 2) ) / 
         ( pow(N[I]+1, 2) + pow(K[I], 2) ) )
  #end        
  }        
      
  Reflective2RGB(TS)   
   
#end  


#macro C_Metal (N,K)
  rgb D_Metal(N, K)
#end  


#macro C_Metal (N,K)
  rgb  ( pow(N[WavelengthIndex]-1, 2) + pow(K[WavelengthIndex], 2) ) / 
       ( pow(N[WavelengthIndex]+1, 2) + pow(K[WavelengthIndex], 2) )
#end               


#macro IOR_Metal (N,K)
  ior pow(K[17],2) / pow(N[17],2)
#end


#macro EXT_Metal (N,K)
  K[17]
#end


//================================================================================
//================================================================================

#end // (WavelengthIndex < 0)


//================================================================================ 
//
// HIGH LEVEL INTERFACE
// ********************
//
// macros that are the same for spectral and rgb preview rendering
// Use them as they are or just as a reference for creating your own.
//
// Note that I'm just a color guy and not an expert in material properties.
// As such those macros do reflect just my poor ideas and do claim by no means
// any accurate real world behaviour or scientific usefulness.
// ... but any hints on this matter would be highly appreciated ;)
//
//================================================================================  


#macro C_Average (Diffuse1, F1, Diffuse2, F2)
  C_Spectral(D_Average(Diffuse1, F1, Diffuse2, F2))
#end


#macro C_RGB (R, G, B)
  C_Spectral(D_RGB(R, G, B))
#end


#macro P_Spectral (Diffuse)
  pigment {C_Spectral(Diffuse)}
#end  


#macro P_Spectral_Filter (Diffuse, Filter)
  pigment {C_Spectral_Filter(Diffuse, Filter)} 
#end       


#macro P_Spectral_Transmit (Diffuse, Transmit)
  pigment {C_Spectral_Transmit(Diffuse, Transmit)} 
#end       


#macro T_Spectral_Matte (Diffuse)
  texture {
    P_Spectral (Diffuse)
    finish {
      ambient 0  emission 0  diffuse 1
    }
  }  
#end  


#macro T_Spectral_Emitting (Emission, Intensity)
  texture {
    P_Spectral (Emission)
    finish {
      ambient 0  emission Intensity  diffuse 0
    }  
  }
#end  


#macro M_Spectral_Matte (Diffuse)
  material {
    T_Spectral_Matte (Diffuse)
  }
#end


#macro M_Spectral_Emitting (Emission, Intensity)
  material {
    T_Spectral_Emitting (Emission, Intensity)
  }
#end  


#macro M_Spectral_Shiny (Diffuse, Reflect, IOR)
  material {                                                           
    texture {
      P_Spectral (Diffuse)
      finish {
        ambient 0  emission 0  diffuse 1-Reflect
        reflection {0, Reflect fresnel on} conserve_energy
      }  
    }
    interior { IOR_Spectral(IOR) }
  }
#end


#macro M_Spectral_Filter (Diffuse, IOR, FadeDist)
  material {                                                           
    texture {
      P_Spectral_Filter (Diffuse, Value_1)
      finish {
        ambient 0  emission 0  diffuse 0
        reflection {0 1 fresnel on} conserve_energy
      }  
    }
    interior {
      IOR_Spectral(IOR)
      fade_power 1001
      fade_distance FadeDist
      FadeColor_Spectral (Diffuse) 
    }
  }
#end    



//================================================================================
// Metals 
// ******
//
// Metals have no measured diffuse reflectance but a complex number where
// N (refraction) is the real part and K (extinction) the imaginary part.
// The actual (diffuse) reflectance is calculated "on the fly" from this complex
// number using this fresnel equation:
// R(w) = ( ( sqr(N(w)-1) + sqr(K(w)) ) / ( sqr(N(w)+1) + sqr(K(w)) ) )
//
//================================================================================


#macro M_Spectral_Metal (N, K, Reflectance)
  material {
    texture {                              
      pigment {C_Metal(N, K)}
      finish {
        ambient 0  emission 0  diffuse (0.5 - Reflectance*0.45)
        reflection {0 Reflectance fresnel on 
          #if (WavelengthIndex < 0) metallic #end  // !!!
        }
        conserve_energy
        brilliance EXT_Metal(N, K) 
        #if (WavelengthIndex < 0) metallic #end
      }
    }
    interior {IOR_Metal(N, K)}
  }
#end



#macro Spectral_Average (Data1, F1, Data2, F2)
  #local F = F1 + F2; 
  
  #if ((F <= 0.0) | (F1 < 0) | (F2 < 0))
    error("Spectral_Average: F1 + F2 has to be > 0.0");
  #end  

  #local FF1 = F1/F;
  #local FF2 = F2/F;

  #local Tmp = array[36];

  #for(I, 0, 35)
    #declare Tmp[I] = Data1[I]*FF1 + Data2[I]*FF2;
  #end
  
  Tmp
#end


#macro M_Spectral_Alloy (N1, K1, F1, N2, K2 F2, Reflectance) 
  #local N = Spectral_Average (N1, F1, N2, F2);
  #local K = Spectral_Average (K1, F1, K2, F2);
  M_Spectral_Metal (N, K, Reflectance)
#end



// wavelength nm                       380      390      400      410      420      430      440      450      460      470      480      490      500      510      520      530      540      550      560      570      580      590      600      610      620      630      640      650      660      670      680      690      700      710      720      730

#declare IOR_N_Aluminium = array[36] { 0.43745, 0.46265, 0.48784, 0.51304, 0.54101, 0.57035, 0.59969, 0.63324, 0.66843, 0.70362, 0.73927, 0.77592, 0.81257, 0.84921, 0.88783, 0.93029, 0.97274, 1.01519, 1.05764, 1.10625, 1.15827, 1.21030, 1.26232, 1.31434, 1.36645, 1.43031, 1.49417, 1.55803, 1.62703, 1.69751, 1.76800, 1.83981, 1.92139, 2.00298, 2.08456, 2.16738 }
#declare IOR_K_Aluminium = array[36] { 4.58661, 4.71107, 4.83552, 4.95998, 5.08430, 5.20855, 5.33281, 5.45435, 5.57484, 5.69533, 5.81463, 5.93134, 6.04806, 6.16477, 6.28104, 6.39645, 6.51187, 6.62728, 6.74270, 6.85478, 6.96502, 7.07526, 7.18550, 7.29573, 7.40592, 7.50808, 7.61025, 7.71241, 7.80423, 7.89304, 7.98186, 8.06881, 8.14197, 8.21514, 8.28831, 8.35851 }

#declare IOR_N_Chrome    = array[36] { 1.91761, 1.96178, 2.01050, 2.06210, 2.12128, 2.18583, 2.25115, 2.32444, 2.40701, 2.49579, 2.59167, 2.69222, 2.78191, 2.85932, 2.94188, 3.03334, 3.11515, 3.17240, 3.20804, 3.22244, 3.22139, 3.21044, 3.19542, 3.17844, 3.15981, 3.14055, 3.12199, 3.10440, 3.08994, 3.07652, 3.06769, 3.05966, 3.05680, 3.05478, 3.05748, 3.06059 }
#declare IOR_K_Chrome    = array[36] { 2.73668, 2.78563, 2.85060, 2.91028, 2.97307, 3.03354, 3.08587, 3.13556, 3.18391, 3.23125, 3.26771, 3.28974, 3.30703, 3.32214, 3.33077, 3.33198, 3.32972, 3.32657, 3.32179, 3.31481, 3.30712, 3.30027, 3.29792, 3.29795, 3.30008, 3.30794, 3.31690, 3.32738, 3.33847, 3.34977, 3.36054, 3.37091, 3.37853, 3.38608, 3.39329, 3.40061 }

#declare IOR_N_Copper    = array[36] { 1.20921, 1.18000, 1.18000, 1.18000, 1.17773, 1.17434, 1.17095, 1.16576, 1.15988, 1.15400, 1.14759, 1.14005, 1.13251, 1.12497, 1.10786, 1.07214, 1.03127, 0.94392, 0.85657, 0.73949, 0.60641, 0.47332, 0.40422, 0.33778, 0.27182, 0.25403, 0.23624, 0.21845, 0.21442, 0.21499, 0.21390, 0.21310, 0.21359, 0.21469, 0.21900, 0.22327 }
#declare IOR_K_Copper    = array[36] { 2.12561, 2.21000, 2.21000, 2.21000, 2.24407, 2.29492, 2.34576, 2.38965, 2.43082, 2.47200, 2.50804, 2.53317, 2.55829, 2.58342, 2.59848, 2.59402, 2.59041, 2.59449, 2.59857, 2.65074, 2.72881, 2.80688, 2.94993, 3.09569, 3.24132, 3.37322, 3.50512, 3.63702, 3.75051, 3.85785, 3.96461, 4.06863, 4.16176, 4.25455, 4.34545, 4.43598 }

#declare IOR_N_Nickel    = array[36] { 1.61000, 1.61000, 1.61000, 1.61000, 1.61472, 1.62000, 1.62000, 1.62878, 1.64090, 1.65220, 1.66163, 1.66689, 1.67792, 1.69725, 1.71604, 1.73382, 1.75184, 1.77224, 1.79265, 1.81194, 1.83060, 1.84925, 1.87603, 1.90315, 1.93031, 1.96137, 1.98941, 2.01294, 2.04471, 2.08000, 2.11158, 2.14350, 2.17850, 2.21350, 2.24850, 2.28364 }
#declare IOR_K_Nickel    = array[36] { 2.25551, 2.31200, 2.36000, 2.42015, 2.47775, 2.53471, 2.59353, 2.65390, 2.71452, 2.77102, 2.82958, 2.89274, 2.95575, 3.01855, 3.07964, 3.13742, 3.19514, 3.25229, 3.30943, 3.36582, 3.42179, 3.47776, 3.53532, 3.59295, 3.65056, 3.70646, 3.75882, 3.80588, 3.85706, 3.91000, 3.95737, 4.00450, 4.04950, 4.09450, 4.13950, 4.18318 }

#declare IOR_N_Silver    = array[36] { 0.19708, 0.18820, 0.17300, 0.17300, 0.16687, 0.15951, 0.15755, 0.15129, 0.14346, 0.13668, 0.13167, 0.13062, 0.13000, 0.13000, 0.12985, 0.12940, 0.12867, 0.12500, 0.12132, 0.12024, 0.12061, 0.12099, 0.12425, 0.12764, 0.13103, 0.13378, 0.13653, 0.13928, 0.14000, 0.14000, 0.14000, 0.14024, 0.14221, 0.14419, 0.14616, 0.14792 }
#declare IOR_K_Silver    = array[36] { 1.72102, 1.83800, 1.95000, 2.07030, 2.18077, 2.28288, 2.37438, 2.47024, 2.56723, 2.65763, 2.74611, 2.83032, 2.91763, 3.00942, 3.09720, 3.17720, 3.25735, 3.33898, 3.42061, 3.50015, 3.57851, 3.65687, 3.73159, 3.80617, 3.88083, 3.96339, 4.04596, 4.12853, 4.20928, 4.28939, 4.36950, 4.44889, 4.52296, 4.59704, 4.67111, 4.74537 }

#declare IOR_N_Gold      = array[36] { 1.68798, 1.67080, 1.65800, 1.64146, 1.62656, 1.60718, 1.57188, 1.50229, 1.41768, 1.31373, 1.18881, 1.01723, 0.85500, 0.70620, 0.57687, 0.48532, 0.39847, 0.35929, 0.32011, 0.29593, 0.28020, 0.26447, 0.24874, 0.23301, 0.21728, 0.20155, 0.18582, 0.17009, 0.16477, 0.16312, 0.16146, 0.16012, 0.16111, 0.16209, 0.16308, 0.16415 }
#declare IOR_K_Gold      = array[36] { 1.91693, 1.94000, 1.95600, 1.95750, 1.94951, 1.93412, 1.91059, 1.87854, 1.84374, 1.81549, 1.80318, 1.82634, 1.89546, 2.03072, 2.18347, 2.37013, 2.55249, 2.69127, 2.83004, 2.89942, 2.92975, 2.96009, 2.99043, 3.02076, 3.05110, 3.08144, 3.11178, 3.14211, 3.28287, 3.46243, 3.64199, 3.81630, 3.95210, 4.08790, 4.22370, 4.35783 }

#declare IOR_N_Platinum  = array[36] { 1.66103, 1.68832, 1.72022, 1.74261, 1.76887, 1.79654, 1.82268, 1.84756, 1.87181, 1.89441, 1.91816, 1.94447, 1.97386, 2.00768, 2.04062, 2.07187, 2.10285, 2.13130, 2.15976, 2.18433, 2.20672, 2.22910, 2.25278, 2.27651, 2.30025, 2.32479, 2.34933, 2.37387, 2.40686, 2.44267, 2.47848, 2.51356, 2.54319, 2.57281, 2.60244, 2.63200 }
#declare IOR_K_Platinum  = array[36] { 2.71573, 2.77664, 2.84060, 2.90030, 2.96246, 3.02471, 3.08353, 3.14390, 3.20452, 3.26102, 3.31958, 3.38274, 3.44377, 3.50174, 3.55973, 3.61777, 3.67407, 3.71472, 3.75537, 3.80582, 3.86179, 3.91776, 3.96881, 4.01966, 4.07058, 4.12887, 4.18715, 4.24543, 4.29512, 4.34196, 4.38879, 4.43593, 4.48531, 4.53469, 4.58407, 4.63322 }


// pure metals

#macro M_Aluminium (Reflectance)
  M_Spectral_Metal(IOR_N_Aluminium, IOR_K_Aluminium, Reflectance)
#end


#macro M_Chrome (Reflectance)
  M_Spectral_Metal(IOR_N_Chrome, IOR_K_Chrome, Reflectance)
#end


#macro M_Copper (Reflectance)
  M_Spectral_Metal(IOR_N_Copper, IOR_K_Copper, Reflectance)
#end


#macro M_Nickel (Reflectance)
  M_Spectral_Metal(IOR_N_Bickel, IOR_K_Nickel, Reflectance)
#end


#macro M_Silver (Reflectance)
  M_Spectral_Metal(IOR_N_Silver, IOR_K_Silver, Reflectance)
#end


#macro M_Gold (Reflectance)
  M_Spectral_Metal(IOR_N_Gold, IOR_K_Gold, Reflectance)
#end


#macro M_Platinum (Reflectance)
  M_Spectral_Metal(IOR_N_Platinum, IOR_K_Platinum, Reflectance)
#end


// commonly used alloys for jewelry

#macro M_GoldCopper_Alloy (CopperPercentage, Reflectance)
  #local GoldPercentage = 100 - CopperPercentage;
  M_Spectral_Alloy (
    IOR_N_Gold,   IOR_K_Gold,   GoldPercentage,
    IOR_N_Copper, IOR_K_Copper, CopperPercentage,
    Reflectance 
  )
#end


#macro M_GoldSilver_Alloy (SilverPercentage, Reflectance)
  #local GoldPercentage = 100 - SilverPercentage;
  M_Spectral_Alloy (
    IOR_N_Gold,   IOR_K_Gold,   GoldPercentage,
    IOR_N_Silver, IOR_K_Silver, SilverPercentage,
    Reflectance 
  )
#end


#macro M_GoldNickel_Alloy (NickelPercentage, Reflectance)
  #local GoldPercentage = 100 - NickelPercentage;
  M_Spectral_Alloy (
    IOR_N_Gold,   IOR_K_Gold,   GoldPercentage,
    IOR_N_Nickel, IOR_K_Nickel, NickelPercentage,
    Reflectance 
  )
#end


#macro M_GoldPlatinum_Alloy (PlatinumPercentage, Reflectance)
  #local GoldPercentage = 100 - PlatinumPercentage;
  M_Spectral_Alloy (
    IOR_N_Gold,   IOR_K_Gold,   GoldPercentage,
    IOR_N_Platinum, IOR_K_Platinum, PlatinumPercentage,
    Reflectance 
  )
#end



//================================================================================

#end  // ifndef (spectral_materials_inc)

