/*
 * @file 		lsystems.pov
 * @desc		A basic L-System parser and printer
 *              Here is a macro which parses L-System language string, useful to generate 
 *              old fashioned but interesting bidimensional fractals
 *
 * @ver 		1.0.2
 * @date 		15/01/2013
 * @note		English translation, a few comments
 * @auth 		Paolo Gibellini
 * @mailto 		p.gibellini@gmail.com
 * @history		1.0.1 - 16/04/2004 - Base version
 *
 * @ref 		An Introduction to Lindenmayer Systems (http://www.biologie.uni-hamburg.de/b-online/e28_3/lsys.html)
 * @ref 		Samples of LSE (http://www.kevs3d.co.uk/dev/lsystems/, http://en.wikipedia.org/wiki/L-system, http://www.nbb.cornell.edu/neurobio/land/OldStudentProjects/cs490-94to95/hwchen/)
 *
 * @todo		Adaptive scale, 3-d extension (http://math.stackexchange.com/questions/123642/representing-a-3d-hilbert-curve-as-an-l-system), a better documentation and a better parser... ;-)
 */

// Includes and global settings -----------------------------------------------
#include "colors.inc"
#include "textures.inc"
#include "metals.inc"
#include "golds.inc"

global_settings {
  assumed_gamma 1.0
}
// ----------------------------------------------------------------------------

// ToDo: adaptive camera ------------------------------------------------------
camera {
  location  <0, 0, -20>
  look_at   <2, 0,  0>
}

light_source {
  <0, 0, 0>
  color rgb <1, 1, 1>
  translate <-5, 10, -20>
}
// ----------------------------------------------------------------------------

// Textures -------------------------------------------------------------------
// Checkered texture
#declare t_purple = texture {
	pigment{color rgb<85,0,6>/255}
	finish {ambient 0.15 diffuse 0.85 phong 1}
}
#local hm=<0.95,0.95,1>;
#declare t_mirror=texture{
    pigment{color rgb hm}
    finish {  
        reflection 0.9
    }
    normal{
        agate 0.02
    }
    scale 0.1
}
#declare t_tex_orange=texture{
	checker
		texture{t_purple} 
		texture{t_mirror}
		scale 0.1
}
// Yellowish texture
#local h=<.3,.2,.1>;
#local h=<0.2,0.5,0.8>;
#local h=<1.0,0.9,0.4>;
#declare t_tex_sun=texture{ 
	pigment{
		function {y*(y-z)-x*(x+y)}
		colour_map{
			[.08 rgb h]
			[.1  rgb h*4.6]
			[.12 rgb h*8]
			[.32  rgb h*2.1]
			[.48 rgb h]
			[0.8 rgb 0]
		}
	}
	finish{
		ambient.35
		brilliance 2
		diffuse.3
		specular.8
		reflection.1
	}
	scale 0.1
}
// Greenish texture
#declare t_tex_green=texture {      
	pigment {
		agate
		color_map {
			[ 0.0     rgbt <0.3663, 0.5682, 0.0687, 1.0> ]
			[ 0.25    rgbt <0.3663, 0.5682, 0.0687, 0.82> ]
			[ 0.31    rgbt <0.0887, 0.9886, 0.4438, 0.01> ]
			[ 0.39    rgbt <0.1308, 0.9083, 0.5524, 0.2> ]
			[ 0.80    rgbt <0.1755, 0.5095, 0.448, 0.43> ]
			[ 1.0     rgbt <0.1755, 0.5095, 0.448, 1.0> ]
		}
		turbulence 0.6
		ramp_wave
	}
	normal {
		agate
		turbulence 0.6
		ramp_wave
	}      
	finish {
		ambient 0.1
		diffuse 0.5
		phong_size 81.0
		roughness 0.0
	}
	scale 0.3
}
// Background texture
#local h2=<0.2,0.5,0.8>;
#declare t_blue_moon_light=texture{
	pigment {
		agate
		color_map{
			[ 0 rgb h2*.98]
			[.3 rgb h2*.76]
			[.5 rgb h2*.65]
			[.7 rgb h2*.76]
			[ 1 rgb h2*.98]
		}
		turbulence .61
		scale 2.3
	}
	normal {
		agate
		turbulence .61
		scale 2.3
	}
}
// Golden texture
#declare t_golden_thing=texture{
 	T_Gold_2A
    normal {granite .75 scale 0.15 }
}
// ----------------------------------------------------------------------------

// Defaults -------------------------------------------------------------------
#declare t_coord=<0,0,0>; 	// Current coordinates
#declare t_angle=0;			// Current angle
#declare t_lpath=1;       	// Path length
#declare t_dir=6;			// Rotation=360/angle increment
#declare t_level=0;			// current level
#declare l_dim=0.05;		// Line diameter
#declare r_init="=";		// Rule init
#declare r_end=";";			// Rule end
#declare n_rules=0;			// Num rules
#declare n_max_rules=100;	// Max num rules
#declare n_max_stacks=200;	// Max stacks number
#declare a_coord_stack=array[n_max_stacks];		// Coordinates Stack
#declare a_angle_stack=array[n_max_stacks];		// Angles stack
#declare n_curr_stack=0;
#declare t_rules="";		// Final rule
// ----------------------------------------------------------------------------

// Drawing functions ----------------------------------------------------------

/**
 * Draw a line between two points with a given diameter (the line is a rounded cylinder of given radius), texture, translation, rotation
 *
 * @param	from			Line starting point
 * @param	to				Line ending point
 * @param	rx		 	 	Line radius
 * @param	tex				Texture
 * @param	transl			Translation
 * @param	rotat			Rotation
 */
#macro lineto(from, to, rx, tex, transl, rotat)
    union {
    	cylinder {from, to, rx}
    	sphere {from, rx}
    	sphere {to, rx}
	    texture{tex}
	    translate transl
	    rotate rotat
	}
#end

/**
 * In a turtle geometry the cursor movement is treated with the analogy of a turtle on a cartesian plane
 *
 * @param	rule			Rule to be followed by the turtle
 * @param	tex				Texture
 * @param	transl			Translation
 * @param	rotat			Rotation
 */
#macro turtle_move(rule, tex, transl, rotat)
	#local n_rules=strlen(rule);
	#local n_currule=0;
	#while(n_currule<n_rules)
		#local n_currule=n_currule+1;
		#switch (asc(substr(rule,n_currule,1)))
			#case (asc("F"))	// Forward and line
				#local old_t_coord=t_coord;
				#declare t_coord=z*t_coord+x*(t_coord+t_lpath*cos(t_angle*pi/180))+y*(t_coord-t_lpath*sin(t_angle*pi/180));
				lineto(old_t_coord,t_coord,l_dim/2, tex, transl, rotat)
			#break
			#case (asc("f"))	// Forward
				#declare t_coord=z*t_coord+x*(t_coord+t_lpath*cos(t_angle*pi/180))+y*(t_coord+t_lpath*sin(t_angle*pi/180));
			#break
			#case (asc("+"))	// Angle increment
				#declare t_angle=t_angle+360/t_dir;
			#break
			#case (asc("-"))	// Angle decrement
				#declare t_angle=t_angle-360/t_dir;
			#break
			#case (asc("|"))	// Angle increment of 180
				#declare t_angle=t_angle+180;
			#break
			#case (asc("*"))	// Increase step size by 10%
				#declare t_lpath=t_lpath*1.1;
			#break
			#case (asc("/"))	// Decrease step size by 10%
				#declare t_lpath=t_lpath/1.1;
			#break
			#case (asc(","))	// Increase step size randomly - ToDo
				#declare t_lpath=t_lpath;
			#break
			#case (asc("["))	// Stack push - only n_max_stacks levels
				#declare a_coord_stack[n_curr_stack]=t_coord;
				#declare a_angle_stack[n_curr_stack]=t_angle;
				#declare n_curr_stack=n_curr_stack+1;
			#break
			#case (asc("]"))	// Stack pop - only n_max_stacks levels
				#declare n_curr_stack=n_curr_stack-1;
				#declare t_coord=a_coord_stack[n_curr_stack];
				#declare t_angle=a_angle_stack[n_curr_stack];
			#break
		#end
	#end
#end
// ----------------------------------------------------------------------------

// String rewriting system ----------------------------------------------------

/**
 * Replacement of a char in a string
 *
 * @param	the_string		Our string
 * @param	old_char		Char to be replaced
 * @param	new_char		Replacement
 *
 * @set		str_replaced	The string with the replacement
 *
 * @todo	Replacement of a substring
 */
#macro str_replace(the_string,old_char,new_char)
	#local n=0;
	#local n_tmp=strlen(the_string);
	#declare str_replaced="";
	#while(n<n_tmp)
		#local n=n+1;
		#if(asc(substr(the_string,n,1))=asc(old_char))
			#declare str_replaced=concat(str_replaced,new_char);
		#else
			#declare str_replaced=concat(str_replaced,substr(the_string,n,1));
		#end
	#end
#end

/**
 * Position of a char in a string
 *
 * @param	the_string		Our string
 * @param	the_char		Our char
 *
 * @set		str_position	The position of the char starting from 1
 *
 * @todo	Position  of a substring
 */
#macro str_pos(the_string,the_char)
	#local n=0;
	#local n_tmp=strlen(the_string);
	#declare str_position=0;
	#while(n<n_tmp)
		#local n=n+1;
		#if(asc(substr(the_string,n,1))=asc(the_char))
		    #declare str_position=n;
		    #local n=n_tmp;
		#end
	#end
#end

/**
 * Split the rule in two arrays (a_rules_k and a_rules_v) for a further use
 *
 * @param	rules	 		Rules are defined by "char=rule;", e.g. "F=F-F++F-F;" or "F=FXF;X=[-F+F+F]+F-F-F+;"
 *
 * @set		a_rules_k		Array with rules variable	(e.g. a_rules_k[0]="F")
 * @set		a_rules_v		Array with rules			(e.g. a_rules_v[0]="F-F++F-F")
 *
 * @oss		There is no formal controls, use a max of n_max_rules rules (see defaults section)
 */
#macro fill_array_rules(rules)
	// Generate temporary array
	#declare n_rules=0;
	#local n_tmp=strlen(rules);
	#local n=0;
	#local k_tmp="";
	#local v_tmp="";
	#local is_k=1;
	#local is_v=0;
	#local a_tmp_k=array[n_max_rules];
	#local a_tmp_v=array[n_max_rules];
	// Count rules to dimensionate array
	#while(n<n_tmp)
		#local n=n+1;
		#switch (asc(substr(rules,n,1)))
			#case(asc(r_init))	// Init a rule
				#local is_k=0;
				#local is_v=1;
			#break
			#case(asc(r_end))	// End a rule: store the data
				#declare n_rules=n_rules+1;
				#local is_k=1;
				#local is_v=0;
				#local a_tmp_k[n_rules-1]=k_tmp;
				#local a_tmp_v[n_rules-1]=v_tmp;
				#local k_tmp="";
				#local v_tmp="";
			#break
			#case(asc(" "))		// Avoid garbage
			#break
			#else				// Concat data
				#if(is_k=1)
					#local k_tmp=concat(k_tmp,substr(rules,n,1));
				#end
				#if(is_v=1)
					#local v_tmp=concat(v_tmp,substr(rules,n,1));
				#end
		#end           
	#end

	// Declares rules array
	#declare a_rules_k=array[n_rules];
	#declare a_rules_v=array[n_rules];
	// Fills array
	#local i=0;
	#while(i<n_rules)
		#declare a_rules_k[i]=a_tmp_k[i];
		#declare a_rules_v[i]=a_tmp_v[i];
		#local i=i+1;
	#end
#end

/**
 * Rewriting system
 *
 * @param	axiom			Initial axiom
 * @param	a_rules_k		Array with rules variable	(e.g. a_rules_k[0]="F")
 * @param	a_rules_v		Array with rules			(e.g. a_rules_v[0]="F-F++F-F")
 * @param	n_step			Number of steps
 *
 * @set		t_rule			Final rule
 *
 * @oss		This rewriting system is probably inadequate for complex cases
 */
#macro parse_string(axiom, a_rules_k, a_rules_v, n_step)
	#local applied_axiom=axiom;
	// Sobstitution of rules to axiom
	#local i_s=0;
	#while(i_s<n_step)
		#local i=0;
		#local init_axiom=applied_axiom;
		#while(i<n_rules)
		    str_pos(init_axiom,a_rules_k[i])
		    #if(str_position>0)
			    str_replace(applied_axiom,a_rules_k[i],a_rules_v[i])
			    #local applied_axiom=str_replaced;
			#end
			#local i=i+1;
		#end
		#local i_s=i_s+1;
	#end
	// Updates final rules
	#declare t_rules=applied_axiom;
#end 
// ----------------------------------------------------------------------------

// Tools ----------------------------------------------------------------------

/**
 * Init the turtle
 *
 * @param	axiom			Initial axiom
 * @param	rules	 		Rules are defined by "char=rule;", e.g. "F=F-F++F-F;" or "F=FXF;X=[-F+F+F]+F-F-F+;"
 * @param	directions		Rotation (360/angle increment)
 * @param	n_step			Number of steps
 * @param	tex				Texture
 * @param	transl			Translation
 * @param	rotat			Rotation
 */
#macro init_turtle(axiom, rules, directions, n_step, tex, transl, rotat)
	#declare t_dir=directions;
	// By now...
	#declare t_lpath=t_lpath/n_step;
	fill_array_rules(rules)
	parse_string(axiom, a_rules_k, a_rules_v, n_step)
	turtle_move(t_rules, tex, transl, rotat)
#end


/**
 * L-System parser
 *
 * @param	axiom			Initial axiom
 * @param	rules	 		Rules are defined by "char=rule;", e.g. "F=F-F++F-F;" or "F=FXF;X=[-F+F+F]+F-F-F+;"
 * @param	directions		Rotation (360/angle increment)
 * @param	n_step			Number of steps
 * @param	tex				Texture
 * @param	transl			Translation
 * @param	rotat			Rotation
 * @param	curr_pos		Current position
 * @param	curr_angle		Current angle 
 * @param	path_len		Path length
 * @param	line_diameter	Line diameter
 */
#macro l_system(axiom, rules, directions, n_step, tex, transl, rotat, curr_pos, curr_angle, path_len, line_diameter)
	#declare t_coord=curr_pos;
	#declare t_angle=curr_angle;
	#declare t_lpath=path_len;
	#declare l_dim=line_diameter;
	union {
		init_turtle(axiom, rules, directions, n_step, tex, transl, rotat)
	}
#end
// ----------------------------------------------------------------------------


// Sample scene ---------------------------------------------------------------

// Background -----------------------------------------------------------------
sky_sphere {
  pigment {
    gradient y
    color_map {
      [0.0 rgb <0.6,0.7,1.0>]
      [0.7 rgb <0.0,0.1,0.8>]
    }
  }
}

plane {
  z, 400
  texture{t_blue_moon_light}
}
// ----------------------------------------------------------------------------

// Sample Output --------------------------------------------------------------
//#declare the_koch=l_system("F+F+F+F", "F=F+F-F-FF+F+F-F;", 4, 2, texture{t_tex_orange scale 0.3}, 0, 0, <0,0,0>, 0, 0.3, 0.05)
//#declare the_dragon=l_system("F+X", "X=X+YF;Y=FX-Y;", 4, 10, t_golden_thing, 0, 0, <0,0,0>, 0, 1, 0.04)
#declare the_sun=l_system("X+X+X+X+X+X+X+X+X+X+X+X+X+X+X+X+X+X+X+X+X+X+X+X", "X=[F+F+F+F[---X-Y]+++++F++++++++F-F-F-F];Y=[F+F+F+F[---Y]+++++F++++++++F-F-F-F];", 24, 2, t_tex_sun, 0, 0, <0,0,0>, 0, 1, 0.1)
#declare candy_cane=l_system("F+F+F", "F=F-F++F-F;", 5, 4, t_tex_orange, <-1.2,-1,-19>, 0, <0,0,0>, 0, 0.5, 0.05)
#declare the_thing=l_system("F++F++F++F++F", "F=F++F++F|F-F++F;", 10, 2, t_golden_thing, 0, 0, <0,0,0>, 0, 2, 0.3)
#declare the_tree=l_system("X", "X=F[+X]F[-X]+X;F=FF;", 20, 6, texture{t_tex_green}, 0, 0, <0,0,0>, 0, 1, 0.2)

// Scene
//object{the_koch scale 5 translate <-2,6,1> no_shadow}
//object{the_dragon scale 1 translate <0,-6,-4> no_shadow}
object{the_sun scale 8 translate <6,60,100> no_shadow}
object{the_tree rotate 70*z translate <-14,-20,0>}
object{the_tree rotate 70*z translate <0,-21,0> scale <-0.7,1.3,1> translate <20,-10,2>}
object{the_thing scale 1  rotate 36*z translate <-3,-8,0> no_shadow}
object{candy_cane rotate -20*z translate <0,0.5,0>}
// ----------------------------------------------------------------------------

