#include "shapes.inc"

// wrap a value
#macro rounded_lprism_WrapV(V, Repeat)
 mod(V+Repeat, Repeat)
#end

#declare rounded_lprism_autofit_en = on;

// line intersection for a pair of 2D lines <A,B>, C,D>
#macro rounded_lprism_LineIntersect2D(A,B,C,D)
 #if(A.x-B.x != 0)
  #local m1 = (A.y-B.y)/(A.x-B.x);
 #else
  #local m1 = 1e10;
 #end
 #if(C.x-D.x != 0)
  #local m2 = (C.y-D.y)/(C.x-D.x);
 #else
  #local m2 = 1e10;
 #end
 #local b1 = A.y - m1*A.x;
 #local b2 = C.y - m2*C.x;
 #local slope_intercept_x = (b2-b1)/(m1-m2);
 #local slope_intercept_y = m1*slope_intercept_x + b1;
 <slope_intercept_x, slope_intercept_y>
#end

// IArc_2D(A,B,C)
// fits an arc into the equal sides of an an Isosceles triangle based on three 2D points
// line A, B: assumed symmetric bisecting line, A = apex, B = bisecting line point
// line A, C: assumed equal side,               A = apex, C = end point
#macro rounded_lprism_IArc_2D(A,B,C)
 #local perpC = C+VPerp_To_Vector(<C.x,C.y,0>-<A.x,A.y,0>);
 #local intersect = rounded_lprism_LineIntersect2D(A,B,C,perpC);
 #local bisector = vnormalize(B-A);
 #declare rounded_lprism_IArc_radius = vlength( intersect - C );
 #declare rounded_lprism_IArc_offset = A+bisector*vlength(A-intersect);
 #declare rounded_lprism_IArc_angle  = VAngleD( A-intersect, C-intersect )*2;
#end // out: IArc_radius, IArc_offset, IArc_angle

#macro rounded_lprism( pts, z1, z2, eRad_default, inset)
 
 // set a minimum value for safety
 #local inset = max(abs(inset)-0.00001,0.00125);
 
 // set a minimum value
 #local eRad_default = max(abs(eRad_default),0.005);
 
 // correct the depths
 #if(z2<z1) #local tmp = z2; #local z2 = z1; #local z1 = tmp; #end
 
 // this would force eRad_default to be a little over inset
 //#local eRad_default = max(abs(eRad_default),inset+0.0001);
 
 #macro rounded_lprism_get_metrics(v2,V)
  
  #local V = rounded_lprism_WrapV(V,dimension_size(pts[v2],1));
  #local VPlus = rounded_lprism_WrapV(V+1,dimension_size(pts[v2],1));
  #local VMinus = rounded_lprism_WrapV(V-1,dimension_size(pts[v2],1));
  
  // if a point-specific arc radius is available, use it, otherwise use default
  #if(pts[v2][V].z!=0)
   #declare rounded_lprism_eRad = pts[v2][V].z;
  #else
   #declare rounded_lprism_eRad = eRad_default;
  #end
  
  // point CCW, middle, CW
  #local ptL   = < pts[v2][VMinus].x, pts[v2][VMinus].y, 0 >;
  #local ptM = < pts[v2][V].x, pts[v2][V].y, 0>;                              // middle point
  #local ptR   = < pts[v2][VPlus].x, pts[v2][VPlus].y, 0 >;
  
  // the actual corner normal, point B for arc
  #declare rounded_lprism_cNA = ptM + vnormalize( vnormalize(ptR-ptM) + vnormalize(ptL-ptM) );
  
  
  // find center of CCW, CW line segments
  #local rounded_lprism_cePtL = (ptL+ptM)/2;
  #local rounded_lprism_cePtR = (ptR+ptM)/2;
  
  // intersection of inset lines
  #local li =
  rounded_lprism_LineIntersect2D(
   ptM, rounded_lprism_cNA,
   rounded_lprism_cePtL+VPerp_To_Vector(ptM-ptL)*inset,
   ptM+VPerp_To_Vector(ptM-ptL)*inset
  );
   
  // length of line from middle point to the point perpendicular to li
  #local ilength =
  vlength(
   (li +VPerp_To_Vector(ptM-ptR)*inset )
   -ptM
  )+0.00001;
   
  #declare rounded_lprism_corner_point_is_outside = 
  VEq(
   ptM + VPerp_To_Vector( (ptM + vnormalize(ptR-ptM)*rounded_lprism_eRad) - (ptM + vnormalize(ptL-ptM)*rounded_lprism_eRad ) ),
   rounded_lprism_cNA
  );
   
  // test for outward angle and enforce minimum distance
  #if( rounded_lprism_corner_point_is_outside )
   #declare rounded_lprism_eRad = max(rounded_lprism_eRad,ilength);
  #end
   
  // the length from middle point to the segment's center
  #local distL = vlength(rounded_lprism_cePtL-ptM);
  #local distR = vlength(rounded_lprism_cePtR-ptM);
   
  // a minimum distance
  #local distMin = min(distL,distR);
  
  // autofit
  #if(rounded_lprism_autofit_en)
  
   // enforce maximum distance
   #declare rounded_lprism_eRad = min(rounded_lprism_eRad,distMin);
   
  #end // ~autofit

  // extend from corners CW, CCW by eRad
  #declare rounded_lprism_eCW  = ptM + vnormalize(ptR-ptM)*rounded_lprism_eRad;
  #declare rounded_lprism_eCCW = ptM + vnormalize(ptL-ptM)*rounded_lprism_eRad;
  
  // perpendicular extension from eCW, point C for arc
  #declare rounded_lprism_epCW =  rounded_lprism_eCW + VPerp_To_Vector( rounded_lprism_eCW - ptM )*inset;
  
  // perpendicular extension from eCCW
  #declare rounded_lprism_epCCW = rounded_lprism_eCCW + VPerp_To_Vector( ptM - rounded_lprism_eCCW)*inset;
  
  // inward corner normal
  #local cN = VPerp_To_Vector( rounded_lprism_eCW - rounded_lprism_eCCW );
  
  // perpendicular extension from epCW
  #local backtrack = rounded_lprism_epCW+VPerp_To_Vector( rounded_lprism_epCW-rounded_lprism_eCW )*rounded_lprism_eRad*2; // might need to be changed for highly-acute angles
  
  // finds point A for arc, among other things
  #local pA = ptM;
  #local pB = ptM+cN*.5;
  #local pC = rounded_lprism_epCW;
  #local pD = backtrack;
  #declare rounded_lprism_cNIsect = rounded_lprism_LineIntersect2D(pA,pB,pC,pD);
  
 #end // get_metrics out: eRad, cNA, eCW, eCCW, epCW, epCCW, cNIsect
 
 // object construction
 union{
  // **** joins ****
  #macro join_object(p1,p2)
   union{
    #local cptrans = transform{Point_At_Trans(p2-p1) translate p1}
    #local join_piece=
    #if(rounded_lprism_type = 1)
     cylinder{
      0,y*vlength(p2-p1),inset open
      // clipping for transparency
      clipped_by{
       plane{-x,0}
       plane{z,0}
      }
      transform{cptrans}
     }
    #else
     box{<0,0,-inset>,<inset,vlength(p2-p1),-inset> matrix<1,0,1,0,1,0,0,0,1,0,0,0> transform{cptrans} inverse}
    #end
    object{ join_piece }
    object{ join_piece scale<1,1,-1> translate z*(z2-z1-inset*2)}
    box{
     <0,0,0>,
     <0,vlength(p2-p1),z2-z1-inset*2>
     translate x*inset
     transform{cptrans}
     inverse
    }
    translate z*inset
   }
  #end
  
  // objects loop
  #local n_poly_pts_total = 0;
  #local v2=0;
  #while(v2<dimension_size(pts,1))
   
   #local n_poly_pts = array[dimension_size(pts,1)];
   #local n_poly_pts[v2]=0;
   
   // number of points in array
   #local npts1  = dimension_size(pts[v2],1);
   #local V      = 0;
   #local new_pt = <0,0>;
   #while(V < npts1)
    rounded_lprism_get_metrics(v2,V)
    rounded_lprism_IArc_2D(rounded_lprism_cNIsect, rounded_lprism_cNA, rounded_lprism_epCW)
    #if(V=0) #local very_first_point = rounded_lprism_epCCW; #end
    #local old_pt = new_pt;
    #local new_pt = rounded_lprism_epCW;
    #if(V>0)
     #if( !VEq(old_pt,rounded_lprism_epCCW) )
      object{join_object(rounded_lprism_epCCW,old_pt)}
     #end
    #end
    #local last_point = very_first_point;
    #local obtrans = transform{Point_At_Trans( rounded_lprism_cNA-rounded_lprism_cNIsect )}
    // round sides
    cylinder{
     0,y*(z2-z1-inset*2),rounded_lprism_IArc_radius+inset*(rounded_lprism_corner_point_is_outside-.5)*2
     open
     clipped_by{
      Wedge(rounded_lprism_IArc_angle)
      rotate -y*rounded_lprism_IArc_angle/2
     }
     rotate x*90
     transform{obtrans}
     translate rounded_lprism_IArc_offset+z*inset
     #if(!rounded_lprism_corner_point_is_outside) inverse #end
    }
    // curved bevel sections
    #local front =
    #if(rounded_lprism_type = 1)
     torus{ rounded_lprism_IArc_radius, inset
    #else
     cone{0,rounded_lprism_IArc_radius+inset*(rounded_lprism_corner_point_is_outside-.5)*2,
     -y*inset,rounded_lprism_IArc_radius open
     #if(!rounded_lprism_corner_point_is_outside) inverse #end
    #end
     clipped_by{
      object{
       Wedge(rounded_lprism_IArc_angle)
       rotate -y*rounded_lprism_IArc_angle/2
      }
      // clipping for transparency
      #if(rounded_lprism_type = 1)
      plane{y,0 }
      
      cylinder{
       -y*(inset+.1),y*(inset+.1),rounded_lprism_IArc_radius
       #if(rounded_lprism_corner_point_is_outside) inverse #end
      }#end
     }
     rotate x*90
     transform{obtrans}
     translate rounded_lprism_IArc_offset
    }
    object{front translate z*inset}
    object{front scale<1,1,-1> translate z*(z2-z1-inset)}
    #local n_poly_pts[v2] = n_poly_pts[v2] + 2 - rounded_lprism_corner_point_is_outside;
    #local V=V+1;
   #end
   #if( !VEq(rounded_lprism_epCW,last_point) )
    object{join_object(last_point,rounded_lprism_epCW)}
   #end
   #local n_poly_pts_total=n_poly_pts_total+n_poly_pts[v2]+1;
   #local v2=v2+1;
  #end
  // ~*** joins ****
  
  // polygon
  #local cap=
  polygon{
   n_poly_pts_total
   // points loop
   #local v2=0;
   #while(v2<dimension_size(pts,1))
    // number of points in array
    #local npts = dimension_size(pts[v2],1);
    #local V = 0;
    #while(V < npts)
     rounded_lprism_get_metrics(v2,V)
     #if(rounded_lprism_corner_point_is_outside)
      rounded_lprism_cNIsect
     #else
      rounded_lprism_epCCW, rounded_lprism_epCW
     #end
     #local V=V+1;
    #end
    #local V=0;
    rounded_lprism_get_metrics(v2,V)
    #if(rounded_lprism_corner_point_is_outside)
      rounded_lprism_cNIsect
     #else
      rounded_lprism_epCCW
    #end
    #local v2=v2+1;
   #end
  
   // clipping loop
   #local v2=0;
   #while(v2<dimension_size(pts,1))
    // number of points in array
    #local npts = dimension_size(pts[v2],1);
    #local V = 0;
    #while(V < npts)
     rounded_lprism_get_metrics(v2,V)
     rounded_lprism_IArc_2D(
      rounded_lprism_cNIsect,
      rounded_lprism_cNA,
      rounded_lprism_epCW
     )
     #if(rounded_lprism_corner_point_is_outside)
      clipped_by{
       difference{
        prism{
         -.1,.1,5,
         <rounded_lprism_IArc_offset.x,rounded_lprism_IArc_offset.y>,
         <rounded_lprism_eCCW.x,rounded_lprism_eCCW.y>,
         rounded_lprism_cNIsect
         <rounded_lprism_eCW.x,rounded_lprism_eCW.y>,
         <rounded_lprism_IArc_offset.x,rounded_lprism_IArc_offset.y>
         rotate x*270
        }
        cylinder{
         -z*.2, z*.2, rounded_lprism_IArc_radius
         translate rounded_lprism_IArc_offset
        }
        inverse
       }
      }
     #else
      clipped_by{
       intersection{
        plane{y,0
         Point_At_Trans( rounded_lprism_cNA-rounded_lprism_cNIsect )
         translate rounded_lprism_epCW 
        }
        cylinder{-z*.11,z*.11,rounded_lprism_IArc_radius
         translate rounded_lprism_IArc_offset
        }
        inverse
       }
      }
     #end
     #local V=V+1;
    #end
    #local v2=v2+1;
   #end
   // ~clipping
  }
  object{cap}
  object{cap translate z*(z2-z1) inverse}
  
  translate z*z1
 }
#end

#macro rounded_lprism_cage(pts, z1, z2, lthk)
 union{
  #local v2=0;
  #while(v2<dimension_size(pts,1))
   #local npts1 = dimension_size(pts[v2],1);
   #local V = 0;
   #while(V < npts1)
    #local p1 = pts[v2][V];
    #local p2 = pts[v2][rounded_lprism_WrapV(V+1,npts1)];
    #if(!VEq(<p1.x,p1.y,0>,<p2.x,p2.y,0>))
     cylinder{<p1.x,p1.y,z1>,<p2.x,p2.y,z1>,lthk}
     cylinder{<p1.x,p1.y,z2>,<p2.x,p2.y,z2>,lthk}
     cylinder{<p1.x,p1.y,z1>,<p1.x,p1.y,z2>,lthk}
     sphere{<p1.x,p1.y,z1>,lthk*2}
     sphere{<p1.x,p1.y,z2>,lthk*2}
    #end
    #local V=V+1;
   #end
   #local v2=v2+1;
  #end
 }
#end

#macro rounded_lprism_prism(pts, z1, z2)
 #local total_pts = 0;
 #local v2=0;
 #while(v2<dimension_size(pts,1))
  #local npts1 = dimension_size(pts[v2],1);
  #local V = 0;
  #while(V < npts1)
   #local total_pts = total_pts+1;
   #local V=V+1;
  #end
  #local v2=v2+1;
 #end
 prism{
  z1,z2,total_pts + dimension_size(pts,1)
  #local v2=0;
  #while(v2<dimension_size(pts,1))
   #local npts1 = dimension_size(pts[v2],1);
   #local V = 0;
   #while(V < npts1)
    #local p1 = pts[v2][V];
    #local p2 = pts[v2][rounded_lprism_WrapV(V+1,npts1)];
    #if(V=0)
     #local first_point = p1;
    #end
    <p1.x,p1.y>
    #local V=V+1;
   #end
   <first_point.x,first_point.y>
   #local v2=v2+1;
  #end
  rotate x*270
  scale<1,1,-1>
  //translate -z*z1
 }
#end

#macro rounded_lprism_autofit(pts, z1, z2, eRad_default, inset)
 #declare rounded_lprism_type = 1;
 #declare rounded_lprism_autofit_en = on;
 rounded_lprism(pts, z1, z2, eRad_default, inset)
#end

#macro rounded_lprism_free(pts, z1, z2, eRad_default, inset)
 #declare rounded_lprism_type = 1;
 #declare rounded_lprism_autofit_en = off;
 rounded_lprism(pts, z1, z2, eRad_default, inset)
#end

#macro rounded_lprism_autofit_chamfered(pts, z1, z2, eRad_default, inset)
 #declare rounded_lprism_type = 2;
 #declare rounded_lprism_autofit_en = on;
 rounded_lprism(pts, z1, z2, eRad_default, inset)
#end

#macro rounded_lprism_free_chamfered(pts, z1, z2, eRad_default, inset)
 #declare rounded_lprism_type = 2;
 #declare rounded_lprism_autofit_en = off;
 rounded_lprism(pts, z1, z2, eRad_default, inset)
#end