/*

 lbMakePoints.pov
 
 You can program your own point sets to make custom effects for
 lb7b.pov with this file. Render at 512x384 or greater to get a
 visualization of your points. lb7b.pov will handle up to 65536
 points, which should be enough (if not very slow).
 
 Each point is a 3D vector. The first two entries correspond to the
 x/y position of the point. The third entry is the "opacity" for
 that point.
 
 For example:
 #declare lb_pts[pt_counter] = <0,0,1>;
 #declare pt_counter = pt_counter + 1;
 
 is a point centered at <0,0> and is opaque (a relative term here).
 It writes to the lb_pts array.
 
 Each time you write a new value to the array, make sure to update
 the pt_counter variable. This will be the total number of elements
 written to the final array.
 
 You should keep the opacity within the 0 - 1 range for most appli-
 cations and the positions should often be kept between the <-1,-1>
 - <1,1> range. This is to maintain consistency with the other
 point sets.
 
 To begin, just place your code in the point creation section
 below. I have included an example for making a disc.
 
 That's it!

*/


/* Here's a nice big array for holding point values. If you need
   more room, just expand it :) Not all of these will be used.   */
#declare lb_pts = array[9216];

// your output file name here
#declare output_file_name = "lbPtsCustom.inc";

// turn file writing on/off
#declare write_file = false;

// change size of point representation
#declare point_size = .01;

// the counter always starts at zero
#declare pt_counter = 0;


// ****************************************************************
// begin point creation - delete all of this for your own routines


// Disc Example

// make center point
#declare lb_pts[pt_counter] = <0,0,1>;

// pt_counter needs to updated when we create a point
#declare pt_counter = pt_counter + 1;

// number of points from center to diameter
#declare rad_n = 18;

// number of points per ring -
// for example, 6 will become 6, 12, 18, 24, etc.
#declare ang_n = 6;

// the disc
#declare radius_var = 1;
#while(radius_var < rad_n + 1)
 #declare rot_div = radius_var*ang_n;
 #declare angle_var = 0;
 #while(angle_var < rot_div)
  
  // Writing the point,
  #declare lb_pts[pt_counter] =
   <
    sin(pi*2/rot_div*angle_var)*radius_var/(rad_n+1), // x pos
    cos(pi*2/rot_div*angle_var)*radius_var/(rad_n+1), // y pos
    1-radius_var/(rad_n+1)                            // opacity
   >;
  
  // updating the point counter.
  #declare pt_counter = pt_counter + 1;
  
  #declare angle_var = angle_var + 1;
 #end
 #declare radius_var = radius_var + 1;
#end


// end point creation
// ****************************************************************

default{finish{ambient 1}}

camera{
 orthographic
 location -z*10
 look_at 0
 angle 15
}

// displays the points
union{
 #declare V=0;
 #declare R=seed(1234);
 #while(V<pt_counter)
  sphere{<lb_pts[V].x,lb_pts[V].y,rand(R)*.01>,point_size pigment{rgb 1 transmit 1-lb_pts[V].z}}
  #declare V=V+1;
 #end
 translate -x*.5/1.5
}

// red frame
union{
 box{<-1,-1,1.1>,<1,1,2> scale<.975,.975,1>pigment{rgb x*.75}}
 box{<-1,-1,1>,<1,1,2> scale<.96,.96,1>}
 translate -x*.5/1.5
}

// writing the actual file
#if(write_file=true)
 #fopen file output_file_name write
 #write(file,concat("#declare lb_pts=array[",str(pt_counter,0,0),"]{\n"))
 #write(file,concat(" <",str(lb_pts[0].x,0,3),",",str(lb_pts[0].y,0,3),",",str(lb_pts[0].z,0,3),">"))
 #declare V=1;
 #while(V<pt_counter)
  #write(file,concat(",\n <",str(lb_pts[V].x,0,3),",",str(lb_pts[V].y,0,3),",",str(lb_pts[V].z,0,3),">"))
  #declare V=V+1;
 #end
 #write(file,"\n}")
#end

// text objects
#declare text_union=
union{
 #if(write_file=true)
  #declare text_object_write=
  text{ttf"crystal.ttf" "file written" 1,0 scale .1 }
  object{text_object_write translate<(1.25-max_extent(text_object_write).x),.95-max_extent(text_object_write).y-.1,0>}
 #else
  #declare text_object_write=
  text{ttf"crystal.ttf" "file not written" 1,0 scale .1 }
  object{text_object_write translate<(1.25-max_extent(text_object_write).x),.95-max_extent(text_object_write).y-.1,0>}
 #end
 #declare text_object_file_name=
 text{ttf"crystal.ttf" output_file_name 1 0 scale .1}
 object{text_object_file_name translate<(1.25-max_extent(text_object_file_name).x),.95-max_extent(text_object_file_name).y,0>}
 #declare text_object_num_points=
 text{ttf"crystal.ttf" concat("point count: ",str(pt_counter,0,0)) 1 0 scale .1}
 object{text_object_num_points translate<(1.25-max_extent(text_object_num_points).x),-1+max_extent(text_object_num_points).y,0>}
}

// white text
object{text_union pigment{rgb 1}}

// black text border
#declare V=0;
#while(V<8)
 object{text_union matrix<1,0,0,0,1,0,sin(pi/4*V)*.25,cos(pi/4*V)*.25,1,0,0,0> translate z*.1}
 #declare V=V+1;
#end