/*

 Luminous Bloom version 7
 2010 Samuel T. Benge (stbenge@hotmail.com)
 
 With this version different effect shapes are available, with the option
 to include your own point sets for even more effects. A large number of
 points may be used (up to 65536), so if you need more samples or a dif-
 ferent effect, you can make your own sets with lbMakePoints.pov.
 
 I have changed the code entirely to work on a similar principle described
 by Tek in his post "Lens flare in scene," March 7, 2010, povray.binaries.
 images. Furthermore, I employed a suggestion for a speed-enhancement made
 by Alain in the same thread. Thank you guys! Because of your posts, this
 version is the fastest yet!
 
 Thanks also to Rune S. Johansen, who has unwittingly provided me with the
 means to detect image names. I took the code from his "illusion.inc."
 
 For details on the keywords used by this file, and for various tips,
 scroll down to the help section.

*/

#declare lb_point_data = "lbPtsStarburst.inc"

#declare lb_image_data = "testPattern.hdr"

#declare lb_interpolate_image = 2;

#declare lb_intensity = 2;

#declare lb_exponent = .5;

#declare lb_falloff_squared = true;

#declare lb_size = .15;

#declare lb_rotation = 0;

#declare lb_assumed_gamma = 2.2;

#declare lb_global_ambient = 1;


/*

 Keyword Help

 
 string lb_point_data
 
  You can specify different point sets with this. It will be used to load
  points which will then determine the shape of the effect. Each point set
  contains position and opacity data.
  
  Just copy and paste any of these for lb_point_data:
   "lbPtsCross.inc"     - an equidistant cross
   "lbPtsDisc.inc"      - a disc with falloff
   "lbPtsDiscHQ.inc"    - a higher-quality version of lbPtsDisc
   "lbPtsLinear.inc"    - a straight line
   "lbPtsSquashed.inc"  - kind of a mix between lbPtsDisc and lbPtsLinear
   "lbPtsStar.inc"      - a six-pointed star
   "lbPtsStarburst.inc" - a semi-random starburst
   
  You can also make your own point sets if you are comfortable writing POV-
  Ray code. See "lbMakePoints.pov" for details.

 
 string lb_image_data
 
  Specify your input image here. Only three types are supported. These are:
  HDR; PNG; TGA. For HDR images you will need either MegaPOV 3.6 or POV-Ray
  3.7 beta or higher.
 
 
 int lb_interpolate_image
  
  0, 1, or 2
  Interpolate the image? Choose 0 for no smoothing. Choose 1 to smooth the
  effect only. Choose 2 to smooth both the effect and the image itself.
  Both 1 or 2 will give good results, at some cost to rendering speed.
 
 
 float lb_intensity
  
  0.0 - ?
  This is the intensity of the effect. Any values may be used, and must be
  adjusted when you change lb_exponent.
 
 
 float lb_exponent
  
  This is the same as the reflection_exponent keyword for finishes. Low
  values (<1.0) will cause the effect to favor very bright areas, while a
  value of 1.0 will produce the standard effect. If you lower this, you
  will probably have to lower lb_intensity as well.
  
 
 bool lb_falloff_squared
 
  true or false
  Use true to give the effect a more realistic falloff. Use false if you
  want more efficient sample usage.

 
 float lb_size
  
  The size of the effect. Keep it above 0.0. Only small values should be
  needed. Larger values may increase the appearance of low-sample
  artifacts (especially with the disc models), so be cautious when
  changing it.


 float lb_rotation
  
  0 - 360
  You can rotate the effect here, in degrees.
  

 float lb_assumed_gamma
  
  1.0 or 2.2
  A shortcut to the ever-controversial assumed_gamma global setting.
 
 
 float lb_global_ambient
 
  You can raise or lower the light level for the entire image here. The
  recommended starting value is 1.0. Try lowering it if your image dev-
  elops overly bright areas.
  
 
 Tips
  
  *HDR images will always work better than other formats.
  
  *You can mix effects by first rendering one effect, and then using the
  rendered result for the next pass.
  
  *Certain point sets render too darkly because the samples are so spread
  out (such as lbPtsDisc & lbPtsDiscHQ). To compensate, try lowering
  lb_exponent and lb_intensity until it looks right.
 
  *Since there is no longer a quality setting, this version will always
  use the maximum number of samples for any given point set. It's best to
  render small portions of the image or the whole image at a lower res-
  olution to preview the effect.
  
  *Try to avoid using images with pure black areas, as these will always
  render more slowly for some reason. An rgb value of <.02,.02,.02> will
  be much faster than absolute black.
  
  *To view the different point sets, open lbViewPoints.pov, include the
  point file you wish to see, and render at 512x384.
  
*/

global_settings{ assumed_gamma lb_assumed_gamma adc_bailout 0 }

#default{finish{ambient lb_global_ambient}}

camera{
 orthographic
 right x*2 up y*2
 location -z*10
 look_at 0
}

// by not including transforms.inc, we save a little parsing time
#macro VDist(V1, V2) vlength(V1 - V2) #end
#macro VPerp_To_Vector(v0) #if (vlength(v0) = 0) #local vN = <0, 0, 0>; #else #local Dm = min(abs(v0.x), abs(v0.y), abs(v0.z)); #if (abs(v0.z) = Dm) #local vN = vnormalize(vcross(v0, z)); #else #if (abs(v0.y) = Dm) #local vN = vnormalize(vcross(v0, y)); #else #local vN = vnormalize(vcross(v0, x)); #end #end #end vN #end
#macro Shear_Trans(A, B, C) transform { matrix < A.x, A.y, A.z, B.x, B.y, B.z, C.x, C.y, C.z, 0, 0, 0> } #end
#macro Point_At_Trans(YAxis) #local Y = vnormalize(YAxis); #local X = VPerp_To_Vector(Y); #local Z = vcross(X, Y); Shear_Trans(X, Y, Z) #end

#include lb_point_data

#declare lb_image_format = strlwr(substr(lb_image_data,strlen(lb_image_data)-3,4))

#macro lb_set_image_map()
 #if(strcmp(lb_image_format,".hdr")=0) hdr #end
 #if(strcmp(lb_image_format,".png")=0) png #end
 #if(strcmp(lb_image_format,".tga")=0) tga #end
 lb_image_data
#end

#macro lb_set_image_transformation()
 warp{repeat x flip x}
 warp{repeat y flip y}
 scale 2 translate-1
#end

#declare lb_img=
pigment{
 image_map{
  lb_set_image_map()
  #if(lb_interpolate_image=2) interpolate 2 #end
 }
 lb_set_image_transformation()
}

#declare lb_img_effect=
pigment{
 image_map{
  lb_set_image_map()
  #if(lb_interpolate_image>0) interpolate 2 #end
 }
 lb_set_image_transformation()
}

/* This image plane will be reflected. It
   results in the luminous bloom effect.  */
plane{z,-20 pigment{lb_img_effect}}

// if intensity = 0, render without flares
#if(lb_intensity=0)
 plane{z,0 pigment{lb_img}}
#else
 plane{z,0
  // 65536 samples are possible with nested average patterns
  texture{
   average
   texture_map{
    // here's a tiny value so Point_At_Trans and VDist don't throw errors
    #declare lb_tiny_V=0.00001;
    #declare lb_n_pts=dimension_size(lb_pts,1);
    #declare lb_subsection_target=256;
    #declare lb_counter=0;
    #declare lb_section=0;
    #while(lb_section<=int(lb_n_pts/256))
     [1
      average
      texture_map{
       #declare lb_subsection=0;
       // determine the number of samples needed for this subsection
       #if(lb_n_pts-lb_counter<256)
        #declare lb_subsection_target=lb_n_pts-lb_counter;
       #end
       #while(lb_subsection<lb_subsection_target)
        #declare lb_V=lb_counter;
        [1
         // combine original image back into effect
         pigment{lb_img}
         // opacity for each point affects the reflection values
         finish{
          #if(lb_falloff_squared=true)
           reflection pow(lb_pts[lb_V].z*lb_intensity,2)/256*lb_subsection_target
          #else
           reflection (lb_pts[lb_V].z*lb_intensity)/256*lb_subsection_target
          #end
          reflection_exponent lb_exponent
         }
         normal{
          planar
          // calculate normal strength based on distance
          bump_size lb_size*VDist(<0,0,lb_tiny_V>,<lb_pts[lb_V].x,lb_pts[lb_V].y,lb_tiny_V>)
          // some transforms
          scale 4 translate -y*2
          rotate -z*lb_rotation
          // calculate angles based on point position
          Point_At_Trans(<lb_pts[lb_V].x+lb_tiny_V,lb_pts[lb_V].y+lb_tiny_V,lb_tiny_V>)
          // image aspect correction
          scale<image_width/image_height,1,1>
         }
        ]
        #declare lb_counter=lb_counter+1;
        #declare lb_subsection=lb_subsection+1;
       #end
      }
     ]
    #declare lb_section=lb_section+1;
    #end
   }
  }
 }
#end