// std.inc
// by Tek
// macros for stuff I usually write in every scene
// MUST be easier than writing them

#ifndef (__STD_INC__)
#declare __STD_INC__ =1;

#version unofficial megapov 1.21;

#include "rad_def.inc"
#include "transforms.inc"


// debug

#macro AssertEq( v1, v2, Message )
	#if ( abs(v1 - v2) > abs(v1+0.0001)*0.0001 )
		#error concat("Assert Fired: ",str(v1,0,3)," != ",str(v2,0,3),Message)
	#end
#end


// mathematical

#macro VRand(Seed) (<rand(Seed),rand(Seed),rand(Seed)>*2-1) #end

// given a value in [-1,1] apply a curve to correct distribution in proportion to length of circumference of a circle at that lattitude
// integral of f(x) = cos(pi*f(x)), so probability of being placed at any given value == length of circumference at that lattitude
// only that still ain't quite right, integral over what range?
// *FUDGE IT* for now, go find my previous version!
#declare SphereAngleDistribute = function(x) { x*pow(abs(x),2) }


#macro RandOrient(Seed)
   // evenly distribute orientations on surface of sphere, and at random angles
   // owing to order of computation I'm basing this on <1,0,0>, though that doesn't matter since all orientations will be equally likely
   <rand(Seed)*360,SphereAngleDistribute(rand(Seed))*90,rand(Seed)*360>
#end


#macro ScreenSpace()
   // at depth z=0: transform something into pixels on screen, measured from middle
   // depth is adjusted in proportion to the other dimensions, to avoid distortion
   scale 1/image_height
   translate z*StoredCamZoom
   scale 1/1000 // just in front of camera
   transform { StoredCamTransform } 
#end

//todo: want macro to place things relative to 2D pos & dims & depth...


#macro ReOrient_Vector( V, From, To )
	#local Dot = vdot(From,To)/(vlength(From)*vlength(To));
	#if ( abs(Dot) < 0.9999 )
		#local Axis = vcross(From,To);
		#local Angle = atan2( vlength(Axis), Dot );
		#local Result = vaxis_rotate( V, Axis, degrees(Angle) );
	#else
		#if ( Dot > 0 )
			#local Result = V;
		#else
			#local Result = -V;
		#end
	#end
	(Result)
#end

#macro Lerp(A,B,C)
   ( A*(1-C) + B*C )
#end

#macro BiLerp(A,B,C,D,U,V)
   ( A*(1-U)*(1-V) + B*U*(1-V) + C*(1-U)*V + D*U*V )
#end

#declare SignPow = function(x,p) { x*pow(abs(x),p-1) }
#declare ReverseSignPow = function(x,p) { select(x,-1+pow(1-abs(x),p),1-pow(1-abs(x),p)) }


#macro AtmosphericFalloff(s,d1,d0)
	function (r)
	{
		//exponential curve tweaked to hit 1 at d1 and 0 at d0
		max( 0, ( exp(s*(d1-r)) - exp(s*(d1-d0)) ) / (1 - exp(s*(d1-d0))) )
	}
#end

#declare UMod = function(x,m){ (x/m-floor(x/m))*m }

#declare Quantize = function(x,n){ floor(x*n)/n }

#declare Ease = function(x){ .5-.5*cos(pi*x) } // suspect trig is faster than { 3*x*x-2*x*x*x } // only valid for 0,1 range

#macro frac(a)
	(a - floor(a))
#end

// rotate a right by i bits within low n bits 
// despite pov's lack of binary instructions this is actually pretty easy
#macro BitRotateRight(a,i,n)
	mod(int(a*pow(2,n-i) + a*pow(2,-i)),pow(2,n))
#end

// rotate a left by i bits within low n bits 
#macro BitRotateLeft(a,i,n)
	mod(int(a*pow(2,i) + a*pow(2,i-n)),pow(2,n))
#end

/*
// unit-test bit rotate
#debug concat("BitRotate test: ",str(1,0,0),"=",str(BitRotateRight(1,0,4),0,0),"=",str(BitRotateRight(2,1,4),0,0),"=",str(BitRotateRight(4,2,4),0,0),"=",str(BitRotateRight(8,3,4) ,0,0),"\n")
#debug concat("BitRotate test: ",str(9,0,0),"=",str(BitRotateRight(9,0,4),0,0),"=",str(BitRotateRight(3,1,4),0,0),"=",str(BitRotateRight(6,2,4),0,0),"=",str(BitRotateRight(12,3,4),0,0),"\n")
#debug concat("BitRotate test: ",str(7,0,0),"=",str(BitRotateRight(7,0,4),0,0),"=",str(BitRotateRight(14,1,4),0,0),"=",str(BitRotateRight(13,2,4),0,0),"=",str(BitRotateRight(11,3,4),0,0),"\n")
#debug concat("BitRotate test: ",str(11,0,0),"=",str(BitRotateRight(11,0,4),0,0),"=",str(BitRotateRight(7,1,4),0,0),"=",str(BitRotateRight(14,2,4),0,0),"=",str(BitRotateRight(13,3,4),0,0),"\n")
*/


// data structures

//STACK FUNCTIONS CAUSE CRASHES! don't use intensely
#macro StackStart()
	#ifdef (Stack) #undef Stack #end
	#declare Stack = array[1]; // this uninitialized entry is useful for error checking, pop too many and you will hit it!
#end

#macro StackPush( Item )
	#local Stack2 = array[dimension_size(Stack,1)+1];
	#local Stack2[0] = Item;
	#local i=0;#while ( i < dimension_size(Stack,1)-1 ) //don't copy the uninited terminator! leave it uninited, for error checking
		#local Stack2[i+1] = Stack[i];
	#local i=i+1;#end
	#ifdef (Stack) #undef Stack #end
	#declare Stack = Stack2;
#end

#macro StackPop()
	#local Item = Stack[0];
	#local Stack2 = array[dimension_size(Stack,1)-1];
	#local i=0;#while ( i < dimension_size(Stack,1)-1 )
		#local Stack2[i] = Stack[i-1];
	#local i=i+1;#end
	#ifdef (Stack) #undef Stack #end
	#declare Stack = Stack2;
	Item
#end

#macro StackSize()
	(dimension_size( Stack, 1 )-1)
#end

#macro StackAsArray()
	// remove the terminator & return
	#local Array = array[dimension_size(Stack,1)-1];
	#local i=0;#while ( i < dimension_size(Stack,1)-1 )
		#local Array[i] = Stack[i];
	#local i=i+1;#end
	Array
#end

// procedural

#macro For(From,To,Step)
   #local i=From;
   #while ( i < To )
      Callback(i)
      #local i=i+Step;
   #end   
#end


// camera setup (call before calling camera macro)

// blur parameters
// Aperture & Focus, are scaled by distance of camera from target (for camera types with targets, 1.0 in front of camera otherwise).
// Focus = 0 => target in focus, focus = -0.5 = half way between camera and target
// Pure - turn on/off a slower render 
#macro CamBlur( RelApe, RelFocus, Samples, Pure )
	#declare StoredBlurRelApe = RelApe;
	#declare StoredBlurRelFocus = RelFocus;
	#declare StoredBlurSamples = Samples;
	#declare StoredBlurPure = Pure;
#end

// internal

#macro DoBlur()
	#ifdef (StoredBlurRelApe)
		
		#local Dir = StoredCamLook-StoredCamLoc; // don't normalize

		#declare StoredBlurAperture = StoredBlurRelApe*vlength(Dir);
		aperture StoredBlurAperture

		#ifndef (StoredCamTransform)
			focal_point StoredCamLook + StoredBlurRelFocus*Dir
		#else
			// pov applies blur settings before transform, regardless of the order of the instructions
			// assume camera was at <0,0,0> pointing along z before transform, and transform doesn't scale.
			focal_point 0 + (StoredBlurRelFocus+1)*vlength(Dir)*z
		#end
		
		blur_samples StoredBlurSamples
		
		//some magic settings to get rid of some nasty artefacts where the system tries to be clever.
		confidence 1 variance 0 // perfect settings - see only noise from num samps
		
	#end
#end

// camera

#macro CamTransform( Transform )
   #declare StoredCamTransform = Transform;
   transform { StoredCamTransform }
#end

#macro CamLocLook( Loc, Look )
   #declare Transform =
      transform {
      	#ifdef (CamTilt) rotate CamTilt*z #end
         #local Dir = Look-Loc;
         Reorient_Trans( z, Dir*<1,0,1> )
         Reorient_Trans( Dir*<1,0,1>, Dir )
         translate Loc
      }
   #declare StoredCamLoc = Loc;
   #declare StoredCamLook = Look;
   CamTransform(Transform)
#end

#macro CamAxes(Zoom)
	right			x*image_width/image_height
	up				y
	direction	z*Zoom
	
	#declare StoredCamZoom = Zoom;
#end

#macro CamAxesY(Zoom)
	CamAxes(Zoom*image_width/image_height)
#end

#macro CamAxesAvg(Zoom)
	CamAxes(Zoom*sqrt(image_height*image_height + image_width*image_width)/image_height)
#end

#macro CamPolar (Origin,Rotation,Distance,Zoom)
	camera {
		CamAxes(Zoom)

      CamTransform (
         transform {		
	      	#ifdef (CamTilt) rotate CamTilt*z #end
      		translate -z*Distance
      		rotate Rotation*z
      		rotate Rotation*<1,1,0>
      		translate Origin
   	   }
   	)
   	
   	#declare StoredCamLoc = vrotate( -z*Distance, Rotation*<1,1,0> ) + Origin;
   	#declare StoredCamLook = Origin;
		
		DoBlur()
	}
#end

#macro CamOrbit (Origin,Direction,Distance,Zoom)
	camera {
		CamAxes(Zoom)
		
		CamLocLook( vnormalize(Direction)*Distance, Origin )
		
		DoBlur()
	}
#end

#macro Cam(Loc,Look,Zoom)
	camera {
		CamAxes(Zoom)
		
		CamLocLook( Loc, Look )
		
		DoBlur()
	}
#end

#macro CamDir(Loc,Dir,Zoom)
	Cam(Loc,Loc+Dir,Zoom)
#end

#macro CamRot(Loc,Rot,Zoom)
	Cam(Loc,Loc+vrotate(z,Rot),Zoom)
#end

// camera/screen debug

// draw a grid using default material, wrap this in object{} to change colour
#macro Grid(Divisions)
	union {
		#local i=0; #while (i <= Divisions)
			#local f = i/Divisions;
			cylinder { <-1,f*2-1,1>*StoredCamDim, <1,f*2-1,1>*StoredCamDim, 1/image_width }
			cylinder { <f*2-1,-1,1>*StoredCamDim, <f*2-1,1,1>*StoredCamDim, 1/image_width }
		#local i=i+1;#end
		scale .001
		transform { StoredCamTransform }
	}
#end


// global setup (lights, materials, background)

#macro Radiosity(RadType)
   #declare StoredRadType = RadType;
#end

#macro Default(Brightness)
	#default { pigment { rgb 1 } finish { diffuse Brightness ambient 0 } }
#end

#macro DefaultNoRad(Ambient)
	#default { pigment { rgb 1 } finish { diffuse .8 ambient rgb Ambient } }
#end

#macro BG(Col) background { rgb Col } #end

#macro Global( DoRad )
	//#version unofficial megapov 1.21;  doesn't work :(, have to put into each file.
	global_settings {
		assumed_gamma 1
		#if ( DoRad )
			#ifndef (StoredRadType) Radiosity(Radiosity_Fast) #end
      	radiosity { Rad_Settings( StoredRadType, on, on ) }
		#end
		max_trace_level 20
	}
	#if ( DoRad )
		Default(.7)
	#else
		DefaultNoRad(.2)
	#end
	
	#declare DefaultTexture = texture{}; // this is kind of buried, but it needs to be placed here, just in case default isn't retroactive
	
	BG(.7)
#end

#macro LightBoxes(Prob,Seed)
	box { -3,3
		pigment { cells colour_map { [Prob rgb 1][Prob rgbt 1] } translate Seed phase Seed }
		finish { ambient 8 diffuse 0 }
		no_image no_shadow hollow inverse
		scale 1000
		no_radiosity
	}
#end


// lights

#macro DoAreaLight(Dist)
	#ifdef (AreaLights)
		area_light x*Dist/20,y*Dist/20,2,2 adaptive 1 jitter orient
	#end
#end

#macro Light( Pos, Col )
	light_source {
		Pos, rgb Col
		#ifdef (StoredCamLoc)
			DoAreaLight(vlength(Pos-StoredCamLoc))
		#else
			DoAreaLight(vlength(Pos-0))
		#end
		//looks_like { Glow( 0, Dist/9, Col*8, .3 ) }
	}
#end

#macro LightPolar( Origin, Rot, Dist, Col )
	light_source {
		vrotate(-z,Rot)*Dist, rgb Col
		DoAreaLight(Dist)
		looks_like { Glow( 0, Dist/9, Col*8, .3 ) }
		translate Origin
	}
#end

#macro FillLight( Pos, Col )
	light_source {
		Pos, rgb Col
		shadowless
	}
#end

#macro SunLight( Dir, Col )
	light_source {
		vnormalize(Dir)*10000, rgb Col parallel point_at 0
		DoAreaLight(10000)
	}
#end

#macro SunLightPolar( Rot, Col )
	light_source {
		vrotate(-z,Rot)*10000, rgb Col parallel point_at 0
		DoAreaLight(10000)
	}
#end

#macro SpotLightPolar( Origin, Rot, Dist, Inner, Outer, Col )
	#local RelPos = vrotate(-z,Rot)*Dist;
	SpotLight( Origin+RelPos, Origin, Inner, Outer, Col )
#end

#macro SpotLight( Pos, Target, Inner, Outer, Col )
	#ifdef (DebugSpotlights)
		SpotLightDebug( Pos, Target, Inner, Outer, Col )
	#else
	light_source {
			#local Dist = vlength(Target-Pos);
			Pos, rgb Col spotlight point_at Target radius degrees(atan2(Inner,Dist)) falloff degrees(atan2(Outer,Dist))
		DoAreaLight(Dist)
//			looks_like { GlowSpot( Pos, Target-Pos, Dist*.2, Col, .3 ) }
	}
	#end
#end

#macro SpotLightDebug( Pos, Target, Inner, Outer, Col )
	cone { Pos, 0, Target, Inner pigment { rgb Col transmit .8 } finish { diffuse 1 ambient 1 } }
	cone { Pos, 0, Target, Outer pigment { rgb Col transmit .8 } finish { diffuse 1 ambient 1 } }
#end

#macro LightBoxPolar( Origin, Rot, Dist )
   box {
      -1,1
      scale <.3,.3,.01>*Dist
      rotate z*45
      translate -z*Dist
      rotate Rot
      translate Origin
      
      pigment { rgb 1 }
      finish { ambient 5 }
      
      no_shadow
      no_radiosity
   }
#end


// objects

#macro Studio( BackDrop )
   BG(0)

   LightBoxPolar( 0, <30,120>, 8 )
   LightBoxPolar( 0, <60,30>, 8 )
   LightBoxPolar( 0, <30,-90>, 8 )
   
   LightPolar( 0, <60,60>, 5, 1 )
   
   difference {
      box { <-.8,-1.1,-.8>, <.8,.8,1.1> }
      superellipsoid { <.2,.2> }
      translate y
      scale 5
      pigment { rgb BackDrop }
      finish { brilliance .2 }
   }
#end


#macro Ground( Dist )

	#local r = 10000;
	sphere {
		-y*(r+abs(Dist)), r
		texture { GroundTex }
	}

#end

#macro Arrow( Base, Tip, ShaftRad, HeadRad, Col )
	union {
		#local HeadRatio = min( 2*HeadRad/vlength(Tip-Base), 1/2 );
		cylinder { Base, Lerp(Tip,Base,HeadRatio), ShaftRad }
		cone { Lerp(Tip,Base,HeadRatio), HeadRad, Tip, 0 }
		pigment { Col }
	} 
#end


// materials

// DefaultTexture - declared in Global()

#macro Gloss(MinRef)
	texture {
		pigment { rgbt 1 }
		finish {
			reflection { MinRef, 1 falloff 5 } conserve_energy
			phong 1 phong_size 200
		}
	}
#end

#macro Flip()
   colour_map { [0 rgb 1][1 rgb 0] }
#end

#macro Outline()
	material {
		texture { pigment { rgbt 1 } }
		interior_texture { pigment { rgb 0 } }
	}
	no_radiosity
	no_reflection
	no_shadow
#end

// very neutral ground texture, because I never know what to use
#declare GroundTex =
	texture {
		pigment { rgb <.2,.13,.08> }
	}


// special fx

#macro Glow(Pos, Rad, Intensity, Falloff)
	sphere {
		0, 1 hollow no_shadow no_radiosity
		pigment { rgbt 1 }
		interior {
			media {
				emission Intensity*Rad //mul by rad to correct density (parts/vol) for area (parts/area)
				density {
					spherical
					poly_wave Falloff
					density_map { [0 rgb 0][1 rgb 1/(Rad*Rad)] }
				}
			}
		}
		scale Rad
		translate Pos
	}
#end

#macro GlowRays(Pos, Rad, Intensity, Falloff, RayRatio)
	sphere {
		0, 1 hollow no_shadow no_radiosity
		pigment { rgbt 1 }
		interior {
			media {
				emission Intensity*Rad //mul by rad to correct density (parts/vol) for area (parts/area)
				density {
					spherical
					poly_wave Falloff
					density_map {
						[0 rgb 0]
						[1
							#local RayControl = function(x,y,z,r) { f_wrinkles(x/r,y/r,z/r) }
							function { RayControl(x,y,z,.3*sqrt(x*x+y*y+z*z)) }
							density_map {
								[0 rgb RayRatio/(Rad*Rad)]
								[1 rgb 1/(Rad*Rad)]
							}
						]
					}
				}
			}
		}
		scale Rad
		translate Pos
	}
#end

#macro GlowSpot(Pos, Dir, Dist, Intensity, Falloff)
	// ideally I'd like a directional component to get a camera-based flare... but such things should really be handled in lens processing
//	todo
#end

#macro Fog(Dist,Col)
fog {
   fog_type 1
   distance Dist
   rgbt Col
}
#end

#macro GroundFog(Bottom,Top,Dist,Col)
fog {
   fog_type 2
   fog_alt Top-Bottom
   fog_offset Bottom
   distance Dist
   rgbt Col
}
#end


#macro Iris(Brightness)
	// linearly scale brightness of whole image
	sphere {
		StoredCamLoc, 0.0001
		hollow
		pigment { rgb 0 transmit Brightness }
	}
#end

#macro Vignetting(Level)
	sphere {
		0, 1
		hollow
		pigment {
			pigment_pattern {
				gradient z
				#local CornerZ = StoredCamZoom / sqrt( pow(StoredCamZoom,2) + pow(.5*image_width/image_height,2) + .25);
				colour_map {
					[CornerZ rgb 0]
					[1 rgb 1]
				}
			}
			cubic_wave
			colour_map {
				[0 rgb 0 transmit 1-Level]
				[1 rgb 0 transmit 1]
			}
		}
		finish { ambient 1 diffuse 0 }

//dammit do these htings in tonemapping so they work with blur

		no_reflection
		no_shadow
		no_radiosity
		
		#ifdef (StoredBlurAperture)
			scale StoredBlurAperture*4 // need to make sure blur doesn't destroy the effect
		#else
			scale .00011 // avoid iris sphere
		#end
		transform { StoredCamTransform }
	}
#end


// full scene presets - these provide a shorthand for testing out things, e.g. creating a single object without fussing around with the scene

// Blue preset
// focus on a unit-sized object at the origin
// this behaves well if you override it's camera (by setting your own camera after calling this)
// optionally creates some light & dark boxes to help reflections "pop"
#macro ObjectPresetBlueA( Highlights )

	Global(on)

	CamPolar( 0, <30,0>, 6, 2.9 )
	
	sky_sphere {
		pigment { planar poly_wave 2 colour_map { [0 rgb <0,.1,.3>][1 rgb <1.0,1.3,1.7>] } }// rotate -30*x }
		#if (Highlights)
			pigment { cells scale .51 rotate -70 colour_map { [.8 rgbt 1][.8 rgb 0][.9 rgb 0][.9 rgb 2] } }
		#end
	}
	
	SunLight( <-2,2,-1>, <1.3,1.17,1>*2)
	
	#default { pigment { rgb .5 } } // grey, because this lighting rig intentionally saturates white objects

#end

#macro ObjectPresetBlue()				ObjectPresetBlueA(off) #end
#macro ObjectPresetBlueHighlights()	ObjectPresetBlueA(on) #end


// Warm preset
// Scale is radius of sphere representing target object at <0,0,0>
// Orient is interpretted as rotation of whole background with camera about y, and only camera about x. i.e. the scene is always built with y up.
// This gives the most natural result whilst allowing very specific lighting & composition.
#macro ObjectPresetWarmA( Scale, Orient, BlurQuality )

	#ifndef (StoredRadType)
		Radiosity(Radiosity_2Bounce) // need a lot of bounced light
	#end
	Global(on)

	// lots of depth of field, almost macro lensy
	#switch (BlurQuality)
		//#default #case (0) #break; do nothing
		#case (1) CamBlur(0.1,0,20,off) #break;
		#case (2) CamBlur(0.1,0,200,on) #break;
	#end
	
	CamPolar(0,Orient,4*Scale,1)
	
	union {
		// built on a 1m scale, then scaled up so target object is much smaller than 1m
		
		#local Dim = <6,3,4>;
		#local Offset = <-3,-.5,-3>; //to -ve corner
		
		// FRONT WALL
		// with big window to let the light in, small overhang on left perhaps
		#local WinOffset = <1,.5,Dim.z>; //relative to Offset
		#local WinDim = Dim-WinOffset; // all the way to the top right
		box { 0,1 scale <Dim.x,WinOffset.y,.2> translate Offset+Dim*z }
		box { 0,1 scale <WinOffset.x,Dim.y,.2> translate Offset+(Dim-.001)*z }
		
		//frames
		#local Step = WinDim.x/8;
		#local f=0; #while ( f < WinDim.x+Step/2 )
			box { y-1,1 scale <.01,WinDim.y,.01> translate Offset+WinOffset+f*x }
		#local f=f+Step; #end
		
		#local Step = WinDim.y/3;
		#local f=0; #while ( f < WinDim.y+Step/2 )
			box { x-1,1 scale <WinDim.x,.01,.01> translate Offset+WinOffset+f*y }
		#local f=f+Step; #end
		
		
		// BACK WALL
		// green speckly band implying noticeboards with art posted, door
		box {
			0,1 scale <Dim.x,Dim.y,-.2> translate Offset
		}
		
		// LEFT WALL
		// green speckly band implying noticeboards with art posted, door
		box {
			0,1 scale <-.2,Dim.y,Dim.z> translate Offset
		}

		// RIGHT WALL
		// blackboards
		box {
			0,1 scale <.2,Dim.y,Dim.z> translate Offset+Dim*x
		}
		
		// FLOOR
		// dull yellow/grey tiles with speckles stains and black grout (flush, no bumps aside from wiggle)
		box { 0, 1 scale <Dim.x,-.2,Dim.z> translate Offset }
		
		// CEILING
		// polystyrene, high, strip lights [off]
		box { 0, 1 scale <Dim.x,.2,Dim.z> translate Offset+Dim*y }
		
		//convert 1m scale to scene scale
		scale 40*Scale
		rotate Orient*y
	}

	// Adjust these in grey first, then colour
	SunLightPolar( <65,-135>+Orient*y, rgb 4*<1.4,1,.7> )
	BG(2*<1.3,1,.8>)//<.9,1,1.3>)
	
	// 2D tree
	#declare R = .5;
	#macro Branch(Start,Dir,L)
		#local End = Start+Dir*L;
		cone { Start, L*.05, End, L*R*.05 }
		sphere { Start, L*.05 }
		sphere { End, L*R*.05 }
		#if ( L > 0.2 )
			// end branches
			#local n = 1.99*rand(rs);
			#local i=0; #while ( i < n )
				Branch(End,vrotate(Dir,(rand(rs)-.5)*60*z),L*R)
			#local i=i+1; #end
			// side branches
			#local n = 3.99*rand(rs);
			#local i=0; #while ( i < n )
				#local T = rand(rs)*.8+.2;
				Branch(Lerp(Start,End,T),vrotate(Dir,select(rand(rs)-.5,-1,1)*((rand(rs)-.5)*30+45)*z),L*R)//Lerp(L,L*R,T))
			#local i=i+1; #end
		#end
	#end
	
	#declare rs = seed(13);
	#local Tree1 = union { Branch(z*2-y*.1,y,1) pigment { rgb .05 } }
	#local Tree2 = union { Branch(z*2-y*.1,y,1) pigment { rgb .05 } }
	object { Tree1 translate x scale 100*Scale rotate Orient*y }
	object { Tree2 translate x+z*1 scale 100*Scale rotate Orient*y }
	object { Tree1 scale <-1,1,1> translate x+z*2 scale 100*Scale rotate Orient*y }

	// nearby building
	box { 0,1 scale <1,3,10> translate <-3,0,5>+Offset+Dim*z scale 40*Scale pigment { rgb .2 }}
	
//angle of shot seems wrong, though foreground is right... => 2/3rdsise?	

	//looks like treetops, 1stfloorish

#end


#macro ObjectPresetWarmNoBlur( A,B ) ObjectPresetWarmA ( A,B,0 ) #end
#macro ObjectPresetWarmLoBlur( A,B ) ObjectPresetWarmA ( A,B,1 ) #end
#macro ObjectPresetWarmHiBlur( A,B ) ObjectPresetWarmA ( A,B,2 ) #end





//IDEAS:
/*
// better pigment/texture/colour map - allows cubic curves between several different values
*needs work on the "pigment/texture/density" map part of it, or just code 4 macros
BetterMapStart(
	Pattern,
	"pigment_map"
)
	BetterMapEntry(0.1) rgb 0.7
	BetterMapRange(0.3, 0.4)
		spotted
		scale .3
BetterMapEnd()


->

pigment_pattern {
	pigment_pattern { Pattern }
	colour_map {
		#local Flip = 0;
		#local i=0; #while ( i < NumVals )
			[ValStart[i],ValEnd[i] rgb Flip, rgb Flip]
			#local Flip = 1-Flip;
		#local i=i+1; #end
	}
}
cubic_wave
MapType {
	[0
		// odd-numbered patterns
		pigment_pattern { Pattern }
		MapType {
			#local i=0; #while ( i < NumVals )
				[ValStart[i],ValStart[i+2] Pattern[i+1],Pattern[i+1]] //this is the one that has to blend with i and i+2
			#local i=i+2; #end
		}
	]
	[1
		// even numbered patterns
		pigment_pattern { Pattern }
		MapType {
			#local i=1; #while ( i < NumVals )
				[ValStart[i],ValStart[i+2] Pattern[i+1],Pattern[i+1]] //this is the one that has to blend with i and i+2
			#local i=i+2; #end
		}
	]
}
*/

#end // __STD_INC__
