/*
lens flare experiment:

A lens flare is basically an out-of-focus distorted faint image of the scene.
So can we pull that off using a part-reflective mirror?
*/

#declare ImageFlare = 1.5;
#declare LightFlare = 5; // Light sources are visible using a specular highlight on the flare surface
#declare EchoStrength = .003;

#declare FlareSize = .5;

#declare FlareSamples = 80; // more = smoother, slower effect. For x filter this should be divisible by 4
#declare NoiseSamples = 20; // This slows parse time but probably not render time, bigger numbers make noise more random.


#include "std.inc"

Global(on)

// The Scene - lots of lights in dark world, high contrast
BG(0)

#if ( final_frame > 0 )
	Cam(x*(-1+1.4*Ease(frame_number/(final_frame-1))), z*4, 1)
#else
	Cam(0, z*4, 1)
#end

Light(<-1,2,20>,<1,1.2,1.4>*2)

plane { y, -1 }

sphere { z*4+x, 1 texture { pigment { rgb <.1,.3,.5> } } Gloss(.05) }

height_field { function 256,256 { pigment { granite scale 5 colour_map { [0 rgb 1][1 rgb 0] } } } scale <2,.2,2> translate <-2,-.8,2> finish { specular 3 roughness 1/500 } }//texture {} Gloss(0.5) }



// The Lens Flare - stick a half-silvered mirror in front of the camera, and a "flare screen" behind it
union {
	// half-silvered mirror
	disc { z*StoredCamZoom, -z, sqrt(2) pigment { rgb <1,1,1> transmit 1 } finish { diffuse 0 ambient 0 reflection { 1/4 } conserve_energy } no_reflection }

	// "flare screen" behind camera, reflecting multiple offset views of the scene
	disc {
		-z*StoredCamZoom, z, 3*sqrt(2)
		
		// X shaped flare, multiple copies of the scene each offset along one of the arms of the X
		#macro FlareTex(i)
			#local I2 = int(2*i/FlareSamples);
			#local F2 = frac((2*i+2*Offset)/FlareSamples)*2-1;
			[pow(1-abs(F2),2)/FlareSamples
				pigment { rgb <1,1,1> } // tint this for colour seperations, etc.
				finish {
					diffuse 0 ambient 0
					reflection { ImageFlare metallic 1 }
					phong LightFlare phong_size Lerp(5000,10000,abs(F2))
					metallic 1
				}
				normal {
					gradient y FlareSize*F2 translate y/2 scale 2*3*sqrt(2) rotate z*(40+90*I2)
				}
			]
		#end

		// coloured repeated images of the scene
		#macro EchoTex(Strength,Col,Displacement)
			[Strength*EchoStrength
				pigment { rgb Col }
				finish {
					diffuse 0 ambient 0
					reflection { ImageFlare metallic 1 }
					phong LightFlare phong_size 5000
					metallic 1
				}
				normal {
					// roughly parabolic
					pigment_pattern { spherical scale 3*sqrt(2) translate -z*StoredCamZoom colour_map { [0 rgb 1][1 rgb 0] } } Displacement
					poly_wave 2
				}
			]
		#end
		
		texture {
			pigment_pattern { cells translate z*frame_number scale 3/image_width }
			texture_map {
				#declare NoiseStep = 1/NoiseSamples;
				#local f=0;#while ( f < 1.0 )
					#local Offset = f + NoiseStep/2;
					#local Tex =
						texture {
							average
							texture_map {
							
								// bloom
								#declare rs = seed(22);
								#local i=0;#while ( i < FlareSamples )
									FlareTex(i)
								#local i=i+1;#end
								
								// echoes
								EchoTex(1,<1,.1,.03>,6)
								EchoTex(1,<.03,1,.1>,7)
								EchoTex(1,<.1,.03,1>,8)
								EchoTex(1,<1,.1,.03>,18)
								EchoTex(1,<.03,1,.1>,16)
								EchoTex(1,<.1,.03,1>,14)
							}
						};
					[f Tex]
					[f+NoiseStep Tex]
				#local f=f+NoiseStep;#end
			}
		}
	}

	scale .001 // small enough to not affect the rest of the scene, and to keep flared image close to actual
	transform { StoredCamTransform }
}
