/*

 hf_tools.inc
 2009 Sam Benge
 
 This is a small utility for manipulating grayscale data,
 mainly for height fields and isosurfaces. It includes
 some functions for loading red/green 16-bit height
 field data into pigments.
 
 The image loading routines are next to useless, as they can
 be awfully slow. It's best to always use 16-bit pngs for
 your height maps.
 
 Contents:
 
 edge_map( pigment, radius, rand_radius, quality )
 angle_map( pigment )
 superellipsoidal( roundness )
 image_16_bit( image_name )
 image_8_bit( image_name )
 hf_grey_16( pigment )
 
 
 -----------------------------------------------------------
 edge_map( pigment_name, radius, rand_radius, quality )
 
 pigment_name:  the pigment to be used
 radius:        blur radius (0-1)
 rand_radius:   randome blur radius (0-1)
 quality:       how many samples (0-1)
 
 This is a proximity pattern, for a pigment. It operates
 as if a pigment were already a height field. You can
 perform edge operations on any pigment, including pre-
 declared images. It returns an "edge map" projected
 between <0,0>, <1,0> along the z axis which is the
 default image space.
 
 If using a red/green grayscale image, you can supply an
 8-bit version with image_8_bit() for extra speed.
 
 examples:
 
 #declare my_pigment=pigment{crackle scale .2 form x}
 edge_map(my_pigment, 0.02, 0.001, 0.3)
 
 #declare my_pigment=pigment{image_map{png"mountains.png"interpolate 2}}
 edge_map(my_pigment, 0.02, 0.001, 0.3)
 
 see:
 edged.pov
 edge_maker.pov
 edged2.pov
 
 -----------------------------------------------------------
 angle_map(pigment_name)
 
 pigment_name:  the pigment to be used
 
 Imagine a radial pigment starting at black and going to
 white around a cone. Now imagine such a radial pigment
 applied to a height field, kind of like a slope pigment
 but starting at rgb 0 = 0 degrees, and ending at rgb
 1 = 360 degrees around the y-axis. The center of this
 pattern is at every peak, not just at the center of the
 whole height field.
 
 With this pigment you can effectively map the directions
 of a height field, allowing special pigments to be
 placed according to different angles.
 
 It returns an "angle map" projected between <0,0>, <1,0>
 along the z axis which is the default image space.
 
 If you are using an image for input, make sure it is a
 16-bit height map.
 
 8-bit images will not work correctly with this pigment.
 
 examples:
 
 #declare my_pigment=pigment{crackle scale .2 form x}
 angle_map(my_pigment)
 
 #declare my_pigment=pigment{image_16_bit("mountains.tga")}
 angle_map(my_pigment)
 
 see:

 angle_mapped.pov
 angle_map_advanced.pov

 ----------------------------------------------------------
 
 superellipsoidal(roundness)
 
 roundness:     the roundness of the edges (0-1)
 
 This is the pattern equivalent of the superellipsoid. It
 is mapped between <0,0>, <0,1> along the z axis.
 
 It can be used to taper the edges of height data so that
 a height field can be placed smoothly on a plane.
 
 see: superellipsoidal.pov
 
 -----------------------------------------------------------
 image_16_bit( image_name )
 
 Loads a red/green 16-bit grayscale file as a usable
 pigment.
 
 Supported image formats: png, tga, bmp
 
 example: pigment{ image_16_bit("mountains.tga") }
 
 -----------------------------------------------------------
 image_8_bit( image_name )

 Loads a red/green 16-bit grayscale file as a usable
 pigment. It converts it to 8-bits, which is faster. 
 
 It's just like any other 8-bit grayscale image: ugly
 when used for height_fields or isosurfaces.
 
 Supported image formats: png, tga, bmp
 
 example: pigment{ image_8_bit("mountains.tga") }
 
 ----------------------------------------------------------

 hf_grey_16( pigment )

 pigment_name:  the pigment to be used
 
 Renders a pigment as a red/green 16-bit image. If you use
 this to make a height map, make sure you use
 image_16_bit() or image_8_bit() when you load the map
 back into POV-Ray. It should not be used with hf_gray_16
 or +fng.
 
 example:
 
 #declare my_pigment=pigment{crackle scale .2 form x}
 hf_grey_16(my_pigment)

*/

// edges //

#declare pts_sphere=array[251]{<-0.333333,-0.666667,-1>,<-1.11022e-016,-0.666667,-1>,<0.333333,-0.666667,-1>,<-0.666667,-0.333333,-1>,<-0.333333,-0.333333,-1>,<-1.11022e-016,-0.333333,-1>,<0.333333,-0.333333,-1>,<0.666667,-0.333333,-1>,<-0.666667,-1.11022e-016,-1>,<-0.333333,-1.11022e-016,-1>,<-1.11022e-016,-1.11022e-016,-1>,<0.333333,-1.11022e-016,-1>,<0.666667,-1.11022e-016,-1>,<-0.666667,0.333333,-1>,<-0.333333,0.333333,-1>,<-1.11022e-016,0.333333,-1>,<0.333333,0.333333,-1>,<0.666667,0.333333,-1>,<-0.333333,0.666667,-1>,<-1.11022e-016,0.666667,-1>,<0.333333,0.666667,-1>,<-0.333333,-1,-0.666667>,<-1.11022e-016,-1,-0.666667>,<0.333333,-1,-0.666667>,<-0.666667,-0.666667,-0.666667>,<-0.333333,-0.666667,-0.666667>,<-1.11022e-016,-0.666667,-0.666667>,<0.333333,-0.666667,-0.666667>,<0.666667,-0.666667,-0.666667>,<-1,-0.333333,-0.666667>,<-0.666667,-0.333333,-0.666667>,<-0.333333,-0.333333,-0.666667>,<-1.11022e-016,-0.333333,-0.666667>,<0.333333,-0.333333,-0.666667>,<0.666667,-0.333333,-0.666667>,<1,-0.333333,-0.666667>,<-1,-1.11022e-016,-0.666667>,<-0.666667,-1.11022e-016,-0.666667>,<-0.333333,-1.11022e-016,-0.666667>,<-1.11022e-016,-1.11022e-016,-0.666667>,<0.333333,-1.11022e-016,-0.666667>,<0.666667,-1.11022e-016,-0.666667>,<1,-1.11022e-016,-0.666667>,<-1,0.333333,-0.666667>,<-0.666667,0.333333,-0.666667>,<-0.333333,0.333333,-0.666667>,<-1.11022e-016,0.333333,-0.666667>,<0.333333,0.333333,-0.666667>,<0.666667,0.333333,-0.666667>,<1,0.333333,-0.666667>,<-0.666667,0.666667,-0.666667>,<-0.333333,0.666667,-0.666667>,<-1.11022e-016,0.666667,-0.666667>,<0.333333,0.666667,-0.666667>,<0.666667,0.666667,-0.666667>,<-0.333333,1,-0.666667>,<-1.11022e-016,1,-0.666667>,<0.333333,1,-0.666667>,<-0.666667,-1,-0.333333>,<-0.333333,-1,-0.333333>,<-1.11022e-016,-1,-0.333333>,<0.333333,-1,-0.333333>,<0.666667,-1,-0.333333>,<-1,-0.666667,-0.333333>,<-0.666667,-0.666667,-0.333333>,<-0.333333,-0.666667,-0.333333>,<-1.11022e-016,-0.666667,-0.333333>,<0.333333,-0.666667,-0.333333>,<0.666667,-0.666667,-0.333333>,<1,-0.666667,-0.333333>,<-1,-0.333333,-0.333333>,<-0.666667,-0.333333,-0.333333>,<-0.333333,-0.333333,-0.333333>,<-1.11022e-016,-0.333333,-0.333333>,<0.333333,-0.333333,-0.333333>,<0.666667,-0.333333,-0.333333>,<1,-0.333333,-0.333333>,<-1,-1.11022e-016,-0.333333>,<-0.666667,-1.11022e-016,-0.333333>,<-0.333333,-1.11022e-016,-0.333333>,<-1.11022e-016,-1.11022e-016,-0.333333>,<0.333333,-1.11022e-016,-0.333333>,<0.666667,-1.11022e-016,-0.333333>,<1,-1.11022e-016,-0.333333>,<-1,0.333333,-0.333333>,<-0.666667,0.333333,-0.333333>,<-0.333333,0.333333,-0.333333>,<-1.11022e-016,0.333333,-0.333333>,<0.333333,0.333333,-0.333333>,<0.666667,0.333333,-0.333333>,<1,0.333333,-0.333333>,<-1,0.666667,-0.333333>,<-0.666667,0.666667,-0.333333>,<-0.333333,0.666667,-0.333333>,<-1.11022e-016,0.666667,-0.333333>,<0.333333,0.666667,-0.333333>,<0.666667,0.666667,-0.333333>,<1,0.666667,-0.333333>,<-0.666667,1,-0.333333>,<-0.333333,1,-0.333333>,<-1.11022e-016,1,-0.333333>,<0.333333,1,-0.333333>,<0.666667,1,-0.333333>,<-0.666667,-1,-1.11022e-016>,<-0.333333,-1,-1.11022e-016>,<-1.11022e-016,-1,-1.11022e-016>,<0.333333,-1,-1.11022e-016>,<0.666667,-1,-1.11022e-016>,<-1,-0.666667,-1.11022e-016>,<-0.666667,-0.666667,-1.11022e-016>,<-0.333333,-0.666667,-1.11022e-016>,<-1.11022e-016,-0.666667,-1.11022e-016>,<0.333333,-0.666667,-1.11022e-016>,<0.666667,-0.666667,-1.11022e-016>,<1,-0.666667,-1.11022e-016>,<-1,-0.333333,-1.11022e-016>,<-0.666667,-0.333333,-1.11022e-016>,<-0.333333,-0.333333,-1.11022e-016>,<-1.11022e-016,-0.333333,-1.11022e-016>,<0.333333,-0.333333,-1.11022e-016>,<0.666667,-0.333333,-1.11022e-016>,<1,-0.333333,-1.11022e-016>,<-1,-1.11022e-016,-1.11022e-016>,<-0.666667,-1.11022e-016,-1.11022e-016>,<-0.333333,-1.11022e-016,-1.11022e-016>,<-1.11022e-016,-1.11022e-016,-1.11022e-016>,<0.333333,-1.11022e-016,-1.11022e-016>,<0.666667,-1.11022e-016,-1.11022e-016>,<1,-1.11022e-016,-1.11022e-016>,<-1,0.333333,-1.11022e-016>,<-0.666667,0.333333,-1.11022e-016>,<-0.333333,0.333333,-1.11022e-016>,<-1.11022e-016,0.333333,-1.11022e-016>,<0.333333,0.333333,-1.11022e-016>,<0.666667,0.333333,-1.11022e-016>,<1,0.333333,-1.11022e-016>,<-1,0.666667,-1.11022e-016>,<-0.666667,0.666667,-1.11022e-016>,<-0.333333,0.666667,-1.11022e-016>,<-1.11022e-016,0.666667,-1.11022e-016>,<0.333333,0.666667,-1.11022e-016>,<0.666667,0.666667,-1.11022e-016>,<1,0.666667,-1.11022e-016>,<-0.666667,1,-1.11022e-016>,<-0.333333,1,-1.11022e-016>,<-1.11022e-016,1,-1.11022e-016>,<0.333333,1,-1.11022e-016>,<0.666667,1,-1.11022e-016>,<-0.666667,-1,0.333333>,<-0.333333,-1,0.333333>,<-1.11022e-016,-1,0.333333>,<0.333333,-1,0.333333>,<0.666667,-1,0.333333>,<-1,-0.666667,0.333333>,<-0.666667,-0.666667,0.333333>,<-0.333333,-0.666667,0.333333>,<-1.11022e-016,-0.666667,0.333333>,<0.333333,-0.666667,0.333333>,<0.666667,-0.666667,0.333333>,<1,-0.666667,0.333333>,<-1,-0.333333,0.333333>,<-0.666667,-0.333333,0.333333>,<-0.333333,-0.333333,0.333333>,<-1.11022e-016,-0.333333,0.333333>,<0.333333,-0.333333,0.333333>,<0.666667,-0.333333,0.333333>,<1,-0.333333,0.333333>,<-1,-1.11022e-016,0.333333>,<-0.666667,-1.11022e-016,0.333333>,<-0.333333,-1.11022e-016,0.333333>,<-1.11022e-016,-1.11022e-016,0.333333>,<0.333333,-1.11022e-016,0.333333>,<0.666667,-1.11022e-016,0.333333>,<1,-1.11022e-016,0.333333>,<-1,0.333333,0.333333>,<-0.666667,0.333333,0.333333>,<-0.333333,0.333333,0.333333>,<-1.11022e-016,0.333333,0.333333>,<0.333333,0.333333,0.333333>,<0.666667,0.333333,0.333333>,<1,0.333333,0.333333>,<-1,0.666667,0.333333>,<-0.666667,0.666667,0.333333>,<-0.333333,0.666667,0.333333>,<-1.11022e-016,0.666667,0.333333>,<0.333333,0.666667,0.333333>,<0.666667,0.666667,0.333333>,<1,0.666667,0.333333>,<-0.666667,1,0.333333>,<-0.333333,1,0.333333>,<-1.11022e-016,1,0.333333>,<0.333333,1,0.333333>,<0.666667,1,0.333333>,<-0.333333,-1,0.666667>,<-1.11022e-016,-1,0.666667>,<0.333333,-1,0.666667>,<-0.666667,-0.666667,0.666667>,<-0.333333,-0.666667,0.666667>,<-1.11022e-016,-0.666667,0.666667>,<0.333333,-0.666667,0.666667>,<0.666667,-0.666667,0.666667>,<-1,-0.333333,0.666667>,<-0.666667,-0.333333,0.666667>,<-0.333333,-0.333333,0.666667>,<-1.11022e-016,-0.333333,0.666667>,<0.333333,-0.333333,0.666667>,<0.666667,-0.333333,0.666667>,<1,-0.333333,0.666667>,<-1,-1.11022e-016,0.666667>,<-0.666667,-1.11022e-016,0.666667>,<-0.333333,-1.11022e-016,0.666667>,<-1.11022e-016,-1.11022e-016,0.666667>,<0.333333,-1.11022e-016,0.666667>,<0.666667,-1.11022e-016,0.666667>,<1,-1.11022e-016,0.666667>,<-1,0.333333,0.666667>,<-0.666667,0.333333,0.666667>,<-0.333333,0.333333,0.666667>,<-1.11022e-016,0.333333,0.666667>,<0.333333,0.333333,0.666667>,<0.666667,0.333333,0.666667>,<1,0.333333,0.666667>,<-0.666667,0.666667,0.666667>,<-0.333333,0.666667,0.666667>,<-1.11022e-016,0.666667,0.666667>,<0.333333,0.666667,0.666667>,<0.666667,0.666667,0.666667>,<-0.333333,1,0.666667>,<-1.11022e-016,1,0.666667>,<0.333333,1,0.666667>,<-0.333333,-0.666667,1>,<-1.11022e-016,-0.666667,1>,<0.333333,-0.666667,1>,<-0.666667,-0.333333,1>,<-0.333333,-0.333333,1>,<-1.11022e-016,-0.333333,1>,<0.333333,-0.333333,1>,<0.666667,-0.333333,1>,<-0.666667,-1.11022e-016,1>,<-0.333333,-1.11022e-016,1>,<-1.11022e-016,-1.11022e-016,1>,<0.333333,-1.11022e-016,1>,<0.666667,-1.11022e-016,1>,<-0.666667,0.333333,1>,<-0.333333,0.333333,1>,<-1.11022e-016,0.333333,1>,<0.333333,0.333333,1>,<0.666667,0.333333,1>,<-0.333333,0.666667,1>,<-1.11022e-016,0.666667,1>,<0.333333,0.666667,1>,}

#declare hf16_Rseed=seed(001);

#macro hf16_edge_pigment(ep_pigment,ep_r, ep_rr, ep_q)
 average
 pigment_map{
  #local V=0;
  #while(V<dimension_size(pts_sphere,1))
   [1
    ep_pigment
    translate (pts_sphere[V])*ep_r-<rand(hf16_Rseed)-rand(hf16_Rseed),rand(hf16_Rseed)-rand(hf16_Rseed),rand(hf16_Rseed)-rand(hf16_Rseed)>*ep_rr
   ]
   #local V=V+1/ep_q;
  #end
 }
#end

#macro edge_map(Pigment, rad, rad_rand, quality)
 #local f_plane=function{select(z,0,1)}
 #local f_image=function{pigment{Pigment}}
 #local f_edges=
 function{
  pigment{
   hf16_edge_pigment(
    pigment{function{f_plane(x,y,z-f_image(x,y,0).grey)}},
    rad,
    rad_rand,
    quality
   )
  }
 }
 pigment_pattern{
  function{f_edges(x,y,z+f_image(x,y,0).grey).grey }
  warp{planar}
 }
 
#end

// angle map //

#macro angle_map(Pigment)
 #local Accuracy=.0001;
 #local f_image=function{pigment{Pigment}}
 #local f_angle=
 function{
  pigment{
   radial
   color_map{[0 rgb 0][1 rgb 1]}
   rotate x*90+z*90 //scale<1,-1,1>
  }
 }
 pigment_pattern{
  function{
   f_angle(
    (pow(f_image(x+Accuracy,y,z).gray,2)-pow(f_image(x-Accuracy,y,z).gray,2))/(Accuracy*64),
    (pow(f_image(x,y+Accuracy,z).gray,2)-pow(f_image(x,y-Accuracy,z).gray,2))/(Accuracy*64),
    0
   ).grey
  }
  warp{planar}
 }
#end

// image loading //

#macro image_load(image_name)
 #declare f_img=
 function{
  pigment{
   /*pigment_pattern{boxed translate(x+y) scale 1/2}
   pigment_map{
    [0 rgb 0]
    [1/256*/
     image_map{
      // string manipulation from Rune's illusion.inc
      #local image_format = strlwr(substr(image_name,strlen(image_name)-3,4))
      #if (strcmp(image_format ,".png")=0) png #end
      #if (strcmp(image_format ,".tga")=0) tga #end
      #if (strcmp(image_format ,".bmp")=0) bmp #end
      image_name
      interpolate 2
      once
     }/*
     warp{planar}
    ]
   }*/
  }
 }
#end

// image functions //

#macro image_16_bit(image_name)
 image_load(image_name)
 function{
  f_img(x,y,0).x
  +f_img(x,y,0).y/255
 }
 #undef f_img
#end

#macro image_8_bit(image_name)
 image_load(image_name)
 function{
  f_img(x,y,0).x
 }
 #undef f_img
#end

// hf_gray_16

#macro hf_grey_16(Pigment)
 #local f_pmt=function{pigment{Pigment}}
 average
 pigment_map{
  [1
   function{f_pmt(x,y,z).grey}
   color_map{[0 rgb 0][1 rgb x*2]}
  ]
  [1
   function{f_pmt(x,y,z).grey*255}
   color_map{[0 rgb 0][1 rgb y*2]}
  ]
 }
#end

// new pigments

#macro superellipsoidal(edge)
 #local edge2=int(1/edge)*2;
 pigment_pattern{
  function{select(1-pow(x,edge2)-pow(y,edge2),0,1-pow(x,edge2)-pow(y,edge2))}
  translate(x+y) scale 1/2
 }
#end