// ------------------------------------------------------------
// Persistence of Vision Ray Tracer Include File
// Transmission Spectral Data library for PoV-Ray
// ------------------------------------------------------------
// Converted from ROSCO data sheets by Christopher Shake
// ------------------------------------------------------------
/*
	These are ROSCOLUX gel colors
	
	Rosco uses 'transmission' to define the amount of light passed
	through each filter, but POV-Ray uses 'filter' to identify transmitted
	light that is colored by the object pigment, so each color should be
	used as a color pigment with filter=1
	
	Each color is specified with 'TS_Rxxxx' for Transmission Spectrum
  All are splines of transmission spectrums, needs CIE.inc to convert
  to usable RGB format with 'ReflectiveSpectrum()'
  
  Contents (in 2007 Swatchbook Order):
  [TS_xxxxx] - [overall % trans] - [Name]
  Added Colors Classic Edition:
    R3410 - 92% - Roscosun 1/8 CTO (5500K to 4900K) 
    R3409 - 81% - Roscosun 1/4 CTO (5500K to 4500K)
    R3408 - 73% - Roscosun 1/2 CTO (5500K to 3800K)
    R3411 - 58% - Roscosun 3/4 CTO (5500K to 3200K)
    R3407 - 47% - Roscosun CTO (5500K to 2900K)
    R3444 - 92% - Eighth Straw 1/8 CTS (5500K to 4900K)
    R3443 - 81% - Quarter Straw 1/4 CTS (5500K to 4500K)
    R3442 - 73% - Half Straw 1/2 CTS (5500K to 3800K)
    R3441 - 50% - Full Straw CTS (5500K to 3200K)
    R3405 - 30% - Roscosun 85N.3 (5500K to 3200K)
    R3406 - 17% - Roscosun 85N.6 (5500K to 3200K)
    R4515 - 80% - CalColor 15 Yellow
    R4530 - 75% - CalColor 30 Yellow
    R4560 - 70% - CalColor 60 Yellow
    R4590 - 66% - CalColor 90 Yellow
    R2002 - 23% - Storaro Orange
    R2003 - 64% - Storaro Yellow
    R4615 - 67% - CalColor 15 Red
    R4630 - 55% - CalColor 30 Red
    R4660 - 39% - CalColor 60 Red
    R4690 - 30% - CalColor 90 Red
    R4815 - 71% - CalColor 15 Pink
    R4830 - 61% - CalColor 30 Pink
    R4860 - 46% - CalColor 60 Pink
    R4890 - 38% - CalColor 90 Pink
    R3318 - 89% - Tough 1/8 Minusgreen
    R3314 - 81% - Tough 1/4 Minusgreen
    R3313 - 71% - Tough 1/2 Minusgreen
    R3308 - 55% - Tough Minusgreen
    R4715 - 65% - CalColor 15 Magenta
    R4730 - 52% - CalColor 30 Magenta
    R4760 - 37% - CalColor 60 Magenta
    R4790 - 28% - CalColor 90 Magenta
    R4915 - 81% - CalColor 15 Lavender
    R4930 - 47% - CalColor 30 Lavender
    R4960 - 29% - CalColor 60 Lavender
    R4990 - 18% - CalColor 90 Lavender
    R3216 - 81% - Eighth Blue CTB (3200K to 3300K)
    R3208 - 74% - Quarter Blue CTB (3200K to 3500K)
    R3206 - 64% - Third Blue CTB (3200K to 3800K)
    R3204 - 52% - Half Blue CTB (3200K to 4100K)
    R3203 - 41% - Three Quarter Blue CTB (3200K to 4700K)
    R3202 - 36% - Full Blue CTB (3200K to 5500K)
    R3220 - 10% - Double Blue CTB (3200K to 10000K)
    R4215 - 57% - CalColor 15 Blue
    R4230 - 42% - CalColor 30 Blue
    R4260 - 24% - CalColor 60 Blue
    R4290 - 15% - CalColor 90 Blue
    R2006 - 9% - Storaro Azure
    R2007 - 10% - Storaro Blue
    R2008 - 4% - Storaro Indigo
    R2009 - 3% - Storaro Violet
    R4307 - 79% - CalColor 7.5 Cyan
    R4315 - 72% - CalColor 15 Cyan
    R4330 - 63% - CalColor 30 Cyan
    R4360 - 50% - CalColor 60 Cyan
    R4390 - 39% - CalColor 90 Cyan
    R2005 - 18% - Storaro Cyan
    R3317 - 93% - Tough 1/8 Plusgreen
    R3316 - 92% - Tough 1/4 Plusgreen
    R3315 - 90% - Tough 1/2 Plusgreen
    R3304 - 76% - Tough Plusgreen
    R4415 - 67% - CalColor 15 Green
    R4430 - 52% - CalColor 30 Green
    R4460 - 32% - CalColor 60 Green
    R4490 - 18% - CalColor 90 Green
    R2004 - 15% - Storaro Green
  Standard Book:
    R00 - 100% - Clear
    R01 - 56% - Light Bastard Amber
    R02 - 78% - Bastard Amber
    R302 - 88% - Pale Bastard Amber
    R03 - 62% - Dark Bastard Amber
    R303 - 55% - Warm Peach
    R04 - 66% - Medium Bastard Amber
    R304 - 79% - Pale Apricot
    R05 - 80% - Rose Tint
    R305 - 75% - Rose Gold
    R06 - 92% - No Color Straw
    R07 - 96% - Pale Yellow
    R08 - 86% - Pale Gold
    R09 - 74% - Pale Amber Gold
    R10 - 92% - Medium Yellow
    R310 - 82% - Daffodil
    R11 - 82% - Light Straw
    R12 - 88% - Straw
    R312 - 85% - Canary
    R13 - 78% - Straw Tint
    R313 - 77% - Light Relief Yellow
    R14 - 68% - Medium Straw
    R15 - 65% - Deep Straw
    R16 - 68% - Light Amber
    R316 - 58% - Gallo Gold
    R17 - 56% - Light Flame
    R317 - 51% - Apricot
    R18 - 56% - Flame
    R318 - 52% - Mayan Sun
    R19 - 20% - Fire
    R20 - 54% - Medium Amber
    R21 - 43% - Golden Amber
    R321 - 39% - Golden Amber
    R22 - 26% - Deep Amber
    R23 - 32% - Orange
    R24 - 22% - Scarlet
    R324 - 31% - Gypsy Red
    R25 - 14% - Orange Red
    R325 - 18% - Henna Sky
    R26 - 12% - Light Red
    R27 - 4% - Medium Red
    R30 - 44% - Light Salmon Pink
    R31 - 46% - Salmon Pink
    R331 - 68% - Shell Pink
    R32 - 28% - Medium Salmon Pink
    R332 - 38% - Cherry Rose
    R33 - 65% - No Color Pink
    R333 - 71% - Blush Pink
    R34 - 45% - Flesh Pink
    R35 - 66% - Light Pink
    R36 - 46% - Medium Pink
    R336 - 48% - Billinton Pink
    R37 - 56% - Pale Rose Pink
    R337 - 55% - True Pink
    R38 - 49% - Light Rose
    R39 - 10% - Skeleton Exotic Sangria
    R339 - 15% - Broadway Pink
    R40 - 34% - Light Salmon
    R41 - 24% - Salmon
    R42 - 8% - Deep Salmon
    R342 - 16% - Rose Pink
    R43 - 28% - Deep Pink
    R343 - 33% - Neon Pink
    R44 - 26% - Middle Rose
    R344 - 21% - Follies Pink
    R45 - 8% - Rose
    R46 - 6% - Magenta
    R346 - 22% - Tropical Magenta
    R47 - 16% - Light Rose Purple
    R347 - 6% - Belladonna Rose
    R48 - 16% - Rose Purple
    R348 - 14% - Purple Jazz
    R49 - 4% - Medium Purple
    R349 - 11% - Fischer Fuchsia
    R50 - 14% - Mauve
    R51 - 54% - Surprise Pink
    R351 - 61% - Lavender Mist
    R52 - 26% - Light Lavender
    R53 - 64% - Pale Lavender
    R353 - 29% - Lilly Lavender
    R54 - 50% - Special Lavender
    R55 - 37% - Lilac
    R355 - 20% - Pale Violet
    R56 - 4% - Gypsy Lavender
    R356 - 27% - Middle Lavender
    R57 - 24% - Lavender
    R357 - 5% - Royal Lavender
    R58 - 10% - Deep Lavender
    R358 - 5% - Rose Indigo
    R59 - 2% - Indigo
    R359 - 9% - Medium Violet
    R60 - 62% - No Color Blue
    R360 - 52% - Clearwater
    R61 - 66% - Mist Blue
    R361 - 18% - Hemsley Blue
    R62 - 54% - Booster Blue
    R362 - 32% - Tipton Blue
    R63 - 56% - Pale Blue
    R363 - 52% - Aquamarine
    R64 - 2% - Light Steel Blue
    R364 - 32% - Blue Bell
    R65 - 35% - Daylight Blue
    R365 - 36% - Tharon Delft Blue
    R66 - 67% - Cool Blue
    R366 - 32% - Jordan Blue
    R67 - 26% - Light Sky Blue
    R367 - 20% - Slate Blue
    R68 - 14% - Parry Sky Blue
    R368 - 16% - Winkler Blue
    R69 - 18% - Brilliant Blue
    R369 - 22% - Tahitian Blue
    R70 - 45% - Nile Blue
    R370 - 31% - Italian Blue
    R71 - 30% - Sea Blue
    R371 - 35% - Theatre Booster 1
    R72 - 44% - Azure Blue
    R372 - 55% - Theare Booster 2
    R73 - 28% - Peacock Blue
    R373 - 72% - Theatre Booster 3
    R74 - 4% - Night Blue
    R374 - 26% - Sea Green
    R75 - 11% - Twilight Blue
    R375 - 32% - Cerulean Blue
    R76 - 9% - Light Green Blue
    R376 - 6% - Bermuda Blue
    R77 - 9% - Green Blue
    R377 - 11% - Iris Purple
    R78 - 19% - Trudy Blue
    R378 - 15% - Alice Blue
    R79 - 8% - Bright Blue
    R80 - 9% - Primary Blue
    R81 - 10% - Urban Blue
    R381 - 6% - Baldassari Blue
    R82 - 6% - Surprise Blue
    R382 - 0.56% - Congo Blue
    R83 - 4% - Medium Blue
    R383 - 4% - Sapphire Blue
    R84 - 14% - Zephyr Blue
    R384 - 2% - Midnight Blue
    R85 - 3% - Deep Blue
    R385 - 4% - Royal Blue
    R86 - 56% - Pea Green
    R386 - 32% - Leaf Green
    R87 - 85% - Pale Yellow Green
    R88 - 82% - Light Green
    R388 - 765 - Gaslight Green
    R89 - 45% - Moss Green
    R389 - 40% - Chroma Green
    R90 - 13% - Dark Yellow Green
    R91 - 7% - Primary Green
    R92 - 59% - Turquoise
    R392 - 20% - Pacific Green
    R93 - 35% - Blue Green
    R393 - 14% - Emerald Green
    R94 - 25% - Kelly Green
    R95 - 15% - Medium Blue Green
    R395 - 13% - Teal Green
    R96 - 98% - Lime
    R97 - 50% - Light Grey
    R397 - 70% - Pale Grey
    R98 - 25% - Medium Grey
    R398 - 40% - Neutral Grey
    R99 - 35% - Chocolate

*/

// Standard include file beginning stuff 
#ifndef (tspd_roscolux_Inc_Temp)
#declare tspd_roscolux_Inc_Temp = version;
#version 3.5;

#ifdef(View_POV_Include_Stack)
  #debug "Including tspd_roscolux.inc\n"
#end                   


//-------------------------------------------------------------
// Begin Added Colors Classic Edition Swatchbook
//-------------------------------------------------------------

// #3410 SUN 1/8 CTO
// Amber Correction Lighting Filter to Lower Color Temperature
// Converts 5500K to 4900K
#declare TS_R3410 = spline{ linear_spline
 360, 0.69
 380, 0.72
 400, 0.71
 420, 0.67
 440, 0.65
 460, 0.68
 480, 0.76
 500, 0.77
 520, 0.76
 540, 0.78
 560, 0.78
 580, 0.82
 600, 0.84
 620, 0.85
 640, 0.85
 660, 0.85
 680, 0.85
 700, 0.86
 720, 0.86
 740, 0.86
}

// #3409 SUN 1/4 CTO
// Amber Correction Lighting Filter to Lower Color Temperature.
// Converts 5500K to 4500K
#declare TS_R3409 = spline{ linear_spline
 360, 0.66
 380, 0.69
 400, 0.66
 420, 0.57
 440, 0.52
 460, 0.57
 480, 0.72
 500, 0.72
 520, 0.69
 540, 0.72
 560, 0.72
 580, 0.79
 600, 0.84
 620, 0.85
 640, 0.85
 660, 0.85
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.87
}

// #3408 SUN 1/2 CTO
// Amber Correction Lighting Filter to Lower Color Temperature.
// Converts 5500K to 3800K
#declare TS_R3408 = spline{ linear_spline
 360, 0.53
 380, 0.56
 400, 0.50
 420, 0.34
 440, 0.26
 460, 0.34
 480, 0.59
 500, 0.59
 520, 0.54
 540, 0.58
 560, 0.59
 580, 0.72
 600, 0.83
 620, 0.84
 640, 0.85
 660, 0.85
 680, 0.85
 700, 0.86
 720, 0.86
 740, 0.86
}

// #3411 SUN 3/4 CTO
// Amber Correction Lighting Filter to Lower Color Temperature.
// Converts 5500K to 3200K
#declare TS_R3411 = spline{ linear_spline
 360, 0.56
 380, 0.55
 400, 0.39
 420, 0.19
 440, 0.12
 460, 0.20
 480, 0.46
 500, 0.46
 520, 0.39
 540, 0.44
 560, 0.45
 580, 0.65
 600, 0.82
 620, 0.84
 640, 0.85
 660, 0.85
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.86
}

// #3407 SUN CTO
// Amber Correction Lighting Filter to Lower Color Temperature.
// Converts 5500K to 2900K
#declare TS_R3407 = spline{ linear_spline
 360, 0.46
 380, 0.47
 400, 0.29
 420, 0.12
 440, 0.06
 460, 0.13
 480, 0.37
 500, 0.37
 520, 0.31
 540, 0.35
 560, 0.36
 580, 0.57
 600, 0.76
 620, 0.77
 640, 0.76
 660, 0.75
 680, 0.74
 700, 0.76
 720, 0.80
 740, 0.84
}

// #3444 EIGHTH STRAW
// Straw Correction Lighting Filter to Lower Color Temperature.
// Converts 5500K to 4900K
#declare TS_R3444 = spline{ linear_spline
 360, 0.68
 380, 0.72
 400, 0.71
 420, 0.68
 440, 0.65
 460, 0.69
 480, 0.78
 500, 0.79
 520, 0.78
 540, 0.80
 560, 0.80
 580, 0.82
 600, 0.84
 620, 0.85
 640, 0.85
 660, 0.85
 680, 0.85
 700, 0.85
 720, 0.86
 740, 0.86
}

// #3443 QUARTER STRAW
// Straw Correction Lighting Filter to Lower Color Temperature.
// Converts 5500K to 4500K
#declare TS_R3443 = spline{ linear_spline
 360, 0.65
 380, 0.68
 400, 0.65
 420, 0.57
 440, 0.52
 460, 0.58
 480, 0.74
 500, 0.75
 520, 0.74
 540, 0.76
 560, 0.76
 580, 0.81
 600, 0.84
 620, 0.85
 640, 0.85
 660, 0.85
 680, 0.85
 700, 0.85
 720, 0.86
 740, 0.86
}

// #3442 HALF STRAW
// Straw Correction Lighting Filter to Lower Color Temperature.
// Converts 5500K to 3800K
#declare TS_R3442 = spline{ linear_spline
 360, 0.62
 380, 0.62
 400, 0.51
 420, 0.34
 440, 0.26
 460, 0.35
 480, 0.61
 500, 0.63
 520, 0.59
 540, 0.63
 560, 0.63
 580, 0.74
 600, 0.82
 620, 0.84
 640, 0.84
 660, 0.84
 680, 0.85
 700, 0.85
 720, 0.85
 740, 0.85
}

// #3441 FULL STRAW
// Straw Correction Lighting Filter to Lower Color Temperature.
// Converts 5500K to 3200K
#declare TS_R3441 = spline{ linear_spline
 360, 0.55
 380, 0.52
 400, 0.33
 420, 0.14
 440, 0.07
 460, 0.15
 480, 0.44
 500, 0.46
 520, 0.40
 540, 0.46
 560, 0.46
 580, 0.66
 600, 0.82
 620, 0.85
 640, 0.85
 660, 0.86
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.86
}

// #3405 SUN 85N.3
// Amber Correction Window Filter with Neutral Density
// Converts 5500K to 3200K
#declare TS_R3405 = spline{ linear_spline
 360, 0.35
 380, 0.37
 400, 0.23
 420, 0.08
 440, 0.03
 460, 0.09
 480, 0.28
 500, 0.26
 520, 0.19
 540, 0.23
 560, 0.21
 580, 0.39
 600, 0.51
 620, 0.44
 640, 0.37
 660, 0.32
 680, 0.28
 700, 0.32
 720, 0.51
 740, 0.71
}

// #3406 SUN 85N.6
// Amber Correction Window Filter with Neutral Density
// Converts 5500K to 3200K
#declare TS_R3406 = spline{ linear_spline
 360, 0.25
 380, 0.27
 400, 0.14
 420, 0.03
 440, 0.01
 460, 0.04
 480, 0.15
 500, 0.13
 520, 0.08
 540, 0.10
 560, 0.09
 580, 0.22
 600, 0.32
 620, 0.24
 640, 0.17
 660, 0.13
 680, 0.10
 700, 0.13
 720, 0.31
 740, 0.59
}

// #4515 CALCOLOR 15 YELLOW
// Color Lighting Filter. Enhances Green and Red by 1/2 Stop.
#declare TS_R4515 = spline{ linear_spline
 360, 0.68
 380, 0.70
 400, 0.64
 420, 0.52
 440, 0.46
 460, 0.55
 480, 0.82
 500, 0.86
 520, 0.87
 540, 0.88
 560, 0.88
 580, 0.89
 600, 0.89
 620, 0.89
 640, 0.89
 660, 0.89
 680, 0.90
 700, 0.90
 720, 0.90
 740, 0.91
}

// #4530 CALCOLOR 30 YELLOW
// Color Lighting Filter. Enhances Green and Red by One Stop.
#declare TS_R4530 = spline{ linear_spline
 360, 0.66
 380, 0.65
 400, 0.49
 420, 0.30
 440, 0.22
 460, 0.35
 480, 0.74
 500, 0.81
 520, 0.84
 540, 0.85
 560, 0.85
 580, 0.85
 600, 0.86
 620, 0.86
 640, 0.86
 660, 0.86
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.87
}

// #4560 CALCOLOR 60 YELLOW
// Color Lighting Filter. Enhances Green and Red by Two Stops.
#declare TS_R4560 = spline{ linear_spline
 360, 0.55
 380, 0.45
 400, 0.19
 420, 0.04
 440, 0.02
 460, 0.10
 480, 0.58
 500, 0.77
 520, 0.82
 540, 0.85
 560, 0.85
 580, 0.85
 600, 0.86
 620, 0.86
 640, 0.86
 660, 0.86
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.86
}

// #4590 CALCOLOR 90 YELLOW
// Color Lighting Filter. Enhances Green and Red by Three Stops.
#declare TS_R4590 = spline{ linear_spline
 360, 0.36
 380, 0.20
 400, 0.03
 420, 0.00
 440, 0.00
 460, 0.02
 480, 0.34
 500, 0.65
 520, 0.78
 540, 0.82
 560, 0.84
 580, 0.84
 600, 0.85
 620, 0.85
 640, 0.85
 660, 0.85
 680, 0.85
 700, 0.85
 720, 0.85
 740, 0.86
}

// #2002 VS ORANGE
// Color Effects Lighting Filter. Deep Orange.
#declare TS_R2002 = spline{ linear_spline
 360, 0.35
 380, 0.31
 400, 0.12
 420, 0.02
 440, 0.01
 460, 0.03
 480, 0.17
 500, 0.16
 520, 0.11
 540, 0.14
 560, 0.14
 580, 0.36
 600, 0.65
 620, 0.67
 640, 0.63
 660, 0.60
 680, 0.58
 700, 0.61
 720, 0.71
 740, 0.80
}

// #2003 VS YELLOW
// Color Effects Lighting Filter. Deep Yellow.
#declare TS_R2003 = spline{ linear_spline
 360, 0.47
 380, 0.44
 400, 0.20
 420, 0.05
 440, 0.01
 460, 0.03
 480, 0.09
 500, 0.16
 520, 0.28
 540, 0.62
 560, 0.82
 580, 0.85
 600, 0.85
 620, 0.86
 640, 0.86
 660, 0.86
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.86
}

// #4615 CALCOLOR 15 RED
// Color Lighting Filter. Enhances Red by 1/2 Stop.
#declare TS_R4615 = spline{ linear_spline
 360, 0.66
 380, 0.70
 400, 0.67
 420, 0.58
 440, 0.52
 460, 0.56
 480, 0.67
 500, 0.64
 520, 0.59
 540, 0.85
 560, 0.61
 580, 0.66
 600, 0.80
 620, 0.85
 640, 0.86
 660, 0.86
 680, 0.86
 700, 0.86
 720, 0.87
 740, 0.87
}

// #4630 CALCOLOR 30 RED
// Color Lighting Filter. Enhances Red by One Stop.
#declare TS_R4630 = spline{ linear_spline
 360, 0.60
 380, 0.64
 400, 0.54
 420, 0.37
 440, 0.28
 460, 0.34
 480, 0.52
 500, 0.46
 520, 0.40
 540, 0.38
 560, 0.42
 580, 0.50
 600, 0.73
 620, 0.83
 640, 0.85
 660, 0.85
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.87
}

// #4660 CALCOLOR 60 RED
// Color Lighting Filter. Enhances Red by Two Stops.
#declare TS_R4660 = spline{ linear_spline
 360, 0.49
 380, 0.52
 400, 0.34
 420, 0.15
 440, 0.08
 460, 0.14
 480, 0.33
 500, 0.27
 520, 0.20
 540, 0.18
 560, 0.23
 580, 0.31
 600, 0.63
 620, 0.81
 640, 0.84
 660, 0.85
 680, 0.85
 700, 0.86
 720, 0.86
 740, 0.87
}

// #4690 CALCOLOR 90 RED
// Color Lighting Filter. Enhances Red by Three Stops.
#declare TS_R4690 = spline{ linear_spline
 360, 0.39
 380, 0.40
 400, 0.18
 420, 0.04
 440, 0.01
 460, 0.05
 480, 0.18
 500, 0.13
 520, 0.09
 540, 0.07
 560, 0.10
 580, 0.17
 600, 0.52
 620, 0.77
 640, 0.83
 660, 0.84
 680, 0.85
 700, 0.86
 720, 0.86
 740, 0.86
}

// #4815 CALCOLOR 15 PINK
// Color Lighting Filter. Enhances Red and Magenta.
#declare TS_R4815 = spline{ linear_spline
 360, 0.67
 380, 0.72
 400, 0.72
 420, 0.69
 440, 0.66
 460, 0.66
 480, 0.67
 500, 0.61
 520, 0.57
 540, 0.55
 560, 0.59
 580, 0.64
 600, 0.78
 620, 0.83
 640, 0.84
 660, 0.84
 680, 0.84
 700, 0.84
 720, 0.84
 740, 0.85
}

// #4830 CALCOLOR 30 PINK
// Color Lighting Filter. Enhances Red and Magenta.
#declare TS_R4830 = spline{ linear_spline
 360, 0.62
 380, 0.69
 400, 0.66
 420, 0.57
 440, 0.50
 460, 0.49
 480, 0.52
 500, 0.43
 520, 0.36
 540, 0.34
 560, 0.39
 580, 0.47
 600, 0.70
 620, 0.81
 640, 0.83
 660, 0.83
 680, 0.83
 700, 0.83
 720, 0.84
 740, 0.84
}

// #4860 CALCOLOR 60 PINK
// Color Lighting Filter. Enhances Red and Magenta.
#declare TS_R4860 = spline{ linear_spline
 360, 0.55
 380, 0.63
 400, 0.57
 420, 0.43
 440, 0.36
 460, 0.33
 480, 0.37
 500, 0.27
 520, 0.20
 540, 0.18
 560, 0.22
 580, 0.31
 600, 0.62
 620, 0.80
 640, 0.83
 660, 0.83
 680, 0.83
 700, 0.83
 720, 0.83
 740, 0.83
}

// #4890 CALCOLOR 90 PINK
// Color Lighting Filter. Enhances Red and Magenta.
#declare TS_R4890 = spline{ linear_spline
 360, 0.47
 380, 0.56
 400, 0.45
 420, 0.28
 440, 0.18
 460, 0.18
 480, 0.23
 500, 0.14
 520, 0.09
 540, 0.07
 560, 0.10
 580, 0.18
 600, 0.53
 620, 0.77
 640, 0.82
 660, 0.82
 680, 0.82
 700, 0.82
 720, 0.83
 740, 0.83
}

// #3318 TOUGH 1/8 MINUSGREEN
// Color Balancing Lighting Filter. Removes Green.
#declare TS_R3318 = spline{ linear_spline
 360, 0.63
 380, 0.68
 400, 0.72
 420, 0.74
 440, 0.75
 460, 0.75
 480, 0.75
 500, 0.74
 520, 0.73
 540, 0.74
 560, 0.75
 580, 0.78
 600, 0.81
 620, 0.81
 640, 0.82
 660, 0.82
 680, 0.83
 700, 0.84
 720, 0.84
 740, 0.84
}

// #3314 TOUGH 1/4 MINUSGREEN
// Color Balancing Lighting Filter. Removes Green.
#declare TS_R3314 = spline{ linear_spline
 360, 0.65
 380, 0.71
 400, 0.75
 420, 0.77
 440, 0.77
 460, 0.76
 480, 0.75
 500, 0.72
 520, 0.69
 540, 0.68
 560, 0.70
 580, 0.72
 600, 0.79
 620, 0.81
 640, 0.81
 660, 0.83
 680, 0.85
 700, 0.86
 720, 0.86
 740, 0.86
}

// #3313 TOUGH 1/2 MINUSGREEN
// Color Balancing Lighting Filter. Removes Green.
#declare TS_R3313 = spline{ linear_spline
 360, 0.61
 380, 0.69
 400, 0.72
 420, 0.74
 440, 0.74
 460, 0.71
 480, 0.67
 500, 0.62
 520, 0.57
 540, 0.56
 560, 0.56
 580, 0.63
 600, 0.74
 620, 0.79
 640, 0.80
 660, 0.82
 680, 0.84
 700, 0.84
 720, 0.85
 740, 0.85
}

// #3308 TOUGH MINUSGREEN
// Color Balancing Lighting Filter. Removes Green.
#declare TS_R3308 = spline{ linear_spline
 360, 0.63
 380, 0.71
 400, 0.74
 420, 0.74
 440, 0.72
 460, 0.67
 480, 0.60
 500, 0.52
 520, 0.45
 540, 0.43
 560, 0.47
 580, 0.53
 600, 0.71
 620, 0.78
 640, 0.79
 660, 0.81
 680, 0.84
 700, 0.85
 720, 0.86
 740, 0.86
}

// #4715 CALCOLOR 15 MAGENTA
// Color Lighting Filter. Enhances Red and Blue by 1/2 Stop.
#declare TS_R4715 = spline{ linear_spline
 360, 0.66
 380, 0.73
 400, 0.76
 420, 0.78
 440, 0.76
 460, 0.72
 480, 0.67
 500, 0.60
 520, 0.55
 540, 0.52
 560, 0.55
 580, 0.59
 600, 0.71
 620, 0.76
 640, 0.77
 660, 0.79
 680, 0.83
 700, 0.85
 720, 0.86
 740, 0.86
}

// #4730 CALCOLOR 30 MAGENTA
// Color Lighting Filter. Enhances Red and Blue by One Stop.
#declare TS_R4730 = spline{ linear_spline
 360, 0.62
 380, 0.72
 400, 0.75
 420, 0.75
 440, 0.71
 460, 0.63
 480, 0.53
 500, 0.42
 520, 0.34
 540, 0.31
 560, 0.34
 580, 0.40
 600, 0.61
 620, 0.70
 640, 0.71
 660, 0.75
 680, 0.82
 700, 0.85
 720, 0.86
 740, 0.86
}

// #4760 CALCOLOR 60 MAGENTA
// Color Lighting Filter. Enhances Red and Blue by Two Stops.
#declare TS_R4760 = spline{ linear_spline
 360, 0.51
 380, 0.66
 400, 0.70
 420, 0.69
 440, 0.61
 460, 0.46
 480, 0.31
 500, 0.19
 520, 0.12
 540, 0.10
 560, 0.12
 580, 0.17
 600, 0.41
 620, 0.56
 640, 0.58
 660, 0.65
 680, 0.77
 700, 0.83
 720, 0.85
 740, 0.85
}

// #4790 CALCOLOR 90 MAGENTA
// Color Lighting Filter. Enhances Red and Blue by Three Stops.
#declare TS_R4790 = spline{ linear_spline
 360, 0.46
 380, 0.62
 400, 0.67
 420, 0.66
 440, 0.56
 460, 0.39
 480, 0.24
 500, 0.12
 520, 0.07
 540, 0.05
 560, 0.07
 580, 0.11
 600, 0.35
 620, 0.51
 640, 0.53
 660, 0.61
 680, 0.75
 700, 0.82
 720, 0.84
 740, 0.85
}

// #4915 CALCOLOR 15 LAVENDER
// Color Lighting Filter. Enhances Blue and Magenta.
#declare TS_R4915 = spline{ linear_spline
 360, 0.64
 380, 0.70
 400, 0.74
 420, 0.75
 440, 0.74
 460, 0.69
 480, 0.63
 500, 0.56
 520, 0.49
 540, 0.49
 560, 0.46
 580, 0.51
 600, 0.56
 620, 0.56
 640, 0.53
 660, 0.64
 680, 0.79
 700, 0.83
 720, 0.84
 740, 0.84
}

// #4930 CALCOLOR 30 LAVENDER
// Color Lighting Filter. Enhances Blue and Magenta.
#declare TS_R4930 = spline{ linear_spline
 360, 0.60
 380, 0.67
 400, 0.72
 420, 0.73
 440, 0.69
 460, 0.61
 480, 0.50
 500, 0.42
 520, 0.32
 540, 0.33
 560, 0.29
 580, 0.37
 600, 0.43
 620, 0.44
 640, 0.41
 660, 0.55
 680, 0.76
 700, 0.82
 720, 0.83
 740, 0.84
}

// #49605 CALCOLOR 60 LAVENDER
// Color Lighting Filter. Enhances Blue and Magenta.
#declare TS_R4960 = spline{ linear_spline
 360, 0.50
 380, 0.61
 400, 0.67
 420, 0.67
 440, 0.59
 460, 0.45
 480, 0.30
 500, 0.21
 520, 0.12
 540, 0.13
 560, 0.10
 580, 0.17
 600, 0.24
 620, 0.24
 640, 0.21
 660, 0.38
 680, 0.68
 700, 0.80
 720, 0.83
 740, 0.83
}

// #4990 CALCOLOR 90 LAVENDER
// Color Lighting Filter. Enhances Blue and Magenta.
#declare TS_R4990 = spline{ linear_spline
 360, 0.37
 380, 0.48
 400, 0.57
 420, 0.58
 440, 0.49
 460, 0.34
 480, 0.20
 500, 0.11
 520, 0.05
 540, 0.05
 560, 0.04
 580, 0.08
 600, 0.12
 620, 0.12
 640, 0.11
 660, 0.21
 680, 0.48
 700, 0.69
 720, 0.78
 740, 0.81
}

// #3216 EIGHTH BLUE CTB
// Blue Correction Lighting Filter to Raise Color Temperature.
// Converts 3200K to 3300K
#declare TS_R3216 = spline{ linear_spline
 360, 0.65
 380, 0.70
 400, 0.74
 420, 0.76
 440, 0.77
 460, 0.77
 480, 0.76
 500, 0.75
 520, 0.72
 540, 0.73
 560, 0.71
 580, 0.72
 600, 0.71
 620, 0.69
 640, 0.68
 660, 0.66
 680, 0.69
 700, 0.76
 720, 0.82
 740, 0.85
}

// #3208 QUARTER BLUE CTB
// Blue Correction Lighting Filter to Raise Color Temperature.
// Converts 3200K to 3500K
#declare TS_R3208 = spline{ linear_spline
 360, 0.58
 380, 0.64
 400, 0.66
 420, 0.72
 440, 0.73
 460, 0.71
 480, 0.68
 500, 0.65
 520, 0.51
 540, 0.60
 560, 0.57
 580, 0.59
 600, 0.57
 620, 0.54
 640, 0.52
 660, 0.50
 680, 0.54
 700, 0.66
 720, 0.76
 740, 0.81
}

// #3206 THIRD BLUE CTB
// Blue Correction Lighting Filter to Raise Color Temperature.
// Converts 3200K to 3800K
#declare TS_R3206 = spline{ linear_spline
 360, 0.54
 380, 0.61
 400, 0.67
 420, 0.72
 440, 0.73
 460, 0.70
 480, 0.65
 500, 0.61
 520, 0.55
 540, 0.55
 560, 0.51
 580, 0.52
 600, 0.49
 620, 0.45
 640, 0.43
 660, 0.40
 680, 0.47
 700, 0.62
 720, 0.75
 740, 0.82
}

// #3204 HALF BLUE CTB
// Blue Correction Lighting Filter to Raise Color Temperature.
// Converts 3200K to 4100K
#declare TS_R3204 = spline{ linear_spline
 360, 0.52
 380, 0.59
 400, 0.65
 420, 0.70
 440, 0.71
 460, 0.68
 480, 0.62
 500, 0.57
 520, 0.51
 540, 0.50
 560, 0.45
 580, 0.46
 600, 0.42
 620, 0.38
 640, 0.35
 660, 0.34
 680, 0.42
 700, 0.59
 720, 0.74
 740, 0.81
}

// #3203 3/4 BLUE CTB
// Blue Correction Lighting Filter to Raise Color Temperature.
// Converts 3200K to 4700K
#declare TS_R3203 = spline{ linear_spline
 360, 0.46
 380, 0.54
 400, 0.60
 420, 0.68
 440, 0.69
 460, 0.64
 480, 0.56
 500, 0.50
 520, 0.42
 540, 0.41
 560, 0.35
 580, 0.36
 600, 0.32
 620, 0.26
 640, 0.24
 660, 0.22
 680, 0.28
 700, 0.47
 720, 0.68
 740, 0.80
}

// #3202 FULL BLUE CTB
// Blue Correction Lighting Filter to Raise Color Temperature.
// Converts 3200K to 5500K
#declare TS_R3202 = spline{ linear_spline
 360, 0.40
 380, 0.49
 400, 0.56
 420, 0.66
 440, 0.66
 460, 0.59
 480, 0.49
 500, 0.41
 520, 0.33
 540, 0.32
 560, 0.26
 580, 0.27
 600, 0.23
 620, 0.18
 640, 0.16
 660, 0.14
 680, 0.19
 700, 0.38
 720, 0.62
 740, 0.76
}

// #3220 DOUBLE BLUE CTB
// Blue Correction Lighting Filter to Raise Color Temperature.
// Converts 3200K to 10000K
#declare TS_R3220 = spline{ linear_spline
 360, 0.23
 380, 0.31
 400, 0.40
 420, 0.53
 440, 0.53
 460, 0.41
 480, 0.27
 500, 0.18
 520, 0.11
 540, 0.10
 560, 0.06
 580, 0.07
 600, 0.05
 620, 0.03
 640, 0.02
 660, 0.02
 680, 0.04
 700, 0.15
 720, 0.42
 740, 0.67
}

// #4215 CALCOLOR 15 BLUE
// Color Lighting Filter. Enhances Blue by 1/2 Stop.
#declare TS_R4215 = spline{ linear_spline
 360, 0.62
 380, 0.69
 400, 0.74
 420, 0.75
 440, 0.72
 460, 0.68
 480, 0.63
 500, 0.56
 520, 0.54
 540, 0.49
 560, 0.49
 580, 0.49
 600, 0.48
 620, 0.46
 640, 0.46
 660, 0.56
 680, 0.73
 700, 0.82
 720, 0.84
 740, 0.85
}

// #4230 CALCOLOR 30 BLUE
// Color Lighting Filter. Enhances Blue by One Stop.
#declare TS_R4230 = spline{ linear_spline
 360, 0.57
 380, 0.66
 400, 0.72
 420, 0.73
 440, 0.71
 460, 0.65
 480, 0.57
 500, 0.50
 520, 0.40
 540, 0.37
 560, 0.31
 580, 0.30
 600, 0.30
 620, 0.29
 640, 0.27
 660, 0.39
 680, 0.64
 700, 0.79
 720, 0.83
 740, 0.85
}

// #4260 CALCOLOR 60 BLUE
// Color Lighting Filter. Enhances Blue by Two Stops.
#declare TS_R4260 = spline{ linear_spline
 360, 0.47
 380, 0.56
 400, 0.66
 420, 0.67
 440, 0.62
 460, 0.52
 480, 0.40
 500, 0.30
 520, 0.20
 540, 0.17
 560, 0.12
 580, 0.12
 600, 0.13
 620, 0.12
 640, 0.11
 660, 0.22
 680, 0.51
 700, 0.73
 720, 0.80
 740, 0.82
}

// #4290 CALCOLOR 90 BLUE
// Color Lighting Filter. Enhances Blue by Three Stops.
#declare TS_R4290 = spline{ linear_spline
 360, 0.39
 380, 0.52
 400, 0.62
 420, 0.63
 440, 0.54
 460, 0.40
 480, 0.26
 500, 0.16
 520, 0.08
 540, 0.06
 560, 0.03
 580, 0.03
 600, 0.03
 620, 0.03
 640, 0.03
 660, 0.11
 680, 0.48
 700, 0.74
 720, 0.81
 740, 0.83
}

// #2006 VS AZURE
// Color Effects Lighting Filter. Deep Greenish-Blue.
#declare TS_R2006 = spline{ linear_spline
 360, 0.35
 380, 0.48
 400, 0.60
 420, 0.63
 440, 0.58
 460, 0.51
 480, 0.44
 500, 0.31
 520, 0.15
 540, 0.05
 560, 0.01
 580, 0.00
 600, 0.00
 620, 0.00
 640, 0.00
 660, 0.02
 680, 0.29
 700, 0.64
 720, 0.77
 740, 0.81
}

// #2007 VS BLUE
// Color Effects Lighting Filter. Deep Reddish-Blue.
#declare TS_R2007 = spline{ linear_spline
 360, 0.23
 380, 0.31
 400, 0.40
 420, 0.53
 440, 0.53
 460, 0.41
 480, 0.27
 500, 0.18
 520, 0.11
 540, 0.10
 560, 0.06
 580, 0.07
 600, 0.05
 620, 0.03
 640, 0.02
 660, 0.02
 680, 0.04
 700, 0.15
 720, 0.42
 740, 0.67
}

// #2008 VS INDIGO
// Color Effects Lighting Filter. Deep Purple.
#declare TS_R2008 = spline{ linear_spline
 360, 0.11
 380, 0.18
 400, 0.24
 420, 0.33
 440, 0.27
 460, 0.16
 480, 0.08
 500, 0.04
 520, 0.01
 540, 0.01
 560, 0.01
 580, 0.02
 600, 0.02
 620, 0.01
 640, 0.01
 660, 0.00
 680, 0.01
 700, 0.08
 720, 0.32
 740, 0.60
}

// #2009 VS VIOLET
// Color Effects Lighting Filter. Deep Violet.
#declare TS_R2009 = spline{ linear_spline
 360, 0.29
 380, 0.44
 400, 0.40
 420, 0.26
 440, 0.15
 460, 0.11
 480, 0.10
 500, 0.04
 520, 0.01
 540, 0.01
 560, 0.01
 580, 0.01
 600, 0.03
 620, 0.06
 640, 0.05
 660, 0.14
 680, 0.48
 700, 0.73
 720, 0.81
 740, 0.83
}

// #4307 CALCOLOR 7.5 CYAN
// Color Lighting Filter. Enhances Blue and Green by 1/4 Stop.
#declare TS_R4307 = spline{ linear_spline
 360, 0.67
 380, 0.72
 400, 0.76
 420, 0.78
 440, 0.79
 460, 0.81
 480, 0.83
 500, 0.83
 520, 0.83
 540, 0.83
 560, 0.80
 580, 0.78
 600, 0.75
 620, 0.71
 640, 0.67
 660, 0.64
 680, 0.62
 700, 0.65
 720, 0.74
 740, 0.82
}

// #4315 CALCOLOR 15 CYAN
// Color Lighting Filter. Enhances Blue and Green by 1/2 Stop.
#declare TS_R4215 = spline{ linear_spline
 360, 0.61
 380, 0.68
 400, 0.71
 420, 0.73
 440, 0.74
 460, 0.76
 480, 0.79
 500, 0.79
 520, 0.79
 540, 0.78
 560, 0.75
 580, 0.71
 600, 0.66
 620, 0.59
 640, 0.53
 660, 0.49
 680, 0.47
 700, 0.51
 720, 0.64
 740, 0.76
}

// #4330 CALCOLOR 30 CYAN
// Color Lighting Filter. Enhances Blue and Green by One Stop.
#declare TS_R4330 = spline{ linear_spline
 360, 0.57
 380, 0.66
 400, 0.72
 420, 0.73
 440, 0.74
 460, 0.77
 480, 0.81
 500, 0.81
 520, 0.79
 540, 0.76
 560, 0.70
 580, 0.63
 600, 0.53
 620, 0.43
 640, 0.35
 660, 0.30
 680, 0.27
 700, 0.31
 720, 0.51
 740, 0.71
}

// #4360 CALCOLOR 60 CYAN
// Color Lighting Filter. Enhances Blue and Green by Two Stops.
#declare TS_R4360 = spline{ linear_spline
 360, 0.43
 380, 0.55
 400, 0.61
 420, 0.59
 440, 0.58
 460, 0.64
 480, 0.76
 500, 0.75
 520, 0.71
 540, 0.65
 560, 0.55
 580, 0.42
 600, 0.30
 620, 0.19
 640, 0.12
 660, 0.09
 680, 0.07
 700, 0.10
 720, 0.27
 740, 0.55
}

// #4390 CALCOLOR 90 CYAN
// Color Lighting Filter. Enhances Blue and Green by Three Stops.
#declare TS_R4390 = spline{ linear_spline
 360, 0.31
 380, 0.45
 400, 0.53
 420, 0.50
 440, 0.47
 460, 0.56
 480, 0.72
 500, 0.70
 520, 0.64
 540, 0.54
 560, 0.41
 580, 0.27
 600, 0.16
 620, 0.07
 640, 0.04
 660, 0.02
 680, 0.02
 700, 0.03
 720, 0.13
 740, 0.41
}

// #2005 VS CYAN
// Color Effects Lighting Filter. Deep Cyan.
#declare TS_R2005 = spline{ linear_spline
 360, 0.20
 380, 0.25
 400, 0.25
 420, 0.26
 440, 0.24
 460, 0.30
 480, 0.47
 500, 0.45
 520, 0.37
 540, 0.27
 560, 0.16
 580, 0.08
 600, 0.03
 620, 0.01
 640, 0.01
 660, 0.01
 680, 0.02
 700, 0.11
 720, 0.37
 740, 0.62
}

// #3317 TOUGH 1/8 PLUSGREEN
// Color Balancing Lighting Filter. Adds Green.
#declare TS_R3317 = spline{ linear_spline
 360, 0.67
 380, 0.72
 400, 0.72
 420, 0.70
 440, 0.68
 460, 0.72
 480, 0.81
 500, 0.82
 520, 0.83
 540, 0.83
 560, 0.83
 580, 0.82
 600, 0.82
 620, 0.82
 640, 0.82
 660, 0.82
 680, 0.83
 700, 0.84
 720, 0.85
 740, 0.85
}

// #3316 TOUGH 1/4 PLUSGREEN
// Color Balancing Lighting Filter. Adds Green.
#declare TS_R3316 = spline{ linear_spline
 360, 0.66
 380, 0.69
 400, 0.69
 420, 0.65
 440, 0.62
 460, 0.67
 480, 0.79
 500, 0.81
 520, 0.81
 540, 0.82
 560, 0.81
 580, 0.80
 600, 0.79
 620, 0.77
 640, 0.75
 660, 0.74
 680, 0.73
 700, 0.74
 720, 0.79
 740, 0.83
}

// #3315 TOUGH 1/2 PLUSGREEN
// Color Balancing Lighting Filter. Adds Green.
#declare TS_R3315 = spline{ linear_spline
 360, 0.60
 380, 0.64
 400, 0.60
 420, 0.51
 440, 0.45
 460, 0.54
 480, 0.76
 500, 0.79
 520, 0.79
 540, 0.79
 560, 0.77
 580, 0.75
 600, 0.71
 620, 0.67
 640, 0.64
 660, 0.61
 680, 0.60
 700, 0.65
 720, 0.74
 740, 0.81
}

// #3304 TOUGH PLUSGREEN
// Color Balancing Lighting Filter. Adds Green.
#declare TS_R3304 = spline{ linear_spline
 360, 0.54
 380, 0.56
 400, 0.46
 420, 0.30
 440, 0.22
 460, 0.34
 480, 0.70
 500, 0.77
 520, 0.77
 540, 0.76
 560, 0.74
 580, 0.70
 600, 0.64
 620, 0.57
 640, 0.52
 660, 0.48
 680, 0.46
 700, 0.51
 720, 0.65
 740, 0.78
}

// #4415 CALCOLOR 15 GREEN
// Color Lighting Filter. Enhances Green by 1/2 Stop.
#declare TS_R4415 = spline{ linear_spline
 360, 0.61
 380, 0.65
 400, 0.59
 420, 0.47
 440, 0.40
 460, 0.50
 480, 0.75
 500, 0.79
 520, 0.80
 540, 0.80
 560, 0.77
 580, 0.74
 600, 0.69
 620, 0.62
 640, 0.56
 660, 0.52
 680, 0.50
 700, 0.54
 720, 0.67
 740, 0.79
}

// #4430 CALCOLOR 30 GREEN
// Color Lighting Filter. Enhances Green by One Stop.
#declare TS_R4430 = spline{ linear_spline
 360, 0.50
 380, 0.52
 400, 0.39
 420, 0.21
 440, 0.14
 460, 0.26
 480, 0.63
 500, 0.71
 520, 0.73
 540, 0.72
 560, 0.68
 580, 0.61
 600, 0.52
 620, 0.41
 640, 0.33
 660, 0.28
 680, 0.25
 700, 0.30
 720, 0.49
 740, 0.69
}

// #4460 CALCOLOR 60 GREEN
// Color Lighting Filter. Enhances Green by Two Stops.
#declare TS_R4460 = spline{ linear_spline
 360, 0.30
 380, 0.29
 400, 0.12
 420, 0.02
 440, 0.01
 460, 0.06
 480, 0.39
 500, 0.55
 520, 0.59
 540, 0.59
 560, 0.52
 580, 0.42
 600, 0.30
 620, 0.18
 640, 0.12
 660, 0.08
 680, 0.07
 700, 0.09
 720, 0.25
 740, 0.53
}

// #4490 CALCOLOR 90 GREEN
// Color Lighting Filter. Enhances Green by Three Stops.
#declare TS_R4490 = spline{ linear_spline
 360, 0.12
 380, 0.09
 400, 0.01
 420, 0.00
 440, 0.00
 460, 0.01
 480, 0.15
 500, 0.32
 520, 0.40
 540, 0.42
 560, 0.35
 580, 0.23
 600, 0.13
 620, 0.05
 640, 0.02
 660, 0.01
 680, 0.01
 700, 0.02
 720, 0.09
 740, 0.34
}

// #2004 VS GREEN
// Color Effects Lighting Filter. Deep Green.
#declare TS_R2004 = spline{ linear_spline
 360, 0.04
 380, 0.02
 400, 0.00
 420, 0.00
 440, 0.00
 460, 0.00
 480, 0.08
 500, 0.24
 520, 0.25
 540, 0.19
 560, 0.10
 580, 0.04
 600, 0.01
 620, 0.00
 640, 0.00
 660, 0.00
 680, 0.01
 700, 0.06
 720, 0.27
 740, 0.57
}

//-------------------------------------------------------------
// Begin Standard Swatchbook
//-------------------------------------------------------------

// #00 CLEAR
#declare TS_R00 = spline{ linear_spline
 360, 1.00
 740, 1.00
}

// #01 LT BASTARD AMBER
// Color Effects Lighting Filter
#declare TS_R01 = spline{ linear_spline
 360, 0.46
 380, 0.59
 400, 0.62
 420, 0.58
 440, 0.47
 460, 0.36
 480, 0.29
 500, 0.27
 520, 0.30
 540, 0.38
 560, 0.50
 580, 0.69
 600, 0.82
 620, 0.86
 640, 0.88
 660, 0.88
 680, 0.88
 700, 0.88
 720, 0.88
 740, 0.88
}

// #02 BASTARD AMBER
// Color Effects Lighting Filter
#declare TS_R02 = spline{ linear_spline
 360, 0.69
 380, 0.73
 400, 0.71
 420, 0.65
 440, 0.56
 460, 0.49
 480, 0.47
 500, 0.49
 520, 0.54
 540, 0.58
 560, 0.74
 580, 0.80
 600, 0.84
 620, 0.85
 640, 0.85
 660, 0.86
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.86
}

// #302 PALE BASTARD AMBER
// Color Effects Lighting Filter
#declare TS_R302 = spline{ linear_spline
 360, 0.78
 380, 0.78
 400, 0.78
 420, 0.74
 440, 0.69
 460, 0.69
 480, 0.72
 500, 0.73
 520, 0.74
 540, 0.78
 560, 0.80
 580, 0.83
 600, 0.86
 620, 0.86
 640, 0.86
 660, 0.86
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.86
}

// #03 DRK BASTARD AMBER
// Color Effects Lighting Filter
#declare TS_R03 = spline{ linear_spline
 360, 0.46
 380, 0.54
 400, 0.53
 420, 0.49
 440, 0.40
 460, 0.33
 480, 0.31
 500, 0.32
 520, 0.38
 540, 0.47
 560, 0.56
 580, 0.72
 600, 0.82
 620, 0.85
 640, 0.85
 660, 0.86
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.86
}

// #303 WARM PEACH
// Color Effects Lighting Filter
#declare TS_R303 = spline{ linear_spline
 360, 0.36
 380, 0.48
 400, 0.46
 420, 0.38
 440, 0.26
 460, 0.18
 480, 0.15
 500, 0.16
 520, 0.20
 540, 0.32
 560, 0.45
 580, 0.69
 600, 0.86
 620, 0.90
 640, 0.90
 660, 0.90
 680, 0.90
 700, 0.91
 720, 0.90
 740, 0.91
}

// #04 MED BASTARD AMBER
// Color Effects Lighting Filter
#declare TS_R04 = spline{ linear_spline
 360, 0.49
 380, 0.54
 400, 0.53
 420, 0.52
 440, 0.48
 460, 0.42
 480, 0.38
 500, 0.38
 520, 0.41
 540, 0.50
 560, 0.60
 580, 0.75
 600, 0.85
 620, 0.87
 640, 0.88
 660, 0.88
 680, 0.88
 700, 0.88
 720, 0.88
 740, 0.88
}

// #304 PALE APRICOT
// Color Effects Lighting Filter
#declare TS_R304 = spline{ linear_spline
 360, 0.60
 380, 0.68
 400, 0.68
 420, 0.64
 440, 0.57
 460, 0.49
 480, 0.45
 500, 0.46
 520, 0.50
 540, 0.58
 560, 0.66
 580, 0.79
 600, 0.86
 620, 0.88
 640, 0.89
 660, 0.89
 680, 0.89
 700, 0.89
 720, 0.89
 740, 0.89
}

// #05 ROSE TINT
// Color Effects Lighting Filter
#declare TS_R05 = spline{ linear_spline
 360, 0.62
 380, 0.72
 400, 0.75
 420, 0.75
 440, 0.70
 460, 0.64
 480, 0.59
 500, 0.57
 520, 0.58
 540, 0.64
 560, 0.71
 580, 0.80
 600, 0.85
 620, 0.87
 640, 0.88
 660, 0.88
 680, 0.88
 700, 0.88
 720, 0.88
 740, 0.88
}

// #305 ROSE GOLD
// Color Effects Lighting Filter
#declare TS_R305 = spline{ linear_spline
 360, 0.59
 380, 0.67
 400, 0.68
 420, 0.66
 440, 0.59
 460, 0.52
 480, 0.47
 500, 0.46
 520, 0.49
 540, 0.56
 560, 0.64
 580, 0.76
 600, 0.83
 620, 0.84
 640, 0.84
 660, 0.85
 680, 0.85
 700, 0.87
 720, 0.87
 740, 0.88
}

// #06 NO COLOR STRAW
// Color Effects Lighting Filter
#declare TS_R06 = spline{ linear_spline
 360, 0.67
 380, 0.66
 400, 0.64
 420, 0.65
 440, 0.72
 460, 0.77
 480, 0.81
 500, 0.84
 520, 0.85
 540, 0.87
 560, 0.88
 580, 0.88
 600, 0.88
 620, 0.88
 640, 0.88
 660, 0.88
 680, 0.88
 700, 0.88
 720, 0.88
 740, 0.88
}

// #07 PALE YELLOW
// Color Effects Lighting Filter
#declare TS_R07 = spline{ linear_spline
 360, 0.52
 380, 0.43
 400, 0.37
 420, 0.40
 440, 0.52
 460, 0.65
 480, 0.74
 500, 0.80
 520, 0.83
 540, 0.86
 560, 0.87
 580, 0.88
 600, 0.88
 620, 0.88
 640, 0.88
 660, 0.88
 680, 0.88
 700, 0.88
 720, 0.88
 740, 0.88
}

// #08 PALE GOLD
// Color Effects Lighting Filter
#declare TS_R08 = spline{ linear_spline
 360, 0.69
 380, 0.70
 400, 0.63
 420, 0.49
 440, 0.42
 460, 0.50
 480, 0.71
 500, 0.72
 520, 0.70
 540, 0.73
 560, 0.72
 580, 0.80
 600, 0.84
 620, 0.85
 640, 0.85
 660, 0.86
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.86
}

// #09 PALE AMBER GOLD
// Color Effects Lighting Filter
#declare TS_R09 = spline{ linear_spline
 360, 0.58
 380, 0.61
 400, 0.55
 420, 0.47
 440, 0.39
 460, 0.35
 480, 0.38
 500, 0.48
 520, 0.57
 540, 0.65
 560, 0.72
 580, 0.81
 600, 0.87
 620, 0.88
 640, 0.88
 660, 0.88
 680, 0.88
 700, 0.88
 720, 0.88
 740, 0.88
}

// #10 MED YELLOW
// Color Effects Lighting Filter
#declare TS_R10 = spline{ linear_spline
 360, 0.34
 380, 0.06
 400, 0.00
 420, 0.00
 440, 0.00
 460, 0.00
 480, 0.17
 500, 0.74
 520, 0.86
 540, 0.87
 560, 0.87
 580, 0.87
 600, 0.88
 620, 0.88
 640, 0.88
 660, 0.88
 680, 0.88
 700, 0.88
 720, 0.88
 740, 0.88
}

// #310 DAFFODIL
// Color Effects Lighting Filter
#declare TS_R310 = spline{ linear_spline
 360, 0.64
 380, 0.56
 400, 0.32
 420, 0.12
 440, 0.06
 460, 0.17
 480, 0.60
 500, 0.73
 520, 0.79
 540, 0.83
 560, 0.85
 580, 0.85
 600, 0.85
 620, 0.86
 640, 0.86
 660, 0.86
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.86
}

// #11 LT STRAW
// Color Effects Lighting Filter
#declare TS_R11 = spline{ linear_spline
 360, 0.55
 380, 0.49
 400, 0.33
 420, 0.22
 440, 0.15
 460, 0.14
 480, 0.21
 500, 0.43
 520, 0.64
 540, 0.75
 560, 0.80
 580, 0.85
 600, 0.87
 620, 0.88
 640, 0.88
 660, 0.88
 680, 0.88
 700, 0.88
 720, 0.88
 740, 0.88
}

// #12 STRAW
// Color Effects Lighting Filter
#declare TS_R12 = spline{ linear_spline
 360, 0.33
 380, 0.16
 400, 0.01
 420, 0.00
 440, 0.00
 460, 0.01
 480, 0.28
 500, 0.62
 520, 0.77
 540, 0.83
 560, 0.84
 580, 0.85
 600, 0.86
 620, 0.86
 640, 0.86
 660, 0.86
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.86
}

// #312 CANARY
// Color Effects Lighting Filter
#declare TS_R312 = spline{ linear_spline
 360, 0.03
 380, 0.01
 400, 0.00
 420, 0.00
 440, 0.00
 460, 0.01
 480, 0.06
 500, 0.30
 520, 0.64
 540, 0.79
 560, 0.85
 580, 0.87
 600, 0.87
 620, 0.88
 640, 0.88
 660, 0.88
 680, 0.87
 700, 0.87
 720, 0.87
 740, 0.87
}

// #13 STRAW TINT
// Color Effects Lighting Filter
#declare TS_R13 = spline{ linear_spline
 360, 0.59
 380, 0.59
 400, 0.49
 420, 0.40
 440, 0.32
 460, 0.29
 480, 0.34
 500, 0.49
 520, 0.61
 540, 0.70
 560, 0.76
 580, 0.84
 600, 0.88
 620, 0.89
 640, 0.89
 660, 0.89
 680, 0.89
 700, 0.89
 720, 0.89
 740, 0.89
}

// #313 LIGHT RELIEF YELLOW
// Color Effects Lighting Filter
#declare TS_R313 = spline{ linear_spline
 360, 0.14
 380, 0.05
 400, 0.02
 420, 0.02
 440, 0.05
 460, 0.12
 480, 0.22
 500, 0.37
 520, 0.55
 540, 0.71
 560, 0.82
 580, 0.87
 600, 0.88
 620, 0.88
 640, 0.88
 660, 0.88
 680, 0.89
 700, 0.88
 720, 0.88
 740, 0.89
}

// #14 MED STRAW
// Color Effects Lighting Filter
#declare TS_R14 = spline{ linear_spline
 360, 0.40
 380, 0.36
 400, 0.22
 420, 0.12
 440, 0.07
 460, 0.06
 480, 0.11
 500, 0.27
 520, 0.48
 540, 0.62
 560, 0.70
 580, 0.80
 600, 0.85
 620, 0.86
 640, 0.87
 660, 0.87
 680, 0.87
 700, 0.87
 720, 0.87
 740, 0.87
}

// #15 DEEP STRAW
// Color Effects Lighting Filter
#declare TS_R15 = spline{ linear_spline
 360, 0.10
 380, 0.03
 400, 0.01
 420, 0.01
 440, 0.01
 460, 0.02
 480, 0.04
 500, 0.10
 520, 0.27
 540, 0.51
 560, 0.75
 580, 0.86
 600, 0.89
 620, 0.89
 640, 0.89
 660, 0.89
 680, 0.89
 700, 0.89
 720, 0.89
 740, 0.89
}

// #16 LT AMBER
// Color Effects Lighting Filter
#declare TS_R16 = spline{ linear_spline
 360, 0.60
 380, 0.67
 400, 0.60
 420, 0.46
 440, 0.30
 460, 0.19
 480, 0.15
 500, 0.19
 520, 0.29
 540, 0.58
 560, 0.74
 580, 0.81
 600, 0.85
 620, 0.85
 640, 0.85
 660, 0.86
 680, 0.85
 700, 0.86
 720, 0.86
 740, 0.86
}

// #316 GALLO GOLD
// Color Effects Lighting Filter
#declare TS_R316 = spline{ linear_spline
 360, 0.54
 380, 0.54
 400, 0.38
 420, 0.17
 440, 0.11
 460, 0.17
 480, 0.45
 500, 0.45
 520, 0.39
 540, 0.43
 560, 0.44
 580, 0.64
 600, 0.80
 620, 0.83
 640, 0.83
 660, 0.84
 680, 0.84
 700, 0.84
 720, 0.84
 740, 0.85
}

// #17 LT FLAME
// Color Effects Lighting Filter
#declare TS_R17 = spline{ linear_spline
 360, 0.57
 380, 0.64
 400, 0.58
 420, 0.45
 440, 0.28
 460, 0.18
 480, 0.14
 500, 0.15
 520, 0.20
 540, 0.42
 560, 0.52
 580, 0.69
 600, 0.82
 620, 0.84
 640, 0.84
 660, 0.84
 680, 0.84
 700, 0.85
 720, 0.85
 740, 0.85
}

// #317 APRICOT
// Color Effects Lighting Filter
#declare TS_R317 = spline{ linear_spline
 360, 0.22
 380, 0.21
 400, 0.18
 420, 0.18
 440, 0.21
 460, 0.21
 480, 0.20
 500, 0.20
 520, 0.24
 540, 0.33
 560, 0.45
 580, 0.67
 600, 0.84
 620, 0.88
 640, 0.89
 660, 0.89
 680, 0.89
 700, 0.89
 720, 0.89
 740, 0.89
}

// #18 FLAME
// Color Effects Lighting Filter
#declare TS_R18 = spline{ linear_spline
 360, 0.38
 380, 0.49
 400, 0.48
 420, 0.36
 440, 0.22
 460, 0.15
 480, 0.12
 500, 0.14
 520, 0.20
 540, 0.41
 560, 0.53
 580, 0.66
 600, 0.76
 620, 0.78
 640, 0.78
 660, 0.78
 680, 0.78
 700, 0.78
 720, 0.79
 740, 0.79
}

// #318 MAYAN SUN
// Color Effects Lighting Filter
#declare TS_R318 = spline{ linear_spline
 360, 0.52
 380, 0.61
 400, 0.56
 420, 0.42
 440, 0.24
 460, 0.13
 480, 0.09
 500, 0.09
 520, 0.11
 540, 0.26
 560, 0.34
 580, 0.58
 600, 0.80
 620, 0.85
 640, 0.86
 660, 0.86
 680, 0.86
 700, 0.86
 720, 0.87
 740, 0.87
}

// #19 FIRE
// Color Effects Lighting Filter
#declare TS_R19 = spline{ linear_spline
 360, 0.01
 380, 0.02
 400, 0.01
 420, 0.01
 440, 0.00
 460, 0.00
 480, 0.00
 500, 0.00
 520, 0.00
 540, 0.00
 560, 0.01
 580, 0.15
 600, 0.54
 620, 0.78
 640, 0.84
 660, 0.86
 680, 0.87
 700, 0.87
 720, 0.87
 740, 0.87
}

// #20 MED AMBER
// Color Effects Lighting Filter
#declare TS_R20 = spline{ linear_spline
 360, 0.28
 380, 0.27
 400, 0.16
 420, 0.08
 440, 0.04
 460, 0.03
 480, 0.04
 500, 0.11
 520, 0.22
 540, 0.35
 560, 0.48
 580, 0.69
 600, 0.84
 620, 0.87
 640, 0.88
 660, 0.88
 680, 0.88
 700, 0.88
 720, 0.88
 740, 0.88
}

// #21 GOLDEN AMBER
// Color Effects Lighting Filter
#declare TS_R21 = spline{ linear_spline
 360, 0.16
 380, 0.18
 400, 0.11
 420, 0.06
 440, 0.02
 460, 0.01
 480, 0.01
 500, 0.03
 520, 0.07
 540, 0.14
 560, 0.26
 580, 0.54
 600, 0.80
 620, 0.87
 640, 0.88
 660, 0.88
 680, 0.88
 700, 0.88
 720, 0.88
 740, 0.88
}

// #321 SFT GOLD AMBER
// Color Effects Lighting Filter
#declare TS_R321 = spline{ linear_spline
 360, 0.47
 380, 0.44
 400, 0.22
 420, 0.06
 440, 0.02
 460, 0.07
 480, 0.27
 500, 0.27
 520, 0.26
 540, 0.25
 560, 0.25
 580, 0.49
 600, 0.72
 620, 0.73
 640, 0.70
 660, 0.68
 680, 0.66
 700, 0.68
 720, 0.76
 740, 0.83
}

// #22 DEEP AMBER
// Color Effects Lighting Filter
#declare TS_R22 = spline{ linear_spline
 360, 0.01
 380, 0.06
 400, 0.06
 420, 0.01
 440, 0.00
 460, 0.00
 480, 0.00
 500, 0.00
 520, 0.00
 540, 0.02
 560, 0.09
 580, 0.32
 600, 0.65
 620, 0.81
 640, 0.85
 660, 0.86
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.87
}

// #23 ORANGE
// Color Effects Lighting Filter
#declare TS_R23 = spline{ linear_spline
 360, 0.12
 380, 0.17
 400, 0.14
 420, 0.09
 440, 0.04
 460, 0.02
 480, 0.01
 500, 0.02
 520, 0.04
 540, 0.08
 560, 0.18
 580, 0.45
 600, 0.75
 620, 0.84
 640, 0.87
 660, 0.87
 680, 0.88
 700, 0.88
 720, 0.88
 740, 0.88
}

// #24 SCARLET
// Color Effects Lighting Filter
#declare TS_R24 = spline{ linear_spline
 360, 0.05
 380, 0.15
 400, 0.24
 420, 0.20
 440, 0.09
 460, 0.03
 480, 0.01
 500, 0.01
 520, 0.01
 540, 0.01
 560, 0.04
 580, 0.19
 600, 0.52
 620, 0.75
 640, 0.83
 660, 0.86
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.87
}

// #324 GYPSY RED
// Color Effects Lighting Filter
#declare TS_R324 = spline{ linear_spline
 360, 0.24
 380, 0.33
 400, 0.32
 420, 0.25
 440, 0.12
 460, 0.05
 480, 0.02
 500, 0.01
 520, 0.01
 540, 0.01
 560, 0.03
 580, 0.17
 600, 0.60
 620, 0.84
 640, 0.88
 660, 0.89
 680, 0.89
 700, 0.89
 720, 0.89
 740, 0.89
}

// #25 ORANGE RED
// Color Effects Lighting Filter
#declare TS_R25 = spline{ linear_spline
 360, 0.02
 380, 0.05
 400, 0.09
 420, 0.06
 440, 0.02
 460, 0.00
 480, 0.00
 500, 0.00
 520, 0.00
 540, 0.00
 560, 0.01
 580, 0.08
 600, 0.40
 620, 0.71
 640, 0.83
 660, 0.87
 680, 0.87
 700, 0.88
 720, 0.88
 740, 0.88
}

// #325 HENNA SKY
// Color Effects Lighting Filter
#declare TS_R325 = spline{ linear_spline
 360, 0.05
 380, 0.05
 400, 0.05
 420, 0.00
 440, 0.00
 460, 0.01
 480, 0.07
 500, 0.07
 520, 0.04
 540, 0.03
 560, 0.04
 580, 0.07
 600, 0.27
 620, 0.44
 640, 0.46
 660, 0.55
 680, 0.72
 700, 0.81
 720, 0.81
 740, 0.81
}

// #26 LT RED
// Color Effects Lighting Filter
#declare TS_R26 = spline{ linear_spline
 360, 0.13
 380, 0.12
 400, 0.03
 420, 0.00
 440, 0.00
 460, 0.00
 480, 0.00
 500, 0.00
 520, 0.00
 540, 0.00
 560, 0.00
 580, 0.00
 600, 0.08
 620, 0.50
 640, 0.73
 660, 0.80
 680, 0.83
 700, 0.84
 720, 0.84
 740, 0.85
}

// #27 MED RED
// Color Effects Lighting Filter
#declare TS_R27 = spline{ linear_spline
 360, 0.01
 380, 0.01
 400, 0.00
 420, 0.00
 440, 0.00
 460, 0.00
 480, 0.00
 500, 0.00
 520, 0.00
 540, 0.00
 560, 0.00
 580, 0.00
 600, 0.02
 620, 0.11
 640, 0.33
 660, 0.59
 680, 0.76
 700, 0.82
 720, 0.85
 740, 0.86
}

// #30 LT SALMON PINK
// Color Effects Lighting Filter
#declare TS_R30 = spline{ linear_spline
 360, 0.33
 380, 0.46
 400, 0.50
 420, 0.46
 440, 0.35
 460, 0.24
 480, 0.19
 500, 0.17
 520, 0.19
 540, 0.26
 560, 0.36
 580, 0.58
 600, 0.77
 620, 0.83
 640, 0.85
 660, 0.85
 680, 0.85
 700, 0.85
 720, 0.85
 740, 0.86
}

// #31 SALMON PINK
// Color Effects Lighting Filter
#declare TS_R31 = spline{ linear_spline
 360, 0.34
 380, 0.51
 400, 0.60
 420, 0.57
 440, 0.46
 460, 0.34
 480, 0.26
 500, 0.22
 520, 0.21
 540, 0.25
 560, 0.33
 580, 0.53
 600, 0.74
 620, 0.84
 640, 0.86
 660, 0.87
 680, 0.87
 700, 0.87
 720, 0.87
 740, 0.88
}

// #331 SHELL PINK
// Color Effects Lighting Filter
#declare TS_R331 = spline{ linear_spline
 360, 0.57
 380, 0.68
 400, 0.70
 420, 0.67
 440, 0.57
 460, 0.47
 480, 0.40
 500, 0.37
 520, 0.37
 540, 0.44
 560, 0.51
 580, 0.69
 600, 0.85
 620, 0.89
 640, 0.90
 660, 0.90
 680, 0.90
 700, 0.90
 720, 0.91
 740, 0.91
}

// #32 MED SALMON PINK
// Color Effects Lighting Filter
#declare TS_R32 = spline{ linear_spline
 360, 0.17
 380, 0.33
 400, 0.43
 420, 0.39
 440, 0.26
 460, 0.15
 480, 0.09
 500, 0.07
 520, 0.06
 540, 0.09
 560, 0.15
 580, 0.37
 600, 0.65
 620, 0.80
 640, 0.85
 660, 0.86
 680, 0.87
 700, 0.87
 720, 0.87
 740, 0.87
}

// #33 NO COLOR PINK
// Color Effects Lighting Filter
#declare TS_R33 = spline{ linear_spline
 360, 0.56
 380, 0.68
 400, 0.75
 420, 0.75
 440, 0.71
 460, 0.64
 480, 0.57
 500, 0.52
 520, 0.50
 540, 0.51
 560, 0.57
 580, 0.69
 600, 0.81
 620, 0.85
 640, 0.87
 660, 0.87
 680, 0.87
 700, 0.87
 720, 0.87
 740, 0.87
}

// #333 BLUSH PINK
// Color Effects Lighting Filter
#declare TS_R333 = spline{ linear_spline
 360, 0.61
 380, 0.69
 400, 0.72
 420, 0.74
 440, 0.74
 460, 0.71
 480, 0.67
 500, 0.62
 520, 0.57
 540, 0.56
 560, 0.58
 580, 0.63
 600, 0.76
 620, 0.79
 640, 0.80
 660, 0.82
 680, 0.84
 700, 0.84
 720, 0.85
 740, 0.85
}

// #34 FLESH PINK
// Color Effects Lighting Filter
#declare TS_R34 = spline{ linear_spline
 360, 0.40
 380, 0.52
 400, 0.56
 420, 0.52
 440, 0.44
 460, 0.40
 480, 0.38
 500, 0.31
 520, 0.24
 540, 0.28
 560, 0.28
 580, 0.51
 600, 0.73
 620, 0.77
 640, 0.78
 660, 0.79
 680, 0.79
 700, 0.79
 720, 0.80
 740, 0.80
}

// #35 LT PINK
// Color Effects Lighting Filter
#declare TS_R35 = spline{ linear_spline
 360, 0.55
 380, 0.49
 400, 0.77
 420, 0.77
 440, 0.73
 460, 0.66
 480, 0.59
 500, 0.53
 520, 0.49
 540, 0.50
 560, 0.56
 580, 0.69
 600, 0.80
 620, 0.86
 640, 0.88
 660, 0.88
 680, 0.88
 700, 0.88
 720, 0.88
 740, 0.88
}

// #36 MED PINK
// Color Effects Lighting Filter
#declare TS_R36 = spline{ linear_spline
 360, 0.38
 380, 0.56
 400, 0.69
 420, 0.67
 440, 0.59
 460, 0.48
 480, 0.38
 500, 0.30
 520, 0.26
 540, 0.27
 560, 0.34
 580, 0.52
 600, 0.71
 620, 0.82
 640, 0.85
 660, 0.86
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.87
}

// #336 BILLINGTON PINK
// Color Effects Lighting Filter
#declare TS_R336 = spline{ linear_spline
 360, 0.46
 380, 0.59
 400, 0.67
 420, 0.69
 440, 0.64
 460, 0.56
 480, 0.46
 500, 0.36
 520, 0.29
 540, 0.27
 560, 0.30
 580, 0.40
 600, 0.62
 620, 0.77
 640, 0.79
 660, 0.80
 680, 0.80
 700, 0.80
 720, 0.81
 740, 0.81
}

// #37 PALE ROSE PINK
// Color Effects Lighting Filter
#declare TS_R37 = spline{ linear_spline
 360, 0.62
 380, 0.70
 400, 0.73
 420, 0.74
 440, 0.72
 460, 0.67
 480, 0.60
 500, 0.52
 520, 0.45
 540, 0.43
 560, 0.46
 580, 0.53
 600, 0.70
 620, 0.78
 640, 0.79
 660, 0.81
 680, 0.84
 700, 0.85
 720, 0.85
 740, 0.85
}

// #337 TRUE PINK
// Color Effects Lighting Filter
#declare TS_R337 = spline{ linear_spline
 360, 0.76
 380, 0.80
 400, 0.80
 420, 0.79
 440, 0.74
 460, 0.67
 480, 0.58
 500, 0.49
 520, 0.42
 540, 0.41
 560, 0.44
 580, 0.56
 600, 0.76
 620, 0.85
 640, 0.87
 660, 0.88
 680, 0.88
 700, 0.88
 720, 0.88
 740, 0.88
}

// #38 LT ROSE
// Color Effects Lighting Filter
#declare TS_R38 = spline{ linear_spline
 360, 0.45
 380, 0.60
 400, 0.67
 420, 0.66
 440, 0.60
 460, 0.53
 480, 0.47
 500, 0.43
 520, 0.40
 540, 0.40
 560, 0.42
 580, 0.48
 600, 0.57
 620, 0.68
 640, 0.76
 660, 0.82
 680, 0.84
 700, 0.85
 720, 0.85
 740, 0.86
}

// #39 EXOTIC SANGRIA
// Color Effects Lighting Filter
#declare TS_R39 = spline{ linear_spline
 360, 0.22
 380, 0.45
 400, 0.52
 420, 0.47
 440, 0.29
 460, 0.11
 480, 0.03
 500, 0.00
 520, 0.00
 540, 0.00
 560, 0.00
 580, 0.01
 600, 0.15
 620, 0.43
 640, 0.50
 660, 0.61
 680, 0.75
 700, 0.80
 720, 0.82
 740, 0.83
}

// #339 BROADWAY PINK
// Color Effects Lighting Filter
#declare TS_R339 = spline{ linear_spline
 360, 0.26
 380, 0.40
 400, 0.47
 420, 0.44
 440, 0.32
 460, 0.19
 480, 0.09
 500, 0.04
 520, 0.02
 540, 0.01
 560, 0.03
 580, 0.04
 600, 0.21
 620, 0.61
 640, 0.75
 660, 0.81
 680, 0.84
 700, 0.85
 720, 0.86
 740, 0.86
}

// #40 LT SALMON
// Color Effects Lighting Filter
#declare TS_R40 = spline{ linear_spline
 360, 0.19
 380, 0.31
 400, 0.36
 420, 0.31
 440, 0.19
 460, 0.10
 480, 0.06
 500, 0.06
 520, 0.07
 540, 0.12
 560, 0.21
 580, 0.47
 600, 0.73
 620, 0.83
 640, 0.86
 660, 0.87
 680, 0.87
 700, 0.87
 720, 0.87
 740, 0.87
}

// #41 SALMON
// Color Effects Lighting Filter
#declare TS_R41 = spline{ linear_spline
 360, 0.31
 380, 0.36
 400, 0.28
 420, 0.12
 440, 0.05
 460, 0.05
 480, 0.07
 500, 0.04
 520, 0.02
 540, 0.03
 560, 0.03
 580, 0.24
 600, 0.68
 620, 0.81
 640, 0.82
 660, 0.83
 680, 0.83
 700, 0.84
 720, 0.84
 740, 0.85
}

// #42 DEEP SALMON
// Color Effects Lighting Filter
#declare TS_R42 = spline{ linear_spline
 360, 0.28
 380, 0.41
 400, 0.30
 420, 0.13
 440, 0.05
 460, 0.04
 480, 0.05
 500, 0.01
 520, 0.01
 540, 0.00
 560, 0.01
 580, 0.03
 600, 0.29
 620, 0.62
 640, 0.69
 660, 0.70
 680, 0.72
 700, 0.75
 720, 0.80
 740, 0.84
}

// #342 ROSE PINK
// Color Effects Lighting Filter
#declare TS_R342 = spline{ linear_spline
 360, 0.04
 380, 0.17
 400, 0.35
 420, 0.30
 440, 0.15
 460, 0.05
 480, 0.01
 500, 0.00
 520, 0.00
 540, 0.00
 560, 0.01
 580, 0.03
 600, 0.23
 620, 0.62
 640, 0.80
 660, 0.85
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.86
}

// #43 DEEP PINK
// Color Effects Lighting Filter
#declare TS_R43 = spline{ linear_spline
 360, 0.17
 380, 0.41
 400, 0.64
 420, 0.63
 440, 0.51
 460, 0.35
 480, 0.22
 500, 0.13
 520, 0.08
 540, 0.07
 560, 0.11
 580, 0.28
 600, 0.56
 620, 0.77
 640, 0.84
 660, 0.86
 680, 0.86
 700, 0.86
 720, 0.86
 740, 0.87
}

// #343 NEON PINK
// Color Effects Lighting Filter
#declare TS_R343 = spline{ linear_spline
 360, 0.01
 380, 0.09
 400, 0.53
 420, 0.60
 440, 0.49
 460, 0.34
 480, 0.19
 500, 0.08
 520, 0.03
 540, 0.03
 560, 0.07
 580, 0.29
 600, 0.65
 620, 0.82
 640, 0.86
 660, 0.87
 680, 0.88
 700, 0.88
 720, 0.88
 740, 0.88
}

// #44 MIDDLE ROSE
// Color Effects Lighting Filter
#declare TS_R44 = spline{ linear_spline
 360, 0.53
 380, 0.68
 400, 0.71
 420, 0.70
 440, 0.62
 460, 0.46
 480, 0.31
 500, 0.18
 520, 0.11
 540, 0.10
 560, 0.13
 580, 0.21
 600, 0.56
 620, 0.79
 640, 0.82
 660, 0.84
 680, 0.84
 700, 0.85
 720, 0.85
 740, 0.85
}

// #344 FOLLIES PINK
// Color Effects Lighting Filter
#declare TS_R344 = spline{ linear_spline
 360, 0.49
 380, 0.63
 400, 0.70
 420, 0.69
 440, 0.62
 460, 0.50
 480, 0.36
 500, 0.22
 520, 0.14
 540, 0.08
 560, 0.10
 580, 0.10
 600, 0.32
 620, 0.72
 640, 0.83
 660, 0.86
 680, 0.87
 700, 0.88
 720, 0.88
 740, 0.88
}

// #45 ROSE
// Color Effects Lighting Filter
#declare TS_R45 = spline{ linear_spline
 360, 0.13
 380, 0.32
 400, 0.38
 420, 0.25
 440, 0.14
 460, 0.07
 480, 0.04
 500, 0.03
 520, 0.02
 540, 0.02
 560, 0.02
 580, 0.05
 600, 0.18
 620, 0.37
 640, 0.59
 660, 0.75
 680, 0.83
 700, 0.86
 720, 0.87
 740, 0.88
}

// #46 MAGENTA
// Color Effects Lighting Filter
#declare TS_R46 = spline{ linear_spline
 360, 0.03
 380, 0.11
 400, 0.20
 420, 0.15
 440, 0.06
 460, 0.02
 480, 0.01
 500, 0.00
 520, 0.00
 540, 0.00
 560, 0.01
 580, 0.02
 600, 0.07
 620, 0.20
 640, 0.43
 660, 0.65
 680, 0.78
 700, 0.83
 720, 0.84
 740, 0.85
}

// #346 TROPICAL MAGENTA
// Color Effects Lighting Filter
#declare TS_R346 = spline{ linear_spline
 360, 0.14
 380, 0.29
 400, 0.46
 420, 0.55
 440, 0.53
 460, 0.38
 480, 0.18
 500, 0.05
 520, 0.02
 540, 0.01
 560, 0.00
 580, 0.03
 600, 0.25
 620, 0.55
 640, 0.73
 660, 0.79
 680, 0.80
 700, 0.80
 720, 0.81
 740, 0.81
}

// #47 LT ROSE PURPLE
// Color Effects Lighting Filter
#declare TS_R47 = spline{ linear_spline
 360, 0.29
 380, 0.45
 400, 0.46
 420, 0.35
 440, 0.25
 460, 0.21
 480, 0.19
 500, 0.10
 520, 0.05
 540, 0.04
 560, 0.05
 580, 0.06
 600, 0.16
 620, 0.17
 640, 0.11
 660, 0.09
 680, 0.07
 700, 0.09
 720, 0.26
 740, 0.55
}

// #347 BELLADONNA ROSE
// Color Effects Lighting Filter
#declare TS_R347 = spline{ linear_spline
 360, 0.03
 380, 0.19
 400, 0.46
 420, 0.39
 440, 0.24
 460, 0.11
 480, 0.04
 500, 0.02
 520, 0.00
 540, 0.00
 560, 0.00
 580, 0.01
 600, 0.00
 620, 0.03
 640, 0.14
 660, 0.35
 680, 0.50
 700, 0.58
 720, 0.69
 740, 0.80
}

// #48 ROSE PURPLE
// Color Effects Lighting Filter
#declare TS_R48 = spline{ linear_spline
 360, 0.20
 380, 0.45
 400, 0.65
 420, 0.62
 440, 0.51
 460, 0.38
 480, 0.27
 500, 0.19
 520, 0.12
 540, 0.09
 560, 0.07
 580, 0.09
 600, 0.17
 620, 0.33
 640, 0.55
 660, 0.73
 680, 0.82
 700, 0.85
 720, 0.87
 740, 0.87
}

// #348 PURPLE JAZZ
// Color Effects Lighting Filter
#declare TS_R348 = spline{ linear_spline
 360, 0.15
 380, 0.42
 400, 0.69
 420, 0.56
 440, 0.53
 460, 0.37
 480, 0.24
 500, 0.14
 520, 0.08
 540, 0.04
 560, 0.03
 580, 0.04
 600, 0.09
 620, 0.23
 640, 0.46
 660, 0.69
 680, 0.83
 700, 0.88
 720, 0.90
 740, 0.90
}

// #49 EXOTIC SANGRIA
// Color Effects Lighting Filter
#declare TS_R49 = spline{ linear_spline
 360, 0.05
 380, 0.25
 400, 0.49
 420, 0.44
 440, 0.29
 460, 0.15
 480, 0.07
 500, 0.03
 520, 0.01
 540, 0.01
 560, 0.00
 580, 0.01
 600, 0.03
 620, 0.11
 640, 0.32
 660, 0.59
 680, 0.76
 700, 0.83
 720, 0.85
 740, 0.86
}

// #349 FISHER FUCHSIA
// Color Effects Lighting Filter
#declare TS_R349 = spline{ linear_spline
 360, 0.01
 380, 0.14
 400, 0.62
 420, 0.65
 440, 0.49
 460, 0.28
 480, 0.12
 500, 0.04
 520, 0.01
 540, 0.01
 560, 0.01
 580, 0.03
 600, 0.11
 620, 0.30
 640, 0.54
 660, 0.73
 680, 0.83
 700, 0.88
 720, 0.89
 740, 0.90
}

// #50 MAUVE
// Color Effects Lighting Filter
#declare TS_R50 = spline{ linear_spline
 360, 0.15
 380, 0.32
 400, 0.34
 420, 0.18
 440, 0.09
 460, 0.06
 480, 0.05
 500, 0.05
 520, 0.06
 540, 0.06
 560, 0.06
 580, 0.09
 600, 0.16
 620, 0.31
 640, 0.53
 660, 0.70
 680, 0.79
 700, 0.83
 720, 0.84
 740, 0.85
}

// #51 SURPRISE PINK
// Color Effects Lighting Filter
#declare TS_R51 = spline{ linear_spline
 360, 0.68
 380, 0.75
 400, 0.78
 420, 0.78
 440, 0.77
 460, 0.72
 480, 0.67
 500, 0.59
 520, 0.53
 540, 0.48
 560, 0.48
 580, 0.48
 600, 0.56
 620, 0.59
 640, 0.58
 660, 0.67
 680, 0.80
 700, 0.84
 720, 0.85
 740, 0.85
}

// #351 LAVENDER MIST
// Color Effects Lighting Filter
#declare TS_R351 = spline{ linear_spline
 360, 0.68
 380, 0.74
 400, 0.77
 420, 0.78
 440, 0.79
 460, 0.76
 480, 0.71
 500, 0.67
 520, 0.61
 540, 0.57
 560, 0.56
 580, 0.57
 600, 0.63
 620, 0.64
 640, 0.64
 660, 0.76
 680, 0.85
 700, 0.87
 720, 0.87
 740, 0.88
}

// #52 LT LAVENDER
// Color Effects Lighting Filter
#declare TS_R52 = spline{ linear_spline
 360, 0.43
 380, 0.61
 400, 0.72
 420, 0.73
 440, 0.70
 460, 0.64
 480, 0.55
 500, 0.47
 520, 0.37
 540, 0.30
 560, 0.26
 580, 0.25
 600, 0.31
 620, 0.40
 640, 0.49
 660, 0.69
 680, 0.82
 700, 0.86
 720, 0.87
 740, 0.88
}

// #53 PALE LAVENDER
// Color Effects Lighting Filter
#declare TS_R53 = spline{ linear_spline
 360, 0.65
 380, 0.73
 400, 0.77
 420, 0.78
 440, 0.79
 460, 0.78
 480, 0.75
 500, 0.71
 520, 0.66
 540, 0.61
 560, 0.57
 580, 0.54
 600, 0.56
 620, 0.59
 640, 0.62
 660, 0.75
 680, 0.83
 700, 0.86
 720, 0.87
 740, 0.87
}

// #353 LILLY LAVENDER
// Color Effects Lighting Filter
#declare TS_R353 = spline{ linear_spline
 360, 0.42
 380, 0.58
 400, 0.69
 420, 0.73
 440, 0.72
 460, 0.67
 480, 0.58
 500, 0.48
 520, 0.37
 540, 0.27
 560, 0.22
 580, 0.18
 600, 0.22
 620, 0.27
 640, 0.32
 660, 0.58
 680, 0.79
 700, 0.86
 720, 0.88
 740, 0.89
}

// #54 SPECIAL LAVENDER
// Color Effects Lighting Filter
#declare TS_R54 = spline{ linear_spline
 360, 0.54
 380, 0.65
 400, 0.73
 420, 0.76
 440, 0.77
 460, 0.75
 480, 0.68
 500, 0.60
 520, 0.52
 540, 0.45
 560, 0.44
 580, 0.43
 600, 0.48
 620, 0.50
 640, 0.50
 660, 0.69
 680, 0.83
 700, 0.86
 720, 0.87
 740, 0.88
}

// #55 LILAC
// Color Effects Lighting Filter
#declare TS_R55 = spline{ linear_spline
 360, 0.46
 380, 0.57
 400, 0.65
 420, 0.69
 440, 0.70
 460, 0.68
 480, 0.61
 500, 0.53
 520, 0.44
 540, 0.34
 560, 0.29
 580, 0.23
 600, 0.26
 620, 0.30
 640, 0.33
 660, 0.57
 680, 0.78
 700, 0.84
 720, 0.85
 740, 0.86
}

// #355 PALE VIOLET
// Color Effects Lighting Filter
#declare TS_R355 = spline{ linear_spline
 360, 0.33
 380, 0.48
 400, 0.59
 420, 0.63
 440, 0.66
 460, 0.62
 480, 0.52
 500, 0.42
 520, 0.30
 540, 0.19
 560, 0.14
 580, 0.10
 600, 0.12
 620, 0.14
 640, 0.17
 660, 0.44
 680, 0.73
 700, 0.84
 720, 0.86
 740, 0.87
}

// #56 GYPSY LAVENDER 
// Color Effects Lighting Filter
#declare TS_R56 = spline{ linear_spline
 360, 0.23
 380, 0.31
 400, 0.38
 420, 0.48
 440, 0.54
 460, 0.46
 480, 0.29
 500, 0.14
 520, 0.06
 540, 0.02
 560, 0.01
 580, 0.01
 600, 0.02
 620, 0.03
 640, 0.03
 660, 0.23
 680, 0.62
 700, 0.80
 720, 0.84
 740, 0.86
}

// #356 MIDDLE LAVENDER
// Color Effects Lighting Filter
#declare TS_R356 = spline{ linear_spline
 360, 0.36
 380, 0.54
 400, 0.67
 420, 0.69
 440, 0.67
 460, 0.60
 480, 0.51
 500, 0.41
 520, 0.31
 540, 0.23
 560, 0.19
 580, 0.17
 600, 0.22
 620, 0.29
 640, 0.37
 660, 0.60
 680, 0.79
 700, 0.84
 720, 0.86
 740, 0.87
}

// #57 LAVENDER
// Color Effects Lighting Filter
#declare TS_R57 = spline{ linear_spline
 360, 0.40
 380, 0.60
 400, 0.75
 420, 0.78
 440, 0.75
 460, 0.62
 480, 0.52
 500, 0.40
 520, 0.32
 540, 0.20
 560, 0.17
 580, 0.15
 600, 0.17
 620, 0.20
 640, 0.45
 660, 0.60
 680, 0.76
 700, 0.81
 720, 0.80
 740, 0.70
}

// #357 ROYAL LAVENDER
// Color Effects Lighting Filter
#declare TS_R357 = spline{ linear_spline
 360, 0.28
 380, 0.28
 400, 0.44
 420, 0.53
 440, 0.58
 460, 0.48
 480, 0.31
 500, 0.15
 520, 0.06
 540, 0.02
 560, 0.02
 580, 0.01
 600, 0.03
 620, 0.05
 640, 0.06
 660, 0.28
 680, 0.66
 700, 0.81
 720, 0.86
 740, 0.87
}

// #58 - DEEP LAVENDER
// Color Effects Lighting Filter
#declare TS_R58 = spline{ linear_spline
 360, 0.17
 380, 0.34
 400, 0.52
 420, 0.60
 440, 0.62
 460, 0.51
 480, 0.34
 500, 0.19
 520, 0.10
 540, 0.006
 560, 0.05
 580, 0.05
 600, 0.09
 620, 0.10
 640, 0.11
 660, 0.37
 680, 0.71
 700, 0.84
 720, 0.87
 740, 0.87
}

// #358 ROSE INDIGO
// Color Effects Lighting Filter
#declare TS_R358 = spline{ linear_spline
 360, 0.21
 380, 0.31
 400, 0.40
 420, 0.47
 440, 0.48
 460, 0.35
 480, 0.18
 500, 0.06
 520, 0.02
 540, 0.01
 560, 0.01
 580, 0.00
 600, 0.02
 620, 0.06
 640, 0.07
 660, 0.31
 680, 0.65
 700, 0.77
 720, 0.80
 740, 0.82
}

// #59 INDIGO
// Color Effects Lighting Filter
#declare TS_R59 = spline{ linear_spline
 360, 0.16
 380, 0.30
 400, 0.41
 420, 0.34
 440, 0.23
 460, 0.15
 480, 0.09
 500, 0.03
 520, 0.01
 540, 0.00
 560, 0.00
 580, 0.00
 600, 0.00
 620, 0.00
 640, 0.00
 660, 0.03
 680, 0.33
 700, 0.64
 720, 0.76
 740, 0.80
}

// #359 MED VIOLET
// Color Effects Lighting Filter
#declare TS_R359 = spline{ linear_spline
 360, 0.24
 380, 0.31
 400, 0.39
 420, 0.50
 440, 0.59
 460, 0.52
 480, 0.35
 500, 0.18
 520, 0.08
 540, 0.03
 560, 0.02
 580, 0.01
 600, 0.01
 620, 0.02
 640, 0.03
 660, 0.19
 680, 0.49
 700, 0.71
 720, 0.82
 740, 0.86
}

// #60 NO COLOR BLUE
// Color Effects Lighting Filter
#declare TS_R60 = spline{ linear_spline
 360, 0.67
 380, 0.72
 400, 0.76
 420, 0.78
 440, 0.77
 460, 0.75
 480, 0.71
 500, 0.67
 520, 0.60
 540, 0.55
 560, 0.48
 580, 0.43
 600, 0.41
 620, 0.40
 640, 0.36
 660, 0.50
 680, 0.75
 700, 0.83
 720, 0.85
 740, 0.85
}

// #360 CLEARWATER
// Color Effects Lighting Filter
#declare TS_R360 = spline{ linear_spline
 360, 0.52
 380, 0.59
 400, 0.65
 420, 0.70
 440, 0.71
 460, 0.68
 480, 0.62
 500, 0.57
 520, 0.51
 540, 0.50
 560, 0.45
 580, 0.46
 600, 0.42
 620, 0.38
 640, 0.35
 660, 0.34
 680, 0.42
 700, 0.59
 720, 0.74
 740, 0.81
}

// #61 MIST BLUE
// Color Effects Lighting Filter
#declare TS_R61 = spline{ linear_spline
 360, 0.72
 380, 0.74
 400, 0.76
 420, 0.79
 440, 0.83
 460, 0.84
 480, 0.83
 500, 0.80
 520, 0.76
 540, 0.68
 560, 0.62
 580, 0.52
 600, 0.51
 620, 0.50
 640, 0.49
 660, 0.68
 680, 0.83
 700, 0.87
 720, 0.88
 740, 0.88
}

// #361 HEMSLEY BLUE
// Color Effects Lighting Filter
#declare TS_R361 = spline{ linear_spline
 360, 0.45
 380, 0.45
 400, 0.45
 420, 0.51
 440, 0.59
 460, 0.60
 480, 0.55
 500, 0.45
 520, 0.31
 540, 0.17
 560, 0.11
 580, 0.04
 600, 0.05
 620, 0.05
 640, 0.05
 660, 0.27
 680, 0.64
 700, 0.80
 720, 0.80
 740, 0.80
}

// #62 BOOSTER BLUE
// Color Effects Lighting Filter
#declare TS_R62 = spline{ linear_spline
 360, 0.48
 380, 0.56
 400, 0.63
 420, 0.71
 440, 0.73
 460, 0.69
 480, 0.63
 500, 0.57
 520, 0.50
 540, 0.48
 560, 0.40
 580, 0.38
 600, 0.32
 620, 0.26
 640, 0.24
 660, 0.22
 680, 0.28
 700, 0.48
 720, 0.68
 740, 0.80
}

// #362 TIPTON BLUE
// Color Effects Lighting Filter
#declare TS_R362 = spline{ linear_spline
 360, 0.61
 380, 0.69
 400, 0.74
 420, 0.76
 440, 0.75
 460, 0.71
 480, 0.66
 500, 0.59
 520, 0.50
 540, 0.41
 560, 0.35
 580, 0.28
 600, 0.27
 620, 0.26
 640, 0.25
 660, 0.37
 680, 0.62
 700, 0.78
 720, 0.82
 740, 0.84
}

// #63 PALE BLUE
// Color Effects Lighting Filter
#declare TS_R63 = spline{ linear_spline
 360, 0.63
 380, 0.66
 400, 0.70
 420, 0.75
 440, 0.79
 460, 0.81
 480, 0.80
 500, 0.75
 520, 0.69
 540, 0.59
 560, 0.51
 580, 0.40
 600, 0.40
 620, 0.38
 640, 0.37
 660, 0.61
 680, 0.81
 700, 0.86
 720, 0.88
 740, 0.88
}

// #363 AQUAMARINE
// Color Effects Lighting Filter
#declare TS_R363 = spline{ linear_spline
 360, 0.08
 380, 0.35
 400, 0.67
 420, 0.76
 440, 0.82
 460, 0.85
 480, 0.85
 500, 0.84
 520, 0.80
 540, 0.71
 560, 0.56
 580, 0.44
 600, 0.38
 620, 0.37
 640, 0.39
 660, 0.43
 680, 0.37
 700, 0.52
 720, 0.59
 740, 0.66
}

// #64 LT STEEL BLUE
// Color Effects Lighting Filter
#declare TS_R64 = spline{ linear_spline
 360, 0.42
 380, 0.47
 400, 0.52
 420, 0.59
 440, 0.68
 460, 0.69
 480, 0.64
 500, 0.53
 520, 0.40
 540, 0.26
 560, 0.18
 580, 0.10
 600, 0.10
 620, 0.09
 640, 0.09
 660, 0.34
 680, 0.71
 700, 0.84
 720, 0.88
 740, 0.88
}

// #364 BLUE BELL
// Color Effects Lighting Filter
#declare TS_R364 = spline{ linear_spline
 360, 0.58
 380, 0.68
 400, 0.73
 420, 0.74
 440, 0.72
 460, 0.67
 480, 0.60
 500, 0.52
 520, 0.41
 540, 0.31
 560, 0.25
 580, 0.18
 600, 0.17
 620, 0.17
 640, 0.16
 660, 0.27
 680, 0.56
 700, 0.76
 720, 0.83
 740, 0.85
}

// #65 DAYLT BLUE
// Color Effects Lighting Filter
#declare TS_R65 = spline{ linear_spline
 360, 0.48
 380, 0.50
 400, 0.55
 420, 0.63
 440, 0.75
 460, 0.78
 480, 0.74
 500, 0.63
 520, 0.50
 540, 0.32
 560, 0.22
 580, 0.11
 600, 0.11
 620, 0.10
 640, 0.09
 660, 0.34
 680, 0.70
 700, 0.84
 720, 0.87
 740, 0.88
}

// #365 THARON DELFT BLUE
// Color Effects Lighting Filter
#declare TS_R365 = spline{ linear_spline
 360, 0.40
 380, 0.49
 400, 0.56
 420, 0.66
 440, 0.86
 460, 0.59
 480, 0.49
 500, 0.41
 520, 0.33
 540, 0.32
 560, 0.26
 580, 0.27
 600, 0.23
 620, 0.18
 640, 0.16
 660, 0.14
 680, 0.19
 700, 0.38
 720, 0.62
 740, 0.76
}

// #66 COOL BLUE
// Color Effects Lighting Filter
#declare TS_R66 = spline{ linear_spline
 360, 0.44
 380, 0.59
 400, 0.70
 420, 0.76
 440, 0.81
 460, 0.83
 480, 0.83
 500, 0.81
 520, 0.77
 540, 0.71
 560, 0.63
 580, 0.53
 600, 0.46
 620, 0.42
 640, 0.41
 660, 0.41
 680, 0.29
 700, 0.64
 720, 0.72
 740, 0.78
}

// #366 JORDAN BLUE
// Color Effects Lighting Filter
#declare TS_R366 = spline{ linear_spline
 360, 0.34
 380, 0.46
 400, 0.58
 420, 0.69
 440, 0.79
 460, 0.83
 480, 0.79
 500, 0.71
 520, 0.59
 540, 0.42
 560, 0.29
 580, 0.16
 600, 0.13
 620, 0.11
 640, 0.11
 660, 0.23
 680, 0.28
 700, 0.55
 720, 0.63
 740, 0.75
}

// #67 LT SKY BLUE
// Color Effects Lighting Filter
#declare TS_R67 = spline{ linear_spline
 360, 0.42
 380, 0.44
 400, 0.49
 420, 0.60
 440, 0.73
 460, 0.77
 480, 0.72
 500, 0.59
 520, 0.43
 540, 0.26
 560, 0.16
 580, 0.07
 600, 0.06
 620, 0.06
 640, 0.05
 660, 0.28
 680, 0.67
 700, 0.84
 720, 0.88
 740, 0.88
}

// #367 SLATE BLUE
// Color Effects Lighting Filter
#declare TS_R367 = spline{ linear_spline
 360, 0.43
 380, 0.47
 400, 0.53
 420, 0.62
 440, 0.73
 460, 0.76
 480, 0.70
 500, 0.58
 520, 0.44
 540, 0.27
 560, 0.18
 580, 0.09
 600, 0.08
 620, 0.08
 640, 0.08
 660, 0.32
 680, 0.66
 700, 0.83
 720, 0.86
 740, 0.88
}

// #68 PARRY SKY BLUE
// Color Effects Lighting Filter
#declare TS_R68 = spline{ linear_spline
 360, 0.27
 380, 0.28
 400, 0.33
 420, 0.46
 440, 0.55
 460, 0.71
 480, 0.62
 500, 0.44
 520, 0.25
 540, 0.10
 560, 0.04
 580, 0.01
 600, 0.01
 620, 0.01
 640, 0.01
 660, 0.13
 680, 0.55
 700, 0.79
 720, 0.86
 740, 0.87
}

// #368 WINKLER BLUE
// Color Effects Lighting Filter
#declare TS_R368 = spline{ linear_spline
 360, 0.46
 380, 0.46
 400, 0.46
 420, 0.57
 440, 0.67
 460, 0.68
 480, 0.60
 500, 0.47
 520, 0.31
 540, 0.17
 560, 0.10
 580, 0.04
 600, 0.04
 620, 0.04
 640, 0.06
 660, 0.27
 680, 0.62
 700, 0.77
 720, 0.77
 740, 0.77
}

// #69 BRILLIANT BLUE
// Color Effects Lighting Filter
#declare TS_R69 = spline{ linear_spline
 360, 0.09
 380, 0.15
 400, 0.28
 420, 0.42
 440, 0.57
 460, 0.64
 480, 0.62
 500, 0.53
 520, 0.37
 540, 0.18
 560, 0.08
 580, 0.02
 600, 0.02
 620, 0.01
 640, 0.01
 660, 0.05
 680, 0.10
 700, 0.21
 720, 0.23
 740, 0.35
}

// #369 TAHITIAN BLUE
// Color Effects Lighting Filter
#declare TS_R369 = spline{ linear_spline
 360, 0.15
 380, 0.25
 400, 0.45
 420, 0.60
 440, 0.74
 460, 0.79
 480, 0.77
 500, 0.67
 520, 0.53
 540, 0.34
 560, 0.20
 580, 0.08
 600, 0.06
 620, 0.04
 640, 0.04
 660, 0.11
 680, 0.15
 700, 0.24
 720, 0.28
 740, 0.49
}

// #70 NILE BLUE
// Color Effects Lighting Filter
#declare TS_R70 = spline{ linear_spline
 360, 0.53
 380, 0.51
 400, 0.49
 420, 0.51
 440, 0.61
 460, 0.72
 480, 0.75
 500, 0.70
 520, 0.60
 540, 0.46
 560, 0.34
 580, 0.22
 600, 0.18
 620, 0.16
 640, 0.15
 660, 0.27
 680, 0.48
 700, 0.69
 720, 0.82
 740, 0.87
}

// #370 ITALIAN BLUE
// Color Effects Lighting Filter
#declare TS_R370 = spline{ linear_spline
 360, 0.03
 380, 0.07
 400, 0.21
 420, 0.31
 440, 0.43
 460, 0.55
 480, 0.63
 500, 0.67
 520, 0.61
 540, 0.43
 560, 0.21
 580, 0.09
 600, 0.04
 620, 0.02
 640, 0.01
 660, 0.02
 680, 0.01
 700, 0.01
 720, 0.02
 740, 0.11
}

// #71 SEA BLUE
// Color Effects Lighting Filter
#declare TS_R71 = spline{ linear_spline
 360, 0.39
 380, 0.36
 400, 0.34
 420, 0.37
 440, 0.49
 460, 0.63
 480, 0.67
 500, 0.80
 520, 0.47
 540, 0.31
 560, 0.19
 580, 0.10
 600, 0.07
 620, 0.06
 640, 0.05
 660, 0.13
 680, 0.33
 700, 0.49
 720, 0.77
 740, 0.84
}

// #371 THEATRE BOOSTER 1
// Color Effects Lighting Filter
#declare TS_R371 = spline{ linear_spline
 360, 0.49
 380, 0.56
 400, 0.63
 420, 0.69
 440, 0.75
 460, 0.75
 480, 0.69
 500, 0.59
 520, 0.68
 540, 0.37
 560, 0.31
 580, 0.23
 600, 0.24
 620, 0.23
 640, 0.23
 660, 0.49
 680, 0.75
 700, 0.84
 720, 0.86
 740, 0.87
}

// #72 AZURE BLUE
// Color Effects Lighting Filter
#declare TS_R72 = spline{ linear_spline
 360, 0.18
 380, 0.28
 400, 0.47
 420, 0.61
 440, 0.72
 460, 0.78
 480, 0.77
 500, 0.72
 520, 0.62
 540, 0.46
 560, 0.31
 580, 0.18
 600, 0.13
 620, 0.11
 640, 0.10
 660, 0.16
 680, 0.18
 700, 0.29
 720, 0.31
 740, 0.43
}

// #372 THEATRE BOOSTER 2
// Color Effects Lighting Filter
#declare TS_R372 = spline{ linear_spline
 360, 0.39
 380, 0.51
 400, 0.67
 420, 0.74
 440, 0.77
 460, 0.77
 480, 0.75
 500, 0.70
 520, 0.64
 540, 0.57
 560, 0.52
 580, 0.46
 600, 0.46
 620, 0.47
 640, 0.48
 660, 0.46
 680, 0.36
 700, 0.52
 720, 0.54
 740, 0.63
}

// #73 PEACOCK BLUE
// Color Effects Lighting Filter
#declare TS_R73 = spline{ linear_spline
 360, 0.38
 380, 0.33
 400, 0.27
 420, 0.28
 440, 0.38
 460, 0.56
 480, 0.65
 500, 0.63
 520, 0.52
 540, 0.36
 560, 0.23
 580, 0.13
 600, 0.08
 620, 0.07
 640, 0.06
 660, 0.08
 680, 0.21
 700, 0.47
 720, 0.72
 740, 0.83
}

// #373 THEATRE BOOSTER 3
// Color Effects Lighting Filter
#declare TS_R373 = spline{ linear_spline
 360, 0.72
 380, 0.76
 400, 0.79
 420, 0.82
 440, 0.84
 460, 0.84
 480, 0.83
 500, 0.81
 520, 0.78
 540, 0.74
 560, 0.72
 580, 0.67
 600, 0.68
 620, 0.67
 640, 0.67
 660, 0.78
 680, 0.85
 700, 0.87
 720, 0.88
 740, 0.88
}

// #74 NIGHT BLUE
// Color Effects Lighting Filter
#declare TS_R74 = spline{ linear_spline
 360, 0.01
 380, 0.01
 400, 0.05
 420, 0.14
 440, 0.39
 460, 0.51
 480, 0.38
 500, 0.17
 520, 0.05
 540, 0.00
 560, 0.00
 580, 0.00
 600, 0.00
 620, 0.00
 640, 0.00
 660, 0.00
 680, 0.02
 700, 0.05
 720, 0.05
 740, 0.17
}

// #374 SEA GREEN
// Color Effects Lighting Filter
#declare TS_R374 = spline{ linear_spline
 360, 0.14
 380, 0.21
 400, 0.26
 420, 0.30
 440, 0.41
 460, 0.54
 480, 0.63
 500, 0.63
 520, 0.56
 540, 0.42
 560, 0.27
 580, 0.14
 600, 0.07
 620, 0.04
 640, 0.03
 660, 0.02
 680, 0.01
 700, 0.05
 720, 0.22
 740, 0.49
}

// #75 TWILIGHT BLUE
// Color Effects Lighting Filter
#declare TS_R75 = spline{ linear_spline
 360, 0.27
 380, 0.25
 400, 0.22
 420, 0.24
 440, 0.36
 460, 0.47
 480, 0.49
 500, 0.42
 520, 0.29
 540, 0.14
 560, 0.06
 580, 0.02
 600, 0.01
 620, 0.01
 640, 0.00
 660, 0.02
 680, 0.05
 700, 0.08
 720, 0.20
 740, 0.44
}

// #375 CERULEAN BLUE
// Color Effects Lighting Filter
#declare TS_R375 = spline{ linear_spline
 360, 0.48
 380, 0.48
 400, 0.48
 420, 0.47
 440, 0.45
 460, 0.53
 480, 0.66
 500, 0.61
 520, 0.50
 540, 0.38
 560, 0.24
 580, 0.12
 600, 0.05
 620, 0.01
 640, 0.00
 660, 0.00
 680, 0.00
 700, 0.00
 720, 0.00
 740, 0.00
}

// #76 LT GREEN BLUE
// Color Effects Lighting Filter
#declare TS_R76 = spline{ linear_spline
 360, 0.14
 380, 0.11
 400, 0.08
 420, 0.09
 440, 0.17
 460, 0.31
 480, 0.38
 500, 0.32
 520, 0.18
 540, 0.06
 560, 0.02
 580, 0.00
 600, 0.00
 620, 0.00
 640, 0.00
 660, 0.01
 680, 0.08
 700, 0.33
 720, 0.64
 740, 0.81
}

// #376 BERMUDA BLUE
// Color Effects Lighting Filter
#declare TS_R376 = spline{ linear_spline
 360, 0.46
 380, 0.54
 400, 0.53
 420, 0.43
 440, 0.35
 460, 0.39
 480, 0.52
 500, 0.46
 520, 0.33
 540, 0.21
 560, 0.12
 580, 0.06
 600, 0.04
 620, 0.04
 640, 0.03
 660, 0.10
 680, 0.39
 700, 0.67
 720, 0.79
 740, 0.82
}

// #77 GREEN BLUE
// Color Effects Lighting Filter
#declare TS_R77 = spline{ linear_spline
 360, 0.36
 380, 0.47
 400, 0.49
 420, 0.40
 440, 0.31
 460, 0.33
 480, 0.42
 500, 0.32
 520, 0.17
 540, 0.06
 560, 0.02
 580, 0.00
 600, 0.00
 620, 0.00
 640, 0.00
 660, 0.602
 680, 0.29
 700, 0.65
 720, 0.78
 740, 0.81
}

// #377 IRIS PURPLE
// Color Effects Lighting Filter
#declare TS_R377 = spline{ linear_spline
 360, 0.28
 380, 0.33
 400, 0.40
 420, 0.52
 440, 0.63
 460, 0.56
 480, 0.39
 500, 0.20
 520, 0.09
 540, 0.03
 560, 0.02
 580, 0.01
 600, 0.01
 620, 0.02
 640, 0.02
 660, 0.21
 680, 0.62
 700, 0.80
 720, 0.84
 740, 0.85
}

// #78 TRUDY BLUE
// Color Effects Lighting Filter
#declare TS_R78 = spline{ linear_spline
 360, 0.34
 380, 0.45
 400, 0.55
 420, 0.62
 440, 0.68
 460, 0.65
 480, 0.56
 500, 0.43
 520, 0.30
 540, 0.18
 560, 0.12
 580, 0.07
 600, 0.08
 620, 0.09
 640, 0.10
 660, 0.36
 680, 0.71
 700, 0.84
 720, 0.87
 740, 0.88
}

// #378 ALICE BLUE
// Color Effects Lighting Filter
#declare TS_R378 = spline{ linear_spline
 360, 0.47
 380, 0.58
 400, 0.67
 420, 0.68
 440, 0.61
 460, 0.49
 480, 0.36
 500, 0.26
 520, 0.16
 540, 0.13
 560, 0.08
 580, 0.08
 600, 0.08
 620, 0.08
 640, 0.08
 660, 0.15
 680, 0.51
 700, 0.75
 720, 0.81
 740, 0.83
}

// #79 BRIGHT BLUE
// Color Effects Lighting Filter
#declare TS_R79 = spline{ linear_spline
 360, 0.14
 380, 0.15
 400, 0.19
 420, 0.31
 440, 0.54
 460, 0.62
 480, 0.50
 500, 0.29
 520, 0.13
 540, 0.03
 560, 0.01
 580, 0.00
 600, 0.00
 620, 0.00
 640, 0.00
 660, 0.05
 680, 0.43
 700, 0.74
 720, 0.84
 740, 0.86
}

// R80 - Primary Blue
#declare TS_R80 = spline{ linear_spline
 360, 0.35
 380, 0.48
 400, 0.60
 420, 0.63
 440, 0.58
 460, 0.51
 480, 0.44
 500, 0.31
 520, 0.15
 540, 0.05
 560, 0.01
 580, 0.00
 600, 0.00
 620, 0.00
 640, 0.00
 660, 0.02
 680, 0.29
 700, 0.64
 720, 0.77
 740, 0.81
}

// #81 URBAN BLUE
// Color Effects Lighting Filter
#declare TS_R81 = spline{ linear_spline
 360, 0.24
 380, 0.30
 400, 0.37
 420, 0.46
 440, 0.60
 460, 0.62
 480, 0.51
 500, 0.35
 520, 0.20
 540, 0.09
 560, 0.05
 580, 0.02
 600, 0.02
 620, 0.02
 640, 0.02
 660, 0.16
 680, 0.53
 700, 0.76
 720, 0.84
 740, 0.87
}

// #381 BALDASSARI BLUE
// Color Effects Lighting Filter
#declare TS_R381 = spline{ linear_spline
 360, 0.28
 380, 0.28
 400, 0.28
 420, 0.44
 440, 0.43
 460, 0.29
 480, 0.16
 500, 0.09
 520, 0.04
 540, 0.03
 560, 0.02
 580, 0.02
 600, 0.01
 620, 0.00
 640, 0.00
 660, 0.00
 680, 0.01
 700, 0.07
 720, 0.07
 740, 0.07
}

// #82 SURPRISE BLUE
// Color Effects Lighting Filter
#declare TS_R82 = spline{ linear_spline
 360, 0.14
 380, 0.23
 400, 0.30
 420, 0.38
 440, 0.46
 460, 0.43
 480, 0.33
 500, 0.21
 520, 0.11
 540, 0.04
 560, 0.02
 580, 0.00
 600, 0.01
 620, 0.01
 640, 0.01
 660, 0.14
 680, 0.56
 700, 0.80
 720, 0.86
 740, 0.87
}

// #382 CONGO BLUE
// Color Effects Lighting Filter
#declare TS_R382 = spline{ linear_spline
 360, 0.01
 380, 0.01
 400, 0.02
 420, 0.07
 440, 0.23
 460, 0.24
 480, 0.11
 500, 0.02
 520, 0.00
 540, 0.00
 560, 0.00
 580, 0.00
 600, 0.00
 620, 0.00
 640, 0.00
 660, 0.00
 680, 0.20
 700, 0.63
 720, 0.81
 740, 0.86
}

// #83 MED BLUE
// Color Effects Lighting Filter
#declare TS_R83 = spline{ linear_spline
 360, 0.05
 380, 0.05
 400, 0.07
 420, 0.15
 440, 0.37
 460, 0.50
 480, 0.38
 500, 0.17
 520, 0.05
 540, 0.01
 560, 0.00
 580, 0.00
 600, 0.00
 620, 0.00
 640, 0.00
 660, 0.01
 680, 0.22
 700, 0.59
 720, 0.78
 740, 0.85
}

// #383 SAPPHIRE BLUE
// Color Effects Lighting Filter
#declare TS_R383 = spline{ linear_spline
 360, 0.05
 380, 0.08
 400, 0.12
 420, 0.21
 440, 0.38
 460, 0.40
 480, 0.26
 500, 0.12
 520, 0.03
 540, 0.01
 560, 0.00
 580, 0.00
 600, 0.00
 620, 0.00
 640, 0.00
 660, 0.02
 680, 0.26
 700, 0.52
 720, 0.67
 740, 0.78
}

// #84 ZEPHYR BLUE
// Color Effects Lighting Filter
#declare TS_R84 = spline{ linear_spline
 360, 0.10
 380, 0.22
 400, 0.41
 420, 0.51
 440, 0.57
 460, 0.52
 480, 0.42
 500, 0.29
 520, 0.18
 540, 0.09
 560, 0.05
 580, 0.02
 600, 0.03
 620, 0.03
 640, 0.04
 660, 0.15
 680, 0.24
 700, 0.41
 720, 0.45
 740, 0.57
}

// #384 MIDNIGHT BLUE
// Color Effects Lighting Filter
#declare TS_R384 = spline{ linear_spline
 360, 0.06
 380, 0.09
 400, 0.13
 420, 0.25
 440, 0.44
 460, 0.40
 480, 0.22
 500, 0.07
 520, 0.02
 540, 0.00
 560, 0.00
 580, 0.00
 600, 0.00
 620, 0.00
 640, 0.00
 660, 0.01
 680, 0.05
 700, 0.14
 720, 0.37
 740, 0.67
}

// #85 DEEP BLUE
// Color Effects Lighting Filter
#declare TS_R85 = spline{ linear_spline
 360, 0.03
 380, 0.03
 400, 0.04
 420, 0.08
 440, 0.25
 460, 0.37
 480, 0.28
 500, 0.12
 520, 0.03
 540, 0.00
 560, 0.00
 580, 0.00
 600, 0.00
 620, 0.00
 640, 0.00
 660, 0.00
 680, 0.14
 700, 0.50
 720, 0.76
 740, 0.86
}

// #385 ROYAL BLUE
// Color Effects Lighting Filter
#declare TS_R385 = spline{ linear_spline
 360, 0.01
 380, 0.01
 400, 0.01
 420, 0.05
 440, 0.26
 460, 0.36
 480, 0.19
 500, 0.04
 520, 0.00
 540, 0.00
 560, 0.00
 580, 0.00
 600, 0.00
 620, 0.00
 640, 0.00
 660, 0.00
 680, 0.14
 700, 0.57
 720, 0.79
 740, 0.86
}

// #86 PEA GREEN
// Color Effects Lighting Filter
#declare TS_R86 = spline{ linear_spline
 360, 0.38
 380, 0.32
 400, 0.18
 420, 0.10
 440, 0.08
 460, 0.08
 480, 0.16
 500, 0.38
 520, 0.60
 540, 0.64
 560, 0.57
 580, 0.44
 600, 0.33
 620, 0.22
 640, 0.15
 660, 0.13
 680, 0.11
 700, 0.12
 720, 0.24
 740, 0.48
}

// #386 LEAF GREEN
// Color Effects Lighting Filter
#declare TS_R386 = spline{ linear_spline
 360, 0.30
 380, 0.29
 400, 0.12
 420, 0.02
 440, 0.01
 460, 0.06
 480, 0.39
 500, 0.55
 520, 0.59
 540, 0.59
 560, 0.52
 580, 0.42
 600, 0.30
 620, 0.18
 640, 0.12
 660, 0.08
 680, 0.07
 700, 0.09
 720, 0.25
 740, 0.53
}

// #87 PALE YELLOW GREEN
// Color Effects Lighting Filter
#declare TS_R87 = spline{ linear_spline
 360, 0.59
 380, 0.63
 400, 0.60
 420, 0.50
 440, 0.45
 460, 0.53
 480, 0.75
 500, 0.78
 520, 0.79
 540, 0.78
 560, 0.77
 580, 0.74
 600, 0.71
 620, 0.67
 640, 0.63
 660, 0.61
 680, 0.60
 700, 0.64
 720, 0.74
 740, 0.81
}

// #88 LT GREEN
// Color Effects Lighting Filter
#declare TS_R88 = spline{ linear_spline
 360, 0.54
 380, 0.57
 400, 0.47
 420, 0.30
 440, 0.23
 460, 0.35
 480, 0.70
 500, 0.77
 520, 0.78
 540, 0.77
 560, 0.74
 580, 0.70
 600, 0.64
 620, 0.58
 640, 0.52
 660, 0.48
 680, 0.46
 700, 0.51
 720, 0.65
 740, 0.78
}

// #388 GASLT GREEN
// Color Effects Lighting Filter
#declare TS_R388 = spline{ linear_spline
 360, 0.13
 380, 0.05
 400, 0.02
 420, 0.03
 440, 0.12
 460, 0.33
 480, 0.58
 500, 0.74
 520, 0.80
 540, 0.79
 560, 0.74
 580, 0.68
 600, 0.62
 620, 0.53
 640, 0.47
 660, 0.45
 680, 0.42
 700, 0.44
 720, 0.55
 740, 0.70
}

// #89 MOSS GREEN
// Color Effects Lighting Filter
#declare TS_R89 = spline{ linear_spline
 360, 0.31
 380, 0.25
 400, 0.12
 420, 0.06
 440, 0.06
 460, 0.08
 480, 0.16
 500, 0.36
 520, 0.53
 540, 0.52
 560, 0.39
 580, 0.23
 600, 0.13
 620, 0.06
 640, 0.03
 660, 0.02
 680, 0.01
 700, 0.02
 720, 0.07
 740, 0.28
}

// #389 CHROMA GREEN
// Color Effects Lighting Filter
#declare TS_R389 = spline{ linear_spline
 360, 0.04
 380, 0.04
 400, 0.03
 420, 0.00
 440, 0.00
 460, 0.03
 480, 0.45
 500, 0.77
 520, 0.72
 540, 0.55
 560, 0.32
 580, 0.18
 600, 0.11
 620, 0.07
 640, 0.07
 660, 0.07
 680, 0.05
 700, 0.07
 720, 0.06
 740, 0.14
}

// #90 DARK YELLOW GREEN
// Color Effects Lighting Filter
#declare TS_R90 = spline{ linear_spline
 360, 0.08
 380, 0.05
 400, 0.01
 420, 0.00
 440, 0.00
 460, 0.01
 480, 0.04
 500, 0.14
 520, 0.26
 540, 0.20
 560, 0.09
 580, 0.02
 600, 0.00
 620, 0.00
 640, 0.00
 660, 0.00
 680, 0.00
 700, 0.00
 720, 0.00
 740, 0.04
}

// #91  PRIMARY GREEN
// Color Effects Lighting Filter
#declare TS_R91 = spline{ linear_spline
 360, 0.13
 380, 0.07
 400, 0.01
 420, 0.00
 440, 0.00
 460, 0.00
 480, 0.08
 500, 0.16
 520, 0.11
 540, 0.06
 560, 0.02
 580, 0.01
 600, 0.00
 620, 0.00
 640, 0.00
 660, 0.02
 680, 0.28
 700, 0.63
 720, 0.77
 740, 0.81
}

// #92 TURQUOISE
// Color Effects Lighting Filter
#declare TS_R92 = spline{ linear_spline
 360, 0.40
 380, 0.51
 400, 0.52
 420, 0.42
 440, 0.37
 460, 0.47
 480, 0.72
 500, 0.74
 520, 0.69
 540, 0.62
 560, 0.50
 580, 0.37
 600, 0.25
 620, 0.14
 640, 0.08
 660, 0.06
 680, 0.04
 700, 0.06
 720, 0.21
 740, 0.51
}

// #392 PACIFIC GREEN
// Color Effects Lighting Filter
#declare TS_R392 = spline{ linear_spline
 360, 0.11
 380, 0.17
 400, 0.20
 420, 0.26
 440, 0.30
 460, 0.45
 480, 0.54
 500, 0.57
 520, 0.51
 540, 0.38
 560, 0.23
 580, 0.12
 600, 0.06
 620, 0.04
 640, 0.03
 660, 0.03
 680, 0.02
 700, 0.07
 720, 0.14
 740, 0.31
}

// #93 BLUE GREEN
// Color Effects Lighting Filter
#declare TS_R93 = spline{ linear_spline
 360, 0.53
 380, 0.55
 400, 0.44
 420, 0.26
 440, 0.19
 460, 0.27
 480, 0.59
 500, 0.58
 520, 0.48
 540, 0.33
 560, 0.22
 580, 0.12
 600, 0.08
 620, 0.08
 640, 0.06
 660, 0.20
 680, 0.59
 700, 0.78
 720, 0.82
 740, 0.84
}

// #393 EMERALD GREEN
// Color Effects Lighting Filter
#declare TS_R393 = spline{ linear_spline
 360, 0.13
 380, 0.14
 400, 0.10
 420, 0.08
 440, 0.13
 460, 0.23
 480, 0.33
 500, 0.40
 520, 0.39
 540, 0.25
 560, 0.12
 580, 0.04
 600, 0.01
 620, 0.00
 640, 0.00
 660, 0.00
 680, 0.00
 700, 0.00
 720, 0.02
 740, 0.15
}

// #94 KELLY GREEN
// Color Effects Lighting Filter
#declare TS_R94 = spline{ linear_spline
 360, 0.21
 380, 0.17
 400, 0.08
 420, 0.05
 440, 0.08
 460, 0.15
 480, 0.26
 500, 0.38
 520, 0.44
 540, 0.34
 560, 0.18
 580, 0.07
 600, 0.02
 620, 0.00
 640, 0.00
 660, 0.00
 680, 0.00
 700, 0.00
 720, 0.01
 740, 0.10
}

// #95 MED BLUE GREEN
// Color Effects Lighting Filter
#declare TS_R95 = spline{ linear_spline
 360, 0.18
 380, 0.11
 400, 0.06
 420, 0.05
 440, 0.10
 460, 0.23
 480, 0.36
 500, 0.39
 520, 0.28
 540, 0.14
 560, 0.05
 580, 0.02
 600, 0.01
 620, 0.00
 640, 0.00
 660, 0.01
 680, 0.04
 700, 0.22
 720, 0.56
 740, 0.77
}

// #395 TEAL GREEN
// Color Effects Lighting Filter
#declare TS_R395 = spline{ linear_spline
 360, 0.19
 380, 0.15
 400, 0.08
 420, 0.06
 440, 0.11
 460, 0.22
 480, 0.34
 500, 0.41
 520, 0.37
 540, 0.24
 560, 0.11
 580, 0.04
 600, 0.01
 620, 0.00
 640, 0.00
 660, 0.00
 680, 0.00
 700, 0.01
 720, 0.04
 740, 0.21
}

// #96 LIME
// Color Effects Lighting Filter
#declare TS_R96 = spline{ linear_spline
 360, 0.74
 380, 0.64
 400, 0.46
 420, 0.28
 440, 0.21
 460, 0.28
 480, 0.62
 500, 0.85
 520, 0.87
 540, 0.87
 560, 0.87
 580, 0.86
 600, 0.85
 620, 0.84
 640, 0.83
 660, 0.82
 680, 0.82
 700, 0.82
 720, 0.84
 740, 0.87
}

// #97 LT GREY
// Color Effects Lighting Filter
#declare TS_R97 = spline{ linear_spline
 360, 0.47
 380, 0.52
 400, 0.49
 420, 0.42
 440, 0.36
 460, 0.40
 480, 0.52
 500, 0.49
 520, 0.43
 540, 0.44
 560, 0.41
 580, 0.47
 600, 0.48
 620, 0.44
 640, 0.42
 660, 0.39
 680, 0.45
 700, 0.59
 720, 0.73
 740, 0.79
}

// #397 PALE GREY
// Color Effects Lighting Filter
#declare TS_R397 = spline{ linear_spline
 360, 0.31
 380, 0.41
 400, 0.57
 420, 0.58
 440, 0.54
 460, 0.58
 480, 0.66
 500, 0.64
 520, 0.60
 540, 0.61
 560, 0.59
 580, 0.64
 600, 0.65
 620, 0.62
 640, 0.60
 660, 0.58
 680, 0.60
 700, 0.68
 720, 0.77
 740, 0.83
}

// #98 MED GREY
// Color Effects Lighting Filter
#declare TS_R98 = spline{ linear_spline
 360, 0.35
 380, 0.39
 400, 0.33
 420, 0.22
 440, 0.15
 460, 0.19
 480, 0.32
 500, 0.28
 520, 0.20
 540, 0.21
 560, 0.18
 580, 0.25
 600, 0.27
 620, 0.22
 640, 0.20
 660, 0.18
 680, 0.24
 700, 0.43
 720, 0.65
 740, 0.78
}

// #398 NEUTRAL GREY
// Color Effects Lighting Filter
#declare TS_R398 = spline{ linear_spline
 360, 0.23
 380, 0.30
 400, 0.35
 420, 0.39
 440, 0.41
 460, 0.43
 480, 0.43
 500, 0.44
 520, 0.43
 540, 0.42
 560, 0.41
 580, 0.40
 600, 0.40
 620, 0.41
 640, 0.41
 660, 0.37
 680, 0.28
 700, 0.44
 720, 0.48
 740, 0.52
}

// #99 CHOCOLATE
// Color Effects Lighting Filter
#declare TS_R99 = spline{ linear_spline
 360, 0.39
 380, 0.46
 400, 0.37
 420, 0.21
 440, 0.13
 460, 0.16
 480, 0.33
 500, 0.28
 520, 0.21
 540, 0.24
 560, 0.22
 580, 0.39
 600, 0.49
 620, 0.42
 640, 0.34
 660, 0.30
 680, 0.28
 700, 0.30
 720, 0.49
 740, 0.70
}

//=============================================================
       
#version tspd_roscolux_Inc_Temp;
#end // eof