/*
  Persistence of Vision Ray Tracer Scene Description File
  
  TheatreSys : main instrument macros
  
  Version 1.0
  
  Macros to create simulations of theatrical lighting instruments with
  real parameters from existing instruments. Uses CIE.inc by Ive for
  color calculations, and LightSysIV by Jamie Vives Piqueres for global
  scaling and intensity.
  
  See readme.theatresys.txt for more information
  
  Christopher Shake, Feb-2009
  
  Changes on version 1: (2009.03.13)
    + Created Ellipsoidals, PARs, Striplights
    + Created Iris, Shutters, Gobo, Gel, and Color Scroller functions
    + Compiled Rosco, Lee, and Apollo Gel libraries
    + Created PowersTheatre backdrop template
  
  Contents:
    Ellipsoidals:
      Altman Instruments:
        A360Q_4p5x6p5(Location,PointAt,Intensity,Sharpness)
        A360Q_6x9(Location,PointAt,Intensity,Sharpness)
        A360Q_6x12(Location,PointAt,Intensity,Sharpness)
        A360Q_6x16(Location,PointAt,Intensity,Sharpness)
        A360Q_6x22(Location,PointAt,Intensity,Sharpness)
        Shakespeare_S65(Location,PointAt,Intensity,Sharpness)
        Shakespeare_S610(Location,PointAt,Intensity,Sharpness)
        Shakespeare_S612(Location,PointAt,Intensity,Sharpness)
        Shakespeare_S620(Location,PointAt,Intensity,Sharpness)
        Shakespeare_S630(Location,PointAt,Intensity,Sharpness)
        Shakespeare_S640(Location,PointAt,Intensity,Sharpness)
        Shakespeare_S650(Location,PointAt,Intensity,Sharpness)
        Shakespeare_S61535Z(Location,PointAt,Intensity,Sharpness,Zoom)
        Shakespeare_S63055Z(Location,PointAt,Intensity,Sharpness,Zoom)
      ETC instruments:
        SourceFour_05(Location,PointAt,Intensity,Sharpness)
        SourceFour_10(Location,PointAt,Intensity,Sharpness)
        SourceFour_14(Location,PointAt,Intensity,Sharpness)
        SourceFour_19(Location,PointAt,Intensity,Sharpness)
        SourceFour_26(Location,PointAt,Intensity,Sharpness)
        SourceFour_36(Location,PointAt,Intensity,Sharpness)
        SourceFour_50(Location,PointAt,Intensity,Sharpness)
        SourceFour_70(Location,PointAt,Intensity,Sharpness)
        SourceFour_90(Location,PointAt,Intensity,Sharpness)
        SourceFour_1530Z(Location,PointAt,Intensity,Sharpness,Zoom)
        SourceFour_2550Z(Location,PointAt,Intensity,Sharpness,Zoom)
      Strand instruments:
        Coolbeam_SL5(Location,PointAt,Intensity,Sharpness)
        Coolbeam_SL10(Location,PointAt,Intensity,Sharpness)
        Coolbeam_SL19(Location,PointAt,Intensity,Sharpness)
        Coolbeam_SL26(Location,PointAt,Intensity,Sharpness)
        Coolbeam_SL36(Location,PointAt,Intensity,Sharpness)
        Coolbeam_SL50(Location,PointAt,Intensity,Sharpness)
        Coolbeam_SL1532Z(Location,PointAt,Intensity,Sharpness,Zoom)
        Coolbeam_SL2350Z(Location,PointAt,Intensity,Sharpness,Zoom)
    Striplights:
      PAR38_6ft_3ckt(Center,Axis,PointAt,Intensity[3],LampName,Color[3])
      PAR38_6ft_4ckt(Center,Axis,PointAt,Intensity[4],LampName,Color[4])
      PAR38_7ft6in_3ckt(Center,Axis,PointAt,Intensity[3],LampName,Color[3])
      PAR38_8ft_4ckt(Center,Axis,PointAt,Intensity[4],LampName,Color[4])
      MR16_ZS_1(Center,Axis,PointAt,Intensity[1],LampName,Color[1])
      MR16_ZS_2(Center,Axis,PointAt,Intensity[2],LampName,Color[2])
      MR16_ZS_3(Center,Axis,PointAt,Intensity[3],LampName,Color[3])
      MR16_ZS_4(Center,Axis,PointAt,Intensity[4],LampName,Color[4])
    Other Instruments:
      PAR64Can(Location,PointAt,Intensity,LongAxis,LampName)
    Bulbs:
      MR-16:
        MR16_EYF(Location,PointAt,Intensity)
        MR16_EYJ(Location,PointAt,Intensity)
        MR16_EYC(Location,PointAt,Intensity)
        MR16_FPA(Location,PointAt,Intensity)
        MR16_FPC(Location,PointAt,Intensity)
        MR16_FPB(Location,PointAt,Intensity)
      PAR38:
        PAR38_250SP(Location,PointAt,Intensity)
        PAR38_250FL(Location,PointAt,Intensity)
        PAR38_120SP(Location,PointAt,Intensity)
        PAR38_120FL(Location,PointAt,Intensity)
      PAR64:
        Q1000PAR64_1(Location,PointAt,Intensity,LongAxis)
        Q1000PAR64_2(Location,PointAt,Intensity,LongAxis)
        Q1000PAR64_5(Location,PointAt,Intensity,LongAxis)
        Q1000PAR64_6(Location,PointAt,Intensity,LongAxis)
        Q1000PAR64_3D(Location,PointAt,Intensity,LongAxis)
        Q1000PAR64_7D(Location,PointAt,Intensity,LongAxis)
        Q1000PAR64_NSP(Location,PointAt,Intensity,LongAxis)
        Q1000PAR64_MFL(Location,PointAt,Intensity,LongAxis)
        Q1000PAR64_WFL(Location,PointAt,Intensity,LongAxis)
    Instrument accessories:
      InsertGobo(GoboImageMap,Rotation)
      InsertGel(GelSpectrum)
      InsertColorScroller(GelArray,Index,SnapTo)
      InsertIris(Amount)
      ShutterCutTop(Amount,Angle)
      ShutterCutBottom(Amount,Angle)
      ShutterCutLeft(Amount,Angle)
      ShutterCutRight(Amount,Angle)

    Other useful functions:
      InsertHaze(Density,Samples,Scale) - call inside bounding object with only shape defined

    Texture Definitions:
      BlackPaintTexture
      BlackDrapeTexture
      CycTexture
      ScrimTexture

    plus various background helper functions.
    
    General Notes:
    + Intensity and Sharpness are all on a scale of 0 to 1
    + Declare 'Global_TheatreSys_Unit' to the base unit you use in POV-Ray:
       Possible values: [1 "cm"| 2 "in"| 3 "m"| 4 "ft"| 5 "mm"]
       Must be declared before file is included, else 1 ('cm') will be used.
    + Declare 'Use_Fading' to 0 or 1 to use distance based light fading
    + Declare 'Use_Soft_Shadows' to 0 or 1, disabling speeds up renders,
      enabling blurs gobos/shutters when focus is soft.
    + Gobos must be inserted before Shutters, for unknown reasons.
    + Shutter cut Amount is 1 for fully open, 0 for cut to the center of the beam
    + Zoom parameter is 0 for narrow, 1 for wide beam
    + PAR Lamps can be created on their own as well as in instruments that take them
    + Striplights need intensities and colors in array form. They will accept arrays
      that are too long, and simply ignore the extra values, but will error if the
      array is too short.
    + Striplight and PARCan instruments take the macro name for a lamp as the
      LampName parameter, eg: LampName = "PAR38_250FL"
    + Bulbs can be called individually, or will be created as part of an instrument.
      When created individually, Gels will not work

    Requires CIE.inc, lightsys.inc, transforms.inc
    
 This work is licensed under the Creative Commons Attribution-Noncommercial 3.0 Unported License
  http://creativecommons.org/licenses/by-nc/3.0/
*/
#ifndef(THEATRESYS_INC_TEMP)
#declare THEATRESYS_INC_TEMP = version;
#version 3.6;
#include "transforms.inc"
#include "CIE.inc"
#include "lightsys.inc"
#default{texture{finish{ambient 0}}}

#ifdef(View_POV_Include_Stack)
	#debug "including theatresys.inc\n";
#end
#ifndef(Global_TheatreSys_Unit)
	#declare Global_TheatreSys_Unit = 1;//"cm";
#end
#if(Global_TheatreSys_Unit=1)//"cm")
  #declare UnitMult = 1;
#end
#if(Global_TheatreSys_Unit=2)//"in")
  #declare UnitMult = 1/2.54;
#end
#if(Global_TheatreSys_Unit=3)//"m")
  #declare UnitMult = 0.01;
#end
#if(Global_TheatreSys_Unit=4)//"ft")
  #declare UnitMult = 1/(12*2.54);
#end
#if(Global_TheatreSys_Unit=5)//"mm")
  #declare UnitMult = 10;
#end
#ifndef(UnitMult)
  #declare UnitMult = 1;
#end

#declare GlobalGoboDistance = 2*UnitMult;
#declare GlobalGoboThickness = 0.001*UnitMult;
#declare GlobalGelDistance = 10*UnitMult;
#declare GlobalGelThickness = 0.001*UnitMult;
#declare DistanceToPAROval = 8*UnitMult;
#declare InstrumentRadius = 10*UnitMult;
#declare DEBUG = no;

#ifndef(Use_Photons)
  #declare Use_Photons = no;
#end
#ifndef(Use_Fading)
	#declare Use_Fading = yes;
#end
#ifndef(Use_Soft_Shadows)
	#declare Use_Soft_Shadows = no;
#end
#ifndef(Gobo_Blur_Amount)
  #declare Gobo_Blur_Amount = 1;
#end
#ifndef(My_ColSys)
	CIE_ChromaticAdaption(0)
	#declare My_ColSys = sRGB_ColSys
	CIE_ColorSystem(My_ColSys)
	CIE_ColorSystemWhitepoint(My_ColSys, Illuminant_D50)
#end

// ==============================================================================
// Instrument Accessories - BEGIN
// ==============================================================================

// Amount 1 is fully open, 0 is closed
// Angle is in degrees from flat
#macro ShutterCutTop(Amount,Angle)
	#ifdef(GoboSlot)
		difference{
			cylinder{0,GlobalGoboThickness*x,0.5}
			box{<-GlobalGoboThickness,-1,-1>,<GlobalGoboThickness*2,0,1> rotate Angle*x translate 0.5*(1-Amount)*y}
			pigment{rgbt 0}
			scale <1,1.1,1.1>
			transform{GoboSlot}
      IncrementGoboPlacement()
		}
	#end
#end

// Amount 1 is fully open, 0 is closed
// Angle is in degrees from flat
#macro ShutterCutBottom(Amount,Angle)
	#ifdef(GoboSlot)
		difference{
			cylinder{0,GlobalGoboThickness*x,0.5}
			box{<-GlobalGoboThickness,1,-1>,<GlobalGoboThickness*2,0,1> rotate Angle*x translate -0.5*(1-Amount)*y}
			pigment{rgbt 0}
			scale <1,1.1,1.1>
			transform{GoboSlot}
      IncrementGoboPlacement()
		}
	#end
#end

// Amount 1 is fully open, 0 is closed
// Angle is in degrees from flat
#macro ShutterCutLeft(Amount,Angle)
	#ifdef(GoboSlot)
		difference{
			cylinder{0,GlobalGoboThickness*x,0.5}
			box{<-GlobalGoboThickness,-1,0>,<GlobalGoboThickness*2,1,-1> rotate Angle*x translate 0.5*(1-Amount)*z}
			pigment{rgbt 0}
			scale <1,1.1,1.1>
			transform{GoboSlot}
      IncrementGoboPlacement()
		}
	#end
#end

// Amount 1 is fully open, 0 is closed
// Angle is in degrees from flat
#macro ShutterCutRight(Amount,Angle)
	#ifdef(GoboSlot)
		difference{
			cylinder{0,GlobalGoboThickness*x,0.5}
			box{<-GlobalGoboThickness,-1,0>,<GlobalGoboThickness*2,1,1> rotate Angle*x translate -0.5*(1-Amount)*z}
			pigment{rgbt 0}
			scale <1,1.1,1.1>
			transform{GoboSlot}
      IncrementGoboPlacement()
		}
	#end
#end

// 0 is closed, 1 is open
#macro InsertIris(Amount)
  #ifdef(GoboSlot)
    #if(Amount > 0)
      difference{
        cylinder{0,GlobalGoboThickness*x,0.5+GlobalGoboThickness}
        cylinder{-GlobalGoboThickness*x,GlobalGoboThickness*2*x,0.5*Amount}
        pigment{rgbt 0}
        scale <1,1.1,1.1>
        transform{GoboSlot}
      }
    #else
      cylinder{0,GlobalGoboThickness*x,0.5+GlobalGoboThickness
        pigment{rgbt 0}
        scale <1,1.1,1.1>
        transform{GoboSlot}
      }
    #end
    IncrementGoboPlacement()
  #end
#end

// Gobo is an image_map, required parameters: once, filter all 1
// example: #local Gobo = pigment{image_map{jpeg "xxxx.jpg" once filter all 1}}
// Rotation is in degrees
#macro InsertGobo(GoboImageMap,Rotation)
	#ifdef(GoboSlot)
		cylinder{0,GlobalGoboThickness*x,0.5
			texture{pigment{GoboImageMap translate <-0.5,-0.5,0> rotate 90*y}}
			scale <1,1.1,1.1>
			rotate Rotation*x
			transform{GoboSlot}
		}
    IncrementGoboPlacement()
	#end
#end

// GelSpectrum is a spline of wavelength transmission, as used in CIE.inc
#macro InsertGel(GelSpectrum)
	#ifdef(GelSlot)
		cylinder{0,GlobalGelThickness*x,InstrumentRadius
			texture{pigment{rgb ReflectiveSpectrum(GelSpectrum) filter 1}}
			transform{GelSlot}
      double_illuminate
		}
    IncrementGelPlacement()
	#end
#end

// GelArray is an array of colors in the string - {TS_Rxxx, TS_Rxxx, TS_Lxxx, etc}
#macro InsertColorScroller(GelArray,Index,SnapTo)
	#ifdef(GelSlot)
		#local incr = 0.001*UnitMult;
		#local frameindex = round(Index/SnapTo)*SnapTo;
		#local numcolors = dimension_size(GelArray,1);
		cylinder{0,GlobalGelThickness*x,InstrumentRadius
			texture{pigment{
				gradient y
				color_map{
					#local i=0;
					#while (i < numcolors-1)
						[(i+1)/numcolors-incr rgb ReflectiveSpectrum(GelArray[i]) filter 1]
						[(i+1)/numcolors+incr rgb ReflectiveSpectrum(GelArray[i+1]) filter 1]
						#local i=i+1;
					#end
				}
				scale <1,numcolors*InstrumentRadius*2,InstrumentRadius*2>
				translate <0,-InstrumentRadius,-InstrumentRadius>
				translate <0,-frameindex*InstrumentRadius*2,0>
			}}
			transform{GelSlot}
		}
    IncrementGelPlacement()
	#end
#end

// ==============================================================================
// Instrument Accessories - END
// ==============================================================================

// ==============================================================================
// Other Functions - BEGIN
// ==============================================================================

// Creates haze in parent object. Scale will usually be 1*UnitMult
#macro InsertHaze(Density,Samples,Scale)
	pigment{rgbt 1}
	hollow
	interior{
		media{
			scattering{2,1}
			density{bozo
				scale Scale
				warp{turbulence 1}
				density_map{
					[0 rgb 0.0001*0]
					[1 rgb 0.0001*Density]
				}
			}
      method 3 samples Samples variance 1.0/256 confidence 0.96
		}
	}
#end

#declare BlackPaintTexture =
	texture{
		pigment{rgb <0.046,0.0476,0.050>/2}
		normal{bumps 0.05*UnitMult scale 0.3*UnitMult rotate 45 }
		finish{diffuse 0.1 ambient 0 reflection{0.00001,0.001}}
	}

#declare BlackDrapeTexture =
	texture{
		pigment{rgb <0.02,0.02,0.022>}
    normal{quilted 0.01*UnitMult scale 0.1*UnitMult}
//		normal{granite 0.01 rotate 45*z scale 30*UnitMult }
		finish{diffuse 0.001 ambient 0 specular 0.33 reflection{0.001,0.01}}
	}

#declare CycTexture = 
  texture{
    pigment{rgb <0.71928,0.78470,0.78537>}
    normal{spotted 0.01*UnitMult scale 0.3*UnitMult}
    finish{diffuse 0.2 ambient 0 specular 0.33 metallic reflection{0.0001,0.01}}
  }

#declare ScrimTexture = 
  texture{
    pigment{checker rgbf <0.58173,0.57781,0.59287,0.9>, rgbf <0.58173,0.57781,0.59287,0.1> scale 0.1*UnitMult}
    normal{quilted 0.1*UnitMult scale 0.3*UnitMult}
    finish{diffuse 0.01 ambient 0 specular 0.33 reflection{0.0001,0.01}}
  }
// ==============================================================================
// Other Functions - END
// ==============================================================================

// ==============================================================================
// Ellipsoidals - BEGIN
// ==============================================================================
// Zoom param is 0 for narrow beam, 1 for wide beam

// ------------------------------------------------------------------------------
// Altman
// ------------------------------------------------------------------------------
#macro A360Q_4p5x6p5(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 55;
	#local BeamAngle = FieldAngle*Sharpness; //22
	#local Cd = 52425*Intensity;
	LightingInstrument(Location,PointAt)
	HP600_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro A360Q_6x9(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 37;
	#local BeamAngle = FieldAngle*Sharpness; //16
	#local Cd = 83700*Intensity;
	LightingInstrument(Location,PointAt)
	HP600_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro A360Q_6x12(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 26;
	#local BeamAngle = FieldAngle*Sharpness; //11
	#local Cd = 164925*Intensity;
	LightingInstrument(Location,PointAt)
	HP600_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro A360Q_6x16(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 19;
	#local BeamAngle = FieldAngle*Sharpness; //8.5
	#local Cd = 231750*Intensity;
	LightingInstrument(Location,PointAt)
	HP600_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro A360Q_6x22(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 11;
	#local BeamAngle = FieldAngle*Sharpness; //8
	#local Cd = 238500*Intensity;
	LightingInstrument(Location,PointAt)
	HP600_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro Shakespeare_S65(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 7;
	#local BeamAngle = FieldAngle*Sharpness; //5
	#local Cd = 974900*Intensity;
	LightingInstrument(Location,PointAt)
	HP600_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro Shakespeare_S610(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 10;
	#local BeamAngle = FieldAngle*Sharpness; //7
	#local Cd = 791000*Intensity;
	LightingInstrument(Location,PointAt)
	HP600_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro Shakespeare_S612(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 12;
	#local BeamAngle = FieldAngle*Sharpness; //7
	#local Cd = 327700*Intensity;
	LightingInstrument(Location,PointAt)
	HP600_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro Shakespeare_S620(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 20;
	#local BeamAngle = FieldAngle*Sharpness; //13
	#local Cd = 189700*Intensity;
	LightingInstrument(Location,PointAt)
	HP600_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro Shakespeare_S630(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 28;
	#local BeamAngle = FieldAngle*Sharpness; //13
	#local Cd = 149200*Intensity;
	LightingInstrument(Location,PointAt)
	HP600_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro Shakespeare_S640(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 38;
	#local BeamAngle = FieldAngle*Sharpness; //20
	#local Cd = 98100*Intensity;
	LightingInstrument(Location,PointAt)
	HP600_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro Shakespeare_S650(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 50;
	#local BeamAngle = FieldAngle*Sharpness; //23
	#local Cd = 46300*Intensity;
	LightingInstrument(Location,PointAt)
	HP600_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro Shakespeare_S61535Z(Location,PointAt,Intensity,Sharpness,Zoom)
	#local FieldAngleNarrow = 15;
	#local FieldAngleWide = 35;
	#local FieldAngle = (FieldAngleWide-FieldAngleNarrow)*Zoom+FieldAngleNarrow;
	#local BeamAngle = FieldAngle*Sharpness;
	#local CdNarrow = 351000;
	#local CdWide = 190000;
	#local Cd = exp((ln(CdNarrow)-ln(CdWide))*(1-Zoom)+ln(CdWide))*Intensity;
	LightingInstrument(Location,PointAt)
	HP600_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro Shakespeare_S63055Z(Location,PointAt,Intensity,Sharpness,Zoom)
	#local FieldAngleNarrow = 30;
	#local FieldAngleWide = 55;
	#local FieldAngle = (FieldAngleWide-FieldAngleNarrow)*Zoom+FieldAngleNarrow;
	#local BeamAngle = FieldAngle*Sharpness;
	#local CdNarrow = 166000;
	#local CdWide = 66000;
	#local Cd = exp((ln(CdNarrow)-ln(CdWide))*(1-Zoom)+ln(CdWide))*Intensity;
	LightingInstrument(Location,PointAt)
	HP600_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

// ------------------------------------------------------------------------------
// Electronic Theatre Controls (ETC)
// ------------------------------------------------------------------------------
#macro SourceFour_05(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 7;
	#local BeamAngle = FieldAngle*Sharpness; //6
	#local Cd = 1345250*Intensity;
//	#local Lm = 9370*Intensity;
	LightingInstrument(Location,PointAt)
	HPL750_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro SourceFour_10(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 11;
	#local BeamAngle = FieldAngle*Sharpness; //8
	#local Cd = 783310*Intensity;
//	#local Lm = 11925*Intensity;
	LightingInstrument(Location,PointAt)
	HPL750_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro SourceFour_14(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 15;
	#local BeamAngle = FieldAngle*Sharpness; //14
	#local Cd = 404780*Intensity;
//	#local Lm = 12850*Intensity;
	LightingInstrument(Location,PointAt)
	HPL750_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro SourceFour_19(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 18;
	#local BeamAngle = FieldAngle*Sharpness; //15
	#local Cd = 243520*Intensity;
//	#local Lm = 11180*Intensity;
	LightingInstrument(Location,PointAt)
	HPL750_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro SourceFour_26(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 25;
	#local BeamAngle = FieldAngle*Sharpness; //18
	#local Cd = 176255*Intensity;
//	#local Lm = 13690*Intensity;
	LightingInstrument(Location,PointAt)
	HPL750_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro SourceFour_36(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 34;
	#local BeamAngle = FieldAngle*Sharpness; //27
	#local Cd = 90885*Intensity;
//	#local Lm = 14240*Intensity;
	LightingInstrument(Location,PointAt)
	HPL750_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro SourceFour_50(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 50;
	#local BeamAngle = FieldAngle*Sharpness; //36
	#local Cd = 45640*Intensity;
//	#local Lm = 13980*Intensity;
	LightingInstrument(Location,PointAt)
	HPL750_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro SourceFour_70(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 70;
	#local BeamAngle = FieldAngle*Sharpness; //60
	#local Cd = 22270*Intensity;
//	#local Lm = 16130*Intensity;
	LightingInstrument(Location,PointAt)
	HPL750_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro SourceFour_90(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 87;
	#local BeamAngle = FieldAngle*Sharpness; //79
	#local Cd = 11330*Intensity;
//	#local Lm = 13205*Intensity;
	LightingInstrument(Location,PointAt)
	HPL750_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro SourceFour_1530Z(Location,PointAt,Intensity,Sharpness,Zoom)
	#local FieldAngleNarrow = 15;
	#local FieldAngleWide = 30;
	#local FieldAngle = (FieldAngleWide-FieldAngleNarrow)*Zoom+FieldAngleNarrow;
	#local BeamAngle = FieldAngle*Sharpness;
	#local CdNarrow = 355000;
	#local CdWide = 110000;
	#local Cd = exp((ln(CdNarrow)-ln(CdWide))*(1-Zoom)+ln(CdWide))*Intensity;
	LightingInstrument(Location,PointAt)
	HPL750_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro SourceFour_2550Z(Location,PointAt,Intensity,Sharpness,Zoom)
	#local FieldAngleNarrow = 25;
	#local FieldAngleWide = 50;
	#local FieldAngle = (FieldAngleWide-FieldAngleNarrow)*Zoom+FieldAngleNarrow;
	#local BeamAngle = FieldAngle*Sharpness;
	#local CdNarrow = 155000;
	#local CdWide = 59500;
	#local Cd = exp((ln(CdNarrow)-ln(CdWide))*(1-Zoom)+ln(CdWide))*Intensity;
	LightingInstrument(Location,PointAt)
	HPL750_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

// ------------------------------------------------------------------------------
// Strand Lighting
// ------------------------------------------------------------------------------

#macro Coolbeam_SL5(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 5.14;
	#local BeamAngle = FieldAngle*Sharpness;
	#local Cd = 2000000*Intensity;
//	#local Lm = CandelaToLumens(Cd);
	LightingInstrument(Location,PointAt)
	GLC575_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro Coolbeam_SL10(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 9.65;
	#local BeamAngle = FieldAngle*Sharpness;
	#local Cd = 660000*Intensity;
//	#local Lm = CandelaToLumens(Cd);
	LightingInstrument(Location,PointAt)
	GLC575_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro Coolbeam_SL19(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 17.8;
	#local BeamAngle = FieldAngle*Sharpness;
	#local Cd = 200000*Intensity;
//	#local Lm = CandelaToLumens(Cd);
	LightingInstrument(Location,PointAt)
	GLC575_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro Coolbeam_SL26(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 24.7;
	#local BeamAngle = FieldAngle*Sharpness;
	#local Cd = 117000*Intensity;
//	#local Lm = CandelaToLumens(Cd);
	LightingInstrument(Location,PointAt)
	GLC575_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro Coolbeam_SL36(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 32.6;
	#local BeamAngle = FieldAngle*Sharpness;
	#local Cd = 66000*Intensity;
//	#local Lm = CandelaToLumens(Cd);
	LightingInstrument(Location,PointAt)
	GLC575_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro Coolbeam_SL50(Location,PointAt,Intensity,Sharpness)
	#local FieldAngle = 42.9;
	#local BeamAngle = FieldAngle*Sharpness;
	#local Cd = 34000*Intensity;
//	#local Lm = CandelaToLumens(Cd);
	LightingInstrument(Location,PointAt)
	GLC575_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro Coolbeam_SL1532Z(Location,PointAt,Intensity,Sharpness,Zoom)
	#local FieldAngleNarrow = 15;
	#local FieldAngleWide = 32;
	#local FieldAngle = (FieldAngleWide-FieldAngleNarrow)*Zoom+FieldAngleNarrow;
	#local BeamAngle = FieldAngle*Sharpness;
	#local CdNarrow = 237000;
	#local CdWide = 126000;
	#local Cd = exp((ln(CdNarrow)-ln(CdWide))*(1-Zoom)+ln(CdWide))*Intensity;
	LightingInstrument(Location,PointAt)
	GLC575_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

#macro Coolbeam_SL2350Z(Location,PointAt,Intensity,Sharpness,Zoom)
	#local FieldAngleNarrow = 23;
	#local FieldAngleWide = 50;
	#local FieldAngle = (FieldAngleWide-FieldAngleNarrow)*Zoom+FieldAngleNarrow;
	#local BeamAngle = FieldAngle*Sharpness;
	#local CdNarrow = 151000;
	#local CdWide = 43000;
	#local Cd = exp((ln(CdNarrow)-ln(CdWide))*(1-Zoom)+ln(CdWide))*Intensity;
	LightingInstrument(Location,PointAt)
	GLC575_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
#end

// ===========================================================
// Ellipsoidals - END
// ===========================================================

// ===========================================================
// Bulbs - BEGIN
// ===========================================================

// ------------------------------------------------------------------------------
// MR-16
// ------------------------------------------------------------------------------

// 75w 14degree
#macro MR16_EYF(Location,PointAt,Intensity)
  #local BeamAngle = 14;
  #local FieldAngle = 30;
  #local Candelas = 11000;
  #local ColorTemp = 3050;
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight radius BeamAngle/2 falloff FieldAngle/2 point_at PointAt
    #if(Use_Fading)
      Light_Fading()
    #end
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

// 75w 25degree
#macro MR16_EYJ(Location,PointAt,Intensity)
  #local BeamAngle = 25;
  #local FieldAngle = 40;
  #local Candelas = 4500;
  #local ColorTemp = 3050;
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight radius BeamAngle/2 falloff FieldAngle/2 point_at PointAt
    #if(Use_Fading)
      Light_Fading()
    #end
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

// 75w 40degree
#macro MR16_EYC(Location,PointAt,Intensity)
  #local BeamAngle = 40;
  #local FieldAngle = 51;
  #local Candelas = 2000;
  #local ColorTemp = 3050;
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight radius BeamAngle/2 falloff FieldAngle/2 point_at PointAt
    #if(Use_Fading)
      Light_Fading()
    #end
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

// 65w 14degree
#macro MR16_FPA(Location,PointAt,Intensity)
  #local BeamAngle = 14;
  #local FieldAngle = 30;
  #local Candelas = 4500;
  #local ColorTemp = 3000;
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight radius BeamAngle/2 falloff FieldAngle/2 point_at PointAt
    #if(Use_Fading)
      Light_Fading()
    #end
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

// 65w 27degree
#macro MR16_FPC(Location,PointAt,Intensity)
  #local BeamAngle = 27;
  #local FieldAngle = 40;
  #local Candelas = 4500;
  #local ColorTemp = 3000;
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight radius BeamAngle/2 falloff FieldAngle/2 point_at PointAt
    #if(Use_Fading)
      Light_Fading()
    #end
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

// 65w 27degree
#macro MR16_FPB(Location,PointAt,Intensity)
  #local BeamAngle = 38;
  #local FieldAngle = 51;
  #local Candelas = 2000;
  #local ColorTemp = 3000;
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight radius BeamAngle/2 falloff FieldAngle/2 point_at PointAt
    #if(Use_Fading)
      Light_Fading()
    #end
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

// ------------------------------------------------------------------------------
// PAR-38
// ------------------------------------------------------------------------------

#macro PAR38_250SP(Location,PointAt,Intensity)
  #local BeamAngle = 10;
  #local FieldAngle = 20;
  #local Candelas = 46500;
  #local ColorTemp = 2950;
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight radius BeamAngle/2 falloff FieldAngle/2 point_at PointAt
    #if(Use_Fading)
      Light_Fading()
    #end
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

#macro PAR38_250FL(Location,PointAt,Intensity)
  #local BeamAngle = 30;
  #local FieldAngle = 60;
  #local Candelas = 9000;
  #local ColorTemp = 3025;
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight radius BeamAngle/2 falloff FieldAngle/2 point_at PointAt
    #if(Use_Fading)
      Light_Fading()
    #end
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

#macro PAR38_120SP(Location,PointAt,Intensity)
  #local BeamAngle = 10;
  #local FieldAngle = 20;
  #local Candelas = 22500;
  #local ColorTemp = 2950;
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight radius BeamAngle/2 falloff FieldAngle/2 point_at PointAt
    #if(Use_Fading)
      Light_Fading()
    #end
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

#macro PAR38_120FL(Location,PointAt,Intensity)
  #local BeamAngle = 30;
  #local FieldAngle = 60;
  #local Candelas = 4600;
  #local ColorTemp = 2950;
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight radius BeamAngle/2 falloff FieldAngle/2 point_at PointAt
    #if(Use_Fading)
      Light_Fading()
    #end
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

// ------------------------------------------------------------------------------
// PAR-64
// ------------------------------------------------------------------------------

// FFN-VNSP
#macro Q1000PAR64_1(Location,PointAt,Intensity,LongAxis)
  #local BeamAngle = array[2] {12,6}
  #local FieldAngle = array[2] {24,10}
  #local Candelas = 400000;
  #local ColorTemp = 3200;
  #local pointVect = vnormalize(PointAt-Location);
  #local LongAxis = vnormalize(vcross(vcross(pointVect,LongAxis),pointVect));
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight
    point_at PointAt
    radius 45 falloff 45
    #if(Use_Fading)
      Light_Fading()
    #end
    CreatePAROval(BeamAngle,FieldAngle,Location,PointAt,LongAxis)
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

// FFP-NSP
#macro Q1000PAR64_2(Location,PointAt,Intensity,LongAxis)
  #local BeamAngle = array[2] {14,7}
  #local FieldAngle = array[2] {26,14}
  #local Candelas = 330000;
  #local ColorTemp = 3200;
  #local pointVect = vnormalize(PointAt-Location);
  #local LongAxis = vnormalize(vcross(vcross(pointVect,LongAxis),pointVect));
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight
    point_at PointAt
    radius 45 falloff 45
    #if(Use_Fading)
      Light_Fading()
    #end
    CreatePAROval(BeamAngle,FieldAngle,Location,PointAt,LongAxis)
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

// FFR-MFL
#macro Q1000PAR64_5(Location,PointAt,Intensity,LongAxis)
  #local BeamAngle = array[2] {28,12}
  #local FieldAngle = array[2] {44,21}
  #local Candelas = 125000;
  #local ColorTemp = 3200;
  #local pointVect = vnormalize(PointAt-Location);
  #local LongAxis = vnormalize(vcross(vcross(pointVect,LongAxis),pointVect));
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight
    point_at PointAt
    radius 45 falloff 45
    #if(Use_Fading)
      Light_Fading()
    #end
    CreatePAROval(BeamAngle,FieldAngle,Location,PointAt,LongAxis)
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

// FFS-WFL
#macro Q1000PAR64_6(Location,PointAt,Intensity,LongAxis)
  #local BeamAngle = array[2] {48,24}
  #local FieldAngle = array[2] {71,45}
  #local Candelas = 40000;
  #local ColorTemp = 3200;
  #local pointVect = vnormalize(PointAt-Location);
  #local LongAxis = vnormalize(vcross(vcross(pointVect,LongAxis),pointVect));
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight
    point_at PointAt
    radius 45 falloff 45
    #if(Use_Fading)
      Light_Fading()
    #end
    CreatePAROval(BeamAngle,FieldAngle,Location,PointAt,LongAxis)
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

// FGM-NSP
#macro Q1000PAR64_3D(Location,PointAt,Intensity,LongAxis)
  #local BeamAngle = array[2] {13,6}
  #local FieldAngle = array[2] {24,12}
  #local Candelas = 200000;
  #local ColorTemp = 5200;
  #local pointVect = vnormalize(PointAt-Location);
  #local LongAxis = vnormalize(vcross(vcross(pointVect,LongAxis),pointVect));
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight
    point_at PointAt
    radius 45 falloff 45
    #if(Use_Fading)
      Light_Fading()
    #end
    CreatePAROval(BeamAngle,FieldAngle,Location,PointAt,LongAxis)
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

// FGN-MFL
#macro Q1000PAR64_7D(Location,PointAt,Intensity,LongAxis)
  #local BeamAngle = array[2] {27,11}
  #local FieldAngle = array[2] {43,20}
  #local Candelas = 70000;
  #local ColorTemp = 5200;
  #local pointVect = vnormalize(PointAt-Location);
  #local LongAxis = vnormalize(vcross(vcross(pointVect,LongAxis),pointVect));
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight
    point_at PointAt
    radius 45 falloff 45
    #if(Use_Fading)
      Light_Fading()
    #end
    CreatePAROval(BeamAngle,FieldAngle,Location,PointAt,LongAxis)
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

// PAR64-NSP
#macro Q1000PAR64_NSP(Location,PointAt,Intensity,LongAxis)
  #local BeamAngle = array[2] {15,8}
  #local FieldAngle = array[2] {31,14}
  #local Candelas = 200000;
  #local ColorTemp = 2950;
  #local pointVect = vnormalize(PointAt-Location);
  #local LongAxis = vnormalize(vcross(vcross(pointVect,LongAxis),pointVect));
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight
    point_at PointAt
    radius 45 falloff 45
    #if(Use_Fading)
      Light_Fading()
    #end
    CreatePAROval(BeamAngle,FieldAngle,Location,PointAt,LongAxis)
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

// PAR64-MFL
#macro Q1000PAR64_MFL(Location,PointAt,Intensity,LongAxis)
  #local BeamAngle = array[2] {28,12}
  #local FieldAngle = array[2] {45,22}
  #local Candelas = 80000;
  #local ColorTemp = 2950;
  #local pointVect = vnormalize(PointAt-Location);
  #local LongAxis = vnormalize(vcross(vcross(pointVect,LongAxis),pointVect));
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight
    point_at PointAt
    radius 45 falloff 45
    #if(Use_Fading)
      Light_Fading()
    #end
    CreatePAROval(BeamAngle,FieldAngle,Location,PointAt,LongAxis)
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

// PAR64-WFL
#macro Q1000PAR64_WFL(Location,PointAt,Intensity,LongAxis)
  #local BeamAngle = array[2] {48,24}
  #local FieldAngle = array[2] {72,45}
  #local Candelas = 33000;
  #local ColorTemp = 2950;
  #local pointVect = vnormalize(PointAt-Location);
  #local LongAxis = vnormalize(vcross(vcross(pointVect,LongAxis),pointVect));
  light_source{ Location Light_Color(Blackbody(ColorTemp),Candelas*Intensity)
    spotlight
    point_at PointAt
    radius 45 falloff 45
    #if(Use_Fading)
      Light_Fading()
    #end
    CreatePAROval(BeamAngle,FieldAngle,Location,PointAt,LongAxis)
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
  }
#end

// ===========================================================
// Bulbs - END
// ===========================================================

// ===========================================================
// Striplights - BEGIN
// ===========================================================

#macro PAR38_6ft_3ckt(Center,Axis,PointAt,Intensity,LampName,Color)
  ParseLiteral(LampName)
  #local numLamps = 12;
  #local numCircuits = 3;
  #local instrumentLength = 6*12*2.54*UnitMult;
  #local dbl=6*2.54*UnitMult; // distance between lamps is 6"
  #local axis=vnormalize(Axis);
  #local pointat=vnormalize(vcross(vcross(Axis,PointAt-Center),Axis));
  #local unit=1;
  #while (unit <= numLamps)
    #local currentPoint = Center+(axis*((unit-(numLamps/2))-0.5)*dbl);
    ParsedLiteral()(currentPoint,currentPoint+pointat,Intensity[mod(unit,numCircuits)])
		cylinder{currentPoint,currentPoint+pointat*0.001*UnitMult,dbl/2
			texture{pigment{rgb ReflectiveSpectrum(Color[mod(unit,numCircuits)]) filter 1}}
			translate pointat*GlobalGelDistance/2
		}
    #local unit = unit + 1;
  #end
  StriplightInstrument(Center,axis,pointat,instrumentLength,dbl)
#end

#macro PAR38_6ft_4ckt(Center,Axis,PointAt,Intensity,LampName,Color)
  ParseLiteral(LampName)
  #local numLamps = 12;
  #local numCircuits = 4;
  #local instrumentLength = 6*12*2.54*UnitMult;
  #local dbl=6*2.54*UnitMult; // distance between lamps is 6"
  #local axis=vnormalize(Axis);
  #local pointat=vnormalize(vcross(vcross(Axis,PointAt-Center),Axis));
  #local unit=1;
  #while (unit <= numLamps)
    #local currentPoint = Center+(axis*((unit-(numLamps/2))-0.5)*dbl);
    ParsedLiteral()(currentPoint,currentPoint+pointat,Intensity[mod(unit,numCircuits)])
		cylinder{currentPoint,currentPoint+pointat*0.001*UnitMult,dbl/2
			texture{pigment{rgb ReflectiveSpectrum(Color[mod(unit,numCircuits)]) filter 1}}
			translate pointat*GlobalGelDistance/2
		}
    #local unit = unit + 1;
  #end
  StriplightInstrument(Center,axis,pointat,instrumentLength,dbl)
#end

#macro PAR38_7ft6in_3ckt(Center,Axis,PointAt,Intensity,LampName,Color)
  ParseLiteral(LampName)
  #local numLamps = 15;
  #local numCircuits = 3;
  #local instrumentLength = 7.5*12*2.54*UnitMult;
  #local dbl=6*2.54*UnitMult; // distance between lamps is 6"
  #local axis=vnormalize(Axis);
  #local pointat=vnormalize(vcross(vcross(Axis,PointAt-Center),Axis));
  #local unit=1;
  #while (unit <= numLamps)
    #local currentPoint = Center+(axis*((unit-(numLamps/2))-0.5)*dbl);
    ParsedLiteral()(currentPoint,currentPoint+pointat,Intensity[mod(unit,numCircuits)])
		cylinder{currentPoint,currentPoint+pointat*0.001*UnitMult,dbl/2
			texture{pigment{rgb ReflectiveSpectrum(Color[mod(unit,numCircuits)]) filter 1}}
			translate pointat*GlobalGelDistance/2
		}
    #local unit = unit + 1;
  #end
  StriplightInstrument(Center,axis,pointat,instrumentLength,dbl)
#end

#macro PAR38_8ft_4ckt(Center,Axis,PointAt,Intensity,LampName,Color)
  ParseLiteral(LampName)
  #local numLamps = 16;
  #local numCircuits = 4;
  #local instrumentLength = 8*12*2.54*UnitMult;
  #local dbl=6*2.54*UnitMult; // distance between lamps is 6"
  #local axis=vnormalize(Axis);
  #local pointat=vnormalize(vcross(vcross(Axis,PointAt-Center),Axis));
  #local unit=1;
  #while (unit <= numLamps)
    #local currentPoint = Center+(axis*((unit-(numLamps/2))-0.5)*dbl);
    ParsedLiteral()(currentPoint,currentPoint+pointat,Intensity[mod(unit,numCircuits)])
		cylinder{currentPoint,currentPoint+pointat*0.001*UnitMult,dbl/2
			texture{pigment{rgb ReflectiveSpectrum(Color[mod(unit,numCircuits)]) filter 1}}
			translate pointat*GlobalGelDistance/2
		}
    #local unit = unit + 1;
  #end
  StriplightInstrument(Center,axis,pointat,instrumentLength,dbl)
#end

#macro MR16_ZS_1(Center,Axis,PointAt,Intensity,LampName,Color)
  ParseLiteral(LampName)
  #local numLamps = 10;
  #local numCircuits = 1;
  #local instrumentLength = (2*12+2)*2.54*UnitMult;
  #local instrumentHeight = 4.75*2.54*UnitMult;
  #local dbs=instrumentLength/(numLamps/2);
  #local dbl=1.5*2.54*UnitMult; // distance from section center to each bulb
  #local axis=vnormalize(Axis);
  #local pointat=vnormalize(vcross(vcross(Axis,PointAt-Center),Axis));
  #local unit=1;
  #while (unit <= numLamps/2)
    #local currentPoint = Center+(axis*((unit-(numLamps/4))-0.5)*dbs);
    ParsedLiteral()(currentPoint+axis*dbl,currentPoint+axis*dbl+pointat,Intensity[mod(unit,numCircuits)])
    ParsedLiteral()(currentPoint-axis*dbl,currentPoint-axis*dbl+pointat,Intensity[mod(unit,numCircuits)])
		box{<0,-instrumentHeight/2,-dbs/2>,<0.001*UnitMult,instrumentHeight/2,dbs/2>
			texture{pigment{rgb ReflectiveSpectrum(Color[mod(unit,numCircuits)]) filter 1}}
      Orient_To_Trans(currentPoint,currentPoint+pointat)
			translate currentPoint+pointat*GlobalGelDistance/2
		}
    #local unit = unit + 1;
  #end
  StriplightInstrument(Center,axis,pointat,instrumentLength,instrumentHeight)
#end

#macro MR16_ZS_2(Center,Axis,PointAt,Intensity,LampName,Color)
  ParseLiteral(LampName)
  #local numLamps = 20;
  #local numCircuits = 2;
  #local instrumentLength = (4*12+3)*2.54*UnitMult;
  #local instrumentHeight = 4.75*2.54*UnitMult;
  #local dbs=instrumentLength/(numLamps/2);
  #local dbl=1.5*2.54*UnitMult; // distance from section center to each bulb
  #local axis=vnormalize(Axis);
  #local pointat=vnormalize(vcross(vcross(Axis,PointAt-Center),Axis));
  #local unit=1;
  #while (unit <= numLamps/2)
    #local currentPoint = Center+(axis*((unit-(numLamps/4))-0.5)*dbs);
    ParsedLiteral()(currentPoint+axis*dbl,currentPoint+axis*dbl+pointat,Intensity[mod(unit,numCircuits)])
    ParsedLiteral()(currentPoint-axis*dbl,currentPoint-axis*dbl+pointat,Intensity[mod(unit,numCircuits)])
		box{<0,-instrumentHeight/2,-dbs/2>,<0.001*UnitMult,instrumentHeight/2,dbs/2>
			texture{pigment{rgb ReflectiveSpectrum(Color[mod(unit,numCircuits)]) filter 1}}
      Orient_To_Trans(currentPoint,currentPoint+pointat)
			translate currentPoint+pointat*GlobalGelDistance/2
		}
    #local unit = unit + 1;
  #end
  StriplightInstrument(Center,axis,pointat,instrumentLength,instrumentHeight)
#end

#macro MR16_ZS_3(Center,Axis,PointAt,Intensity,LampName,Color)
  ParseLiteral(LampName)
  #local numLamps = 30;
  #local numCircuits = 3;
  #local instrumentLength = (6*12+4)*2.54*UnitMult;
  #local instrumentHeight = 4.75*2.54*UnitMult;
  #local dbs=instrumentLength/(numLamps/2);
  #local dbl=1.5*2.54*UnitMult; // distance from section center to each bulb
  #local axis=vnormalize(Axis);
  #local pointat=vnormalize(vcross(vcross(Axis,PointAt-Center),Axis));
  #local unit=1;
  #while (unit <= numLamps/2)
    #local currentPoint = Center+(axis*((unit-(numLamps/4))-0.5)*dbs);
    ParsedLiteral()(currentPoint+axis*dbl,currentPoint+axis*dbl+pointat,Intensity[mod(unit,numCircuits)])
    ParsedLiteral()(currentPoint-axis*dbl,currentPoint-axis*dbl+pointat,Intensity[mod(unit,numCircuits)])
		box{<0,-instrumentHeight/2,-dbs/2>,<0.001*UnitMult,instrumentHeight/2,dbs/2>
			texture{pigment{rgb ReflectiveSpectrum(Color[mod(unit,numCircuits)]) filter 1}}
      Orient_To_Trans(currentPoint,currentPoint+pointat)
			translate currentPoint+pointat*GlobalGelDistance/2
		}
    #local unit = unit + 1;
  #end
  StriplightInstrument(Center,axis,pointat,instrumentLength,instrumentHeight)
#end

#macro MR16_ZS_4(Center,Axis,PointAt,Intensity,LampName,Color)
  ParseLiteral(LampName)
  #local numLamps = 40;
  #local numCircuits = 4;
  #local instrumentLength = (8*12+5)*2.54*UnitMult;
  #local instrumentHeight = 4.75*2.54*UnitMult;
  #local dbs=instrumentLength/(numLamps/2);
  #local dbl=1.5*2.54*UnitMult; // distance from section center to each bulb
  #local axis=vnormalize(Axis);
  #local pointat=vnormalize(vcross(vcross(Axis,PointAt-Center),Axis));
  #local unit=1;
  #while (unit <= numLamps/2)
    #local currentPoint = Center+(axis*((unit-(numLamps/4))-0.5)*dbs);
    ParsedLiteral()(currentPoint+axis*dbl,currentPoint+axis*dbl+pointat,Intensity[mod(unit,numCircuits)])
    ParsedLiteral()(currentPoint-axis*dbl,currentPoint-axis*dbl+pointat,Intensity[mod(unit,numCircuits)])
		box{<0,-instrumentHeight/2,-dbs/2>,<0.001*UnitMult,instrumentHeight/2,dbs/2>
			texture{pigment{rgb ReflectiveSpectrum(Color[mod(unit,numCircuits)]) filter 1}}
      Orient_To_Trans(currentPoint,currentPoint+pointat)
			translate currentPoint+pointat*GlobalGelDistance/2
		}
    #local unit = unit + 1;
  #end
  StriplightInstrument(Center,axis,pointat,instrumentLength,instrumentHeight)
#end

// ===========================================================
// Striplights - END
// ===========================================================

// ===========================================================
// Other Instruments - BEGIN
// ===========================================================

#macro PAR64Can(Location,PointAt,Intensity,LongAxis,LampName)
  ParseLiteral(LampName)
  ParsedLiteral()(Location,PointAt,Intensity,LongAxis)
	LightingInstrument(Location,PointAt)
#end

// ===========================================================
// Other Instruments - BEGIN
// ===========================================================

// ===========================================================
// Internal Functions - BEGIN
// ===========================================================

#macro IncrementGoboPlacement()
  #declare GoboDistance = GoboDistance + GlobalGoboThickness*2;
  #declare GoboSlot = transform{translate GlobalGoboThickness*x*2 scale <1,1+GlobalGoboThickness,1+GlobalGoboThickness> GoboSlot}
#end
#macro IncrementGelPlacement()
  #declare GelDistance = GelDistance + GlobalGelThickness*2;
  #declare GelSlot = transform{translate GlobalGelThickness*x*2 GelSlot}
#end

// Transforms calling object to be at Location, looking at PointAt,
// with +y as Up by default, or if vertical +z is Up
#macro Orient_To_Trans(Location,PointAt)
  #local yworks=vcross(y,PointAt-Location);
  #if(vlength(yworks)=0)
    #local yUp1=<0,0,1>;
  #else
    #local yUp1=<0,1,0>;
  #end
  #local yUp=vnormalize(vcross(vcross(PointAt-Location,yUp1),PointAt-Location));
  #local xAt=vnormalize(PointAt-Location);
  #local zNew=vnormalize(vcross(xAt,yUp));
  #if(DEBUG)
    #debug concat("yUp <",
      str(yUp.x,0,2),",",str(yUp.y,0,2),",",str(yUp.z,0,2),">, xAt <",
      str(xAt.x,0,2),",",str(xAt.y,0,2),",",str(xAt.z,0,2),">, zNew <",
      str(zNew.x,0,2),",",str(zNew.y,0,2),",",str(zNew.z,0,2),">\n")
  #end
  Shear_Trans(xAt,yUp,zNew)
#end

// Defines transformation to place Gobo or accessory
#macro GoboLocation(FieldAngle,Location,PointAt)
	#declare GoboSlot = transform{
		translate x*GoboDistance
		#local s = (2)*GoboDistance*tan(radians(FieldAngle/2));
		scale <1,s,s>
		Orient_To_Trans(Location,PointAt)
		translate Location
	}
#end

// Defines transformation to place Gel
#macro GelLocation(Location,PointAt)
	#declare GelSlot = transform{
		translate x*GelDistance
		Orient_To_Trans(Location,PointAt)
		translate Location
	}
#end

// thanks to clipka for this macro:
// Usage - ParseLiteral("macro_name") ... ParsedLiteral()(macro arguments)
// Each call to ParseLiteral(s) will overwrite the macro referred to by ParsedLiteral()
#macro ParseLiteral(s)
  #declare ParsedLiteralFile = "ParseLiteral_$$$.inc";
  #fopen ParseLiteral_FD ParsedLiteralFile write
  #write (ParseLiteral_FD,s)
  #fclose ParseLiteral_FD
#end

#macro ParsedLiteral()
  #include ParsedLiteralFile
#end

// Creates the oval beam for PAR 64 Lamps
#macro CreatePAROval(BeamAngle,FieldAngle,Location,PointAt,LongAxis)
  projected_through{cylinder{0,-0.001*UnitMult*y,1}
    scale <DistanceToPAROval*tan(BeamAngle[0]/2*pi/180),1,DistanceToPAROval*tan(BeamAngle[1]/2*pi/180)>
  #local pointVect = vnormalize(PointAt-Location);
  #local LongAxis = vnormalize(vcross(vcross(pointVect,LongAxis),pointVect));
  #local zVect = vnormalize(vcross(LongAxis,PointAt-Location));
//  #debug concat("pointVect = <",str(pointVect.x,0,3),",",str(pointVect.y,0,3),",",str(pointVect.z,0,3),">\n")
//  #debug concat("LongAxis = <",str(LongAxis.x,0,3),",",str(LongAxis.y,0,3),",",str(LongAxis.z,0,3),">\n")
//  #debug concat("zVect = <",str(zVect.x,0,3),",",str(zVect.y,0,3),",",str(zVect.z,0,3),">\n")
    Shear_Trans(LongAxis,pointVect,zVect)
    translate DistanceToPAROval*pointVect
    translate Location
  }
  #local multx = FieldAngle[0]/BeamAngle[0]*DistanceToPAROval*tan(BeamAngle[0]/2*pi/180)+DistanceToPAROval*tan(BeamAngle[0]/2*pi/180);
  #local multz = FieldAngle[1]/BeamAngle[1]*DistanceToPAROval*tan(BeamAngle[1]/2*pi/180)+DistanceToPAROval*tan(BeamAngle[1]/2*pi/180);
  area_light LongAxis*multx*.9,vnormalize(vcross(LongAxis,pointVect))*multz*.9,25,25 adaptive 5 circular
#end

// Internal function
#macro HP600_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
	#local mult = 0.01*UnitMult*Gobo_Blur_Amount;
	light_source{
		Location Light_Color(Blackbody(3200),Cd)
		spotlight radius BeamAngle/2 falloff FieldAngle/2 point_at PointAt
		#if(Use_Fading)
			Light_Fading()
		#end
		#if(Sharpness<1 & Use_Soft_Shadows = 1)
			area_light
			y*pow((1-Sharpness),2)*mult,z*pow((1-Sharpness),2)*mult,6,6
			adaptive 2
			circular orient
		#end
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
	}
#end

// Internal function
#macro HPL750_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
	#local mult = 0.01*UnitMult*Gobo_Blur_Amount;
	light_source{
		Location Light_Color(Blackbody(3250),Cd)
		spotlight radius BeamAngle/2 falloff FieldAngle/2 point_at PointAt
		#if(Use_Fading)
			Light_Fading()
		#end
		#if(Sharpness<1 & Use_Soft_Shadows = 1)
			area_light
			y*pow((1-Sharpness),2)*mult,z*pow((1-Sharpness),2)*mult,6,6
			adaptive 2
			circular orient
		#end
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
    looks_like{sphere{0,5*UnitMult pigment{color rgb 0.5}}}
	}
#end

// Internal function
#macro GLC575_Lamp(Location,PointAt,Cd,BeamAngle,FieldAngle,Sharpness)
	#local mult = 0.01*UnitMult*Gobo_Blur_Amount;
	light_source{
		Location Light_Color(Blackbody(3200),Cd)
		spotlight radius BeamAngle/2 falloff FieldAngle/2 point_at PointAt
		#if(Use_Fading)
			Light_Fading()
		#end
		#if(Sharpness<1 & Use_Soft_Shadows = 1)
			area_light
			y*pow((1-Sharpness),2)*mult,z*pow((1-Sharpness),2)*mult,6,6
			adaptive 2
			circular orient
		#end
    #if(Use_Photons)
      photons{reflection on refraction on}
    #end
	}
#end

// Creates an instrument can and prepares transforms for accessories and gels
#macro LightingInstrument(Location,PointAt)
	#declare GoboDistance = GlobalGoboDistance;
	#declare GelDistance = GlobalGelDistance;
	GoboLocation(FieldAngle,Location,PointAt)
	GelLocation(Location,PointAt)
	object{
		difference{
			cylinder{<-8*UnitMult,0,0>,<GelDistance,0,0>,InstrumentRadius}
			cylinder{<-7.999*UnitMult,0,0>,<GelDistance+0.001*UnitMult,0,0>,InstrumentRadius*0.99}
			texture {pigment{color rgb 0.02} finish{metallic}}
		}
		Orient_To_Trans(Location,PointAt)
		translate Location
	}
#end

// Creates body for striplights
#macro StriplightInstrument(Center,Axis,PointAt,Length,Width)
  object{
    difference{
      box{<-Length/2,-4*UnitMult,-Width/2>,<+Length/2,GlobalGelDistance/2,Width/2>}
      box{<-(Length/2-0.1*UnitMult),-3.9*UnitMult,-(Width/2-0.1*UnitMult)>,<+(Length/2-0.1*UnitMult),GlobalGelDistance+1,(Width/2-0.1*UnitMult)>}
      texture {pigment{color rgb 0.02} finish{metallic}}
    }
    Shear_Trans(Axis,PointAt,vnormalize(vcross(Axis,PointAt)))
    translate Center
  }
#end

// Really quick and dirty hack, but comes closer to the field lumens vs. candela of
// all the Source4 units than the 'official' formula of lm=cd*2*pi*(1-cos(theta/2))
#macro CandelaToLumens(Cd,FieldAngle) // unused!
  #local Lm = Cd*(FieldAngle^2/90^2);
  (Lm)
#end

// ==============================================================================
// Internal Functions - END
// ==============================================================================

#version THEATRESYS_INC_TEMP;
#end
