/*
  Persistence of Vision Ray Tracer include file
  
  PowersPositions.inc
  - macros for lighting positions and soft goods placement in PowersTheatre
  
  Christopher Shake, March 2009
  
  For use with PowersTheatreSmooth.inc or PowersTheatreRough.inc, and needs
  TheatreSys already included.
  
  version 0.1 (2009.03.13)
  
  General Info:
    Powers has 5 on-stage electrics, with 1st being just upstage of the proscenium
      and 5th being near the back.
    The back wall is 25' or so upstage, and the stage is about 32' wide.
    The proscenium is 18' high.
  
  Contents:
    InsertPowersTheatre()
      - Creates the theatre itself.
    SetTrimHeight(ElecNum,NewHeight)
      - Changes hanging height of electric, used before inserting masking or lights
      - Optional, basic trim is already set for defaults.
    ElectricPosition(ElecNum,DistanceFromCenter)
      - Creates a vector that gives the position of an instrument.
      - DistanceFromCenter is positive going house right.
    DanceBoomPosition(ElecNum,Side,Height)
      - Creates a vector that gives the position of an instrument.
      - Side is +1 for house right, -1 for house left.
      - Height is height above the floor.
    InsertMasking(ElecNum,DistanceDownstage)
      - Creates legs and border 'DistanceDownstage' of a given electric.
    InsertFullBlack(DistanceUpstage)
      - Full width black curtain.
    InsertCyc(DistanceUpstage)
      - Full width white cyclorama.
    InsertScrim(DistanceUpstage)
      - Full width semi-transparent mesh screen.
    InsertTraveler(DistanceUpstage,Height,PercentOpen)
      - Full width black curtain that opens in the middle.
      - Height is how high to the top of the curtain.
      - PercentOpen is 0 for closed, 1 for fully open.
    SitAudienceCenter()
      - Places a camera where it can see the whole stage
    
  TODO:
    Hanging positions for house beams
*/

#ifndef(POWERSTHEATREPOSITIONS_INC_TEMP)
#declare POWERSTHEATREPOSITIONS_INC_TEMP = version;
#version 3.5;

#declare inFeet = 12*2.54*UnitMult;

#ifndef(InsideLegLocation)
	#declare InsideLegLocation = 16*inFeet;
#end
#ifndef(LegWidth)
	#declare LegWidth = 8*inFeet;
#end
#ifndef(LegHeight)
	#declare LegHeight = 22*inFeet;
#end
#ifndef(BorderHeight)
	#declare BorderHeight = 4*inFeet;
#end
#ifndef(BorderWidth)
	#declare BorderWidth = 40*inFeet;
#end
#ifndef(BorderOverlap) // distance below trim height of upstage electric
	#declare BorderOverlap = 1*inFeet;
#end

#declare trimHeights = array[5]
	{19*inFeet,19*inFeet,19*inFeet,19*inFeet,19*inFeet}
#declare electricPositions = array[5]
	{3.5*inFeet,8.2*inFeet,12*inFeet,16.8*inFeet,20.1*inFeet}

#macro SetTrimHeight(ElecNum,NewHeight)
	trimHeights[ElecNum-1] = NewHeight;
#end

#macro ElectricPosition(ElecNum,DistanceFromCenter)
  #if(ElecNum=0)
    #local vec = ProPipePosition(DistanceFromCenter);
  #else
    #local vec = <DistanceFromCenter,trimHeights[ElecNum-1],electricPositions[ElecNum-1]>;
  #end
  (vec)
#end

#macro ProPipePosition(DistanceFromCenter)
  <DistanceFromCenter,19*inFeet,-5*inFeet>
#end

// Side is -1 for house left, 1 for house right
#macro DanceBoomPosition(ElecNum,Side,Height)
	<23*inFeet*Side,Height,electricPositions[ElecNum-1]>
#end

// AmountDownstage is the distance downstage of given electric
#macro InsertMasking(ElecNum,DistanceDownstage)
	#local zlocation = electricPositions[ElecNum-1]-DistanceDownstage;
	#local Thickness = 0.001*UnitMult; //3 mm
	box{<-InsideLegLocation,0,zlocation>,<-(InsideLegLocation+LegWidth),LegHeight,zlocation+Thickness> texture{BlackDrapeTexture}}
	box{<InsideLegLocation,0,zlocation>,<(InsideLegLocation+LegWidth),LegHeight,zlocation+Thickness> texture{BlackDrapeTexture}}
	box{<-BorderWidth/2,trimHeights[ElecNum-1]-BorderOverlap,zlocation>,<BorderWidth/2,trimHeights[ElecNum-1]+BorderHeight-BorderOverlap,zlocation+Thickness> texture{BlackDrapeTexture}}
#end

#macro InsertPowersTheatre()
	object{PowersTheatre() texture{BlackPaintTexture}}
#end

// back wall is 25ft
#macro InsertFullBlack(DistanceUpstage)
	box{<-25*inFeet,0,DistanceUpstage>,<25*inFeet,25*inFeet,DistanceUpstage+0.3*UnitMult> texture{BlackDrapeTexture}}
#end

#macro InsertCyc(DistanceUpstage)
	box{<-25*inFeet,0,DistanceUpstage>,<25*inFeet,25*inFeet,DistanceUpstage+0.3*UnitMult> texture{CycTexture}}
#end

#macro InsertScrim(DistanceUpstage)
	box{<-25*inFeet,0,DistanceUpstage>,<25*inFeet,25*inFeet,DistanceUpstage+0.*UnitMult> texture{ScrimTexture} double_illuminate}
#end

// PercentOpen is 0 to 1, where 1 is open
#macro InsertTraveler(DistanceUpstage,Height,PercentOpen)
	#local Thickness = 0.3*UnitMult; //3 mm
	box{<-InsideLegLocation*PercentOpen,0,DistanceUpstage>,<-25*inFeet,Height,DistanceUpstage+Thickness> texture{BlackDrapeTexture}}
	box{<InsideLegLocation*PercentOpen,0,DistanceUpstage>,<25*inFeet,Height,DistanceUpstage+Thickness> texture{BlackDrapeTexture}}
#end

#macro SitAudienceCenter()
  camera{
    location <0,4,-36>*inFeet
    look_at <0,8,0>*inFeet
  }
#end

#version POWERSTHEATREPOSITIONS_INC_TEMP;
#end