/*
allstars.inc

This POVRAY include file defines all naked eye stars as distant povray
spheres, with a colour to correspond to the brightness. It is seriously
huge, but as stars are in decreasing brightness order you can safely trim
the last few thousand off. I suggest about a thousand stars for a minimum
convincing sky, though this does depend on the camera zoom you are using.
The last half would only be visible in a very dark sky on a moonless night.

You will also need to adjust the scale of the spheres used for the stars.

This depends on two factors:

1. The resolution at which you are rendering.
2.  The zoom ('direction') used on the camera.

The objective is for the stars to be just over a pixel across, though
bigger values make them more prominent. For higher resolution images,
several pixels across may look good. You can adjust set this for all
brightnesses with the starscale define. If you get lost, try a large
value, (such as 1000), then reduce until they look good.

The how and why of this file...

I do a lot of astronomical and space stuff in Povray, and while it's not
that hard to do a easonably convincing sky with textures, it occured to me
it would be nice to do a real one, with the stars in the correct places.

The raw information in this file is taken from the Yale Bright Star catalogue,
with the last 50 or so objects at the end removed. (Its rather incomplete for
the faintest stars). I then sucked the info into Excel, and cleaned up the
numbers - for example turning the astronomical angles measured in degrees,
minutes and seconds into decimal degrees, and killing off unwanted columns.

I used a simple calculation to turn the brightness of the star into a colour,
with the brightest stars as pure white, and the faintest as dark gray.
Each star is represented by a distant sphere.

The sphere is moved out to a distant position on the X axis, and then rotated
into place. The fainter stars are moved further than the brightest by a small
amount, to ensure that the fainter ones don't block the brighter, when two
stars very close.

Known problems:
Close double stars will come out as too faint, if they end up within one
pixel of each other. Effectively you will only see the brighter one - this
is only seriously inaccurate for double stars which are almost the same
brightness.

Possible enhancements:
I rather like the idea of adjusting the colour to reflect the actual colour
of the stars. It should be subtle...
If anyone has an accurate image map of the Milky Way, it would be neat to
add that as a background texture.
I also rather like the idea of a shell which you pass your position, a date,
and a time - and Povray would then render an accurate night sky for you.
Adding a few of the very brightest deep sky objects would also be fun!

This file by Nick Stevens, starbase1@cix.compulink.co.uk
*/

#declare starscale=20;
#declare star=sphere {<0,0,0> starscale  finish {ambient 1}};


#declare realsky=union{

object {star scale 2.30 translate <9999.01,0,0> rotate <0,0,-16.72,> rotate <0,-101.29,0>  pigment {rgb <1,1,1>}}
object {star scale 2.25 translate <9999.28,0,0> rotate <0,0,-52.68,> rotate <0,-95.99,0>  pigment {rgb <1,1,1>}}
object {star scale 2.15 translate <9999.89,0,0> rotate <0,0,19.18,> rotate <0,-213.92,0>  pigment {rgb <1,1,1>}}
object {star scale 2.13 translate <10000.00,0,0> rotate <0,0,41.08,> rotate <0,-75.62,0>  pigment {rgb <1,1,1>}}
object {star scale 2.13 translate <10000.04,0,0> rotate <0,0,38.78,> rotate <0,-279.23,0>  pigment {rgb <0.995,0.995,0.995>}}
object {star scale 2.12 translate <10000.06,0,0> rotate <0,0,46.00,> rotate <0,-79.17,0>  pigment {rgb <0.9925,0.9925,0.9925>}}
object {star scale 2.11 translate <10000.15,0,0> rotate <0,0,-8.20,> rotate <0,-78.63,0>  pigment {rgb <0.98125,0.98125,0.98125>}}
object {star scale 2.08 translate <10000.33,0,0> rotate <0,0,-60.83,> rotate <0,-219.9,0>  pigment {rgb <0.95875,0.95875,0.95875>}}
object {star scale 2.07 translate <10000.36,0,0> rotate <0,0,5.23,> rotate <0,-114.83,0>  pigment {rgb <0.955,0.955,0.955>}}
object {star scale 2.05 translate <10000.49,0,0> rotate <0,0,-57.25,> rotate <0,-24.43,0>  pigment {rgb <0.93875,0.93875,0.93875>}}
object {star scale 2.03 translate <10000.61,0,0> rotate <0,0,-60.37,> rotate <0,-210.96,0>  pigment {rgb <0.92375,0.92375,0.92375>}}
object {star scale 2.01 translate <10000.75,0,0> rotate <0,0,8.87,> rotate <0,-297.7,0>  pigment {rgb <0.90625,0.90625,0.90625>}}
object {star scale 2.00 translate <10000.80,0,0> rotate <0,0,-63.10,> rotate <0,-186.65,0>  pigment {rgb <0.9,0.9,0.9>}}
object {star scale 2.00 translate <10000.80,0,0> rotate <0,0,7.40,> rotate <0,-88.79,0>  pigment {rgb <0.9,0.9,0.9>}}
object {star scale 1.99 translate <10000.86,0,0> rotate <0,0,16.50,> rotate <0,-68.98,0>  pigment {rgb <0.8925,0.8925,0.8925>}}
object {star scale 1.97 translate <10000.97,0,0> rotate <0,0,-11.15,> rotate <0,-201.3,0>  pigment {rgb <0.87875,0.87875,0.87875>}}
object {star scale 1.95 translate <10001.08,0,0> rotate <0,0,-26.43,> rotate <0,-247.35,0>  pigment {rgb <0.865,0.865,0.865>}}
object {star scale 1.94 translate <10001.15,0,0> rotate <0,0,28.02,> rotate <0,-116.33,0>  pigment {rgb <0.85625,0.85625,0.85625>}}
object {star scale 1.94 translate <10001.15,0,0> rotate <0,0,-29.62,> rotate <0,-344.41,0>  pigment {rgb <0.85625,0.85625,0.85625>}}
object {star scale 1.93 translate <10001.25,0,0> rotate <0,0,45.27,> rotate <0,-310.36,0>  pigment {rgb <0.84375,0.84375,0.84375>}}
object {star scale 1.92 translate <10001.27,0,0> rotate <0,0,-59.70,> rotate <0,-191.93,0>  pigment {rgb <0.84125,0.84125,0.84125>}}
object {star scale 1.91 translate <10001.35,0,0> rotate <0,0,11.97,> rotate <0,-152.09,0>  pigment {rgb <0.83125,0.83125,0.83125>}}
object {star scale 1.88 translate <10001.50,0,0> rotate <0,0,-28.97,> rotate <0,-104.66,0>  pigment {rgb <0.8125,0.8125,0.8125>}}
object {star scale 1.86 translate <10001.62,0,0> rotate <0,0,-37.10,> rotate <0,-263.4,0>  pigment {rgb <0.7975,0.7975,0.7975>}}
object {star scale 1.86 translate <10001.63,0,0> rotate <0,0,6.35,> rotate <0,-81.28,0>  pigment {rgb <0.79625,0.79625,0.79625>}}
object {star scale 1.86 translate <10001.65,0,0> rotate <0,0,28.60,> rotate <0,-81.57,0>  pigment {rgb <0.79375,0.79375,0.79375>}}
object {star scale 1.86 translate <10001.66,0,0> rotate <0,0,-57.12,> rotate <0,-187.79,0>  pigment {rgb <0.7925,0.7925,0.7925>}}
object {star scale 1.85 translate <10001.68,0,0> rotate <0,0,-69.72,> rotate <0,-138.3,0>  pigment {rgb <0.79,0.79,0.79>}}
object {star scale 1.85 translate <10001.69,0,0> rotate <0,0,-1.20,> rotate <0,-84.05,0>  pigment {rgb <0.78875,0.78875,0.78875>}}
object {star scale 1.84 translate <10001.74,0,0> rotate <0,0,-46.97,> rotate <0,-332.06,0>  pigment {rgb <0.7825,0.7825,0.7825>}}
object {star scale 1.84 translate <10001.79,0,0> rotate <0,0,61.75,> rotate <0,-165.93,0>  pigment {rgb <0.77625,0.77625,0.77625>}}
object {star scale 1.84 translate <10001.79,0,0> rotate <0,0,55.95,> rotate <0,-193.51,0>  pigment {rgb <0.77625,0.77625,0.77625>}}
object {star scale 1.83 translate <10001.80,0,0> rotate <0,0,49.85,> rotate <0,-51.08,0>  pigment {rgb <0.775,0.775,0.775>}}
object {star scale 1.83 translate <10001.83,0,0> rotate <0,0,-34.38,> rotate <0,-276.04,0>  pigment {rgb <0.77125,0.77125,0.77125>}}
object {star scale 1.83 translate <10001.83,0,0> rotate <0,0,-47.35,> rotate <0,-122.38,0>  pigment {rgb <0.77125,0.77125,0.77125>}}
object {star scale 1.83 translate <10001.84,0,0> rotate <0,0,-26.40,> rotate <0,-107.1,0>  pigment {rgb <0.77,0.77,0.77>}}
object {star scale 1.82 translate <10001.86,0,0> rotate <0,0,-43.00,> rotate <0,-264.33,0>  pigment {rgb <0.7675,0.7675,0.7675>}}
object {star scale 1.82 translate <10001.88,0,0> rotate <0,0,49.32,> rotate <0,-206.88,0>  pigment {rgb <0.765,0.765,0.765>}}
object {star scale 1.82 translate <10001.88,0,0> rotate <0,0,-59.50,> rotate <0,-125.63,0>  pigment {rgb <0.765,0.765,0.765>}}
object {star scale 1.82 translate <10001.90,0,0> rotate <0,0,44.95,> rotate <0,-89.88,0>  pigment {rgb <0.7625,0.7625,0.7625>}}
object {star scale 1.82 translate <10001.90,0,0> rotate <0,0,19.85,> rotate <0,-155,0>  pigment {rgb <0.7625,0.7625,0.7625>}}
object {star scale 1.81 translate <10001.92,0,0> rotate <0,0,-56.73,> rotate <0,-306.41,0>  pigment {rgb <0.76,0.76,0.76>}}
object {star scale 1.81 translate <10001.93,0,0> rotate <0,0,16.40,> rotate <0,-99.43,0>  pigment {rgb <0.75875,0.75875,0.75875>}}
object {star scale 1.81 translate <10001.93,0,0> rotate <0,0,-69.03,> rotate <0,-252.17,0>  pigment {rgb <0.75875,0.75875,0.75875>}}
object {star scale 1.81 translate <10001.95,0,0> rotate <0,0,-54.72,> rotate <0,-131.18,0>  pigment {rgb <0.75625,0.75625,0.75625>}}
object {star scale 1.80 translate <10001.98,0,0> rotate <0,0,-17.95,> rotate <0,-95.68,0>  pigment {rgb <0.7525,0.7525,0.7525>}}
object {star scale 1.80 translate <10001.98,0,0> rotate <0,0,-8.67,> rotate <0,-141.9,0>  pigment {rgb <0.7525,0.7525,0.7525>}}
object {star scale 1.80 translate <10001.99,0,0> rotate <0,0,31.88,> rotate <0,-113.65,0>  pigment {rgb <0.75125,0.75125,0.75125>}}
object {star scale 1.80 translate <10002.00,0,0> rotate <0,0,-2.98,> rotate <0,-34.84,0>  pigment {rgb <0.75,0.75,0.75>}}
object {star scale 1.80 translate <10002.00,0,0> rotate <0,0,25.92,> rotate <0,-239.88,0>  pigment {rgb <0.75,0.75,0.75>}}
object {star scale 1.80 translate <10002.01,0,0> rotate <0,0,23.45,> rotate <0,-31.79,0>  pigment {rgb <0.74875,0.74875,0.74875>}}
object {star scale 1.80 translate <10002.03,0,0> rotate <0,0,-17.98,> rotate <0,-10.9,0>  pigment {rgb <0.74625,0.74625,0.74625>}}
object {star scale 1.79 translate <10002.04,0,0> rotate <0,0,89.25,> rotate <0,-37.8,0>  pigment {rgb <0.745,0.745,0.745>}}
object {star scale 1.79 translate <10002.05,0,0> rotate <0,0,-1.95,> rotate <0,-85.19,0>  pigment {rgb <0.74375,0.74375,0.74375>}}
object {star scale 1.79 translate <10002.06,0,0> rotate <0,0,35.62,> rotate <0,-17.43,0>  pigment {rgb <0.7425,0.7425,0.7425>}}
object {star scale 1.79 translate <10002.06,0,0> rotate <0,0,-9.67,> rotate <0,-86.94,0>  pigment {rgb <0.7425,0.7425,0.7425>}}
object {star scale 1.79 translate <10002.06,0,0> rotate <0,0,-26.30,> rotate <0,-283.82,0>  pigment {rgb <0.7425,0.7425,0.7425>}}
object {star scale 1.79 translate <10002.07,0,0> rotate <0,0,29.08,> rotate <0,-2.1,0>  pigment {rgb <0.74125,0.74125,0.74125>}}
object {star scale 1.79 translate <10002.07,0,0> rotate <0,0,12.57,> rotate <0,-263.73,0>  pigment {rgb <0.74125,0.74125,0.74125>}}
object {star scale 1.79 translate <10002.07,0,0> rotate <0,0,-36.38,> rotate <0,-211.67,0>  pigment {rgb <0.74125,0.74125,0.74125>}}
object {star scale 1.79 translate <10002.07,0,0> rotate <0,0,74.15,> rotate <0,-222.68,0>  pigment {rgb <0.74125,0.74125,0.74125>}}
object {star scale 1.79 translate <10002.09,0,0> rotate <0,0,-63.10,> rotate <0,-186.65,0>  pigment {rgb <0.73875,0.73875,0.73875>}}
object {star scale 1.78 translate <10002.10,0,0> rotate <0,0,42.33,> rotate <0,-30.97,0>  pigment {rgb <0.7375,0.7375,0.7375>}}
object {star scale 1.78 translate <10002.10,0,0> rotate <0,0,-46.88,> rotate <0,-340.67,0>  pigment {rgb <0.7375,0.7375,0.7375>}}
object {star scale 1.78 translate <10002.13,0,0> rotate <0,0,14.57,> rotate <0,-177.27,0>  pigment {rgb <0.73375,0.73375,0.73375>}}
object {star scale 1.78 translate <10002.15,0,0> rotate <0,0,40.95,> rotate <0,-47.05,0>  pigment {rgb <0.73125,0.73125,0.73125>}}
object {star scale 1.77 translate <10002.16,0,0> rotate <0,0,-48.97,> rotate <0,-190.38,0>  pigment {rgb <0.73,0.73,0.73>}}
object {star scale 1.77 translate <10002.20,0,0> rotate <0,0,-0.30,> rotate <0,-83,0>  pigment {rgb <0.725,0.725,0.725>}}
object {star scale 1.77 translate <10002.21,0,0> rotate <0,0,40.25,> rotate <0,-305.55,0>  pigment {rgb <0.72375,0.72375,0.72375>}}
object {star scale 1.76 translate <10002.22,0,0> rotate <0,0,56.53,> rotate <0,-10.13,0>  pigment {rgb <0.7225,0.7225,0.7225>}}
object {star scale 1.76 translate <10002.23,0,0> rotate <0,0,51.48,> rotate <0,-269.15,0>  pigment {rgb <0.72125,0.72125,0.72125>}}
object {star scale 1.76 translate <10002.23,0,0> rotate <0,0,26.72,> rotate <0,-233.67,0>  pigment {rgb <0.72125,0.72125,0.72125>}}
object {star scale 1.76 translate <10002.24,0,0> rotate <0,0,-43.43,> rotate <0,-137,0>  pigment {rgb <0.72,0.72,0.72>}}
object {star scale 1.76 translate <10002.25,0,0> rotate <0,0,-59.27,> rotate <0,-139.28,0>  pigment {rgb <0.71875,0.71875,0.71875>}}
object {star scale 1.76 translate <10002.25,0,0> rotate <0,0,-40.00,> rotate <0,-120.9,0>  pigment {rgb <0.71875,0.71875,0.71875>}}
object {star scale 1.76 translate <10002.27,0,0> rotate <0,0,54.93,> rotate <0,-200.98,0>  pigment {rgb <0.71625,0.71625,0.71625>}}
object {star scale 1.76 translate <10002.27,0,0> rotate <0,0,59.15,> rotate <0,-2.29,0>  pigment {rgb <0.71625,0.71625,0.71625>}}
object {star scale 1.75 translate <10002.30,0,0> rotate <0,0,-53.47,> rotate <0,-204.97,0>  pigment {rgb <0.7125,0.7125,0.7125>}}
object {star scale 1.75 translate <10002.30,0,0> rotate <0,0,-22.62,> rotate <0,-240.08,0>  pigment {rgb <0.7125,0.7125,0.7125>}}
object {star scale 1.75 translate <10002.30,0,0> rotate <0,0,-34.30,> rotate <0,-252.54,0>  pigment {rgb <0.7125,0.7125,0.7125>}}
object {star scale 1.75 translate <10002.31,0,0> rotate <0,0,-47.40,> rotate <0,-220.48,0>  pigment {rgb <0.71125,0.71125,0.71125>}}
object {star scale 1.75 translate <10002.32,0,0> rotate <0,0,-42.15,> rotate <0,-218.88,0>  pigment {rgb <0.71,0.71,0.71>}}
object {star scale 1.74 translate <10002.38,0,0> rotate <0,0,56.38,> rotate <0,-165.46,0>  pigment {rgb <0.7025,0.7025,0.7025>}}
object {star scale 1.74 translate <10002.39,0,0> rotate <0,0,-42.30,> rotate <0,-6.57,0>  pigment {rgb <0.70125,0.70125,0.70125>}}
object {star scale 1.73 translate <10002.40,0,0> rotate <0,0,27.08,> rotate <0,-221.25,0>  pigment {rgb <0.7,0.7,0.7>}}
object {star scale 1.73 translate <10002.40,0,0> rotate <0,0,9.88,> rotate <0,-326.05,0>  pigment {rgb <0.7,0.7,0.7>}}
object {star scale 1.73 translate <10002.41,0,0> rotate <0,0,-39.03,> rotate <0,-265.62,0>  pigment {rgb <0.69875,0.69875,0.69875>}}
object {star scale 1.73 translate <10002.43,0,0> rotate <0,0,-15.72,> rotate <0,-257.6,0>  pigment {rgb <0.69625,0.69625,0.69625>}}
object {star scale 1.73 translate <10002.43,0,0> rotate <0,0,53.70,> rotate <0,-178.45,0>  pigment {rgb <0.69625,0.69625,0.69625>}}
object {star scale 1.73 translate <10002.44,0,0> rotate <0,0,62.58,> rotate <0,-319.65,0>  pigment {rgb <0.695,0.695,0.695>}}
object {star scale 1.73 translate <10002.44,0,0> rotate <0,0,-29.30,> rotate <0,-111.02,0>  pigment {rgb <0.695,0.695,0.695>}}
object {star scale 1.72 translate <10002.46,0,0> rotate <0,0,33.97,> rotate <0,-311.55,0>  pigment {rgb <0.6925,0.6925,0.6925>}}
object {star scale 1.72 translate <10002.49,0,0> rotate <0,0,-55.02,> rotate <0,-140.53,0>  pigment {rgb <0.68875,0.68875,0.68875>}}
object {star scale 1.72 translate <10002.50,0,0> rotate <0,0,15.20,> rotate <0,-346.19,0>  pigment {rgb <0.6875,0.6875,0.6875>}}
object {star scale 1.71 translate <10002.52,0,0> rotate <0,0,4.10,> rotate <0,-45.57,0>  pigment {rgb <0.685,0.685,0.685>}}
object {star scale 1.71 translate <10002.54,0,0> rotate <0,0,-47.30,> rotate <0,-208.88,0>  pigment {rgb <0.6825,0.6825,0.6825>}}
object {star scale 1.71 translate <10002.56,0,0> rotate <0,0,28.08,> rotate <0,-345.95,0>  pigment {rgb <0.68,0.68,0.68>}}
object {star scale 1.71 translate <10002.56,0,0> rotate <0,0,20.52,> rotate <0,-168.53,0>  pigment {rgb <0.68,0.68,0.68>}}
object {star scale 1.71 translate <10002.57,0,0> rotate <0,0,60.72,> rotate <0,-14.18,0>  pigment {rgb <0.67875,0.67875,0.67875>}}
object {star scale 1.71 translate <10002.57,0,0> rotate <0,0,-10.57,> rotate <0,-249.29,0>  pigment {rgb <0.67875,0.67875,0.67875>}}
object {star scale 1.70 translate <10002.58,0,0> rotate <0,0,-17.83,> rotate <0,-83.18,0>  pigment {rgb <0.6775,0.6775,0.6775>}}
object {star scale 1.70 translate <10002.58,0,0> rotate <0,0,-50.72,> rotate <0,-182.09,0>  pigment {rgb <0.6775,0.6775,0.6775>}}
object {star scale 1.70 translate <10002.59,0,0> rotate <0,0,-17.53,> rotate <0,-183.95,0>  pigment {rgb <0.67625,0.67625,0.67625>}}
object {star scale 1.70 translate <10002.60,0,0> rotate <0,0,-29.87,> rotate <0,-285.65,0>  pigment {rgb <0.675,0.675,0.675>}}
object {star scale 1.70 translate <10002.61,0,0> rotate <0,0,-9.38,> rotate <0,-229.25,0>  pigment {rgb <0.67375,0.67375,0.67375>}}
object {star scale 1.70 translate <10002.63,0,0> rotate <0,0,-19.80,> rotate <0,-241.36,0>  pigment {rgb <0.67125,0.67125,0.67125>}}
object {star scale 1.70 translate <10002.63,0,0> rotate <0,0,-34.08,> rotate <0,-84.91,0>  pigment {rgb <0.67125,0.67125,0.67125>}}
object {star scale 1.70 translate <10002.63,0,0> rotate <0,0,20.80,> rotate <0,-28.66,0>  pigment {rgb <0.67125,0.67125,0.67125>}}
object {star scale 1.69 translate <10002.65,0,0> rotate <0,0,6.42,> rotate <0,-236.07,0>  pigment {rgb <0.66875,0.66875,0.66875>}}
object {star scale 1.69 translate <10002.66,0,0> rotate <0,0,37.20,> rotate <0,-89.93,0>  pigment {rgb <0.6675,0.6675,0.6675>}}
object {star scale 1.69 translate <10002.67,0,0> rotate <0,0,-43.13,> rotate <0,-224.63,0>  pigment {rgb <0.66625,0.66625,0.66625>}}
object {star scale 1.69 translate <10002.67,0,0> rotate <0,0,-23.40,> rotate <0,-188.6,0>  pigment {rgb <0.66625,0.66625,0.66625>}}
object {star scale 1.69 translate <10002.68,0,0> rotate <0,0,-49.43,> rotate <0,-161.69,0>  pigment {rgb <0.665,0.665,0.665>}}
object {star scale 1.69 translate <10002.68,0,0> rotate <0,0,60.23,> rotate <0,-21.45,0>  pigment {rgb <0.665,0.665,0.665>}}
object {star scale 1.69 translate <10002.68,0,0> rotate <0,0,18.40,> rotate <0,-208.67,0>  pigment {rgb <0.665,0.665,0.665>}}
object {star scale 1.69 translate <10002.69,0,0> rotate <0,0,-29.82,> rotate <0,-275.25,0>  pigment {rgb <0.66375,0.66375,0.66375>}}
object {star scale 1.69 translate <10002.69,0,0> rotate <0,0,33.15,> rotate <0,-74.25,0>  pigment {rgb <0.66375,0.66375,0.66375>}}
object {star scale 1.68 translate <10002.70,0,0> rotate <0,0,-69.13,> rotate <0,-189.3,0>  pigment {rgb <0.6625,0.6625,0.6625>}}
object {star scale 1.68 translate <10002.70,0,0> rotate <0,0,-37.30,> rotate <0,-262.69,0>  pigment {rgb <0.6625,0.6625,0.6625>}}
object {star scale 1.68 translate <10002.70,0,0> rotate <0,0,-37.10,> rotate <0,-109.29,0>  pigment {rgb <0.6625,0.6625,0.6625>}}
object {star scale 1.68 translate <10002.72,0,0> rotate <0,0,10.62,> rotate <0,-296.56,0>  pigment {rgb <0.66,0.66,0.66>}}
object {star scale 1.68 translate <10002.73,0,0> rotate <0,0,61.50,> rotate <0,-246,0>  pigment {rgb <0.65875,0.65875,0.65875>}}
object {star scale 1.68 translate <10002.73,0,0> rotate <0,0,-3.68,> rotate <0,-243.58,0>  pigment {rgb <0.65875,0.65875,0.65875>}}
object {star scale 1.68 translate <10002.75,0,0> rotate <0,0,-16.05,> rotate <0,-222.72,0>  pigment {rgb <0.65625,0.65625,0.65625>}}
object {star scale 1.68 translate <10002.75,0,0> rotate <0,0,-36.72,> rotate <0,-200.15,0>  pigment {rgb <0.65625,0.65625,0.65625>}}
object {star scale 1.67 translate <10002.76,0,0> rotate <0,0,-5.92,> rotate <0,-83.86,0>  pigment {rgb <0.655,0.655,0.655>}}
object {star scale 1.67 translate <10002.76,0,0> rotate <0,0,-64.38,> rotate <0,-160.74,0>  pigment {rgb <0.655,0.655,0.655>}}
object {star scale 1.67 translate <10002.77,0,0> rotate <0,0,-41.17,> rotate <0,-233.79,0>  pigment {rgb <0.65375,0.65375,0.65375>}}
object {star scale 1.67 translate <10002.77,0,0> rotate <0,0,-5.08,> rotate <0,-76.96,0>  pigment {rgb <0.65375,0.65375,0.65375>}}
object {star scale 1.67 translate <10002.77,0,0> rotate <0,0,4.57,> rotate <0,-265.87,0>  pigment {rgb <0.65375,0.65375,0.65375>}}
object {star scale 1.67 translate <10002.78,0,0> rotate <0,0,21.48,> rotate <0,-247.55,0>  pigment {rgb <0.6525,0.6525,0.6525>}}
object {star scale 1.67 translate <10002.79,0,0> rotate <0,0,52.32,> rotate <0,-262.61,0>  pigment {rgb <0.65125,0.65125,0.65125>}}
object {star scale 1.67 translate <10002.80,0,0> rotate <0,0,-77.25,> rotate <0,-6.44,0>  pigment {rgb <0.65,0.65,0.65>}}
object {star scale 1.67 translate <10002.81,0,0> rotate <0,0,-24.30,> rotate <0,-121.89,0>  pigment {rgb <0.64875,0.64875,0.64875>}}
object {star scale 1.67 translate <10002.81,0,0> rotate <0,0,31.60,> rotate <0,-250.32,0>  pigment {rgb <0.64875,0.64875,0.64875>}}
object {star scale 1.67 translate <10002.81,0,0> rotate <0,0,-58.75,> rotate <0,-183.79,0>  pigment {rgb <0.64875,0.64875,0.64875>}}
object {star scale 1.66 translate <10002.83,0,0> rotate <0,0,-16.13,> rotate <0,-326.76,0>  pigment {rgb <0.64625,0.64625,0.64625>}}
object {star scale 1.66 translate <10002.83,0,0> rotate <0,0,-28.22,> rotate <0,-248.97,0>  pigment {rgb <0.64625,0.64625,0.64625>}}
object {star scale 1.66 translate <10002.83,0,0> rotate <0,0,15.18,> rotate <0,-3.31,0>  pigment {rgb <0.64625,0.64625,0.64625>}}
object {star scale 1.66 translate <10002.83,0,0> rotate <0,0,10.97,> rotate <0,-195.55,0>  pigment {rgb <0.64625,0.64625,0.64625>}}
object {star scale 1.66 translate <10002.83,0,0> rotate <0,0,-25.43,> rotate <0,-276.99,0>  pigment {rgb <0.64625,0.64625,0.64625>}}
object {star scale 1.66 translate <10002.84,0,0> rotate <0,0,-20.75,> rotate <0,-82.06,0>  pigment {rgb <0.645,0.645,0.645>}}
object {star scale 1.66 translate <10002.84,0,0> rotate <0,0,-63.43,> rotate <0,-238.79,0>  pigment {rgb <0.645,0.645,0.645>}}
object {star scale 1.66 translate <10002.84,0,0> rotate <0,0,-55.53,> rotate <0,-261.33,0>  pigment {rgb <0.645,0.645,0.645>}}
object {star scale 1.66 translate <10002.85,0,0> rotate <0,0,31.88,> rotate <0,-58.53,0>  pigment {rgb <0.64375,0.64375,0.64375>}}
object {star scale 1.66 translate <10002.85,0,0> rotate <0,0,-60.25,> rotate <0,-334.63,0>  pigment {rgb <0.64375,0.64375,0.64375>}}
object {star scale 1.66 translate <10002.86,0,0> rotate <0,0,-61.57,> rotate <0,-29.69,0>  pigment {rgb <0.6425,0.6425,0.6425>}}
object {star scale 1.66 translate <10002.87,0,0> rotate <0,0,45.13,> rotate <0,-296.24,0>  pigment {rgb <0.64125,0.64125,0.64125>}}
object {star scale 1.66 translate <10002.87,0,0> rotate <0,0,24.12,> rotate <0,-56.87,0>  pigment {rgb <0.64125,0.64125,0.64125>}}
object {star scale 1.65 translate <10002.88,0,0> rotate <0,0,-25.58,> rotate <0,-245.3,0>  pigment {rgb <0.64,0.64,0.64>}}
object {star scale 1.65 translate <10002.88,0,0> rotate <0,0,38.32,> rotate <0,-194.01,0>  pigment {rgb <0.64,0.64,0.64>}}
object {star scale 1.65 translate <10002.88,0,0> rotate <0,0,-5.58,> rotate <0,-322.89,0>  pigment {rgb <0.64,0.64,0.64>}}
object {star scale 1.65 translate <10002.89,0,0> rotate <0,0,40.00,> rotate <0,-59.46,0>  pigment {rgb <0.63875,0.63875,0.63875>}}
object {star scale 1.65 translate <10002.89,0,0> rotate <0,0,-21.02,> rotate <0,-287.44,0>  pigment {rgb <0.63875,0.63875,0.63875>}}
object {star scale 1.65 translate <10002.89,0,0> rotate <0,0,22.52,> rotate <0,-95.74,0>  pigment {rgb <0.63875,0.63875,0.63875>}}
object {star scale 1.65 translate <10002.89,0,0> rotate <0,0,-68.68,> rotate <0,-229.73,0>  pigment {rgb <0.63875,0.63875,0.63875>}}
object {star scale 1.65 translate <10002.89,0,0> rotate <0,0,8.28,> rotate <0,-111.79,0>  pigment {rgb <0.63875,0.63875,0.63875>}}
object {star scale 1.65 translate <10002.90,0,0> rotate <0,0,-26.12,> rotate <0,-239.71,0>  pigment {rgb <0.6375,0.6375,0.6375>}}
object {star scale 1.65 translate <10002.92,0,0> rotate <0,0,-50.62,> rotate <0,-102.48,0>  pigment {rgb <0.635,0.635,0.635>}}
object {star scale 1.65 translate <10002.93,0,0> rotate <0,0,-0.32,> rotate <0,-331.45,0>  pigment {rgb <0.63375,0.63375,0.63375>}}
object {star scale 1.64 translate <10002.94,0,0> rotate <0,0,-49.88,> rotate <0,-262.96,0>  pigment {rgb <0.6325,0.6325,0.6325>}}
object {star scale 1.64 translate <10002.94,0,0> rotate <0,0,53.50,> rotate <0,-46.2,0>  pigment {rgb <0.6325,0.6325,0.6325>}}
object {star scale 1.64 translate <10002.95,0,0> rotate <0,0,-16.52,> rotate <0,-187.46,0>  pigment {rgb <0.63125,0.63125,0.63125>}}
object {star scale 1.64 translate <10002.95,0,0> rotate <0,0,30.22,> rotate <0,-340.75,0>  pigment {rgb <0.63125,0.63125,0.63125>}}
object {star scale 1.64 translate <10002.95,0,0> rotate <0,0,-13.52,> rotate <0,-59.51,0>  pigment {rgb <0.63125,0.63125,0.63125>}}
object {star scale 1.64 translate <10002.97,0,0> rotate <0,0,23.77,> rotate <0,-146.46,0>  pigment {rgb <0.62875,0.62875,0.62875>}}
object {star scale 1.64 translate <10002.98,0,0> rotate <0,0,43.83,> rotate <0,-75.49,0>  pigment {rgb <0.6275,0.6275,0.6275>}}
object {star scale 1.64 translate <10002.98,0,0> rotate <0,0,21.15,> rotate <0,-84.41,0>  pigment {rgb <0.6275,0.6275,0.6275>}}
object {star scale 1.64 translate <10002.98,0,0> rotate <0,0,-23.18,> rotate <0,-199.73,0>  pigment {rgb <0.6275,0.6275,0.6275>}}
object {star scale 1.64 translate <10002.99,0,0> rotate <0,0,13.87,> rotate <0,-286.35,0>  pigment {rgb <0.62625,0.62625,0.62625>}}
object {star scale 1.64 translate <10002.99,0,0> rotate <0,0,25.13,> rotate <0,-100.98,0>  pigment {rgb <0.62625,0.62625,0.62625>}}
object {star scale 1.64 translate <10002.99,0,0> rotate <0,0,-30.43,> rotate <0,-271.45,0>  pigment {rgb <0.62625,0.62625,0.62625>}}
object {star scale 1.64 translate <10002.99,0,0> rotate <0,0,-22.62,> rotate <0,-182.53,0>  pigment {rgb <0.62625,0.62625,0.62625>}}
object {star scale 1.63 translate <10003.00,0,0> rotate <0,0,34.98,> rotate <0,-32.38,0>  pigment {rgb <0.625,0.625,0.625>}}
object {star scale 1.63 translate <10003.00,0,0> rotate <0,0,-37.37,> rotate <0,-328.48,0>  pigment {rgb <0.625,0.625,0.625>}}
object {star scale 1.63 translate <10003.00,0,0> rotate <0,0,44.48,> rotate <0,-167.42,0>  pigment {rgb <0.625,0.625,0.625>}}
object {star scale 1.63 translate <10003.02,0,0> rotate <0,0,-30.07,> rotate <0,-95.08,0>  pigment {rgb <0.6225,0.6225,0.6225>}}
object {star scale 1.63 translate <10003.02,0,0> rotate <0,0,47.78,> rotate <0,-55.73,0>  pigment {rgb <0.6225,0.6225,0.6225>}}
object {star scale 1.63 translate <10003.03,0,0> rotate <0,0,-40.12,> rotate <0,-266.9,0>  pigment {rgb <0.62125,0.62125,0.62125>}}
object {star scale 1.63 translate <10003.03,0,0> rotate <0,0,-23.83,> rotate <0,-105.76,0>  pigment {rgb <0.62125,0.62125,0.62125>}}
object {star scale 1.63 translate <10003.03,0,0> rotate <0,0,41.50,> rotate <0,-155.58,0>  pigment {rgb <0.62125,0.62125,0.62125>}}
object {star scale 1.63 translate <10003.04,0,0> rotate <0,0,38.32,> rotate <0,-218.02,0>  pigment {rgb <0.62,0.62,0.62>}}
object {star scale 1.63 translate <10003.04,0,0> rotate <0,0,71.83,> rotate <0,-230.18,0>  pigment {rgb <0.62,0.62,0.62>}}
object {star scale 1.63 translate <10003.05,0,0> rotate <0,0,-68.12,> rotate <0,-191.57,0>  pigment {rgb <0.61875,0.61875,0.61875>}}
object {star scale 1.62 translate <10003.07,0,0> rotate <0,0,-14.78,> rotate <0,-305.25,0>  pigment {rgb <0.61625,0.61625,0.61625>}}
object {star scale 1.62 translate <10003.07,0,0> rotate <0,0,67.67,> rotate <0,-288.14,0>  pigment {rgb <0.61625,0.61625,0.61625>}}
object {star scale 1.62 translate <10003.08,0,0> rotate <0,0,27.97,> rotate <0,-292.68,0>  pigment {rgb <0.615,0.615,0.615>}}
object {star scale 1.62 translate <10003.11,0,0> rotate <0,0,-47.28,> rotate <0,-309.39,0>  pigment {rgb <0.61125,0.61125,0.61125>}}
object {star scale 1.62 translate <10003.11,0,0> rotate <0,0,-36.77,> rotate <0,-274.41,0>  pigment {rgb <0.61125,0.61125,0.61125>}}
object {star scale 1.62 translate <10003.11,0,0> rotate <0,0,5.95,> rotate <0,-133.85,0>  pigment {rgb <0.61125,0.61125,0.61125>}}
object {star scale 1.62 translate <10003.11,0,0> rotate <0,0,-35.77,> rotate <0,-87.74,0>  pigment {rgb <0.61125,0.61125,0.61125>}}
object {star scale 1.62 translate <10003.11,0,0> rotate <0,0,-16.18,> rotate <0,-162.4,0>  pigment {rgb <0.61125,0.61125,0.61125>}}
object {star scale 1.61 translate <10003.12,0,0> rotate <0,0,-57.03,> rotate <0,-142.8,0>  pigment {rgb <0.61,0.61,0.61>}}
object {star scale 1.61 translate <10003.12,0,0> rotate <0,0,-55.98,> rotate <0,-254.66,0>  pigment {rgb <0.61,0.61,0.61>}}
object {star scale 1.61 translate <10003.13,0,0> rotate <0,0,-42.10,> rotate <0,-224.79,0>  pigment {rgb <0.60875,0.60875,0.60875>}}
object {star scale 1.61 translate <10003.13,0,0> rotate <0,0,-63.02,> rotate <0,-173.95,0>  pigment {rgb <0.60875,0.60875,0.60875>}}
object {star scale 1.61 translate <10003.14,0,0> rotate <0,0,24.83,> rotate <0,-258.76,0>  pigment {rgb <0.6075,0.6075,0.6075>}}
object {star scale 1.61 translate <10003.14,0,0> rotate <0,0,-38.05,> rotate <0,-252.97,0>  pigment {rgb <0.6075,0.6075,0.6075>}}
object {star scale 1.61 translate <10003.14,0,0> rotate <0,0,34.40,> rotate <0,-140.26,0>  pigment {rgb <0.6075,0.6075,0.6075>}}
object {star scale 1.61 translate <10003.15,0,0> rotate <0,0,48.03,> rotate <0,-134.8,0>  pigment {rgb <0.60625,0.60625,0.60625>}}
object {star scale 1.61 translate <10003.15,0,0> rotate <0,0,-65.07,> rotate <0,-146.78,0>  pigment {rgb <0.60625,0.60625,0.60625>}}
object {star scale 1.61 translate <10003.16,0,0> rotate <0,0,36.80,> rotate <0,-258.76,0>  pigment {rgb <0.605,0.605,0.605>}}
object {star scale 1.61 translate <10003.17,0,0> rotate <0,0,41.23,> rotate <0,-76.63,0>  pigment {rgb <0.60375,0.60375,0.60375>}}
object {star scale 1.61 translate <10003.17,0,0> rotate <0,0,-43.18,> rotate <0,-99.44,0>  pigment {rgb <0.60375,0.60375,0.60375>}}
object {star scale 1.60 translate <10003.18,0,0> rotate <0,0,70.55,> rotate <0,-322.16,0>  pigment {rgb <0.6025,0.6025,0.6025>}}
object {star scale 1.60 translate <10003.18,0,0> rotate <0,0,-64.97,> rotate <0,-220.63,0>  pigment {rgb <0.6025,0.6025,0.6025>}}
object {star scale 1.60 translate <10003.18,0,0> rotate <0,0,51.68,> rotate <0,-143.21,0>  pigment {rgb <0.6025,0.6025,0.6025>}}
object {star scale 1.60 translate <10003.18,0,0> rotate <0,0,65.72,> rotate <0,-257.2,0>  pigment {rgb <0.6025,0.6025,0.6025>}}
object {star scale 1.60 translate <10003.18,0,0> rotate <0,0,-27.00,> rotate <0,-281.42,0>  pigment {rgb <0.6025,0.6025,0.6025>}}
object {star scale 1.60 translate <10003.19,0,0> rotate <0,0,6.95,> rotate <0,-72.46,0>  pigment {rgb <0.60125,0.60125,0.60125>}}
object {star scale 1.60 translate <10003.19,0,0> rotate <0,0,30.23,> rotate <0,-318.23,0>  pigment {rgb <0.60125,0.60125,0.60125>}}
object {star scale 1.60 translate <10003.19,0,0> rotate <0,0,-22.37,> rotate <0,-76.37,0>  pigment {rgb <0.60125,0.60125,0.60125>}}
object {star scale 1.60 translate <10003.20,0,0> rotate <0,0,22.50,> rotate <0,-93.72,0>  pigment {rgb <0.6,0.6,0.6>}}
object {star scale 1.60 translate <10003.20,0,0> rotate <0,0,-37.05,> rotate <0,-267.46,0>  pigment {rgb <0.6,0.6,0.6>}}
object {star scale 1.60 translate <10003.20,0,0> rotate <0,0,9.38,> rotate <0,-254.43,0>  pigment {rgb <0.6,0.6,0.6>}}
object {star scale 1.60 translate <10003.21,0,0> rotate <0,0,-40.65,> rotate <0,-230.34,0>  pigment {rgb <0.59875,0.59875,0.59875>}}
object {star scale 1.60 translate <10003.22,0,0> rotate <0,0,77.62,> rotate <0,-354.83,0>  pigment {rgb <0.5975,0.5975,0.5975>}}
object {star scale 1.60 translate <10003.23,0,0> rotate <0,0,-4.70,> rotate <0,-244.58,0>  pigment {rgb <0.59625,0.59625,0.59625>}}
object {star scale 1.59 translate <10003.24,0,0> rotate <0,0,-43.30,> rotate <0,-112.31,0>  pigment {rgb <0.595,0.595,0.595>}}
object {star scale 1.59 translate <10003.24,0,0> rotate <0,0,-0.82,> rotate <0,-302.83,0>  pigment {rgb <0.595,0.595,0.595>}}
object {star scale 1.59 translate <10003.25,0,0> rotate <0,0,32.68,> rotate <0,-284.73,0>  pigment {rgb <0.59375,0.59375,0.59375>}}
object {star scale 1.59 translate <10003.25,0,0> rotate <0,0,30.87,> rotate <0,-9.83,0>  pigment {rgb <0.59375,0.59375,0.59375>}}
object {star scale 1.59 translate <10003.25,0,0> rotate <0,0,-2.88,> rotate <0,-275.33,0>  pigment {rgb <0.59375,0.59375,0.59375>}}
object {star scale 1.59 translate <10003.26,0,0> rotate <0,0,-55.05,> rotate <0,-68.5,0>  pigment {rgb <0.5925,0.5925,0.5925>}}
object {star scale 1.59 translate <10003.26,0,0> rotate <0,0,-61.93,> rotate <0,-102.05,0>  pigment {rgb <0.5925,0.5925,0.5925>}}
object {star scale 1.59 translate <10003.26,0,0> rotate <0,0,-26.68,> rotate <0,-211.6,0>  pigment {rgb <0.5925,0.5925,0.5925>}}
object {star scale 1.59 translate <10003.27,0,0> rotate <0,0,-15.82,> rotate <0,-343.66,0>  pigment {rgb <0.59125,0.59125,0.59125>}}
object {star scale 1.59 translate <10003.27,0,0> rotate <0,0,-74.25,> rotate <0,-56.81,0>  pigment {rgb <0.59125,0.59125,0.59125>}}
object {star scale 1.59 translate <10003.28,0,0> rotate <0,0,-25.00,> rotate <0,-260.5,0>  pigment {rgb <0.59,0.59,0.59>}}
object {star scale 1.58 translate <10003.30,0,0> rotate <0,0,-46.72,> rotate <0,-16.52,0>  pigment {rgb <0.5875,0.5875,0.5875>}}
object {star scale 1.58 translate <10003.30,0,0> rotate <0,0,58.97,> rotate <0,-231.23,0>  pigment {rgb <0.5875,0.5875,0.5875>}}
object {star scale 1.58 translate <10003.31,0,0> rotate <0,0,57.03,> rotate <0,-183.86,0>  pigment {rgb <0.58625,0.58625,0.58625>}}
object {star scale 1.58 translate <10003.31,0,0> rotate <0,0,-61.68,> rotate <0,-158,0>  pigment {rgb <0.58625,0.58625,0.58625>}}
object {star scale 1.58 translate <10003.31,0,0> rotate <0,0,-70.03,> rotate <0,-153.44,0>  pigment {rgb <0.58625,0.58625,0.58625>}}
object {star scale 1.58 translate <10003.31,0,0> rotate <0,0,-27.67,> rotate <0,-286.73,0>  pigment {rgb <0.58625,0.58625,0.58625>}}
object {star scale 1.58 translate <10003.32,0,0> rotate <0,0,-56.38,> rotate <0,-261.35,0>  pigment {rgb <0.585,0.585,0.585>}}
object {star scale 1.58 translate <10003.32,0,0> rotate <0,0,-16.20,> rotate <0,-78.23,0>  pigment {rgb <0.585,0.585,0.585>}}
object {star scale 1.58 translate <10003.32,0,0> rotate <0,0,-25.28,> rotate <0,-226.02,0>  pigment {rgb <0.585,0.585,0.585>}}
object {star scale 1.58 translate <10003.33,0,0> rotate <0,0,-43.23,> rotate <0,-258.04,0>  pigment {rgb <0.58375,0.58375,0.58375>}}
object {star scale 1.58 translate <10003.34,0,0> rotate <0,0,-62.47,> rotate <0,-63.6,0>  pigment {rgb <0.5825,0.5825,0.5825>}}
object {star scale 1.58 translate <10003.34,0,0> rotate <0,0,15.43,> rotate <0,-168.56,0>  pigment {rgb <0.5825,0.5825,0.5825>}}
object {star scale 1.58 translate <10003.34,0,0> rotate <0,0,-9.78,> rotate <0,-269.75,0>  pigment {rgb <0.5825,0.5825,0.5825>}}
object {star scale 1.58 translate <10003.35,0,0> rotate <0,0,-2.38,> rotate <0,-81.12,0>  pigment {rgb <0.58125,0.58125,0.58125>}}
object {star scale 1.58 translate <10003.35,0,0> rotate <0,0,-24.87,> rotate <0,-117.32,0>  pigment {rgb <0.58125,0.58125,0.58125>}}
object {star scale 1.58 translate <10003.35,0,0> rotate <0,0,60.72,> rotate <0,-127.57,0>  pigment {rgb <0.58125,0.58125,0.58125>}}
object {star scale 1.58 translate <10003.35,0,0> rotate <0,0,12.90,> rotate <0,-101.33,0>  pigment {rgb <0.58125,0.58125,0.58125>}}
object {star scale 1.58 translate <10003.35,0,0> rotate <0,0,58.20,> rotate <0,-332.71,0>  pigment {rgb <0.58125,0.58125,0.58125>}}
object {star scale 1.57 translate <10003.36,0,0> rotate <0,0,-44.70,> rotate <0,-230.67,0>  pigment {rgb <0.58,0.58,0.58>}}
object {star scale 1.57 translate <10003.36,0,0> rotate <0,0,-0.60,> rotate <0,-203.68,0>  pigment {rgb <0.58,0.58,0.58>}}
object {star scale 1.57 translate <10003.36,0,0> rotate <0,0,3.12,> rotate <0,-291.37,0>  pigment {rgb <0.58,0.58,0.58>}}
object {star scale 1.57 translate <10003.37,0,0> rotate <0,0,6.42,> rotate <0,-131.7,0>  pigment {rgb <0.57875,0.57875,0.57875>}}
object {star scale 1.57 translate <10003.37,0,0> rotate <0,0,63.68,> rotate <0,-28.6,0>  pigment {rgb <0.57875,0.57875,0.57875>}}
object {star scale 1.57 translate <10003.37,0,0> rotate <0,0,3.38,> rotate <0,-193.9,0>  pigment {rgb <0.57875,0.57875,0.57875>}}
object {star scale 1.57 translate <10003.38,0,0> rotate <0,0,-61.33,> rotate <0,-154.27,0>  pigment {rgb <0.5775,0.5775,0.5775>}}
object {star scale 1.57 translate <10003.39,0,0> rotate <0,0,10.83,> rotate <0,-340.36,0>  pigment {rgb <0.57625,0.57625,0.57625>}}
object {star scale 1.57 translate <10003.40,0,0> rotate <0,0,-38.40,> rotate <0,-240.03,0>  pigment {rgb <0.575,0.575,0.575>}}
object {star scale 1.57 translate <10003.40,0,0> rotate <0,0,-52.10,> rotate <0,-228.07,0>  pigment {rgb <0.575,0.575,0.575>}}
object {star scale 1.57 translate <10003.40,0,0> rotate <0,0,-62.48,> rotate <0,-83.4,0>  pigment {rgb <0.575,0.575,0.575>}}
object {star scale 1.57 translate <10003.40,0,0> rotate <0,0,-62.52,> rotate <0,-146.31,0>  pigment {rgb <0.575,0.575,0.575>}}
object {star scale 1.57 translate <10003.40,0,0> rotate <0,0,-43.32,> rotate <0,-22.09,0>  pigment {rgb <0.575,0.575,0.575>}}
object {star scale 1.57 translate <10003.41,0,0> rotate <0,0,27.73,> rotate <0,-266.62,0>  pigment {rgb <0.57375,0.57375,0.57375>}}
object {star scale 1.57 translate <10003.41,0,0> rotate <0,0,15.87,> rotate <0,-67.17,0>  pigment {rgb <0.57375,0.57375,0.57375>}}
object {star scale 1.57 translate <10003.41,0,0> rotate <0,0,-41.68,> rotate <0,-207.38,0>  pigment {rgb <0.57375,0.57375,0.57375>}}
object {star scale 1.57 translate <10003.41,0,0> rotate <0,0,12.48,> rotate <0,-60.17,0>  pigment {rgb <0.57375,0.57375,0.57375>}}
object {star scale 1.57 translate <10003.41,0,0> rotate <0,0,29.58,> rotate <0,-28.27,0>  pigment {rgb <0.57375,0.57375,0.57375>}}
object {star scale 1.56 translate <10003.42,0,0> rotate <0,0,-40.30,> rotate <0,-44.56,0>  pigment {rgb <0.5725,0.5725,0.5725>}}
object {star scale 1.56 translate <10003.42,0,0> rotate <0,0,61.83,> rotate <0,-311.32,0>  pigment {rgb <0.5725,0.5725,0.5725>}}
object {star scale 1.56 translate <10003.42,0,0> rotate <0,0,-66.20,> rotate <0,-311.24,0>  pigment {rgb <0.5725,0.5725,0.5725>}}
object {star scale 1.56 translate <10003.43,0,0> rotate <0,0,33.37,> rotate <0,-282.52,0>  pigment {rgb <0.57125,0.57125,0.57125>}}
object {star scale 1.56 translate <10003.43,0,0> rotate <0,0,-4.88,> rotate <0,-286.56,0>  pigment {rgb <0.57125,0.57125,0.57125>}}
object {star scale 1.56 translate <10003.43,0,0> rotate <0,0,23.42,> rotate <0,-154.18,0>  pigment {rgb <0.57125,0.57125,0.57125>}}
object {star scale 1.56 translate <10003.44,0,0> rotate <0,0,57.82,> rotate <0,-12.28,0>  pigment {rgb <0.57,0.57,0.57>}}
object {star scale 1.56 translate <10003.44,0,0> rotate <0,0,-58.97,> rotate <0,-137.74,0>  pigment {rgb <0.57,0.57,0.57>}}
object {star scale 1.56 translate <10003.44,0,0> rotate <0,0,-10.18,> rotate <0,-17.15,0>  pigment {rgb <0.57,0.57,0.57>}}
object {star scale 1.56 translate <10003.45,0,0> rotate <0,0,42.92,> rotate <0,-154.28,0>  pigment {rgb <0.56875,0.56875,0.56875>}}
object {star scale 1.56 translate <10003.46,0,0> rotate <0,0,3.23,> rotate <0,-40.83,0>  pigment {rgb <0.5675,0.5675,0.5675>}}
object {star scale 1.56 translate <10003.46,0,0> rotate <0,0,33.08,> rotate <0,-169.62,0>  pigment {rgb <0.5675,0.5675,0.5675>}}
object {star scale 1.56 translate <10003.46,0,0> rotate <0,0,-27.93,> rotate <0,-105.43,0>  pigment {rgb <0.5675,0.5675,0.5675>}}
object {star scale 1.56 translate <10003.46,0,0> rotate <0,0,-52.98,> rotate <0,-119.2,0>  pigment {rgb <0.5675,0.5675,0.5675>}}
object {star scale 1.56 translate <10003.47,0,0> rotate <0,0,-42.48,> rotate <0,-207.4,0>  pigment {rgb <0.56625,0.56625,0.56625>}}
object {star scale 1.55 translate <10003.48,0,0> rotate <0,0,9.90,> rotate <0,-145.29,0>  pigment {rgb <0.565,0.565,0.565>}}
object {star scale 1.55 translate <10003.48,0,0> rotate <0,0,33.32,> rotate <0,-228.88,0>  pigment {rgb <0.565,0.565,0.565>}}
object {star scale 1.55 translate <10003.48,0,0> rotate <0,0,-51.32,> rotate <0,-342.14,0>  pigment {rgb <0.565,0.565,0.565>}}
object {star scale 1.55 translate <10003.48,0,0> rotate <0,0,19.48,> rotate <0,-299.69,0>  pigment {rgb <0.565,0.565,0.565>}}
object {star scale 1.55 translate <10003.49,0,0> rotate <0,0,40.38,> rotate <0,-225.49,0>  pigment {rgb <0.56375,0.56375,0.56375>}}
object {star scale 1.55 translate <10003.49,0,0> rotate <0,0,24.60,> rotate <0,-342.5,0>  pigment {rgb <0.56375,0.56375,0.56375>}}
object {star scale 1.55 translate <10003.50,0,0> rotate <0,0,14.38,> rotate <0,-258.66,0>  pigment {rgb <0.5625,0.5625,0.5625>}}
object {star scale 1.55 translate <10003.50,0,0> rotate <0,0,-45.97,> rotate <0,-276.75,0>  pigment {rgb <0.5625,0.5625,0.5625>}}
object {star scale 1.55 translate <10003.50,0,0> rotate <0,0,1.00,> rotate <0,-298.12,0>  pigment {rgb <0.5625,0.5625,0.5625>}}
object {star scale 1.55 translate <10003.50,0,0> rotate <0,0,-15.93,> rotate <0,-26.02,0>  pigment {rgb <0.5625,0.5625,0.5625>}}
object {star scale 1.55 translate <10003.50,0,0> rotate <0,0,-21.10,> rotate <0,-284.43,0>  pigment {rgb <0.5625,0.5625,0.5625>}}
object {star scale 1.55 translate <10003.50,0,0> rotate <0,0,66.20,> rotate <0,-342.42,0>  pigment {rgb <0.5625,0.5625,0.5625>}}
object {star scale 1.55 translate <10003.50,0,0> rotate <0,0,38.83,> rotate <0,-46.3,0>  pigment {rgb <0.5625,0.5625,0.5625>}}
object {star scale 1.55 translate <10003.51,0,0> rotate <0,0,6.20,> rotate <0,-332.55,0>  pigment {rgb <0.56125,0.56125,0.56125>}}
object {star scale 1.55 translate <10003.52,0,0> rotate <0,0,9.20,> rotate <0,-124.13,0>  pigment {rgb <0.56,0.56,0.56>}}
object {star scale 1.55 translate <10003.53,0,0> rotate <0,0,21.98,> rotate <0,-110.03,0>  pigment {rgb <0.55875,0.55875,0.55875>}}
object {star scale 1.55 translate <10003.53,0,0> rotate <0,0,16.77,> rotate <0,-151.83,0>  pigment {rgb <0.55875,0.55875,0.55875>}}
object {star scale 1.55 translate <10003.53,0,0> rotate <0,0,38.93,> rotate <0,-250.73,0>  pigment {rgb <0.55875,0.55875,0.55875>}}
object {star scale 1.55 translate <10003.53,0,0> rotate <0,0,-9.77,> rotate <0,-55.81,0>  pigment {rgb <0.55875,0.55875,0.55875>}}
object {star scale 1.55 translate <10003.53,0,0> rotate <0,0,19.18,> rotate <0,-67.15,0>  pigment {rgb <0.55875,0.55875,0.55875>}}
object {star scale 1.54 translate <10003.54,0,0> rotate <0,0,-54.58,> rotate <0,-149.22,0>  pigment {rgb <0.5575,0.5575,0.5575>}}
object {star scale 1.54 translate <10003.54,0,0> rotate <0,0,-15.40,> rotate <0,-264.4,0>  pigment {rgb <0.5575,0.5575,0.5575>}}
object {star scale 1.54 translate <10003.54,0,0> rotate <0,0,-31.85,> rotate <0,-173.25,0>  pigment {rgb <0.5575,0.5575,0.5575>}}
object {star scale 1.54 translate <10003.54,0,0> rotate <0,0,-14.83,> rotate <0,-86.74,0>  pigment {rgb <0.5575,0.5575,0.5575>}}
object {star scale 1.54 translate <10003.54,0,0> rotate <0,0,-8.83,> rotate <0,-4.86,0>  pigment {rgb <0.5575,0.5575,0.5575>}}
object {star scale 1.54 translate <10003.55,0,0> rotate <0,0,-51.52,> rotate <0,-34.13,0>  pigment {rgb <0.55625,0.55625,0.55625>}}
object {star scale 1.54 translate <10003.55,0,0> rotate <0,0,-33.80,> rotate <0,-64.48,0>  pigment {rgb <0.55625,0.55625,0.55625>}}
object {star scale 1.54 translate <10003.55,0,0> rotate <0,0,-36.27,> rotate <0,-230.45,0>  pigment {rgb <0.55625,0.55625,0.55625>}}
object {star scale 1.54 translate <10003.55,0,0> rotate <0,0,-3.42,> rotate <0,-237.4,0>  pigment {rgb <0.55625,0.55625,0.55625>}}
object {star scale 1.54 translate <10003.55,0,0> rotate <0,0,-46.07,> rotate <0,-214.85,0>  pigment {rgb <0.55625,0.55625,0.55625>}}
object {star scale 1.54 translate <10003.56,0,0> rotate <0,0,72.73,> rotate <0,-275.27,0>  pigment {rgb <0.555,0.555,0.555>}}
object {star scale 1.54 translate <10003.56,0,0> rotate <0,0,-70.50,> rotate <0,-107.19,0>  pigment {rgb <0.555,0.555,0.555>}}
object {star scale 1.54 translate <10003.56,0,0> rotate <0,0,24.40,> rotate <0,-116.11,0>  pigment {rgb <0.555,0.555,0.555>}}
object {star scale 1.54 translate <10003.56,0,0> rotate <0,0,-12.53,> rotate <0,-304.51,0>  pigment {rgb <0.555,0.555,0.555>}}
object {star scale 1.54 translate <10003.56,0,0> rotate <0,0,-38.02,> rotate <0,-253.08,0>  pigment {rgb <0.555,0.555,0.555>}}
object {star scale 1.54 translate <10003.56,0,0> rotate <0,0,-14.77,> rotate <0,-169.83,0>  pigment {rgb <0.555,0.555,0.555>}}
object {star scale 1.54 translate <10003.57,0,0> rotate <0,0,47.15,> rotate <0,-135.9,0>  pigment {rgb <0.55375,0.55375,0.55375>}}
object {star scale 1.54 translate <10003.57,0,0> rotate <0,0,16.53,> rotate <0,-109.53,0>  pigment {rgb <0.55375,0.55375,0.55375>}}
object {star scale 1.54 translate <10003.57,0,0> rotate <0,0,30.38,> rotate <0,-217.96,0>  pigment {rgb <0.55375,0.55375,0.55375>}}
object {star scale 1.54 translate <10003.57,0,0> rotate <0,0,-66.18,> rotate <0,-302.18,0>  pigment {rgb <0.55375,0.55375,0.55375>}}
object {star scale 1.54 translate <10003.57,0,0> rotate <0,0,48.62,> rotate <0,-24.5,0>  pigment {rgb <0.55375,0.55375,0.55375>}}
object {star scale 1.54 translate <10003.58,0,0> rotate <0,0,-6.85,> rotate <0,-79.4,0>  pigment {rgb <0.5525,0.5525,0.5525>}}
object {star scale 1.54 translate <10003.58,0,0> rotate <0,0,-40.47,> rotate <0,-142.68,0>  pigment {rgb <0.5525,0.5525,0.5525>}}
object {star scale 1.54 translate <10003.58,0,0> rotate <0,0,14.60,> rotate <0,-309.39,0>  pigment {rgb <0.5525,0.5525,0.5525>}}
object {star scale 1.54 translate <10003.58,0,0> rotate <0,0,-28.13,> rotate <0,-234.25,0>  pigment {rgb <0.5525,0.5525,0.5525>}}
object {star scale 1.54 translate <10003.59,0,0> rotate <0,0,-60.68,> rotate <0,-262.78,0>  pigment {rgb <0.55125,0.55125,0.55125>}}
object {star scale 1.54 translate <10003.59,0,0> rotate <0,0,9.03,> rotate <0,-51.2,0>  pigment {rgb <0.55125,0.55125,0.55125>}}
object {star scale 1.54 translate <10003.59,0,0> rotate <0,0,-37.98,> rotate <0,-116.32,0>  pigment {rgb <0.55125,0.55125,0.55125>}}
object {star scale 1.54 translate <10003.59,0,0> rotate <0,0,-60.40,> rotate <0,-185.34,0>  pigment {rgb <0.55125,0.55125,0.55125>}}
object {star scale 1.53 translate <10003.60,0,0> rotate <0,0,33.97,> rotate <0,-103.2,0>  pigment {rgb <0.55,0.55,0.55>}}
object {star scale 1.53 translate <10003.60,0,0> rotate <0,0,-22.45,> rotate <0,-86.12,0>  pigment {rgb <0.55,0.55,0.55>}}
object {star scale 1.53 translate <10003.60,0,0> rotate <0,0,-8.18,> rotate <0,-21,0>  pigment {rgb <0.55,0.55,0.55>}}
object {star scale 1.53 translate <10003.61,0,0> rotate <0,0,27.27,> rotate <0,-42.5,0>  pigment {rgb <0.54875,0.54875,0.54875>}}
object {star scale 1.53 translate <10003.61,0,0> rotate <0,0,-12.37,> rotate <0,-152.65,0>  pigment {rgb <0.54875,0.54875,0.54875>}}
object {star scale 1.53 translate <10003.61,0,0> rotate <0,0,1.77,> rotate <0,-177.68,0>  pigment {rgb <0.54875,0.54875,0.54875>}}
object {star scale 1.53 translate <10003.61,0,0> rotate <0,0,-71.55,> rotate <0,-195.57,0>  pigment {rgb <0.54875,0.54875,0.54875>}}
object {star scale 1.53 translate <10003.62,0,0> rotate <0,0,-70.50,> rotate <0,-107.18,0>  pigment {rgb <0.5475,0.5475,0.5475>}}
object {star scale 1.53 translate <10003.62,0,0> rotate <0,0,15.35,> rotate <0,-22.87,0>  pigment {rgb <0.5475,0.5475,0.5475>}}
object {star scale 1.53 translate <10003.62,0,0> rotate <0,0,-52.92,> rotate <0,-130.08,0>  pigment {rgb <0.5475,0.5475,0.5475>}}
object {star scale 1.53 translate <10003.62,0,0> rotate <0,0,42.32,> rotate <0,-345.48,0>  pigment {rgb <0.5475,0.5475,0.5475>}}
object {star scale 1.53 translate <10003.62,0,0> rotate <0,0,-64.73,> rotate <0,-266.43,0>  pigment {rgb <0.5475,0.5475,0.5475>}}
object {star scale 1.53 translate <10003.63,0,0> rotate <0,0,-66.72,> rotate <0,-176.4,0>  pigment {rgb <0.54625,0.54625,0.54625>}}
object {star scale 1.53 translate <10003.63,0,0> rotate <0,0,24.05,> rotate <0,-57.29,0>  pigment {rgb <0.54625,0.54625,0.54625>}}
object {star scale 1.53 translate <10003.63,0,0> rotate <0,0,15.62,> rotate <0,-64.95,0>  pigment {rgb <0.54625,0.54625,0.54625>}}
object {star scale 1.53 translate <10003.63,0,0> rotate <0,0,-42.35,> rotate <0,-253.65,0>  pigment {rgb <0.54625,0.54625,0.54625>}}
object {star scale 1.53 translate <10003.64,0,0> rotate <0,0,53.90,> rotate <0,-9.24,0>  pigment {rgb <0.545,0.545,0.545>}}
object {star scale 1.53 translate <10003.64,0,0> rotate <0,0,-58.45,> rotate <0,-313.7,0>  pigment {rgb <0.545,0.545,0.545>}}
object {star scale 1.53 translate <10003.65,0,0> rotate <0,0,64.37,> rotate <0,-211.1,0>  pigment {rgb <0.54375,0.54375,0.54375>}}
object {star scale 1.53 translate <10003.65,0,0> rotate <0,0,-29.78,> rotate <0,-234.67,0>  pigment {rgb <0.54375,0.54375,0.54375>}}
object {star scale 1.52 translate <10003.66,0,0> rotate <0,0,63.07,> rotate <0,-142.88,0>  pigment {rgb <0.5425,0.5425,0.5425>}}
object {star scale 1.52 translate <10003.66,0,0> rotate <0,0,-0.02,> rotate <0,-337.21,0>  pigment {rgb <0.5425,0.5425,0.5425>}}
object {star scale 1.52 translate <10003.66,0,0> rotate <0,0,9.93,> rotate <0,-83.78,0>  pigment {rgb <0.5425,0.5425,0.5425>}}
object {star scale 1.52 translate <10003.66,0,0> rotate <0,0,-16.67,> rotate <0,-325.02,0>  pigment {rgb <0.5425,0.5425,0.5425>}}
object {star scale 1.52 translate <10003.66,0,0> rotate <0,0,-21.17,> rotate <0,-347.36,0>  pigment {rgb <0.5425,0.5425,0.5425>}}
object {star scale 1.52 translate <10003.67,0,0> rotate <0,0,15.42,> rotate <0,-236.55,0>  pigment {rgb <0.54125,0.54125,0.54125>}}
object {star scale 1.52 translate <10003.67,0,0> rotate <0,0,-50.10,> rotate <0,-271.66,0>  pigment {rgb <0.54125,0.54125,0.54125>}}
object {star scale 1.52 translate <10003.67,0,0> rotate <0,0,29.10,> rotate <0,-231.95,0>  pigment {rgb <0.54125,0.54125,0.54125>}}
object {star scale 1.52 translate <10003.68,0,0> rotate <0,0,-1.45,> rotate <0,-190.42,0>  pigment {rgb <0.54,0.54,0.54>}}
object {star scale 1.52 translate <10003.68,0,0> rotate <0,0,5.60,> rotate <0,-72.8,0>  pigment {rgb <0.54,0.54,0.54>}}
object {star scale 1.52 translate <10003.69,0,0> rotate <0,0,-51.60,> rotate <0,-28.99,0>  pigment {rgb <0.53875,0.53875,0.53875>}}
object {star scale 1.52 translate <10003.69,0,0> rotate <0,0,-33.20,> rotate <0,-130.9,0>  pigment {rgb <0.53875,0.53875,0.53875>}}
object {star scale 1.52 translate <10003.69,0,0> rotate <0,0,3.28,> rotate <0,-349.29,0>  pigment {rgb <0.53875,0.53875,0.53875>}}
object {star scale 1.52 translate <10003.69,0,0> rotate <0,0,29.25,> rotate <0,-269.44,0>  pigment {rgb <0.53875,0.53875,0.53875>}}
object {star scale 1.52 translate <10003.70,0,0> rotate <0,0,-21.75,> rotate <0,-49.88,0>  pigment {rgb <0.5375,0.5375,0.5375>}}
object {star scale 1.52 translate <10003.70,0,0> rotate <0,0,24.12,> rotate <0,-56.22,0>  pigment {rgb <0.5375,0.5375,0.5375>}}
object {star scale 1.52 translate <10003.70,0,0> rotate <0,0,47.78,> rotate <0,-176.51,0>  pigment {rgb <0.5375,0.5375,0.5375>}}
object {star scale 1.52 translate <10003.71,0,0> rotate <0,0,6.40,> rotate <0,-298.83,0>  pigment {rgb <0.53625,0.53625,0.53625>}}
object {star scale 1.52 translate <10003.71,0,0> rotate <0,0,4.48,> rotate <0,-237.7,0>  pigment {rgb <0.53625,0.53625,0.53625>}}
object {star scale 1.52 translate <10003.71,0,0> rotate <0,0,-14.17,> rotate <0,-89.1,0>  pigment {rgb <0.53625,0.53625,0.53625>}}
object {star scale 1.52 translate <10003.71,0,0> rotate <0,0,-40.58,> rotate <0,-118.05,0>  pigment {rgb <0.53625,0.53625,0.53625>}}
object {star scale 1.52 translate <10003.71,0,0> rotate <0,0,54.28,> rotate <0,-89.88,0>  pigment {rgb <0.53625,0.53625,0.53625>}}
object {star scale 1.51 translate <10003.72,0,0> rotate <0,0,2.45,> rotate <0,-73.56,0>  pigment {rgb <0.535,0.535,0.535>}}
object {star scale 1.51 translate <10003.72,0,0> rotate <0,0,9.73,> rotate <0,-51.79,0>  pigment {rgb <0.535,0.535,0.535>}}
object {star scale 1.51 translate <10003.72,0,0> rotate <0,0,-10.33,> rotate <0,-27.86,0>  pigment {rgb <0.535,0.535,0.535>}}
object {star scale 1.51 translate <10003.72,0,0> rotate <0,0,43.93,> rotate <0,-316.23,0>  pigment {rgb <0.535,0.535,0.535>}}
object {star scale 1.51 translate <10003.73,0,0> rotate <0,0,9.57,> rotate <0,-271.84,0>  pigment {rgb <0.53375,0.53375,0.53375>}}
object {star scale 1.51 translate <10003.73,0,0> rotate <0,0,38.05,> rotate <0,-318.7,0>  pigment {rgb <0.53375,0.53375,0.53375>}}
object {star scale 1.51 translate <10003.73,0,0> rotate <0,0,-2.60,> rotate <0,-84.69,0>  pigment {rgb <0.53375,0.53375,0.53375>}}
object {star scale 1.51 translate <10003.73,0,0> rotate <0,0,1.90,> rotate <0,-221.56,0>  pigment {rgb <0.53375,0.53375,0.53375>}}
object {star scale 1.51 translate <10003.73,0,0> rotate <0,0,-9.47,> rotate <0,-53.23,0>  pigment {rgb <0.53375,0.53375,0.53375>}}
object {star scale 1.51 translate <10003.74,0,0> rotate <0,0,19.15,> rotate <0,-245.48,0>  pigment {rgb <0.5325,0.5325,0.5325>}}
object {star scale 1.51 translate <10003.74,0,0> rotate <0,0,-48.22,> rotate <0,-159.33,0>  pigment {rgb <0.5325,0.5325,0.5325>}}
object {star scale 1.51 translate <10003.74,0,0> rotate <0,0,-22.42,> rotate <0,-321.67,0>  pigment {rgb <0.5325,0.5325,0.5325>}}
object {star scale 1.51 translate <10003.74,0,0> rotate <0,0,2.72,> rotate <0,-266.98,0>  pigment {rgb <0.5325,0.5325,0.5325>}}
object {star scale 1.51 translate <10003.74,0,0> rotate <0,0,-47.10,> rotate <0,-136.04,0>  pigment {rgb <0.5325,0.5325,0.5325>}}
object {star scale 1.51 translate <10003.74,0,0> rotate <0,0,56.87,> rotate <0,-268.38,0>  pigment {rgb <0.5325,0.5325,0.5325>}}
object {star scale 1.51 translate <10003.75,0,0> rotate <0,0,50.28,> rotate <0,-337.82,0>  pigment {rgb <0.53125,0.53125,0.53125>}}
object {star scale 1.51 translate <10003.75,0,0> rotate <0,0,-59.05,> rotate <0,-252.45,0>  pigment {rgb <0.53125,0.53125,0.53125>}}
object {star scale 1.51 translate <10003.76,0,0> rotate <0,0,-21.73,> rotate <0,-286.17,0>  pigment {rgb <0.53,0.53,0.53>}}
object {star scale 1.51 translate <10003.76,0,0> rotate <0,0,25.33,> rotate <0,-331.75,0>  pigment {rgb <0.53,0.53,0.53>}}
object {star scale 1.51 translate <10003.76,0,0> rotate <0,0,17.53,> rotate <0,-65.73,0>  pigment {rgb <0.53,0.53,0.53>}}
object {star scale 1.51 translate <10003.76,0,0> rotate <0,0,-66.13,> rotate <0,-126.43,0>  pigment {rgb <0.53,0.53,0.53>}}
object {star scale 1.51 translate <10003.76,0,0> rotate <0,0,-7.58,> rotate <0,-343.15,0>  pigment {rgb <0.53,0.53,0.53>}}
object {star scale 1.51 translate <10003.77,0,0> rotate <0,0,15.92,> rotate <0,-309.91,0>  pigment {rgb <0.52875,0.52875,0.52875>}}
object {star scale 1.51 translate <10003.77,0,0> rotate <0,0,42.58,> rotate <0,-56.3,0>  pigment {rgb <0.52875,0.52875,0.52875>}}
object {star scale 1.51 translate <10003.77,0,0> rotate <0,0,55.90,> rotate <0,-42.68,0>  pigment {rgb <0.52875,0.52875,0.52875>}}
object {star scale 1.51 translate <10003.77,0,0> rotate <0,0,-9.50,> rotate <0,-311.92,0>  pigment {rgb <0.52875,0.52875,0.52875>}}
object {star scale 1.51 translate <10003.77,0,0> rotate <0,0,51.73,> rotate <0,-292.43,0>  pigment {rgb <0.52875,0.52875,0.52875>}}
object {star scale 1.51 translate <10003.77,0,0> rotate <0,0,53.37,> rotate <0,-289.28,0>  pigment {rgb <0.52875,0.52875,0.52875>}}
object {star scale 1.50 translate <10003.78,0,0> rotate <0,0,-77.38,> rotate <0,-325.37,0>  pigment {rgb <0.5275,0.5275,0.5275>}}
object {star scale 1.50 translate <10003.78,0,0> rotate <0,0,46.73,> rotate <0,-303.41,0>  pigment {rgb <0.5275,0.5275,0.5275>}}
object {star scale 1.50 translate <10003.78,0,0> rotate <0,0,-20.87,> rotate <0,-87.83,0>  pigment {rgb <0.5275,0.5275,0.5275>}}
object {star scale 1.50 translate <10003.78,0,0> rotate <0,0,-58.85,> rotate <0,-163.38,0>  pigment {rgb <0.5275,0.5275,0.5275>}}
object {star scale 1.50 translate <10003.78,0,0> rotate <0,0,27.80,> rotate <0,-111.43,0>  pigment {rgb <0.5275,0.5275,0.5275>}}
object {star scale 1.50 translate <10003.79,0,0> rotate <0,0,59.05,> rotate <0,-147.75,0>  pigment {rgb <0.52625,0.52625,0.52625>}}
object {star scale 1.50 translate <10003.79,0,0> rotate <0,0,44.87,> rotate <0,-47.38,0>  pigment {rgb <0.52625,0.52625,0.52625>}}
object {star scale 1.50 translate <10003.80,0,0> rotate <0,0,46.02,> rotate <0,-264.86,0>  pigment {rgb <0.525,0.525,0.525>}}
object {star scale 1.50 translate <10003.80,0,0> rotate <0,0,18.53,> rotate <0,-296.85,0>  pigment {rgb <0.525,0.525,0.525>}}
object {star scale 1.50 translate <10003.81,0,0> rotate <0,0,1.98,> rotate <0,-247.73,0>  pigment {rgb <0.52375,0.52375,0.52375>}}
object {star scale 1.50 translate <10003.81,0,0> rotate <0,0,36.82,> rotate <0,-139.71,0>  pigment {rgb <0.52375,0.52375,0.52375>}}
object {star scale 1.50 translate <10003.81,0,0> rotate <0,0,-58.75,> rotate <0,-156.97,0>  pigment {rgb <0.52375,0.52375,0.52375>}}
object {star scale 1.50 translate <10003.81,0,0> rotate <0,0,34.22,> rotate <0,-163.33,0>  pigment {rgb <0.52375,0.52375,0.52375>}}
object {star scale 1.50 translate <10003.81,0,0> rotate <0,0,-30.57,> rotate <0,-68.89,0>  pigment {rgb <0.52375,0.52375,0.52375>}}
object {star scale 1.50 translate <10003.81,0,0> rotate <0,0,-16.85,> rotate <0,-156.52,0>  pigment {rgb <0.52375,0.52375,0.52375>}}
object {star scale 1.50 translate <10003.82,0,0> rotate <0,0,26.30,> rotate <0,-235.69,0>  pigment {rgb <0.5225,0.5225,0.5225>}}
object {star scale 1.50 translate <10003.82,0,0> rotate <0,0,-42.10,> rotate <0,-209.57,0>  pigment {rgb <0.5225,0.5225,0.5225>}}
object {star scale 1.50 translate <10003.82,0,0> rotate <0,0,-26.77,> rotate <0,-108.7,0>  pigment {rgb <0.5225,0.5225,0.5225>}}
object {star scale 1.50 translate <10003.83,0,0> rotate <0,0,32.28,> rotate <0,-56.08,0>  pigment {rgb <0.52125,0.52125,0.52125>}}
object {star scale 1.50 translate <10003.83,0,0> rotate <0,0,70.27,> rotate <0,-297.04,0>  pigment {rgb <0.52125,0.52125,0.52125>}}
object {star scale 1.50 translate <10003.83,0,0> rotate <0,0,28.77,> rotate <0,-271.88,0>  pigment {rgb <0.52125,0.52125,0.52125>}}
object {star scale 1.50 translate <10003.83,0,0> rotate <0,0,69.33,> rotate <0,-172.85,0>  pigment {rgb <0.52125,0.52125,0.52125>}}
object {star scale 1.49 translate <10003.84,0,0> rotate <0,0,-60.65,> rotate <0,-133.76,0>  pigment {rgb <0.52,0.52,0.52>}}
object {star scale 1.49 translate <10003.84,0,0> rotate <0,0,-1.38,> rotate <0,-335.41,0>  pigment {rgb <0.52,0.52,0.52>}}
object {star scale 1.49 translate <10003.84,0,0> rotate <0,0,-42.13,> rotate <0,-153.68,0>  pigment {rgb <0.52,0.52,0.52>}}
object {star scale 1.49 translate <10003.84,0,0> rotate <0,0,-51.07,> rotate <0,-86.82,0>  pigment {rgb <0.52,0.52,0.52>}}
object {star scale 1.49 translate <10003.84,0,0> rotate <0,0,15.65,> rotate <0,-239.11,0>  pigment {rgb <0.52,0.52,0.52>}}
object {star scale 1.49 translate <10003.84,0,0> rotate <0,0,-63.68,> rotate <0,-243.86,0>  pigment {rgb <0.52,0.52,0.52>}}
object {star scale 1.49 translate <10003.84,0,0> rotate <0,0,-33.43,> rotate <0,-95.53,0>  pigment {rgb <0.52,0.52,0.52>}}
object {star scale 1.49 translate <10003.84,0,0> rotate <0,0,-64.80,> rotate <0,-56.05,0>  pigment {rgb <0.52,0.52,0.52>}}
object {star scale 1.49 translate <10003.84,0,0> rotate <0,0,37.25,> rotate <0,-269.06,0>  pigment {rgb <0.52,0.52,0.52>}}
object {star scale 1.49 translate <10003.84,0,0> rotate <0,0,21.77,> rotate <0,-275.93,0>  pigment {rgb <0.52,0.52,0.52>}}
object {star scale 1.49 translate <10003.84,0,0> rotate <0,0,-8.25,> rotate <0,-278.8,0>  pigment {rgb <0.52,0.52,0.52>}}
object {star scale 1.49 translate <10003.85,0,0> rotate <0,0,-46.65,> rotate <0,-130.16,0>  pigment {rgb <0.51875,0.51875,0.51875>}}
object {star scale 1.49 translate <10003.85,0,0> rotate <0,0,-28.98,> rotate <0,-48.02,0>  pigment {rgb <0.51875,0.51875,0.51875>}}
object {star scale 1.49 translate <10003.85,0,0> rotate <0,0,-48.53,> rotate <0,-189.43,0>  pigment {rgb <0.51875,0.51875,0.51875>}}
object {star scale 1.49 translate <10003.85,0,0> rotate <0,0,9.30,> rotate <0,-158.2,0>  pigment {rgb <0.51875,0.51875,0.51875>}}
object {star scale 1.49 translate <10003.85,0,0> rotate <0,0,15.95,> rotate <0,-67.15,0>  pigment {rgb <0.51875,0.51875,0.51875>}}
object {star scale 1.49 translate <10003.85,0,0> rotate <0,0,-42.28,> rotate <0,-63.5,0>  pigment {rgb <0.51875,0.51875,0.51875>}}
object {star scale 1.49 translate <10003.85,0,0> rotate <0,0,-79.03,> rotate <0,-221.96,0>  pigment {rgb <0.51875,0.51875,0.51875>}}
object {star scale 1.49 translate <10003.86,0,0> rotate <0,0,38.50,> rotate <0,-14.19,0>  pigment {rgb <0.5175,0.5175,0.5175>}}
object {star scale 1.49 translate <10003.86,0,0> rotate <0,0,-21.05,> rotate <0,-273.44,0>  pigment {rgb <0.5175,0.5175,0.5175>}}
object {star scale 1.49 translate <10003.86,0,0> rotate <0,0,-48.73,> rotate <0,-227.98,0>  pigment {rgb <0.5175,0.5175,0.5175>}}
object {star scale 1.49 translate <10003.86,0,0> rotate <0,0,-14.30,> rotate <0,-69.55,0>  pigment {rgb <0.5175,0.5175,0.5175>}}
object {star scale 1.49 translate <10003.86,0,0> rotate <0,0,-44.80,> rotate <0,-209.67,0>  pigment {rgb <0.5175,0.5175,0.5175>}}
object {star scale 1.49 translate <10003.86,0,0> rotate <0,0,-35.48,> rotate <0,-82.8,0>  pigment {rgb <0.5175,0.5175,0.5175>}}
object {star scale 1.49 translate <10003.86,0,0> rotate <0,0,-24.20,> rotate <0,-103.53,0>  pigment {rgb <0.5175,0.5175,0.5175>}}
object {star scale 1.49 translate <10003.87,0,0> rotate <0,0,69.78,> rotate <0,-188.37,0>  pigment {rgb <0.51625,0.51625,0.51625>}}
object {star scale 1.49 translate <10003.87,0,0> rotate <0,0,-45.75,> rotate <0,-2.35,0>  pigment {rgb <0.51625,0.51625,0.51625>}}
object {star scale 1.49 translate <10003.88,0,0> rotate <0,0,-29.22,> rotate <0,-239.23,0>  pigment {rgb <0.515,0.515,0.515>}}
object {star scale 1.49 translate <10003.88,0,0> rotate <0,0,-54.50,> rotate <0,-170.25,0>  pigment {rgb <0.515,0.515,0.515>}}
object {star scale 1.49 translate <10003.88,0,0> rotate <0,0,2.32,> rotate <0,-138.59,0>  pigment {rgb <0.515,0.515,0.515>}}
object {star scale 1.49 translate <10003.88,0,0> rotate <0,0,-39.40,> rotate <0,-202.76,0>  pigment {rgb <0.515,0.515,0.515>}}
object {star scale 1.49 translate <10003.88,0,0> rotate <0,0,-0.67,> rotate <0,-184.98,0>  pigment {rgb <0.515,0.515,0.515>}}
object {star scale 1.49 translate <10003.88,0,0> rotate <0,0,-72.13,> rotate <0,-188.12,0>  pigment {rgb <0.515,0.515,0.515>}}
object {star scale 1.49 translate <10003.88,0,0> rotate <0,0,24.37,> rotate <0,-56.45,0>  pigment {rgb <0.515,0.515,0.515>}}
object {star scale 1.49 translate <10003.88,0,0> rotate <0,0,-5.65,> rotate <0,-220.76,0>  pigment {rgb <0.515,0.515,0.515>}}
object {star scale 1.49 translate <10003.88,0,0> rotate <0,0,46.47,> rotate <0,-354.39,0>  pigment {rgb <0.515,0.515,0.515>}}
object {star scale 1.49 translate <10003.88,0,0> rotate <0,0,-78.88,> rotate <0,-248.36,0>  pigment {rgb <0.515,0.515,0.515>}}
object {star scale 1.49 translate <10003.88,0,0> rotate <0,0,-1.13,> rotate <0,-144.96,0>  pigment {rgb <0.515,0.515,0.515>}}
object {star scale 1.49 translate <10003.89,0,0> rotate <0,0,46.32,> rotate <0,-244.93,0>  pigment {rgb <0.51375,0.51375,0.51375>}}
object {star scale 1.49 translate <10003.89,0,0> rotate <0,0,35.08,> rotate <0,-299.75,0>  pigment {rgb <0.51375,0.51375,0.51375>}}
object {star scale 1.49 translate <10003.89,0,0> rotate <0,0,-3.92,> rotate <0,-126.42,0>  pigment {rgb <0.51375,0.51375,0.51375>}}
object {star scale 1.49 translate <10003.89,0,0> rotate <0,0,-45.25,> rotate <0,-347.59,0>  pigment {rgb <0.51375,0.51375,0.51375>}}
object {star scale 1.49 translate <10003.89,0,0> rotate <0,0,-8.90,> rotate <0,-44.1,0>  pigment {rgb <0.51375,0.51375,0.51375>}}
object {star scale 1.49 translate <10003.89,0,0> rotate <0,0,26.02,> rotate <0,-148.19,0>  pigment {rgb <0.51375,0.51375,0.51375>}}
object {star scale 1.48 translate <10003.90,0,0> rotate <0,0,-14.78,> rotate <0,-233.88,0>  pigment {rgb <0.5125,0.5125,0.5125>}}
object {star scale 1.48 translate <10003.90,0,0> rotate <0,0,-46.05,> rotate <0,-131.5,0>  pigment {rgb <0.5125,0.5125,0.5125>}}
object {star scale 1.48 translate <10003.90,0,0> rotate <0,0,6.00,> rotate <0,-60.79,0>  pigment {rgb <0.5125,0.5125,0.5125>}}
object {star scale 1.48 translate <10003.90,0,0> rotate <0,0,-67.25,> rotate <0,-284.24,0>  pigment {rgb <0.5125,0.5125,0.5125>}}
object {star scale 1.48 translate <10003.90,0,0> rotate <0,0,43.95,> rotate <0,-283.84,0>  pigment {rgb <0.5125,0.5125,0.5125>}}
object {star scale 1.48 translate <10003.91,0,0> rotate <0,0,30.92,> rotate <0,-255.08,0>  pigment {rgb <0.51125,0.51125,0.51125>}}
object {star scale 1.48 translate <10003.91,0,0> rotate <0,0,5.25,> rotate <0,-318.96,0>  pigment {rgb <0.51125,0.51125,0.51125>}}
object {star scale 1.48 translate <10003.92,0,0> rotate <0,0,-3.35,> rotate <0,-69.08,0>  pigment {rgb <0.51,0.51,0.51>}}
object {star scale 1.48 translate <10003.92,0,0> rotate <0,0,-50.23,> rotate <0,-187.01,0>  pigment {rgb <0.51,0.51,0.51>}}
object {star scale 1.48 translate <10003.93,0,0> rotate <0,0,18.15,> rotate <0,-131.17,0>  pigment {rgb <0.50875,0.50875,0.50875>}}
object {star scale 1.48 translate <10003.93,0,0> rotate <0,0,41.17,> rotate <0,-314.3,0>  pigment {rgb <0.50875,0.50875,0.50875>}}
object {star scale 1.48 translate <10003.93,0,0> rotate <0,0,-43.68,> rotate <0,-6.55,0>  pigment {rgb <0.50875,0.50875,0.50875>}}
object {star scale 1.48 translate <10003.93,0,0> rotate <0,0,-17.85,> rotate <0,-290.42,0>  pigment {rgb <0.50875,0.50875,0.50875>}}
object {star scale 1.48 translate <10003.93,0,0> rotate <0,0,-9.55,> rotate <0,-115.31,0>  pigment {rgb <0.50875,0.50875,0.50875>}}
object {star scale 1.48 translate <10003.93,0,0> rotate <0,0,-49.08,> rotate <0,-22.81,0>  pigment {rgb <0.50875,0.50875,0.50875>}}
object {star scale 1.48 translate <10003.94,0,0> rotate <0,0,-55.25,> rotate <0,-17.1,0>  pigment {rgb <0.5075,0.5075,0.5075>}}
object {star scale 1.48 translate <10003.94,0,0> rotate <0,0,-72.60,> rotate <0,-115.45,0>  pigment {rgb <0.5075,0.5075,0.5075>}}
object {star scale 1.48 translate <10003.94,0,0> rotate <0,0,-33.62,> rotate <0,-237.74,0>  pigment {rgb <0.5075,0.5075,0.5075>}}
object {star scale 1.48 translate <10003.94,0,0> rotate <0,0,-32.52,> rotate <0,-102.46,0>  pigment {rgb <0.5075,0.5075,0.5075>}}
object {star scale 1.48 translate <10003.94,0,0> rotate <0,0,-58.98,> rotate <0,-167.15,0>  pigment {rgb <0.5075,0.5075,0.5075>}}
object {star scale 1.48 translate <10003.94,0,0> rotate <0,0,-19.25,> rotate <0,-99.17,0>  pigment {rgb <0.5075,0.5075,0.5075>}}
object {star scale 1.48 translate <10003.95,0,0> rotate <0,0,10.53,> rotate <0,-170.98,0>  pigment {rgb <0.50625,0.50625,0.50625>}}
object {star scale 1.48 translate <10003.95,0,0> rotate <0,0,52.77,> rotate <0,-43.57,0>  pigment {rgb <0.50625,0.50625,0.50625>}}
object {star scale 1.48 translate <10003.95,0,0> rotate <0,0,72.42,> rotate <0,-30.86,0>  pigment {rgb <0.50625,0.50625,0.50625>}}
object {star scale 1.48 translate <10003.95,0,0> rotate <0,0,23.57,> rotate <0,-341.63,0>  pigment {rgb <0.50625,0.50625,0.50625>}}
object {star scale 1.48 translate <10003.95,0,0> rotate <0,0,-42.82,> rotate <0,-89.79,0>  pigment {rgb <0.50625,0.50625,0.50625>}}
object {star scale 1.48 translate <10003.95,0,0> rotate <0,0,-34.02,> rotate <0,-66.01,0>  pigment {rgb <0.50625,0.50625,0.50625>}}
object {star scale 1.47 translate <10003.96,0,0> rotate <0,0,2.93,> rotate <0,-270.16,0>  pigment {rgb <0.505,0.505,0.505>}}
object {star scale 1.47 translate <10003.96,0,0> rotate <0,0,41.78,> rotate <0,-135.16,0>  pigment {rgb <0.505,0.505,0.505>}}
object {star scale 1.47 translate <10003.96,0,0> rotate <0,0,-72.90,> rotate <0,-300.15,0>  pigment {rgb <0.505,0.505,0.505>}}
object {star scale 1.47 translate <10003.96,0,0> rotate <0,0,-28.95,> rotate <0,-115.95,0>  pigment {rgb <0.505,0.505,0.505>}}
object {star scale 1.47 translate <10003.96,0,0> rotate <0,0,-52.37,> rotate <0,-182.91,0>  pigment {rgb <0.505,0.505,0.505>}}
object {star scale 1.47 translate <10003.96,0,0> rotate <0,0,-40.62,> rotate <0,-290.98,0>  pigment {rgb <0.505,0.505,0.505>}}
object {star scale 1.47 translate <10003.96,0,0> rotate <0,0,-43.48,> rotate <0,-337.32,0>  pigment {rgb <0.505,0.505,0.505>}}
object {star scale 1.47 translate <10003.96,0,0> rotate <0,0,-20.10,> rotate <0,-350.74,0>  pigment {rgb <0.505,0.505,0.505>}}
object {star scale 1.47 translate <10003.97,0,0> rotate <0,0,-35.30,> rotate <0,-130.03,0>  pigment {rgb <0.50375,0.50375,0.50375>}}
object {star scale 1.47 translate <10003.97,0,0> rotate <0,0,-6.28,> rotate <0,-93.72,0>  pigment {rgb <0.50375,0.50375,0.50375>}}
object {star scale 1.47 translate <10003.97,0,0> rotate <0,0,-20.67,> rotate <0,-241.7,0>  pigment {rgb <0.50375,0.50375,0.50375>}}
object {star scale 1.47 translate <10003.97,0,0> rotate <0,0,-67.95,> rotate <0,-109.21,0>  pigment {rgb <0.50375,0.50375,0.50375>}}
object {star scale 1.47 translate <10003.98,0,0> rotate <0,0,39.15,> rotate <0,-87.88,0>  pigment {rgb <0.5025,0.5025,0.5025>}}
object {star scale 1.47 translate <10003.98,0,0> rotate <0,0,-62.32,> rotate <0,-137.82,0>  pigment {rgb <0.5025,0.5025,0.5025>}}
object {star scale 1.47 translate <10003.98,0,0> rotate <0,0,-58.23,> rotate <0,-349.36,0>  pigment {rgb <0.5025,0.5025,0.5025>}}
object {star scale 1.47 translate <10003.98,0,0> rotate <0,0,-74.03,> rotate <0,-156.1,0>  pigment {rgb <0.5025,0.5025,0.5025>}}
object {star scale 1.47 translate <10003.98,0,0> rotate <0,0,47.70,> rotate <0,-303.87,0>  pigment {rgb <0.5025,0.5025,0.5025>}}
object {star scale 1.47 translate <10003.99,0,0> rotate <0,0,58.78,> rotate <0,-325.88,0>  pigment {rgb <0.50125,0.50125,0.50125>}}
object {star scale 1.47 translate <10003.99,0,0> rotate <0,0,-21.08,> rotate <0,-30,0>  pigment {rgb <0.50125,0.50125,0.50125>}}
object {star scale 1.47 translate <10004.00,0,0> rotate <0,0,-59.70,> rotate <0,-161.27,0>  pigment {rgb <0.5,0.5,0.5>}}
object {star scale 1.47 translate <10004.00,0,0> rotate <0,0,-19.47,> rotate <0,-243,0>  pigment {rgb <0.5,0.5,0.5>}}
object {star scale 1.47 translate <10004.00,0,0> rotate <0,0,-71.43,> rotate <0,-280.76,0>  pigment {rgb <0.5,0.5,0.5>}}
object {star scale 1.47 translate <10004.00,0,0> rotate <0,0,-62.80,> rotate <0,-143.06,0>  pigment {rgb <0.5,0.5,0.5>}}
object {star scale 1.47 translate <10004.00,0,0> rotate <0,0,33.85,> rotate <0,-34.33,0>  pigment {rgb <0.5,0.5,0.5>}}
object {star scale 1.47 translate <10004.00,0,0> rotate <0,0,55.00,> rotate <0,-201.3,0>  pigment {rgb <0.5,0.5,0.5>}}
object {star scale 1.47 translate <10004.00,0,0> rotate <0,0,-66.40,> rotate <0,-135.61,0>  pigment {rgb <0.5,0.5,0.5>}}
object {star scale 1.47 translate <10004.00,0,0> rotate <0,0,-37.80,> rotate <0,-220.49,0>  pigment {rgb <0.5,0.5,0.5>}}
object {star scale 1.47 translate <10004.00,0,0> rotate <0,0,-49.90,> rotate <0,-43.47,0>  pigment {rgb <0.5,0.5,0.5>}}
object {star scale 1.47 translate <10004.00,0,0> rotate <0,0,32.92,> rotate <0,-297.64,0>  pigment {rgb <0.5,0.5,0.5>}}
object {star scale 1.47 translate <10004.01,0,0> rotate <0,0,-44.47,> rotate <0,-290.66,0>  pigment {rgb <0.49875,0.49875,0.49875>}}
object {star scale 1.47 translate <10004.01,0,0> rotate <0,0,30.37,> rotate <0,-307.35,0>  pigment {rgb <0.49875,0.49875,0.49875>}}
object {star scale 1.47 translate <10004.01,0,0> rotate <0,0,58.57,> rotate <0,-240.48,0>  pigment {rgb <0.49875,0.49875,0.49875>}}
object {star scale 1.47 translate <10004.01,0,0> rotate <0,0,-27.70,> rotate <0,-132.63,0>  pigment {rgb <0.49875,0.49875,0.49875>}}
object {star scale 1.47 translate <10004.01,0,0> rotate <0,0,-13.58,> rotate <0,-342.4,0>  pigment {rgb <0.49875,0.49875,0.49875>}}
object {star scale 1.46 translate <10004.02,0,0> rotate <0,0,-3.25,> rotate <0,-71.38,0>  pigment {rgb <0.4975,0.4975,0.4975>}}
object {star scale 1.46 translate <10004.02,0,0> rotate <0,0,-0.12,> rotate <0,-338.84,0>  pigment {rgb <0.4975,0.4975,0.4975>}}
object {star scale 1.46 translate <10004.02,0,0> rotate <0,0,6.87,> rotate <0,-359.83,0>  pigment {rgb <0.4975,0.4975,0.4975>}}
object {star scale 1.46 translate <10004.02,0,0> rotate <0,0,45.60,> rotate <0,-323.5,0>  pigment {rgb <0.4975,0.4975,0.4975>}}
object {star scale 1.46 translate <10004.02,0,0> rotate <0,0,-5.75,> rotate <0,-285.42,0>  pigment {rgb <0.4975,0.4975,0.4975>}}
object {star scale 1.46 translate <10004.02,0,0> rotate <0,0,15.07,> rotate <0,-284.9,0>  pigment {rgb <0.4975,0.4975,0.4975>}}
object {star scale 1.46 translate <10004.03,0,0> rotate <0,0,-57.18,> rotate <0,-193.65,0>  pigment {rgb <0.49625,0.49625,0.49625>}}
object {star scale 1.46 translate <10004.03,0,0> rotate <0,0,28.77,> rotate <0,-131.68,0>  pigment {rgb <0.49625,0.49625,0.49625>}}
object {star scale 1.46 translate <10004.03,0,0> rotate <0,0,47.72,> rotate <0,-62.17,0>  pigment {rgb <0.49625,0.49625,0.49625>}}
object {star scale 1.46 translate <10004.03,0,0> rotate <0,0,11.30,> rotate <0,-308.3,0>  pigment {rgb <0.49625,0.49625,0.49625>}}
object {star scale 1.46 translate <10004.03,0,0> rotate <0,0,6.03,> rotate <0,-170.28,0>  pigment {rgb <0.49625,0.49625,0.49625>}}
object {star scale 1.46 translate <10004.03,0,0> rotate <0,0,-24.72,> rotate <0,-182.1,0>  pigment {rgb <0.49625,0.49625,0.49625>}}
object {star scale 1.46 translate <10004.03,0,0> rotate <0,0,6.52,> rotate <0,-176.46,0>  pigment {rgb <0.49625,0.49625,0.49625>}}
object {star scale 1.46 translate <10004.04,0,0> rotate <0,0,-37.90,> rotate <0,-215.14,0>  pigment {rgb <0.495,0.495,0.495>}}
object {star scale 1.46 translate <10004.04,0,0> rotate <0,0,60.45,> rotate <0,-75.85,0>  pigment {rgb <0.495,0.495,0.495>}}
object {star scale 1.46 translate <10004.04,0,0> rotate <0,0,-50.17,> rotate <0,-244.96,0>  pigment {rgb <0.495,0.495,0.495>}}
object {star scale 1.46 translate <10004.04,0,0> rotate <0,0,-49.42,> rotate <0,-219.47,0>  pigment {rgb <0.495,0.495,0.495>}}
object {star scale 1.46 translate <10004.04,0,0> rotate <0,0,-35.18,> rotate <0,-220.91,0>  pigment {rgb <0.495,0.495,0.495>}}
object {star scale 1.46 translate <10004.04,0,0> rotate <0,0,15.80,> rotate <0,-207.37,0>  pigment {rgb <0.495,0.495,0.495>}}
object {star scale 1.46 translate <10004.04,0,0> rotate <0,0,35.78,> rotate <0,-59.74,0>  pigment {rgb <0.495,0.495,0.495>}}
object {star scale 1.46 translate <10004.05,0,0> rotate <0,0,-45.28,> rotate <0,-227.21,0>  pigment {rgb <0.49375,0.49375,0.49375>}}
object {star scale 1.46 translate <10004.05,0,0> rotate <0,0,-42.65,> rotate <0,-131.1,0>  pigment {rgb <0.49375,0.49375,0.49375>}}
object {star scale 1.46 translate <10004.05,0,0> rotate <0,0,-6.83,> rotate <0,-62.97,0>  pigment {rgb <0.49375,0.49375,0.49375>}}
object {star scale 1.46 translate <10004.05,0,0> rotate <0,0,51.85,> rotate <0,-216.3,0>  pigment {rgb <0.49375,0.49375,0.49375>}}
object {star scale 1.46 translate <10004.05,0,0> rotate <0,0,49.62,> rotate <0,-47.27,0>  pigment {rgb <0.49375,0.49375,0.49375>}}
object {star scale 1.46 translate <10004.05,0,0> rotate <0,0,-53.15,> rotate <0,-254.9,0>  pigment {rgb <0.49375,0.49375,0.49375>}}
object {star scale 1.46 translate <10004.06,0,0> rotate <0,0,-17.68,> rotate <0,-171.22,0>  pigment {rgb <0.4925,0.4925,0.4925>}}
object {star scale 1.46 translate <10004.06,0,0> rotate <0,0,-64.00,> rotate <0,-184.61,0>  pigment {rgb <0.4925,0.4925,0.4925>}}
object {star scale 1.46 translate <10004.06,0,0> rotate <0,0,-17.23,> rotate <0,-316.49,0>  pigment {rgb <0.4925,0.4925,0.4925>}}
object {star scale 1.46 translate <10004.06,0,0> rotate <0,0,-58.80,> rotate <0,-229.38,0>  pigment {rgb <0.4925,0.4925,0.4925>}}
object {star scale 1.46 translate <10004.06,0,0> rotate <0,0,50.68,> rotate <0,-25.91,0>  pigment {rgb <0.4925,0.4925,0.4925>}}
object {star scale 1.46 translate <10004.06,0,0> rotate <0,0,0.33,> rotate <0,-39.87,0>  pigment {rgb <0.4925,0.4925,0.4925>}}
object {star scale 1.46 translate <10004.06,0,0> rotate <0,0,-76.92,> rotate <0,-124.63,0>  pigment {rgb <0.4925,0.4925,0.4925>}}
object {star scale 1.46 translate <10004.06,0,0> rotate <0,0,24.27,> rotate <0,-11.83,0>  pigment {rgb <0.4925,0.4925,0.4925>}}
object {star scale 1.46 translate <10004.06,0,0> rotate <0,0,-12.05,> rotate <0,-103.55,0>  pigment {rgb <0.4925,0.4925,0.4925>}}
object {star scale 1.46 translate <10004.06,0,0> rotate <0,0,26.90,> rotate <0,-113.98,0>  pigment {rgb <0.4925,0.4925,0.4925>}}
object {star scale 1.46 translate <10004.07,0,0> rotate <0,0,13.50,> rotate <0,-74.09,0>  pigment {rgb <0.49125,0.49125,0.49125>}}
object {star scale 1.46 translate <10004.07,0,0> rotate <0,0,-23.62,> rotate <0,-45.6,0>  pigment {rgb <0.49125,0.49125,0.49125>}}
object {star scale 1.46 translate <10004.07,0,0> rotate <0,0,-59.23,> rotate <0,-143.61,0>  pigment {rgb <0.49125,0.49125,0.49125>}}
object {star scale 1.46 translate <10004.07,0,0> rotate <0,0,-18.30,> rotate <0,-164.94,0>  pigment {rgb <0.49125,0.49125,0.49125>}}
object {star scale 1.45 translate <10004.08,0,0> rotate <0,0,19.82,> rotate <0,-320.52,0>  pigment {rgb <0.49,0.49,0.49>}}
object {star scale 1.45 translate <10004.08,0,0> rotate <0,0,-68.67,> rotate <0,-35.44,0>  pigment {rgb <0.49,0.49,0.49>}}
object {star scale 1.45 translate <10004.08,0,0> rotate <0,0,-6.00,> rotate <0,-214,0>  pigment {rgb <0.49,0.49,0.49>}}
object {star scale 1.45 translate <10004.08,0,0> rotate <0,0,9.28,> rotate <0,-84.23,0>  pigment {rgb <0.49,0.49,0.49>}}
object {star scale 1.45 translate <10004.09,0,0> rotate <0,0,82.12,> rotate <0,-101.13,0>  pigment {rgb <0.48875,0.48875,0.48875>}}
object {star scale 1.45 translate <10004.09,0,0> rotate <0,0,41.40,> rotate <0,-24.2,0>  pigment {rgb <0.48875,0.48875,0.48875>}}
object {star scale 1.45 translate <10004.10,0,0> rotate <0,0,-46.37,> rotate <0,-117.31,0>  pigment {rgb <0.4875,0.4875,0.4875>}}
object {star scale 1.45 translate <10004.10,0,0> rotate <0,0,-66.32,> rotate <0,-234.18,0>  pigment {rgb <0.4875,0.4875,0.4875>}}
object {star scale 1.45 translate <10004.10,0,0> rotate <0,0,-37.92,> rotate <0,-287.37,0>  pigment {rgb <0.4875,0.4875,0.4875>}}
object {star scale 1.45 translate <10004.10,0,0> rotate <0,0,-68.27,> rotate <0,-39.9,0>  pigment {rgb <0.4875,0.4875,0.4875>}}
object {star scale 1.45 translate <10004.10,0,0> rotate <0,0,-61.17,> rotate <0,-176.63,0>  pigment {rgb <0.4875,0.4875,0.4875>}}
object {star scale 1.45 translate <10004.10,0,0> rotate <0,0,-39.33,> rotate <0,-287.51,0>  pigment {rgb <0.4875,0.4875,0.4875>}}
object {star scale 1.45 translate <10004.10,0,0> rotate <0,0,-14.85,> rotate <0,-147.87,0>  pigment {rgb <0.4875,0.4875,0.4875>}}
object {star scale 1.45 translate <10004.10,0,0> rotate <0,0,-39.85,> rotate <0,-40.17,0>  pigment {rgb <0.4875,0.4875,0.4875>}}
object {star scale 1.45 translate <10004.10,0,0> rotate <0,0,18.13,> rotate <0,-237.18,0>  pigment {rgb <0.4875,0.4875,0.4875>}}
object {star scale 1.45 translate <10004.11,0,0> rotate <0,0,49.22,> rotate <0,-41.05,0>  pigment {rgb <0.48625,0.48625,0.48625>}}
object {star scale 1.45 translate <10004.11,0,0> rotate <0,0,-15.63,> rotate <0,-105.94,0>  pigment {rgb <0.48625,0.48625,0.48625>}}
object {star scale 1.45 translate <10004.11,0,0> rotate <0,0,-52.75,> rotate <0,-345.22,0>  pigment {rgb <0.48625,0.48625,0.48625>}}
object {star scale 1.45 translate <10004.11,0,0> rotate <0,0,12.95,> rotate <0,-52.72,0>  pigment {rgb <0.48625,0.48625,0.48625>}}
object {star scale 1.45 translate <10004.11,0,0> rotate <0,0,-26.93,> rotate <0,-312.95,0>  pigment {rgb <0.48625,0.48625,0.48625>}}
object {star scale 1.45 translate <10004.12,0,0> rotate <0,0,9.65,> rotate <0,-90.6,0>  pigment {rgb <0.485,0.485,0.485>}}
object {star scale 1.45 translate <10004.12,0,0> rotate <0,0,-43.75,> rotate <0,-337.44,0>  pigment {rgb <0.485,0.485,0.485>}}
object {star scale 1.45 translate <10004.12,0,0> rotate <0,0,8.73,> rotate <0,-181.3,0>  pigment {rgb <0.485,0.485,0.485>}}
object {star scale 1.45 translate <10004.12,0,0> rotate <0,0,-41.87,> rotate <0,-298.82,0>  pigment {rgb <0.485,0.485,0.485>}}
object {star scale 1.45 translate <10004.12,0,0> rotate <0,0,-49.07,> rotate <0,-277.21,0>  pigment {rgb <0.485,0.485,0.485>}}
object {star scale 1.45 translate <10004.13,0,0> rotate <0,0,48.40,> rotate <0,-63.73,0>  pigment {rgb <0.48375,0.48375,0.48375>}}
object {star scale 1.45 translate <10004.13,0,0> rotate <0,0,-42.98,> rotate <0,-129.41,0>  pigment {rgb <0.48375,0.48375,0.48375>}}
object {star scale 1.45 translate <10004.13,0,0> rotate <0,0,-25.27,> rotate <0,-311.53,0>  pigment {rgb <0.48375,0.48375,0.48375>}}
object {star scale 1.45 translate <10004.13,0,0> rotate <0,0,5.63,> rotate <0,-354.99,0>  pigment {rgb <0.48375,0.48375,0.48375>}}
object {star scale 1.45 translate <10004.13,0,0> rotate <0,0,-7.80,> rotate <0,-80.99,0>  pigment {rgb <0.48375,0.48375,0.48375>}}
object {star scale 1.45 translate <10004.13,0,0> rotate <0,0,37.75,> rotate <0,-334,0>  pigment {rgb <0.48375,0.48375,0.48375>}}
object {star scale 1.45 translate <10004.13,0,0> rotate <0,0,-67.95,> rotate <0,-184.39,0>  pigment {rgb <0.48375,0.48375,0.48375>}}
object {star scale 1.44 translate <10004.14,0,0> rotate <0,0,-0.50,> rotate <0,-107.96,0>  pigment {rgb <0.4825,0.4825,0.4825>}}
object {star scale 1.44 translate <10004.14,0,0> rotate <0,0,44.33,> rotate <0,-355.1,0>  pigment {rgb <0.4825,0.4825,0.4825>}}
object {star scale 1.44 translate <10004.14,0,0> rotate <0,0,-64.60,> rotate <0,-181.72,0>  pigment {rgb <0.4825,0.4825,0.4825>}}
object {star scale 1.44 translate <10004.14,0,0> rotate <0,0,25.65,> rotate <0,-326.16,0>  pigment {rgb <0.4825,0.4825,0.4825>}}
object {star scale 1.44 translate <10004.14,0,0> rotate <0,0,26.88,> rotate <0,-239.4,0>  pigment {rgb <0.4825,0.4825,0.4825>}}
object {star scale 1.44 translate <10004.14,0,0> rotate <0,0,5.70,> rotate <0,-129.42,0>  pigment {rgb <0.4825,0.4825,0.4825>}}
object {star scale 1.44 translate <10004.14,0,0> rotate <0,0,-81.38,> rotate <0,-341.51,0>  pigment {rgb <0.4825,0.4825,0.4825>}}
object {star scale 1.44 translate <10004.14,0,0> rotate <0,0,-16.73,> rotate <0,-238.46,0>  pigment {rgb <0.4825,0.4825,0.4825>}}
object {star scale 1.44 translate <10004.14,0,0> rotate <0,0,-78.60,> rotate <0,-158.87,0>  pigment {rgb <0.4825,0.4825,0.4825>}}
object {star scale 1.44 translate <10004.15,0,0> rotate <0,0,20.22,> rotate <0,-97.24,0>  pigment {rgb <0.48125,0.48125,0.48125>}}
object {star scale 1.44 translate <10004.15,0,0> rotate <0,0,20.57,> rotate <0,-106.03,0>  pigment {rgb <0.48125,0.48125,0.48125>}}
object {star scale 1.44 translate <10004.15,0,0> rotate <0,0,-7.78,> rotate <0,-334.2,0>  pigment {rgb <0.48125,0.48125,0.48125>}}
object {star scale 1.44 translate <10004.15,0,0> rotate <0,0,-35.25,> rotate <0,-249.1,0>  pigment {rgb <0.48125,0.48125,0.48125>}}
object {star scale 1.44 translate <10004.16,0,0> rotate <0,0,-11.38,> rotate <0,-241.09,0>  pigment {rgb <0.48,0.48,0.48>}}
object {star scale 1.44 translate <10004.16,0,0> rotate <0,0,23.27,> rotate <0,-91.03,0>  pigment {rgb <0.48,0.48,0.48>}}
object {star scale 1.44 translate <10004.16,0,0> rotate <0,0,62.93,> rotate <0,-8.25,0>  pigment {rgb <0.48,0.48,0.48>}}
object {star scale 1.44 translate <10004.16,0,0> rotate <0,0,31.37,> rotate <0,-233.23,0>  pigment {rgb <0.48,0.48,0.48>}}
object {star scale 1.44 translate <10004.16,0,0> rotate <0,0,-36.20,> rotate <0,-57.37,0>  pigment {rgb <0.48,0.48,0.48>}}
object {star scale 1.44 translate <10004.17,0,0> rotate <0,0,31.78,> rotate <0,-112.28,0>  pigment {rgb <0.47875,0.47875,0.47875>}}
object {star scale 1.44 translate <10004.17,0,0> rotate <0,0,-24.18,> rotate <0,-261.59,0>  pigment {rgb <0.47875,0.47875,0.47875>}}
object {star scale 1.44 translate <10004.17,0,0> rotate <0,0,-27.05,> rotate <0,-340.17,0>  pigment {rgb <0.47875,0.47875,0.47875>}}
object {star scale 1.44 translate <10004.18,0,0> rotate <0,0,46.08,> rotate <0,-214.1,0>  pigment {rgb <0.4775,0.4775,0.4775>}}
object {star scale 1.44 translate <10004.18,0,0> rotate <0,0,23.95,> rotate <0,-56.58,0>  pigment {rgb <0.4775,0.4775,0.4775>}}
object {star scale 1.44 translate <10004.18,0,0> rotate <0,0,-10.28,> rotate <0,-213.23,0>  pigment {rgb <0.4775,0.4775,0.4775>}}
object {star scale 1.44 translate <10004.19,0,0> rotate <0,0,82.03,> rotate <0,-251.49,0>  pigment {rgb <0.47625,0.47625,0.47625>}}
object {star scale 1.44 translate <10004.19,0,0> rotate <0,0,57.05,> rotate <0,-333.76,0>  pigment {rgb <0.47625,0.47625,0.47625>}}
object {star scale 1.43 translate <10004.20,0,0> rotate <0,0,5.95,> rotate <0,-82.7,0>  pigment {rgb <0.475,0.475,0.475>}}
object {star scale 1.43 translate <10004.20,0,0> rotate <0,0,20.55,> rotate <0,-281.41,0>  pigment {rgb <0.475,0.475,0.475>}}
object {star scale 1.43 translate <10004.20,0,0> rotate <0,0,12.18,> rotate <0,-341.68,0>  pigment {rgb <0.475,0.475,0.475>}}
object {star scale 1.43 translate <10004.20,0,0> rotate <0,0,-37.05,> rotate <0,-286.6,0>  pigment {rgb <0.475,0.475,0.475>}}
object {star scale 1.43 translate <10004.20,0,0> rotate <0,0,36.70,> rotate <0,-156.97,0>  pigment {rgb <0.475,0.475,0.475>}}
object {star scale 1.43 translate <10004.20,0,0> rotate <0,0,-32.53,> rotate <0,-343.99,0>  pigment {rgb <0.475,0.475,0.475>}}
object {star scale 1.43 translate <10004.20,0,0> rotate <0,0,30.72,> rotate <0,-311.42,0>  pigment {rgb <0.475,0.475,0.475>}}
object {star scale 1.43 translate <10004.20,0,0> rotate <0,0,2.50,> rotate <0,-271.36,0>  pigment {rgb <0.475,0.475,0.475>}}
object {star scale 1.43 translate <10004.20,0,0> rotate <0,0,42.45,> rotate <0,-248.53,0>  pigment {rgb <0.475,0.475,0.475>}}
object {star scale 1.43 translate <10004.20,0,0> rotate <0,0,-22.88,> rotate <0,-119.22,0>  pigment {rgb <0.475,0.475,0.475>}}
object {star scale 1.43 translate <10004.20,0,0> rotate <0,0,-27.83,> rotate <0,-266.89,0>  pigment {rgb <0.475,0.475,0.475>}}
object {star scale 1.43 translate <10004.20,0,0> rotate <0,0,18.58,> rotate <0,-83.05,0>  pigment {rgb <0.475,0.475,0.475>}}
object {star scale 1.43 translate <10004.20,0,0> rotate <0,0,-34.45,> rotate <0,-207.36,0>  pigment {rgb <0.475,0.475,0.475>}}
object {star scale 1.43 translate <10004.21,0,0> rotate <0,0,71.33,> rotate <0,-275.19,0>  pigment {rgb <0.47375,0.47375,0.47375>}}
object {star scale 1.43 translate <10004.21,0,0> rotate <0,0,-62.18,> rotate <0,-283.05,0>  pigment {rgb <0.47375,0.47375,0.47375>}}
object {star scale 1.43 translate <10004.21,0,0> rotate <0,0,59.95,> rotate <0,-52.27,0>  pigment {rgb <0.47375,0.47375,0.47375>}}
object {star scale 1.43 translate <10004.21,0,0> rotate <0,0,62.98,> rotate <0,-307.4,0>  pigment {rgb <0.47375,0.47375,0.47375>}}
object {star scale 1.43 translate <10004.21,0,0> rotate <0,0,-36.80,> rotate <0,-241.65,0>  pigment {rgb <0.47375,0.47375,0.47375>}}
object {star scale 1.43 translate <10004.22,0,0> rotate <0,0,-9.08,> rotate <0,-348.97,0>  pigment {rgb <0.4725,0.4725,0.4725>}}
object {star scale 1.43 translate <10004.22,0,0> rotate <0,0,22.30,> rotate <0,-66.34,0>  pigment {rgb <0.4725,0.4725,0.4725>}}
object {star scale 1.43 translate <10004.22,0,0> rotate <0,0,38.32,> rotate <0,-42.65,0>  pigment {rgb <0.4725,0.4725,0.4725>}}
object {star scale 1.43 translate <10004.22,0,0> rotate <0,0,-23.25,> rotate <0,-56.71,0>  pigment {rgb <0.4725,0.4725,0.4725>}}
object {star scale 1.43 translate <10004.22,0,0> rotate <0,0,-4.75,> rotate <0,-281.79,0>  pigment {rgb <0.4725,0.4725,0.4725>}}
object {star scale 1.43 translate <10004.22,0,0> rotate <0,0,-6.05,> rotate <0,-348.58,0>  pigment {rgb <0.4725,0.4725,0.4725>}}
object {star scale 1.43 translate <10004.23,0,0> rotate <0,0,11.87,> rotate <0,-134.62,0>  pigment {rgb <0.47125,0.47125,0.47125>}}
object {star scale 1.43 translate <10004.23,0,0> rotate <0,0,10.53,> rotate <0,-233.7,0>  pigment {rgb <0.47125,0.47125,0.47125>}}
object {star scale 1.43 translate <10004.23,0,0> rotate <0,0,-77.52,> rotate <0,-250.77,0>  pigment {rgb <0.47125,0.47125,0.47125>}}
object {star scale 1.43 translate <10004.23,0,0> rotate <0,0,-48.12,> rotate <0,-118.33,0>  pigment {rgb <0.47125,0.47125,0.47125>}}
object {star scale 1.43 translate <10004.23,0,0> rotate <0,0,-34.70,> rotate <0,-247.85,0>  pigment {rgb <0.47125,0.47125,0.47125>}}
object {star scale 1.43 translate <10004.23,0,0> rotate <0,0,49.32,> rotate <0,-326.7,0>  pigment {rgb <0.47125,0.47125,0.47125>}}
object {star scale 1.43 translate <10004.23,0,0> rotate <0,0,48.20,> rotate <0,-54.13,0>  pigment {rgb <0.47125,0.47125,0.47125>}}
object {star scale 1.43 translate <10004.23,0,0> rotate <0,0,39.40,> rotate <0,-319.35,0>  pigment {rgb <0.47125,0.47125,0.47125>}}
object {star scale 1.43 translate <10004.23,0,0> rotate <0,0,-33.03,> rotate <0,-206.42,0>  pigment {rgb <0.47125,0.47125,0.47125>}}
object {star scale 1.43 translate <10004.23,0,0> rotate <0,0,-65.37,> rotate <0,-321.61,0>  pigment {rgb <0.47125,0.47125,0.47125>}}
object {star scale 1.43 translate <10004.23,0,0> rotate <0,0,41.27,> rotate <0,-69.18,0>  pigment {rgb <0.47125,0.47125,0.47125>}}
object {star scale 1.43 translate <10004.23,0,0> rotate <0,0,86.25,> rotate <0,-17.19,0>  pigment {rgb <0.47125,0.47125,0.47125>}}
object {star scale 1.43 translate <10004.24,0,0> rotate <0,0,47.25,> rotate <0,-17.38,0>  pigment {rgb <0.47,0.47,0.47>}}
object {star scale 1.43 translate <10004.24,0,0> rotate <0,0,-12.50,> rotate <0,-304.41,0>  pigment {rgb <0.47,0.47,0.47>}}
object {star scale 1.43 translate <10004.24,0,0> rotate <0,0,-47.70,> rotate <0,-36.75,0>  pigment {rgb <0.47,0.47,0.47>}}
object {star scale 1.43 translate <10004.24,0,0> rotate <0,0,-13.87,> rotate <0,-41.03,0>  pigment {rgb <0.47,0.47,0.47>}}
object {star scale 1.43 translate <10004.24,0,0> rotate <0,0,-51.48,> rotate <0,-64,0>  pigment {rgb <0.47,0.47,0.47>}}
object {star scale 1.43 translate <10004.24,0,0> rotate <0,0,-64.88,> rotate <0,-5.02,0>  pigment {rgb <0.47,0.47,0.47>}}
object {star scale 1.43 translate <10004.24,0,0> rotate <0,0,-31.08,> rotate <0,-156.79,0>  pigment {rgb <0.47,0.47,0.47>}}
object {star scale 1.43 translate <10004.25,0,0> rotate <0,0,10.15,> rotate <0,-68.92,0>  pigment {rgb <0.46875,0.46875,0.46875>}}
object {star scale 1.43 translate <10004.25,0,0> rotate <0,0,-47.35,> rotate <0,-122.37,0>  pigment {rgb <0.46875,0.46875,0.46875>}}
object {star scale 1.43 translate <10004.25,0,0> rotate <0,0,-68.87,> rotate <0,-18.94,0>  pigment {rgb <0.46875,0.46875,0.46875>}}
object {star scale 1.43 translate <10004.25,0,0> rotate <0,0,75.68,> rotate <0,-216.88,0>  pigment {rgb <0.46875,0.46875,0.46875>}}
object {star scale 1.43 translate <10004.25,0,0> rotate <0,0,44.93,> rotate <0,-242.19,0>  pigment {rgb <0.46875,0.46875,0.46875>}}
object {star scale 1.43 translate <10004.25,0,0> rotate <0,0,-12.87,> rotate <0,-265.35,0>  pigment {rgb <0.46875,0.46875,0.46875>}}
object {star scale 1.43 translate <10004.25,0,0> rotate <0,0,-7.83,> rotate <0,-94.93,0>  pigment {rgb <0.46875,0.46875,0.46875>}}
object {star scale 1.43 translate <10004.25,0,0> rotate <0,0,28.88,> rotate <0,-115.83,0>  pigment {rgb <0.46875,0.46875,0.46875>}}
object {star scale 1.43 translate <10004.25,0,0> rotate <0,0,43.20,> rotate <0,-125.71,0>  pigment {rgb <0.46875,0.46875,0.46875>}}
object {star scale 1.43 translate <10004.25,0,0> rotate <0,0,45.95,> rotate <0,-89.98,0>  pigment {rgb <0.46875,0.46875,0.46875>}}
object {star scale 1.42 translate <10004.26,0,0> rotate <0,0,1.55,> rotate <0,-210.41,0>  pigment {rgb <0.4675,0.4675,0.4675>}}
object {star scale 1.42 translate <10004.26,0,0> rotate <0,0,-49.90,> rotate <0,-196.73,0>  pigment {rgb <0.4675,0.4675,0.4675>}}
object {star scale 1.42 translate <10004.26,0,0> rotate <0,0,-47.87,> rotate <0,-229.63,0>  pigment {rgb <0.4675,0.4675,0.4675>}}
object {star scale 1.42 translate <10004.26,0,0> rotate <0,0,-55.60,> rotate <0,-159.83,0>  pigment {rgb <0.4675,0.4675,0.4675>}}
object {star scale 1.42 translate <10004.26,0,0> rotate <0,0,-40.18,> rotate <0,-193.36,0>  pigment {rgb <0.4675,0.4675,0.4675>}}
object {star scale 1.42 translate <10004.26,0,0> rotate <0,0,-13.87,> rotate <0,-331.61,0>  pigment {rgb <0.4675,0.4675,0.4675>}}
object {star scale 1.42 translate <10004.26,0,0> rotate <0,0,-21.63,> rotate <0,-53.45,0>  pigment {rgb <0.4675,0.4675,0.4675>}}
object {star scale 1.42 translate <10004.26,0,0> rotate <0,0,10.12,> rotate <0,-41.23,0>  pigment {rgb <0.4675,0.4675,0.4675>}}
object {star scale 1.42 translate <10004.26,0,0> rotate <0,0,27.88,> rotate <0,-197.97,0>  pigment {rgb <0.4675,0.4675,0.4675>}}
object {star scale 1.42 translate <10004.26,0,0> rotate <0,0,-43.07,> rotate <0,-49.98,0>  pigment {rgb <0.4675,0.4675,0.4675>}}
object {star scale 1.42 translate <10004.26,0,0> rotate <0,0,9.15,> rotate <0,-26.35,0>  pigment {rgb <0.4675,0.4675,0.4675>}}
object {star scale 1.42 translate <10004.27,0,0> rotate <0,0,-43.53,> rotate <0,-346.72,0>  pigment {rgb <0.46625,0.46625,0.46625>}}
object {star scale 1.42 translate <10004.27,0,0> rotate <0,0,16.13,> rotate <0,-311.66,0>  pigment {rgb <0.46625,0.46625,0.46625>}}
object {star scale 1.42 translate <10004.27,0,0> rotate <0,0,12.52,> rotate <0,-69.54,0>  pigment {rgb <0.46625,0.46625,0.46625>}}
object {star scale 1.42 translate <10004.27,0,0> rotate <0,0,-8.75,> rotate <0,-77.29,0>  pigment {rgb <0.46625,0.46625,0.46625>}}
object {star scale 1.42 translate <10004.27,0,0> rotate <0,0,-79.30,> rotate <0,-184.58,0>  pigment {rgb <0.46625,0.46625,0.46625>}}
object {star scale 1.42 translate <10004.27,0,0> rotate <0,0,43.27,> rotate <0,-354.53,0>  pigment {rgb <0.46625,0.46625,0.46625>}}
object {star scale 1.42 translate <10004.27,0,0> rotate <0,0,8.47,> rotate <0,-37.04,0>  pigment {rgb <0.46625,0.46625,0.46625>}}
object {star scale 1.42 translate <10004.27,0,0> rotate <0,0,41.35,> rotate <0,-188.44,0>  pigment {rgb <0.46625,0.46625,0.46625>}}
object {star scale 1.42 translate <10004.27,0,0> rotate <0,0,-16.83,> rotate <0,-320.56,0>  pigment {rgb <0.46625,0.46625,0.46625>}}
object {star scale 1.42 translate <10004.27,0,0> rotate <0,0,7.88,> rotate <0,-15.73,0>  pigment {rgb <0.46625,0.46625,0.46625>}}
object {star scale 1.42 translate <10004.27,0,0> rotate <0,0,6.38,> rotate <0,-351.99,0>  pigment {rgb <0.46625,0.46625,0.46625>}}
object {star scale 1.42 translate <10004.28,0,0> rotate <0,0,-32.35,> rotate <0,-337.88,0>  pigment {rgb <0.465,0.465,0.465>}}
object {star scale 1.42 translate <10004.28,0,0> rotate <0,0,-33.90,> rotate <0,-178.23,0>  pigment {rgb <0.465,0.465,0.465>}}
object {star scale 1.42 translate <10004.28,0,0> rotate <0,0,-16.62,> rotate <0,-247.78,0>  pigment {rgb <0.465,0.465,0.465>}}
object {star scale 1.42 translate <10004.28,0,0> rotate <0,0,61.13,> rotate <0,-326.36,0>  pigment {rgb <0.465,0.465,0.465>}}
object {star scale 1.42 translate <10004.28,0,0> rotate <0,0,56.57,> rotate <0,-303.35,0>  pigment {rgb <0.465,0.465,0.465>}}
object {star scale 1.42 translate <10004.28,0,0> rotate <0,0,22.97,> rotate <0,-70.56,0>  pigment {rgb <0.465,0.465,0.465>}}
object {star scale 1.42 translate <10004.28,0,0> rotate <0,0,-44.80,> rotate <0,-290.8,0>  pigment {rgb <0.465,0.465,0.465>}}
object {star scale 1.42 translate <10004.28,0,0> rotate <0,0,0.40,> rotate <0,-54.22,0>  pigment {rgb <0.465,0.465,0.465>}}
object {star scale 1.42 translate <10004.28,0,0> rotate <0,0,-38.63,> rotate <0,-264.14,0>  pigment {rgb <0.465,0.465,0.465>}}
object {star scale 1.42 translate <10004.28,0,0> rotate <0,0,81.32,> rotate <0,-144.27,0>  pigment {rgb <0.465,0.465,0.465>}}
object {star scale 1.42 translate <10004.29,0,0> rotate <0,0,64.62,> rotate <0,-330.95,0>  pigment {rgb <0.46375,0.46375,0.46375>}}
object {star scale 1.42 translate <10004.29,0,0> rotate <0,0,-13.18,> rotate <0,-79.89,0>  pigment {rgb <0.46375,0.46375,0.46375>}}
object {star scale 1.42 translate <10004.29,0,0> rotate <0,0,3.38,> rotate <0,-130.81,0>  pigment {rgb <0.46375,0.46375,0.46375>}}
object {star scale 1.42 translate <10004.29,0,0> rotate <0,0,8.90,> rotate <0,-63.88,0>  pigment {rgb <0.46375,0.46375,0.46375>}}
object {star scale 1.42 translate <10004.29,0,0> rotate <0,0,50.35,> rotate <0,-61.65,0>  pigment {rgb <0.46375,0.46375,0.46375>}}
object {star scale 1.42 translate <10004.29,0,0> rotate <0,0,22.82,> rotate <0,-66.58,0>  pigment {rgb <0.46375,0.46375,0.46375>}}
object {star scale 1.42 translate <10004.29,0,0> rotate <0,0,-29.87,> rotate <0,-261.84,0>  pigment {rgb <0.46375,0.46375,0.46375>}}
object {star scale 1.42 translate <10004.30,0,0> rotate <0,0,17.93,> rotate <0,-66.37,0>  pigment {rgb <0.4625,0.4625,0.4625>}}
object {star scale 1.42 translate <10004.30,0,0> rotate <0,0,-49.23,> rotate <0,-119.56,0>  pigment {rgb <0.4625,0.4625,0.4625>}}
object {star scale 1.42 translate <10004.30,0,0> rotate <0,0,-29.58,> rotate <0,-271.25,0>  pigment {rgb <0.4625,0.4625,0.4625>}}
object {star scale 1.42 translate <10004.30,0,0> rotate <0,0,36.90,> rotate <0,-283.63,0>  pigment {rgb <0.4625,0.4625,0.4625>}}
object {star scale 1.42 translate <10004.30,0,0> rotate <0,0,-44.65,> rotate <0,-108.38,0>  pigment {rgb <0.4625,0.4625,0.4625>}}
object {star scale 1.42 translate <10004.30,0,0> rotate <0,0,77.80,> rotate <0,-236.01,0>  pigment {rgb <0.4625,0.4625,0.4625>}}
object {star scale 1.42 translate <10004.30,0,0> rotate <0,0,24.47,> rotate <0,-56.3,0>  pigment {rgb <0.4625,0.4625,0.4625>}}
object {star scale 1.42 translate <10004.30,0,0> rotate <0,0,-29.37,> rotate <0,-14.65,0>  pigment {rgb <0.4625,0.4625,0.4625>}}
object {star scale 1.42 translate <10004.30,0,0> rotate <0,0,-16.20,> rotate <0,-188.02,0>  pigment {rgb <0.4625,0.4625,0.4625>}}
object {star scale 1.42 translate <10004.30,0,0> rotate <0,0,33.17,> rotate <0,-332.5,0>  pigment {rgb <0.4625,0.4625,0.4625>}}
object {star scale 1.42 translate <10004.30,0,0> rotate <0,0,-0.82,> rotate <0,-174.24,0>  pigment {rgb <0.4625,0.4625,0.4625>}}
object {star scale 1.42 translate <10004.30,0,0> rotate <0,0,66.33,> rotate <0,-73.51,0>  pigment {rgb <0.4625,0.4625,0.4625>}}
object {star scale 1.42 translate <10004.31,0,0> rotate <0,0,37.38,> rotate <0,-231.13,0>  pigment {rgb <0.46125,0.46125,0.46125>}}
object {star scale 1.42 translate <10004.31,0,0> rotate <0,0,-13.55,> rotate <0,-131.59,0>  pigment {rgb <0.46125,0.46125,0.46125>}}
object {star scale 1.42 translate <10004.31,0,0> rotate <0,0,-1.10,> rotate <0,-309.58,0>  pigment {rgb <0.46125,0.46125,0.46125>}}
object {star scale 1.42 translate <10004.31,0,0> rotate <0,0,-19.67,> rotate <0,-70.11,0>  pigment {rgb <0.46125,0.46125,0.46125>}}
object {star scale 1.42 translate <10004.31,0,0> rotate <0,0,-63.78,> rotate <0,-177.42,0>  pigment {rgb <0.46125,0.46125,0.46125>}}
object {star scale 1.42 translate <10004.31,0,0> rotate <0,0,-83.67,> rotate <0,-216.73,0>  pigment {rgb <0.46125,0.46125,0.46125>}}
object {star scale 1.42 translate <10004.31,0,0> rotate <0,0,22.98,> rotate <0,-142.93,0>  pigment {rgb <0.46125,0.46125,0.46125>}}
object {star scale 1.41 translate <10004.32,0,0> rotate <0,0,-63.30,> rotate <0,-180.76,0>  pigment {rgb <0.46,0.46,0.46>}}
object {star scale 1.41 translate <10004.32,0,0> rotate <0,0,-59.75,> rotate <0,-130.15,0>  pigment {rgb <0.46,0.46,0.46>}}
object {star scale 1.41 translate <10004.32,0,0> rotate <0,0,8.93,> rotate <0,-112.04,0>  pigment {rgb <0.46,0.46,0.46>}}
object {star scale 1.41 translate <10004.32,0,0> rotate <0,0,-18.47,> rotate <0,-246.76,0>  pigment {rgb <0.46,0.46,0.46>}}
object {star scale 1.41 translate <10004.32,0,0> rotate <0,0,-56.40,> rotate <0,-215.08,0>  pigment {rgb <0.46,0.46,0.46>}}
object {star scale 1.41 translate <10004.32,0,0> rotate <0,0,-20.87,> rotate <0,-241.85,0>  pigment {rgb <0.46,0.46,0.46>}}
object {star scale 1.41 translate <10004.32,0,0> rotate <0,0,17.35,> rotate <0,-326.13,0>  pigment {rgb <0.46,0.46,0.46>}}
object {star scale 1.41 translate <10004.32,0,0> rotate <0,0,40.68,> rotate <0,-28.72,0>  pigment {rgb <0.46,0.46,0.46>}}
object {star scale 1.41 translate <10004.32,0,0> rotate <0,0,-48.95,> rotate <0,-193.28,0>  pigment {rgb <0.46,0.46,0.46>}}
object {star scale 1.41 translate <10004.32,0,0> rotate <0,0,36.07,> rotate <0,-274.96,0>  pigment {rgb <0.46,0.46,0.46>}}
object {star scale 1.41 translate <10004.33,0,0> rotate <0,0,2.77,> rotate <0,-30.51,0>  pigment {rgb <0.45875,0.45875,0.45875>}}
object {star scale 1.41 translate <10004.33,0,0> rotate <0,0,-12.85,> rotate <0,-260.2,0>  pigment {rgb <0.45875,0.45875,0.45875>}}
object {star scale 1.41 translate <10004.33,0,0> rotate <0,0,-43.58,> rotate <0,-222.91,0>  pigment {rgb <0.45875,0.45875,0.45875>}}
object {star scale 1.41 translate <10004.33,0,0> rotate <0,0,-42.57,> rotate <0,-234.51,0>  pigment {rgb <0.45875,0.45875,0.45875>}}
object {star scale 1.41 translate <10004.33,0,0> rotate <0,0,-33.03,> rotate <0,-326.24,0>  pigment {rgb <0.45875,0.45875,0.45875>}}
object {star scale 1.41 translate <10004.33,0,0> rotate <0,0,55.15,> rotate <0,-17.78,0>  pigment {rgb <0.45875,0.45875,0.45875>}}
object {star scale 1.41 translate <10004.33,0,0> rotate <0,0,29.50,> rotate <0,-93.84,0>  pigment {rgb <0.45875,0.45875,0.45875>}}
object {star scale 1.41 translate <10004.33,0,0> rotate <0,0,-30.15,> rotate <0,-229.46,0>  pigment {rgb <0.45875,0.45875,0.45875>}}
object {star scale 1.41 translate <10004.33,0,0> rotate <0,0,-57.53,> rotate <0,-139.05,0>  pigment {rgb <0.45875,0.45875,0.45875>}}
object {star scale 1.41 translate <10004.34,0,0> rotate <0,0,-49.35,> rotate <0,-144.21,0>  pigment {rgb <0.4575,0.4575,0.4575>}}
object {star scale 1.41 translate <10004.34,0,0> rotate <0,0,-23.42,> rotate <0,-97.96,0>  pigment {rgb <0.4575,0.4575,0.4575>}}
object {star scale 1.41 translate <10004.34,0,0> rotate <0,0,-68.62,> rotate <0,-121.99,0>  pigment {rgb <0.4575,0.4575,0.4575>}}
object {star scale 1.41 translate <10004.34,0,0> rotate <0,0,58.42,> rotate <0,-337.29,0>  pigment {rgb <0.4575,0.4575,0.4575>}}
object {star scale 1.41 translate <10004.34,0,0> rotate <0,0,-2.98,> rotate <0,-122.15,0>  pigment {rgb <0.4575,0.4575,0.4575>}}
object {star scale 1.41 translate <10004.34,0,0> rotate <0,0,-77.48,> rotate <0,-125.16,0>  pigment {rgb <0.4575,0.4575,0.4575>}}
object {star scale 1.41 translate <10004.34,0,0> rotate <0,0,-63.67,> rotate <0,-272.14,0>  pigment {rgb <0.4575,0.4575,0.4575>}}
object {star scale 1.41 translate <10004.34,0,0> rotate <0,0,-65.73,> rotate <0,-86.2,0>  pigment {rgb <0.4575,0.4575,0.4575>}}
object {star scale 1.41 translate <10004.34,0,0> rotate <0,0,-45.60,> rotate <0,-210.43,0>  pigment {rgb <0.4575,0.4575,0.4575>}}
object {star scale 1.41 translate <10004.34,0,0> rotate <0,0,4.15,> rotate <0,-261.63,0>  pigment {rgb <0.4575,0.4575,0.4575>}}
object {star scale 1.41 translate <10004.34,0,0> rotate <0,0,47.72,> rotate <0,-337.38,0>  pigment {rgb <0.4575,0.4575,0.4575>}}
object {star scale 1.41 translate <10004.35,0,0> rotate <0,0,37.60,> rotate <0,-281.2,0>  pigment {rgb <0.45625,0.45625,0.45625>}}
object {star scale 1.41 translate <10004.35,0,0> rotate <0,0,5.83,> rotate <0,-132.11,0>  pigment {rgb <0.45625,0.45625,0.45625>}}
object {star scale 1.41 translate <10004.35,0,0> rotate <0,0,-62.97,> rotate <0,-7.89,0>  pigment {rgb <0.45625,0.45625,0.45625>}}
object {star scale 1.41 translate <10004.35,0,0> rotate <0,0,-45.38,> rotate <0,-216.55,0>  pigment {rgb <0.45625,0.45625,0.45625>}}
object {star scale 1.41 translate <10004.35,0,0> rotate <0,0,58.42,> rotate <0,-104.32,0>  pigment {rgb <0.45625,0.45625,0.45625>}}
object {star scale 1.41 translate <10004.35,0,0> rotate <0,0,18.18,> rotate <0,-281.75,0>  pigment {rgb <0.45625,0.45625,0.45625>}}
object {star scale 1.41 translate <10004.35,0,0> rotate <0,0,86.58,> rotate <0,-263.05,0>  pigment {rgb <0.45625,0.45625,0.45625>}}
object {star scale 1.41 translate <10004.35,0,0> rotate <0,0,-35.28,> rotate <0,-299.93,0>  pigment {rgb <0.45625,0.45625,0.45625>}}
object {star scale 1.41 translate <10004.35,0,0> rotate <0,0,-41.18,> rotate <0,-211.51,0>  pigment {rgb <0.45625,0.45625,0.45625>}}
object {star scale 1.41 translate <10004.35,0,0> rotate <0,0,38.13,> rotate <0,-289.09,0>  pigment {rgb <0.45625,0.45625,0.45625>}}
object {star scale 1.41 translate <10004.35,0,0> rotate <0,0,28.27,> rotate <0,-186.73,0>  pigment {rgb <0.45625,0.45625,0.45625>}}
object {star scale 1.41 translate <10004.35,0,0> rotate <0,0,19.73,> rotate <0,-47.91,0>  pigment {rgb <0.45625,0.45625,0.45625>}}
object {star scale 1.41 translate <10004.35,0,0> rotate <0,0,48.00,> rotate <0,-52.64,0>  pigment {rgb <0.45625,0.45625,0.45625>}}
object {star scale 1.41 translate <10004.36,0,0> rotate <0,0,-57.47,> rotate <0,-10.84,0>  pigment {rgb <0.455,0.455,0.455>}}
object {star scale 1.41 translate <10004.36,0,0> rotate <0,0,20.82,> rotate <0,-272.19,0>  pigment {rgb <0.455,0.455,0.455>}}
object {star scale 1.41 translate <10004.36,0,0> rotate <0,0,-35.28,> rotate <0,-89.39,0>  pigment {rgb <0.455,0.455,0.455>}}
object {star scale 1.41 translate <10004.36,0,0> rotate <0,0,-1.30,> rotate <0,-294.18,0>  pigment {rgb <0.455,0.455,0.455>}}
object {star scale 1.41 translate <10004.36,0,0> rotate <0,0,33.72,> rotate <0,-9.22,0>  pigment {rgb <0.455,0.455,0.455>}}
object {star scale 1.41 translate <10004.36,0,0> rotate <0,0,-12.95,> rotate <0,-78.31,0>  pigment {rgb <0.455,0.455,0.455>}}
object {star scale 1.41 translate <10004.36,0,0> rotate <0,0,18.02,> rotate <0,-295.03,0>  pigment {rgb <0.455,0.455,0.455>}}
object {star scale 1.41 translate <10004.36,0,0> rotate <0,0,-61.48,> rotate <0,-275.81,0>  pigment {rgb <0.455,0.455,0.455>}}
object {star scale 1.41 translate <10004.36,0,0> rotate <0,0,8.90,> rotate <0,-72.65,0>  pigment {rgb <0.455,0.455,0.455>}}
object {star scale 1.41 translate <10004.36,0,0> rotate <0,0,8.85,> rotate <0,-33.25,0>  pigment {rgb <0.455,0.455,0.455>}}
object {star scale 1.41 translate <10004.36,0,0> rotate <0,0,-35.13,> rotate <0,-94.14,0>  pigment {rgb <0.455,0.455,0.455>}}
object {star scale 1.41 translate <10004.36,0,0> rotate <0,0,22.08,> rotate <0,-61.18,0>  pigment {rgb <0.455,0.455,0.455>}}
object {star scale 1.41 translate <10004.37,0,0> rotate <0,0,-5.53,> rotate <0,-197.49,0>  pigment {rgb <0.45375,0.45375,0.45375>}}
object {star scale 1.41 translate <10004.37,0,0> rotate <0,0,77.72,> rotate <0,-302.22,0>  pigment {rgb <0.45375,0.45375,0.45375>}}
object {star scale 1.41 translate <10004.37,0,0> rotate <0,0,10.00,> rotate <0,-151.98,0>  pigment {rgb <0.45375,0.45375,0.45375>}}
object {star scale 1.41 translate <10004.37,0,0> rotate <0,0,-37.82,> rotate <0,-353.25,0>  pigment {rgb <0.45375,0.45375,0.45375>}}
object {star scale 1.41 translate <10004.37,0,0> rotate <0,0,10.17,> rotate <0,-253.5,0>  pigment {rgb <0.45375,0.45375,0.45375>}}
object {star scale 1.41 translate <10004.37,0,0> rotate <0,0,17.48,> rotate <0,-295.26,0>  pigment {rgb <0.45375,0.45375,0.45375>}}
object {star scale 1.41 translate <10004.37,0,0> rotate <0,0,29.32,> rotate <0,-9.64,0>  pigment {rgb <0.45375,0.45375,0.45375>}}
object {star scale 1.40 translate <10004.38,0,0> rotate <0,0,-53.45,> rotate <0,-319.96,0>  pigment {rgb <0.4525,0.4525,0.4525>}}
object {star scale 1.40 translate <10004.38,0,0> rotate <0,0,39.13,> rotate <0,-288.44,0>  pigment {rgb <0.4525,0.4525,0.4525>}}
object {star scale 1.40 translate <10004.38,0,0> rotate <0,0,-42.22,> rotate <0,-165.04,0>  pigment {rgb <0.4525,0.4525,0.4525>}}
object {star scale 1.40 translate <10004.38,0,0> rotate <0,0,-5.45,> rotate <0,-73.23,0>  pigment {rgb <0.4525,0.4525,0.4525>}}
object {star scale 1.40 translate <10004.38,0,0> rotate <0,0,-17.05,> rotate <0,-104.04,0>  pigment {rgb <0.4525,0.4525,0.4525>}}
object {star scale 1.40 translate <10004.38,0,0> rotate <0,0,-52.98,> rotate <0,-98.74,0>  pigment {rgb <0.4525,0.4525,0.4525>}}
object {star scale 1.40 translate <10004.38,0,0> rotate <0,0,2.35,> rotate <0,-120.57,0>  pigment {rgb <0.4525,0.4525,0.4525>}}
object {star scale 1.40 translate <10004.39,0,0> rotate <0,0,-54.98,> rotate <0,-329.48,0>  pigment {rgb <0.45125,0.45125,0.45125>}}
object {star scale 1.40 translate <10004.39,0,0> rotate <0,0,30.25,> rotate <0,-107.79,0>  pigment {rgb <0.45125,0.45125,0.45125>}}
object {star scale 1.40 translate <10004.39,0,0> rotate <0,0,-24.95,> rotate <0,-109.68,0>  pigment {rgb <0.45125,0.45125,0.45125>}}
object {star scale 1.40 translate <10004.39,0,0> rotate <0,0,30.20,> rotate <0,-269.63,0>  pigment {rgb <0.45125,0.45125,0.45125>}}
object {star scale 1.40 translate <10004.39,0,0> rotate <0,0,-53.62,> rotate <0,-102.46,0>  pigment {rgb <0.45125,0.45125,0.45125>}}
object {star scale 1.40 translate <10004.39,0,0> rotate <0,0,23.42,> rotate <0,-14.3,0>  pigment {rgb <0.45125,0.45125,0.45125>}}
object {star scale 1.40 translate <10004.39,0,0> rotate <0,0,-20.63,> rotate <0,-351.51,0>  pigment {rgb <0.45125,0.45125,0.45125>}}
object {star scale 1.40 translate <10004.40,0,0> rotate <0,0,-21.10,> rotate <0,-260.25,0>  pigment {rgb <0.45,0.45,0.45>}}
object {star scale 1.40 translate <10004.40,0,0> rotate <0,0,-19.25,> rotate <0,-122.26,0>  pigment {rgb <0.45,0.45,0.45>}}
object {star scale 1.40 translate <10004.40,0,0> rotate <0,0,-9.18,> rotate <0,-349.48,0>  pigment {rgb <0.45,0.45,0.45>}}
object {star scale 1.40 translate <10004.40,0,0> rotate <0,0,20.27,> rotate <0,-88.6,0>  pigment {rgb <0.45,0.45,0.45>}}
object {star scale 1.40 translate <10004.40,0,0> rotate <0,0,2.08,> rotate <0,-225.73,0>  pigment {rgb <0.45,0.45,0.45>}}
object {star scale 1.40 translate <10004.40,0,0> rotate <0,0,26.10,> rotate <0,-262.68,0>  pigment {rgb <0.45,0.45,0.45>}}
object {star scale 1.40 translate <10004.40,0,0> rotate <0,0,-18.93,> rotate <0,-3.66,0>  pigment {rgb <0.45,0.45,0.45>}}
object {star scale 1.40 translate <10004.40,0,0> rotate <0,0,-46.32,> rotate <0,-28.41,0>  pigment {rgb <0.45,0.45,0.45>}}
object {star scale 1.40 translate <10004.40,0,0> rotate <0,0,11.43,> rotate <0,-146.89,0>  pigment {rgb <0.45,0.45,0.45>}}
object {star scale 1.40 translate <10004.41,0,0> rotate <0,0,31.53,> rotate <0,-169.55,0>  pigment {rgb <0.44875,0.44875,0.44875>}}
object {star scale 1.40 translate <10004.41,0,0> rotate <0,0,75.38,> rotate <0,-346.98,0>  pigment {rgb <0.44875,0.44875,0.44875>}}
object {star scale 1.40 translate <10004.41,0,0> rotate <0,0,20.18,> rotate <0,-165.58,0>  pigment {rgb <0.44875,0.44875,0.44875>}}
object {star scale 1.40 translate <10004.41,0,0> rotate <0,0,9.48,> rotate <0,-83.7,0>  pigment {rgb <0.44875,0.44875,0.44875>}}
object {star scale 1.40 translate <10004.41,0,0> rotate <0,0,-50.47,> rotate <0,-218.15,0>  pigment {rgb <0.44875,0.44875,0.44875>}}
object {star scale 1.40 translate <10004.41,0,0> rotate <0,0,-39.50,> rotate <0,-215.76,0>  pigment {rgb <0.44875,0.44875,0.44875>}}
object {star scale 1.40 translate <10004.41,0,0> rotate <0,0,14.77,> rotate <0,-91.9,0>  pigment {rgb <0.44875,0.44875,0.44875>}}
object {star scale 1.40 translate <10004.41,0,0> rotate <0,0,23.40,> rotate <0,-351.34,0>  pigment {rgb <0.44875,0.44875,0.44875>}}
object {star scale 1.40 translate <10004.41,0,0> rotate <0,0,-32.53,> rotate <0,-349.71,0>  pigment {rgb <0.44875,0.44875,0.44875>}}
object {star scale 1.40 translate <10004.41,0,0> rotate <0,0,-18.23,> rotate <0,-99.48,0>  pigment {rgb <0.44875,0.44875,0.44875>}}
object {star scale 1.40 translate <10004.41,0,0> rotate <0,0,-27.97,> rotate <0,-222.57,0>  pigment {rgb <0.44875,0.44875,0.44875>}}
object {star scale 1.40 translate <10004.41,0,0> rotate <0,0,-5.03,> rotate <0,-311.93,0>  pigment {rgb <0.44875,0.44875,0.44875>}}
object {star scale 1.40 translate <10004.41,0,0> rotate <0,0,-6.02,> rotate <0,-0.49,0>  pigment {rgb <0.44875,0.44875,0.44875>}}
object {star scale 1.40 translate <10004.42,0,0> rotate <0,0,-40.30,> rotate <0,-44.57,0>  pigment {rgb <0.4475,0.4475,0.4475>}}
object {star scale 1.40 translate <10004.42,0,0> rotate <0,0,-26.35,> rotate <0,-108.57,0>  pigment {rgb <0.4475,0.4475,0.4475>}}
object {star scale 1.40 translate <10004.42,0,0> rotate <0,0,15.08,> rotate <0,-310.86,0>  pigment {rgb <0.4475,0.4475,0.4475>}}
object {star scale 1.40 translate <10004.42,0,0> rotate <0,0,52.23,> rotate <0,-335.89,0>  pigment {rgb <0.4475,0.4475,0.4475>}}
object {star scale 1.40 translate <10004.42,0,0> rotate <0,0,32.18,> rotate <0,-305.97,0>  pigment {rgb <0.4475,0.4475,0.4475>}}
object {star scale 1.40 translate <10004.42,0,0> rotate <0,0,24.67,> rotate <0,-292.18,0>  pigment {rgb <0.4475,0.4475,0.4475>}}
object {star scale 1.40 translate <10004.42,0,0> rotate <0,0,65.53,> rotate <0,-57.38,0>  pigment {rgb <0.4475,0.4475,0.4475>}}
object {star scale 1.40 translate <10004.42,0,0> rotate <0,0,-12.10,> rotate <0,-56.54,0>  pigment {rgb <0.4475,0.4475,0.4475>}}
object {star scale 1.40 translate <10004.43,0,0> rotate <0,0,53.77,> rotate <0,-74.32,0>  pigment {rgb <0.44625,0.44625,0.44625>}}
object {star scale 1.40 translate <10004.43,0,0> rotate <0,0,13.72,> rotate <0,-220.29,0>  pigment {rgb <0.44625,0.44625,0.44625>}}
object {star scale 1.40 translate <10004.43,0,0> rotate <0,0,-41.87,> rotate <0,-70.14,0>  pigment {rgb <0.44625,0.44625,0.44625>}}
object {star scale 1.40 translate <10004.43,0,0> rotate <0,0,-40.35,> rotate <0,-123.51,0>  pigment {rgb <0.44625,0.44625,0.44625>}}
object {star scale 1.40 translate <10004.43,0,0> rotate <0,0,-7.67,> rotate <0,-63.82,0>  pigment {rgb <0.44625,0.44625,0.44625>}}
object {star scale 1.40 translate <10004.43,0,0> rotate <0,0,7.35,> rotate <0,-236.61,0>  pigment {rgb <0.44625,0.44625,0.44625>}}
object {star scale 1.40 translate <10004.43,0,0> rotate <0,0,3.35,> rotate <0,-129.69,0>  pigment {rgb <0.44625,0.44625,0.44625>}}
object {star scale 1.40 translate <10004.43,0,0> rotate <0,0,7.58,> rotate <0,-12.17,0>  pigment {rgb <0.44625,0.44625,0.44625>}}
object {star scale 1.39 translate <10004.44,0,0> rotate <0,0,1.30,> rotate <0,-270.44,0>  pigment {rgb <0.445,0.445,0.445>}}
object {star scale 1.39 translate <10004.44,0,0> rotate <0,0,4.60,> rotate <0,-95.94,0>  pigment {rgb <0.445,0.445,0.445>}}
object {star scale 1.39 translate <10004.44,0,0> rotate <0,0,-59.32,> rotate <0,-64.12,0>  pigment {rgb <0.445,0.445,0.445>}}
object {star scale 1.39 translate <10004.44,0,0> rotate <0,0,7.38,> rotate <0,-293.52,0>  pigment {rgb <0.445,0.445,0.445>}}
object {star scale 1.39 translate <10004.44,0,0> rotate <0,0,-36.67,> rotate <0,-124.64,0>  pigment {rgb <0.445,0.445,0.445>}}
object {star scale 1.39 translate <10004.44,0,0> rotate <0,0,-80.55,> rotate <0,-161.45,0>  pigment {rgb <0.445,0.445,0.445>}}
object {star scale 1.39 translate <10004.44,0,0> rotate <0,0,-57.55,> rotate <0,-158.9,0>  pigment {rgb <0.445,0.445,0.445>}}
object {star scale 1.39 translate <10004.44,0,0> rotate <0,0,5.48,> rotate <0,-25.36,0>  pigment {rgb <0.445,0.445,0.445>}}
object {star scale 1.39 translate <10004.45,0,0> rotate <0,0,-11.87,> rotate <0,-78.08,0>  pigment {rgb <0.44375,0.44375,0.44375>}}
object {star scale 1.39 translate <10004.45,0,0> rotate <0,0,59.02,> rotate <0,-94.9,0>  pigment {rgb <0.44375,0.44375,0.44375>}}
object {star scale 1.39 translate <10004.45,0,0> rotate <0,0,-22.30,> rotate <0,-113.51,0>  pigment {rgb <0.44375,0.44375,0.44375>}}
object {star scale 1.39 translate <10004.45,0,0> rotate <0,0,-41.25,> rotate <0,-135.02,0>  pigment {rgb <0.44375,0.44375,0.44375>}}
object {star scale 1.39 translate <10004.45,0,0> rotate <0,0,-32.42,> rotate <0,-42.27,0>  pigment {rgb <0.44375,0.44375,0.44375>}}
object {star scale 1.39 translate <10004.45,0,0> rotate <0,0,-39.62,> rotate <0,-122.84,0>  pigment {rgb <0.44375,0.44375,0.44375>}}
object {star scale 1.39 translate <10004.45,0,0> rotate <0,0,73.35,> rotate <0,-288.89,0>  pigment {rgb <0.44375,0.44375,0.44375>}}
object {star scale 1.39 translate <10004.46,0,0> rotate <0,0,-32.87,> rotate <0,-343.13,0>  pigment {rgb <0.4425,0.4425,0.4425>}}
object {star scale 1.39 translate <10004.46,0,0> rotate <0,0,34.90,> rotate <0,-319.48,0>  pigment {rgb <0.4425,0.4425,0.4425>}}
object {star scale 1.39 translate <10004.46,0,0> rotate <0,0,-47.57,> rotate <0,-246.8,0>  pigment {rgb <0.4425,0.4425,0.4425>}}
object {star scale 1.39 translate <10004.46,0,0> rotate <0,0,-50.65,> rotate <0,-182.02,0>  pigment {rgb <0.4425,0.4425,0.4425>}}
object {star scale 1.39 translate <10004.46,0,0> rotate <0,0,26.18,> rotate <0,-141.16,0>  pigment {rgb <0.4425,0.4425,0.4425>}}
object {star scale 1.39 translate <10004.46,0,0> rotate <0,0,2.87,> rotate <0,-78.33,0>  pigment {rgb <0.4425,0.4425,0.4425>}}
object {star scale 1.39 translate <10004.46,0,0> rotate <0,0,-21.47,> rotate <0,-248.03,0>  pigment {rgb <0.4425,0.4425,0.4425>}}
object {star scale 1.39 translate <10004.46,0,0> rotate <0,0,-3.65,> rotate <0,-169.17,0>  pigment {rgb <0.4425,0.4425,0.4425>}}
object {star scale 1.39 translate <10004.46,0,0> rotate <0,0,29.75,> rotate <0,-218.67,0>  pigment {rgb <0.4425,0.4425,0.4425>}}
object {star scale 1.39 translate <10004.46,0,0> rotate <0,0,2.40,> rotate <0,-101.97,0>  pigment {rgb <0.4425,0.4425,0.4425>}}
object {star scale 1.39 translate <10004.46,0,0> rotate <0,0,1.72,> rotate <0,-74.64,0>  pigment {rgb <0.4425,0.4425,0.4425>}}
object {star scale 1.39 translate <10004.46,0,0> rotate <0,0,-45.17,> rotate <0,-177.79,0>  pigment {rgb <0.4425,0.4425,0.4425>}}
object {star scale 1.39 translate <10004.47,0,0> rotate <0,0,-25.93,> rotate <0,-117.02,0>  pigment {rgb <0.44125,0.44125,0.44125>}}
object {star scale 1.39 translate <10004.47,0,0> rotate <0,0,-64.95,> rotate <0,-336.83,0>  pigment {rgb <0.44125,0.44125,0.44125>}}
object {star scale 1.39 translate <10004.47,0,0> rotate <0,0,-23.75,> rotate <0,-346.67,0>  pigment {rgb <0.44125,0.44125,0.44125>}}
object {star scale 1.39 translate <10004.47,0,0> rotate <0,0,51.60,> rotate <0,-137.22,0>  pigment {rgb <0.44125,0.44125,0.44125>}}
object {star scale 1.39 translate <10004.47,0,0> rotate <0,0,-22.83,> rotate <0,-167.91,0>  pigment {rgb <0.44125,0.44125,0.44125>}}
object {star scale 1.39 translate <10004.47,0,0> rotate <0,0,14.22,> rotate <0,-92.98,0>  pigment {rgb <0.44125,0.44125,0.44125>}}
object {star scale 1.39 translate <10004.47,0,0> rotate <0,0,-32.58,> rotate <0,-97.04,0>  pigment {rgb <0.44125,0.44125,0.44125>}}
object {star scale 1.39 translate <10004.47,0,0> rotate <0,0,-18.58,> rotate <0,-41.28,0>  pigment {rgb <0.44125,0.44125,0.44125>}}
object {star scale 1.39 translate <10004.47,0,0> rotate <0,0,-39.55,> rotate <0,-331.53,0>  pigment {rgb <0.44125,0.44125,0.44125>}}
object {star scale 1.39 translate <10004.47,0,0> rotate <0,0,29.05,> rotate <0,-50.08,0>  pigment {rgb <0.44125,0.44125,0.44125>}}
object {star scale 1.39 translate <10004.48,0,0> rotate <0,0,-14.55,> rotate <0,-355.68,0>  pigment {rgb <0.44,0.44,0.44>}}
object {star scale 1.39 translate <10004.48,0,0> rotate <0,0,10.00,> rotate <0,-318.62,0>  pigment {rgb <0.44,0.44,0.44>}}
object {star scale 1.39 translate <10004.48,0,0> rotate <0,0,7.33,> rotate <0,-98.23,0>  pigment {rgb <0.44,0.44,0.44>}}
object {star scale 1.39 translate <10004.48,0,0> rotate <0,0,35.25,> rotate <0,-151.86,0>  pigment {rgb <0.44,0.44,0.44>}}
object {star scale 1.39 translate <10004.48,0,0> rotate <0,0,-38.87,> rotate <0,-118.16,0>  pigment {rgb <0.44,0.44,0.44>}}
object {star scale 1.39 translate <10004.48,0,0> rotate <0,0,-46.77,> rotate <0,-108.14,0>  pigment {rgb <0.44,0.44,0.44>}}
object {star scale 1.39 translate <10004.48,0,0> rotate <0,0,-4.33,> rotate <0,-224.3,0>  pigment {rgb <0.44,0.44,0.44>}}
object {star scale 1.39 translate <10004.48,0,0> rotate <0,0,-72.60,> rotate <0,-136.29,0>  pigment {rgb <0.44,0.44,0.44>}}
object {star scale 1.39 translate <10004.48,0,0> rotate <0,0,44.27,> rotate <0,-340.13,0>  pigment {rgb <0.44,0.44,0.44>}}
object {star scale 1.39 translate <10004.49,0,0> rotate <0,0,67.40,> rotate <0,-37.27,0>  pigment {rgb <0.43875,0.43875,0.43875>}}
object {star scale 1.39 translate <10004.49,0,0> rotate <0,0,50.22,> rotate <0,-294.11,0>  pigment {rgb <0.43875,0.43875,0.43875>}}
object {star scale 1.39 translate <10004.49,0,0> rotate <0,0,13.23,> rotate <0,-101,0>  pigment {rgb <0.43875,0.43875,0.43875>}}
object {star scale 1.39 translate <10004.49,0,0> rotate <0,0,39.72,> rotate <0,-333.47,0>  pigment {rgb <0.43875,0.43875,0.43875>}}
object {star scale 1.39 translate <10004.49,0,0> rotate <0,0,-0.37,> rotate <0,-151.98,0>  pigment {rgb <0.43875,0.43875,0.43875>}}
object {star scale 1.39 translate <10004.49,0,0> rotate <0,0,52.05,> rotate <0,-143.71,0>  pigment {rgb <0.43875,0.43875,0.43875>}}
object {star scale 1.39 translate <10004.49,0,0> rotate <0,0,-33.00,> rotate <0,-332.1,0>  pigment {rgb <0.43875,0.43875,0.43875>}}
object {star scale 1.39 translate <10004.49,0,0> rotate <0,0,-56.77,> rotate <0,-131.68,0>  pigment {rgb <0.43875,0.43875,0.43875>}}
object {star scale 1.39 translate <10004.49,0,0> rotate <0,0,43.13,> rotate <0,-337.62,0>  pigment {rgb <0.43875,0.43875,0.43875>}}
object {star scale 1.39 translate <10004.49,0,0> rotate <0,0,-65.58,> rotate <0,-359.98,0>  pigment {rgb <0.43875,0.43875,0.43875>}}
object {star scale 1.39 translate <10004.49,0,0> rotate <0,0,15.62,> rotate <0,-66.59,0>  pigment {rgb <0.43875,0.43875,0.43875>}}
object {star scale 1.39 translate <10004.49,0,0> rotate <0,0,-20.03,> rotate <0,-246.03,0>  pigment {rgb <0.43875,0.43875,0.43875>}}
object {star scale 1.38 translate <10004.50,0,0> rotate <0,0,24.75,> rotate <0,-163.9,0>  pigment {rgb <0.4375,0.4375,0.4375>}}
object {star scale 1.38 translate <10004.50,0,0> rotate <0,0,-56.03,> rotate <0,-155.23,0>  pigment {rgb <0.4375,0.4375,0.4375>}}
object {star scale 1.38 translate <10004.50,0,0> rotate <0,0,-59.33,> rotate <0,-230.85,0>  pigment {rgb <0.4375,0.4375,0.4375>}}
object {star scale 1.38 translate <10004.50,0,0> rotate <0,0,-26.80,> rotate <0,-114.7,0>  pigment {rgb <0.4375,0.4375,0.4375>}}
object {star scale 1.38 translate <10004.50,0,0> rotate <0,0,17.45,> rotate <0,-206.82,0>  pigment {rgb <0.4375,0.4375,0.4375>}}
object {star scale 1.38 translate <10004.50,0,0> rotate <0,0,-25.00,> rotate <0,-316.78,0>  pigment {rgb <0.4375,0.4375,0.4375>}}
object {star scale 1.38 translate <10004.50,0,0> rotate <0,0,1.78,> rotate <0,-355.51,0>  pigment {rgb <0.4375,0.4375,0.4375>}}
object {star scale 1.38 translate <10004.50,0,0> rotate <0,0,-51.93,> rotate <0,-311.01,0>  pigment {rgb <0.4375,0.4375,0.4375>}}
object {star scale 1.38 translate <10004.50,0,0> rotate <0,0,-21.82,> rotate <0,-322.18,0>  pigment {rgb <0.4375,0.4375,0.4375>}}
object {star scale 1.38 translate <10004.50,0,0> rotate <0,0,-29.77,> rotate <0,-68.38,0>  pigment {rgb <0.4375,0.4375,0.4375>}}
object {star scale 1.38 translate <10004.50,0,0> rotate <0,0,30.10,> rotate <0,-17.92,0>  pigment {rgb <0.4375,0.4375,0.4375>}}
object {star scale 1.38 translate <10004.50,0,0> rotate <0,0,-56.15,> rotate <0,-87.46,0>  pigment {rgb <0.4375,0.4375,0.4375>}}
object {star scale 1.38 translate <10004.50,0,0> rotate <0,0,-35.95,> rotate <0,-142.31,0>  pigment {rgb <0.4375,0.4375,0.4375>}}
object {star scale 1.38 translate <10004.50,0,0> rotate <0,0,-27.70,> rotate <0,-300.67,0>  pigment {rgb <0.4375,0.4375,0.4375>}}
object {star scale 1.38 translate <10004.51,0,0> rotate <0,0,19.68,> rotate <0,-235.38,0>  pigment {rgb <0.43625,0.43625,0.43625>}}
object {star scale 1.38 translate <10004.51,0,0> rotate <0,0,39.18,> rotate <0,-87.3,0>  pigment {rgb <0.43625,0.43625,0.43625>}}
object {star scale 1.38 translate <10004.51,0,0> rotate <0,0,4.12,> rotate <0,-84.8,0>  pigment {rgb <0.43625,0.43625,0.43625>}}
object {star scale 1.38 translate <10004.51,0,0> rotate <0,0,-61.33,> rotate <0,-144.84,0>  pigment {rgb <0.43625,0.43625,0.43625>}}
object {star scale 1.38 translate <10004.51,0,0> rotate <0,0,57.58,> rotate <0,-311.34,0>  pigment {rgb <0.43625,0.43625,0.43625>}}
object {star scale 1.38 translate <10004.51,0,0> rotate <0,0,-11.38,> rotate <0,-317.4,0>  pigment {rgb <0.43625,0.43625,0.43625>}}
object {star scale 1.38 translate <10004.51,0,0> rotate <0,0,29.25,> rotate <0,-41.98,0>  pigment {rgb <0.43625,0.43625,0.43625>}}
object {star scale 1.38 translate <10004.51,0,0> rotate <0,0,67.87,> rotate <0,-300.7,0>  pigment {rgb <0.43625,0.43625,0.43625>}}
object {star scale 1.38 translate <10004.51,0,0> rotate <0,0,-62.15,> rotate <0,-60.23,0>  pigment {rgb <0.43625,0.43625,0.43625>}}
object {star scale 1.38 translate <10004.52,0,0> rotate <0,0,37.13,> rotate <0,-260.92,0>  pigment {rgb <0.435,0.435,0.435>}}
object {star scale 1.38 translate <10004.52,0,0> rotate <0,0,-60.98,> rotate <0,-200.66,0>  pigment {rgb <0.435,0.435,0.435>}}
object {star scale 1.38 translate <10004.52,0,0> rotate <0,0,36.48,> rotate <0,-311.85,0>  pigment {rgb <0.435,0.435,0.435>}}
object {star scale 1.38 translate <10004.52,0,0> rotate <0,0,-34.98,> rotate <0,-114.34,0>  pigment {rgb <0.435,0.435,0.435>}}
object {star scale 1.38 translate <10004.52,0,0> rotate <0,0,-13.38,> rotate <0,-214.78,0>  pigment {rgb <0.435,0.435,0.435>}}
object {star scale 1.38 translate <10004.52,0,0> rotate <0,0,36.78,> rotate <0,-4.58,0>  pigment {rgb <0.435,0.435,0.435>}}
object {star scale 1.38 translate <10004.52,0,0> rotate <0,0,3.82,> rotate <0,-345.97,0>  pigment {rgb <0.435,0.435,0.435>}}
object {star scale 1.38 translate <10004.52,0,0> rotate <0,0,41.08,> rotate <0,-12.45,0>  pigment {rgb <0.435,0.435,0.435>}}
object {star scale 1.38 translate <10004.52,0,0> rotate <0,0,49.42,> rotate <0,-348.14,0>  pigment {rgb <0.435,0.435,0.435>}}
object {star scale 1.38 translate <10004.52,0,0> rotate <0,0,-64.53,> rotate <0,-201,0>  pigment {rgb <0.435,0.435,0.435>}}
object {star scale 1.38 translate <10004.52,0,0> rotate <0,0,-3.08,> rotate <0,-90.01,0>  pigment {rgb <0.435,0.435,0.435>}}
object {star scale 1.38 translate <10004.52,0,0> rotate <0,0,26.95,> rotate <0,-226.11,0>  pigment {rgb <0.435,0.435,0.435>}}
object {star scale 1.38 translate <10004.52,0,0> rotate <0,0,27.80,> rotate <0,-303.94,0>  pigment {rgb <0.435,0.435,0.435>}}
object {star scale 1.38 translate <10004.52,0,0> rotate <0,0,-27.17,> rotate <0,-299.24,0>  pigment {rgb <0.435,0.435,0.435>}}
object {star scale 1.38 translate <10004.52,0,0> rotate <0,0,9.40,> rotate <0,-346.75,0>  pigment {rgb <0.435,0.435,0.435>}}
object {star scale 1.38 translate <10004.53,0,0> rotate <0,0,-19.80,> rotate <0,-228.05,0>  pigment {rgb <0.43375,0.43375,0.43375>}}
object {star scale 1.38 translate <10004.53,0,0> rotate <0,0,51.78,> rotate <0,-213.37,0>  pigment {rgb <0.43375,0.43375,0.43375>}}
object {star scale 1.38 translate <10004.53,0,0> rotate <0,0,-45.95,> rotate <0,-272.8,0>  pigment {rgb <0.43375,0.43375,0.43375>}}
object {star scale 1.38 translate <10004.53,0,0> rotate <0,0,47.65,> rotate <0,-316.65,0>  pigment {rgb <0.43375,0.43375,0.43375>}}
object {star scale 1.38 translate <10004.53,0,0> rotate <0,0,-36.87,> rotate <0,-230.79,0>  pigment {rgb <0.43375,0.43375,0.43375>}}
object {star scale 1.38 translate <10004.53,0,0> rotate <0,0,-5.08,> rotate <0,-261.65,0>  pigment {rgb <0.43375,0.43375,0.43375>}}
object {star scale 1.38 translate <10004.53,0,0> rotate <0,0,12.02,> rotate <0,-112.45,0>  pigment {rgb <0.43375,0.43375,0.43375>}}
object {star scale 1.38 translate <10004.54,0,0> rotate <0,0,48.28,> rotate <0,-11.18,0>  pigment {rgb <0.4325,0.4325,0.4325>}}
object {star scale 1.38 translate <10004.54,0,0> rotate <0,0,-44.95,> rotate <0,-233.97,0>  pigment {rgb <0.4325,0.4325,0.4325>}}
object {star scale 1.38 translate <10004.54,0,0> rotate <0,0,33.37,> rotate <0,-79.55,0>  pigment {rgb <0.4325,0.4325,0.4325>}}
object {star scale 1.38 translate <10004.54,0,0> rotate <0,0,-35.48,> rotate <0,-76.1,0>  pigment {rgb <0.4325,0.4325,0.4325>}}
object {star scale 1.38 translate <10004.54,0,0> rotate <0,0,20.22,> rotate <0,-177,0>  pigment {rgb <0.4325,0.4325,0.4325>}}
object {star scale 1.38 translate <10004.54,0,0> rotate <0,0,-22.97,> rotate <0,-98.76,0>  pigment {rgb <0.4325,0.4325,0.4325>}}
object {star scale 1.38 translate <10004.54,0,0> rotate <0,0,-24.17,> rotate <0,-245.16,0>  pigment {rgb <0.4325,0.4325,0.4325>}}
object {star scale 1.38 translate <10004.54,0,0> rotate <0,0,-17.35,> rotate <0,-0.93,0>  pigment {rgb <0.4325,0.4325,0.4325>}}
object {star scale 1.38 translate <10004.54,0,0> rotate <0,0,12.77,> rotate <0,-352.29,0>  pigment {rgb <0.4325,0.4325,0.4325>}}
object {star scale 1.38 translate <10004.54,0,0> rotate <0,0,76.97,> rotate <0,-105.02,0>  pigment {rgb <0.4325,0.4325,0.4325>}}
object {star scale 1.38 translate <10004.54,0,0> rotate <0,0,35.07,> rotate <0,-42.88,0>  pigment {rgb <0.4325,0.4325,0.4325>}}
object {star scale 1.38 translate <10004.55,0,0> rotate <0,0,27.60,> rotate <0,-88.33,0>  pigment {rgb <0.43125,0.43125,0.43125>}}
object {star scale 1.38 translate <10004.55,0,0> rotate <0,0,47.52,> rotate <0,-314.95,0>  pigment {rgb <0.43125,0.43125,0.43125>}}
object {star scale 1.38 translate <10004.55,0,0> rotate <0,0,23.63,> rotate <0,-322.49,0>  pigment {rgb <0.43125,0.43125,0.43125>}}
object {star scale 1.38 translate <10004.55,0,0> rotate <0,0,49.22,> rotate <0,-111.68,0>  pigment {rgb <0.43125,0.43125,0.43125>}}
object {star scale 1.38 translate <10004.55,0,0> rotate <0,0,-45.22,> rotate <0,-216.53,0>  pigment {rgb <0.43125,0.43125,0.43125>}}
object {star scale 1.38 translate <10004.55,0,0> rotate <0,0,69.83,> rotate <0,-143.62,0>  pigment {rgb <0.43125,0.43125,0.43125>}}
object {star scale 1.38 translate <10004.55,0,0> rotate <0,0,36.38,> rotate <0,-143.55,0>  pigment {rgb <0.43125,0.43125,0.43125>}}
object {star scale 1.38 translate <10004.55,0,0> rotate <0,0,-28.58,> rotate <0,-208.57,0>  pigment {rgb <0.43125,0.43125,0.43125>}}
object {star scale 1.38 translate <10004.55,0,0> rotate <0,0,71.32,> rotate <0,-325.48,0>  pigment {rgb <0.43125,0.43125,0.43125>}}
object {star scale 1.38 translate <10004.55,0,0> rotate <0,0,-61.40,> rotate <0,-59.69,0>  pigment {rgb <0.43125,0.43125,0.43125>}}
object {star scale 1.37 translate <10004.56,0,0> rotate <0,0,-28.13,> rotate <0,-357.23,0>  pigment {rgb <0.43,0.43,0.43>}}
object {star scale 1.37 translate <10004.56,0,0> rotate <0,0,58.88,> rotate <0,-52.48,0>  pigment {rgb <0.43,0.43,0.43>}}
object {star scale 1.37 translate <10004.56,0,0> rotate <0,0,-1.18,> rotate <0,-143,0>  pigment {rgb <0.43,0.43,0.43>}}
object {star scale 1.37 translate <10004.56,0,0> rotate <0,0,49.47,> rotate <0,-336.13,0>  pigment {rgb <0.43,0.43,0.43>}}
object {star scale 1.37 translate <10004.56,0,0> rotate <0,0,-28.45,> rotate <0,-272.02,0>  pigment {rgb <0.43,0.43,0.43>}}
object {star scale 1.37 translate <10004.56,0,0> rotate <0,0,38.45,> rotate <0,-136.63,0>  pigment {rgb <0.43,0.43,0.43>}}
object {star scale 1.37 translate <10004.56,0,0> rotate <0,0,-55.03,> rotate <0,-154.9,0>  pigment {rgb <0.43,0.43,0.43>}}
object {star scale 1.37 translate <10004.56,0,0> rotate <0,0,-60.57,> rotate <0,-160.88,0>  pigment {rgb <0.43,0.43,0.43>}}
object {star scale 1.37 translate <10004.57,0,0> rotate <0,0,24.08,> rotate <0,-298.37,0>  pigment {rgb <0.42875,0.42875,0.42875>}}
object {star scale 1.37 translate <10004.57,0,0> rotate <0,0,54.07,> rotate <0,-148.03,0>  pigment {rgb <0.42875,0.42875,0.42875>}}
object {star scale 1.37 translate <10004.57,0,0> rotate <0,0,46.53,> rotate <0,-335.26,0>  pigment {rgb <0.42875,0.42875,0.42875>}}
object {star scale 1.37 translate <10004.57,0,0> rotate <0,0,22.65,> rotate <0,-283.69,0>  pigment {rgb <0.42875,0.42875,0.42875>}}
object {star scale 1.37 translate <10004.57,0,0> rotate <0,0,-27.93,> rotate <0,-243.08,0>  pigment {rgb <0.42875,0.42875,0.42875>}}
object {star scale 1.37 translate <10004.57,0,0> rotate <0,0,-46.55,> rotate <0,-147.92,0>  pigment {rgb <0.42875,0.42875,0.42875>}}
object {star scale 1.37 translate <10004.57,0,0> rotate <0,0,27.08,> rotate <0,-313.03,0>  pigment {rgb <0.42875,0.42875,0.42875>}}
object {star scale 1.37 translate <10004.57,0,0> rotate <0,0,-40.27,> rotate <0,-54.27,0>  pigment {rgb <0.42875,0.42875,0.42875>}}
object {star scale 1.37 translate <10004.58,0,0> rotate <0,0,-37.32,> rotate <0,-55.71,0>  pigment {rgb <0.4275,0.4275,0.4275>}}
object {star scale 1.37 translate <10004.58,0,0> rotate <0,0,-28.40,> rotate <0,-115.88,0>  pigment {rgb <0.4275,0.4275,0.4275>}}
object {star scale 1.37 translate <10004.58,0,0> rotate <0,0,64.72,> rotate <0,-207.86,0>  pigment {rgb <0.4275,0.4275,0.4275>}}
object {star scale 1.37 translate <10004.58,0,0> rotate <0,0,-46.50,> rotate <0,-263.92,0>  pigment {rgb <0.4275,0.4275,0.4275>}}
object {star scale 1.37 translate <10004.58,0,0> rotate <0,0,65.72,> rotate <0,-290.17,0>  pigment {rgb <0.4275,0.4275,0.4275>}}
object {star scale 1.37 translate <10004.58,0,0> rotate <0,0,23.75,> rotate <0,-350.16,0>  pigment {rgb <0.4275,0.4275,0.4275>}}
object {star scale 1.37 translate <10004.58,0,0> rotate <0,0,-46.08,> rotate <0,-10.33,0>  pigment {rgb <0.4275,0.4275,0.4275>}}
object {star scale 1.37 translate <10004.58,0,0> rotate <0,0,-40.50,> rotate <0,-287.08,0>  pigment {rgb <0.4275,0.4275,0.4275>}}
object {star scale 1.37 translate <10004.58,0,0> rotate <0,0,65.93,> rotate <0,-224.4,0>  pigment {rgb <0.4275,0.4275,0.4275>}}
object {star scale 1.37 translate <10004.59,0,0> rotate <0,0,14.03,> rotate <0,-246.35,0>  pigment {rgb <0.42625,0.42625,0.42625>}}
object {star scale 1.37 translate <10004.59,0,0> rotate <0,0,-4.83,> rotate <0,-83.85,0>  pigment {rgb <0.42625,0.42625,0.42625>}}
object {star scale 1.37 translate <10004.59,0,0> rotate <0,0,3.10,> rotate <0,-81.71,0>  pigment {rgb <0.42625,0.42625,0.42625>}}
object {star scale 1.37 translate <10004.59,0,0> rotate <0,0,-59.92,> rotate <0,-198.08,0>  pigment {rgb <0.42625,0.42625,0.42625>}}
object {star scale 1.37 translate <10004.59,0,0> rotate <0,0,-24.88,> rotate <0,-294.18,0>  pigment {rgb <0.42625,0.42625,0.42625>}}
object {star scale 1.37 translate <10004.59,0,0> rotate <0,0,72.15,> rotate <0,-265.48,0>  pigment {rgb <0.42625,0.42625,0.42625>}}
object {star scale 1.37 translate <10004.59,0,0> rotate <0,0,-25.85,> rotate <0,-137.01,0>  pigment {rgb <0.42625,0.42625,0.42625>}}
object {star scale 1.37 translate <10004.59,0,0> rotate <0,0,50.30,> rotate <0,-64.56,0>  pigment {rgb <0.42625,0.42625,0.42625>}}
object {star scale 1.37 translate <10004.59,0,0> rotate <0,0,-13.07,> rotate <0,-151.28,0>  pigment {rgb <0.42625,0.42625,0.42625>}}
object {star scale 1.37 translate <10004.59,0,0> rotate <0,0,-2.77,> rotate <0,-142.28,0>  pigment {rgb <0.42625,0.42625,0.42625>}}
object {star scale 1.37 translate <10004.59,0,0> rotate <0,0,57.50,> rotate <0,-358.6,0>  pigment {rgb <0.42625,0.42625,0.42625>}}
object {star scale 1.37 translate <10004.59,0,0> rotate <0,0,-37.13,> rotate <0,-164.18,0>  pigment {rgb <0.42625,0.42625,0.42625>}}
object {star scale 1.37 translate <10004.59,0,0> rotate <0,0,64.33,> rotate <0,-130.05,0>  pigment {rgb <0.42625,0.42625,0.42625>}}
object {star scale 1.37 translate <10004.60,0,0> rotate <0,0,-25.33,> rotate <0,-238.4,0>  pigment {rgb <0.425,0.425,0.425>}}
object {star scale 1.37 translate <10004.60,0,0> rotate <0,0,-7.03,> rotate <0,-97.2,0>  pigment {rgb <0.425,0.425,0.425>}}
object {star scale 1.37 translate <10004.60,0,0> rotate <0,0,-60.32,> rotate <0,-168.15,0>  pigment {rgb <0.425,0.425,0.425>}}
object {star scale 1.37 translate <10004.60,0,0> rotate <0,0,-38.73,> rotate <0,-231.33,0>  pigment {rgb <0.425,0.425,0.425>}}
object {star scale 1.37 translate <10004.60,0,0> rotate <0,0,38.68,> rotate <0,-4.28,0>  pigment {rgb <0.425,0.425,0.425>}}
object {star scale 1.37 translate <10004.60,0,0> rotate <0,0,-62.42,> rotate <0,-166.63,0>  pigment {rgb <0.425,0.425,0.425>}}
object {star scale 1.37 translate <10004.60,0,0> rotate <0,0,16.97,> rotate <0,-221.31,0>  pigment {rgb <0.425,0.425,0.425>}}
object {star scale 1.37 translate <10004.60,0,0> rotate <0,0,35.25,> rotate <0,-308.48,0>  pigment {rgb <0.425,0.425,0.425>}}
object {star scale 1.37 translate <10004.61,0,0> rotate <0,0,19.10,> rotate <0,-222.85,0>  pigment {rgb <0.42375,0.42375,0.42375>}}
object {star scale 1.37 translate <10004.61,0,0> rotate <0,0,-15.97,> rotate <0,-290.43,0>  pigment {rgb <0.42375,0.42375,0.42375>}}
object {star scale 1.37 translate <10004.61,0,0> rotate <0,0,-3.68,> rotate <0,-270.12,0>  pigment {rgb <0.42375,0.42375,0.42375>}}
object {star scale 1.37 translate <10004.61,0,0> rotate <0,0,42.45,> rotate <0,-238.17,0>  pigment {rgb <0.42375,0.42375,0.42375>}}
object {star scale 1.37 translate <10004.61,0,0> rotate <0,0,3.20,> rotate <0,-28.39,0>  pigment {rgb <0.42375,0.42375,0.42375>}}
object {star scale 1.37 translate <10004.61,0,0> rotate <0,0,39.62,> rotate <0,-47.82,0>  pigment {rgb <0.42375,0.42375,0.42375>}}
object {star scale 1.37 translate <10004.61,0,0> rotate <0,0,-5.70,> rotate <0,-1.33,0>  pigment {rgb <0.42375,0.42375,0.42375>}}
object {star scale 1.37 translate <10004.61,0,0> rotate <0,0,-10.07,> rotate <0,-233.55,0>  pigment {rgb <0.42375,0.42375,0.42375>}}
object {star scale 1.36 translate <10004.62,0,0> rotate <0,0,4.20,> rotate <0,-284.05,0>  pigment {rgb <0.4225,0.4225,0.4225>}}
object {star scale 1.36 translate <10004.62,0,0> rotate <0,0,-26.80,> rotate <0,-114.71,0>  pigment {rgb <0.4225,0.4225,0.4225>}}
object {star scale 1.36 translate <10004.62,0,0> rotate <0,0,-37.42,> rotate <0,-138.94,0>  pigment {rgb <0.4225,0.4225,0.4225>}}
object {star scale 1.36 translate <10004.62,0,0> rotate <0,0,37.30,> rotate <0,-259.42,0>  pigment {rgb <0.4225,0.4225,0.4225>}}
object {star scale 1.36 translate <10004.62,0,0> rotate <0,0,-18.38,> rotate <0,-119.97,0>  pigment {rgb <0.4225,0.4225,0.4225>}}
object {star scale 1.36 translate <10004.62,0,0> rotate <0,0,-7.32,> rotate <0,-82.98,0>  pigment {rgb <0.4225,0.4225,0.4225>}}
object {star scale 1.36 translate <10004.62,0,0> rotate <0,0,-49.62,> rotate <0,-118.27,0>  pigment {rgb <0.4225,0.4225,0.4225>}}
object {star scale 1.36 translate <10004.62,0,0> rotate <0,0,20.13,> rotate <0,-90.98,0>  pigment {rgb <0.4225,0.4225,0.4225>}}
object {star scale 1.36 translate <10004.62,0,0> rotate <0,0,-8.12,> rotate <0,-264.46,0>  pigment {rgb <0.4225,0.4225,0.4225>}}
object {star scale 1.36 translate <10004.62,0,0> rotate <0,0,-54.27,> rotate <0,-173.69,0>  pigment {rgb <0.4225,0.4225,0.4225>}}
object {star scale 1.36 translate <10004.62,0,0> rotate <0,0,-44.67,> rotate <0,-235.3,0>  pigment {rgb <0.4225,0.4225,0.4225>}}
object {star scale 1.36 translate <10004.62,0,0> rotate <0,0,-7.23,> rotate <0,-130.92,0>  pigment {rgb <0.4225,0.4225,0.4225>}}
object {star scale 1.36 translate <10004.62,0,0> rotate <0,0,26.07,> rotate <0,-237.4,0>  pigment {rgb <0.4225,0.4225,0.4225>}}
object {star scale 1.36 translate <10004.62,0,0> rotate <0,0,59.17,> rotate <0,-14.17,0>  pigment {rgb <0.4225,0.4225,0.4225>}}
object {star scale 1.36 translate <10004.63,0,0> rotate <0,0,-8.37,> rotate <0,-246.95,0>  pigment {rgb <0.42125,0.42125,0.42125>}}
object {star scale 1.36 translate <10004.63,0,0> rotate <0,0,-28.37,> rotate <0,-113.85,0>  pigment {rgb <0.42125,0.42125,0.42125>}}
object {star scale 1.36 translate <10004.63,0,0> rotate <0,0,7.35,> rotate <0,-166.25,0>  pigment {rgb <0.42125,0.42125,0.42125>}}
object {star scale 1.36 translate <10004.63,0,0> rotate <0,0,27.77,> rotate <0,-300.28,0>  pigment {rgb <0.42125,0.42125,0.42125>}}
object {star scale 1.36 translate <10004.63,0,0> rotate <0,0,51.55,> rotate <0,-339.34,0>  pigment {rgb <0.42125,0.42125,0.42125>}}
object {star scale 1.36 translate <10004.63,0,0> rotate <0,0,-59.15,> rotate <0,-193.66,0>  pigment {rgb <0.42125,0.42125,0.42125>}}
object {star scale 1.36 translate <10004.63,0,0> rotate <0,0,-24.60,> rotate <0,-58.43,0>  pigment {rgb <0.42125,0.42125,0.42125>}}
object {star scale 1.36 translate <10004.63,0,0> rotate <0,0,-62.27,> rotate <0,-277.84,0>  pigment {rgb <0.42125,0.42125,0.42125>}}
object {star scale 1.36 translate <10004.63,0,0> rotate <0,0,-42.32,> rotate <0,-278.38,0>  pigment {rgb <0.42125,0.42125,0.42125>}}
object {star scale 1.36 translate <10004.63,0,0> rotate <0,0,23.08,> rotate <0,-168.8,0>  pigment {rgb <0.42125,0.42125,0.42125>}}
object {star scale 1.36 translate <10004.64,0,0> rotate <0,0,-56.48,> rotate <0,-191.59,0>  pigment {rgb <0.42,0.42,0.42>}}
object {star scale 1.36 translate <10004.64,0,0> rotate <0,0,-24.02,> rotate <0,-59.98,0>  pigment {rgb <0.42,0.42,0.42>}}
object {star scale 1.36 translate <10004.64,0,0> rotate <0,0,6.62,> rotate <0,-180.22,0>  pigment {rgb <0.42,0.42,0.42>}}
object {star scale 1.36 translate <10004.64,0,0> rotate <0,0,21.60,> rotate <0,-75.78,0>  pigment {rgb <0.42,0.42,0.42>}}
object {star scale 1.36 translate <10004.64,0,0> rotate <0,0,-25.75,> rotate <0,-237.75,0>  pigment {rgb <0.42,0.42,0.42>}}
object {star scale 1.36 translate <10004.64,0,0> rotate <0,0,-39.98,> rotate <0,-189.97,0>  pigment {rgb <0.42,0.42,0.42>}}
object {star scale 1.36 translate <10004.64,0,0> rotate <0,0,-10.68,> rotate <0,-27.4,0>  pigment {rgb <0.42,0.42,0.42>}}
object {star scale 1.36 translate <10004.64,0,0> rotate <0,0,-30.97,> rotate <0,-112.68,0>  pigment {rgb <0.42,0.42,0.42>}}
object {star scale 1.36 translate <10004.64,0,0> rotate <0,0,-49.23,> rotate <0,-240.8,0>  pigment {rgb <0.42,0.42,0.42>}}
object {star scale 1.36 translate <10004.64,0,0> rotate <0,0,8.73,> rotate <0,-271.83,0>  pigment {rgb <0.42,0.42,0.42>}}
object {star scale 1.36 translate <10004.64,0,0> rotate <0,0,-51.43,> rotate <0,-206.66,0>  pigment {rgb <0.42,0.42,0.42>}}
object {star scale 1.36 translate <10004.64,0,0> rotate <0,0,50.05,> rotate <0,-346.05,0>  pigment {rgb <0.42,0.42,0.42>}}
object {star scale 1.36 translate <10004.64,0,0> rotate <0,0,-63.08,> rotate <0,-88.53,0>  pigment {rgb <0.42,0.42,0.42>}}
object {star scale 1.36 translate <10004.64,0,0> rotate <0,0,-27.88,> rotate <0,-109.14,0>  pigment {rgb <0.42,0.42,0.42>}}
object {star scale 1.36 translate <10004.64,0,0> rotate <0,0,25.13,> rotate <0,-359.44,0>  pigment {rgb <0.42,0.42,0.42>}}
object {star scale 1.36 translate <10004.65,0,0> rotate <0,0,-57.78,> rotate <0,-240.88,0>  pigment {rgb <0.41875,0.41875,0.41875>}}
object {star scale 1.36 translate <10004.65,0,0> rotate <0,0,-59.18,> rotate <0,-159.69,0>  pigment {rgb <0.41875,0.41875,0.41875>}}
object {star scale 1.36 translate <10004.65,0,0> rotate <0,0,24.58,> rotate <0,-18.44,0>  pigment {rgb <0.41875,0.41875,0.41875>}}
object {star scale 1.36 translate <10004.65,0,0> rotate <0,0,59.38,> rotate <0,-282.8,0>  pigment {rgb <0.41875,0.41875,0.41875>}}
object {star scale 1.36 translate <10004.65,0,0> rotate <0,0,9.88,> rotate <0,-100.24,0>  pigment {rgb <0.41875,0.41875,0.41875>}}
object {star scale 1.36 translate <10004.65,0,0> rotate <0,0,71.32,> rotate <0,-57.59,0>  pigment {rgb <0.41875,0.41875,0.41875>}}
object {star scale 1.36 translate <10004.65,0,0> rotate <0,0,49.02,> rotate <0,-203.61,0>  pigment {rgb <0.41875,0.41875,0.41875>}}
object {star scale 1.36 translate <10004.65,0,0> rotate <0,0,27.72,> rotate <0,-40.86,0>  pigment {rgb <0.41875,0.41875,0.41875>}}
object {star scale 1.36 translate <10004.65,0,0> rotate <0,0,14.85,> rotate <0,-68.46,0>  pigment {rgb <0.41875,0.41875,0.41875>}}
object {star scale 1.36 translate <10004.65,0,0> rotate <0,0,-10.78,> rotate <0,-252.46,0>  pigment {rgb <0.41875,0.41875,0.41875>}}
object {star scale 1.36 translate <10004.65,0,0> rotate <0,0,30.15,> rotate <0,-294.85,0>  pigment {rgb <0.41875,0.41875,0.41875>}}
object {star scale 1.36 translate <10004.65,0,0> rotate <0,0,21.03,> rotate <0,-17.87,0>  pigment {rgb <0.41875,0.41875,0.41875>}}
object {star scale 1.36 translate <10004.65,0,0> rotate <0,0,-8.00,> rotate <0,-189.81,0>  pigment {rgb <0.41875,0.41875,0.41875>}}
object {star scale 1.36 translate <10004.65,0,0> rotate <0,0,-27.05,> rotate <0,-274.52,0>  pigment {rgb <0.41875,0.41875,0.41875>}}
object {star scale 1.36 translate <10004.66,0,0> rotate <0,0,-20.15,> rotate <0,-103.9,0>  pigment {rgb <0.4175,0.4175,0.4175>}}
object {star scale 1.36 translate <10004.66,0,0> rotate <0,0,-32.27,> rotate <0,-315.33,0>  pigment {rgb <0.4175,0.4175,0.4175>}}
object {star scale 1.36 translate <10004.66,0,0> rotate <0,0,14.68,> rotate <0,-308.83,0>  pigment {rgb <0.4175,0.4175,0.4175>}}
object {star scale 1.36 translate <10004.66,0,0> rotate <0,0,1.37,> rotate <0,-336.32,0>  pigment {rgb <0.4175,0.4175,0.4175>}}
object {star scale 1.36 translate <10004.66,0,0> rotate <0,0,0.33,> rotate <0,-291.63,0>  pigment {rgb <0.4175,0.4175,0.4175>}}
object {star scale 1.36 translate <10004.66,0,0> rotate <0,0,-34.42,> rotate <0,-234.94,0>  pigment {rgb <0.4175,0.4175,0.4175>}}
object {star scale 1.36 translate <10004.66,0,0> rotate <0,0,16.75,> rotate <0,-270.01,0>  pigment {rgb <0.4175,0.4175,0.4175>}}
object {star scale 1.36 translate <10004.66,0,0> rotate <0,0,-48.82,> rotate <0,-190.65,0>  pigment {rgb <0.4175,0.4175,0.4175>}}
object {star scale 1.36 translate <10004.66,0,0> rotate <0,0,54.68,> rotate <0,-205.19,0>  pigment {rgb <0.4175,0.4175,0.4175>}}
object {star scale 1.36 translate <10004.67,0,0> rotate <0,0,63.52,> rotate <0,-137.73,0>  pigment {rgb <0.41625,0.41625,0.41625>}}
object {star scale 1.36 translate <10004.67,0,0> rotate <0,0,-19.47,> rotate <0,-324.27,0>  pigment {rgb <0.41625,0.41625,0.41625>}}
object {star scale 1.36 translate <10004.67,0,0> rotate <0,0,49.52,> rotate <0,-52.34,0>  pigment {rgb <0.41625,0.41625,0.41625>}}
object {star scale 1.36 translate <10004.67,0,0> rotate <0,0,10.13,> rotate <0,-317.59,0>  pigment {rgb <0.41625,0.41625,0.41625>}}
object {star scale 1.36 translate <10004.67,0,0> rotate <0,0,15.40,> rotate <0,-76.14,0>  pigment {rgb <0.41625,0.41625,0.41625>}}
object {star scale 1.36 translate <10004.67,0,0> rotate <0,0,10.17,> rotate <0,-73.72,0>  pigment {rgb <0.41625,0.41625,0.41625>}}
object {star scale 1.36 translate <10004.67,0,0> rotate <0,0,-14.95,> rotate <0,-91.54,0>  pigment {rgb <0.41625,0.41625,0.41625>}}
object {star scale 1.36 translate <10004.67,0,0> rotate <0,0,21.47,> rotate <0,-130.82,0>  pigment {rgb <0.41625,0.41625,0.41625>}}
object {star scale 1.36 translate <10004.67,0,0> rotate <0,0,43.18,> rotate <0,-163.49,0>  pigment {rgb <0.41625,0.41625,0.41625>}}
object {star scale 1.36 translate <10004.67,0,0> rotate <0,0,4.37,> rotate <0,-270.07,0>  pigment {rgb <0.41625,0.41625,0.41625>}}
object {star scale 1.36 translate <10004.67,0,0> rotate <0,0,51.18,> rotate <0,-325.53,0>  pigment {rgb <0.41625,0.41625,0.41625>}}
object {star scale 1.36 translate <10004.67,0,0> rotate <0,0,31.98,> rotate <0,-159.68,0>  pigment {rgb <0.41625,0.41625,0.41625>}}
object {star scale 1.36 translate <10004.67,0,0> rotate <0,0,4.65,> rotate <0,-144.61,0>  pigment {rgb <0.41625,0.41625,0.41625>}}
object {star scale 1.36 translate <10004.67,0,0> rotate <0,0,-18.83,> rotate <0,-340.9,0>  pigment {rgb <0.41625,0.41625,0.41625>}}
object {star scale 1.36 translate <10004.67,0,0> rotate <0,0,-56.78,> rotate <0,-330.84,0>  pigment {rgb <0.41625,0.41625,0.41625>}}
object {star scale 1.35 translate <10004.68,0,0> rotate <0,0,-25.37,> rotate <0,-114.58,0>  pigment {rgb <0.415,0.415,0.415>}}
object {star scale 1.35 translate <10004.68,0,0> rotate <0,0,15.92,> rotate <0,-69.82,0>  pigment {rgb <0.415,0.415,0.415>}}
object {star scale 1.35 translate <10004.68,0,0> rotate <0,0,-7.85,> rotate <0,-324.44,0>  pigment {rgb <0.415,0.415,0.415>}}
object {star scale 1.35 translate <10004.68,0,0> rotate <0,0,-57.65,> rotate <0,-156.85,0>  pigment {rgb <0.415,0.415,0.415>}}
object {star scale 1.35 translate <10004.68,0,0> rotate <0,0,-67.63,> rotate <0,-28.73,0>  pigment {rgb <0.415,0.415,0.415>}}
object {star scale 1.35 translate <10004.68,0,0> rotate <0,0,-8.93,> rotate <0,-275.92,0>  pigment {rgb <0.415,0.415,0.415>}}
object {star scale 1.35 translate <10004.68,0,0> rotate <0,0,-1.40,> rotate <0,-120.3,0>  pigment {rgb <0.415,0.415,0.415>}}
object {star scale 1.35 translate <10004.68,0,0> rotate <0,0,-6.25,> rotate <0,-202.99,0>  pigment {rgb <0.415,0.415,0.415>}}
object {star scale 1.35 translate <10004.69,0,0> rotate <0,0,-20.92,> rotate <0,-353.32,0>  pigment {rgb <0.41375,0.41375,0.41375>}}
object {star scale 1.35 translate <10004.69,0,0> rotate <0,0,-52.72,> rotate <0,-134.08,0>  pigment {rgb <0.41375,0.41375,0.41375>}}
object {star scale 1.35 translate <10004.69,0,0> rotate <0,0,-22.45,> rotate <0,-347.48,0>  pigment {rgb <0.41375,0.41375,0.41375>}}
object {star scale 1.35 translate <10004.69,0,0> rotate <0,0,-29.30,> rotate <0,-31.13,0>  pigment {rgb <0.41375,0.41375,0.41375>}}
object {star scale 1.35 translate <10004.69,0,0> rotate <0,0,14.72,> rotate <0,-66.65,0>  pigment {rgb <0.41375,0.41375,0.41375>}}
object {star scale 1.35 translate <10004.69,0,0> rotate <0,0,69.65,> rotate <0,-293.09,0>  pigment {rgb <0.41375,0.41375,0.41375>}}
object {star scale 1.35 translate <10004.69,0,0> rotate <0,0,-22.33,> rotate <0,-141.83,0>  pigment {rgb <0.41375,0.41375,0.41375>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,-42.62,> rotate <0,-353.77,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,-21.23,> rotate <0,-80.11,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,70.90,> rotate <0,-30.49,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,-48.47,> rotate <0,-196.57,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,-62.93,> rotate <0,-52.35,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,-1.08,> rotate <0,-82.43,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,-23.28,> rotate <0,-202.43,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,-32.17,> rotate <0,-319.49,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,39.67,> rotate <0,-44.69,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,-14.57,> rotate <0,-277.3,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,-70.53,> rotate <0,-136.41,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,8.92,> rotate <0,-44.93,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,-34.73,> rotate <0,-175.05,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,-5.72,> rotate <0,-281.87,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,40.48,> rotate <0,-63.73,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,-26.30,> rotate <0,-298.96,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,-28.83,> rotate <0,-140.8,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,59.23,> rotate <0,-23.48,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,-47.08,> rotate <0,-117.09,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,8.47,> rotate <0,-298.56,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,-63.68,> rotate <0,-209.41,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,84.35,> rotate <0,-343.6,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,8.03,> rotate <0,-150.05,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.70,0,0> rotate <0,0,17.88,> rotate <0,-7.01,0>  pigment {rgb <0.4125,0.4125,0.4125>}}
object {star scale 1.35 translate <10004.71,0,0> rotate <0,0,40.10,> rotate <0,-79.78,0>  pigment {rgb <0.41125,0.41125,0.41125>}}
object {star scale 1.35 translate <10004.71,0,0> rotate <0,0,-12.93,> rotate <0,-122.82,0>  pigment {rgb <0.41125,0.41125,0.41125>}}
object {star scale 1.35 translate <10004.71,0,0> rotate <0,0,-63.17,> rotate <0,-181.08,0>  pigment {rgb <0.41125,0.41125,0.41125>}}
object {star scale 1.35 translate <10004.71,0,0> rotate <0,0,-57.48,> rotate <0,-76.38,0>  pigment {rgb <0.41125,0.41125,0.41125>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,-33.00,> rotate <0,-207.95,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,-31.93,> rotate <0,-208.3,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,-47.05,> rotate <0,-226.28,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,-9.05,> rotate <0,-280.57,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,68.12,> rotate <0,-21.48,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,-60.98,> rotate <0,-191.41,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,-45.17,> rotate <0,-241.62,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,-8.98,> rotate <0,-313.17,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,-45.50,> rotate <0,-354.46,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,-0.38,> rotate <0,-80.44,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,-36.73,> rotate <0,-109.58,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,40.57,> rotate <0,-199.39,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,-18.87,> rotate <0,-325.67,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,-18.35,> rotate <0,-176.19,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,50.28,> rotate <0,-36.4,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,-19.67,> rotate <0,-235.49,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,-25.97,> rotate <0,-140.38,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.72,0,0> rotate <0,0,14.25,> rotate <0,-73.13,0>  pigment {rgb <0.41,0.41,0.41>}}
object {star scale 1.35 translate <10004.73,0,0> rotate <0,0,28.73,> rotate <0,-326.03,0>  pigment {rgb <0.40875,0.40875,0.40875>}}
object {star scale 1.35 translate <10004.73,0,0> rotate <0,0,17.80,> rotate <0,-185.18,0>  pigment {rgb <0.40875,0.40875,0.40875>}}
object {star scale 1.35 translate <10004.73,0,0> rotate <0,0,-0.45,> rotate <0,-259.15,0>  pigment {rgb <0.40875,0.40875,0.40875>}}
object {star scale 1.35 translate <10004.73,0,0> rotate <0,0,-78.70,> rotate <0,-245.09,0>  pigment {rgb <0.40875,0.40875,0.40875>}}
object {star scale 1.35 translate <10004.73,0,0> rotate <0,0,-72.00,> rotate <0,-157.58,0>  pigment {rgb <0.40875,0.40875,0.40875>}}
object {star scale 1.35 translate <10004.73,0,0> rotate <0,0,34.45,> rotate <0,-292.94,0>  pigment {rgb <0.40875,0.40875,0.40875>}}
object {star scale 1.35 translate <10004.73,0,0> rotate <0,0,-5.07,> rotate <0,-52.65,0>  pigment {rgb <0.40875,0.40875,0.40875>}}
object {star scale 1.35 translate <10004.73,0,0> rotate <0,0,33.78,> rotate <0,-156.48,0>  pigment {rgb <0.40875,0.40875,0.40875>}}
object {star scale 1.35 translate <10004.73,0,0> rotate <0,0,-66.82,> rotate <0,-177.06,0>  pigment {rgb <0.40875,0.40875,0.40875>}}
object {star scale 1.34 translate <10004.74,0,0> rotate <0,0,-42.88,> rotate <0,-39.95,0>  pigment {rgb <0.4075,0.4075,0.4075>}}
object {star scale 1.34 translate <10004.74,0,0> rotate <0,0,-42.98,> rotate <0,-122.86,0>  pigment {rgb <0.4075,0.4075,0.4075>}}
object {star scale 1.34 translate <10004.74,0,0> rotate <0,0,-23.58,> rotate <0,-145.32,0>  pigment {rgb <0.4075,0.4075,0.4075>}}
object {star scale 1.34 translate <10004.74,0,0> rotate <0,0,13.17,> rotate <0,-103.66,0>  pigment {rgb <0.4075,0.4075,0.4075>}}
object {star scale 1.34 translate <10004.74,0,0> rotate <0,0,68.12,> rotate <0,-349.66,0>  pigment {rgb <0.4075,0.4075,0.4075>}}
object {star scale 1.34 translate <10004.74,0,0> rotate <0,0,46.03,> rotate <0,-240.7,0>  pigment {rgb <0.4075,0.4075,0.4075>}}
object {star scale 1.34 translate <10004.74,0,0> rotate <0,0,-42.10,> rotate <0,-285.78,0>  pigment {rgb <0.4075,0.4075,0.4075>}}
object {star scale 1.34 translate <10004.74,0,0> rotate <0,0,62.08,> rotate <0,-324.48,0>  pigment {rgb <0.4075,0.4075,0.4075>}}
object {star scale 1.34 translate <10004.74,0,0> rotate <0,0,-34.70,> rotate <0,-235.67,0>  pigment {rgb <0.4075,0.4075,0.4075>}}
object {star scale 1.34 translate <10004.74,0,0> rotate <0,0,-18.30,> rotate <0,-199.6,0>  pigment {rgb <0.4075,0.4075,0.4075>}}
object {star scale 1.34 translate <10004.74,0,0> rotate <0,0,-75.07,> rotate <0,-42.62,0>  pigment {rgb <0.4075,0.4075,0.4075>}}
object {star scale 1.34 translate <10004.74,0,0> rotate <0,0,34.43,> rotate <0,-207.95,0>  pigment {rgb <0.4075,0.4075,0.4075>}}
object {star scale 1.34 translate <10004.74,0,0> rotate <0,0,-2.48,> rotate <0,-165.46,0>  pigment {rgb <0.4075,0.4075,0.4075>}}
object {star scale 1.34 translate <10004.74,0,0> rotate <0,0,37.32,> rotate <0,-87.76,0>  pigment {rgb <0.4075,0.4075,0.4075>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,40.42,> rotate <0,-158.31,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,51.37,> rotate <0,-214.04,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,-48.27,> rotate <0,-108.66,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,19.28,> rotate <0,-28.38,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,-12.75,> rotate <0,-305.17,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,-53.45,> rotate <0,-212.48,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,1.85,> rotate <0,-88.11,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,83.15,> rotate <0,-341.88,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,-23.82,> rotate <0,-269.95,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,-2.15,> rotate <0,-330.83,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,-9.80,> rotate <0,-174.17,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,-15.25,> rotate <0,-38.02,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,-61.30,> rotate <0,-122.25,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,56.72,> rotate <0,-46.38,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,-10.60,> rotate <0,-11.05,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,-15.97,> rotate <0,-201.86,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,8.05,> rotate <0,-101.83,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.75,0,0> rotate <0,0,20.98,> rotate <0,-237.81,0>  pigment {rgb <0.40625,0.40625,0.40625>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,-48.80,> rotate <0,-7.85,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,47.67,> rotate <0,-225.95,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,-21.00,> rotate <0,-42.76,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,27.27,> rotate <0,-19.87,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,-47.32,> rotate <0,-130.3,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,22.20,> rotate <0,-66.36,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,-45.20,> rotate <0,-215.18,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,-42.37,> rotate <0,-253.5,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,32.18,> rotate <0,-83.18,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,-60.58,> rotate <0,-308.9,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,-49.42,> rotate <0,-265.1,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,25.47,> rotate <0,-346.78,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,36.70,> rotate <0,-73.16,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,-3.00,> rotate <0,-172.58,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,-37.93,> rotate <0,-300.89,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,-27.77,> rotate <0,-215.78,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.76,0,0> rotate <0,0,67.63,> rotate <0,-135.64,0>  pigment {rgb <0.405,0.405,0.405>}}
object {star scale 1.34 translate <10004.77,0,0> rotate <0,0,21.40,> rotate <0,-289.05,0>  pigment {rgb <0.40375,0.40375,0.40375>}}
object {star scale 1.34 translate <10004.77,0,0> rotate <0,0,23.60,> rotate <0,-29.48,0>  pigment {rgb <0.40375,0.40375,0.40375>}}
object {star scale 1.34 translate <10004.77,0,0> rotate <0,0,-17.82,> rotate <0,-307.21,0>  pigment {rgb <0.40375,0.40375,0.40375>}}
object {star scale 1.34 translate <10004.77,0,0> rotate <0,0,38.18,> rotate <0,-169.78,0>  pigment {rgb <0.40375,0.40375,0.40375>}}
object {star scale 1.34 translate <10004.77,0,0> rotate <0,0,57.05,> rotate <0,-278.14,0>  pigment {rgb <0.40375,0.40375,0.40375>}}
object {star scale 1.34 translate <10004.77,0,0> rotate <0,0,51.57,> rotate <0,-186,0>  pigment {rgb <0.40375,0.40375,0.40375>}}
object {star scale 1.34 translate <10004.77,0,0> rotate <0,0,-67.77,> rotate <0,-260.5,0>  pigment {rgb <0.40375,0.40375,0.40375>}}
object {star scale 1.34 translate <10004.77,0,0> rotate <0,0,-77.07,> rotate <0,-0.4,0>  pigment {rgb <0.40375,0.40375,0.40375>}}
object {star scale 1.34 translate <10004.77,0,0> rotate <0,0,-1.13,> rotate <0,-13.25,0>  pigment {rgb <0.40375,0.40375,0.40375>}}
object {star scale 1.34 translate <10004.77,0,0> rotate <0,0,17.42,> rotate <0,-194.73,0>  pigment {rgb <0.40375,0.40375,0.40375>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,-6.00,> rotate <0,-83.76,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,25.58,> rotate <0,-303.82,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,-27.77,> rotate <0,-146.05,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,-11.97,> rotate <0,-139.94,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,36.50,> rotate <0,-242.24,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,69.32,> rotate <0,-94.71,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,-28.62,> rotate <0,-244.58,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,-30.33,> rotate <0,-119.42,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,16.20,> rotate <0,-67.64,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,-64.87,> rotate <0,-281.36,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,50.52,> rotate <0,-10.52,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,-35.90,> rotate <0,-123.38,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,44.40,> rotate <0,-313.31,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,-12.53,> rotate <0,-74.98,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,63.35,> rotate <0,-56.51,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,-41.35,> rotate <0,-333.9,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.78,0,0> rotate <0,0,72.35,> rotate <0,-332.45,0>  pigment {rgb <0.4025,0.4025,0.4025>}}
object {star scale 1.34 translate <10004.79,0,0> rotate <0,0,28.33,> rotate <0,-335.33,0>  pigment {rgb <0.40125,0.40125,0.40125>}}
object {star scale 1.34 translate <10004.79,0,0> rotate <0,0,54.52,> rotate <0,-7.94,0>  pigment {rgb <0.40125,0.40125,0.40125>}}
object {star scale 1.34 translate <10004.79,0,0> rotate <0,0,-25.12,> rotate <0,-247.55,0>  pigment {rgb <0.40125,0.40125,0.40125>}}
object {star scale 1.34 translate <10004.79,0,0> rotate <0,0,-67.90,> rotate <0,-198.81,0>  pigment {rgb <0.40125,0.40125,0.40125>}}
object {star scale 1.34 translate <10004.79,0,0> rotate <0,0,4.70,> rotate <0,-336.97,0>  pigment {rgb <0.40125,0.40125,0.40125>}}
object {star scale 1.34 translate <10004.79,0,0> rotate <0,0,42.50,> rotate <0,-99.83,0>  pigment {rgb <0.40125,0.40125,0.40125>}}
object {star scale 1.34 translate <10004.79,0,0> rotate <0,0,5.47,> rotate <0,-199.4,0>  pigment {rgb <0.40125,0.40125,0.40125>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,51.52,> rotate <0,-122.11,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,-40.08,> rotate <0,-267.55,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,-59.23,> rotate <0,-134.25,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,67.13,> rotate <0,-137.6,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,61.13,> rotate <0,-13.27,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,-2.23,> rotate <0,-217.05,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,51.38,> rotate <0,-359.6,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,-8.82,> rotate <0,-48.96,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,21.20,> rotate <0,-309.63,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,-23.97,> rotate <0,-262.85,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,-7.22,> rotate <0,-84.72,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,37.85,> rotate <0,-32.12,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,17.45,> rotate <0,-66.03,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,25.95,> rotate <0,-89.5,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,-54.97,> rotate <0,-92.58,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,-7.17,> rotate <0,-75.36,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,-64.47,> rotate <0,-160.56,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,-13.38,> rotate <0,-159.39,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,68.75,> rotate <0,-264.24,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,19.48,> rotate <0,-154.93,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,-62.42,> rotate <0,-140.24,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,35.50,> rotate <0,-214.5,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,25.00,> rotate <0,-225.53,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,27.63,> rotate <0,-196.8,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,-51.42,> rotate <0,-105.22,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,20.20,> rotate <0,-3.65,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.80,0,0> rotate <0,0,-9.55,> rotate <0,-193.59,0>  pigment {rgb <0.4,0.4,0.4>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,-51.45,> rotate <0,-186.63,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,25.85,> rotate <0,-185.63,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,-9.55,> rotate <0,-140.12,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,-20.55,> rotate <0,-276.34,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,-40.82,> rotate <0,-320.19,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,22.63,> rotate <0,-188.71,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,29.30,> rotate <0,-340.44,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,-63.57,> rotate <0,-120.08,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,-44.50,> rotate <0,-228.2,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,1.03,> rotate <0,-245.52,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,15.60,> rotate <0,-77.42,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,39.62,> rotate <0,-143.77,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,35.65,> rotate <0,-237.81,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,-34.90,> rotate <0,-79.37,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,71.30,> rotate <0,-283.6,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,26.65,> rotate <0,-281.52,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,-4.22,> rotate <0,-255.26,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.81,0,0> rotate <0,0,-10.87,> rotate <0,-171.15,0>  pigment {rgb <0.39875,0.39875,0.39875>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,-48.48,> rotate <0,-125.63,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,-22.75,> rotate <0,-283.54,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,58.97,> rotate <0,-13.75,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,-23.32,> rotate <0,-109.15,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,22.80,> rotate <0,-240.57,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,54.02,> rotate <0,-139.05,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,-20.22,> rotate <0,-103.38,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,-39.67,> rotate <0,-107.21,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,-63.95,> rotate <0,-161.03,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,-31.70,> rotate <0,-267.3,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,-17.82,> rotate <0,-355.44,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,-33.05,> rotate <0,-125.35,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,34.22,> rotate <0,-49.68,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,65.57,> rotate <0,-276.5,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,-67.18,> rotate <0,-78.44,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,-5.85,> rotate <0,-284.26,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,77.55,> rotate <0,-212.21,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,26.52,> rotate <0,-220.86,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.82,0,0> rotate <0,0,-33.02,> rotate <0,-6.98,0>  pigment {rgb <0.3975,0.3975,0.3975>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,45.98,> rotate <0,-252.31,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,46.82,> rotate <0,-303.33,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,-19.85,> rotate <0,-316.1,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,36.28,> rotate <0,-204.37,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,46.12,> rotate <0,-312.23,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,33.10,> rotate <0,-259.33,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,45.40,> rotate <0,-21.91,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,9.27,> rotate <0,-63.48,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,-41.42,> rotate <0,-340.88,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,49.02,> rotate <0,-349.43,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,-10.67,> rotate <0,-337.66,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,8.27,> rotate <0,-176.32,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,-67.62,> rotate <0,-41.39,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,38.03,> rotate <0,-304.45,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,65.65,> rotate <0,-50,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,55.98,> rotate <0,-157.66,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,25.10,> rotate <0,-212.6,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,-47.38,> rotate <0,-29.29,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,-26.20,> rotate <0,-299.74,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,-41.65,> rotate <0,-155.58,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,64.60,> rotate <0,-250.23,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,-14.13,> rotate <0,-99.82,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,44.23,> rotate <0,-33.3,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.83,0,0> rotate <0,0,6.15,> rotate <0,-22.55,0>  pigment {rgb <0.39625,0.39625,0.39625>}}
object {star scale 1.33 translate <10004.84,0,0> rotate <0,0,-23.02,> rotate <0,-112.47,0>  pigment {rgb <0.395,0.395,0.395>}}
object {star scale 1.33 translate <10004.84,0,0> rotate <0,0,16.53,> rotate <0,-85.33,0>  pigment {rgb <0.395,0.395,0.395>}}
object {star scale 1.33 translate <10004.84,0,0> rotate <0,0,58.55,> rotate <0,-352.51,0>  pigment {rgb <0.395,0.395,0.395>}}
object {star scale 1.33 translate <10004.84,0,0> rotate <0,0,-11.88,> rotate <0,-39.89,0>  pigment {rgb <0.395,0.395,0.395>}}
object {star scale 1.33 translate <10004.84,0,0> rotate <0,0,-53.50,> rotate <0,-341.41,0>  pigment {rgb <0.395,0.395,0.395>}}
object {star scale 1.33 translate <10004.84,0,0> rotate <0,0,74.40,> rotate <0,-47.98,0>  pigment {rgb <0.395,0.395,0.395>}}
object {star scale 1.33 translate <10004.84,0,0> rotate <0,0,-49.52,> rotate <0,-195.89,0>  pigment {rgb <0.395,0.395,0.395>}}
object {star scale 1.33 translate <10004.84,0,0> rotate <0,0,-38.32,> rotate <0,-114.86,0>  pigment {rgb <0.395,0.395,0.395>}}
object {star scale 1.33 translate <10004.84,0,0> rotate <0,0,-63.12,> rotate <0,-186.63,0>  pigment {rgb <0.395,0.395,0.395>}}
object {star scale 1.33 translate <10004.84,0,0> rotate <0,0,-37.80,> rotate <0,-198.01,0>  pigment {rgb <0.395,0.395,0.395>}}
object {star scale 1.33 translate <10004.84,0,0> rotate <0,0,3.37,> rotate <0,-49.84,0>  pigment {rgb <0.395,0.395,0.395>}}
object {star scale 1.33 translate <10004.84,0,0> rotate <0,0,75.72,> rotate <0,-158.77,0>  pigment {rgb <0.395,0.395,0.395>}}
object {star scale 1.33 translate <10004.84,0,0> rotate <0,0,3.62,> rotate <0,-165.14,0>  pigment {rgb <0.395,0.395,0.395>}}
object {star scale 1.33 translate <10004.84,0,0> rotate <0,0,11.48,> rotate <0,-248.15,0>  pigment {rgb <0.395,0.395,0.395>}}
object {star scale 1.33 translate <10004.84,0,0> rotate <0,0,5.05,> rotate <0,-331.42,0>  pigment {rgb <0.395,0.395,0.395>}}
object {star scale 1.33 translate <10004.85,0,0> rotate <0,0,-25.27,> rotate <0,-288.89,0>  pigment {rgb <0.39375,0.39375,0.39375>}}
object {star scale 1.33 translate <10004.85,0,0> rotate <0,0,3.38,> rotate <0,-275.21,0>  pigment {rgb <0.39375,0.39375,0.39375>}}
object {star scale 1.33 translate <10004.85,0,0> rotate <0,0,-15.95,> rotate <0,-130.43,0>  pigment {rgb <0.39375,0.39375,0.39375>}}
object {star scale 1.33 translate <10004.85,0,0> rotate <0,0,-51.80,> rotate <0,-152.24,0>  pigment {rgb <0.39375,0.39375,0.39375>}}
object {star scale 1.33 translate <10004.85,0,0> rotate <0,0,-64.38,> rotate <0,-161.72,0>  pigment {rgb <0.39375,0.39375,0.39375>}}
object {star scale 1.33 translate <10004.85,0,0> rotate <0,0,46.50,> rotate <0,-65.39,0>  pigment {rgb <0.39375,0.39375,0.39375>}}
object {star scale 1.33 translate <10004.85,0,0> rotate <0,0,-58.02,> rotate <0,-128.83,0>  pigment {rgb <0.39375,0.39375,0.39375>}}
object {star scale 1.33 translate <10004.85,0,0> rotate <0,0,-44.35,> rotate <0,-269.2,0>  pigment {rgb <0.39375,0.39375,0.39375>}}
object {star scale 1.33 translate <10004.85,0,0> rotate <0,0,30.88,> rotate <0,-245.53,0>  pigment {rgb <0.39375,0.39375,0.39375>}}
object {star scale 1.33 translate <10004.85,0,0> rotate <0,0,-63.60,> rotate <0,-229.41,0>  pigment {rgb <0.39375,0.39375,0.39375>}}
object {star scale 1.33 translate <10004.85,0,0> rotate <0,0,16.32,> rotate <0,-214.94,0>  pigment {rgb <0.39375,0.39375,0.39375>}}
object {star scale 1.33 translate <10004.85,0,0> rotate <0,0,-22.53,> rotate <0,-29.17,0>  pigment {rgb <0.39375,0.39375,0.39375>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,-37.63,> rotate <0,-57.15,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,21.93,> rotate <0,-81.91,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,-53.12,> rotate <0,-130.6,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,34.22,> rotate <0,-34.26,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,-29.55,> rotate <0,-129.93,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,-3.57,> rotate <0,-359.67,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,-10.25,> rotate <0,-63.6,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,38.48,> rotate <0,-78.36,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,-14.27,> rotate <0,-239.55,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,59.42,> rotate <0,-346.65,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,-34.12,> rotate <0,-256.21,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,45.43,> rotate <0,-191.28,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,56.78,> rotate <0,-251.33,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,24.57,> rotate <0,-87.25,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,8.15,> rotate <0,-220.41,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,-66.78,> rotate <0,-199.31,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.86,0,0> rotate <0,0,-19.77,> rotate <0,-296.59,0>  pigment {rgb <0.3925,0.3925,0.3925>}}
object {star scale 1.32 translate <10004.87,0,0> rotate <0,0,59.45,> rotate <0,-101.56,0>  pigment {rgb <0.39125,0.39125,0.39125>}}
object {star scale 1.32 translate <10004.87,0,0> rotate <0,0,-35.63,> rotate <0,-281.08,0>  pigment {rgb <0.39125,0.39125,0.39125>}}
object {star scale 1.32 translate <10004.87,0,0> rotate <0,0,-22.52,> rotate <0,-49.59,0>  pigment {rgb <0.39125,0.39125,0.39125>}}
object {star scale 1.32 translate <10004.87,0,0> rotate <0,0,5.58,> rotate <0,-38.97,0>  pigment {rgb <0.39125,0.39125,0.39125>}}
object {star scale 1.32 translate <10004.87,0,0> rotate <0,0,39.05,> rotate <0,-339.81,0>  pigment {rgb <0.39125,0.39125,0.39125>}}
object {star scale 1.32 translate <10004.87,0,0> rotate <0,0,-33.80,> rotate <0,-88.28,0>  pigment {rgb <0.39125,0.39125,0.39125>}}
object {star scale 1.32 translate <10004.87,0,0> rotate <0,0,-37.10,> rotate <0,-284.68,0>  pigment {rgb <0.39125,0.39125,0.39125>}}
object {star scale 1.32 translate <10004.87,0,0> rotate <0,0,-21.68,> rotate <0,-265.86,0>  pigment {rgb <0.39125,0.39125,0.39125>}}
object {star scale 1.32 translate <10004.87,0,0> rotate <0,0,45.52,> rotate <0,-20.58,0>  pigment {rgb <0.39125,0.39125,0.39125>}}
object {star scale 1.32 translate <10004.87,0,0> rotate <0,0,58.65,> rotate <0,-356.77,0>  pigment {rgb <0.39125,0.39125,0.39125>}}
object {star scale 1.32 translate <10004.87,0,0> rotate <0,0,-41.72,> rotate <0,-269.45,0>  pigment {rgb <0.39125,0.39125,0.39125>}}
object {star scale 1.32 translate <10004.87,0,0> rotate <0,0,-61.98,> rotate <0,-338.25,0>  pigment {rgb <0.39125,0.39125,0.39125>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,55.18,> rotate <0,-263.04,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,55.17,> rotate <0,-263.07,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,-27.68,> rotate <0,-133.88,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,55.75,> rotate <0,-359.75,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,-53.72,> rotate <0,-157.84,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,-33.78,> rotate <0,-312.49,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,-45.18,> rotate <0,-108.3,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,10.23,> rotate <0,-190.47,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,-12.28,> rotate <0,-36.49,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,-15.47,> rotate <0,-2.81,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,-48.10,> rotate <0,-293.8,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,38.53,> rotate <0,-323.69,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,-25.93,> rotate <0,-148.55,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,37.50,> rotate <0,-72.48,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,-2.55,> rotate <0,-309.18,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,18.50,> rotate <0,-116.53,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,-46.23,> rotate <0,-312.37,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,-27.42,> rotate <0,-159.31,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.88,0,0> rotate <0,0,-16.30,> rotate <0,-212.71,0>  pigment {rgb <0.39,0.39,0.39>}}
object {star scale 1.32 translate <10004.89,0,0> rotate <0,0,-65.20,> rotate <0,-177.97,0>  pigment {rgb <0.38875,0.38875,0.38875>}}
object {star scale 1.32 translate <10004.89,0,0> rotate <0,0,-8.27,> rotate <0,-280.88,0>  pigment {rgb <0.38875,0.38875,0.38875>}}
object {star scale 1.32 translate <10004.89,0,0> rotate <0,0,30.78,> rotate <0,-195.07,0>  pigment {rgb <0.38875,0.38875,0.38875>}}
object {star scale 1.32 translate <10004.89,0,0> rotate <0,0,45.08,> rotate <0,-104.4,0>  pigment {rgb <0.38875,0.38875,0.38875>}}
object {star scale 1.32 translate <10004.89,0,0> rotate <0,0,-38.60,> rotate <0,-240.85,0>  pigment {rgb <0.38875,0.38875,0.38875>}}
object {star scale 1.32 translate <10004.89,0,0> rotate <0,0,12.65,> rotate <0,-87.39,0>  pigment {rgb <0.38875,0.38875,0.38875>}}
object {star scale 1.32 translate <10004.89,0,0> rotate <0,0,34.58,> rotate <0,-114.79,0>  pigment {rgb <0.38875,0.38875,0.38875>}}
object {star scale 1.32 translate <10004.89,0,0> rotate <0,0,65.13,> rotate <0,-254.01,0>  pigment {rgb <0.38875,0.38875,0.38875>}}
object {star scale 1.32 translate <10004.89,0,0> rotate <0,0,-70.10,> rotate <0,-247.12,0>  pigment {rgb <0.38875,0.38875,0.38875>}}
object {star scale 1.32 translate <10004.89,0,0> rotate <0,0,-14.60,> rotate <0,-21.4,0>  pigment {rgb <0.38875,0.38875,0.38875>}}
object {star scale 1.32 translate <10004.89,0,0> rotate <0,0,-2.95,> rotate <0,-95,0>  pigment {rgb <0.38875,0.38875,0.38875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,12.75,> rotate <0,-256.34,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,-78.22,> rotate <0,-179.9,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,-28.23,> rotate <0,-38.46,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,18.65,> rotate <0,-76.86,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,48.92,> rotate <0,-249.69,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,-34.00,> rotate <0,-192.68,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,21.03,> rotate <0,-48.73,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,8.82,> rotate <0,-343.81,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,-81.00,> rotate <0,-214.55,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,-44.05,> rotate <0,-248.52,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,-20.05,> rotate <0,-75.35,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,50.97,> rotate <0,-12.21,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,-31.52,> rotate <0,-228.66,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,40.18,> rotate <0,-40.56,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,-40.50,> rotate <0,-176.63,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,37.35,> rotate <0,-296.07,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,-16.87,> rotate <0,-159.65,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,1.48,> rotate <0,-85.62,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,25.27,> rotate <0,-311.22,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,-2.88,> rotate <0,-307.41,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,63.63,> rotate <0,-329.16,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,61.52,> rotate <0,-94.48,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,82.40,> rotate <0,-112.77,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.90,0,0> rotate <0,0,41.88,> rotate <0,-247.16,0>  pigment {rgb <0.3875,0.3875,0.3875>}}
object {star scale 1.32 translate <10004.91,0,0> rotate <0,0,52.43,> rotate <0,-298.91,0>  pigment {rgb <0.38625,0.38625,0.38625>}}
object {star scale 1.32 translate <10004.91,0,0> rotate <0,0,-59.10,> rotate <0,-198.56,0>  pigment {rgb <0.38625,0.38625,0.38625>}}
object {star scale 1.32 translate <10004.91,0,0> rotate <0,0,-35.18,> rotate <0,-221.25,0>  pigment {rgb <0.38625,0.38625,0.38625>}}
object {star scale 1.32 translate <10004.91,0,0> rotate <0,0,-59.68,> rotate <0,-190.48,0>  pigment {rgb <0.38625,0.38625,0.38625>}}
object {star scale 1.32 translate <10004.91,0,0> rotate <0,0,50.70,> rotate <0,-283.31,0>  pigment {rgb <0.38625,0.38625,0.38625>}}
object {star scale 1.32 translate <10004.91,0,0> rotate <0,0,-4.23,> rotate <0,-107.56,0>  pigment {rgb <0.38625,0.38625,0.38625>}}
object {star scale 1.32 translate <10004.91,0,0> rotate <0,0,-0.05,> rotate <0,-67.97,0>  pigment {rgb <0.38625,0.38625,0.38625>}}
object {star scale 1.32 translate <10004.91,0,0> rotate <0,0,21.27,> rotate <0,-207.43,0>  pigment {rgb <0.38625,0.38625,0.38625>}}
object {star scale 1.32 translate <10004.91,0,0> rotate <0,0,39.32,> rotate <0,-107.92,0>  pigment {rgb <0.38625,0.38625,0.38625>}}
object {star scale 1.31 translate <10004.92,0,0> rotate <0,0,-19.80,> rotate <0,-241.36,0>  pigment {rgb <0.385,0.385,0.385>}}
object {star scale 1.31 translate <10004.92,0,0> rotate <0,0,-48.22,> rotate <0,-99.66,0>  pigment {rgb <0.385,0.385,0.385>}}
object {star scale 1.31 translate <10004.92,0,0> rotate <0,0,-58.45,> rotate <0,-215.65,0>  pigment {rgb <0.385,0.385,0.385>}}
object {star scale 1.31 translate <10004.92,0,0> rotate <0,0,1.25,> rotate <0,-351.73,0>  pigment {rgb <0.385,0.385,0.385>}}
object {star scale 1.31 translate <10004.92,0,0> rotate <0,0,-8.52,> rotate <0,-225.25,0>  pigment {rgb <0.385,0.385,0.385>}}
object {star scale 1.31 translate <10004.92,0,0> rotate <0,0,-49.58,> rotate <0,-105.98,0>  pigment {rgb <0.385,0.385,0.385>}}
object {star scale 1.31 translate <10004.92,0,0> rotate <0,0,-45.30,> rotate <0,-132.45,0>  pigment {rgb <0.385,0.385,0.385>}}
object {star scale 1.31 translate <10004.92,0,0> rotate <0,0,-16.48,> rotate <0,-91.25,0>  pigment {rgb <0.385,0.385,0.385>}}
object {star scale 1.31 translate <10004.92,0,0> rotate <0,0,-41.98,> rotate <0,-62.71,0>  pigment {rgb <0.385,0.385,0.385>}}
object {star scale 1.31 translate <10004.92,0,0> rotate <0,0,-32.55,> rotate <0,-332.53,0>  pigment {rgb <0.385,0.385,0.385>}}
object {star scale 1.31 translate <10004.92,0,0> rotate <0,0,-17.75,> rotate <0,-250.39,0>  pigment {rgb <0.385,0.385,0.385>}}
object {star scale 1.31 translate <10004.92,0,0> rotate <0,0,22.10,> rotate <0,-79.82,0>  pigment {rgb <0.385,0.385,0.385>}}
object {star scale 1.31 translate <10004.92,0,0> rotate <0,0,40.15,> rotate <0,-198.43,0>  pigment {rgb <0.385,0.385,0.385>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,56.37,> rotate <0,-195.18,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,21.23,> rotate <0,-193.32,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,29.35,> rotate <0,-356,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,3.65,> rotate <0,-203.53,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,-10.07,> rotate <0,-243,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,-1.13,> rotate <0,-85.21,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,27.27,> rotate <0,-186.6,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,24.88,> rotate <0,-226.83,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,-10.33,> rotate <0,-231.05,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,-23.92,> rotate <0,-145.56,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,-62.08,> rotate <0,-175.23,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,34.57,> rotate <0,-65.1,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,-38.57,> rotate <0,-138.9,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,-52.53,> rotate <0,-113.91,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,-3.67,> rotate <0,-119.93,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,32.13,> rotate <0,-285,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.93,0,0> rotate <0,0,-52.88,> rotate <0,-301.85,0>  pigment {rgb <0.38375,0.38375,0.38375>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,37.88,> rotate <0,-74.81,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,16.42,> rotate <0,-220.18,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,-27.28,> rotate <0,-166.33,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,30.30,> rotate <0,-33.09,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,-73.22,> rotate <0,-157.76,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,43.32,> rotate <0,-343.01,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,65.57,> rotate <0,-156.03,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,36.80,> rotate <0,-303.63,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,43.33,> rotate <0,-50.36,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,75.75,> rotate <0,-244.38,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,1.83,> rotate <0,-81.18,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,22.60,> rotate <0,-297.77,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,36.85,> rotate <0,-302.36,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,16.13,> rotate <0,-93.01,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,38.48,> rotate <0,-298.96,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,27.53,> rotate <0,-192.93,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,-2.30,> rotate <0,-222.75,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,23.95,> rotate <0,-184.08,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,70.02,> rotate <0,-188.68,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,35.18,> rotate <0,-44.77,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,27.78,> rotate <0,-120.88,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,-15.27,> rotate <0,-115.1,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,21.97,> rotate <0,-275.08,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,-48.72,> rotate <0,-104.07,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.94,0,0> rotate <0,0,-19.00,> rotate <0,-148.72,0>  pigment {rgb <0.3825,0.3825,0.3825>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,48.95,> rotate <0,-307.52,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,-10.60,> rotate <0,-90.46,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,40.57,> rotate <0,-25.14,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,27.35,> rotate <0,-65.09,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,55.70,> rotate <0,-88.71,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,15.20,> rotate <0,-303.57,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,47.03,> rotate <0,-10.87,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,-7.03,> rotate <0,-294.23,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,-45.92,> rotate <0,-277.94,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,54.75,> rotate <0,-239.45,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,-45.53,> rotate <0,-18.8,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,42.62,> rotate <0,-25.45,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,46.42,> rotate <0,-356.51,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,2.85,> rotate <0,-171.99,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,3.32,> rotate <0,-185.09,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,-23.83,> rotate <0,-235.07,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.95,0,0> rotate <0,0,-24.57,> rotate <0,-109.67,0>  pigment {rgb <0.38125,0.38125,0.38125>}}
object {star scale 1.31 translate <10004.96,0,0> rotate <0,0,21.60,> rotate <0,-270.38,0>  pigment {rgb <0.38,0.38,0.38>}}
object {star scale 1.31 translate <10004.96,0,0> rotate <0,0,33.97,> rotate <0,-55.59,0>  pigment {rgb <0.38,0.38,0.38>}}
object {star scale 1.31 translate <10004.96,0,0> rotate <0,0,-29.50,> rotate <0,-217.05,0>  pigment {rgb <0.38,0.38,0.38>}}
object {star scale 1.31 translate <10004.96,0,0> rotate <0,0,34.37,> rotate <0,-311.79,0>  pigment {rgb <0.38,0.38,0.38>}}
object {star scale 1.31 translate <10004.96,0,0> rotate <0,0,20.58,> rotate <0,-64.32,0>  pigment {rgb <0.38,0.38,0.38>}}
object {star scale 1.31 translate <10004.96,0,0> rotate <0,0,26.75,> rotate <0,-118.38,0>  pigment {rgb <0.38,0.38,0.38>}}
object {star scale 1.31 translate <10004.96,0,0> rotate <0,0,-19.02,> rotate <0,-110.55,0>  pigment {rgb <0.38,0.38,0.38>}}
object {star scale 1.31 translate <10004.96,0,0> rotate <0,0,-70.22,> rotate <0,-177.48,0>  pigment {rgb <0.38,0.38,0.38>}}
object {star scale 1.31 translate <10004.96,0,0> rotate <0,0,-18.95,> rotate <0,-289.41,0>  pigment {rgb <0.38,0.38,0.38>}}
object {star scale 1.31 translate <10004.96,0,0> rotate <0,0,-37.12,> rotate <0,-88.87,0>  pigment {rgb <0.38,0.38,0.38>}}
object {star scale 1.31 translate <10004.96,0,0> rotate <0,0,11.30,> rotate <0,-142.98,0>  pigment {rgb <0.38,0.38,0.38>}}
object {star scale 1.31 translate <10004.96,0,0> rotate <0,0,16.35,> rotate <0,-67.11,0>  pigment {rgb <0.38,0.38,0.38>}}
object {star scale 1.31 translate <10004.96,0,0> rotate <0,0,-61.08,> rotate <0,-60.33,0>  pigment {rgb <0.38,0.38,0.38>}}
object {star scale 1.31 translate <10004.96,0,0> rotate <0,0,58.85,> rotate <0,-298.98,0>  pigment {rgb <0.38,0.38,0.38>}}
object {star scale 1.31 translate <10004.96,0,0> rotate <0,0,49.73,> rotate <0,-344.11,0>  pigment {rgb <0.38,0.38,0.38>}}
object {star scale 1.31 translate <10004.96,0,0> rotate <0,0,77.35,> rotate <0,-232.85,0>  pigment {rgb <0.38,0.38,0.38>}}
object {star scale 1.31 translate <10004.97,0,0> rotate <0,0,4.20,> rotate <0,-284.06,0>  pigment {rgb <0.37875,0.37875,0.37875>}}
object {star scale 1.31 translate <10004.97,0,0> rotate <0,0,31.32,> rotate <0,-353.49,0>  pigment {rgb <0.37875,0.37875,0.37875>}}
object {star scale 1.31 translate <10004.97,0,0> rotate <0,0,29.63,> rotate <0,-291.03,0>  pigment {rgb <0.37875,0.37875,0.37875>}}
object {star scale 1.31 translate <10004.97,0,0> rotate <0,0,63.08,> rotate <0,-59.36,0>  pigment {rgb <0.37875,0.37875,0.37875>}}
object {star scale 1.31 translate <10004.97,0,0> rotate <0,0,68.77,> rotate <0,-247,0>  pigment {rgb <0.37875,0.37875,0.37875>}}
object {star scale 1.31 translate <10004.97,0,0> rotate <0,0,13.78,> rotate <0,-202.1,0>  pigment {rgb <0.37875,0.37875,0.37875>}}
object {star scale 1.31 translate <10004.97,0,0> rotate <0,0,43.48,> rotate <0,-170.7,0>  pigment {rgb <0.37875,0.37875,0.37875>}}
object {star scale 1.31 translate <10004.97,0,0> rotate <0,0,-23.12,> rotate <0,-197.26,0>  pigment {rgb <0.37875,0.37875,0.37875>}}
object {star scale 1.31 translate <10004.97,0,0> rotate <0,0,5.10,> rotate <0,-136.5,0>  pigment {rgb <0.37875,0.37875,0.37875>}}
object {star scale 1.31 translate <10004.97,0,0> rotate <0,0,-12.47,> rotate <0,-130,0>  pigment {rgb <0.37875,0.37875,0.37875>}}
object {star scale 1.31 translate <10004.97,0,0> rotate <0,0,62.28,> rotate <0,-351.21,0>  pigment {rgb <0.37875,0.37875,0.37875>}}
object {star scale 1.31 translate <10004.97,0,0> rotate <0,0,31.40,> rotate <0,-272.98,0>  pigment {rgb <0.37875,0.37875,0.37875>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,-9.60,> rotate <0,-349.74,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,58.72,> rotate <0,-115.75,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,-54.55,> rotate <0,-205.43,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,51.60,> rotate <0,-76.67,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,58.23,> rotate <0,-20.02,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,-64.07,> rotate <0,-44.7,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,-4.25,> rotate <0,-105.73,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,-66.90,> rotate <0,-155.74,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,68.68,> rotate <0,-29,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,37.18,> rotate <0,-203.7,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,17.38,> rotate <0,-81.1,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,-50.07,> rotate <0,-244.25,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,-15.78,> rotate <0,-123.33,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,-41.73,> rotate <0,-239.88,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,44.40,> rotate <0,-24.84,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,-34.37,> rotate <0,-102.72,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,57.70,> rotate <0,-288.48,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,-3.70,> rotate <0,-25.68,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,14.08,> rotate <0,-255.78,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.98,0,0> rotate <0,0,43.93,> rotate <0,-319.61,0>  pigment {rgb <0.3775,0.3775,0.3775>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,58.80,> rotate <0,-275.98,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,6.10,> rotate <0,-165.19,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,40.90,> rotate <0,-232.95,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,3.55,> rotate <0,-80.71,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,33.73,> rotate <0,-296.61,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,-14.05,> rotate <0,-104.03,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,37.40,> rotate <0,-81.16,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,29.85,> rotate <0,-240.36,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,-64.30,> rotate <0,-359.4,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,26.83,> rotate <0,-186.75,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,49.07,> rotate <0,-52.01,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,-31.93,> rotate <0,-55.56,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,25.93,> rotate <0,-32.36,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,-47.93,> rotate <0,-230.53,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,43.47,> rotate <0,-271.87,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,33.05,> rotate <0,-184.13,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,-22.68,> rotate <0,-283.78,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,-34.05,> rotate <0,-332.11,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10004.99,0,0> rotate <0,0,-14.52,> rotate <0,-113.45,0>  pigment {rgb <0.37625,0.37625,0.37625>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,2.42,> rotate <0,-213.07,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-54.63,> rotate <0,-243.37,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-2.95,> rotate <0,-58.57,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,17.05,> rotate <0,-242.02,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-32.05,> rotate <0,-301.08,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,27.92,> rotate <0,-112.45,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,61.12,> rotate <0,-59.28,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-55.13,> rotate <0,-184.75,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-12.12,> rotate <0,-69.73,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-14.23,> rotate <0,-354.95,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,12.20,> rotate <0,-335.38,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-44.88,> rotate <0,-137.77,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-51.27,> rotate <0,-143.54,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-13.72,> rotate <0,-93.94,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,54.48,> rotate <0,-30.58,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-49.95,> rotate <0,-128.68,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-5.42,> rotate <0,-290.14,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,9.27,> rotate <0,-111.41,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-62.15,> rotate <0,-91.77,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-21.12,> rotate <0,-143.3,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-38.65,> rotate <0,-275.58,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-73.65,> rotate <0,-35.72,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,69.08,> rotate <0,-160.77,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,41.78,> rotate <0,-102.69,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,28.07,> rotate <0,-313.64,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-25.87,> rotate <0,-240.84,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-36.17,> rotate <0,-170.8,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-76.65,> rotate <0,-224.47,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,49.28,> rotate <0,-96.23,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-32.68,> rotate <0,-121.07,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,18.07,> rotate <0,-260.08,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-8.70,> rotate <0,-205.4,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,69.20,> rotate <0,-187.53,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,16.17,> rotate <0,-108.35,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-20.07,> rotate <0,-5.45,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.00,0,0> rotate <0,0,-59.38,> rotate <0,-300.43,0>  pigment {rgb <0.375,0.375,0.375>}}
object {star scale 1.30 translate <10005.01,0,0> rotate <0,0,-30.90,> rotate <0,-326.93,0>  pigment {rgb <0.37375,0.37375,0.37375>}}
object {star scale 1.30 translate <10005.01,0,0> rotate <0,0,-34.70,> rotate <0,-118.07,0>  pigment {rgb <0.37375,0.37375,0.37375>}}
object {star scale 1.30 translate <10005.01,0,0> rotate <0,0,-47.00,> rotate <0,-158.24,0>  pigment {rgb <0.37375,0.37375,0.37375>}}
object {star scale 1.30 translate <10005.01,0,0> rotate <0,0,-20.17,> rotate <0,-238.33,0>  pigment {rgb <0.37375,0.37375,0.37375>}}
object {star scale 1.30 translate <10005.01,0,0> rotate <0,0,2.05,> rotate <0,-281.21,0>  pigment {rgb <0.37375,0.37375,0.37375>}}
object {star scale 1.30 translate <10005.01,0,0> rotate <0,0,-37.25,> rotate <0,-91.88,0>  pigment {rgb <0.37375,0.37375,0.37375>}}
object {star scale 1.30 translate <10005.01,0,0> rotate <0,0,76.97,> rotate <0,-267.36,0>  pigment {rgb <0.37375,0.37375,0.37375>}}
object {star scale 1.30 translate <10005.01,0,0> rotate <0,0,39.02,> rotate <0,-186.46,0>  pigment {rgb <0.37375,0.37375,0.37375>}}
object {star scale 1.30 translate <10005.01,0,0> rotate <0,0,-26.28,> rotate <0,-75.54,0>  pigment {rgb <0.37375,0.37375,0.37375>}}
object {star scale 1.30 translate <10005.01,0,0> rotate <0,0,6.83,> rotate <0,-144.3,0>  pigment {rgb <0.37375,0.37375,0.37375>}}
object {star scale 1.30 translate <10005.01,0,0> rotate <0,0,27.50,> rotate <0,-209.14,0>  pigment {rgb <0.37375,0.37375,0.37375>}}
object {star scale 1.30 translate <10005.01,0,0> rotate <0,0,-70.13,> rotate <0,-318.33,0>  pigment {rgb <0.37375,0.37375,0.37375>}}
object {star scale 1.30 translate <10005.01,0,0> rotate <0,0,-65.58,> rotate <0,-92.81,0>  pigment {rgb <0.37375,0.37375,0.37375>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,19.13,> rotate <0,-297.24,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,-24.50,> rotate <0,-291.32,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,32.68,> rotate <0,-78.85,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,-23.45,> rotate <0,-246.4,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,19.77,> rotate <0,-293.64,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,18.38,> rotate <0,-188.78,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,-23.70,> rotate <0,-272.93,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,67.80,> rotate <0,-356.98,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,-15.48,> rotate <0,-299.49,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,49.47,> rotate <0,-109.63,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,13.05,> rotate <0,-67.21,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,-36.60,> rotate <0,-109.21,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,-29.73,> rotate <0,-0.58,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,-37.80,> rotate <0,-223.21,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,64.40,> rotate <0,-273.47,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,33.50,> rotate <0,-163.93,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,-18.13,> rotate <0,-207.47,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,-4.23,> rotate <0,-339.44,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,71.82,> rotate <0,-229.28,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,-2.78,> rotate <0,-292.67,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.02,0,0> rotate <0,0,-49.58,> rotate <0,-76.24,0>  pigment {rgb <0.3725,0.3725,0.3725>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,28.65,> rotate <0,-34.74,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,46.95,> rotate <0,-285.36,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,-32.15,> rotate <0,-255.47,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,12.27,> rotate <0,-94.11,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,-26.28,> rotate <0,-90.82,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,-9.57,> rotate <0,-89.77,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,22.65,> rotate <0,-31.64,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,-53.52,> rotate <0,-26.53,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,50.78,> rotate <0,-267.27,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,40.42,> rotate <0,-324.24,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,33.95,> rotate <0,-80,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,28.12,> rotate <0,-112.34,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,-52.93,> rotate <0,-284.62,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,46.07,> rotate <0,-2.58,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,73.18,> rotate <0,-329.81,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,-61.52,> rotate <0,-309.4,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,50.93,> rotate <0,-49.05,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,25.05,> rotate <0,-110.87,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,-16.22,> rotate <0,-72.55,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,52.98,> rotate <0,-297.65,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,-76.52,> rotate <0,-181.2,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,10.87,> rotate <0,-259.65,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,44.52,> rotate <0,-100.77,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,40.83,> rotate <0,-232.73,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.30 translate <10005.03,0,0> rotate <0,0,-7.72,> rotate <0,-349.21,0>  pigment {rgb <0.37125,0.37125,0.37125>}}
object {star scale 1.29 translate <10005.04,0,0> rotate <0,0,-54.43,> rotate <0,-290.71,0>  pigment {rgb <0.37,0.37,0.37>}}
object {star scale 1.29 translate <10005.04,0,0> rotate <0,0,43.95,> rotate <0,-17,0>  pigment {rgb <0.37,0.37,0.37>}}
object {star scale 1.29 translate <10005.04,0,0> rotate <0,0,23.20,> rotate <0,-160.85,0>  pigment {rgb <0.37,0.37,0.37>}}
object {star scale 1.29 translate <10005.04,0,0> rotate <0,0,-14.57,> rotate <0,-116.4,0>  pigment {rgb <0.37,0.37,0.37>}}
object {star scale 1.29 translate <10005.04,0,0> rotate <0,0,-15.62,> rotate <0,-283.68,0>  pigment {rgb <0.37,0.37,0.37>}}
object {star scale 1.29 translate <10005.04,0,0> rotate <0,0,50.10,> rotate <0,-49.78,0>  pigment {rgb <0.37,0.37,0.37>}}
object {star scale 1.29 translate <10005.04,0,0> rotate <0,0,59.17,> rotate <0,-61.11,0>  pigment {rgb <0.37,0.37,0.37>}}
object {star scale 1.29 translate <10005.04,0,0> rotate <0,0,-16.20,> rotate <0,-198.01,0>  pigment {rgb <0.37,0.37,0.37>}}
object {star scale 1.29 translate <10005.04,0,0> rotate <0,0,-37.13,> rotate <0,-70.51,0>  pigment {rgb <0.37,0.37,0.37>}}
object {star scale 1.29 translate <10005.04,0,0> rotate <0,0,-74.90,> rotate <0,-201.28,0>  pigment {rgb <0.37,0.37,0.37>}}
object {star scale 1.29 translate <10005.04,0,0> rotate <0,0,24.65,> rotate <0,-252.94,0>  pigment {rgb <0.37,0.37,0.37>}}
object {star scale 1.29 translate <10005.04,0,0> rotate <0,0,-45.25,> rotate <0,-121.67,0>  pigment {rgb <0.37,0.37,0.37>}}
object {star scale 1.29 translate <10005.04,0,0> rotate <0,0,59.42,> rotate <0,-332.88,0>  pigment {rgb <0.37,0.37,0.37>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,-8.10,> rotate <0,-148.13,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,79.23,> rotate <0,-80.64,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,-22.65,> rotate <0,-129.78,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,10.08,> rotate <0,-309.78,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,53.08,> rotate <0,-69.98,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,10.10,> rotate <0,-213.71,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,50.12,> rotate <0,-300.34,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,44.07,> rotate <0,-312.52,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,55.53,> rotate <0,-280.66,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,-51.17,> rotate <0,-202.35,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,-6.55,> rotate <0,-92.97,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,-4.77,> rotate <0,-96.99,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,-80.12,> rotate <0,-215.6,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,-34.12,> rotate <0,-104.6,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,24.12,> rotate <0,-309.63,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,24.13,> rotate <0,-57.3,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,-68.83,> rotate <0,-92.19,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,3.48,> rotate <0,-356.6,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,-16.13,> rotate <0,-295.63,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,-62.48,> rotate <0,-175.88,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,40.43,> rotate <0,-164.87,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,-45.17,> rotate <0,-115.74,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,5.38,> rotate <0,-350.09,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,17.67,> rotate <0,-114.87,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,49.12,> rotate <0,-275.38,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.05,0,0> rotate <0,0,19.12,> rotate <0,-358.12,0>  pigment {rgb <0.36875,0.36875,0.36875>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,55.45,> rotate <0,-52.5,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,39.67,> rotate <0,-281.09,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,-57.08,> rotate <0,-213.74,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,11.13,> rotate <0,-57.07,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,1.77,> rotate <0,-229.83,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,-1.20,> rotate <0,-48.19,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,-24.77,> rotate <0,-80.44,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,30.18,> rotate <0,-327.46,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,-20.65,> rotate <0,-284.33,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,25.92,> rotate <0,-328.27,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,-44.95,> rotate <0,-67.71,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,-14.33,> rotate <0,-145.08,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,-0.63,> rotate <0,-157.58,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,-26.93,> rotate <0,-78.85,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,39.22,> rotate <0,-165.21,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,-65.62,> rotate <0,-124.58,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,68.13,> rotate <0,-262.99,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,-0.88,> rotate <0,-81.12,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,-9.00,> rotate <0,-101.91,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,-74.92,> rotate <0,-12.15,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,10.33,> rotate <0,-355.84,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.06,0,0> rotate <0,0,22.23,> rotate <0,-271.51,0>  pigment {rgb <0.3675,0.3675,0.3675>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,-45.73,> rotate <0,-147.49,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,36.65,> rotate <0,-234.85,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,-42.67,> rotate <0,-172.15,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,-44.57,> rotate <0,-257.68,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,-47.37,> rotate <0,-151.55,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,-56.20,> rotate <0,-24.95,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,9.72,> rotate <0,-142.99,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,-23.75,> rotate <0,-158.5,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,15.80,> rotate <0,-69.79,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,23.62,> rotate <0,-301.73,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,-13.55,> rotate <0,-328.33,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,73.63,> rotate <0,-338.94,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,16.93,> rotate <0,-12.24,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,6.95,> rotate <0,-158.7,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,-59.57,> rotate <0,-159.09,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,34.47,> rotate <0,-81.91,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,81.20,> rotate <0,-75.09,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,40.77,> rotate <0,-257.39,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.07,0,0> rotate <0,0,87.02,> rotate <0,-115.11,0>  pigment {rgb <0.36625,0.36625,0.36625>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,42.33,> rotate <0,-30.98,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,42.75,> rotate <0,-345.65,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,-38.52,> rotate <0,-116.85,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,-44.10,> rotate <0,-119.33,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,52.93,> rotate <0,-249.06,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,41.82,> rotate <0,-341.03,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,12.32,> rotate <0,-350.77,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,-5.42,> rotate <0,-83.85,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,5.82,> rotate <0,-216.05,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,-36.32,> rotate <0,-123.49,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,-53.08,> rotate <0,-126.9,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,-18.78,> rotate <0,-170.84,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,56.95,> rotate <0,-345.02,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,-74.75,> rotate <0,-92.56,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,-41.62,> rotate <0,-334.11,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,-68.60,> rotate <0,-238.87,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,-9.07,> rotate <0,-326.25,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.08,0,0> rotate <0,0,-60.95,> rotate <0,-229.24,0>  pigment {rgb <0.365,0.365,0.365>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,-46.52,> rotate <0,-132.64,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,-43.98,> rotate <0,-312.12,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,56.80,> rotate <0,-339.66,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,14.90,> rotate <0,-184,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,33.08,> rotate <0,-57.38,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,-1.23,> rotate <0,-98.41,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,11.07,> rotate <0,-286.74,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,54.17,> rotate <0,-9.03,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,46.02,> rotate <0,-147.15,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,-51.02,> rotate <0,-111.59,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,2.25,> rotate <0,-324.89,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,-39.75,> rotate <0,-201.53,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,17.82,> rotate <0,-29.34,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,19.98,> rotate <0,-301.29,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,54.58,> rotate <0,-163.39,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,20.73,> rotate <0,-50.69,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,-31.08,> rotate <0,-173.23,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.29 translate <10005.09,0,0> rotate <0,0,-80.43,> rotate <0,-335,0>  pigment {rgb <0.36375,0.36375,0.36375>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-49.02,> rotate <0,-143.43,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-64.95,> rotate <0,-170.84,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-4.65,> rotate <0,-76.69,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,30.57,> rotate <0,-271.76,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,80.68,> rotate <0,-62.51,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,1.08,> rotate <0,-289.63,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,20.45,> rotate <0,-110.49,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,43.27,> rotate <0,-325.05,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,31.93,> rotate <0,-44.32,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-3.08,> rotate <0,-237.81,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,18.83,> rotate <0,-72.84,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-36.75,> rotate <0,-109.66,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-53.38,> rotate <0,-141.58,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-66.92,> rotate <0,-87.48,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-80.95,> rotate <0,-143.47,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-34.73,> rotate <0,-58.41,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-3.03,> rotate <0,-0.45,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-59.75,> rotate <0,-45.9,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-31.52,> rotate <0,-199.22,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-52.73,> rotate <0,-359.73,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-23.30,> rotate <0,-119.77,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-54.72,> rotate <0,-316.31,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-18.13,> rotate <0,-310.01,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-8.22,> rotate <0,-68.55,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,-63.40,> rotate <0,-76.89,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,45.38,> rotate <0,-324.01,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.10,0,0> rotate <0,0,63.22,> rotate <0,-55.54,0>  pigment {rgb <0.3625,0.3625,0.3625>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,-21.63,> rotate <0,-22.4,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,9.47,> rotate <0,-65.97,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,27.97,> rotate <0,-292.69,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,-7.93,> rotate <0,-18.6,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,-59.43,> rotate <0,-172.95,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,62.28,> rotate <0,-331.29,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,-41.02,> rotate <0,-188.94,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,13.12,> rotate <0,-121.27,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,-42.50,> rotate <0,-28.59,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,39.50,> rotate <0,-276.06,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,20.05,> rotate <0,-272.22,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,-44.17,> rotate <0,-261.05,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,-34.75,> rotate <0,-345.88,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,45.52,> rotate <0,-295.21,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,-60.55,> rotate <0,-310.01,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,-82.02,> rotate <0,-358.03,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,-10.57,> rotate <0,-293.78,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,-44.27,> rotate <0,-139.1,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,65.72,> rotate <0,-160.49,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,25.62,> rotate <0,-267.2,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.11,0,0> rotate <0,0,-26.75,> rotate <0,-177.19,0>  pigment {rgb <0.36125,0.36125,0.36125>}}
object {star scale 1.28 translate <10005.12,0,0> rotate <0,0,8.45,> rotate <0,-215.85,0>  pigment {rgb <0.36,0.36,0.36>}}
object {star scale 1.28 translate <10005.12,0,0> rotate <0,0,36.77,> rotate <0,-110.51,0>  pigment {rgb <0.36,0.36,0.36>}}
object {star scale 1.28 translate <10005.12,0,0> rotate <0,0,77.62,> rotate <0,-183.05,0>  pigment {rgb <0.36,0.36,0.36>}}
object {star scale 1.28 translate <10005.12,0,0> rotate <0,0,-67.10,> rotate <0,-251.67,0>  pigment {rgb <0.36,0.36,0.36>}}
object {star scale 1.28 translate <10005.12,0,0> rotate <0,0,28.87,> rotate <0,-275.25,0>  pigment {rgb <0.36,0.36,0.36>}}
object {star scale 1.28 translate <10005.12,0,0> rotate <0,0,-46.98,> rotate <0,-123.4,0>  pigment {rgb <0.36,0.36,0.36>}}
object {star scale 1.28 translate <10005.12,0,0> rotate <0,0,57.53,> rotate <0,-298.32,0>  pigment {rgb <0.36,0.36,0.36>}}
object {star scale 1.28 translate <10005.12,0,0> rotate <0,0,68.88,> rotate <0,-103.43,0>  pigment {rgb <0.36,0.36,0.36>}}
object {star scale 1.28 translate <10005.12,0,0> rotate <0,0,41.05,> rotate <0,-149.42,0>  pigment {rgb <0.36,0.36,0.36>}}
object {star scale 1.28 translate <10005.12,0,0> rotate <0,0,-4.45,> rotate <0,-77.18,0>  pigment {rgb <0.36,0.36,0.36>}}
object {star scale 1.28 translate <10005.12,0,0> rotate <0,0,-51.83,> rotate <0,-266.03,0>  pigment {rgb <0.36,0.36,0.36>}}
object {star scale 1.28 translate <10005.12,0,0> rotate <0,0,56.85,> rotate <0,-287.92,0>  pigment {rgb <0.36,0.36,0.36>}}
object {star scale 1.28 translate <10005.12,0,0> rotate <0,0,17.82,> rotate <0,-240.31,0>  pigment {rgb <0.36,0.36,0.36>}}
object {star scale 1.28 translate <10005.12,0,0> rotate <0,0,-60.20,> rotate <0,-284.65,0>  pigment {rgb <0.36,0.36,0.36>}}
object {star scale 1.28 translate <10005.12,0,0> rotate <0,0,10.95,> rotate <0,-105.91,0>  pigment {rgb <0.36,0.36,0.36>}}
object {star scale 1.28 translate <10005.12,0,0> rotate <0,0,16.57,> rotate <0,-191.66,0>  pigment {rgb <0.36,0.36,0.36>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,-36.25,> rotate <0,-206.74,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,-38.32,> rotate <0,-280.94,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,2.28,> rotate <0,-288.43,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,10.42,> rotate <0,-297.75,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,7.55,> rotate <0,-126.48,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,-7.78,> rotate <0,-3.62,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,-5.38,> rotate <0,-83.82,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,36.32,> rotate <0,-291.54,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,-33.82,> rotate <0,-37.01,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,11.35,> rotate <0,-52.6,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,-42.43,> rotate <0,-180.92,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,-46.60,> rotate <0,-102.48,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,-4.10,> rotate <0,-114.32,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,27.22,> rotate <0,-125.01,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,-58.92,> rotate <0,-81.58,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,-10.97,> rotate <0,-278.76,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,-2.27,> rotate <0,-214.88,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,-21.60,> rotate <0,-335.4,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,-16.82,> rotate <0,-94.43,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,-29.42,> rotate <0,-242.76,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,-27.25,> rotate <0,-213.19,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,44.48,> rotate <0,-9.19,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,-44.72,> rotate <0,-30.43,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.13,0,0> rotate <0,0,39.02,> rotate <0,-233.81,0>  pigment {rgb <0.35875,0.35875,0.35875>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,-42.63,> rotate <0,-166.82,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,-63.03,> rotate <0,-8.18,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,39.60,> rotate <0,-281.1,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,-67.52,> rotate <0,-185.53,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,16.13,> rotate <0,-311.66,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,19.80,> rotate <0,-291.37,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,33.42,> rotate <0,-116.88,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,8.68,> rotate <0,-347.38,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,57.55,> rotate <0,-80.87,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,49.67,> rotate <0,-199.56,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,24.50,> rotate <0,-260.23,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,19.70,> rotate <0,-90.86,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,-66.77,> rotate <0,-310.49,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,1.77,> rotate <0,-117.93,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,-61.82,> rotate <0,-174.53,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,-77.02,> rotate <0,-316.18,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,67.35,> rotate <0,-228.66,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,22.05,> rotate <0,-137.34,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,-48.93,> rotate <0,-107.7,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,-61.28,> rotate <0,-174.25,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,-1.52,> rotate <0,-208.68,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,-40.93,> rotate <0,-115.93,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,45.02,> rotate <0,-331.51,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.14,0,0> rotate <0,0,8.58,> rotate <0,-251.46,0>  pigment {rgb <0.3575,0.3575,0.3575>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,-62.92,> rotate <0,-123.82,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,6.82,> rotate <0,-320.73,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,5.40,> rotate <0,-294.8,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,-59.08,> rotate <0,-134.85,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,40.02,> rotate <0,-268.33,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,-66.38,> rotate <0,-153.38,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,43.05,> rotate <0,-180.53,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,15.82,> rotate <0,-113.4,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,17.65,> rotate <0,-100.6,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,8.73,> rotate <0,-347.93,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,1.83,> rotate <0,-232.16,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,-49.83,> rotate <0,-130.92,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,50.22,> rotate <0,-49.66,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,67.57,> rotate <0,-102.74,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,-24.97,> rotate <0,-209.63,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,-61.93,> rotate <0,-167.14,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,57.08,> rotate <0,-158.79,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,76.57,> rotate <0,-287.29,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,-16.00,> rotate <0,-222.67,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,37.97,> rotate <0,-5.28,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,-52.33,> rotate <0,-286.58,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,11.95,> rotate <0,-291.24,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,43.73,> rotate <0,-132.99,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,-45.57,> rotate <0,-119.47,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,-28.05,> rotate <0,-233.65,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,36.15,> rotate <0,-38.03,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.28 translate <10005.15,0,0> rotate <0,0,66.87,> rotate <0,-137.1,0>  pigment {rgb <0.35625,0.35625,0.35625>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,-31.07,> rotate <0,-101.12,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,-45.07,> rotate <0,-157.99,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,-65.40,> rotate <0,-174.87,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,0.48,> rotate <0,-309.85,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,-3.72,> rotate <0,-44.16,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,-39.70,> rotate <0,-278.09,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,3.62,> rotate <0,-19.45,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,31.43,> rotate <0,-17.78,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,25.88,> rotate <0,-84.93,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,-51.20,> rotate <0,-194.27,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,28.02,> rotate <0,-98.8,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,9.83,> rotate <0,-343.1,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,54.92,> rotate <0,-17.07,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,72.82,> rotate <0,-39.51,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,-52.12,> rotate <0,-87.72,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,13.72,> rotate <0,-313.9,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,72.25,> rotate <0,-145.74,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,72.33,> rotate <0,-325.77,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,-10.52,> rotate <0,-1.13,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,-16.72,> rotate <0,-232.06,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.16,0,0> rotate <0,0,-60.60,> rotate <0,-119.91,0>  pigment {rgb <0.355,0.355,0.355>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,-18.40,> rotate <0,-277.86,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,-47.23,> rotate <0,-134.72,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,76.12,> rotate <0,-31.38,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,-18.92,> rotate <0,-357.84,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,-56.75,> rotate <0,-106.08,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,-17.15,> rotate <0,-179,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,-3.75,> rotate <0,-65.92,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,-59.52,> rotate <0,-172.95,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,20.42,> rotate <0,-76.95,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,26.10,> rotate <0,-186.08,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,64.87,> rotate <0,-319.84,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,-50.23,> rotate <0,-356.82,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,-9.18,> rotate <0,-233.61,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,34.98,> rotate <0,-304.66,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,-63.97,> rotate <0,-171.43,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,61.42,> rotate <0,-138.58,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,23.50,> rotate <0,-303.88,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,24.20,> rotate <0,-105.6,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,-28.98,> rotate <0,-5.38,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,-52.10,> rotate <0,-283.17,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,44.68,> rotate <0,-294.16,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,40.67,> rotate <0,-111.03,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,-10.73,> rotate <0,-196.97,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,-75.37,> rotate <0,-181.96,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,-67.92,> rotate <0,-104.96,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,2.02,> rotate <0,-169.33,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,15.43,> rotate <0,-231.45,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.17,0,0> rotate <0,0,-32.32,> rotate <0,-86.5,0>  pigment {rgb <0.35375,0.35375,0.35375>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,55.83,> rotate <0,-35.59,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,-57.18,> rotate <0,-193.65,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,-13.90,> rotate <0,-117.94,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,21.60,> rotate <0,-270.37,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,-12.38,> rotate <0,-97.85,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,-76.35,> rotate <0,-82.97,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,70.62,> rotate <0,-25.73,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,-23.78,> rotate <0,-7.6,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,26.27,> rotate <0,-290.71,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,46.20,> rotate <0,-160.88,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,-14.05,> rotate <0,-325.39,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,50.00,> rotate <0,-36.1,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,-32.33,> rotate <0,-316.6,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,-17.22,> rotate <0,-117.42,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,16.03,> rotate <0,-77.93,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,-9.17,> rotate <0,-100.48,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.18,0,0> rotate <0,0,37.52,> rotate <0,-116.66,0>  pigment {rgb <0.3525,0.3525,0.3525>}}
object {star scale 1.27 translate <10005.19,0,0> rotate <0,0,-70.38,> rotate <0,-129.77,0>  pigment {rgb <0.35125,0.35125,0.35125>}}
object {star scale 1.27 translate <10005.19,0,0> rotate <0,0,-15.03,> rotate <0,-350.66,0>  pigment {rgb <0.35125,0.35125,0.35125>}}
object {star scale 1.27 translate <10005.19,0,0> rotate <0,0,-36.48,> rotate <0,-125.34,0>  pigment {rgb <0.35125,0.35125,0.35125>}}
object {star scale 1.27 translate <10005.19,0,0> rotate <0,0,-40.27,> rotate <0,-130.08,0>  pigment {rgb <0.35125,0.35125,0.35125>}}
object {star scale 1.27 translate <10005.19,0,0> rotate <0,0,19.15,> rotate <0,-93.71,0>  pigment {rgb <0.35125,0.35125,0.35125>}}
object {star scale 1.27 translate <10005.19,0,0> rotate <0,0,27.60,> rotate <0,-61.65,0>  pigment {rgb <0.35125,0.35125,0.35125>}}
object {star scale 1.27 translate <10005.19,0,0> rotate <0,0,74.97,> rotate <0,-307.88,0>  pigment {rgb <0.35125,0.35125,0.35125>}}
object {star scale 1.27 translate <10005.19,0,0> rotate <0,0,44.38,> rotate <0,-7.06,0>  pigment {rgb <0.35125,0.35125,0.35125>}}
object {star scale 1.27 translate <10005.19,0,0> rotate <0,0,53.62,> rotate <0,-64.18,0>  pigment {rgb <0.35125,0.35125,0.35125>}}
object {star scale 1.27 translate <10005.19,0,0> rotate <0,0,62.60,> rotate <0,-236.66,0>  pigment {rgb <0.35125,0.35125,0.35125>}}
object {star scale 1.27 translate <10005.19,0,0> rotate <0,0,11.43,> rotate <0,-73.7,0>  pigment {rgb <0.35125,0.35125,0.35125>}}
object {star scale 1.27 translate <10005.19,0,0> rotate <0,0,28.68,> rotate <0,-303.56,0>  pigment {rgb <0.35125,0.35125,0.35125>}}
object {star scale 1.27 translate <10005.19,0,0> rotate <0,0,-10.65,> rotate <0,-12.53,0>  pigment {rgb <0.35125,0.35125,0.35125>}}
object {star scale 1.27 translate <10005.19,0,0> rotate <0,0,-23.82,> rotate <0,-35.64,0>  pigment {rgb <0.35125,0.35125,0.35125>}}
object {star scale 1.27 translate <10005.19,0,0> rotate <0,0,0.30,> rotate <0,-96.81,0>  pigment {rgb <0.35125,0.35125,0.35125>}}
object {star scale 1.27 translate <10005.19,0,0> rotate <0,0,-60.12,> rotate <0,-223.9,0>  pigment {rgb <0.35125,0.35125,0.35125>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,29.75,> rotate <0,-7.53,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,0.57,> rotate <0,-89.71,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-41.48,> rotate <0,-16.95,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-53.05,> rotate <0,-129.99,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-64.18,> rotate <0,-168.19,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-3.60,> rotate <0,-8.81,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-24.30,> rotate <0,-259.5,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,38.72,> rotate <0,-297.64,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-42.33,> rotate <0,-106.01,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,7.67,> rotate <0,-191.4,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,59.88,> rotate <0,-88.74,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,15.32,> rotate <0,-134.31,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,12.43,> rotate <0,-41.24,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-44.13,> rotate <0,-122.4,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,37.03,> rotate <0,-299.98,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,39.90,> rotate <0,-99.7,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,35.80,> rotate <0,-196.44,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,12.43,> rotate <0,-149.56,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-20.70,> rotate <0,-338.67,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-54.55,> rotate <0,-40.17,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-64.03,> rotate <0,-226.2,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-32.77,> rotate <0,-132.47,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-41.48,> rotate <0,-229.02,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-61.63,> rotate <0,-247.7,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-41.07,> rotate <0,-226.33,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-10.17,> rotate <0,-203.24,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-51.62,> rotate <0,-312.88,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-45.47,> rotate <0,-289.09,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,59.63,> rotate <0,-108.98,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-53.97,> rotate <0,-346.17,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-16.25,> rotate <0,-226.66,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-5.73,> rotate <0,-105.48,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,33.80,> rotate <0,-245.59,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,57.12,> rotate <0,-146.63,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,68.48,> rotate <0,-317.38,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.20,0,0> rotate <0,0,-69.50,> rotate <0,-322.19,0>  pigment {rgb <0.35,0.35,0.35>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,-2.47,> rotate <0,-69.4,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,32.50,> rotate <0,-286.86,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,-25.43,> rotate <0,-221.5,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,9.43,> rotate <0,-199.2,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,0.20,> rotate <0,-276.8,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,-13.45,> rotate <0,-349.78,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,-52.38,> rotate <0,-221.75,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,58.42,> rotate <0,-96.7,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,-32.50,> rotate <0,-175.43,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,-61.28,> rotate <0,-214.96,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,63.58,> rotate <0,-339.66,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,10.02,> rotate <0,-63.65,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,-50.98,> rotate <0,-12.67,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,39.23,> rotate <0,-352.82,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,27.93,> rotate <0,-133.91,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,-56.85,> rotate <0,-97.37,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,-19.95,> rotate <0,-199,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,48.80,> rotate <0,-101.91,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.27 translate <10005.21,0,0> rotate <0,0,-11.53,> rotate <0,-96.04,0>  pigment {rgb <0.34875,0.34875,0.34875>}}
object {star scale 1.26 translate <10005.22,0,0> rotate <0,0,-2.78,> rotate <0,-44.68,0>  pigment {rgb <0.3475,0.3475,0.3475>}}
object {star scale 1.26 translate <10005.22,0,0> rotate <0,0,-44.72,> rotate <0,-127.36,0>  pigment {rgb <0.3475,0.3475,0.3475>}}
object {star scale 1.26 translate <10005.22,0,0> rotate <0,0,-52.20,> rotate <0,-135.43,0>  pigment {rgb <0.3475,0.3475,0.3475>}}
object {star scale 1.26 translate <10005.22,0,0> rotate <0,0,-35.90,> rotate <0,-149.72,0>  pigment {rgb <0.3475,0.3475,0.3475>}}
object {star scale 1.26 translate <10005.22,0,0> rotate <0,0,11.82,> rotate <0,-295.64,0>  pigment {rgb <0.3475,0.3475,0.3475>}}
object {star scale 1.26 translate <10005.22,0,0> rotate <0,0,17.52,> rotate <0,-197.5,0>  pigment {rgb <0.3475,0.3475,0.3475>}}
object {star scale 1.26 translate <10005.22,0,0> rotate <0,0,-36.07,> rotate <0,-171.37,0>  pigment {rgb <0.3475,0.3475,0.3475>}}
object {star scale 1.26 translate <10005.22,0,0> rotate <0,0,-2.75,> rotate <0,-157.37,0>  pigment {rgb <0.3475,0.3475,0.3475>}}
object {star scale 1.26 translate <10005.22,0,0> rotate <0,0,-17.47,> rotate <0,-54.07,0>  pigment {rgb <0.3475,0.3475,0.3475>}}
object {star scale 1.26 translate <10005.22,0,0> rotate <0,0,-6.70,> rotate <0,-91.06,0>  pigment {rgb <0.3475,0.3475,0.3475>}}
object {star scale 1.26 translate <10005.22,0,0> rotate <0,0,58.97,> rotate <0,-76.54,0>  pigment {rgb <0.3475,0.3475,0.3475>}}
object {star scale 1.26 translate <10005.22,0,0> rotate <0,0,40.22,> rotate <0,-340.37,0>  pigment {rgb <0.3475,0.3475,0.3475>}}
object {star scale 1.26 translate <10005.22,0,0> rotate <0,0,41.80,> rotate <0,-80.45,0>  pigment {rgb <0.3475,0.3475,0.3475>}}
object {star scale 1.26 translate <10005.22,0,0> rotate <0,0,6.07,> rotate <0,-287.25,0>  pigment {rgb <0.3475,0.3475,0.3475>}}
object {star scale 1.26 translate <10005.22,0,0> rotate <0,0,-50.65,> rotate <0,-261.5,0>  pigment {rgb <0.3475,0.3475,0.3475>}}
object {star scale 1.26 translate <10005.22,0,0> rotate <0,0,-11.83,> rotate <0,-243.46,0>  pigment {rgb <0.3475,0.3475,0.3475>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,21.77,> rotate <0,-102.89,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,-51.73,> rotate <0,-126.38,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,6.95,> rotate <0,-112.01,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,24.93,> rotate <0,-300.5,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,40.37,> rotate <0,-304.23,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,38.73,> rotate <0,-316.73,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,5.25,> rotate <0,-251.94,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,-23.87,> rotate <0,-56.92,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,11.55,> rotate <0,-97.95,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,-46.62,> rotate <0,-116.88,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,10.67,> rotate <0,-136.94,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,-22.00,> rotate <0,-11.19,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,21.20,> rotate <0,-33.2,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,-39.28,> rotate <0,-119.87,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,-62.50,> rotate <0,-49.55,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,-40.42,> rotate <0,-281.93,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,-63.38,> rotate <0,-65.47,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,-46.73,> rotate <0,-232.35,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,46.53,> rotate <0,-322.36,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,-6.35,> rotate <0,-139.18,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,67.20,> rotate <0,-345.89,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,64.58,> rotate <0,-51.17,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.23,0,0> rotate <0,0,-12.70,> rotate <0,-201.68,0>  pigment {rgb <0.34625,0.34625,0.34625>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,32.55,> rotate <0,-282.47,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,7.58,> rotate <0,-18.43,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-47.47,> rotate <0,-260.82,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-43.23,> rotate <0,-138.6,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-64.27,> rotate <0,-161.63,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,17.28,> rotate <0,-327.53,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-20.13,> rotate <0,-163.38,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,43.58,> rotate <0,-101.68,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-23.27,> rotate <0,-325.5,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-78.67,> rotate <0,-245.11,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,86.10,> rotate <0,-333.3,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-39.22,> rotate <0,-109.64,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,65.43,> rotate <0,-193.87,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-54.50,> rotate <0,-264.53,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,1.92,> rotate <0,-113.03,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-13.92,> rotate <0,-80.88,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-44.12,> rotate <0,-276.08,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-45.42,> rotate <0,-130.49,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-1.17,> rotate <0,-56.13,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-8.07,> rotate <0,-154.41,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-47.63,> rotate <0,-173.98,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-35.13,> rotate <0,-2.93,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,56.85,> rotate <0,-332.95,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-37.70,> rotate <0,-98.09,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,54.55,> rotate <0,-226.57,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,20.48,> rotate <0,-247.64,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,40.35,> rotate <0,-234.45,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-20.65,> rotate <0,-318.9,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,74.08,> rotate <0,-281.44,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,20.27,> rotate <0,-25.63,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-32.33,> rotate <0,-99.45,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-6.15,> rotate <0,-253.65,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,69.43,> rotate <0,-213.02,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.24,0,0> rotate <0,0,-7.60,> rotate <0,-246.93,0>  pigment {rgb <0.345,0.345,0.345>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,21.33,> rotate <0,-44.8,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,25.58,> rotate <0,-57.58,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,-18.20,> rotate <0,-306.83,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,-22.22,> rotate <0,-185.14,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,-37.93,> rotate <0,-101.84,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,21.45,> rotate <0,-111.93,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,61.87,> rotate <0,-263.75,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,-18.20,> rotate <0,-213.85,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,-62.80,> rotate <0,-42.25,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,10.53,> rotate <0,-162.31,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,33.57,> rotate <0,-255.4,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,40.23,> rotate <0,-144.59,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,-58.23,> rotate <0,-128.81,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,-56.77,> rotate <0,-161.74,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,-51.05,> rotate <0,-139.53,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,-43.37,> rotate <0,-197.85,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,69.33,> rotate <0,-174.01,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,69.10,> rotate <0,-245.45,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,2.27,> rotate <0,-37.88,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,55.63,> rotate <0,-176.73,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,17.83,> rotate <0,-275.7,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,-8.98,> rotate <0,-68.55,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,49.47,> rotate <0,-212.07,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.25,0,0> rotate <0,0,43.85,> rotate <0,-211.98,0>  pigment {rgb <0.34375,0.34375,0.34375>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,56.77,> rotate <0,-72,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,-26.25,> rotate <0,-129.47,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,6.45,> rotate <0,-87,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,-61.12,> rotate <0,-171.65,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,-4.85,> rotate <0,-83.92,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,13.62,> rotate <0,-284.77,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,-6.10,> rotate <0,-46.64,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,-57.25,> rotate <0,-163.13,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,29.17,> rotate <0,-228.62,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,-7.62,> rotate <0,-46.07,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,36.10,> rotate <0,-286.83,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,15.08,> rotate <0,-65.15,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,19.17,> rotate <0,-21.56,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,-39.43,> rotate <0,-318.26,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,82.55,> rotate <0,-157.77,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,-19.62,> rotate <0,-341.89,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,2.20,> rotate <0,-237.58,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,47.00,> rotate <0,-22.53,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,-64.18,> rotate <0,-157.22,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,-55.57,> rotate <0,-138.58,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,10.02,> rotate <0,-234.12,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,21.35,> rotate <0,-175.2,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,-32.33,> rotate <0,-25.54,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,-15.45,> rotate <0,-355.62,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.26,0,0> rotate <0,0,-17.95,> rotate <0,-3.04,0>  pigment {rgb <0.3425,0.3425,0.3425>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,64.62,> rotate <0,-29.91,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,13.90,> rotate <0,-86.93,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,-18.68,> rotate <0,-356.5,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,50.68,> rotate <0,-59.15,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,-65.82,> rotate <0,-152.18,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,-30.27,> rotate <0,-269.77,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,19.23,> rotate <0,-216.61,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,50.43,> rotate <0,-116.02,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,-30.73,> rotate <0,-33.23,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,49.83,> rotate <0,-148.93,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,-59.73,> rotate <0,-71.09,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,-51.23,> rotate <0,-153.35,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,20.83,> rotate <0,-270.6,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,-1.55,> rotate <0,-60.38,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,21.15,> rotate <0,-50.31,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,-18.28,> rotate <0,-356.05,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,-26.58,> rotate <0,-109.71,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,-79.12,> rotate <0,-37.92,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,-82.23,> rotate <0,-2.51,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,26.23,> rotate <0,-284.94,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,-87.60,> rotate <0,-283.7,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,24.43,> rotate <0,-274.8,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,62.78,> rotate <0,-331.25,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,56.73,> rotate <0,-138.95,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,64.82,> rotate <0,-300.37,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.26 translate <10005.27,0,0> rotate <0,0,0.38,> rotate <0,-35.48,0>  pigment {rgb <0.34125,0.34125,0.34125>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,-74.88,> rotate <0,-139.35,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,39.33,> rotate <0,-172.27,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,-73.40,> rotate <0,-125.52,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,83.40,> rotate <0,-192.31,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,52.35,> rotate <0,-45.22,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,-45.83,> rotate <0,-261.72,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,7.72,> rotate <0,-63.39,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,-72.25,> rotate <0,-336.15,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,-82.72,> rotate <0,-327.73,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,73.05,> rotate <0,-24.63,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,-19.12,> rotate <0,-304.85,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,-24.03,> rotate <0,-126.27,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,-1.90,> rotate <0,-131.81,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,11.42,> rotate <0,-299.06,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,77.28,> rotate <0,-31.28,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,11.60,> rotate <0,-289.45,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,29.67,> rotate <0,-37.04,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,31.70,> rotate <0,-253.24,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,-19.65,> rotate <0,-180.21,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,-22.95,> rotate <0,-103.95,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,-14.42,> rotate <0,-101.71,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,33.37,> rotate <0,-33.98,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,-52.63,> rotate <0,-88.71,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,-28.68,> rotate <0,-84.43,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,65.15,> rotate <0,-65.17,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,-26.08,> rotate <0,-221.93,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,-41.18,> rotate <0,-323.03,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,31.45,> rotate <0,-66.53,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,56.02,> rotate <0,-249.5,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,-15.17,> rotate <0,-318.94,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.28,0,0> rotate <0,0,63.12,> rotate <0,-330.97,0>  pigment {rgb <0.34,0.34,0.34>}}
object {star scale 1.25 translate <10005.29,0,0> rotate <0,0,25.92,> rotate <0,-187.23,0>  pigment {rgb <0.33875,0.33875,0.33875>}}
object {star scale 1.25 translate <10005.29,0,0> rotate <0,0,-21.20,> rotate <0,-317.14,0>  pigment {rgb <0.33875,0.33875,0.33875>}}
object {star scale 1.25 translate <10005.29,0,0> rotate <0,0,-26.60,> rotate <0,-258.84,0>  pigment {rgb <0.33875,0.33875,0.33875>}}
object {star scale 1.25 translate <10005.29,0,0> rotate <0,0,48.83,> rotate <0,-321.72,0>  pigment {rgb <0.33875,0.33875,0.33875>}}
object {star scale 1.25 translate <10005.29,0,0> rotate <0,0,-0.90,> rotate <0,-331.2,0>  pigment {rgb <0.33875,0.33875,0.33875>}}
object {star scale 1.25 translate <10005.29,0,0> rotate <0,0,68.78,> rotate <0,-17.66,0>  pigment {rgb <0.33875,0.33875,0.33875>}}
object {star scale 1.25 translate <10005.29,0,0> rotate <0,0,10.93,> rotate <0,-358.15,0>  pigment {rgb <0.33875,0.33875,0.33875>}}
object {star scale 1.25 translate <10005.29,0,0> rotate <0,0,49.20,> rotate <0,-50.8,0>  pigment {rgb <0.33875,0.33875,0.33875>}}
object {star scale 1.25 translate <10005.29,0,0> rotate <0,0,-12.32,> rotate <0,-80,0>  pigment {rgb <0.33875,0.33875,0.33875>}}
object {star scale 1.25 translate <10005.29,0,0> rotate <0,0,-45.68,> rotate <0,-176.43,0>  pigment {rgb <0.33875,0.33875,0.33875>}}
object {star scale 1.25 translate <10005.29,0,0> rotate <0,0,38.45,> rotate <0,-99.14,0>  pigment {rgb <0.33875,0.33875,0.33875>}}
object {star scale 1.25 translate <10005.29,0,0> rotate <0,0,-12.82,> rotate <0,-152.52,0>  pigment {rgb <0.33875,0.33875,0.33875>}}
object {star scale 1.25 translate <10005.29,0,0> rotate <0,0,51.07,> rotate <0,-33.4,0>  pigment {rgb <0.33875,0.33875,0.33875>}}
object {star scale 1.25 translate <10005.29,0,0> rotate <0,0,22.95,> rotate <0,-326.52,0>  pigment {rgb <0.33875,0.33875,0.33875>}}
object {star scale 1.25 translate <10005.29,0,0> rotate <0,0,2.22,> rotate <0,-119.59,0>  pigment {rgb <0.33875,0.33875,0.33875>}}
object {star scale 1.25 translate <10005.29,0,0> rotate <0,0,25.78,> rotate <0,-116.03,0>  pigment {rgb <0.33875,0.33875,0.33875>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,27.07,> rotate <0,-40.17,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,23.03,> rotate <0,-289.43,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-14.95,> rotate <0,-309.82,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-25.05,> rotate <0,-26.41,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-64.48,> rotate <0,-201.31,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-64.07,> rotate <0,-246.99,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-57.80,> rotate <0,-336.23,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-46.60,> rotate <0,-86.61,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-20.32,> rotate <0,-282.42,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-45.72,> rotate <0,-183.51,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-59.92,> rotate <0,-214.14,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,34.32,> rotate <0,-79.08,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-40.50,> rotate <0,-108.07,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-3.43,> rotate <0,-132.34,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,8.25,> rotate <0,-176.98,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,10.77,> rotate <0,-116.57,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,43.35,> rotate <0,-70.73,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-14.78,> rotate <0,-101.5,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,11.67,> rotate <0,-111.24,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-52.55,> rotate <0,-39.35,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,21.97,> rotate <0,-39.7,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,37.12,> rotate <0,-321.84,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,16.13,> rotate <0,-93.85,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,50.47,> rotate <0,-354.78,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,21.58,> rotate <0,-121.94,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-25.72,> rotate <0,-178.68,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,39.47,> rotate <0,-10.28,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-50.48,> rotate <0,-70.7,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-44.52,> rotate <0,-308.48,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-71.32,> rotate <0,-75.68,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-32.08,> rotate <0,-355.16,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-66.95,> rotate <0,-300.47,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-13.57,> rotate <0,-185.23,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-24.65,> rotate <0,-223.59,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,66.75,> rotate <0,-175.62,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-11.27,> rotate <0,-14.01,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,28.73,> rotate <0,-20.28,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-53.82,> rotate <0,-244.18,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,-43.18,> rotate <0,-279.9,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.30,0,0> rotate <0,0,34.27,> rotate <0,-39.26,0>  pigment {rgb <0.3375,0.3375,0.3375>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,-47.52,> rotate <0,-133.46,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,-57.98,> rotate <0,-145.18,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,-7.68,> rotate <0,-45.68,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,-1.07,> rotate <0,-262.6,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,-16.28,> rotate <0,-295.18,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,-36.10,> rotate <0,-302.8,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,-39.88,> rotate <0,-297.96,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,44.40,> rotate <0,-219.71,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,-19.97,> rotate <0,-94.56,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,5.45,> rotate <0,-60.94,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,1.00,> rotate <0,-69.3,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,-34.67,> rotate <0,-85.56,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,-35.40,> rotate <0,-185.9,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,46.05,> rotate <0,-53.11,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,5.33,> rotate <0,-305.79,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,-37.42,> rotate <0,-235.66,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,-15.83,> rotate <0,-275.03,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,48.98,> rotate <0,-184.95,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,-39.40,> rotate <0,-139.24,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.31,0,0> rotate <0,0,-19.17,> rotate <0,-91.93,0>  pigment {rgb <0.33625,0.33625,0.33625>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,-41.82,> rotate <0,-247.93,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,-47.93,> rotate <0,-127.27,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,-50.23,> rotate <0,-97.46,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,46.38,> rotate <0,-346.91,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,73.95,> rotate <0,-78.09,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,58.40,> rotate <0,-187.49,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,79.48,> rotate <0,-121.2,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,43.22,> rotate <0,-138.45,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,47.38,> rotate <0,-34.82,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,-38.00,> rotate <0,-155.88,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,29.00,> rotate <0,-61.75,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,29.47,> rotate <0,-293.71,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,-24.92,> rotate <0,-117.26,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,68.47,> rotate <0,-123.2,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,-24.80,> rotate <0,-216.2,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,24.67,> rotate <0,-304.2,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,-68.43,> rotate <0,-287.47,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,4.95,> rotate <0,-228.8,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,7.78,> rotate <0,-73.7,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,66.60,> rotate <0,-194.98,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,65.72,> rotate <0,-93.21,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,13.30,> rotate <0,-168.97,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.32,0,0> rotate <0,0,2.52,> rotate <0,-73.35,0>  pigment {rgb <0.335,0.335,0.335>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,31.82,> rotate <0,-350.48,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,-44.27,> rotate <0,-64.82,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,30.97,> rotate <0,-113.79,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,-31.82,> rotate <0,-111.18,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,10.58,> rotate <0,-258.12,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,12.85,> rotate <0,-235.45,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,-48.68,> rotate <0,-182.06,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,24.40,> rotate <0,-147.98,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,-1.17,> rotate <0,-83.38,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,-16.20,> rotate <0,-111.17,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,-45.77,> rotate <0,-278.01,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,47.88,> rotate <0,-58.99,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,16.78,> rotate <0,-299.44,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,12.55,> rotate <0,-93.94,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,-34.18,> rotate <0,-116.4,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,80.82,> rotate <0,-66.76,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,-64.23,> rotate <0,-83.25,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,34.20,> rotate <0,-175.26,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,-17.73,> rotate <0,-200.75,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,34.62,> rotate <0,-333.2,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,13.67,> rotate <0,-199.32,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,20.45,> rotate <0,-128.18,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,-5.73,> rotate <0,-26.5,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.25 translate <10005.33,0,0> rotate <0,0,22.32,> rotate <0,-314.57,0>  pigment {rgb <0.33375,0.33375,0.33375>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,8.50,> rotate <0,-76.97,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-34.70,> rotate <0,-299.96,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-33.85,> rotate <0,-223.93,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-32.98,> rotate <0,-277.77,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-1.60,> rotate <0,-83.18,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,6.05,> rotate <0,-56.42,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,17.47,> rotate <0,-42.33,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,21.47,> rotate <0,-16.42,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-8.18,> rotate <0,-270.77,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,32.90,> rotate <0,-284.26,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-40.65,> rotate <0,-143.08,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,0.72,> rotate <0,-230.25,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-68.32,> rotate <0,-181.16,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-56.37,> rotate <0,-297,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-7.52,> rotate <0,-87.84,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-56.02,> rotate <0,-274.28,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-7.93,> rotate <0,-288.17,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-30.70,> rotate <0,-108.84,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-80.12,> rotate <0,-342.6,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-64.52,> rotate <0,-161.57,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-29.00,> rotate <0,-154.53,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-60.32,> rotate <0,-36.23,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-66.78,> rotate <0,-132.65,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,2.60,> rotate <0,-79.8,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-1.05,> rotate <0,-38.04,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-30.00,> rotate <0,-30.31,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,3.12,> rotate <0,-108.58,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,57.17,> rotate <0,-101.7,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,-39.82,> rotate <0,-313.42,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,5.67,> rotate <0,-325.56,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,8.13,> rotate <0,-174.61,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.34,0,0> rotate <0,0,22.50,> rotate <0,-353.37,0>  pigment {rgb <0.3325,0.3325,0.3325>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,-62.88,> rotate <0,-221.33,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,52.35,> rotate <0,-278.49,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,3.55,> rotate <0,-205.77,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,38.47,> rotate <0,-91.65,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,-58.90,> rotate <0,-299.28,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,29.32,> rotate <0,-154.06,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,3.67,> rotate <0,-179.99,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,75.43,> rotate <0,-281.59,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,42.42,> rotate <0,-293.68,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,3.77,> rotate <0,-83.57,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,-49.10,> rotate <0,-168.14,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,-63.07,> rotate <0,-190.71,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,-7.82,> rotate <0,-335.05,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,30.68,> rotate <0,-161.47,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,21.15,> rotate <0,-64.86,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,19.88,> rotate <0,-118.92,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,-38.63,> rotate <0,-315.74,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,-24.28,> rotate <0,-270.71,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,2.82,> rotate <0,-61.04,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,-46.40,> rotate <0,-15.7,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,-42.70,> rotate <0,-284.07,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,-7.77,> rotate <0,-122.89,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,58.93,> rotate <0,-92.5,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,-9.38,> rotate <0,-94.71,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,-61.55,> rotate <0,-152.34,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,-28.15,> rotate <0,-260.84,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,18.10,> rotate <0,-127.9,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,43.15,> rotate <0,-238.66,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.35,0,0> rotate <0,0,55.45,> rotate <0,-295.52,0>  pigment {rgb <0.33125,0.33125,0.33125>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,-49.52,> rotate <0,-217.59,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,53.48,> rotate <0,-70,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,-12.85,> rotate <0,-284.85,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,5.78,> rotate <0,-335.12,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,-71.52,> rotate <0,-124.95,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,-53.63,> rotate <0,-334.56,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,-6.07,> rotate <0,-141.95,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,21.43,> rotate <0,-9.98,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,18.43,> rotate <0,-253.84,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,24.03,> rotate <0,-83.86,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,-47.37,> rotate <0,-243.82,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,19.32,> rotate <0,-324.44,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,-80.78,> rotate <0,-141.04,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,53.45,> rotate <0,-95.45,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,49.68,> rotate <0,-112.49,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,-53.23,> rotate <0,-255.78,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,31.92,> rotate <0,-150.25,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,-61.78,> rotate <0,-16.83,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,35.98,> rotate <0,-301.59,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,48.60,> rotate <0,-264.16,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,14.93,> rotate <0,-138.81,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,8.18,> rotate <0,-5.15,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,-3.47,> rotate <0,-242.46,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,15.48,> rotate <0,-11.64,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.36,0,0> rotate <0,0,-41.23,> rotate <0,-252.89,0>  pigment {rgb <0.33,0.33,0.33>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,-33.97,> rotate <0,-239.23,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,-20.55,> rotate <0,-121.83,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,46.17,> rotate <0,-315.29,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,-4.20,> rotate <0,-91.66,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,11.70,> rotate <0,-71.51,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,33.75,> rotate <0,-79.75,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,45.45,> rotate <0,-333.45,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,-34.02,> rotate <0,-332.48,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,-16.38,> rotate <0,-62.32,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,-24.53,> rotate <0,-238.47,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,23.00,> rotate <0,-66.83,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,-58.78,> rotate <0,-205.5,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,32.93,> rotate <0,-230.45,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,-13.52,> rotate <0,-265.95,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,55.67,> rotate <0,-285.18,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,45.83,> rotate <0,-130.26,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.37,0,0> rotate <0,0,35.10,> rotate <0,-142.88,0>  pigment {rgb <0.32875,0.32875,0.32875>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-14.15,> rotate <0,-222.33,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-39.70,> rotate <0,-231.19,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-40.58,> rotate <0,-173.4,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-11.28,> rotate <0,-106.67,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,50.35,> rotate <0,-310.55,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-15.15,> rotate <0,-102.24,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-49.17,> rotate <0,-75.7,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-68.68,> rotate <0,-139.32,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,59.45,> rotate <0,-103.27,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,15.50,> rotate <0,-301.36,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,21.77,> rotate <0,-64.9,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,53.40,> rotate <0,-286.23,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,75.17,> rotate <0,-184.71,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-27.93,> rotate <0,-42.48,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-20.65,> rotate <0,-65.16,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-52.80,> rotate <0,-224.07,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,13.03,> rotate <0,-308.49,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,25.53,> rotate <0,-260.04,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,7.87,> rotate <0,-69.78,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-20.72,> rotate <0,-273.8,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-46.23,> rotate <0,-247.43,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-32.20,> rotate <0,-110.88,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-33.02,> rotate <0,-278.49,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,50.53,> rotate <0,-285.06,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,12.20,> rotate <0,-70.01,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-57.67,> rotate <0,-185.7,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,9.83,> rotate <0,-347.5,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,9.13,> rotate <0,-279.12,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,17.72,> rotate <0,-86.86,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-65.47,> rotate <0,-10.62,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-39.20,> rotate <0,-246,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-69.68,> rotate <0,-96.37,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,1.40,> rotate <0,-171.01,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-19.30,> rotate <0,-234.73,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-1.98,> rotate <0,-277.42,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-27.83,> rotate <0,-110.87,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,0.67,> rotate <0,-247.14,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-50.35,> rotate <0,-0.33,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.38,0,0> rotate <0,0,-39.57,> rotate <0,-158.8,0>  pigment {rgb <0.3275,0.3275,0.3275>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,35.93,> rotate <0,-257.01,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,25.63,> rotate <0,-65.65,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,14.38,> rotate <0,-258.67,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,17.52,> rotate <0,-300.01,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,-36.67,> rotate <0,-275.73,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,-15.60,> rotate <0,-109.06,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,20.08,> rotate <0,-261.7,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,-70.95,> rotate <0,-102.86,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,-0.27,> rotate <0,-60.65,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,64.25,> rotate <0,-12.68,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,2.13,> rotate <0,-347.17,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,21.07,> rotate <0,-189.78,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,60.77,> rotate <0,-333,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,-51.27,> rotate <0,-101.72,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,-15.40,> rotate <0,-24,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,61.98,> rotate <0,-301.38,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,33.70,> rotate <0,-245.62,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,68.02,> rotate <0,-301.23,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,-26.33,> rotate <0,-242.03,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.24 translate <10005.39,0,0> rotate <0,0,41.15,> rotate <0,-325.78,0>  pigment {rgb <0.32625,0.32625,0.32625>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,18.40,> rotate <0,-354.49,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,50.83,> rotate <0,-332.79,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,9.95,> rotate <0,-94.28,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,53.92,> rotate <0,-68,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,17.97,> rotate <0,-81.79,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-70.12,> rotate <0,-260.52,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-28.00,> rotate <0,-2.34,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,38.33,> rotate <0,-43.43,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,17.37,> rotate <0,-284.56,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,9.07,> rotate <0,-142.11,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,32.47,> rotate <0,-260.16,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,30.50,> rotate <0,-84.66,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-46.13,> rotate <0,-219.33,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-21.07,> rotate <0,-333.58,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,20.30,> rotate <0,-8.15,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,70.88,> rotate <0,-57.31,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-72.50,> rotate <0,-297.36,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,42.82,> rotate <0,-294.86,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,21.70,> rotate <0,-209.66,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,59.95,> rotate <0,-202.11,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,52.93,> rotate <0,-204.88,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,50.05,> rotate <0,-64.81,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-15.67,> rotate <0,-236.02,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,11.15,> rotate <0,-71.11,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-47.95,> rotate <0,-122.43,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,66.80,> rotate <0,-321.94,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,61.83,> rotate <0,-6.2,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-1.80,> rotate <0,-236.52,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,35.23,> rotate <0,-25.51,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-16.38,> rotate <0,-109.87,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,26.48,> rotate <0,-62.71,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,28.25,> rotate <0,-312.87,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,13.00,> rotate <0,-214.82,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,16.63,> rotate <0,-299.01,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-18.85,> rotate <0,-275.35,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-50.80,> rotate <0,-40.64,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-48.12,> rotate <0,-276.73,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-2.50,> rotate <0,-19.15,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-41.50,> rotate <0,-21.17,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,20.95,> rotate <0,-253.73,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-53.67,> rotate <0,-213.32,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-68.77,> rotate <0,-312.32,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,40.73,> rotate <0,-28.32,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,-27.63,> rotate <0,-318.33,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.40,0,0> rotate <0,0,20.15,> rotate <0,-88.95,0>  pigment {rgb <0.325,0.325,0.325>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-19.57,> rotate <0,-127.88,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-28.45,> rotate <0,-330.21,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-29.70,> rotate <0,-246.17,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,15.92,> rotate <0,-107.09,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-4.03,> rotate <0,-286.24,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-0.93,> rotate <0,-49.59,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-35.80,> rotate <0,-166.23,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-3.70,> rotate <0,-285.73,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-0.08,> rotate <0,-168.44,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,27.60,> rotate <0,-321.92,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-0.88,> rotate <0,-35.55,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-23.98,> rotate <0,-238.48,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,40.37,> rotate <0,-299.31,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,74.85,> rotate <0,-11.94,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-37.33,> rotate <0,-284.17,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-85.67,> rotate <0,-134.17,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,13.72,> rotate <0,-67.66,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,33.30,> rotate <0,-240.26,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-4.83,> rotate <0,-15.76,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-57.68,> rotate <0,-176.83,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-20.87,> rotate <0,-321.04,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,73.92,> rotate <0,-120.05,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-6.90,> rotate <0,-217.17,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-36.23,> rotate <0,-98.45,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,-20.83,> rotate <0,-29.94,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.41,0,0> rotate <0,0,14.03,> rotate <0,-145.93,0>  pigment {rgb <0.32375,0.32375,0.32375>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,55.22,> rotate <0,-11.32,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,62.22,> rotate <0,-357.21,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,-62.77,> rotate <0,-224.18,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,17.05,> rotate <0,-83.06,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,-37.57,> rotate <0,-246.13,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,33.47,> rotate <0,-279.15,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,4.30,> rotate <0,-314.77,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,35.82,> rotate <0,-143.92,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,0.97,> rotate <0,-344.86,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,-56.85,> rotate <0,-193.98,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,24.45,> rotate <0,-135.69,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,-8.55,> rotate <0,-243.03,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,41.08,> rotate <0,-80.06,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,0.33,> rotate <0,-262.21,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,-38.80,> rotate <0,-112.28,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,-38.53,> rotate <0,-115.32,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,-43.50,> rotate <0,-119.24,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,-59.22,> rotate <0,-160.67,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,-31.93,> rotate <0,-110.75,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,-56.32,> rotate <0,-179.56,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,26.47,> rotate <0,-309.27,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,29.65,> rotate <0,-137,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,-36.37,> rotate <0,-118.27,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,72.57,> rotate <0,-183.93,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,29.32,> rotate <0,-267.6,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,14.15,> rotate <0,-98.4,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,-23.95,> rotate <0,-291.38,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,-39.92,> rotate <0,-7.11,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,-39.15,> rotate <0,-342.76,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.42,0,0> rotate <0,0,49.22,> rotate <0,-307.83,0>  pigment {rgb <0.3225,0.3225,0.3225>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,26.32,> rotate <0,-160.76,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,-7.70,> rotate <0,-346.29,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,-55.35,> rotate <0,-227.82,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,-60.50,> rotate <0,-229.7,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,9.83,> rotate <0,-77.33,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,49.83,> rotate <0,-86.48,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,-28.08,> rotate <0,-167.18,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,-17.30,> rotate <0,-161.72,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,0.13,> rotate <0,-30.8,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,24.45,> rotate <0,-305.51,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,15.08,> rotate <0,-42.87,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,-17.98,> rotate <0,-319.49,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,-27.77,> rotate <0,-333.58,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,-52.38,> rotate <0,-234.7,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,44.30,> rotate <0,-41.02,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,45.60,> rotate <0,-142.17,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,32.52,> rotate <0,-236,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,33.25,> rotate <0,-188.41,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,-8.15,> rotate <0,-97.96,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.43,0,0> rotate <0,0,-47.75,> rotate <0,-174.39,0>  pigment {rgb <0.32125,0.32125,0.32125>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-53.67,> rotate <0,-144.3,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-42.68,> rotate <0,-244.83,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,39.72,> rotate <0,-244.98,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,3.30,> rotate <0,-82.81,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,6.68,> rotate <0,-279.16,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-27.13,> rotate <0,-189.43,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-62.20,> rotate <0,-63.7,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,7.25,> rotate <0,-274.79,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-1.13,> rotate <0,-103.6,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,67.80,> rotate <0,-241.58,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-79.43,> rotate <0,-104.14,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,11.38,> rotate <0,-309.45,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-23.87,> rotate <0,-44.35,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-0.30,> rotate <0,-107.85,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-32.65,> rotate <0,-225.75,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-24.83,> rotate <0,-239.65,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,83.82,> rotate <0,-67.05,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-51.52,> rotate <0,-142.52,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-36.77,> rotate <0,-231.83,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,23.42,> rotate <0,-57.09,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-39.03,> rotate <0,-187.09,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-59.42,> rotate <0,-187.92,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,26.05,> rotate <0,-268.85,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-50.62,> rotate <0,-79.84,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-6.92,> rotate <0,-62.6,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-69.53,> rotate <0,-13.75,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-13.53,> rotate <0,-79.42,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-32.63,> rotate <0,-84.96,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,48.63,> rotate <0,-349.88,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-76.78,> rotate <0,-146.59,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,34.33,> rotate <0,-231.57,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,20.30,> rotate <0,-238.64,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-44.40,> rotate <0,-234.05,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,60.70,> rotate <0,-326.85,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.44,0,0> rotate <0,0,-17.87,> rotate <0,-206.85,0>  pigment {rgb <0.32,0.32,0.32>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,77.73,> rotate <0,-50.08,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,15.95,> rotate <0,-265.49,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,-85.13,> rotate <0,-193.75,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,-26.98,> rotate <0,-292.47,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,24.57,> rotate <0,-187.75,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,-37.27,> rotate <0,-329.1,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,-23.60,> rotate <0,-182.77,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,32.92,> rotate <0,-134.23,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,-88.95,> rotate <0,-317.06,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,-9.32,> rotate <0,-211.68,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,43.30,> rotate <0,-25.17,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,24.28,> rotate <0,-56.2,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,48.53,> rotate <0,-136.35,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,-4.65,> rotate <0,-294.45,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,37.95,> rotate <0,-238.95,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,8.20,> rotate <0,-60.99,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,-8.78,> rotate <0,-137.4,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,-14.35,> rotate <0,-69.83,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,55.90,> rotate <0,-342.45,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,60.00,> rotate <0,-93.92,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,18.88,> rotate <0,-161.6,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,59.30,> rotate <0,-222.86,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,-50.32,> rotate <0,-206.91,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.23 translate <10005.45,0,0> rotate <0,0,34.68,> rotate <0,-38.95,0>  pigment {rgb <0.31875,0.31875,0.31875>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-23.12,> rotate <0,-91.63,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-24.63,> rotate <0,-104.39,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-47.08,> rotate <0,-82.54,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,41.60,> rotate <0,-283.72,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-80.47,> rotate <0,-161.32,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-39.13,> rotate <0,-337.17,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-62.85,> rotate <0,-129.33,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,23.62,> rotate <0,-13.74,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-24.00,> rotate <0,-13.17,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,79.42,> rotate <0,-46.53,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,65.15,> rotate <0,-128.65,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-75.03,> rotate <0,-275.9,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,11.35,> rotate <0,-79.02,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,58.65,> rotate <0,-261.52,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,65.13,> rotate <0,-336.77,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-42.17,> rotate <0,-126.47,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-41.35,> rotate <0,-273.3,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-11.55,> rotate <0,-332.66,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,64.20,> rotate <0,-17.86,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-8.75,> rotate <0,-139.17,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-78.97,> rotate <0,-130.33,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-79.37,> rotate <0,-87.57,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,72.00,> rotate <0,-268.8,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-16.53,> rotate <0,-240.08,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-27.50,> rotate <0,-107.58,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,83.33,> rotate <0,-67.5,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-4.08,> rotate <0,-269.2,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,81.43,> rotate <0,-307.06,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,32.42,> rotate <0,-134.88,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,36.03,> rotate <0,-180.41,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-44.20,> rotate <0,-273.98,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-71.60,> rotate <0,-141.78,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-73.08,> rotate <0,-142.9,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-11.40,> rotate <0,-224.19,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,66.78,> rotate <0,-191.89,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-1.02,> rotate <0,-303.31,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.46,0,0> rotate <0,0,-74.93,> rotate <0,-73.8,0>  pigment {rgb <0.3175,0.3175,0.3175>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,-5.37,> rotate <0,-58.17,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,30.57,> rotate <0,-133.56,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,43.63,> rotate <0,-174.59,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,35.08,> rotate <0,-59.12,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,-45.92,> rotate <0,-131.63,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,-41.22,> rotate <0,-182.23,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,48.02,> rotate <0,-53.03,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,44.02,> rotate <0,-49.45,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,15.87,> rotate <0,-81.94,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,-9.45,> rotate <0,-188.45,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,32.58,> rotate <0,-337.5,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,79.57,> rotate <0,-101.56,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,47.22,> rotate <0,-44.96,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,-62.00,> rotate <0,-272.6,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,14.18,> rotate <0,-161.61,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,-2.13,> rotate <0,-163.43,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,25.77,> rotate <0,-295.93,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,38.55,> rotate <0,-206.75,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,80.55,> rotate <0,-311.92,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,-38.52,> rotate <0,-83.21,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,-21.70,> rotate <0,-273.56,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,28.97,> rotate <0,-101.19,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,5.03,> rotate <0,-243.32,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,-33.27,> rotate <0,-254.3,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,-35.67,> rotate <0,-42.67,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.47,0,0> rotate <0,0,-42.92,> rotate <0,-193.83,0>  pigment {rgb <0.31625,0.31625,0.31625>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-73.40,> rotate <0,-232.88,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,30.57,> rotate <0,-278.21,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-59.18,> rotate <0,-105.81,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-30.92,> rotate <0,-244.89,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-13.20,> rotate <0,-174.67,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,61.08,> rotate <0,-173.09,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-14.48,> rotate <0,-87.4,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-26.58,> rotate <0,-142.48,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-58.73,> rotate <0,-159.36,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,54.02,> rotate <0,-219.56,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-42.15,> rotate <0,-91.97,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-40.32,> rotate <0,-132.41,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-43.67,> rotate <0,-282.21,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,78.97,> rotate <0,-246.43,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,37.38,> rotate <0,-246.35,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,15.68,> rotate <0,-67.66,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-12.87,> rotate <0,-321.05,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-23.08,> rotate <0,-111.75,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,48.68,> rotate <0,-344.27,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-52.75,> rotate <0,-200.16,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,75.88,> rotate <0,-242.7,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,21.42,> rotate <0,-283.07,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-0.90,> rotate <0,-290.15,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,24.12,> rotate <0,-187.36,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,12.97,> rotate <0,-213.53,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,30.48,> rotate <0,-97.14,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,20.77,> rotate <0,-344.37,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,13.38,> rotate <0,-1.43,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-22.52,> rotate <0,-11.55,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-2.77,> rotate <0,-356.98,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,18.20,> rotate <0,-2.26,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,37.27,> rotate <0,-222.63,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,17.15,> rotate <0,-74.35,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-87.48,> rotate <0,-352.01,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-39.68,> rotate <0,-281.24,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-28.33,> rotate <0,-191,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-3.55,> rotate <0,-321.32,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,33.43,> rotate <0,-313.48,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.48,0,0> rotate <0,0,-50.82,> rotate <0,-26.53,0>  pigment {rgb <0.315,0.315,0.315>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,-45.32,> rotate <0,-217.53,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,4.42,> rotate <0,-337.49,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,35.05,> rotate <0,-114.64,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,37.72,> rotate <0,-20.92,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,54.43,> rotate <0,-206.65,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,-22.43,> rotate <0,-92.24,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,-35.88,> rotate <0,-118.55,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,-53.43,> rotate <0,-129.85,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,-49.57,> rotate <0,-245.62,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,10.60,> rotate <0,-36.2,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,30.98,> rotate <0,-299.66,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,31.93,> rotate <0,-280.96,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,49.85,> rotate <0,-217.16,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,35.40,> rotate <0,-9.34,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,-43.18,> rotate <0,-144.5,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,-70.98,> rotate <0,-248.58,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,19.62,> rotate <0,-62.29,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,36.40,> rotate <0,-272.01,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,5.15,> rotate <0,-78.68,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,-6.78,> rotate <0,-116.51,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,36.65,> rotate <0,-239.74,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,23.28,> rotate <0,-275.53,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.49,0,0> rotate <0,0,-46.60,> rotate <0,-283.11,0>  pigment {rgb <0.31375,0.31375,0.31375>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,1.22,> rotate <0,-252.85,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,33.28,> rotate <0,-30.74,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,24.27,> rotate <0,-77.03,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,11.13,> rotate <0,-2.51,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-71.92,> rotate <0,-45.57,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,70.13,> rotate <0,-332.66,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-8.37,> rotate <0,-243.9,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-11.30,> rotate <0,-165.81,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-19.67,> rotate <0,-233.15,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,7.25,> rotate <0,-252.58,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-31.05,> rotate <0,-286.1,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,30.55,> rotate <0,-48.84,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,70.37,> rotate <0,-351.82,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,15.65,> rotate <0,-178.92,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-29.47,> rotate <0,-344.9,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-77.52,> rotate <0,-334.46,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-16.85,> rotate <0,-233.23,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,65.02,> rotate <0,-17.92,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-52.08,> rotate <0,-110.16,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-5.18,> rotate <0,-74.1,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-69.63,> rotate <0,-5.16,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,31.80,> rotate <0,-15.7,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,19.65,> rotate <0,-17.45,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,18.02,> rotate <0,-44.11,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-15.47,> rotate <0,-295.89,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-46.65,> rotate <0,-123,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-16.93,> rotate <0,-71.9,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-23.63,> rotate <0,-50.35,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,52.98,> rotate <0,-282.9,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-6.42,> rotate <0,-34.25,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-1.18,> rotate <0,-233.24,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-36.87,> rotate <0,-23.23,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-38.38,> rotate <0,-329.83,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,29.62,> rotate <0,-230.04,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-49.13,> rotate <0,-173.74,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,19.23,> rotate <0,-21.68,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,17.08,> rotate <0,-112.95,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,3.32,> rotate <0,-272.67,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-46.55,> rotate <0,-341.42,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,45.53,> rotate <0,-165.06,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-69.63,> rotate <0,-327.7,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,12.65,> rotate <0,-126.68,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-58.98,> rotate <0,-186.87,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,59.48,> rotate <0,-189.1,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-76.80,> rotate <0,-211.33,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,-8.52,> rotate <0,-30.11,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.50,0,0> rotate <0,0,8.23,> rotate <0,-286.59,0>  pigment {rgb <0.3125,0.3125,0.3125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,55.27,> rotate <0,-73.77,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,-48.10,> rotate <0,-130.57,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,-46.63,> rotate <0,-259.76,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,52.37,> rotate <0,-235.71,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,-65.10,> rotate <0,-160.05,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,-82.22,> rotate <0,-150.18,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,53.23,> rotate <0,-125.95,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,-31.55,> rotate <0,-15.61,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,41.03,> rotate <0,-80.71,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,26.90,> rotate <0,-302.64,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,-34.15,> rotate <0,-95.15,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,21.95,> rotate <0,-150.7,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,36.98,> rotate <0,-283.43,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,26.30,> rotate <0,-285.33,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,-5.22,> rotate <0,-99.15,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,28.80,> rotate <0,-328.13,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,-68.83,> rotate <0,-346.22,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,5.65,> rotate <0,-17.09,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,-77.38,> rotate <0,-48.99,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,46.95,> rotate <0,-77.68,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,-40.82,> rotate <0,-349.54,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,28.83,> rotate <0,-259.7,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,-65.50,> rotate <0,-248.94,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,7.28,> rotate <0,-301.03,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,53.67,> rotate <0,-159.78,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,36.43,> rotate <0,-242.95,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,57.45,> rotate <0,-356.75,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,26.82,> rotate <0,-302.95,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,59.43,> rotate <0,-314.85,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,60.73,> rotate <0,-65.45,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,-2.77,> rotate <0,-225.33,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,13.73,> rotate <0,-154.17,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.22 translate <10005.51,0,0> rotate <0,0,59.75,> rotate <0,-244.31,0>  pigment {rgb <0.31125,0.31125,0.31125>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-66.67,> rotate <0,-289.3,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,48.18,> rotate <0,-112.22,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-12.20,> rotate <0,-116.99,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,38.05,> rotate <0,-62.15,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-43.60,> rotate <0,-105.99,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,1.95,> rotate <0,-166.73,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-18.67,> rotate <0,-99.1,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-30.73,> rotate <0,-272.53,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-70.10,> rotate <0,-126.82,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,55.35,> rotate <0,-203.53,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-18.67,> rotate <0,-71.03,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,78.83,> rotate <0,-337.47,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,59.58,> rotate <0,-124.46,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,55.15,> rotate <0,-28,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-5.22,> rotate <0,-55.16,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-52.73,> rotate <0,-351.65,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,19.27,> rotate <0,-263.35,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-48.00,> rotate <0,-8.92,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-14.35,> rotate <0,-110.24,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,27.97,> rotate <0,-87.74,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,35.22,> rotate <0,-225.78,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-39.38,> rotate <0,-251.7,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,11.98,> rotate <0,-11.75,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-0.17,> rotate <0,-224.39,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-17.15,> rotate <0,-271.95,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-53.10,> rotate <0,-121.27,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,70.78,> rotate <0,-336.5,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-79.10,> rotate <0,-111.41,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-37.92,> rotate <0,-312.75,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,-7.83,> rotate <0,-39,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.52,0,0> rotate <0,0,51.43,> rotate <0,-108.35,0>  pigment {rgb <0.31,0.31,0.31>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,-7.17,> rotate <0,-132.9,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,-53.10,> rotate <0,-130.58,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,61.68,> rotate <0,-245.95,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,-5.83,> rotate <0,-230.28,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,76.80,> rotate <0,-240.88,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,83.72,> rotate <0,-13.72,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,34.98,> rotate <0,-158.38,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,28.62,> rotate <0,-222.49,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,66.67,> rotate <0,-310.8,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,-39.20,> rotate <0,-311.58,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,-70.93,> rotate <0,-70.77,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,12.37,> rotate <0,-289.91,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,39.27,> rotate <0,-224.9,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,-13.80,> rotate <0,-122.67,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,6.67,> rotate <0,-48.11,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,-62.57,> rotate <0,-49.44,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,57.98,> rotate <0,-24.53,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,-59.42,> rotate <0,-138.23,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,-6.52,> rotate <0,-330.82,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,-30.17,> rotate <0,-56.98,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,44.47,> rotate <0,-314.58,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,-37.73,> rotate <0,-94.25,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,14.42,> rotate <0,-239.31,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,-12.53,> rotate <0,-126.68,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,22.47,> rotate <0,-268.96,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,56.80,> rotate <0,-149.97,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,63.07,> rotate <0,-82.54,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.53,0,0> rotate <0,0,-33.55,> rotate <0,-258.83,0>  pigment {rgb <0.30875,0.30875,0.30875>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-59.20,> rotate <0,-297.69,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,12.57,> rotate <0,-313.91,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,25.05,> rotate <0,-74.54,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,25.93,> rotate <0,-30.92,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-56.40,> rotate <0,-117.28,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-29.15,> rotate <0,-112,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-3.82,> rotate <0,-237.23,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,14.18,> rotate <0,-88.09,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,24.32,> rotate <0,-298.63,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-20.87,> rotate <0,-82.78,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-8.42,> rotate <0,-240.2,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-3.48,> rotate <0,-348.89,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,36.47,> rotate <0,-56.13,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-67.50,> rotate <0,-337.16,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,60.45,> rotate <0,-322.75,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,50.93,> rotate <0,-65.05,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-19.30,> rotate <0,-287.07,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,18.53,> rotate <0,-83.38,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-44.77,> rotate <0,-146.63,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,14.30,> rotate <0,-83.48,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-41.87,> rotate <0,-135.34,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,34.13,> rotate <0,-66.12,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,1.38,> rotate <0,-67.13,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-5.50,> rotate <0,-313.03,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,87.30,> rotate <0,-351.76,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-31.38,> rotate <0,-89.09,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,27.62,> rotate <0,-50.55,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,28.98,> rotate <0,-14.46,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,24.32,> rotate <0,-265.84,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,45.18,> rotate <0,-313.33,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,2.90,> rotate <0,-96.84,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-71.98,> rotate <0,-192.44,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,71.23,> rotate <0,-204.3,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-33.43,> rotate <0,-310.08,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-8.98,> rotate <0,-197.14,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-0.30,> rotate <0,-56.24,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,24.57,> rotate <0,-265.62,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-33.92,> rotate <0,-90.32,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-0.28,> rotate <0,-96.82,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-46.25,> rotate <0,-219.08,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,39.40,> rotate <0,-99.67,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,-67.83,> rotate <0,-33.56,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.54,0,0> rotate <0,0,14.13,> rotate <0,-158.05,0>  pigment {rgb <0.3075,0.3075,0.3075>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,44.65,> rotate <0,-330.73,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-30.62,> rotate <0,-157.4,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-65.30,> rotate <0,-197.03,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,60.35,> rotate <0,-14.2,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-41.32,> rotate <0,-120.68,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-10.77,> rotate <0,-297.7,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-34.88,> rotate <0,-268.35,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-53.90,> rotate <0,-145.93,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-32.83,> rotate <0,-186.71,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,24.98,> rotate <0,-298,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,70.90,> rotate <0,-348.9,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,28.62,> rotate <0,-286.66,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,17.05,> rotate <0,-84.27,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-22.40,> rotate <0,-290.16,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-36.33,> rotate <0,-113.46,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-2.47,> rotate <0,-44.92,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-32.67,> rotate <0,-259.26,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,53.20,> rotate <0,-349.18,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,34.48,> rotate <0,-106.55,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,20.57,> rotate <0,-229.6,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-10.45,> rotate <0,-180.18,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,69.25,> rotate <0,-145.56,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-62.75,> rotate <0,-147.73,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,14.32,> rotate <0,-87.62,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-12.83,> rotate <0,-334.2,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-42.03,> rotate <0,-29.91,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,50.32,> rotate <0,-292.83,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,40.15,> rotate <0,-200.08,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-5.92,> rotate <0,-143.63,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-29.57,> rotate <0,-8.42,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-14.58,> rotate <0,-92.39,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,60.13,> rotate <0,-350.64,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,17.32,> rotate <0,-120.2,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,2.92,> rotate <0,-357.99,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,47.55,> rotate <0,-118.68,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,57.87,> rotate <0,-185.21,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.55,0,0> rotate <0,0,-69.98,> rotate <0,-95.66,0>  pigment {rgb <0.30625,0.30625,0.30625>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,48.15,> rotate <0,-226.36,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-56.98,> rotate <0,-178.04,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-43.62,> rotate <0,-138.13,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,21.47,> rotate <0,-16.43,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-78.98,> rotate <0,-46.88,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,10.75,> rotate <0,-204.9,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,17.65,> rotate <0,-123.05,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,24.12,> rotate <0,-61.09,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-54.32,> rotate <0,-291.95,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-55.93,> rotate <0,-34.98,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,52.32,> rotate <0,-291.86,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-12.67,> rotate <0,-52.4,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-37.23,> rotate <0,-82.07,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,12.08,> rotate <0,-329.24,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-84.47,> rotate <0,-235.83,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,37.00,> rotate <0,-304.62,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,12.15,> rotate <0,-24.28,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-58.33,> rotate <0,-251.83,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-70.87,> rotate <0,-166.71,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-62.45,> rotate <0,-179.42,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-34.78,> rotate <0,-215.58,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-11.23,> rotate <0,-263.69,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-31.92,> rotate <0,-296.5,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,64.18,> rotate <0,-1.61,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,19.70,> rotate <0,-55.58,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-21.72,> rotate <0,-12,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,2.53,> rotate <0,-284.32,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,28.42,> rotate <0,-262.96,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,62.87,> rotate <0,-258.14,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,61.52,> rotate <0,-43.99,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-52.38,> rotate <0,-8.61,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-53.75,> rotate <0,-25.62,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-20.08,> rotate <0,-125.34,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,3.07,> rotate <0,-54.97,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,20.55,> rotate <0,-183.04,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,11.65,> rotate <0,-220.43,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,6.62,> rotate <0,-283.87,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-44.83,> rotate <0,-323.35,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,33.50,> rotate <0,-353.66,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,26.63,> rotate <0,-185.08,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,18.12,> rotate <0,-284.03,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,33.72,> rotate <0,-156.04,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-16.27,> rotate <0,-343.69,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-41.38,> rotate <0,-316.6,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,11.62,> rotate <0,-133.98,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-27.80,> rotate <0,-2.9,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,-40.35,> rotate <0,-92.55,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,31.17,> rotate <0,-144.18,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.56,0,0> rotate <0,0,58.00,> rotate <0,-330.52,0>  pigment {rgb <0.305,0.305,0.305>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,6.00,> rotate <0,-311.95,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,74.98,> rotate <0,-11.41,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,6.80,> rotate <0,-190.49,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,-20.58,> rotate <0,-195.94,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,-47.32,> rotate <0,-327.07,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,-24.77,> rotate <0,-9.33,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,-36.70,> rotate <0,-96,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,68.03,> rotate <0,-25.59,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,-9.85,> rotate <0,-16.52,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,4.83,> rotate <0,-289.13,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,24.30,> rotate <0,-264.38,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,-85.78,> rotate <0,-205.23,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,-10.78,> rotate <0,-35.5,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,32.38,> rotate <0,-351.21,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,-57.93,> rotate <0,-242.33,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,61.22,> rotate <0,-0.4,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,-27.67,> rotate <0,-61.4,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,25.05,> rotate <0,-33.93,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,1.75,> rotate <0,-34.51,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,-16.18,> rotate <0,-206.13,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,15.08,> rotate <0,-288.83,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,-24.48,> rotate <0,-70.03,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,-11.20,> rotate <0,-53.99,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,-54.88,> rotate <0,-156.7,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,14.58,> rotate <0,-292.35,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,23.85,> rotate <0,-320.27,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,8.45,> rotate <0,-178.76,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,16.85,> rotate <0,-343.26,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,57.17,> rotate <0,-348.32,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,0.60,> rotate <0,-330.27,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,9.13,> rotate <0,-337.28,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.21 translate <10005.57,0,0> rotate <0,0,-42.20,> rotate <0,-140.46,0>  pigment {rgb <0.30375,0.30375,0.30375>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,44.75,> rotate <0,-343.42,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-30.35,> rotate <0,-137.48,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-42.05,> rotate <0,-305.61,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-25.42,> rotate <0,-104.65,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,52.02,> rotate <0,-6.06,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,10.23,> rotate <0,-83.8,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,33.53,> rotate <0,-51.12,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,15.63,> rotate <0,-67.53,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,14.93,> rotate <0,-16.28,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,30.28,> rotate <0,-230.8,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,2.72,> rotate <0,-262.84,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,33.17,> rotate <0,-332.31,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,22.08,> rotate <0,-256.58,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,49.35,> rotate <0,-9.79,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-10.48,> rotate <0,-55.89,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,50.17,> rotate <0,-35.24,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-11.37,> rotate <0,-326.63,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,20.73,> rotate <0,-16.99,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,26.85,> rotate <0,-107.85,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-72.25,> rotate <0,-171.05,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,49.85,> rotate <0,-52.22,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,37.57,> rotate <0,-55.28,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,42.17,> rotate <0,-273.91,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-71.45,> rotate <0,-1.17,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-36.05,> rotate <0,-115.8,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-53.02,> rotate <0,-130.07,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-33.55,> rotate <0,-242.47,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,64.38,> rotate <0,-30.75,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-13.60,> rotate <0,-157.75,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-40.78,> rotate <0,-229.74,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-4.00,> rotate <0,-126.61,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,3.28,> rotate <0,-113.3,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-10.53,> rotate <0,-257.45,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-16.38,> rotate <0,-283.88,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-25.18,> rotate <0,-333.43,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,47.23,> rotate <0,-108.96,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,64.60,> rotate <0,-134.15,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,18.42,> rotate <0,-172.62,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-35.45,> rotate <0,-124.57,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-43.12,> rotate <0,-153.88,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-36.95,> rotate <0,-265.71,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,22.63,> rotate <0,-255.24,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-21.77,> rotate <0,-291.58,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,24.72,> rotate <0,-51.08,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-1.90,> rotate <0,-112.33,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-29.67,> rotate <0,-157.37,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-5.12,> rotate <0,-141.35,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-7.05,> rotate <0,-156.43,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.58,0,0> rotate <0,0,-42.85,> rotate <0,-249.09,0>  pigment {rgb <0.3025,0.3025,0.3025>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,-5.87,> rotate <0,-98.1,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,-8.58,> rotate <0,-137.18,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,-60.30,> rotate <0,-119.45,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,-73.20,> rotate <0,-223.31,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,47.83,> rotate <0,-311.96,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,62.52,> rotate <0,-279.39,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,28.97,> rotate <0,-331.4,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,51.95,> rotate <0,-230.02,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,33.82,> rotate <0,-52.09,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,40.85,> rotate <0,-198.88,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,-57.63,> rotate <0,-132.9,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,42.92,> rotate <0,-351.78,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,46.72,> rotate <0,-321.33,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,14.03,> rotate <0,-64.99,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,-51.82,> rotate <0,-97.83,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,17.93,> rotate <0,-208.3,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,-41.58,> rotate <0,-196.65,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,-30.75,> rotate <0,-276.26,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,-38.92,> rotate <0,-15.33,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,52.83,> rotate <0,-7.93,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,59.32,> rotate <0,-162.85,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.59,0,0> rotate <0,0,46.23,> rotate <0,-260.09,0>  pigment {rgb <0.30125,0.30125,0.30125>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,30.20,> rotate <0,-317.16,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-23.15,> rotate <0,-254.2,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-38.90,> rotate <0,-346.73,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,41.60,> rotate <0,-344.1,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,45.68,> rotate <0,-56.5,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-34.68,> rotate <0,-236.68,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-36.78,> rotate <0,-98.85,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-53.47,> rotate <0,-72.73,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-19.93,> rotate <0,-250.48,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,48.72,> rotate <0,-222.42,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,45.60,> rotate <0,-247.95,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,61.70,> rotate <0,-342.84,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,32.30,> rotate <0,-310.26,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,45.27,> rotate <0,-196.47,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,40.67,> rotate <0,-184.03,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,32.47,> rotate <0,-98.12,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-56.37,> rotate <0,-95.73,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-34.07,> rotate <0,-162.49,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,20.10,> rotate <0,-291.55,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,19.90,> rotate <0,-34.53,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-68.18,> rotate <0,-216.28,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,40.23,> rotate <0,-353.66,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-24.40,> rotate <0,-75.98,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,45.23,> rotate <0,-110.32,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,41.03,> rotate <0,-3.38,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,40.60,> rotate <0,-297.65,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-3.75,> rotate <0,-126.15,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-14.30,> rotate <0,-294.39,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,7.08,> rotate <0,-96.3,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,39.97,> rotate <0,-260.43,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-41.83,> rotate <0,-213.68,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,31.43,> rotate <0,-72.3,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-55.55,> rotate <0,-101.83,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-9.18,> rotate <0,-117.54,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,62.65,> rotate <0,-80.09,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,11.80,> rotate <0,-146.6,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-12.57,> rotate <0,-59.88,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-10.33,> rotate <0,-81.26,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-17.13,> rotate <0,-151.79,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-59.63,> rotate <0,-331.46,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-18.03,> rotate <0,-355.39,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,31.75,> rotate <0,-286.24,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-33.60,> rotate <0,-205.44,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-22.67,> rotate <0,-320.75,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,-24.12,> rotate <0,-49.9,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,18.10,> rotate <0,-311.37,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.60,0,0> rotate <0,0,15.13,> rotate <0,-237.68,0>  pigment {rgb <0.3,0.3,0.3>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-42.55,> rotate <0,-321.76,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,50.45,> rotate <0,-314.63,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,55.28,> rotate <0,-110.72,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-5.12,> rotate <0,-349.85,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,38.32,> rotate <0,-194,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-45.93,> rotate <0,-335.78,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,11.82,> rotate <0,-297.18,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,4.40,> rotate <0,-336.65,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-63.05,> rotate <0,-273.92,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,5.45,> rotate <0,-236.35,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-5.73,> rotate <0,-263.38,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,0.28,> rotate <0,-298.68,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-64.33,> rotate <0,-179.7,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,16.93,> rotate <0,-277.77,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-15.67,> rotate <0,-23.66,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,14.48,> rotate <0,-86.8,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-26.27,> rotate <0,-238.88,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-24.67,> rotate <0,-116.14,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-35.42,> rotate <0,-289.91,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-32.72,> rotate <0,-98.65,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,16.45,> rotate <0,-171.4,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,16.38,> rotate <0,-224.3,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,75.77,> rotate <0,-124.88,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,84.90,> rotate <0,-53.08,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-21.17,> rotate <0,-117.93,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,39.77,> rotate <0,-145.5,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-24.77,> rotate <0,-335.88,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-46.20,> rotate <0,-147.83,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-36.23,> rotate <0,-275.87,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-64.72,> rotate <0,-327.5,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,56.58,> rotate <0,-162.8,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-56.85,> rotate <0,-351.33,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-5.33,> rotate <0,-106.02,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,-4.10,> rotate <0,-30.92,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,5.50,> rotate <0,-316.15,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,63.50,> rotate <0,-73.03,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,55.72,> rotate <0,-186.9,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,8.78,> rotate <0,-156.31,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.61,0,0> rotate <0,0,0.10,> rotate <0,-358.69,0>  pigment {rgb <0.29875,0.29875,0.29875>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,32.22,> rotate <0,-301.15,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,38.28,> rotate <0,-325.85,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,13.12,> rotate <0,-330.27,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,-61.70,> rotate <0,-204.3,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,23.60,> rotate <0,-278.88,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,29.87,> rotate <0,-275.24,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,79.68,> rotate <0,-18.07,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,-57.90,> rotate <0,-243.96,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,0.38,> rotate <0,-228.95,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,5.52,> rotate <0,-248.15,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,84.25,> rotate <0,-157.43,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,29.97,> rotate <0,-145.89,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,-19.03,> rotate <0,-103.33,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,-25.20,> rotate <0,-105.28,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,4.37,> rotate <0,-45.59,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,61.85,> rotate <0,-64.23,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,50.80,> rotate <0,-28.04,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,62.25,> rotate <0,-304.9,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,55.70,> rotate <0,-359.29,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,49.62,> rotate <0,-224.1,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,8.57,> rotate <0,-32.84,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,82.50,> rotate <0,-222.59,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,4.17,> rotate <0,-91.24,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,65.25,> rotate <0,-284.11,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,54.78,> rotate <0,-173.77,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,-58.48,> rotate <0,-351.81,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,63.87,> rotate <0,-26.94,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,-32.18,> rotate <0,-144.29,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,-57.30,> rotate <0,-118.72,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,-12.28,> rotate <0,-288.31,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,-25.57,> rotate <0,-95.98,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,23.60,> rotate <0,-103,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,-29.50,> rotate <0,-359.87,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,22.62,> rotate <0,-196.6,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.62,0,0> rotate <0,0,47.02,> rotate <0,-298,0>  pigment {rgb <0.2975,0.2975,0.2975>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-32.72,> rotate <0,-160.68,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,8.65,> rotate <0,-158.76,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-42.43,> rotate <0,-305.97,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-17.87,> rotate <0,-111.78,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-31.70,> rotate <0,-102.6,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-47.87,> rotate <0,-224.13,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,29.65,> rotate <0,-123.29,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,16.78,> rotate <0,-65.85,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-35.85,> rotate <0,-196.73,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,2.68,> rotate <0,-326.81,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,12.12,> rotate <0,-131.73,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,60.05,> rotate <0,-261.43,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-55.52,> rotate <0,-140.46,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,38.45,> rotate <0,-306.89,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-80.80,> rotate <0,-100.01,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,9.90,> rotate <0,-241.9,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-0.88,> rotate <0,-256.38,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-32.17,> rotate <0,-91.08,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,23.10,> rotate <0,-299.79,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-0.50,> rotate <0,-93.89,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,65.02,> rotate <0,-129.8,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-48.32,> rotate <0,-230.45,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,12.12,> rotate <0,-108.63,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,48.13,> rotate <0,-115.3,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-19.93,> rotate <0,-198.55,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,22.50,> rotate <0,-205.26,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-12.80,> rotate <0,-61.1,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-47.70,> rotate <0,-155.07,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-5.33,> rotate <0,-177.76,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,56.62,> rotate <0,-338.42,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-26.98,> rotate <0,-350.32,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-37.83,> rotate <0,-321.6,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-8.43,> rotate <0,-152.73,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,25.88,> rotate <0,-182.96,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,65.70,> rotate <0,-85.61,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,27.88,> rotate <0,-126.62,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-39.95,> rotate <0,-88.72,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-40.43,> rotate <0,-173.2,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,-23.70,> rotate <0,-187.57,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.20 translate <10005.63,0,0> rotate <0,0,42.80,> rotate <0,-79.57,0>  pigment {rgb <0.29625,0.29625,0.29625>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-71.50,> rotate <0,-125,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,4.87,> rotate <0,-120.31,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-14.70,> rotate <0,-116.37,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-12.75,> rotate <0,-241.9,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-54.60,> rotate <0,-255.03,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,24.67,> rotate <0,-167.2,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,60.00,> rotate <0,-317.95,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-11.78,> rotate <0,-95.35,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,36.35,> rotate <0,-343.94,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-73.45,> rotate <0,-235.09,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-19.70,> rotate <0,-81.5,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-35.67,> rotate <0,-208.39,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-37.63,> rotate <0,-88.14,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-32.15,> rotate <0,-127.62,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-3.10,> rotate <0,-298.33,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,56.28,> rotate <0,-96.61,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,48.30,> rotate <0,-70.35,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-16.98,> rotate <0,-81.12,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,21.22,> rotate <0,-288.83,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,52.87,> rotate <0,-74.03,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,54.28,> rotate <0,-136,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-49.65,> rotate <0,-250.42,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-43.43,> rotate <0,-282.39,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,1.78,> rotate <0,-146.6,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,27.63,> rotate <0,-111.14,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-75.90,> rotate <0,-174.32,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,12.45,> rotate <0,-39.16,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-60.67,> rotate <0,-230.79,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,8.03,> rotate <0,-276.41,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-80.47,> rotate <0,-84.29,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-3.40,> rotate <0,-39.42,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,21.42,> rotate <0,-306.42,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-83.22,> rotate <0,-225.45,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-83.03,> rotate <0,-226.2,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,57.87,> rotate <0,-64.28,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-24.85,> rotate <0,-285.61,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,39.58,> rotate <0,-230.65,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,8.27,> rotate <0,-330.29,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,15.75,> rotate <0,-251.35,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.64,0,0> rotate <0,0,-62.87,> rotate <0,-13.4,0>  pigment {rgb <0.295,0.295,0.295>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,52.90,> rotate <0,-249.05,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,-14.48,> rotate <0,-114.02,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,14.55,> rotate <0,-289.11,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,-16.38,> rotate <0,-74.75,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,-9.87,> rotate <0,-308.1,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,16.45,> rotate <0,-294.33,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,57.48,> rotate <0,-324.74,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,-10.10,> rotate <0,-101.66,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,-39.52,> rotate <0,-258.07,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,53.72,> rotate <0,-208.46,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,5.58,> rotate <0,-68.53,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,-38.98,> rotate <0,-277.11,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,-33.08,> rotate <0,-339.72,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,21.58,> rotate <0,-64.6,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,24.08,> rotate <0,-127.88,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,47.87,> rotate <0,-11.11,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,52.88,> rotate <0,-330.46,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,55.12,> rotate <0,-40.76,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,24.85,> rotate <0,-56.29,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,-48.60,> rotate <0,-114.58,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,-33.70,> rotate <0,-300.08,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,-62.00,> rotate <0,-349.24,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,42.08,> rotate <0,-17.58,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,-13.77,> rotate <0,-163.58,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,31.18,> rotate <0,-207.16,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,-15.37,> rotate <0,-203.22,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,63.35,> rotate <0,-230.66,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,27.90,> rotate <0,-283.55,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,-46.78,> rotate <0,-109.06,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,-52.45,> rotate <0,-303.58,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.65,0,0> rotate <0,0,30.42,> rotate <0,-350.21,0>  pigment {rgb <0.29375,0.29375,0.29375>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,-64.67,> rotate <0,-154.77,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,17.70,> rotate <0,-266.78,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,20.02,> rotate <0,-40.59,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,-0.62,> rotate <0,-295.18,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,5.77,> rotate <0,-324.63,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,10.00,> rotate <0,-60.44,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,-28.42,> rotate <0,-243.07,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,6.53,> rotate <0,-58,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,33.73,> rotate <0,-303.85,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,-70.62,> rotate <0,-201.46,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,53.57,> rotate <0,-317.56,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,10.08,> rotate <0,-131.19,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,-24.72,> rotate <0,-294,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,-51.22,> rotate <0,-90.21,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,-36.13,> rotate <0,-220.26,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,-63.12,> rotate <0,-352.25,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,-47.67,> rotate <0,-166.13,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,-19.70,> rotate <0,-114.17,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,-52.17,> rotate <0,-208.8,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,2.10,> rotate <0,-354.1,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,31.75,> rotate <0,-175.39,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,32.22,> rotate <0,-322.03,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,38.18,> rotate <0,-350.23,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,44.97,> rotate <0,-57.52,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,-27.60,> rotate <0,-192.11,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,53.40,> rotate <0,-129.59,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,5.43,> rotate <0,-91.24,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,-19.42,> rotate <0,-113.33,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,57.82,> rotate <0,-284.19,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,70.82,> rotate <0,-105.34,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,-31.45,> rotate <0,-4.03,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,-11.78,> rotate <0,-88.68,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,11.55,> rotate <0,-198.14,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,45.50,> rotate <0,-269.98,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,-20.75,> rotate <0,-142.3,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.66,0,0> rotate <0,0,29.82,> rotate <0,-252.66,0>  pigment {rgb <0.2925,0.2925,0.2925>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,-6.02,> rotate <0,-83.75,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,-0.42,> rotate <0,-80.38,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,-16.25,> rotate <0,-122.37,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,-2.40,> rotate <0,-33.2,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,-73.18,> rotate <0,-317.34,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,32.45,> rotate <0,-320.21,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,11.33,> rotate <0,-198.63,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,-25.62,> rotate <0,-221.84,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,27.27,> rotate <0,-48.06,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,64.83,> rotate <0,-205.38,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,42.95,> rotate <0,-333.68,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,82.53,> rotate <0,-310.65,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,-23.15,> rotate <0,-126.23,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,65.10,> rotate <0,-154.51,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,22.95,> rotate <0,-261.03,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,15.58,> rotate <0,-134.4,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,16.03,> rotate <0,-300.87,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,41.72,> rotate <0,-310.49,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,53.05,> rotate <0,-6.28,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,-58.60,> rotate <0,-247.06,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,0.72,> rotate <0,-221.38,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,19.80,> rotate <0,-93,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,26.18,> rotate <0,-321.14,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,49.20,> rotate <0,-29.64,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,-0.48,> rotate <0,-135.49,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,49.40,> rotate <0,-320.5,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,67.17,> rotate <0,-1.18,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,-24.83,> rotate <0,-185.84,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,9.53,> rotate <0,-191.6,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,-30.77,> rotate <0,-70.79,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,68.47,> rotate <0,-112.72,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,30.33,> rotate <0,-309.75,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,-50.95,> rotate <0,-345.28,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,26.30,> rotate <0,-227.1,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,1.28,> rotate <0,-325.54,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,-0.70,> rotate <0,-301.1,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,38.33,> rotate <0,-115.06,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.67,0,0> rotate <0,0,-43.05,> rotate <0,-253.43,0>  pigment {rgb <0.29125,0.29125,0.29125>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,55.38,> rotate <0,-236.91,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-27.65,> rotate <0,-224.66,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,28.95,> rotate <0,-184.38,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-41.80,> rotate <0,-253.51,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-32.02,> rotate <0,-98.16,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,55.80,> rotate <0,-328,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-52.82,> rotate <0,-208.02,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,6.95,> rotate <0,-8.1,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,3.12,> rotate <0,-272.48,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-29.90,> rotate <0,-24.03,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,80.00,> rotate <0,-270.04,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-12.23,> rotate <0,-159.14,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-53.20,> rotate <0,-128.02,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,37.25,> rotate <0,-29.04,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,12.82,> rotate <0,-89.73,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-9.70,> rotate <0,-314.23,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,58.62,> rotate <0,-319.82,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,2.08,> rotate <0,-200.43,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,19.67,> rotate <0,-328.4,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,47.90,> rotate <0,-90.24,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,58.43,> rotate <0,-32.17,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,46.82,> rotate <0,-139.38,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-21.27,> rotate <0,-24.72,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,12.67,> rotate <0,-130.8,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,47.92,> rotate <0,-298.03,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-40.83,> rotate <0,-251.18,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,15.05,> rotate <0,-73.96,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-29.73,> rotate <0,-291.73,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,24.28,> rotate <0,-320.99,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,8.48,> rotate <0,-0.63,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-48.22,> rotate <0,-7.61,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-13.07,> rotate <0,-21.72,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,53.90,> rotate <0,-151.15,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-26.30,> rotate <0,-314.2,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,23.92,> rotate <0,-222.56,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,62.50,> rotate <0,-124.82,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,3.68,> rotate <0,-50.28,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,48.73,> rotate <0,-72.79,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,29.63,> rotate <0,-149.9,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,10.82,> rotate <0,-202.3,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-84.77,> rotate <0,-24.37,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,13.18,> rotate <0,-46.6,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-52.40,> rotate <0,-101.47,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-80.92,> rotate <0,-126.08,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-80.22,> rotate <0,-64.5,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-30.12,> rotate <0,-316.5,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-33.53,> rotate <0,-2.01,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-50.38,> rotate <0,-53.15,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-67.75,> rotate <0,-33.87,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,40.47,> rotate <0,-220.94,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,16.08,> rotate <0,-105.07,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,17.08,> rotate <0,-189.24,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,24.60,> rotate <0,-204.25,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.68,0,0> rotate <0,0,-11.85,> rotate <0,-77.85,0>  pigment {rgb <0.29,0.29,0.29>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-48.75,> rotate <0,-227.99,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-56.50,> rotate <0,-10.44,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,26.93,> rotate <0,-84.29,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-22.77,> rotate <0,-92.45,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-49.08,> rotate <0,-1.58,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-43.08,> rotate <0,-175.33,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-28.70,> rotate <0,-203.15,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-62.65,> rotate <0,-177.61,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,16.30,> rotate <0,-261.13,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,33.92,> rotate <0,-155.78,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-9.73,> rotate <0,-321.3,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-0.15,> rotate <0,-80.93,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-54.37,> rotate <0,-118.13,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-58.07,> rotate <0,-152.94,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,37.23,> rotate <0,-23.57,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-22.85,> rotate <0,-111.93,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-15.05,> rotate <0,-291.55,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,47.42,> rotate <0,-313.96,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,61.97,> rotate <0,-133.34,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-14.83,> rotate <0,-118.08,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,22.47,> rotate <0,-59.22,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-0.98,> rotate <0,-18.7,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-10.27,> rotate <0,-74.96,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,20.57,> rotate <0,-267.1,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,31.17,> rotate <0,-262.73,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-8.82,> rotate <0,-62.7,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-16.73,> rotate <0,-73.78,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-63.70,> rotate <0,-41.36,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-78.35,> rotate <0,-52.5,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-2.05,> rotate <0,-131.51,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-72.42,> rotate <0,-241.48,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,16.32,> rotate <0,-263.41,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-25.30,> rotate <0,-144.25,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-21.37,> rotate <0,-283.5,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-41.50,> rotate <0,-201.73,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,-45.18,> rotate <0,-130.15,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.19 translate <10005.69,0,0> rotate <0,0,46.18,> rotate <0,-114.13,0>  pigment {rgb <0.28875,0.28875,0.28875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,28.97,> rotate <0,-68.66,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,19.20,> rotate <0,-13.65,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,28.62,> rotate <0,-70.33,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-33.73,> rotate <0,-164.8,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-6.07,> rotate <0,-84.15,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-45.28,> rotate <0,-148.57,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,1.20,> rotate <0,-259.13,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-8.22,> rotate <0,-298.66,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-14.05,> rotate <0,-341.93,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,21.47,> rotate <0,-181.07,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-18.55,> rotate <0,-49.67,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-0.70,> rotate <0,-40.31,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-40.35,> rotate <0,-58.6,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-0.15,> rotate <0,-225.45,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-47.37,> rotate <0,-173.8,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-46.85,> rotate <0,-108.69,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,54.63,> rotate <0,-233.99,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-43.45,> rotate <0,-292.35,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,55.20,> rotate <0,-246.1,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,14.10,> rotate <0,-236.83,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-24.28,> rotate <0,-151.09,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-13.85,> rotate <0,-188.15,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-25.27,> rotate <0,-44.9,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,23.87,> rotate <0,-277.4,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,38.27,> rotate <0,-284.51,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,20.68,> rotate <0,-69.57,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-53.45,> rotate <0,-196.91,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-50.65,> rotate <0,-254.58,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,11.38,> rotate <0,-330.83,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-66.05,> rotate <0,-91.54,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,15.30,> rotate <0,-41.14,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-27.28,> rotate <0,-166.49,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,31.18,> rotate <0,-316.63,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,32.68,> rotate <0,-27.18,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-54.97,> rotate <0,-133.8,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,5.10,> rotate <0,-94.32,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,23.57,> rotate <0,-28.96,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-48.82,> rotate <0,-0.27,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-19.75,> rotate <0,-138,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-60.80,> rotate <0,-25.45,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-30.47,> rotate <0,-99.93,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-47.22,> rotate <0,-277.48,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-35.68,> rotate <0,-51.89,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,58.33,> rotate <0,-23.36,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,15.33,> rotate <0,-105.57,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-67.67,> rotate <0,-208.7,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-4.28,> rotate <0,-328.54,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-33.28,> rotate <0,-117.4,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-64.53,> rotate <0,-231.89,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-51.88,> rotate <0,-351.05,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,69.28,> rotate <0,-234.42,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.70,0,0> rotate <0,0,-69.73,> rotate <0,-321.07,0>  pigment {rgb <0.2875,0.2875,0.2875>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,17.90,> rotate <0,-93.62,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-26.17,> rotate <0,-324.05,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,35.37,> rotate <0,-139.61,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-6.73,> rotate <0,-68.48,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-36.75,> rotate <0,-241.82,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-43.07,> rotate <0,-215.04,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-4.98,> rotate <0,-343.8,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-33.57,> rotate <0,-174.15,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-3.97,> rotate <0,-7.51,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-32.58,> rotate <0,-263.68,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-7.98,> rotate <0,-128.87,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-20.88,> rotate <0,-107.93,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-50.23,> rotate <0,-148.71,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-36.83,> rotate <0,-25.51,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,36.42,> rotate <0,-128.68,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,24.25,> rotate <0,-286.66,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,40.88,> rotate <0,-109.51,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,30.78,> rotate <0,-264.15,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,6.20,> rotate <0,-277,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,30.20,> rotate <0,-81.78,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-48.82,> rotate <0,-114.18,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,20.27,> rotate <0,-324.76,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,59.05,> rotate <0,-120.34,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,46.12,> rotate <0,-222.33,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,5.53,> rotate <0,-62.83,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-16.42,> rotate <0,-73.83,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-43.52,> rotate <0,-277.98,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-21.28,> rotate <0,-71.27,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-64.42,> rotate <0,-356.05,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,18.82,> rotate <0,-243.87,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,32.60,> rotate <0,-102.42,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,43.63,> rotate <0,-219.55,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-24.03,> rotate <0,-278.47,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-22.40,> rotate <0,-229.1,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,-39.67,> rotate <0,-80.85,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,27.90,> rotate <0,-108.99,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.71,0,0> rotate <0,0,1.85,> rotate <0,-189.59,0>  pigment {rgb <0.28625,0.28625,0.28625>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,-55.05,> rotate <0,-237.78,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,-36.98,> rotate <0,-99.31,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,11.73,> rotate <0,-344.8,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,4.00,> rotate <0,-272.39,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,19.52,> rotate <0,-339.72,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,21.62,> rotate <0,-67,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,5.82,> rotate <0,-182.52,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,14.07,> rotate <0,-65.51,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,-18.55,> rotate <0,-85.84,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,-18.32,> rotate <0,-137.27,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,52.07,> rotate <0,-205.98,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,49.98,> rotate <0,-323.24,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,13.60,> rotate <0,-255.92,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,42.58,> rotate <0,-73.2,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,29.57,> rotate <0,-80.3,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,18.47,> rotate <0,-235.48,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,-52.80,> rotate <0,-192.74,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,20.90,> rotate <0,-187.43,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,-65.08,> rotate <0,-282.16,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,24.80,> rotate <0,-300.43,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,29.08,> rotate <0,-47.4,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,18.20,> rotate <0,-278.8,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,1.18,> rotate <0,-250.43,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,34.93,> rotate <0,-177.43,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,72.20,> rotate <0,-143.73,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,-5.40,> rotate <0,-203.88,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,34.03,> rotate <0,-163.74,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,-47.35,> rotate <0,-56.31,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,-82.18,> rotate <0,-359.39,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,-46.93,> rotate <0,-147.68,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,-40.07,> rotate <0,-204.95,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,45.78,> rotate <0,-305.53,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,-17.58,> rotate <0,-125.48,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,16.43,> rotate <0,-144.26,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,31.80,> rotate <0,-36.87,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,-26.15,> rotate <0,-76.32,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,70.20,> rotate <0,-183.79,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,-28.85,> rotate <0,-345.33,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,14.65,> rotate <0,-95.02,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,13.32,> rotate <0,-309.68,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,33.58,> rotate <0,-62.75,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,2.10,> rotate <0,-126.4,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,80.13,> rotate <0,-259.9,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,49.37,> rotate <0,-218.67,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,14.22,> rotate <0,-115.51,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.72,0,0> rotate <0,0,8.53,> rotate <0,-242.12,0>  pigment {rgb <0.285,0.285,0.285>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,4.22,> rotate <0,-250.16,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,41.07,> rotate <0,-325.59,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,55.40,> rotate <0,-304.6,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,-22.78,> rotate <0,-270.48,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,-60.90,> rotate <0,-229.15,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,-38.15,> rotate <0,-114.93,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,37.58,> rotate <0,-281.2,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,33.97,> rotate <0,-66.15,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,61.53,> rotate <0,-4.24,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,-62.52,> rotate <0,-66.94,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,59.97,> rotate <0,-55.68,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,73.27,> rotate <0,-79.55,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,53.80,> rotate <0,-266,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,-47.43,> rotate <0,-221.62,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,-59.62,> rotate <0,-168.38,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,-12.02,> rotate <0,-275.8,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,16.20,> rotate <0,-102.45,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,-38.25,> rotate <0,-285.82,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,19.47,> rotate <0,-331.87,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,49.30,> rotate <0,-346.94,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,13.02,> rotate <0,-291.6,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,-46.05,> rotate <0,-140.6,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,55.85,> rotate <0,-171.49,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,29.90,> rotate <0,-300.91,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,25.52,> rotate <0,-122.61,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,-68.30,> rotate <0,-185.55,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,32.25,> rotate <0,-271.45,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,45.63,> rotate <0,-134.21,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,-11.28,> rotate <0,-52,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,-60.52,> rotate <0,-119.08,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,68.45,> rotate <0,-160.45,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,15.28,> rotate <0,-33.26,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.73,0,0> rotate <0,0,19.32,> rotate <0,-315.11,0>  pigment {rgb <0.28375,0.28375,0.28375>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-53.27,> rotate <0,-318.94,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-66.58,> rotate <0,-214.16,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,23.12,> rotate <0,-92.43,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-35.12,> rotate <0,-127,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-2.28,> rotate <0,-102.32,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-65.03,> rotate <0,-239.75,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,14.55,> rotate <0,-340.22,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-13.07,> rotate <0,-354.41,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-3.88,> rotate <0,-108.55,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-69.17,> rotate <0,-299.68,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,31.52,> rotate <0,-4.66,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-60.33,> rotate <0,-192.83,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-14.87,> rotate <0,-278.41,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-67.93,> rotate <0,-244.28,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,19.33,> rotate <0,-282.23,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,15.37,> rotate <0,-82.61,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-19.42,> rotate <0,-143.09,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-13.05,> rotate <0,-67.28,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-18.53,> rotate <0,-332.25,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-40.22,> rotate <0,-219.18,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-9.58,> rotate <0,-3.73,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,25.37,> rotate <0,-103.83,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,78.65,> rotate <0,-201.74,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-5.38,> rotate <0,-334.28,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,64.22,> rotate <0,-232.73,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-30.05,> rotate <0,-39.04,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-44.05,> rotate <0,-89.66,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,7.48,> rotate <0,-106.95,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-18.23,> rotate <0,-133.8,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-27.03,> rotate <0,-303.82,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-9.00,> rotate <0,-358.21,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-33.08,> rotate <0,-83.82,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,41.25,> rotate <0,-263.28,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-78.80,> rotate <0,-356.17,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-69.40,> rotate <0,-207.95,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,5.28,> rotate <0,-12.1,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-27.43,> rotate <0,-210.59,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,13.42,> rotate <0,-102.61,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,77.55,> rotate <0,-277.44,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,17.60,> rotate <0,-347.68,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-7.87,> rotate <0,-38.68,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-10.32,> rotate <0,-112.34,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,11.62,> rotate <0,-332.66,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,36.30,> rotate <0,-167.33,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.74,0,0> rotate <0,0,-57.33,> rotate <0,-48.14,0>  pigment {rgb <0.2825,0.2825,0.2825>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-41.00,> rotate <0,-321.1,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-61.95,> rotate <0,-141.36,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-20.42,> rotate <0,-103.76,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,59.33,> rotate <0,-347.43,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,49.97,> rotate <0,-70.84,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-48.18,> rotate <0,-96.43,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-24.47,> rotate <0,-172.41,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,38.23,> rotate <0,-319.84,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,25.25,> rotate <0,-37.63,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-5.43,> rotate <0,-118.2,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,27.08,> rotate <0,-0.54,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-54.98,> rotate <0,-298.16,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-31.67,> rotate <0,-339.15,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,37.30,> rotate <0,-38.91,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,76.23,> rotate <0,-338.07,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-60.48,> rotate <0,-239.03,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,1.07,> rotate <0,-357.36,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,14.12,> rotate <0,-192.23,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-44.32,> rotate <0,-182.22,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-23.83,> rotate <0,-106.85,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-38.27,> rotate <0,-114.95,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-38.92,> rotate <0,-183.35,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,31.77,> rotate <0,-281.99,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-46.90,> rotate <0,-207.95,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,37.73,> rotate <0,-17.79,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-37.15,> rotate <0,-131.22,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,42.57,> rotate <0,-238.88,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,46.27,> rotate <0,-104.13,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-23.50,> rotate <0,-279.63,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,16.95,> rotate <0,-27.04,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-25.00,> rotate <0,-220.81,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-67.08,> rotate <0,-227.38,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-46.15,> rotate <0,-131.83,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-5.15,> rotate <0,-201.14,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,67.20,> rotate <0,-56.5,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,62.08,> rotate <0,-302.9,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-7.02,> rotate <0,-23.43,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-14.55,> rotate <0,-247.45,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,1.93,> rotate <0,-6.35,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-7.65,> rotate <0,-45.29,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,36.47,> rotate <0,-272.5,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-35.97,> rotate <0,-78.62,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,25.40,> rotate <0,-120.23,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-63.93,> rotate <0,-317.14,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-35.13,> rotate <0,-83.28,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-2.33,> rotate <0,-249.09,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-66.95,> rotate <0,-300.1,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-22.78,> rotate <0,-75.69,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-53.23,> rotate <0,-168.42,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,57.90,> rotate <0,-34.5,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-1.32,> rotate <0,-102.08,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,53.42,> rotate <0,-260.44,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.18 translate <10005.75,0,0> rotate <0,0,-31.60,> rotate <0,-310.35,0>  pigment {rgb <0.28125,0.28125,0.28125>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-60.18,> rotate <0,-238.88,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,32.48,> rotate <0,-133.15,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,34.35,> rotate <0,-57.98,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-8.67,> rotate <0,-77.08,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-53.20,> rotate <0,-237.53,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-0.53,> rotate <0,-102.7,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,33.87,> rotate <0,-243.67,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-80.97,> rotate <0,-308.33,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,57.98,> rotate <0,-58.43,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,28.98,> rotate <0,-99.6,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-13.63,> rotate <0,-300.5,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,48.27,> rotate <0,-261.68,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,31.70,> rotate <0,-88.75,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-28.07,> rotate <0,-269.18,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,8.38,> rotate <0,-103.2,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,44.55,> rotate <0,-341.54,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-31.45,> rotate <0,-112.27,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-25.23,> rotate <0,-107.43,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-25.02,> rotate <0,-281.21,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-60.68,> rotate <0,-261.08,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,9.37,> rotate <0,-53.15,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,24.57,> rotate <0,-56.48,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-44.13,> rotate <0,-259.7,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,68.75,> rotate <0,-155.26,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,6.10,> rotate <0,-268.31,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,36.62,> rotate <0,-240.83,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,3.07,> rotate <0,-173.59,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-51.25,> rotate <0,-315.09,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-16.32,> rotate <0,-72.14,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,37.00,> rotate <0,-298.7,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-35.25,> rotate <0,-97.8,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,38.27,> rotate <0,-70.46,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-17.47,> rotate <0,-97.16,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,6.13,> rotate <0,-65.17,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-12.83,> rotate <0,-188.39,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,55.82,> rotate <0,-238.07,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,56.45,> rotate <0,-123.46,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-3.98,> rotate <0,-323.82,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-46.87,> rotate <0,-200.24,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-49.08,> rotate <0,-225.95,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-18.73,> rotate <0,-277.55,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,40.38,> rotate <0,-343.53,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,4.62,> rotate <0,-153.2,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-85.97,> rotate <0,-337.91,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,8.83,> rotate <0,-260.99,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-4.77,> rotate <0,-271.56,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-41.48,> rotate <0,-346,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-41.12,> rotate <0,-348.75,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,55.48,> rotate <0,-236.65,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,-85.00,> rotate <0,-3.33,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,32.70,> rotate <0,-93.08,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.76,0,0> rotate <0,0,19.50,> rotate <0,-32.66,0>  pigment {rgb <0.28,0.28,0.28>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-4.85,> rotate <0,-336.03,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-51.77,> rotate <0,-153.37,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,39.78,> rotate <0,-338.11,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-43.43,> rotate <0,-271.71,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,38.42,> rotate <0,-296.87,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-60.28,> rotate <0,-117.3,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,11.07,> rotate <0,-348.36,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,56.85,> rotate <0,-99.41,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,78.80,> rotate <0,-336.68,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-42.63,> rotate <0,-52.48,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,17.43,> rotate <0,-23.98,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,1.80,> rotate <0,-82.48,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-64.23,> rotate <0,-160.96,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-60.35,> rotate <0,-133.45,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,49.50,> rotate <0,-319.87,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,58.27,> rotate <0,-17.14,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-34.40,> rotate <0,-94.92,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-25.22,> rotate <0,-111.36,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-35.18,> rotate <0,-186.34,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-60.33,> rotate <0,-193.34,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-13.75,> rotate <0,-111.28,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-70.43,> rotate <0,-336.29,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-9.45,> rotate <0,-40.05,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-5.67,> rotate <0,-72.15,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,36.42,> rotate <0,-357.42,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-44.67,> rotate <0,-188.68,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-12.35,> rotate <0,-137.3,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,25.52,> rotate <0,-255.58,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-27.02,> rotate <0,-113.65,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-60.17,> rotate <0,-267.9,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,48.65,> rotate <0,-58.41,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-60.97,> rotate <0,-135.19,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-64.55,> rotate <0,-280.9,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-35.48,> rotate <0,-123.55,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-52.30,> rotate <0,-261.99,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,51.88,> rotate <0,-111.24,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,9.18,> rotate <0,-106.41,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,42.08,> rotate <0,-349.97,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,56.72,> rotate <0,-140.43,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.77,0,0> rotate <0,0,-32.55,> rotate <0,-21.74,0>  pigment {rgb <0.27875,0.27875,0.27875>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,19.97,> rotate <0,-205.17,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-8.90,> rotate <0,-162.58,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-3.80,> rotate <0,-194.91,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-26.50,> rotate <0,-204.2,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,19.82,> rotate <0,-327.89,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,60.55,> rotate <0,-25.83,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-58.77,> rotate <0,-97.99,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,48.87,> rotate <0,-283.7,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-29.27,> rotate <0,-173.07,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-63.82,> rotate <0,-223.15,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,9.52,> rotate <0,-86.72,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-11.55,> rotate <0,-111.97,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-36.87,> rotate <0,-269.73,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,8.18,> rotate <0,-142.12,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,4.75,> rotate <0,-160.84,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-15.83,> rotate <0,-139.89,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-58.35,> rotate <0,-142.6,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-56.78,> rotate <0,-170.79,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,60.82,> rotate <0,-248.11,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-69.27,> rotate <0,-254.89,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-46.92,> rotate <0,-265.32,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-20.35,> rotate <0,-62.9,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-81.57,> rotate <0,-65.25,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,8.43,> rotate <0,-80.43,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-46.58,> rotate <0,-137.89,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-38.32,> rotate <0,-109.13,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-25.78,> rotate <0,-312.32,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,21.70,> rotate <0,-331.96,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,0.48,> rotate <0,-100.27,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,46.70,> rotate <0,-308.48,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,36.93,> rotate <0,-293.95,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-66.18,> rotate <0,-117.42,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-38.90,> rotate <0,-185.94,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,57.18,> rotate <0,-160.93,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-64.58,> rotate <0,-205.05,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,49.38,> rotate <0,-306.76,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,41.40,> rotate <0,-35.71,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,46.80,> rotate <0,-234.57,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-24.98,> rotate <0,-255.04,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-42.23,> rotate <0,-197.79,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,17.07,> rotate <0,-301.03,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,9.87,> rotate <0,-87.51,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-34.57,> rotate <0,-39.24,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-12.37,> rotate <0,-230.97,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,24.77,> rotate <0,-292.24,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-26.35,> rotate <0,-115.7,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-56.53,> rotate <0,-260.78,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,81.43,> rotate <0,-41.95,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-17.08,> rotate <0,-102.59,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-24.25,> rotate <0,-222.33,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-43.52,> rotate <0,-32.29,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-79.50,> rotate <0,-131.48,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-59.42,> rotate <0,-147.8,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,26.67,> rotate <0,-331.3,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-9.03,> rotate <0,-334.22,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,62.80,> rotate <0,-334.55,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,65.75,> rotate <0,-39.4,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,23.62,> rotate <0,-278.19,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-8.57,> rotate <0,-298.53,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,-73.73,> rotate <0,-83.68,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.78,0,0> rotate <0,0,18.97,> rotate <0,-228.02,0>  pigment {rgb <0.2775,0.2775,0.2775>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,-39.87,> rotate <0,-189,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,-30.95,> rotate <0,-101.38,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,-47.55,> rotate <0,-11.44,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,-57.52,> rotate <0,-277.49,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,8.05,> rotate <0,-168.51,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,-21.42,> rotate <0,-224.36,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,63.98,> rotate <0,-305.3,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,10.27,> rotate <0,-183.35,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,86.97,> rotate <0,-262.68,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,-60.27,> rotate <0,-96.06,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,-40.58,> rotate <0,-227.05,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,1.95,> rotate <0,-292.25,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,-35.52,> rotate <0,-91.36,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,67.82,> rotate <0,-41.21,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,-16.92,> rotate <0,-28.22,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,16.03,> rotate <0,-69.54,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,-36.50,> rotate <0,-114.68,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,61.30,> rotate <0,-1.28,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,0.88,> rotate <0,-98.81,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,-50.97,> rotate <0,-128.97,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,-20.08,> rotate <0,-323.71,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,25.78,> rotate <0,-33.94,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,43.17,> rotate <0,-172.63,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,29.93,> rotate <0,-37.2,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,50.42,> rotate <0,-234.64,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,-11.38,> rotate <0,-14.68,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,34.53,> rotate <0,-38.22,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,-27.15,> rotate <0,-92.65,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,26.77,> rotate <0,-337.29,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,-44.38,> rotate <0,-168.31,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,-21.20,> rotate <0,-321.81,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,54.43,> rotate <0,-84.15,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,6.70,> rotate <0,-146.54,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,7.35,> rotate <0,-320.27,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.79,0,0> rotate <0,0,9.32,> rotate <0,-357.84,0>  pigment {rgb <0.27625,0.27625,0.27625>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-68.68,> rotate <0,-152.38,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,27.55,> rotate <0,-192.32,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-10.80,> rotate <0,-277.86,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-32.37,> rotate <0,-97.16,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-45.83,> rotate <0,-123.6,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,44.43,> rotate <0,-354.38,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,38.20,> rotate <0,-140.25,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,54.47,> rotate <0,-256.33,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-53.17,> rotate <0,-171.7,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-32.47,> rotate <0,-120.77,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-22.40,> rotate <0,-281.59,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,39.90,> rotate <0,-53.4,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,3.37,> rotate <0,-113.57,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-29.77,> rotate <0,-91.53,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-2.15,> rotate <0,-128.5,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-32.37,> rotate <0,-167.47,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-26.12,> rotate <0,-206.4,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-0.32,> rotate <0,-279.4,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,60.33,> rotate <0,-9.11,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-56.08,> rotate <0,-122.39,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,20.25,> rotate <0,-111.74,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-21.40,> rotate <0,-279.48,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-45.12,> rotate <0,-300.2,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-60.82,> rotate <0,-119.71,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-20.92,> rotate <0,-94.75,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,56.60,> rotate <0,-328.72,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-3.20,> rotate <0,-68.15,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-40.88,> rotate <0,-106.78,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-43.80,> rotate <0,-165,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,52.20,> rotate <0,-279.97,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-26.65,> rotate <0,-221.99,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-11.62,> rotate <0,-343.37,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,28.17,> rotate <0,-237.14,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,18.33,> rotate <0,-125.84,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-36.05,> rotate <0,-116.04,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-55.55,> rotate <0,-243.35,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,40.20,> rotate <0,-134.13,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-25.00,> rotate <0,-210,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,20.67,> rotate <0,-44.52,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-56.12,> rotate <0,-154.66,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-45.80,> rotate <0,-282.36,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-3.45,> rotate <0,-82.35,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,33.67,> rotate <0,-142.68,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-57.25,> rotate <0,-145.26,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,76.90,> rotate <0,-181.31,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,10.02,> rotate <0,-197.3,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-25.90,> rotate <0,-288.3,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-77.38,> rotate <0,-353.33,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-16.53,> rotate <0,-1.08,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-27.03,> rotate <0,-108.71,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,32.90,> rotate <0,-5.19,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,78.00,> rotate <0,-100.12,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,34.08,> rotate <0,-199.62,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,22.20,> rotate <0,-247.81,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,17.25,> rotate <0,-82,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,8.00,> rotate <0,-108.92,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,39.67,> rotate <0,-279.53,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.80,0,0> rotate <0,0,-15.28,> rotate <0,-355.96,0>  pigment {rgb <0.275,0.275,0.275>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,-46.13,> rotate <0,-216.8,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,83.42,> rotate <0,-192.28,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,33.45,> rotate <0,-272.94,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,9.58,> rotate <0,-263.65,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,46.82,> rotate <0,-356.89,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,-10.13,> rotate <0,-281.68,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,-42.85,> rotate <0,-347.49,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,72.15,> rotate <0,-265.49,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,19.90,> rotate <0,-291.62,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,23.87,> rotate <0,-201.28,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,0.67,> rotate <0,-269.08,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,-61.05,> rotate <0,-174.09,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,-4.52,> rotate <0,-319.55,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,38.65,> rotate <0,-327.12,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,-40.72,> rotate <0,-84.68,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,44.43,> rotate <0,-230.18,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,-5.98,> rotate <0,-110.6,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,41.23,> rotate <0,-155.55,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,-30.28,> rotate <0,-22.93,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,7.73,> rotate <0,-39.15,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,-58.00,> rotate <0,-96.77,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,-14.57,> rotate <0,-139.28,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,46.03,> rotate <0,-201.57,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,-46.60,> rotate <0,-209.08,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,25.10,> rotate <0,-227.15,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,52.43,> rotate <0,-287.11,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,-13.90,> rotate <0,-291.34,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,15.78,> rotate <0,-119.25,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,-20.90,> rotate <0,-57.15,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,56.78,> rotate <0,-186.26,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.17 translate <10005.81,0,0> rotate <0,0,6.18,> rotate <0,-164.01,0>  pigment {rgb <0.27375,0.27375,0.27375>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-4.57,> rotate <0,-93.65,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,5.50,> rotate <0,-281.37,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,47.90,> rotate <0,-27.32,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-41.52,> rotate <0,-218.88,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,50.73,> rotate <0,-314.1,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,23.10,> rotate <0,-154.31,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-50.70,> rotate <0,-346.81,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-12.43,> rotate <0,-206.49,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-67.72,> rotate <0,-217.82,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-35.90,> rotate <0,-271.21,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,24.90,> rotate <0,-108.68,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-41.17,> rotate <0,-253.74,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,25.17,> rotate <0,-351.92,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,1.82,> rotate <0,-285.88,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,21.70,> rotate <0,-76.98,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-16.62,> rotate <0,-94.03,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-66.50,> rotate <0,-52.71,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,48.12,> rotate <0,-52.65,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,25.18,> rotate <0,-42.19,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-44.00,> rotate <0,-109.52,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,86.63,> rotate <0,-62.51,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-9.77,> rotate <0,-282.74,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,54.97,> rotate <0,-294.67,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-29.57,> rotate <0,-204.68,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,70.27,> rotate <0,-22.81,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-30.18,> rotate <0,-39.58,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-40.27,> rotate <0,-318.05,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,43.40,> rotate <0,-290.99,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-9.95,> rotate <0,-45.48,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-3.75,> rotate <0,-93.48,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-20.83,> rotate <0,-183.75,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-0.27,> rotate <0,-290.59,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,72.47,> rotate <0,-264.28,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,49.47,> rotate <0,-169.18,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-23.15,> rotate <0,-234.45,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,4.42,> rotate <0,-240.21,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-46.98,> rotate <0,-45.73,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-23.60,> rotate <0,-44.53,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,11.43,> rotate <0,-171.25,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,15.42,> rotate <0,-172.43,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-24.88,> rotate <0,-65.78,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-50.50,> rotate <0,-117.6,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-38.73,> rotate <0,-133.2,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,-56.52,> rotate <0,-187.48,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,27.78,> rotate <0,-29.43,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.82,0,0> rotate <0,0,15.13,> rotate <0,-221.53,0>  pigment {rgb <0.2725,0.2725,0.2725>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,27.78,> rotate <0,-174.08,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,11.20,> rotate <0,-66.87,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,56.67,> rotate <0,-315.54,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,23.95,> rotate <0,-74.45,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,33.17,> rotate <0,-179.83,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,-7.80,> rotate <0,-280,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,40.73,> rotate <0,-304.53,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,74.23,> rotate <0,-348.66,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,54.97,> rotate <0,-53.41,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,-48.92,> rotate <0,-237.49,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,32.42,> rotate <0,-295.69,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,-48.47,> rotate <0,-122.8,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,15.23,> rotate <0,-9.2,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,-42.87,> rotate <0,-227.16,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,61.77,> rotate <0,-172.27,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,51.93,> rotate <0,-27.74,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,24.13,> rotate <0,-108.11,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,61.60,> rotate <0,-16.08,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,-16.13,> rotate <0,-134.68,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,19.90,> rotate <0,-346.63,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,35.00,> rotate <0,-205.68,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,-5.25,> rotate <0,-2.58,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,66.15,> rotate <0,-10.51,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,-5.47,> rotate <0,-59.72,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,-33.42,> rotate <0,-149.15,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,-12.58,> rotate <0,-2.68,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,20.75,> rotate <0,-125.09,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,6.90,> rotate <0,-39.02,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,-46.15,> rotate <0,-190.35,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,3.02,> rotate <0,-345.18,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,-30.95,> rotate <0,-20.88,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,55.40,> rotate <0,-218.13,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,53.20,> rotate <0,-184.38,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.83,0,0> rotate <0,0,47.20,> rotate <0,-193.74,0>  pigment {rgb <0.27125,0.27125,0.27125>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-59.33,> rotate <0,-162.35,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-40.58,> rotate <0,-339.25,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-26.97,> rotate <0,-110.23,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-11.32,> rotate <0,-25.44,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-73.67,> rotate <0,-273.14,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,22.28,> rotate <0,-27.54,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,42.38,> rotate <0,-242.95,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-13.55,> rotate <0,-12.35,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-43.98,> rotate <0,-199.31,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-24.72,> rotate <0,-113.29,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-28.07,> rotate <0,-225.53,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,42.98,> rotate <0,-91.27,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-46.85,> rotate <0,-117.3,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-7.60,> rotate <0,-65.18,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-44.15,> rotate <0,-138.53,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-8.33,> rotate <0,-271.53,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,14.45,> rotate <0,-224.05,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,2.92,> rotate <0,-292.09,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,34.03,> rotate <0,-250.96,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,25.70,> rotate <0,-206.68,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,53.17,> rotate <0,-301.56,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-30.83,> rotate <0,-177.92,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-78.45,> rotate <0,-198.58,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,51.23,> rotate <0,-293.58,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,46.62,> rotate <0,-249.05,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,10.83,> rotate <0,-314.61,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-11.88,> rotate <0,-223.6,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-34.43,> rotate <0,-268.08,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-82.47,> rotate <0,-74.71,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-47.75,> rotate <0,-49.36,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,71.63,> rotate <0,-353.75,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,56.60,> rotate <0,-178.99,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,54.52,> rotate <0,-234.38,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,8.93,> rotate <0,-149.11,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,25.38,> rotate <0,-154.18,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-4.20,> rotate <0,-75.69,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,7.28,> rotate <0,-135.69,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-28.67,> rotate <0,-306.36,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,-65.82,> rotate <0,-216.78,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.84,0,0> rotate <0,0,33.23,> rotate <0,-270.15,0>  pigment {rgb <0.27,0.27,0.27>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,32.38,> rotate <0,-157.97,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,-34.90,> rotate <0,-72.87,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,-18.10,> rotate <0,-151.01,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,13.00,> rotate <0,-99.95,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,-0.10,> rotate <0,-65.36,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,-75.90,> rotate <0,-272.81,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,6.95,> rotate <0,-246.04,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,60.20,> rotate <0,-225.36,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,63.28,> rotate <0,-57.4,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,62.78,> rotate <0,-191.83,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,41.08,> rotate <0,-206.56,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,16.12,> rotate <0,-234.12,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,-20.97,> rotate <0,-239.42,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,-37.95,> rotate <0,-116.54,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,38.78,> rotate <0,-273.27,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,43.97,> rotate <0,-57.28,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,-41.12,> rotate <0,-242.83,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,-41.37,> rotate <0,-272.77,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,27.90,> rotate <0,-73.2,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,8.87,> rotate <0,-118.88,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,60.32,> rotate <0,-192.16,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,37.92,> rotate <0,-159.78,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,-37.60,> rotate <0,-138.74,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,-7.38,> rotate <0,-54.62,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,78.07,> rotate <0,-205.66,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,38.82,> rotate <0,-259.6,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,-9.85,> rotate <0,-162.43,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,-28.83,> rotate <0,-347.09,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,-25.17,> rotate <0,-345.03,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,33.58,> rotate <0,-7.86,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,52.12,> rotate <0,-109.39,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,-45.93,> rotate <0,-82.9,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,-15.58,> rotate <0,-142.59,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,-7.35,> rotate <0,-13.93,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,17.80,> rotate <0,-305.09,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,63.30,> rotate <0,-333.1,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.85,0,0> rotate <0,0,30.98,> rotate <0,-137.66,0>  pigment {rgb <0.26875,0.26875,0.26875>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-15.33,> rotate <0,-36.5,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-40.95,> rotate <0,-81.78,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,54.10,> rotate <0,-194.08,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,2.07,> rotate <0,-251.79,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,26.10,> rotate <0,-271.96,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,14.28,> rotate <0,-177.16,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,28.27,> rotate <0,-99.89,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,56.07,> rotate <0,-307.36,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-23.47,> rotate <0,-113.58,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-25.82,> rotate <0,-214.75,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,66.10,> rotate <0,-0.65,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-0.50,> rotate <0,-19.95,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-62.83,> rotate <0,-68.4,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,26.47,> rotate <0,-45.48,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,5.17,> rotate <0,-200.54,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,64.27,> rotate <0,-69.1,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-74.73,> rotate <0,-139.36,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-5.83,> rotate <0,-189.2,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,62.72,> rotate <0,-190.39,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,17.33,> rotate <0,-259.52,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,34.70,> rotate <0,-261.69,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-22.37,> rotate <0,-85.56,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-1.25,> rotate <0,-353.53,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-1.52,> rotate <0,-96.67,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,28.63,> rotate <0,-208.29,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-16.03,> rotate <0,-314.42,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-48.68,> rotate <0,-122.29,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,16.45,> rotate <0,-120.38,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,49.88,> rotate <0,-239.77,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-10.23,> rotate <0,-91.36,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-55.02,> rotate <0,-218.38,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-7.55,> rotate <0,-112.36,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-29.82,> rotate <0,-276.96,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-82.97,> rotate <0,-24.48,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-33.15,> rotate <0,-250.44,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,4.77,> rotate <0,-128.43,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,38.50,> rotate <0,-206.58,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-16.62,> rotate <0,-232.67,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-4.57,> rotate <0,-320.27,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,26.62,> rotate <0,-292.84,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-21.80,> rotate <0,-304.5,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,72.68,> rotate <0,-12.04,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-48.95,> rotate <0,-198.68,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-52.68,> rotate <0,-218.38,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-58.02,> rotate <0,-260.73,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,17.13,> rotate <0,-195.29,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-38.42,> rotate <0,-11.05,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-4.62,> rotate <0,-88.88,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-14.08,> rotate <0,-165.05,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-4.82,> rotate <0,-269.9,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,30.52,> rotate <0,-288.85,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,41.77,> rotate <0,-295.94,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,16.67,> rotate <0,-105.64,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,15.27,> rotate <0,-214.37,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.86,0,0> rotate <0,0,-25.08,> rotate <0,-254.99,0>  pigment {rgb <0.2675,0.2675,0.2675>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,2.52,> rotate <0,-236.01,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-44.48,> rotate <0,-349.17,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,39.45,> rotate <0,-341.02,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-45.27,> rotate <0,-293.34,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-31.70,> rotate <0,-162.06,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,56.50,> rotate <0,-65.47,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-22.97,> rotate <0,-87.48,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-37.25,> rotate <0,-94.29,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,21.55,> rotate <0,-288.15,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,64.90,> rotate <0,-30.72,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,38.92,> rotate <0,-157.53,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,14.22,> rotate <0,-127.16,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-41.88,> rotate <0,-287.49,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,30.93,> rotate <0,-5.1,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-37.93,> rotate <0,-116.15,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,26.48,> rotate <0,-303,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,11.88,> rotate <0,-47.66,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,7.55,> rotate <0,-84.5,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,66.35,> rotate <0,-14.63,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,22.63,> rotate <0,-106.33,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-9.78,> rotate <0,-16.94,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,38.73,> rotate <0,-39.24,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-0.93,> rotate <0,-96.32,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,33.90,> rotate <0,-137.21,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,3.10,> rotate <0,-30.04,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,23.50,> rotate <0,-154.13,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,36.28,> rotate <0,-269.68,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-29.45,> rotate <0,-132.51,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-73.00,> rotate <0,-188.04,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-30.42,> rotate <0,-189.76,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-37.17,> rotate <0,-247.06,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,58.17,> rotate <0,-97.7,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,4.87,> rotate <0,-98.08,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,21.25,> rotate <0,-9.84,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,18.28,> rotate <0,-42.14,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-20.05,> rotate <0,-36.65,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-18.23,> rotate <0,-294.26,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-30.65,> rotate <0,-340.1,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-35.72,> rotate <0,-142.89,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,46.68,> rotate <0,-97.51,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-68.68,> rotate <0,-135.29,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,18.47,> rotate <0,-23.7,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-16.35,> rotate <0,-164.88,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,34.67,> rotate <0,-207.79,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-80.87,> rotate <0,-262.87,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.16 translate <10005.87,0,0> rotate <0,0,-25.88,> rotate <0,-110.27,0>  pigment {rgb <0.26625,0.26625,0.26625>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,72.53,> rotate <0,-68.38,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-0.97,> rotate <0,-281.62,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,29.15,> rotate <0,-244.18,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,1.85,> rotate <0,-89.6,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,65.33,> rotate <0,-328.88,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-68.77,> rotate <0,-285.88,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,0.60,> rotate <0,-54.2,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-67.20,> rotate <0,-258.33,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,72.40,> rotate <0,-201.53,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,32.60,> rotate <0,-72.33,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,7.25,> rotate <0,-355.49,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,36.45,> rotate <0,-307.34,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,29.82,> rotate <0,-276.49,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,7.62,> rotate <0,-296.42,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,35.45,> rotate <0,-53.17,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-62.87,> rotate <0,-261,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-53.38,> rotate <0,-208.43,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-57.72,> rotate <0,-256.1,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,10.82,> rotate <0,-325.64,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,15.82,> rotate <0,-86.69,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-69.18,> rotate <0,-180.66,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,64.05,> rotate <0,-46.83,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,13.65,> rotate <0,-92.87,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-42.48,> rotate <0,-253.61,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-12.62,> rotate <0,-303.11,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-9.95,> rotate <0,-299.95,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-44.15,> rotate <0,-193.75,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,46.83,> rotate <0,-42.93,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-28.08,> rotate <0,-45.41,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-1.57,> rotate <0,-338.51,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,57.22,> rotate <0,-334.11,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,2.73,> rotate <0,-109.85,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-48.57,> rotate <0,-134.48,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-73.97,> rotate <0,-278.23,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-11.68,> rotate <0,-355.28,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-17.67,> rotate <0,-35.52,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-33.32,> rotate <0,-191.69,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-68.30,> rotate <0,-231.56,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,51.82,> rotate <0,-265.34,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,53.28,> rotate <0,-315.95,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,55.77,> rotate <0,-114.2,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-18.07,> rotate <0,-11.93,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-4.92,> rotate <0,-200.83,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-0.47,> rotate <0,-229.61,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.88,0,0> rotate <0,0,-14.87,> rotate <0,-278.18,0>  pigment {rgb <0.265,0.265,0.265>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-61.88,> rotate <0,-328.8,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-26.35,> rotate <0,-125.71,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,1.03,> rotate <0,-77.94,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-8.88,> rotate <0,-113.03,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-5.82,> rotate <0,-316.02,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,21.05,> rotate <0,-29.9,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-40.35,> rotate <0,-153.44,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-18.72,> rotate <0,-214.66,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,28.83,> rotate <0,-357.42,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-66.23,> rotate <0,-198.2,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,62.28,> rotate <0,-1.05,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-21.52,> rotate <0,-311.54,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-29.33,> rotate <0,-56.62,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-0.78,> rotate <0,-184.67,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,3.93,> rotate <0,-100.91,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-22.52,> rotate <0,-116.8,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,53.00,> rotate <0,-281.68,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-48.37,> rotate <0,-133.16,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,9.95,> rotate <0,-104.11,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-25.25,> rotate <0,-238.66,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-54.15,> rotate <0,-120.21,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-23.17,> rotate <0,-284,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,47.95,> rotate <0,-4.29,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,20.98,> rotate <0,-65.89,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-23.68,> rotate <0,-242.18,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,67.42,> rotate <0,-161.27,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,71.43,> rotate <0,-316.6,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,57.73,> rotate <0,-125.11,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,18.20,> rotate <0,-60.2,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,12.53,> rotate <0,-312.41,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-2.50,> rotate <0,-77.83,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-59.45,> rotate <0,-10.1,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-66.95,> rotate <0,-173.08,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-44.62,> rotate <0,-158.14,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,-58.68,> rotate <0,-198.55,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,62.75,> rotate <0,-280.23,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,41.03,> rotate <0,-44.92,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,72.87,> rotate <0,-149.6,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,22.18,> rotate <0,-92.39,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,53.93,> rotate <0,-233.82,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.89,0,0> rotate <0,0,43.20,> rotate <0,-167.41,0>  pigment {rgb <0.26375,0.26375,0.26375>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-53.62,> rotate <0,-267.62,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-69.95,> rotate <0,-197.96,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,39.40,> rotate <0,-75.08,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,0.73,> rotate <0,-163.93,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-37.87,> rotate <0,-241.15,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-62.07,> rotate <0,-69.19,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,13.90,> rotate <0,-284.69,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-47.97,> rotate <0,-344.2,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-66.68,> rotate <0,-142.13,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-22.87,> rotate <0,-143.36,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-15.67,> rotate <0,-286.43,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-8.40,> rotate <0,-152.53,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-84.80,> rotate <0,-227.79,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-63.27,> rotate <0,-178.75,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,45.77,> rotate <0,-299.83,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,48.77,> rotate <0,-114.48,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-50.25,> rotate <0,-206.86,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-40.03,> rotate <0,-293.53,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,1.05,> rotate <0,-209.12,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,41.78,> rotate <0,-349.6,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,21.13,> rotate <0,-346.87,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-14.75,> rotate <0,-326,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-50.77,> rotate <0,-164.28,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-48.92,> rotate <0,-130.27,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-66.58,> rotate <0,-222.18,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,19.75,> rotate <0,-88.73,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-57.25,> rotate <0,-159.51,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-25.95,> rotate <0,-108.05,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-34.13,> rotate <0,-111.68,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-51.55,> rotate <0,-139.68,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-68.30,> rotate <0,-250.35,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-46.43,> rotate <0,-204.35,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-34.80,> rotate <0,-268.05,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,78.12,> rotate <0,-316.37,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-28.78,> rotate <0,-49.51,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,14.73,> rotate <0,-67.1,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,26.92,> rotate <0,-250.4,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,60.52,> rotate <0,-310.07,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,45.57,> rotate <0,-304,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-0.83,> rotate <0,-213.42,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-70.15,> rotate <0,-183.2,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,7.13,> rotate <0,-109.95,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,5.87,> rotate <0,-114.14,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-69.42,> rotate <0,-190.52,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-63.78,> rotate <0,-240.3,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,27.08,> rotate <0,-49.98,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,3.68,> rotate <0,-27.11,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-4.35,> rotate <0,-10.18,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-6.05,> rotate <0,-78.2,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,48.22,> rotate <0,-48.35,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-41.17,> rotate <0,-33.63,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,46.68,> rotate <0,-104.24,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-50.05,> rotate <0,-264.36,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,23.67,> rotate <0,-309.65,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,24.43,> rotate <0,-92.88,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,1.65,> rotate <0,-169.73,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-34.65,> rotate <0,-38.28,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-11.92,> rotate <0,-356.82,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,38.02,> rotate <0,-128.23,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,18.30,> rotate <0,-219.56,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,65.47,> rotate <0,-116.67,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-9.93,> rotate <0,-108.56,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,28.62,> rotate <0,-333.41,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,25.47,> rotate <0,-17.58,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,9.98,> rotate <0,-98.83,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,60.67,> rotate <0,-231.96,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,3.45,> rotate <0,-242.25,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,8.57,> rotate <0,-300.25,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,-81.30,> rotate <0,-309.58,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,75.37,> rotate <0,-339.31,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.90,0,0> rotate <0,0,51.43,> rotate <0,-4.43,0>  pigment {rgb <0.2625,0.2625,0.2625>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,57.13,> rotate <0,-48.95,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,13.85,> rotate <0,-93.79,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-6.92,> rotate <0,-21.09,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,24.37,> rotate <0,-222.1,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-35.83,> rotate <0,-42.56,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-35.00,> rotate <0,-259.74,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-11.15,> rotate <0,-224.73,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,31.28,> rotate <0,-145.4,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-53.18,> rotate <0,-293.23,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-34.78,> rotate <0,-268.44,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,69.35,> rotate <0,-296.08,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,21.82,> rotate <0,-309.79,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,17.27,> rotate <0,-236.18,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-1.18,> rotate <0,-201.55,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-51.10,> rotate <0,-35.73,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-37.85,> rotate <0,-18.19,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,37.47,> rotate <0,-305.93,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-47.35,> rotate <0,-138.4,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,24.02,> rotate <0,-125.13,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,22.27,> rotate <0,-218.14,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-38.15,> rotate <0,-255.96,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,37.08,> rotate <0,-343.76,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-31.80,> rotate <0,-101.34,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-38.43,> rotate <0,-43.39,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-15.32,> rotate <0,-30.75,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-20.42,> rotate <0,-253.35,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-21.05,> rotate <0,-279.73,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-23.15,> rotate <0,-221.53,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,21.15,> rotate <0,-245.02,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-16.13,> rotate <0,-310.14,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-55.15,> rotate <0,-245.1,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,49.20,> rotate <0,-314.1,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,6.82,> rotate <0,-56.54,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-20.28,> rotate <0,-93.78,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,71.75,> rotate <0,-100.13,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-62.27,> rotate <0,-142.2,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-60.52,> rotate <0,-163.83,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,-17.92,> rotate <0,-313.7,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,52.72,> rotate <0,-129.82,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,32.80,> rotate <0,-129.58,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.91,0,0> rotate <0,0,0.22,> rotate <0,-57.16,0>  pigment {rgb <0.26125,0.26125,0.26125>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-37.22,> rotate <0,-249.78,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,10.90,> rotate <0,-307.58,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-13.22,> rotate <0,-203.67,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,35.12,> rotate <0,-199.77,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,21.18,> rotate <0,-147.46,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,61.07,> rotate <0,-15.9,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,15.83,> rotate <0,-309.77,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-45.08,> rotate <0,-91.17,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-44.43,> rotate <0,-48.11,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,69.85,> rotate <0,-163.38,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-3.17,> rotate <0,-358.23,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,17.33,> rotate <0,-62,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-61.83,> rotate <0,-163.63,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,13.55,> rotate <0,-305,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-30.48,> rotate <0,-60.17,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-29.72,> rotate <0,-261.9,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-26.88,> rotate <0,-315.44,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,37.52,> rotate <0,-192.55,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,0.33,> rotate <0,-85.27,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-15.58,> rotate <0,-264.4,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-51.13,> rotate <0,-148.46,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-8.50,> rotate <0,-71.02,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,8.60,> rotate <0,-101.63,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-26.82,> rotate <0,-331.15,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,13.40,> rotate <0,-62.25,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,19.87,> rotate <0,-88.1,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-62.95,> rotate <0,-183.09,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,41.43,> rotate <0,-281.55,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,12.57,> rotate <0,-95.65,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-31.87,> rotate <0,-142.89,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-0.50,> rotate <0,-8.89,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,37.80,> rotate <0,-222.28,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,61.48,> rotate <0,-207.44,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,49.93,> rotate <0,-89.84,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,57.42,> rotate <0,-149.3,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,36.03,> rotate <0,-315.3,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-7.20,> rotate <0,-335.88,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-13.53,> rotate <0,-336.11,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,29.33,> rotate <0,-105.88,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,7.30,> rotate <0,-12.07,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-37.22,> rotate <0,-260.73,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-1.83,> rotate <0,-32.84,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,0.72,> rotate <0,-75.46,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-1.47,> rotate <0,-83.52,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-10.08,> rotate <0,-97.55,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-72.20,> rotate <0,-194.13,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-77.15,> rotate <0,-225.05,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-10.93,> rotate <0,-264.54,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-56.22,> rotate <0,-210.86,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,16.40,> rotate <0,-24.92,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,38.62,> rotate <0,-44.51,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-54.97,> rotate <0,-193.27,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-2.25,> rotate <0,-17.93,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-28.47,> rotate <0,-178.92,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-14.40,> rotate <0,-357.64,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-64.37,> rotate <0,-21.27,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,4.57,> rotate <0,-224.85,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.92,0,0> rotate <0,0,-1.97,> rotate <0,-162.17,0>  pigment {rgb <0.26,0.26,0.26>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-41.82,> rotate <0,-236.1,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,59.37,> rotate <0,-95.52,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-22.52,> rotate <0,-1.94,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,24.47,> rotate <0,-53.61,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-18.73,> rotate <0,-203.01,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,41.33,> rotate <0,-15.73,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,36.67,> rotate <0,-321.45,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-16.23,> rotate <0,-286.72,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-12.35,> rotate <0,-171.79,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-26.20,> rotate <0,-22.6,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-51.03,> rotate <0,-285.99,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,24.63,> rotate <0,-10.4,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-14.58,> rotate <0,-277.45,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-1.00,> rotate <0,-277.99,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-50.20,> rotate <0,-27.73,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-16.87,> rotate <0,-99.65,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-52.12,> rotate <0,-125.73,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,64.33,> rotate <0,-170.71,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,31.28,> rotate <0,-287.94,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-37.00,> rotate <0,-214.85,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,22.30,> rotate <0,-267.7,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,0.82,> rotate <0,-217.46,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,18.37,> rotate <0,-22.47,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-42.67,> rotate <0,-11.24,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,13.53,> rotate <0,-220.18,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,41.95,> rotate <0,-342.59,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-27.77,> rotate <0,-97.7,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-34.32,> rotate <0,-91.77,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,18.13,> rotate <0,-274.53,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-60.42,> rotate <0,-150.5,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,42.55,> rotate <0,-185.95,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-57.15,> rotate <0,-88.09,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,33.68,> rotate <0,-104.25,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-27.32,> rotate <0,-51.59,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,6.20,> rotate <0,-52.69,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,64.62,> rotate <0,-255.56,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,66.85,> rotate <0,-304.38,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,82.75,> rotate <0,-205.6,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-31.77,> rotate <0,-75.6,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,54.90,> rotate <0,-8.29,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-36.53,> rotate <0,-24.61,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,37.95,> rotate <0,-27.16,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-38.53,> rotate <0,-315.61,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-4.72,> rotate <0,-345.38,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,21.88,> rotate <0,-188.78,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-46.88,> rotate <0,-58.39,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,16.83,> rotate <0,-197.45,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-10.33,> rotate <0,-197.44,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,-57.92,> rotate <0,-253.5,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,13.48,> rotate <0,-115.47,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,57.63,> rotate <0,-290.07,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,22.18,> rotate <0,-322.25,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.15 translate <10005.93,0,0> rotate <0,0,62.47,> rotate <0,-326.22,0>  pigment {rgb <0.25875,0.25875,0.25875>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,81.47,> rotate <0,-47.93,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,27.57,> rotate <0,-52.83,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-15.82,> rotate <0,-269.08,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,74.07,> rotate <0,-76.17,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,70.53,> rotate <0,-97.06,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-18.58,> rotate <0,-307.48,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,26.67,> rotate <0,-218.09,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-47.13,> rotate <0,-208.49,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,16.80,> rotate <0,-173.68,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-19.38,> rotate <0,-238.75,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-8.40,> rotate <0,-105.1,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-76.12,> rotate <0,-330.76,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-33.28,> rotate <0,-223.66,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,34.38,> rotate <0,-81.7,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-11.60,> rotate <0,-186.3,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-76.08,> rotate <0,-235.48,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-52.28,> rotate <0,-254.12,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,63.53,> rotate <0,-299.62,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,24.22,> rotate <0,-114.8,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-16.68,> rotate <0,-184.26,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,8.25,> rotate <0,-216,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,12.38,> rotate <0,-250.21,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,12.03,> rotate <0,-275.65,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,75.28,> rotate <0,-354.8,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,9.65,> rotate <0,-129.28,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-63.50,> rotate <0,-187.98,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-44.17,> rotate <0,-127.28,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-58.50,> rotate <0,-251.59,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,66.77,> rotate <0,-356.65,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-60.38,> rotate <0,-193.46,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-27.87,> rotate <0,-291.13,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-41.42,> rotate <0,-108.74,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-56.92,> rotate <0,-262.85,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,31.15,> rotate <0,-81.16,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,11.68,> rotate <0,-90.85,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-23.60,> rotate <0,-241.53,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,51.70,> rotate <0,-323.62,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,11.05,> rotate <0,-27.72,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-58.57,> rotate <0,-156.25,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-30.40,> rotate <0,-257.2,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,25.53,> rotate <0,-332.07,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-27.10,> rotate <0,-337.44,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,5.97,> rotate <0,-316.36,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-59.90,> rotate <0,-234.99,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,1.17,> rotate <0,-86.65,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-36.03,> rotate <0,-271.6,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,22.00,> rotate <0,-137.59,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,6.32,> rotate <0,-265.38,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,61.48,> rotate <0,-99.42,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-6.47,> rotate <0,-64.33,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-52.65,> rotate <0,-112.5,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-21.02,> rotate <0,-234.57,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,33.97,> rotate <0,-14.56,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-22.38,> rotate <0,-43.4,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-16.70,> rotate <0,-134.15,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-26.33,> rotate <0,-227.58,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-38.28,> rotate <0,-221.77,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,56.68,> rotate <0,-254.84,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-80.23,> rotate <0,-277.33,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-26.03,> rotate <0,-3.43,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-3.10,> rotate <0,-162.77,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,42.10,> rotate <0,-202.82,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-82.67,> rotate <0,-208.91,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,37.38,> rotate <0,-152.8,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,1.90,> rotate <0,-182.42,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,0.00,> rotate <0,-165.9,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-7.83,> rotate <0,-173.2,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-61.23,> rotate <0,-66.27,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,11.03,> rotate <0,-84.27,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-54.40,> rotate <0,-112.63,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-13.52,> rotate <0,-143.23,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,1.32,> rotate <0,-268.15,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,41.02,> rotate <0,-305.69,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,5.97,> rotate <0,-148.43,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,-40.05,> rotate <0,-233.51,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.94,0,0> rotate <0,0,18.33,> rotate <0,-43.95,0>  pigment {rgb <0.2575,0.2575,0.2575>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-38.73,> rotate <0,-278.35,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,13.27,> rotate <0,-252.4,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,59.97,> rotate <0,-7.58,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-18.13,> rotate <0,-79.71,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,3.60,> rotate <0,-104.74,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-35.65,> rotate <0,-67.67,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,31.03,> rotate <0,-90.29,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,49.90,> rotate <0,-90.7,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,3.63,> rotate <0,-115.4,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-41.67,> rotate <0,-154.62,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-9.05,> rotate <0,-88.03,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-26.85,> rotate <0,-128.87,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,72.95,> rotate <0,-138.97,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,32.62,> rotate <0,-320.34,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,39.30,> rotate <0,-49.44,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-37.28,> rotate <0,-120.4,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,35.53,> rotate <0,-133.48,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-27.88,> rotate <0,-84.32,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-6.58,> rotate <0,-84.66,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,29.22,> rotate <0,-84.83,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-28.77,> rotate <0,-269.66,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-38.85,> rotate <0,-128.41,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,16.13,> rotate <0,-18.53,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,62.03,> rotate <0,-230.65,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,35.08,> rotate <0,-145.68,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,47.00,> rotate <0,-289.21,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-69.33,> rotate <0,-51.4,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,43.53,> rotate <0,-347.61,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-36.18,> rotate <0,-239.34,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,22.00,> rotate <0,-61.33,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,11.63,> rotate <0,-298.02,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,67.30,> rotate <0,-127.44,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-25.55,> rotate <0,-6.81,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-67.62,> rotate <0,-174.45,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,43.45,> rotate <0,-310.02,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,47.00,> rotate <0,-120.01,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,6.38,> rotate <0,-242.3,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,25.38,> rotate <0,-296.95,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,54.87,> rotate <0,-325.18,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,28.33,> rotate <0,-133.15,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-19.45,> rotate <0,-67.16,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-5.63,> rotate <0,-54.75,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,4.43,> rotate <0,-87.55,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-42.40,> rotate <0,-119.24,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,2.45,> rotate <0,-17.64,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,52.92,> rotate <0,-240.52,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-68.43,> rotate <0,-292.8,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,60.63,> rotate <0,-303.36,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-53.97,> rotate <0,-175.18,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-7.55,> rotate <0,-125.73,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-56.37,> rotate <0,-185.49,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,-48.88,> rotate <0,-257.91,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.95,0,0> rotate <0,0,44.93,> rotate <0,-314.15,0>  pigment {rgb <0.25625,0.25625,0.25625>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,42.43,> rotate <0,-65.9,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-28.50,> rotate <0,-251.25,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,0.07,> rotate <0,-269.27,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,32.82,> rotate <0,-282.44,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-36.55,> rotate <0,-108.11,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-66.27,> rotate <0,-210.22,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,77.52,> rotate <0,-250.78,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-6.80,> rotate <0,-85.72,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-22.17,> rotate <0,-185.05,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,50.53,> rotate <0,-295.45,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-38.60,> rotate <0,-258.9,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,33.97,> rotate <0,-283.72,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-21.45,> rotate <0,-261.18,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-40.78,> rotate <0,-267.89,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,47.20,> rotate <0,-232.18,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,20.60,> rotate <0,-307.74,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,9.67,> rotate <0,-354.98,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-4.07,> rotate <0,-155.86,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,81.30,> rotate <0,-32.35,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-34.47,> rotate <0,-268.61,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,22.93,> rotate <0,-110.87,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-36.22,> rotate <0,-102.93,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,9.13,> rotate <0,-105.83,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,79.60,> rotate <0,-290.42,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-62.97,> rotate <0,-359.33,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,64.23,> rotate <0,-157.61,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,27.90,> rotate <0,-135.45,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,12.30,> rotate <0,-29.86,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-79.78,> rotate <0,-160.46,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-39.68,> rotate <0,-192.98,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,63.62,> rotate <0,-195.45,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,54.13,> rotate <0,-117.78,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,8.38,> rotate <0,-44.06,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-12.48,> rotate <0,-76.85,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,39.27,> rotate <0,-191.25,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-5.78,> rotate <0,-111.46,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-4.08,> rotate <0,-87.15,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,11.52,> rotate <0,-89.2,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-38.15,> rotate <0,-99.26,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-64.60,> rotate <0,-126.47,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,31.02,> rotate <0,-205.07,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,74.00,> rotate <0,-354.84,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-13.30,> rotate <0,-334.75,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-12.63,> rotate <0,-124.6,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-1.58,> rotate <0,-190.91,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,17.65,> rotate <0,-233.89,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-77.43,> rotate <0,-5.37,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,41.47,> rotate <0,-82.7,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-22.83,> rotate <0,-89.06,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-52.50,> rotate <0,-108.84,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-36.10,> rotate <0,-144.37,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,53.85,> rotate <0,-340.08,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,14.52,> rotate <0,-340.49,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-37.18,> rotate <0,-147.37,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,54.28,> rotate <0,-272.63,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,23.75,> rotate <0,-258.92,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-57.43,> rotate <0,-340.2,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,18.52,> rotate <0,-300.82,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-18.07,> rotate <0,-349.85,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,43.22,> rotate <0,-251.3,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-13.32,> rotate <0,-99.19,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,-60.95,> rotate <0,-181.24,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.96,0,0> rotate <0,0,23.48,> rotate <0,-242.91,0>  pigment {rgb <0.255,0.255,0.255>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,-18.50,> rotate <0,-168.13,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,-51.52,> rotate <0,-212.4,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,53.53,> rotate <0,-40.75,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,-5.90,> rotate <0,-282.42,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,-58.82,> rotate <0,-159.75,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,65.50,> rotate <0,-61.66,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,16.25,> rotate <0,-177.31,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,20.93,> rotate <0,-56.12,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,-14.93,> rotate <0,-126.48,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,81.23,> rotate <0,-318.34,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,27.22,> rotate <0,-14,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,23.17,> rotate <0,-59.27,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,34.00,> rotate <0,-115.68,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,36.62,> rotate <0,-2.17,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,17.33,> rotate <0,-58.29,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,19.05,> rotate <0,-66.24,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,2.03,> rotate <0,-87.63,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,-52.18,> rotate <0,-95.91,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,-32.83,> rotate <0,-173.63,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,-44.15,> rotate <0,-204.28,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,-38.88,> rotate <0,-217.8,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,-53.15,> rotate <0,-251.16,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,-20.87,> rotate <0,-345.68,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,19.33,> rotate <0,-117.99,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,68.75,> rotate <0,-273.82,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,-41.72,> rotate <0,-152.66,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.97,0,0> rotate <0,0,-14.88,> rotate <0,-244.75,0>  pigment {rgb <0.25375,0.25375,0.25375>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-4.92,> rotate <0,-93.88,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,7.52,> rotate <0,-315.02,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,38.25,> rotate <0,-166.13,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-42.78,> rotate <0,-126.24,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-70.72,> rotate <0,-163.43,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-59.87,> rotate <0,-196.85,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-42.07,> rotate <0,-205.92,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,36.10,> rotate <0,-296.41,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,38.50,> rotate <0,-197.51,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-25.73,> rotate <0,-65.38,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-5.63,> rotate <0,-312.86,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,21.15,> rotate <0,-196.59,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-13.02,> rotate <0,-190.32,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-3.37,> rotate <0,-195.15,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,37.27,> rotate <0,-28.98,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-20.38,> rotate <0,-273.8,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-27.37,> rotate <0,-79.85,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-19.48,> rotate <0,-200.38,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,73.38,> rotate <0,-243.64,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,48.23,> rotate <0,-301.12,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-47.38,> rotate <0,-52.65,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,8.88,> rotate <0,-210.33,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,75.95,> rotate <0,-72.21,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-42.08,> rotate <0,-132.59,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,6.72,> rotate <0,-330.03,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,0.48,> rotate <0,-73.71,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-45.22,> rotate <0,-119.75,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,42.38,> rotate <0,-48.04,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,0.18,> rotate <0,-110.52,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,52.90,> rotate <0,-321.69,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,9.50,> rotate <0,-89.12,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-56.18,> rotate <0,-190.79,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,12.13,> rotate <0,-322.79,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,47.72,> rotate <0,-84.07,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,43.92,> rotate <0,-103.81,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,65.48,> rotate <0,-279.05,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,63.18,> rotate <0,-217.7,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,42.65,> rotate <0,-359.26,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-64.58,> rotate <0,-169.82,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,23.52,> rotate <0,-282.07,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-6.82,> rotate <0,-52.41,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-51.97,> rotate <0,-106.8,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,34.63,> rotate <0,-138.81,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,80.52,> rotate <0,-321.21,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-8.83,> rotate <0,-2.07,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-61.65,> rotate <0,-141.03,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,6.75,> rotate <0,-11.85,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-45.20,> rotate <0,-288.66,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,30.03,> rotate <0,-25.41,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,48.40,> rotate <0,-154.86,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-41.40,> rotate <0,-205.73,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-53.17,> rotate <0,-215.95,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,81.08,> rotate <0,-307.37,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,16.05,> rotate <0,-332.97,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-0.68,> rotate <0,-302.01,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-38.07,> rotate <0,-225.3,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,13.78,> rotate <0,-237.06,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-17.37,> rotate <0,-274.3,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-60.27,> rotate <0,-10.67,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-53.68,> rotate <0,-243.32,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-9.32,> rotate <0,-320.74,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,33.92,> rotate <0,-88.17,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,42.23,> rotate <0,-251.83,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.98,0,0> rotate <0,0,-24.37,> rotate <0,-270.97,0>  pigment {rgb <0.2525,0.2525,0.2525>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,49.75,> rotate <0,-257.92,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-33.95,> rotate <0,-323.06,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-54.52,> rotate <0,-120.35,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-8.40,> rotate <0,-80.83,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,25.70,> rotate <0,-31.7,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,6.62,> rotate <0,-128.96,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-48.65,> rotate <0,-255.11,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,57.52,> rotate <0,-34.52,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-39.82,> rotate <0,-94.14,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,62.42,> rotate <0,-335.75,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-11.65,> rotate <0,-193.58,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-58.67,> rotate <0,-158.2,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,19.75,> rotate <0,-161.06,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,34.80,> rotate <0,-255.98,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-3.57,> rotate <0,-84.88,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-14.40,> rotate <0,-325.77,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-18.25,> rotate <0,-189.68,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-12.10,> rotate <0,-58.82,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-32.43,> rotate <0,-258.25,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-37.58,> rotate <0,-114.99,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-55.20,> rotate <0,-145.45,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-39.27,> rotate <0,-137.92,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,55.93,> rotate <0,-56.88,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-9.75,> rotate <0,-127.21,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-76.18,> rotate <0,-310.51,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,14.77,> rotate <0,-326.13,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,19.62,> rotate <0,-332.58,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-5.45,> rotate <0,-133.58,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,26.63,> rotate <0,-137.2,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-38.38,> rotate <0,-40.53,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-19.05,> rotate <0,-298.05,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,40.32,> rotate <0,-71.68,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,40.58,> rotate <0,-46.34,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,16.52,> rotate <0,-123.25,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,75.47,> rotate <0,-194.7,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-57.75,> rotate <0,-247.44,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,24.97,> rotate <0,-333.03,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-76.87,> rotate <0,-354.6,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-12.55,> rotate <0,-312.68,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,68.68,> rotate <0,-61.52,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,-37.18,> rotate <0,-175.86,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,31.25,> rotate <0,-305.65,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,5.65,> rotate <0,-16.22,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,62.70,> rotate <0,-329.73,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.14 translate <10005.99,0,0> rotate <0,0,68.88,> rotate <0,-305.02,0>  pigment {rgb <0.25125,0.25125,0.25125>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,42.08,> rotate <0,-1.15,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,64.15,> rotate <0,-84.31,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,28.03,> rotate <0,-77.44,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-12.53,> rotate <0,-154.82,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,1.85,> rotate <0,-28.98,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-47.20,> rotate <0,-340.65,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-33.57,> rotate <0,-121.44,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-43.92,> rotate <0,-245.62,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-0.33,> rotate <0,-30.95,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,22.82,> rotate <0,-52.11,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-55.17,> rotate <0,-262.16,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,33.25,> rotate <0,-117.76,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,60.33,> rotate <0,-120.43,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,23.33,> rotate <0,-96.39,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-65.72,> rotate <0,-156.85,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-59.92,> rotate <0,-162.02,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-72.77,> rotate <0,-228.14,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,19.25,> rotate <0,-267.2,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-27.93,> rotate <0,-54.7,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-8.18,> rotate <0,-115.15,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,41.60,> rotate <0,-156.87,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,12.17,> rotate <0,-208.08,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,0.27,> rotate <0,-37.69,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,48.77,> rotate <0,-282.07,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-14.48,> rotate <0,-314.68,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-4.23,> rotate <0,-147.81,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-10.30,> rotate <0,-108.62,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,57.40,> rotate <0,-358.89,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,36.25,> rotate <0,-299.18,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,36.53,> rotate <0,-282.9,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-71.02,> rotate <0,-326.37,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-63.78,> rotate <0,-186.85,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-14.80,> rotate <0,-74.9,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-58.13,> rotate <0,-23.81,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-10.05,> rotate <0,-32.84,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-4.70,> rotate <0,-126.15,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,23.52,> rotate <0,-21.4,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,68.88,> rotate <0,-43,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,43.47,> rotate <0,-21.58,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,0.70,> rotate <0,-256.32,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,27.13,> rotate <0,-153.46,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,52.62,> rotate <0,-322.87,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,23.97,> rotate <0,-94.08,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,16.03,> rotate <0,-235.25,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,9.42,> rotate <0,-68.45,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-54.70,> rotate <0,-209.14,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-3.00,> rotate <0,-274.22,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-37.70,> rotate <0,-300.07,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,37.42,> rotate <0,-342.05,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,0.97,> rotate <0,-88.69,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-50.58,> rotate <0,-121.18,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,5.48,> rotate <0,-207.6,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,11.25,> rotate <0,-233.97,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-66.65,> rotate <0,-285.01,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,20.20,> rotate <0,-288.76,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-4.28,> rotate <0,-332.64,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,38.67,> rotate <0,-352.67,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-44.85,> rotate <0,-352.86,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-44.80,> rotate <0,-9.97,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,45.30,> rotate <0,-133.05,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,19.15,> rotate <0,-223.35,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-47.05,> rotate <0,-237.88,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,47.40,> rotate <0,-253.33,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,37.43,> rotate <0,-13.37,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-1.45,> rotate <0,-141.59,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-14.25,> rotate <0,-358.13,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,34.02,> rotate <0,-106.84,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,37.42,> rotate <0,-197.41,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,29.67,> rotate <0,-274.96,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,77.00,> rotate <0,-318.93,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-50.20,> rotate <0,-123.39,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,17.73,> rotate <0,-105.6,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-47.55,> rotate <0,-297.55,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,36.10,> rotate <0,-164.89,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-9.27,> rotate <0,-326.57,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-20.72,> rotate <0,-64.57,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,2.38,> rotate <0,-274.03,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-24.20,> rotate <0,-91.44,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,11.73,> rotate <0,-124.14,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-7.20,> rotate <0,-203.25,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.00,0,0> rotate <0,0,-49.95,> rotate <0,-205,0>  pigment {rgb <0.25,0.25,0.25>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-63.18,> rotate <0,-108.01,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,5.23,> rotate <0,-115.03,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,79.72,> rotate <0,-2.33,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,8.82,> rotate <0,-3.75,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,15.17,> rotate <0,-61.93,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,69.92,> rotate <0,-338.26,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,20.85,> rotate <0,-335.92,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-37.92,> rotate <0,-236.87,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-58.67,> rotate <0,-166.62,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-57.63,> rotate <0,-204.7,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-16.35,> rotate <0,-159.07,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,24.25,> rotate <0,-89.24,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-30.92,> rotate <0,-226.64,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,59.72,> rotate <0,-298.4,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,2.45,> rotate <0,-148.05,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-21.32,> rotate <0,-292.73,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,56.10,> rotate <0,-86.63,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,19.58,> rotate <0,-11.8,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,36.63,> rotate <0,-318.36,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-55.82,> rotate <0,-200.2,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,61.68,> rotate <0,-29.64,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-58.38,> rotate <0,-138.82,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-26.28,> rotate <0,-262.94,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,55.42,> rotate <0,-322.22,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-58.93,> rotate <0,-105.27,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-21.05,> rotate <0,-126.33,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-33.50,> rotate <0,-195.14,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,29.67,> rotate <0,-16.11,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,15.05,> rotate <0,-38.23,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-40.30,> rotate <0,-358.66,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,35.38,> rotate <0,-91.53,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-74.83,> rotate <0,-212.12,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,47.35,> rotate <0,-358.89,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-11.23,> rotate <0,-108.28,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,33.82,> rotate <0,-285.23,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,42.00,> rotate <0,-133.04,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,38.05,> rotate <0,-104.76,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,18.73,> rotate <0,-71.57,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,4.77,> rotate <0,-217.69,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,14.58,> rotate <0,-310.32,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,3.05,> rotate <0,-192.9,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,85.37,> rotate <0,-342.76,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.01,0,0> rotate <0,0,-15.02,> rotate <0,-94.7,0>  pigment {rgb <0.24875,0.24875,0.24875>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,18.45,> rotate <0,-226.83,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-16.23,> rotate <0,-107.39,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-46.27,> rotate <0,-123.13,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-43.85,> rotate <0,-118.95,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-68.62,> rotate <0,-81.75,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,8.45,> rotate <0,-128.55,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,37.58,> rotate <0,-339.89,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,11.68,> rotate <0,-98.1,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-50.77,> rotate <0,-215.83,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-37.75,> rotate <0,-171.39,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,23.48,> rotate <0,-37.3,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,18.92,> rotate <0,-214.02,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-56.65,> rotate <0,-133.27,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-67.43,> rotate <0,-250.84,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-55.88,> rotate <0,-330.1,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,17.20,> rotate <0,-242.19,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-48.97,> rotate <0,-120.06,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,16.70,> rotate <0,-80.91,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,25.32,> rotate <0,-55.2,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-10.53,> rotate <0,-86.86,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,11.53,> rotate <0,-289.97,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-71.47,> rotate <0,-195.77,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-43.77,> rotate <0,-116.33,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,44.85,> rotate <0,-11.54,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,52.05,> rotate <0,-299.81,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,1.02,> rotate <0,-251.38,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,58.97,> rotate <0,-88.07,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,4.70,> rotate <0,-41.34,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,43.07,> rotate <0,-68.35,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-14.18,> rotate <0,-333.1,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,26.08,> rotate <0,-104.7,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,15.85,> rotate <0,-67.97,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,63.10,> rotate <0,-120.63,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,41.80,> rotate <0,-65.06,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-39.72,> rotate <0,-75.4,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-28.53,> rotate <0,-103.39,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-21.00,> rotate <0,-33.25,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,7.00,> rotate <0,-18.68,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-38.80,> rotate <0,-227.53,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,37.30,> rotate <0,-263.93,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-18.32,> rotate <0,-290.46,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-62.90,> rotate <0,-43.08,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,72.13,> rotate <0,-63.44,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,31.85,> rotate <0,-336.94,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,26.90,> rotate <0,-47.61,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,12.58,> rotate <0,-105.96,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,10.88,> rotate <0,-45.18,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,49.05,> rotate <0,-244.8,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.02,0,0> rotate <0,0,-41.73,> rotate <0,-187.49,0>  pigment {rgb <0.2475,0.2475,0.2475>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,9.75,> rotate <0,-156.91,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-21.98,> rotate <0,-111.33,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-44.90,> rotate <0,-139.02,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,40.68,> rotate <0,-284.69,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,16.47,> rotate <0,-272.54,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-15.00,> rotate <0,-112.34,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,42.92,> rotate <0,-165.09,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-75.02,> rotate <0,-336.46,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-19.23,> rotate <0,-285.77,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,19.38,> rotate <0,-321.61,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,22.27,> rotate <0,-3.73,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-12.45,> rotate <0,-159.72,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-64.33,> rotate <0,-187.08,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-29.28,> rotate <0,-213.76,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,60.60,> rotate <0,-310.67,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-17.45,> rotate <0,-316.94,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,20.80,> rotate <0,-51.11,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,33.20,> rotate <0,-3.51,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-25.42,> rotate <0,-90.3,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,30.67,> rotate <0,-295.79,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,27.28,> rotate <0,-182.69,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-86.37,> rotate <0,-255.25,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-5.93,> rotate <0,-84.37,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-17.80,> rotate <0,-276.98,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-39.87,> rotate <0,-239.03,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,35.50,> rotate <0,-325.5,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,10.20,> rotate <0,-7.08,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-19.75,> rotate <0,-190.46,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-35.63,> rotate <0,-268.24,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-67.93,> rotate <0,-219.44,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-48.60,> rotate <0,-343.41,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-2.82,> rotate <0,-103.75,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-29.53,> rotate <0,-342.84,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-12.78,> rotate <0,-43.13,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-39.35,> rotate <0,-71.48,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,8.63,> rotate <0,-119.31,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,62.95,> rotate <0,-342.18,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,3.60,> rotate <0,-72.19,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-36.78,> rotate <0,-262.23,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-72.03,> rotate <0,-199.83,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,45.27,> rotate <0,-231.02,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,-59.70,> rotate <0,-259.8,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,8.30,> rotate <0,-149.53,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,55.80,> rotate <0,-319.31,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,39.98,> rotate <0,-268.02,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,25.93,> rotate <0,-32.8,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,14.65,> rotate <0,-209.67,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,47.90,> rotate <0,-296.86,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.03,0,0> rotate <0,0,31.20,> rotate <0,-264.99,0>  pigment {rgb <0.24625,0.24625,0.24625>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,59.37,> rotate <0,-52.54,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-23.68,> rotate <0,-99.9,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-50.70,> rotate <0,-198.35,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-17.75,> rotate <0,-259.97,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,44.70,> rotate <0,-15.02,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-39.13,> rotate <0,-234.68,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,29.80,> rotate <0,-310.15,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-48.42,> rotate <0,-149.43,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-72.13,> rotate <0,-200.72,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,36.23,> rotate <0,-292.7,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,16.08,> rotate <0,-238.4,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,80.00,> rotate <0,-270.02,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-28.78,> rotate <0,-297.3,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,11.27,> rotate <0,-294.22,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,4.53,> rotate <0,-313.92,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-70.30,> rotate <0,-212.63,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-85.62,> rotate <0,-180.59,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,25.23,> rotate <0,-176.05,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-9.72,> rotate <0,-206.8,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,38.73,> rotate <0,-316.73,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,20.82,> rotate <0,-65.6,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-3.80,> rotate <0,-47.83,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-3.75,> rotate <0,-157.18,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,3.45,> rotate <0,-292.64,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-22.53,> rotate <0,-284.6,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-33.18,> rotate <0,-313,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-66.92,> rotate <0,-49.5,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-64.93,> rotate <0,-141.68,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-28.65,> rotate <0,-274.35,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-79.07,> rotate <0,-130.8,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,67.20,> rotate <0,-167.41,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-47.92,> rotate <0,-279.81,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,27.97,> rotate <0,-162.48,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-65.83,> rotate <0,-131.13,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-42.88,> rotate <0,-117.92,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,8.60,> rotate <0,-66.59,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-3.05,> rotate <0,-76.23,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,20.47,> rotate <0,-83.41,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,31.35,> rotate <0,-85.15,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,37.27,> rotate <0,-127.83,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,54.42,> rotate <0,-342.2,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-21.00,> rotate <0,-101.76,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-78.82,> rotate <0,-83.9,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-65.60,> rotate <0,-297.75,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-57.10,> rotate <0,-59.68,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-28.25,> rotate <0,-157.95,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-70.07,> rotate <0,-343.66,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,35.78,> rotate <0,-103.26,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-13.05,> rotate <0,-125.7,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-3.22,> rotate <0,-40.45,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-63.48,> rotate <0,-59.02,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-77.65,> rotate <0,-69.59,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-37.57,> rotate <0,-140.13,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,23.93,> rotate <0,-186.31,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-35.93,> rotate <0,-255.15,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,17.32,> rotate <0,-306.6,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-23.98,> rotate <0,-338.9,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,48.73,> rotate <0,-24.11,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,42.02,> rotate <0,-164.06,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-51.68,> rotate <0,-179.8,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,47.15,> rotate <0,-44.14,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,0.13,> rotate <0,-170.83,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-52.03,> rotate <0,-78.47,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,7.60,> rotate <0,-187.84,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,-40.60,> rotate <0,-347.54,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.04,0,0> rotate <0,0,22.63,> rotate <0,-121.58,0>  pigment {rgb <0.245,0.245,0.245>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-1.08,> rotate <0,-257.05,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-73.00,> rotate <0,-282.43,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-38.47,> rotate <0,-10.68,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-39.43,> rotate <0,-141.07,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-32.13,> rotate <0,-123.55,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,34.15,> rotate <0,-295.97,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,27.58,> rotate <0,-90.25,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,39.03,> rotate <0,-32.72,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-52.32,> rotate <0,-110.09,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,51.27,> rotate <0,-140.18,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-62.50,> rotate <0,-11.13,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,47.48,> rotate <0,-33.51,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,61.08,> rotate <0,-75.4,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-22.72,> rotate <0,-94.26,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,3.93,> rotate <0,-124.95,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-26.18,> rotate <0,-228.47,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-52.45,> rotate <0,-237.21,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,0.47,> rotate <0,-314.3,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,33.32,> rotate <0,-270.4,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,72.40,> rotate <0,-125.17,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-67.83,> rotate <0,-169.33,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-13.12,> rotate <0,-76.65,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-53.28,> rotate <0,-114.75,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-65.68,> rotate <0,-184.28,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,16.65,> rotate <0,-242.87,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-38.15,> rotate <0,-250.95,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-57.40,> rotate <0,-178.55,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-37.87,> rotate <0,-182.64,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,46.48,> rotate <0,-34.8,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-69.72,> rotate <0,-212.76,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,44.70,> rotate <0,-324.37,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,11.95,> rotate <0,-191.81,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,24.15,> rotate <0,-127.15,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-22.85,> rotate <0,-9.03,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,6.07,> rotate <0,-93.92,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-28.63,> rotate <0,-86.77,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,52.42,> rotate <0,-257.63,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-60.88,> rotate <0,-28.95,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,31.95,> rotate <0,-130.08,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-14.55,> rotate <0,-40.39,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-16.27,> rotate <0,-173.31,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,7.60,> rotate <0,-261.58,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,42.50,> rotate <0,-48.73,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-17.35,> rotate <0,-318.78,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,38.67,> rotate <0,-315.77,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,56.58,> rotate <0,-85.76,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-54.82,> rotate <0,-8.89,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,12.63,> rotate <0,-51.04,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-56.42,> rotate <0,-147.17,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,34.67,> rotate <0,-234.7,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-1.07,> rotate <0,-304.93,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,13.25,> rotate <0,-119.9,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,57.55,> rotate <0,-104.3,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,54.22,> rotate <0,-155.13,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,23.02,> rotate <0,-115.24,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-56.88,> rotate <0,-258.55,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,-17.58,> rotate <0,-2.14,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,59.42,> rotate <0,-240.79,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.13 translate <10006.05,0,0> rotate <0,0,31.92,> rotate <0,-85.18,0>  pigment {rgb <0.24375,0.24375,0.24375>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,10.03,> rotate <0,-317.63,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,40.58,> rotate <0,-309.89,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,31.78,> rotate <0,-217.46,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,43.78,> rotate <0,-4.68,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-80.17,> rotate <0,-26.95,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-41.33,> rotate <0,-72.57,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-65.70,> rotate <0,-181.6,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-47.52,> rotate <0,-272.77,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-24.13,> rotate <0,-12.31,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,9.57,> rotate <0,-37.4,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-76.03,> rotate <0,-269.4,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-22.97,> rotate <0,-64.65,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-35.25,> rotate <0,-117.31,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-0.85,> rotate <0,-222.23,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-14.85,> rotate <0,-243.97,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-50.62,> rotate <0,-261.8,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-36.47,> rotate <0,-268.78,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,40.43,> rotate <0,-287.85,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,52.95,> rotate <0,-322.58,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,57.45,> rotate <0,-63.76,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,20.48,> rotate <0,-96.99,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,36.18,> rotate <0,-74.08,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,17.88,> rotate <0,-46.86,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,40.78,> rotate <0,-71.05,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-25.47,> rotate <0,-243.73,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,13.97,> rotate <0,-283.01,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,44.07,> rotate <0,-73.71,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,11.90,> rotate <0,-104.36,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-24.47,> rotate <0,-250.4,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,36.45,> rotate <0,-128.34,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,21.17,> rotate <0,-153.63,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-65.13,> rotate <0,-199.19,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-25.85,> rotate <0,-96.8,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-31.83,> rotate <0,-165.17,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-67.32,> rotate <0,-301.38,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,2.35,> rotate <0,-246.71,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,75.92,> rotate <0,-313.68,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-21.38,> rotate <0,-81.9,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,32.18,> rotate <0,-49.15,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-6.67,> rotate <0,-175.98,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-70.50,> rotate <0,-356.1,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,24.18,> rotate <0,-33.15,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-50.88,> rotate <0,-43.53,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,1.88,> rotate <0,-46.16,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,36.20,> rotate <0,-217.07,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-76.73,> rotate <0,-217.4,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-49.48,> rotate <0,-235.65,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,58.25,> rotate <0,-298.84,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,35.58,> rotate <0,-339.03,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-28.08,> rotate <0,-347.43,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-2.22,> rotate <0,-6.13,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,5.63,> rotate <0,-135.38,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,13.57,> rotate <0,-255.99,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-44.25,> rotate <0,-340.68,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,24.85,> rotate <0,-250.25,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,59.27,> rotate <0,-325.69,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,6.55,> rotate <0,-155.75,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,74.27,> rotate <0,-75.59,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-74.48,> rotate <0,-159.82,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,31.78,> rotate <0,-228.53,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,57.08,> rotate <0,-115.2,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,-1.40,> rotate <0,-221.3,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,49.25,> rotate <0,-293.42,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.06,0,0> rotate <0,0,6.13,> rotate <0,-99.35,0>  pigment {rgb <0.2425,0.2425,0.2425>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,-46.65,> rotate <0,-354.87,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,-48.32,> rotate <0,-215.66,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,20.87,> rotate <0,-87.09,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,4.98,> rotate <0,-240.94,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,-26.47,> rotate <0,-92.81,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,-62.87,> rotate <0,-175.97,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,40.92,> rotate <0,-279.89,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,51.30,> rotate <0,-214.34,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,21.28,> rotate <0,-138.41,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,-28.43,> rotate <0,-275.5,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,53.88,> rotate <0,-280.87,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,-19.65,> rotate <0,-228.33,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,-57.97,> rotate <0,-125.3,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,49.10,> rotate <0,-51.49,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,9.82,> rotate <0,-122.82,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,22.25,> rotate <0,-57.48,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,-14.73,> rotate <0,-266.9,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,79.13,> rotate <0,-146.83,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,-61.27,> rotate <0,-182.77,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,55.97,> rotate <0,-268.85,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,-46.58,> rotate <0,-219.8,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,16.85,> rotate <0,-286.99,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,-36.42,> rotate <0,-318.33,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,-33.00,> rotate <0,-243.59,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,52.50,> rotate <0,-16.01,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,3.75,> rotate <0,-276.96,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,-6.38,> rotate <0,-357.13,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,42.02,> rotate <0,-126.18,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.07,0,0> rotate <0,0,25.10,> rotate <0,-231.91,0>  pigment {rgb <0.24125,0.24125,0.24125>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-18.67,> rotate <0,-273.88,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-34.75,> rotate <0,-268.48,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,1.97,> rotate <0,-78.38,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-37.62,> rotate <0,-254.72,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-6.47,> rotate <0,-202.61,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-72.47,> rotate <0,-172.08,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,4.02,> rotate <0,-86.26,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-13.02,> rotate <0,-190.32,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-52.85,> rotate <0,-64.67,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-4.98,> rotate <0,-224.72,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-51.60,> rotate <0,-232.11,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-43.67,> rotate <0,-158.8,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,20.13,> rotate <0,-79.81,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-19.67,> rotate <0,-115.06,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,35.02,> rotate <0,-296.47,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-71.80,> rotate <0,-321.33,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-63.18,> rotate <0,-343.04,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,15.90,> rotate <0,-72.43,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,2.88,> rotate <0,-292.54,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,31.05,> rotate <0,-58.85,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,6.03,> rotate <0,-59.25,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,5.67,> rotate <0,-107.96,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,57.97,> rotate <0,-175.11,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-78.60,> rotate <0,-210.14,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,65.92,> rotate <0,-225.99,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-6.53,> rotate <0,-249.5,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,16.82,> rotate <0,-354.42,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-11.08,> rotate <0,-166.39,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,19.87,> rotate <0,-37.66,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-63.42,> rotate <0,-303.61,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-33.22,> rotate <0,-241.08,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-43.17,> rotate <0,-135.1,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-66.00,> rotate <0,-223.68,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,47.32,> rotate <0,-35.17,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,38.15,> rotate <0,-45.05,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,42.68,> rotate <0,-319.35,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,33.60,> rotate <0,-91.39,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,18.83,> rotate <0,-121.19,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,68.40,> rotate <0,-199.12,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-19.28,> rotate <0,-109.76,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,66.52,> rotate <0,-33.48,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,18.52,> rotate <0,-346.57,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,57.57,> rotate <0,-263.38,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-21.23,> rotate <0,-332.76,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,24.08,> rotate <0,-70.8,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,40.18,> rotate <0,-84.22,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,74.02,> rotate <0,-33.34,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-22.75,> rotate <0,-300.35,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-45.08,> rotate <0,-356,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-44.47,> rotate <0,-333.9,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-44.23,> rotate <0,-81.23,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-6.55,> rotate <0,-132.02,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-16.53,> rotate <0,-148,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-11.20,> rotate <0,-154.04,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,11.40,> rotate <0,-246.55,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-6.25,> rotate <0,-259.18,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-9.98,> rotate <0,-357.56,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,19.67,> rotate <0,-285.72,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,37.73,> rotate <0,-62.07,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-2.45,> rotate <0,-2.05,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-6.83,> rotate <0,-68.56,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,71.00,> rotate <0,-292.75,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-65.85,> rotate <0,-295.4,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-43.73,> rotate <0,-291.09,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,-60.05,> rotate <0,-350.74,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,21.67,> rotate <0,-358.1,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,37.83,> rotate <0,-297.7,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.08,0,0> rotate <0,0,20.83,> rotate <0,-353.48,0>  pigment {rgb <0.24,0.24,0.24>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,48.72,> rotate <0,-92.9,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,32.30,> rotate <0,-223.99,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,33.97,> rotate <0,-294.94,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-55.60,> rotate <0,-221.3,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-28.78,> rotate <0,-141.68,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-9.03,> rotate <0,-93.86,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,16.98,> rotate <0,-278.97,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,61.05,> rotate <0,-281.08,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-29.30,> rotate <0,-291.27,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,73.32,> rotate <0,-333.22,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,25.48,> rotate <0,-163.68,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,51.50,> rotate <0,-28.21,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-25.38,> rotate <0,-131.46,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,18.85,> rotate <0,-132.69,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,14.98,> rotate <0,-155.46,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,40.80,> rotate <0,-325.39,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,52.35,> rotate <0,-151.29,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,56.68,> rotate <0,-299.08,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-30.82,> rotate <0,-108.02,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-3.32,> rotate <0,-282.84,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-51.73,> rotate <0,-141,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-27.53,> rotate <0,-104.43,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,23.22,> rotate <0,-85.83,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-49.63,> rotate <0,-88.67,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-59.92,> rotate <0,-153.25,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,34.80,> rotate <0,-349.86,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-36.10,> rotate <0,-228.28,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-42.88,> rotate <0,-264.53,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,58.75,> rotate <0,-10.63,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-41.17,> rotate <0,-263.28,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,32.00,> rotate <0,-114.98,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,13.62,> rotate <0,-253.82,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,17.05,> rotate <0,-126.46,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-60.25,> rotate <0,-102.5,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-51.77,> rotate <0,-29.25,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-46.95,> rotate <0,-66.78,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-43.17,> rotate <0,-224.86,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-46.53,> rotate <0,-40.53,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,49.58,> rotate <0,-325.66,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,11.20,> rotate <0,-4.57,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,17.28,> rotate <0,-63.13,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-39.62,> rotate <0,-73.73,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-0.57,> rotate <0,-77.51,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,48.97,> rotate <0,-90.43,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-10.58,> rotate <0,-108.93,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,35.02,> rotate <0,-126.27,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-50.37,> rotate <0,-209.82,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-43.12,> rotate <0,-350.94,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-9.98,> rotate <0,-16.4,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,42.92,> rotate <0,-90.83,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,49.57,> rotate <0,-289.66,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-66.07,> rotate <0,-29.92,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,-70.45,> rotate <0,-204.69,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,61.82,> rotate <0,-12.82,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.09,0,0> rotate <0,0,16.45,> rotate <0,-7.05,0>  pigment {rgb <0.23875,0.23875,0.23875>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,62.80,> rotate <0,-87.27,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,21.00,> rotate <0,-264,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,13.82,> rotate <0,-295.28,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-34.35,> rotate <0,-80.32,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-34.10,> rotate <0,-274.4,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,14.55,> rotate <0,-74.75,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,19.62,> rotate <0,-256.17,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-45.40,> rotate <0,-237.57,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-0.87,> rotate <0,-80.96,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-78.52,> rotate <0,-23.41,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,25.72,> rotate <0,-253.76,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-41.42,> rotate <0,-210.86,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-7.12,> rotate <0,-138.11,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-42.25,> rotate <0,-163.75,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-15.33,> rotate <0,-193.98,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-48.30,> rotate <0,-286.73,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-11.77,> rotate <0,-84.29,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-64.67,> rotate <0,-202.28,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-21.17,> rotate <0,-131.23,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-72.57,> rotate <0,-132.46,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-20.73,> rotate <0,-232.65,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,83.17,> rotate <0,-276.04,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-33.05,> rotate <0,-324.2,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,8.95,> rotate <0,-84.33,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-52.18,> rotate <0,-200.57,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,21.88,> rotate <0,-302.84,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-58.95,> rotate <0,-255.45,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,22.92,> rotate <0,-270.63,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,9.97,> rotate <0,-72.93,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-55.80,> rotate <0,-136.64,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,34.42,> rotate <0,-301.93,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,22.40,> rotate <0,-63.21,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,42.15,> rotate <0,-64.54,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-20.15,> rotate <0,-105.03,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,4.92,> rotate <0,-106.78,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-36.33,> rotate <0,-123.49,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,42.37,> rotate <0,-0.43,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-56.13,> rotate <0,-80.6,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-29.08,> rotate <0,-207.53,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-54.62,> rotate <0,-213.42,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-54.02,> rotate <0,-239.98,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-27.75,> rotate <0,-258.1,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-33.52,> rotate <0,-289.14,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-33.47,> rotate <0,-113.55,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,21.70,> rotate <0,-287.01,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,49.08,> rotate <0,-50.47,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-23.18,> rotate <0,-158.74,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,23.58,> rotate <0,-281.17,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-36.13,> rotate <0,-315,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-10.32,> rotate <0,-183.8,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,14.08,> rotate <0,-208.96,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-3.57,> rotate <0,-193.3,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-6.13,> rotate <0,-268.16,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-38.38,> rotate <0,-106.51,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-57.90,> rotate <0,-244.71,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-59.73,> rotate <0,-164.45,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,13.20,> rotate <0,-238.3,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-42.50,> rotate <0,-186.29,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,46.90,> rotate <0,-130.75,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-57.00,> rotate <0,-15.51,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,58.77,> rotate <0,-187.52,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,46.18,> rotate <0,-194.28,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-47.32,> rotate <0,-84.01,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-67.05,> rotate <0,-139.46,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-49.73,> rotate <0,-168.24,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-26.53,> rotate <0,-247.84,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-38.58,> rotate <0,-28.35,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-2.80,> rotate <0,-306.43,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,54.05,> rotate <0,-324.41,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-41.77,> rotate <0,-26.82,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-70.45,> rotate <0,-101.23,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-49.42,> rotate <0,-157.01,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,22.35,> rotate <0,-164.07,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-61.75,> rotate <0,-228.25,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-69.62,> rotate <0,-308.97,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,60.10,> rotate <0,-343.26,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-27.77,> rotate <0,-13.98,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-58.27,> rotate <0,-24.19,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,-22.53,> rotate <0,-37.64,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,4.48,> rotate <0,-44.27,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.10,0,0> rotate <0,0,56.50,> rotate <0,-311.09,0>  pigment {rgb <0.2375,0.2375,0.2375>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,57.53,> rotate <0,-26.08,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-24.08,> rotate <0,-66.74,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,24.45,> rotate <0,-323.86,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,44.12,> rotate <0,-350.18,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-40.35,> rotate <0,-100.31,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,25.25,> rotate <0,-46.36,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-12.97,> rotate <0,-96.09,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-56.73,> rotate <0,-116.4,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,1.37,> rotate <0,-15.95,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-42.75,> rotate <0,-159.71,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-31.62,> rotate <0,-208,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-35.75,> rotate <0,-259.09,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-22.93,> rotate <0,-125.11,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,24.35,> rotate <0,-203.2,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-30.72,> rotate <0,-218.29,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,43.37,> rotate <0,-303.43,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,29.52,> rotate <0,-91.59,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-46.63,> rotate <0,-150.84,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-15.67,> rotate <0,-252.37,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-15.55,> rotate <0,-230.03,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-79.48,> rotate <0,-347.1,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-42.67,> rotate <0,-171.09,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-27.35,> rotate <0,-108.4,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-52.65,> rotate <0,-149.29,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,34.45,> rotate <0,-278.81,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-31.93,> rotate <0,-358.82,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,66.52,> rotate <0,-7.85,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-46.33,> rotate <0,-127.44,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,52.75,> rotate <0,-202,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,25.35,> rotate <0,-215.78,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-9.37,> rotate <0,-89.75,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,18.73,> rotate <0,-65.1,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,56.33,> rotate <0,-301.34,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-58.53,> rotate <0,-73.72,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-32.00,> rotate <0,-240.89,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-59.00,> rotate <0,-329.63,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-41.37,> rotate <0,-52.56,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-12.88,> rotate <0,-11.37,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-1.13,> rotate <0,-55,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-62.70,> rotate <0,-348.53,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,30.70,> rotate <0,-131.34,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,13.23,> rotate <0,-227.23,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,8.23,> rotate <0,-4.14,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,73.63,> rotate <0,-129.93,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,29.02,> rotate <0,-1.65,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-50.83,> rotate <0,-32.59,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,31.60,> rotate <0,-39.18,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-48.73,> rotate <0,-47.61,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-1.05,> rotate <0,-69.95,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-12.98,> rotate <0,-99.42,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,37.15,> rotate <0,-100.81,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-64.10,> rotate <0,-125.28,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-73.37,> rotate <0,-128.18,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-9.57,> rotate <0,-150.92,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-58.48,> rotate <0,-221.23,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-40.80,> rotate <0,-306.45,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,60.77,> rotate <0,-320.14,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-22.03,> rotate <0,-226.61,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,26.93,> rotate <0,-316.6,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,35.47,> rotate <0,-81.73,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-15.55,> rotate <0,-289.75,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,32.28,> rotate <0,-212.82,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,36.63,> rotate <0,-218.66,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-21.32,> rotate <0,-221.8,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,22.92,> rotate <0,-93.08,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,-2.88,> rotate <0,-45.21,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,6.47,> rotate <0,-14.96,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.12 translate <10006.11,0,0> rotate <0,0,13.48,> rotate <0,-30.65,0>  pigment {rgb <0.23625,0.23625,0.23625>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,27.53,> rotate <0,-121.4,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-39.43,> rotate <0,-245.14,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-54.12,> rotate <0,-209.08,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,38.78,> rotate <0,-203.59,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-3.35,> rotate <0,-307.1,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-22.45,> rotate <0,-100.69,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-34.77,> rotate <0,-106.38,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-36.60,> rotate <0,-129.85,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,1.85,> rotate <0,-143.17,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,51.47,> rotate <0,-303.13,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,16.92,> rotate <0,-261.48,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-48.88,> rotate <0,-120.12,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,83.63,> rotate <0,-307.26,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,16.40,> rotate <0,-100.34,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-19.87,> rotate <0,-155.55,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-49.38,> rotate <0,-166.27,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-14.92,> rotate <0,-252.11,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,53.98,> rotate <0,-319.26,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-12.72,> rotate <0,-327.42,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-14.83,> rotate <0,-239.14,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,78.72,> rotate <0,-20.94,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,54.02,> rotate <0,-231.63,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,67.78,> rotate <0,-227.68,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,5.68,> rotate <0,-131.26,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,27.38,> rotate <0,-210.29,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-71.27,> rotate <0,-8.35,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-54.57,> rotate <0,-240.28,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-19.72,> rotate <0,-121.17,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,58.12,> rotate <0,-79.87,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-65.37,> rotate <0,-252.98,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,13.30,> rotate <0,-296.14,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,81.25,> rotate <0,-111.34,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-13.35,> rotate <0,-117.37,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,1.07,> rotate <0,-343.75,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-54.67,> rotate <0,-321.57,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,64.98,> rotate <0,-146.15,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,14.28,> rotate <0,-265.84,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,38.58,> rotate <0,-335.98,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-47.58,> rotate <0,-304.83,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,33.30,> rotate <0,-132.64,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,34.67,> rotate <0,-1.22,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-35.52,> rotate <0,-344.65,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-31.88,> rotate <0,-23.71,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,54.88,> rotate <0,-281.23,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,36.57,> rotate <0,-280.9,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,10.58,> rotate <0,-88.55,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-50.48,> rotate <0,-288.19,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-20.38,> rotate <0,-61.17,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,54.57,> rotate <0,-89.95,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,10.32,> rotate <0,-97.08,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,13.38,> rotate <0,-117.26,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-19.53,> rotate <0,-117.52,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-10.35,> rotate <0,-142.98,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-8.52,> rotate <0,-143.26,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-39.68,> rotate <0,-178.98,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,53.43,> rotate <0,-183.68,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,40.93,> rotate <0,-274.28,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,33.50,> rotate <0,-290.64,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,21.15,> rotate <0,-302.76,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-36.22,> rotate <0,-318.94,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-67.47,> rotate <0,-349.58,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,33.83,> rotate <0,-37.57,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-42.75,> rotate <0,-156.54,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-34.13,> rotate <0,-268.2,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,21.58,> rotate <0,-304.08,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-5.33,> rotate <0,-125.08,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-75.10,> rotate <0,-164.32,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-19.92,> rotate <0,-68.75,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,41.52,> rotate <0,-213.6,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,18.70,> rotate <0,-281.67,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,49.13,> rotate <0,-352.53,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,51.83,> rotate <0,-301.28,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-36.63,> rotate <0,-222.16,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-21.82,> rotate <0,-91.74,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,-21.18,> rotate <0,-329.68,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.12,0,0> rotate <0,0,46.75,> rotate <0,-331.32,0>  pigment {rgb <0.235,0.235,0.235>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,59.62,> rotate <0,-65.74,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,38.87,> rotate <0,-103.26,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,11.25,> rotate <0,-97.79,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,50.75,> rotate <0,-57.1,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-61.88,> rotate <0,-150.89,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-60.75,> rotate <0,-238.72,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-7.13,> rotate <0,-169.24,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,30.23,> rotate <0,-134.49,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-35.85,> rotate <0,-152.38,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,23.13,> rotate <0,-117.14,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-2.25,> rotate <0,-163.35,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-24.55,> rotate <0,-42.46,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,62.93,> rotate <0,-181.42,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,81.13,> rotate <0,-172.96,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,53.22,> rotate <0,-78.69,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,3.87,> rotate <0,-138.05,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-26.75,> rotate <0,-192.99,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,15.27,> rotate <0,-81.81,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,22.28,> rotate <0,-78.87,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,65.98,> rotate <0,-287.44,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-2.20,> rotate <0,-142.35,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-0.75,> rotate <0,-165.81,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-10.53,> rotate <0,-142.91,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,5.45,> rotate <0,-275.37,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,1.92,> rotate <0,-271.15,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,18.62,> rotate <0,-238.92,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,16.07,> rotate <0,-233.04,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-63.30,> rotate <0,-119.3,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,14.67,> rotate <0,-23.95,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,33.75,> rotate <0,-79.58,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,26.02,> rotate <0,-185.08,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-1.02,> rotate <0,-230.93,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-14.87,> rotate <0,-7.47,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-9.15,> rotate <0,-49.15,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,1.10,> rotate <0,-321.62,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,51.53,> rotate <0,-187.51,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-6.55,> rotate <0,-337.83,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-23.82,> rotate <0,-10.14,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-28.90,> rotate <0,-44.78,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,4.23,> rotate <0,-134.15,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,14.57,> rotate <0,-305.09,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,12.95,> rotate <0,-16.64,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-46.52,> rotate <0,-67.33,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,2.48,> rotate <0,-284.11,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,61.53,> rotate <0,-328.84,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-15.85,> rotate <0,-177.58,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,41.68,> rotate <0,-205.62,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-40.22,> rotate <0,-210.33,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,50.07,> rotate <0,-227.08,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,32.57,> rotate <0,-252.68,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-31.63,> rotate <0,-343.96,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,51.17,> rotate <0,-93.44,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-76.67,> rotate <0,-138.05,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,60.65,> rotate <0,-255.32,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-23.42,> rotate <0,-295.03,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,30.32,> rotate <0,-320.68,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,45.83,> rotate <0,-14.42,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-2.53,> rotate <0,-99.58,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-34.85,> rotate <0,-118.67,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,27.43,> rotate <0,-243.95,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,21.88,> rotate <0,-273.32,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,67.27,> rotate <0,-144.87,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-2.47,> rotate <0,-297.8,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.13,0,0> rotate <0,0,-43.05,> rotate <0,-300.11,0>  pigment {rgb <0.23375,0.23375,0.23375>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-36.08,> rotate <0,-184.45,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,7.15,> rotate <0,-298.67,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,1.07,> rotate <0,-306.16,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,18.18,> rotate <0,-101.85,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-37.22,> rotate <0,-256.59,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,60.50,> rotate <0,-295.05,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-51.82,> rotate <0,-165.04,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-43.40,> rotate <0,-249.61,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-51.23,> rotate <0,-145.86,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-19.55,> rotate <0,-254.27,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-45.77,> rotate <0,-272.12,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-61.32,> rotate <0,-164.81,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-34.72,> rotate <0,-268.33,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,45.00,> rotate <0,-12.58,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-6.75,> rotate <0,-92.76,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,17.32,> rotate <0,-94.76,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,35.42,> rotate <0,-118.92,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-26.77,> rotate <0,-137.18,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,32.52,> rotate <0,-229.88,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-23.83,> rotate <0,-280.47,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,82.42,> rotate <0,-126.14,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-52.82,> rotate <0,-126.61,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-3.20,> rotate <0,-214.13,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-63.27,> rotate <0,-253.85,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,11.93,> rotate <0,-263.06,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,57.08,> rotate <0,-26.19,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-52.58,> rotate <0,-41.05,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-9.78,> rotate <0,-44.7,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-63.13,> rotate <0,-246.34,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-69.63,> rotate <0,-51.01,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,41.45,> rotate <0,-75.83,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-6.18,> rotate <0,-125.63,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,53.30,> rotate <0,-331.86,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-39.62,> rotate <0,-125.35,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-1.60,> rotate <0,-334.14,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-18.87,> rotate <0,-294.11,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-29.12,> rotate <0,-307.73,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,46.10,> rotate <0,-56.17,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-63.02,> rotate <0,-276.38,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-14.25,> rotate <0,-74.44,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,39.82,> rotate <0,-336.86,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,0.50,> rotate <0,-81.45,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-4.60,> rotate <0,-96.65,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-61.87,> rotate <0,-97.8,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,0.98,> rotate <0,-102.27,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-27.35,> rotate <0,-238.63,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-17.77,> rotate <0,-228.64,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,24.60,> rotate <0,-74.07,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-18.62,> rotate <0,-327.93,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,50.57,> rotate <0,-36.97,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-1.40,> rotate <0,-78.83,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-0.32,> rotate <0,-177.25,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-15.67,> rotate <0,-21.17,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-2.85,> rotate <0,-21.2,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-31.10,> rotate <0,-37.15,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-69.27,> rotate <0,-46.95,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-57.55,> rotate <0,-266.23,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-7.07,> rotate <0,-280.65,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,62.05,> rotate <0,-196.6,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,67.15,> rotate <0,-243.1,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,17.53,> rotate <0,-310.49,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,1.85,> rotate <0,-67.01,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-76.12,> rotate <0,-332.99,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,35.40,> rotate <0,-120.48,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,50.62,> rotate <0,-174.48,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-38.80,> rotate <0,-219.58,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-17.62,> rotate <0,-257.06,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-31.03,> rotate <0,-284.59,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-25.90,> rotate <0,-334.15,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-78.77,> rotate <0,-338.86,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-73.60,> rotate <0,-347.15,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,19.67,> rotate <0,-55.95,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,19.82,> rotate <0,-120.2,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-47.82,> rotate <0,-27.58,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-42.90,> rotate <0,-243.85,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,21.83,> rotate <0,-241.84,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-54.42,> rotate <0,-76.25,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,6.98,> rotate <0,-317.12,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-4.63,> rotate <0,-11.35,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-40.65,> rotate <0,-117.04,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,-6.58,> rotate <0,-345.63,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.14,0,0> rotate <0,0,28.25,> rotate <0,-133.12,0>  pigment {rgb <0.2325,0.2325,0.2325>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,43.60,> rotate <0,-4.09,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-45.57,> rotate <0,-22.91,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,14.38,> rotate <0,-94.52,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,40.25,> rotate <0,-295.49,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-53.82,> rotate <0,-350.98,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-13.15,> rotate <0,-97.65,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-59.82,> rotate <0,-198.23,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-56.42,> rotate <0,-187.14,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-1.35,> rotate <0,-105.47,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-14.58,> rotate <0,-258.83,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,48.53,> rotate <0,-56.03,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,31.52,> rotate <0,-32.85,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,61.85,> rotate <0,-78.26,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,50.97,> rotate <0,-210.75,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-57.50,> rotate <0,-180.37,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,4.97,> rotate <0,-99.47,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-33.80,> rotate <0,-272.73,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-58.20,> rotate <0,-158.35,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,22.03,> rotate <0,-320.06,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,66.70,> rotate <0,-132.2,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,39.70,> rotate <0,-135.42,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-10.55,> rotate <0,-342.13,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-20.55,> rotate <0,-317.39,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-78.35,> rotate <0,-28.96,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,8.40,> rotate <0,-205.55,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,36.33,> rotate <0,-159.09,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,53.07,> rotate <0,-187.71,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-53.72,> rotate <0,-11.25,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,32.22,> rotate <0,-56.95,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-68.83,> rotate <0,-191.26,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-8.75,> rotate <0,-13.58,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-29.82,> rotate <0,-48.41,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,5.48,> rotate <0,-108.03,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-37.80,> rotate <0,-286.72,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-2.40,> rotate <0,-344.57,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,45.05,> rotate <0,-353.43,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,5.48,> rotate <0,-226.92,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,77.77,> rotate <0,-164.99,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,67.08,> rotate <0,-170.23,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,22.03,> rotate <0,-99.77,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-39.60,> rotate <0,-219.1,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-16.02,> rotate <0,-47.82,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,43.43,> rotate <0,-252.42,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-2.90,> rotate <0,-337.83,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-15.25,> rotate <0,-353.7,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-52.17,> rotate <0,-153.1,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,-38.40,> rotate <0,-203.02,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,29.42,> rotate <0,-162.24,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,20.47,> rotate <0,-20.85,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,81.72,> rotate <0,-182.75,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.15,0,0> rotate <0,0,56.77,> rotate <0,-307.95,0>  pigment {rgb <0.23125,0.23125,0.23125>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-48.75,> rotate <0,-144.11,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-34.70,> rotate <0,-182.51,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,38.82,> rotate <0,-59.92,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,25.32,> rotate <0,-321.03,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-60.97,> rotate <0,-200.65,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-5.92,> rotate <0,-49,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-14.78,> rotate <0,-305.19,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-1.58,> rotate <0,-276.24,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,25.73,> rotate <0,-25.33,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,37.75,> rotate <0,-39.58,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-26.25,> rotate <0,-356.12,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-63.88,> rotate <0,-274.92,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-29.13,> rotate <0,-240.67,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,2.65,> rotate <0,-97.31,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,10.63,> rotate <0,-125.98,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-35.45,> rotate <0,-256.62,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,55.68,> rotate <0,-250.75,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,2.18,> rotate <0,-259.06,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,37.45,> rotate <0,-289.75,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-6.12,> rotate <0,-216.85,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,25.08,> rotate <0,-120.43,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-18.92,> rotate <0,-103.34,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-35.07,> rotate <0,-178.36,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-83.32,> rotate <0,-306.23,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,45.25,> rotate <0,-331.55,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-26.32,> rotate <0,-333.24,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,19.68,> rotate <0,-339.95,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-54.67,> rotate <0,-211.45,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-5.92,> rotate <0,-326.91,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-54.98,> rotate <0,-153.82,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-26.97,> rotate <0,-267.12,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,16.70,> rotate <0,-81.53,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,52.22,> rotate <0,-47.02,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,23.63,> rotate <0,-71.43,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-37.53,> rotate <0,-295.91,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,57.28,> rotate <0,-335.75,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,13.05,> rotate <0,-57.82,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,10.57,> rotate <0,-113.53,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-28.60,> rotate <0,-132.59,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,41.93,> rotate <0,-314.63,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-5.70,> rotate <0,-92.4,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-38.13,> rotate <0,-25.36,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,54.83,> rotate <0,-62.34,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,43.17,> rotate <0,-76.71,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,73.68,> rotate <0,-99.48,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,59.07,> rotate <0,-133.28,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,13.65,> rotate <0,-122.18,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,8.18,> rotate <0,-335.23,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,34.58,> rotate <0,-21.53,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,32.63,> rotate <0,-90.73,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,36.45,> rotate <0,-226.65,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,13.87,> rotate <0,-65.22,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-60.45,> rotate <0,-250.21,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,9.47,> rotate <0,-76.91,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-5.90,> rotate <0,-350.17,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,1.68,> rotate <0,-4.45,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,4.47,> rotate <0,-119.1,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-35.90,> rotate <0,-176.78,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-10.98,> rotate <0,-254.36,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-18.17,> rotate <0,-230.22,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,59.40,> rotate <0,-78.8,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-22.93,> rotate <0,-87.87,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,52.82,> rotate <0,-346.79,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,26.83,> rotate <0,-347.95,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,8.88,> rotate <0,-174.53,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-65.27,> rotate <0,-226.99,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-61.10,> rotate <0,-281.3,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,41.90,> rotate <0,-252.65,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,48.15,> rotate <0,-3,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-43.15,> rotate <0,-198.49,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,7.18,> rotate <0,-202.5,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-55.05,> rotate <0,-305.14,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,11.20,> rotate <0,-319.72,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-26.75,> rotate <0,-173.1,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,-22.38,> rotate <0,-307.38,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.16,0,0> rotate <0,0,31.70,> rotate <0,-160.55,0>  pigment {rgb <0.23,0.23,0.23>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,52.75,> rotate <0,-106.42,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,37.32,> rotate <0,-77.58,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,40.02,> rotate <0,-67.85,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-41.10,> rotate <0,-250.97,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-33.00,> rotate <0,-207.96,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-4.65,> rotate <0,-92.93,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-43.48,> rotate <0,-212.22,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-31.68,> rotate <0,-210.75,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-61.53,> rotate <0,-99.5,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,38.48,> rotate <0,-301.29,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,3.28,> rotate <0,-117.69,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-22.42,> rotate <0,-86.11,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-23.93,> rotate <0,-298.57,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-42.30,> rotate <0,-91.67,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-69.83,> rotate <0,-116.05,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-10.07,> rotate <0,-302.13,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-23.68,> rotate <0,-336.55,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-17.40,> rotate <0,-1.83,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-3.95,> rotate <0,-244.23,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-23.12,> rotate <0,-1.71,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-40.83,> rotate <0,-254.15,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,39.30,> rotate <0,-344.42,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-20.48,> rotate <0,-256.19,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,39.53,> rotate <0,-327.77,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,23.72,> rotate <0,-57.43,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,1.17,> rotate <0,-76.35,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,75.30,> rotate <0,-255.42,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,29.18,> rotate <0,-82.42,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,50.72,> rotate <0,-202.19,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-79.43,> rotate <0,-324.74,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,27.93,> rotate <0,-289,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-40.85,> rotate <0,-36.14,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-0.92,> rotate <0,-125.05,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,45.20,> rotate <0,-273.88,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,17.83,> rotate <0,-53.53,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-34.92,> rotate <0,-3.74,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,33.27,> rotate <0,-81.72,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,19.35,> rotate <0,-156.75,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,51.90,> rotate <0,-164.83,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-36.37,> rotate <0,-198.79,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-9.27,> rotate <0,-352.25,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,2.70,> rotate <0,-99.42,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,17.15,> rotate <0,-134.28,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-77.63,> rotate <0,-114.02,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-0.27,> rotate <0,-222.75,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,32.90,> rotate <0,-140.36,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,8.10,> rotate <0,-216.08,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-54.37,> rotate <0,-86.8,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,1.47,> rotate <0,-136.75,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,80.87,> rotate <0,-180.08,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,37.03,> rotate <0,-200.98,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.11 translate <10006.17,0,0> rotate <0,0,-29.37,> rotate <0,-340.59,0>  pigment {rgb <0.22875,0.22875,0.22875>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-42.73,> rotate <0,-266.18,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,27.38,> rotate <0,-276.25,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-24.60,> rotate <0,-128.27,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-63.93,> rotate <0,-82.57,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,37.65,> rotate <0,-355.01,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-59.78,> rotate <0,-200.15,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-8.07,> rotate <0,-208.74,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,0.20,> rotate <0,-115.77,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-56.47,> rotate <0,-117.08,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,32.88,> rotate <0,-296.64,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,53.30,> rotate <0,-275.95,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-30.53,> rotate <0,-85.55,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-51.30,> rotate <0,-199.64,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,3.80,> rotate <0,-267.08,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,2.05,> rotate <0,-289.45,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-13.28,> rotate <0,-319.3,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-84.58,> rotate <0,-164.81,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-73.97,> rotate <0,-156.18,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-28.08,> rotate <0,-71.61,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,0.38,> rotate <0,-214.92,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-44.10,> rotate <0,-256.45,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-13.70,> rotate <0,-297.01,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-27.83,> rotate <0,-46.96,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-60.18,> rotate <0,-150.75,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,73.13,> rotate <0,-254.07,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-16.52,> rotate <0,-308.88,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-38.38,> rotate <0,-142.32,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,56.88,> rotate <0,-314.07,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-66.98,> rotate <0,-157.54,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,40.17,> rotate <0,-343.3,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,32.45,> rotate <0,-67.66,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,15.12,> rotate <0,-112.2,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-29.15,> rotate <0,-224.31,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,58.92,> rotate <0,-238.96,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,46.83,> rotate <0,-304.98,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-9.77,> rotate <0,-59.16,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,34.03,> rotate <0,-179.99,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,60.23,> rotate <0,-217.93,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-40.95,> rotate <0,-7.62,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,23.33,> rotate <0,-69.12,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,32.03,> rotate <0,-209.05,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,48.00,> rotate <0,-214.46,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,36.72,> rotate <0,-355.17,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-6.50,> rotate <0,-45.54,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-19.22,> rotate <0,-134.92,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,14.53,> rotate <0,-305.72,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,29.90,> rotate <0,-318.54,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-0.05,> rotate <0,-6.66,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-12.00,> rotate <0,-10.96,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-52.33,> rotate <0,-98.36,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,1.37,> rotate <0,-299.85,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,7.47,> rotate <0,-38.77,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,9.38,> rotate <0,-134.42,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-18.68,> rotate <0,-351.03,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-5.05,> rotate <0,-187.91,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-55.43,> rotate <0,-292.47,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-57.98,> rotate <0,-294.6,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,61.17,> rotate <0,-76.62,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,40.45,> rotate <0,-79.67,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,3.03,> rotate <0,-100.78,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-4.93,> rotate <0,-129.37,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-36.52,> rotate <0,-153.84,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-42.48,> rotate <0,-156.33,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-10.58,> rotate <0,-324.87,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,64.22,> rotate <0,-346.99,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,33.12,> rotate <0,-19.08,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-35.32,> rotate <0,-208.47,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,48.82,> rotate <0,-256.21,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,38.40,> rotate <0,-216.37,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-77.92,> rotate <0,-234.83,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-7.20,> rotate <0,-343.64,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,-9.87,> rotate <0,-95.9,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,27.00,> rotate <0,-36.78,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.18,0,0> rotate <0,0,75.05,> rotate <0,-84.93,0>  pigment {rgb <0.2275,0.2275,0.2275>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-48.07,> rotate <0,-228.97,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-0.38,> rotate <0,-324.39,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,6.62,> rotate <0,-324.43,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,41.03,> rotate <0,-319.73,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,76.95,> rotate <0,-4.06,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,58.62,> rotate <0,-25.57,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,37.77,> rotate <0,-334.73,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-31.08,> rotate <0,-53.49,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-3.30,> rotate <0,-328.65,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-7.80,> rotate <0,-50.83,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,16.55,> rotate <0,-54.85,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-15.05,> rotate <0,-307.77,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-65.80,> rotate <0,-209.63,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-27.72,> rotate <0,-315.79,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,0.53,> rotate <0,-179.76,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-37.62,> rotate <0,-128.63,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-7.63,> rotate <0,-157.74,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,38.02,> rotate <0,-353.7,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-15.02,> rotate <0,-138.85,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,24.77,> rotate <0,-185.55,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-48.37,> rotate <0,-283.26,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,54.37,> rotate <0,-289.9,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,7.07,> rotate <0,-358.78,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-15.77,> rotate <0,-206.4,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-84.78,> rotate <0,-82.56,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,66.10,> rotate <0,-22.72,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,14.63,> rotate <0,-122.75,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-64.82,> rotate <0,-324.51,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-40.50,> rotate <0,-141.62,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-4.80,> rotate <0,-84.47,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-69.10,> rotate <0,-149.25,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-46.97,> rotate <0,-121.33,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-69.62,> rotate <0,-202.19,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-26.28,> rotate <0,-234.37,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,12.73,> rotate <0,-51.83,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,65.13,> rotate <0,-162.21,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-75.03,> rotate <0,-224.98,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,63.17,> rotate <0,-333.46,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-50.15,> rotate <0,-351.79,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,20.83,> rotate <0,-350.67,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-20.52,> rotate <0,-189,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-37.75,> rotate <0,-329.26,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,47.40,> rotate <0,-297.83,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,34.38,> rotate <0,-21.77,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,24.92,> rotate <0,-291.35,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-42.00,> rotate <0,-268.22,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,9.70,> rotate <0,-210.63,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,6.85,> rotate <0,-328.49,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-14.50,> rotate <0,-90.64,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-38.78,> rotate <0,-114.64,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-45.60,> rotate <0,-267.94,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,25.52,> rotate <0,-324.69,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-37.10,> rotate <0,-229.88,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,49.97,> rotate <0,-0.33,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,13.58,> rotate <0,-252.04,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-72.55,> rotate <0,-317.84,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,59.55,> rotate <0,-0.13,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-20.00,> rotate <0,-38.42,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,41.85,> rotate <0,-91.86,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-26.70,> rotate <0,-92.45,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-22.03,> rotate <0,-106.2,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-32.63,> rotate <0,-224.13,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-62.60,> rotate <0,-238.35,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,38.32,> rotate <0,-309.35,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,37.45,> rotate <0,-107.15,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-52.95,> rotate <0,-144.2,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-24.25,> rotate <0,-261.28,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-40.92,> rotate <0,-97.75,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-22.67,> rotate <0,-108.35,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,-28.63,> rotate <0,-286.88,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,53.90,> rotate <0,-340.58,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,15.68,> rotate <0,-17.55,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,1.72,> rotate <0,-20.65,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.19,0,0> rotate <0,0,24.57,> rotate <0,-13.81,0>  pigment {rgb <0.22625,0.22625,0.22625>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,52.70,> rotate <0,-13.45,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,74.73,> rotate <0,-129.2,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,25.80,> rotate <0,-308,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-50.30,> rotate <0,-350.21,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-14.47,> rotate <0,-174.96,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-40.43,> rotate <0,-240.23,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-13.65,> rotate <0,-67.86,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,32.82,> rotate <0,-346.86,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-63.25,> rotate <0,-64.42,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,1.53,> rotate <0,-315.76,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,36.93,> rotate <0,-307.75,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,38.70,> rotate <0,-297.36,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-7.05,> rotate <0,-292.34,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,80.47,> rotate <0,-198.11,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,50.52,> rotate <0,-295.47,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,12.07,> rotate <0,-235.05,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,3.45,> rotate <0,-39.65,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,42.98,> rotate <0,-305.73,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,16.95,> rotate <0,-97.79,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,51.58,> rotate <0,-12.89,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-41.12,> rotate <0,-131.6,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-33.32,> rotate <0,-179.27,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-26.50,> rotate <0,-256.72,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,31.52,> rotate <0,-266.42,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-24.95,> rotate <0,-308.22,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,8.55,> rotate <0,-334,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,43.27,> rotate <0,-121.79,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-24.55,> rotate <0,-103.48,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,10.62,> rotate <0,-111.62,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-40.77,> rotate <0,-230.4,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,66.70,> rotate <0,-84.32,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-8.02,> rotate <0,-21.01,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-1.77,> rotate <0,-103.68,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-16.72,> rotate <0,-85.42,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-23.95,> rotate <0,-149.78,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-53.37,> rotate <0,-150.42,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-36.12,> rotate <0,-56.96,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,18.68,> rotate <0,-93.39,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-22.92,> rotate <0,-111.08,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,31.57,> rotate <0,-309.38,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-65.83,> rotate <0,-185.12,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,23.93,> rotate <0,-271.17,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,20.23,> rotate <0,-338.15,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-14.60,> rotate <0,-78.5,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,13.70,> rotate <0,-249.45,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,33.45,> rotate <0,-171.61,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,11.13,> rotate <0,-268.56,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,6.25,> rotate <0,-284.6,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-12.32,> rotate <0,-219.25,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,28.15,> rotate <0,-184.87,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-58.43,> rotate <0,-193.59,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-4.73,> rotate <0,-315.14,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,0.18,> rotate <0,-345.16,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-35.45,> rotate <0,-122.29,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,5.62,> rotate <0,-151.7,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,61.55,> rotate <0,-179.22,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,2.93,> rotate <0,-307.07,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,10.98,> rotate <0,-103.35,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-7.92,> rotate <0,-18.59,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-4.77,> rotate <0,-25.98,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-58.22,> rotate <0,-115.72,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-26.67,> rotate <0,-135.3,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-57.35,> rotate <0,-150.49,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-61.78,> rotate <0,-187.1,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-42.70,> rotate <0,-198.21,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-43.10,> rotate <0,-211.55,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,50.82,> rotate <0,-271.72,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,45.95,> rotate <0,-294.89,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,52.30,> rotate <0,-307.84,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,5.53,> rotate <0,-312.5,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-40.50,> rotate <0,-318.81,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-24.95,> rotate <0,-106.72,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-35.88,> rotate <0,-124,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-42.77,> rotate <0,-303.1,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,57.10,> rotate <0,-345.38,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,31.95,> rotate <0,-65.04,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,4.25,> rotate <0,-282.01,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-7.07,> rotate <0,-345.35,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,42.18,> rotate <0,-228.54,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,44.80,> rotate <0,-315.6,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,76.48,> rotate <0,-299.9,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,6.98,> rotate <0,-189.63,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.20,0,0> rotate <0,0,-24.88,> rotate <0,-268.73,0>  pigment {rgb <0.225,0.225,0.225>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,14.40,> rotate <0,-266.07,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,57.58,> rotate <0,-66.75,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,48.47,> rotate <0,-270.79,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,73.83,> rotate <0,-30.79,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-59.90,> rotate <0,-154.01,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-20.17,> rotate <0,-232.93,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-47.88,> rotate <0,-119.33,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-58.73,> rotate <0,-131.27,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-35.50,> rotate <0,-104.33,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-69.52,> rotate <0,-13.1,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-2.43,> rotate <0,-174.6,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,51.42,> rotate <0,-247.18,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,80.62,> rotate <0,-191.11,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,38.10,> rotate <0,-39.62,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-45.77,> rotate <0,-104.68,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-72.60,> rotate <0,-184.1,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-23.15,> rotate <0,-234.84,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-30.22,> rotate <0,-258.96,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-1.80,> rotate <0,-284.09,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-32.53,> rotate <0,-41.09,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-41.92,> rotate <0,-285.53,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,75.10,> rotate <0,-141.97,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,22.98,> rotate <0,-228.38,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,25.23,> rotate <0,-69.85,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-2.83,> rotate <0,-85.16,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-14.33,> rotate <0,-113.34,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-37.77,> rotate <0,-101.55,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,41.42,> rotate <0,-286.57,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,0.10,> rotate <0,-309.33,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-8.57,> rotate <0,-94.78,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,28.18,> rotate <0,-106.85,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,30.97,> rotate <0,-109.52,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,41.38,> rotate <0,-283.03,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-2.42,> rotate <0,-309.8,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-12.90,> rotate <0,-90.08,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-4.62,> rotate <0,-186.97,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,51.87,> rotate <0,-308.71,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-12.92,> rotate <0,-337.51,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-18.42,> rotate <0,-58.3,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,35.93,> rotate <0,-100.41,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,8.90,> rotate <0,-120.46,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-58.80,> rotate <0,-146.48,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-58.62,> rotate <0,-220.48,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,19.48,> rotate <0,-231.47,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,50.68,> rotate <0,-342.55,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-13.60,> rotate <0,-67.54,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,50.82,> rotate <0,-284.75,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-11.70,> rotate <0,-215.86,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,77.92,> rotate <0,-30.74,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,24.45,> rotate <0,-59.36,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,21.28,> rotate <0,-76.09,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-1.00,> rotate <0,-89.55,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,22.42,> rotate <0,-115.84,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-70.72,> rotate <0,-258.08,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,7.97,> rotate <0,-22.1,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,11.95,> rotate <0,-267.68,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,53.47,> rotate <0,-85.34,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,1.23,> rotate <0,-89.48,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-60.83,> rotate <0,-182.1,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-4.38,> rotate <0,-329.73,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,4.20,> rotate <0,-82.58,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,27.55,> rotate <0,-196.98,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-43.73,> rotate <0,-168.73,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-7.67,> rotate <0,-180.72,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,14.73,> rotate <0,-315.76,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,30.73,> rotate <0,-286.24,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-65.47,> rotate <0,-15.68,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,-10.35,> rotate <0,-107.33,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.21,0,0> rotate <0,0,1.62,> rotate <0,-99.66,0>  pigment {rgb <0.22375,0.22375,0.22375>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-37.50,> rotate <0,-252.75,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,39.40,> rotate <0,-305.06,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,18.02,> rotate <0,-68.38,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,51.90,> rotate <0,-312.18,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-26.82,> rotate <0,-165.6,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,32.55,> rotate <0,-218.55,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,18.38,> rotate <0,-195.16,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-7.90,> rotate <0,-339.59,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-30.92,> rotate <0,-123.97,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,30.07,> rotate <0,-18.25,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,13.48,> rotate <0,-5.6,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-19.63,> rotate <0,-325.8,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,50.90,> rotate <0,-299.19,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-35.60,> rotate <0,-307.2,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-16.30,> rotate <0,-124.35,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,48.97,> rotate <0,-247.53,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,21.55,> rotate <0,-224.26,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,15.17,> rotate <0,-265.3,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-60.70,> rotate <0,-14.59,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,31.17,> rotate <0,-58.02,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,24.25,> rotate <0,-257.76,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,25.88,> rotate <0,-309.03,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,15.30,> rotate <0,-129.03,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,43.22,> rotate <0,-279.19,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,40.52,> rotate <0,-256.95,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-6.70,> rotate <0,-82.84,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,39.22,> rotate <0,-284.8,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,10.22,> rotate <0,-63.4,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-35.68,> rotate <0,-181.48,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,24.37,> rotate <0,-114.56,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-57.88,> rotate <0,-338.97,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-1.08,> rotate <0,-74.32,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,84.18,> rotate <0,-138.84,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,17.20,> rotate <0,-326.77,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,46.83,> rotate <0,-174.64,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-2.98,> rotate <0,-110.58,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,44.10,> rotate <0,-191.11,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,13.33,> rotate <0,-264.74,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,66.82,> rotate <0,-75.71,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-24.18,> rotate <0,-288.89,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-61.83,> rotate <0,-188.87,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-3.23,> rotate <0,-20.14,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-18.35,> rotate <0,-35.74,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-3.40,> rotate <0,-54.91,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,16.23,> rotate <0,-97.12,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,57.25,> rotate <0,-355.48,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,77.02,> rotate <0,-7.73,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-5.23,> rotate <0,-112.71,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-20.28,> rotate <0,-175.52,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-68.65,> rotate <0,-181.58,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,24.97,> rotate <0,-230.28,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,25.50,> rotate <0,-164.15,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,31.23,> rotate <0,-81.16,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,87.15,> rotate <0,-19.06,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,28.08,> rotate <0,-205.16,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-56.67,> rotate <0,-222.28,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,85.18,> rotate <0,-90.34,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,-63.68,> rotate <0,-95.75,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,61.27,> rotate <0,-327.33,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,43.08,> rotate <0,-295.17,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,45.35,> rotate <0,-269.2,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.22,0,0> rotate <0,0,18.67,> rotate <0,-298.09,0>  pigment {rgb <0.2225,0.2225,0.2225>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,1.62,> rotate <0,-75.5,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-30.90,> rotate <0,-109.24,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,59.90,> rotate <0,-110.57,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-66.40,> rotate <0,-19.26,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-19.45,> rotate <0,-243,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-26.63,> rotate <0,-276.93,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-1.03,> rotate <0,-83.52,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,33.63,> rotate <0,-285.45,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-9.23,> rotate <0,-121.61,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,15.63,> rotate <0,-261.14,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-18.68,> rotate <0,-107.54,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-9.40,> rotate <0,-24.4,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,36.95,> rotate <0,-261.11,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-11.02,> rotate <0,-131.53,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,53.58,> rotate <0,-305.13,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,12.42,> rotate <0,-193.46,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-48.47,> rotate <0,-192.58,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-62.53,> rotate <0,-240.72,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,50.62,> rotate <0,-348.56,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-19.03,> rotate <0,-314.9,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,10.07,> rotate <0,-128.16,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-22.48,> rotate <0,-148.63,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-12.10,> rotate <0,-152.48,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,55.18,> rotate <0,-232.24,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,13.68,> rotate <0,-82.15,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-1.00,> rotate <0,-250.3,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,10.72,> rotate <0,-302.16,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,22.40,> rotate <0,-90.42,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-26.87,> rotate <0,-114.86,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-5.53,> rotate <0,-81.51,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-55.95,> rotate <0,-190.71,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-2.08,> rotate <0,-245.66,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,65.60,> rotate <0,-147.6,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,15.77,> rotate <0,-151.42,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,4.90,> rotate <0,-210.98,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,36.62,> rotate <0,-232.84,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-46.08,> rotate <0,-250.76,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-39.90,> rotate <0,-278.25,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,32.88,> rotate <0,-39.28,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,4.35,> rotate <0,-147.63,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-20.77,> rotate <0,-172.95,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-41.38,> rotate <0,-333.66,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,59.02,> rotate <0,-76.55,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,68.47,> rotate <0,-88.23,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,30.50,> rotate <0,-76.06,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,53.30,> rotate <0,-101.05,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-23.47,> rotate <0,-101.35,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-48.28,> rotate <0,-102.89,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-51.37,> rotate <0,-182.88,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,18.88,> rotate <0,-221.33,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-6.13,> rotate <0,-236.69,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,24.38,> rotate <0,-254.43,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,17.72,> rotate <0,-37.73,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,2.08,> rotate <0,-66.75,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,18.13,> rotate <0,-92.76,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-60.98,> rotate <0,-158.55,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,29.65,> rotate <0,-313.28,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,64.32,> rotate <0,-44.1,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-5.75,> rotate <0,-75.2,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,78.32,> rotate <0,-267.54,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-10.87,> rotate <0,-297.26,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.10 translate <10006.23,0,0> rotate <0,0,-50.50,> rotate <0,-265.52,0>  pigment {rgb <0.22125,0.22125,0.22125>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,34.68,> rotate <0,-49.01,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-45.55,> rotate <0,-138.81,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-23.42,> rotate <0,-294.95,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,27.28,> rotate <0,-3.79,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-24.08,> rotate <0,-102.43,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-22.17,> rotate <0,-282.71,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-67.20,> rotate <0,-189.72,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,46.95,> rotate <0,-52.36,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,50.83,> rotate <0,-257.07,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-1.62,> rotate <0,-253.55,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,31.62,> rotate <0,-152.07,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-41.38,> rotate <0,-186.19,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-1.70,> rotate <0,-171.98,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-37.42,> rotate <0,-306.72,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,59.75,> rotate <0,-321.86,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,52.65,> rotate <0,-92.95,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-7.32,> rotate <0,-152.83,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-4.22,> rotate <0,-243.24,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-74.53,> rotate <0,-259.15,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-18.47,> rotate <0,-261.15,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,0.85,> rotate <0,-282.4,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-9.87,> rotate <0,-53.66,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,56.07,> rotate <0,-338.97,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-24.23,> rotate <0,-358.34,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,57.08,> rotate <0,-42.38,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,44.85,> rotate <0,-103.28,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-45.27,> rotate <0,-294.93,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-38.63,> rotate <0,-233.02,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-21.37,> rotate <0,-352.93,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,30.25,> rotate <0,-184.63,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-64.40,> rotate <0,-183.57,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,66.17,> rotate <0,-329.3,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,63.63,> rotate <0,-0.85,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-25.78,> rotate <0,-102.65,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,20.98,> rotate <0,-308.54,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,13.77,> rotate <0,-274.51,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,19.78,> rotate <0,-284.69,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-60.98,> rotate <0,-249.72,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-39.00,> rotate <0,-287.76,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,12.20,> rotate <0,-295.55,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-24.67,> rotate <0,-287.06,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,12.58,> rotate <0,-230.6,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,38.52,> rotate <0,-103.49,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-26.10,> rotate <0,-48.75,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-27.03,> rotate <0,-153.33,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,6.12,> rotate <0,-40.06,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,39.75,> rotate <0,-214.1,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,46.23,> rotate <0,-26.95,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,32.90,> rotate <0,-354.13,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,32.02,> rotate <0,-17,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-20.03,> rotate <0,-0.74,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,58.00,> rotate <0,-295.81,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-13.72,> rotate <0,-308.55,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-2.25,> rotate <0,-77.74,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-28.80,> rotate <0,-134.82,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,1.12,> rotate <0,-351.81,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,16.07,> rotate <0,-96.22,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,8.43,> rotate <0,-307.03,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,25.45,> rotate <0,-2.22,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-10.78,> rotate <0,-68.34,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-18.48,> rotate <0,-93.83,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,9.07,> rotate <0,-94.67,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-35.07,> rotate <0,-96.38,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,53.12,> rotate <0,-128.14,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-7.20,> rotate <0,-143.33,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-41.92,> rotate <0,-183.88,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,2.03,> rotate <0,-264.79,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-22.48,> rotate <0,-266.94,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,33.38,> rotate <0,-188.45,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-46.98,> rotate <0,-128.38,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-13.97,> rotate <0,-160.63,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,-41.07,> rotate <0,-70.93,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,54.13,> rotate <0,-124.57,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,55.17,> rotate <0,-175.43,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,45.33,> rotate <0,-49.02,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.24,0,0> rotate <0,0,22.80,> rotate <0,-284.99,0>  pigment {rgb <0.22,0.22,0.22>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,2.93,> rotate <0,-200.17,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,2.57,> rotate <0,-266.14,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-60.62,> rotate <0,-161.57,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-70.87,> rotate <0,-161.08,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-4.42,> rotate <0,-83.84,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-45.05,> rotate <0,-133.83,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,27.47,> rotate <0,-288.99,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-44.35,> rotate <0,-92.15,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,43.03,> rotate <0,-113.98,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,32.27,> rotate <0,-179.53,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-6.30,> rotate <0,-191.89,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-66.72,> rotate <0,-40.86,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,29.15,> rotate <0,-304.38,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-10.90,> rotate <0,-21.94,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-13.23,> rotate <0,-133.13,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-4.27,> rotate <0,-86.51,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,26.65,> rotate <0,-1.23,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,19.70,> rotate <0,-255.97,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,32.13,> rotate <0,-87.86,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,16.95,> rotate <0,-291.09,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,48.47,> rotate <0,-192.18,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,8.88,> rotate <0,-96.01,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,9.63,> rotate <0,-297.82,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,57.22,> rotate <0,-205.09,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-38.02,> rotate <0,-169.3,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-19.78,> rotate <0,-188.35,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-63.83,> rotate <0,-96.23,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-21.60,> rotate <0,-104.91,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,56.93,> rotate <0,-54.58,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-40.05,> rotate <0,-113.3,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-66.52,> rotate <0,-189.98,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-29.50,> rotate <0,-287.83,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,55.62,> rotate <0,-72.03,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,48.18,> rotate <0,-260.14,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,25.95,> rotate <0,-84.13,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,26.73,> rotate <0,-287.88,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,38.45,> rotate <0,-103.3,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-55.02,> rotate <0,-301.9,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-10.32,> rotate <0,-177.1,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-6.25,> rotate <0,-65.16,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-4.23,> rotate <0,-142.38,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-48.90,> rotate <0,-186.7,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-14.13,> rotate <0,-237.91,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,15.12,> rotate <0,-270.24,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,36.00,> rotate <0,-41.74,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,49.03,> rotate <0,-89.27,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-54.42,> rotate <0,-295.08,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-50.60,> rotate <0,-103.51,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,7.53,> rotate <0,-210.9,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-44.53,> rotate <0,-21.17,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,5.35,> rotate <0,-45.47,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,12.75,> rotate <0,-63.46,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-42.95,> rotate <0,-120.87,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-23.33,> rotate <0,-142.46,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-49.07,> rotate <0,-176.3,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-25.50,> rotate <0,-206.19,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,16.68,> rotate <0,-275.76,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-36.00,> rotate <0,-276.34,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-49.35,> rotate <0,-300.11,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,64.08,> rotate <0,-337.08,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-15.85,> rotate <0,-359.59,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,49.72,> rotate <0,-275.28,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,45.87,> rotate <0,-323.33,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-1.75,> rotate <0,-160.35,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,0.23,> rotate <0,-292.33,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,47.38,> rotate <0,-118.62,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-4.53,> rotate <0,-352.38,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-59.12,> rotate <0,-119.21,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,59.98,> rotate <0,-153.78,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,18.00,> rotate <0,-331.88,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-58.83,> rotate <0,-152.9,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-41.18,> rotate <0,-196.2,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.25,0,0> rotate <0,0,-24.92,> rotate <0,-275.38,0>  pigment {rgb <0.21875,0.21875,0.21875>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,0.87,> rotate <0,-303.15,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-7.52,> rotate <0,-96.69,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-52.20,> rotate <0,-221.8,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,63.88,> rotate <0,-301.18,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-1.25,> rotate <0,-142.26,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,73.03,> rotate <0,-196.2,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-22.90,> rotate <0,-108.45,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,63.63,> rotate <0,-346.95,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,39.67,> rotate <0,-38.87,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,28.75,> rotate <0,-260.38,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,50.52,> rotate <0,-205.1,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,32.93,> rotate <0,-331.14,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,61.13,> rotate <0,-148.77,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-47.75,> rotate <0,-235.24,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,14.28,> rotate <0,-272.14,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,23.03,> rotate <0,-184.83,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-69.20,> rotate <0,-289.12,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-73.03,> rotate <0,-76.54,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-23.75,> rotate <0,-108.95,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-15.60,> rotate <0,-152.15,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-75.15,> rotate <0,-169.9,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-59.85,> rotate <0,-263.9,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-3.90,> rotate <0,-332.59,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,20.32,> rotate <0,-116.29,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,29.53,> rotate <0,-95.3,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-28.70,> rotate <0,-84.05,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,36.10,> rotate <0,-101.05,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,7.62,> rotate <0,-104.66,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-8.87,> rotate <0,-61.48,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-40.58,> rotate <0,-339.12,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,33.53,> rotate <0,-193.55,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,59.72,> rotate <0,-346.8,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,56.60,> rotate <0,-36.32,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,27.95,> rotate <0,-80.25,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-81.80,> rotate <0,-280.55,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-35.95,> rotate <0,-48.25,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-30.15,> rotate <0,-155.8,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-13.77,> rotate <0,-72.43,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-72.43,> rotate <0,-161.12,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-49.98,> rotate <0,-305.42,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,20.08,> rotate <0,-26.23,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,75.73,> rotate <0,-54.85,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,63.82,> rotate <0,-186.28,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,53.83,> rotate <0,-32.53,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,78.92,> rotate <0,-249.47,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-46.70,> rotate <0,-12.23,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,37.12,> rotate <0,-28.74,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-55.72,> rotate <0,-104.67,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-66.73,> rotate <0,-142.88,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-41.95,> rotate <0,-155.92,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-38.07,> rotate <0,-264.36,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,37.10,> rotate <0,-300.31,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-9.37,> rotate <0,-317.69,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-39.40,> rotate <0,-135.53,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,52.65,> rotate <0,-344.79,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,65.62,> rotate <0,-157.48,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,79.67,> rotate <0,-218.41,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,53.15,> rotate <0,-74.94,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,72.53,> rotate <0,-307.5,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,36.48,> rotate <0,-143.38,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,68.68,> rotate <0,-210.46,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,-17.78,> rotate <0,-32.19,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,11.00,> rotate <0,-100.32,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.26,0,0> rotate <0,0,40.70,> rotate <0,-296.2,0>  pigment {rgb <0.2175,0.2175,0.2175>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-28.62,> rotate <0,-153.01,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,49.72,> rotate <0,-271.78,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-18.83,> rotate <0,-198.36,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-19.02,> rotate <0,-111.21,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-38.40,> rotate <0,-100.57,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,11.50,> rotate <0,-139.46,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,15.98,> rotate <0,-318.37,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-26.55,> rotate <0,-149.2,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,51.58,> rotate <0,-91.12,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-67.48,> rotate <0,-230.17,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-39.07,> rotate <0,-127.85,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,61.25,> rotate <0,-220.52,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,55.37,> rotate <0,-35.97,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,7.58,> rotate <0,-18.44,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-57.07,> rotate <0,-66.05,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,52.42,> rotate <0,-311.97,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-8.05,> rotate <0,-102.68,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,33.38,> rotate <0,-177.79,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-41.07,> rotate <0,-229.54,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,45.50,> rotate <0,-349.48,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,49.22,> rotate <0,-52.87,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-4.80,> rotate <0,-88.9,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-53.35,> rotate <0,-263.83,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,60.42,> rotate <0,-272.78,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,58.48,> rotate <0,-341.85,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,2.08,> rotate <0,-358.27,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-23.37,> rotate <0,-12.39,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,34.72,> rotate <0,-83.41,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-41.92,> rotate <0,-277.76,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,14.38,> rotate <0,-167.93,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,48.23,> rotate <0,-331.46,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-49.38,> rotate <0,-201.84,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,42.07,> rotate <0,-335.46,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,23.73,> rotate <0,-94.25,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-55.18,> rotate <0,-139.88,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-63.80,> rotate <0,-122.1,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-25.47,> rotate <0,-272.23,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-51.07,> rotate <0,-274.25,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,32.55,> rotate <0,-137.02,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-47.07,> rotate <0,-179.18,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-49.72,> rotate <0,-23.15,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-72.42,> rotate <0,-73.28,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,1.25,> rotate <0,-215.81,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,45.65,> rotate <0,-126.9,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-5.73,> rotate <0,-277.56,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,58.57,> rotate <0,-345.84,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,19.37,> rotate <0,-341.37,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,8.47,> rotate <0,-47.16,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,68.50,> rotate <0,-63.22,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-11.17,> rotate <0,-98.2,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-35.35,> rotate <0,-104.19,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-55.78,> rotate <0,-130.59,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,2.30,> rotate <0,-211.15,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,31.88,> rotate <0,-255.57,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,24.53,> rotate <0,-321.1,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,18.45,> rotate <0,-336.42,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-34.15,> rotate <0,-341.83,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,40.72,> rotate <0,-4.93,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,18.72,> rotate <0,-198.3,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,3.80,> rotate <0,-344.38,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-8.30,> rotate <0,-114.07,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,45.77,> rotate <0,-210.55,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,38.27,> rotate <0,-228.4,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-5.50,> rotate <0,-228.71,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,18.25,> rotate <0,-30.93,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,22.70,> rotate <0,-205.94,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,2.48,> rotate <0,-161.29,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-36.27,> rotate <0,-226.56,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,-13.07,> rotate <0,-344.9,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,23.43,> rotate <0,-143.5,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,18.80,> rotate <0,-47.09,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.27,0,0> rotate <0,0,18.27,> rotate <0,-309.48,0>  pigment {rgb <0.21625,0.21625,0.21625>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-42.65,> rotate <0,-122.45,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,57.93,> rotate <0,-242.26,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-47.17,> rotate <0,-255.92,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,58.22,> rotate <0,-284.37,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,25.63,> rotate <0,-40.96,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,35.55,> rotate <0,-41.77,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,55.70,> rotate <0,-102.05,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-26.67,> rotate <0,-159.02,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-44.30,> rotate <0,-0.08,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,52.27,> rotate <0,-285.53,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-78.30,> rotate <0,-75.06,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-21.83,> rotate <0,-180.18,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,33.35,> rotate <0,-242.92,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,56.73,> rotate <0,-172.43,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-63.63,> rotate <0,-227.35,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,8.10,> rotate <0,-241.4,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,8.32,> rotate <0,-103.9,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,63.43,> rotate <0,-165.28,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-34.70,> rotate <0,-261.26,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,47.62,> rotate <0,-266.78,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,6.48,> rotate <0,-269.23,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,50.22,> rotate <0,-303.93,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-40.82,> rotate <0,-300.36,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,8.57,> rotate <0,-238.67,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-26.75,> rotate <0,-277.03,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,49.52,> rotate <0,-355.36,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,0.57,> rotate <0,-178.46,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,29.55,> rotate <0,-337.89,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-7.08,> rotate <0,-265.95,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,20.47,> rotate <0,-327.36,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,35.18,> rotate <0,-290.14,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,25.92,> rotate <0,-104.86,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,17.30,> rotate <0,-60.15,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-30.72,> rotate <0,-69.21,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-52.85,> rotate <0,-132,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,50.28,> rotate <0,-290.85,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,38.88,> rotate <0,-304.01,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-56.18,> rotate <0,-241.85,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-55.40,> rotate <0,-267.16,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-65.57,> rotate <0,-97.51,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,36.92,> rotate <0,-131.04,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,87.70,> rotate <0,-183.83,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,79.92,> rotate <0,-19.13,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,57.05,> rotate <0,-218.57,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-32.98,> rotate <0,-174.25,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,85.58,> rotate <0,-181.12,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-52.37,> rotate <0,-325,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-5.37,> rotate <0,-105.08,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-30.58,> rotate <0,-223.14,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-29.53,> rotate <0,-15.82,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,77.58,> rotate <0,-20.08,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-2.38,> rotate <0,-260.71,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,57.03,> rotate <0,-295.92,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-14.97,> rotate <0,-211.11,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-26.62,> rotate <0,-213.3,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,34.25,> rotate <0,-20.9,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,59.90,> rotate <0,-62.37,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,8.87,> rotate <0,-124.38,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,47.93,> rotate <0,-172.73,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,34.57,> rotate <0,-225.9,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-0.50,> rotate <0,-1.27,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,47.43,> rotate <0,-19.54,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-78.33,> rotate <0,-54.13,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-44.75,> rotate <0,-117.37,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-42.28,> rotate <0,-138.33,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-23.02,> rotate <0,-209.37,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,39.70,> rotate <0,-239.38,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-40.65,> rotate <0,-239.99,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,72.62,> rotate <0,-247.87,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-30.53,> rotate <0,-299.73,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-39.57,> rotate <0,-310.72,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-33.68,> rotate <0,-324.78,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,38.83,> rotate <0,-59.62,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-36.38,> rotate <0,-151.32,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-6.68,> rotate <0,-109.38,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,63.95,> rotate <0,-141.43,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,33.63,> rotate <0,-174.64,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,68.32,> rotate <0,-207.75,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-39.90,> rotate <0,-224.65,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-2.48,> rotate <0,-311.77,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,25.92,> rotate <0,-145.41,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,40.65,> rotate <0,-152.75,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,25.57,> rotate <0,-326.6,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,42.73,> rotate <0,-304.38,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,57.50,> rotate <0,-283.44,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,-47.72,> rotate <0,-304.73,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.28,0,0> rotate <0,0,22.63,> rotate <0,-359.18,0>  pigment {rgb <0.215,0.215,0.215>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-41.12,> rotate <0,-250.94,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,2.27,> rotate <0,-95.36,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-62.83,> rotate <0,-121.18,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,76.48,> rotate <0,-78.65,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-33.08,> rotate <0,-234.05,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,32.85,> rotate <0,-48.95,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,27.72,> rotate <0,-12.47,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,51.58,> rotate <0,-141.23,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-27.33,> rotate <0,-148.25,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,7.35,> rotate <0,-279.96,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-54.27,> rotate <0,-56.14,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,59.73,> rotate <0,-195.67,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-52.57,> rotate <0,-197.74,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,21.75,> rotate <0,-84.86,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,57.07,> rotate <0,-170.46,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,17.05,> rotate <0,-248.86,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,24.28,> rotate <0,-188.39,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,79.95,> rotate <0,-281.41,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-21.43,> rotate <0,-271.8,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,11.00,> rotate <0,-111.67,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-39.40,> rotate <0,-85.88,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,68.43,> rotate <0,-296.69,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,16.48,> rotate <0,-43.3,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,74.67,> rotate <0,-182.45,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-51.10,> rotate <0,-38.48,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,22.77,> rotate <0,-323.64,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-61.42,> rotate <0,-227.69,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,36.90,> rotate <0,-7.24,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-24.00,> rotate <0,-16.53,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-61.02,> rotate <0,-53.21,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-44.65,> rotate <0,-170.6,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-9.65,> rotate <0,-305.75,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-38.55,> rotate <0,-326.12,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-51.22,> rotate <0,-166.52,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-36.42,> rotate <0,-38.06,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,16.20,> rotate <0,-279.29,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,16.58,> rotate <0,-141.38,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,0.03,> rotate <0,-31.62,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,19.82,> rotate <0,-80.24,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,22.47,> rotate <0,-82.68,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-5.32,> rotate <0,-103.1,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,13.25,> rotate <0,-128.44,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-76.30,> rotate <0,-158.85,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-65.77,> rotate <0,-186.33,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,61.15,> rotate <0,-190.77,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-50.78,> rotate <0,-205.73,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-29.38,> rotate <0,-283.15,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,14.38,> rotate <0,-294.06,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,41.77,> rotate <0,-308.45,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-43.55,> rotate <0,-221.88,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,51.35,> rotate <0,-274.99,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,8.38,> rotate <0,-285.59,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,57.97,> rotate <0,-14.05,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,63.08,> rotate <0,-249.23,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.09 translate <10006.29,0,0> rotate <0,0,-51.20,> rotate <0,-154.17,0>  pigment {rgb <0.21375,0.21375,0.21375>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,14.93,> rotate <0,-255.39,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-19.58,> rotate <0,-143.89,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,26.93,> rotate <0,-126.7,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-44.73,> rotate <0,-127.66,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,38.10,> rotate <0,-299.64,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-35.85,> rotate <0,-110.99,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-45.88,> rotate <0,-169.12,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,20.12,> rotate <0,-214.14,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,26.97,> rotate <0,-3.35,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-3.55,> rotate <0,-340.2,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,28.28,> rotate <0,-8.21,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-28.33,> rotate <0,-339.93,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,27.25,> rotate <0,-9.08,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,45.42,> rotate <0,-24.63,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,31.28,> rotate <0,-232.6,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,15.50,> rotate <0,-249.18,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,19.55,> rotate <0,-130.11,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-32.50,> rotate <0,-44.91,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-66.57,> rotate <0,-84.23,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-45.28,> rotate <0,-92.66,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,55.22,> rotate <0,-118.15,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-38.62,> rotate <0,-138.36,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-37.23,> rotate <0,-174.17,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-21.57,> rotate <0,-256.55,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-64.68,> rotate <0,-319.5,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,20.28,> rotate <0,-291.34,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-46.62,> rotate <0,-321.09,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,69.63,> rotate <0,-42.23,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-39.20,> rotate <0,-100.85,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-10.18,> rotate <0,-124.81,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-9.77,> rotate <0,-274.35,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,30.57,> rotate <0,-332.72,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,10.42,> rotate <0,-190.39,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,23.33,> rotate <0,-166.92,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,23.18,> rotate <0,-34.74,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-40.18,> rotate <0,-356.5,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,47.82,> rotate <0,-298.25,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-56.77,> rotate <0,-205.73,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,50.07,> rotate <0,-338.97,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-31.73,> rotate <0,-111.43,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,14.28,> rotate <0,-291.89,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-55.47,> rotate <0,-120.38,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,36.98,> rotate <0,-60.31,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-40.28,> rotate <0,-96.19,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,66.33,> rotate <0,-111.86,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-10.57,> rotate <0,-144.95,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,28.20,> rotate <0,-323.83,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,28.53,> rotate <0,-183,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-22.42,> rotate <0,-210.97,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-48.17,> rotate <0,-239.27,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-34.28,> rotate <0,-263.1,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,27.20,> rotate <0,-255.29,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-12.50,> rotate <0,-261.76,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-6.82,> rotate <0,-280.96,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,24.00,> rotate <0,-269.31,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-41.73,> rotate <0,-75.98,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-45.00,> rotate <0,-213.91,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,13.68,> rotate <0,-14.48,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-43.20,> rotate <0,-35.55,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,78.15,> rotate <0,-146.38,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-22.77,> rotate <0,-194.39,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-25.92,> rotate <0,-341.98,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,61.48,> rotate <0,-86.04,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-4.82,> rotate <0,-344.33,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-1.78,> rotate <0,-180.26,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,37.33,> rotate <0,-289.91,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,28.77,> rotate <0,-282.9,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,60.68,> rotate <0,-259.13,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-11.25,> rotate <0,-20.12,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,27.67,> rotate <0,-83.98,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,33.57,> rotate <0,-84.44,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-35.67,> rotate <0,-86.83,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,32.57,> rotate <0,-96.9,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,71.82,> rotate <0,-108.5,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,69.32,> rotate <0,-128.22,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-36.80,> rotate <0,-154.66,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-56.45,> rotate <0,-220.13,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,31.28,> rotate <0,-265.17,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,75.17,> rotate <0,-268.61,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-7.07,> rotate <0,-276.18,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-56.88,> rotate <0,-281.35,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-12.25,> rotate <0,-293.89,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-52.88,> rotate <0,-297.23,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,19.87,> rotate <0,-306.5,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,53.00,> rotate <0,-311.59,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,59.08,> rotate <0,-332.98,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,27.80,> rotate <0,-334.12,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,51.52,> rotate <0,-87.73,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-60.30,> rotate <0,-140.87,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-1.60,> rotate <0,-85.79,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-52.90,> rotate <0,-278.63,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,0.28,> rotate <0,-350.88,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,45.73,> rotate <0,-332.18,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,42.42,> rotate <0,-122.35,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,5.90,> rotate <0,-243.55,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,21.65,> rotate <0,-271.38,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,25.67,> rotate <0,-50.11,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,36.43,> rotate <0,-298.07,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-0.20,> rotate <0,-163.15,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,32.80,> rotate <0,-221.31,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,13.22,> rotate <0,-304.87,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,8.25,> rotate <0,-31.55,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.30,0,0> rotate <0,0,-46.75,> rotate <0,-22.38,0>  pigment {rgb <0.2125,0.2125,0.2125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,37.40,> rotate <0,-19.7,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-49.15,> rotate <0,-201.78,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-53.08,> rotate <0,-244.34,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-4.37,> rotate <0,-83.88,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,26.68,> rotate <0,-243.19,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-46.05,> rotate <0,-262.95,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,2.37,> rotate <0,-156.05,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,44.58,> rotate <0,-90.08,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,38.73,> rotate <0,-275.99,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,30.42,> rotate <0,-215.03,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,70.98,> rotate <0,-8.33,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-51.47,> rotate <0,-113.67,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,3.28,> rotate <0,-189.52,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-21.10,> rotate <0,-243.62,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,27.90,> rotate <0,-136.04,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-41.63,> rotate <0,-51.55,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,10.75,> rotate <0,-40.62,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-9.43,> rotate <0,-43.21,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,52.83,> rotate <0,-73.29,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-27.47,> rotate <0,-149.23,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,8.95,> rotate <0,-0.6,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,28.28,> rotate <0,-45.88,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-2.07,> rotate <0,-75.16,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-17.47,> rotate <0,-319.57,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-1.42,> rotate <0,-79.9,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-43.08,> rotate <0,-117.84,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-30.33,> rotate <0,-108.49,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,2.37,> rotate <0,-81.15,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,20.27,> rotate <0,-292.34,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,49.42,> rotate <0,-52.96,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,64.40,> rotate <0,-289.94,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-39.33,> rotate <0,-116.77,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-11.35,> rotate <0,-122.24,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,14.28,> rotate <0,-204.53,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,69.33,> rotate <0,-11.66,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-41.08,> rotate <0,-97.18,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-9.07,> rotate <0,-155,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-2.65,> rotate <0,-252.59,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-19.90,> rotate <0,-207.65,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-16.23,> rotate <0,-86.78,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-14.07,> rotate <0,-241.77,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-61.17,> rotate <0,-83.74,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-13.43,> rotate <0,-165.9,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-15.28,> rotate <0,-285.39,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,24.83,> rotate <0,-192.2,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,32.18,> rotate <0,-25.96,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-43.82,> rotate <0,-32.52,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,71.93,> rotate <0,-186.6,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-43.23,> rotate <0,-4.68,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,47.82,> rotate <0,-33.99,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,17.77,> rotate <0,-95.36,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,48.78,> rotate <0,-172.79,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,20.43,> rotate <0,-173.77,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-5.92,> rotate <0,-260,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,9.62,> rotate <0,-289.72,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-37.17,> rotate <0,-26.95,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,54.02,> rotate <0,-61.65,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,10.12,> rotate <0,-64.91,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,42.52,> rotate <0,-86.81,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,3.23,> rotate <0,-88.56,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-57.58,> rotate <0,-138.9,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,45.40,> rotate <0,-149.48,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-9.53,> rotate <0,-197.31,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,36.82,> rotate <0,-232.61,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,0.88,> rotate <0,-236.42,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,68.57,> rotate <0,-244.54,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,30.85,> rotate <0,-280.01,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-7.40,> rotate <0,-290.77,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,20.80,> rotate <0,-294.58,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,24.25,> rotate <0,-198.04,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,0.55,> rotate <0,-78.45,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-68.22,> rotate <0,-130.98,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-50.95,> rotate <0,-166.61,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-67.68,> rotate <0,-253.07,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,48.38,> rotate <0,-349.93,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-35.55,> rotate <0,-352,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-42.32,> rotate <0,-216.56,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-17.15,> rotate <0,-270.35,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,10.07,> rotate <0,-306.43,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,18.25,> rotate <0,-204.76,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-36.93,> rotate <0,-97.9,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,-38.28,> rotate <0,-125.82,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,3.28,> rotate <0,-118.03,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.31,0,0> rotate <0,0,16.93,> rotate <0,-343.65,0>  pigment {rgb <0.21125,0.21125,0.21125>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-48.28,> rotate <0,-203.62,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,8.45,> rotate <0,-255.5,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-68.23,> rotate <0,-274.5,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,64.87,> rotate <0,-357.16,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-52.32,> rotate <0,-81.19,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,65.65,> rotate <0,-49.38,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-47.93,> rotate <0,-200.72,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-29.18,> rotate <0,-305.12,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,5.33,> rotate <0,-133.1,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,65.45,> rotate <0,-277.81,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,38.53,> rotate <0,-197.43,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-3.62,> rotate <0,-69,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-32.80,> rotate <0,-342.5,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-6.28,> rotate <0,-241.43,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,37.97,> rotate <0,-91.26,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-54.02,> rotate <0,-2.26,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-83.10,> rotate <0,-175.26,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-31.22,> rotate <0,-229.68,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-79.15,> rotate <0,-25.34,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-19.73,> rotate <0,-129.67,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-29.85,> rotate <0,-44.3,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-48.10,> rotate <0,-193.02,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,58.53,> rotate <0,-207.61,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-25.25,> rotate <0,-258.06,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-25.93,> rotate <0,-261.73,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-10.22,> rotate <0,-275.76,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,7.33,> rotate <0,-344.68,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,59.97,> rotate <0,-357.23,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-2.78,> rotate <0,-36.24,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-2.95,> rotate <0,-71.6,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,68.27,> rotate <0,-168.05,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,11.92,> rotate <0,-262.59,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,83.47,> rotate <0,-288.78,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-22.80,> rotate <0,-345.35,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,36.63,> rotate <0,-207.44,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-61.18,> rotate <0,-298.67,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,26.88,> rotate <0,-261.5,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,43.92,> rotate <0,-200.51,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-7.43,> rotate <0,-82.59,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,57.12,> rotate <0,-310.8,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-50.68,> rotate <0,-254.04,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,33.40,> rotate <0,-289.76,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-43.48,> rotate <0,-229.04,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,42.58,> rotate <0,-54.5,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,68.50,> rotate <0,-57.93,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-60.22,> rotate <0,-119.92,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-65.53,> rotate <0,-181.47,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,51.07,> rotate <0,-314.05,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,40.33,> rotate <0,-21.94,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,47.42,> rotate <0,-96.96,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-36.37,> rotate <0,-270.45,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,22.58,> rotate <0,-294.04,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,14.73,> rotate <0,-96.37,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-42.37,> rotate <0,-103.61,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,53.57,> rotate <0,-125.12,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-63.30,> rotate <0,-197.97,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,62.83,> rotate <0,-116.61,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,46.87,> rotate <0,-315.93,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-5.43,> rotate <0,-329.55,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,64.92,> rotate <0,-77.44,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,55.87,> rotate <0,-206.3,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,15.40,> rotate <0,-63.95,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-15.62,> rotate <0,-140.23,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-6.33,> rotate <0,-120.61,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-52.20,> rotate <0,-101.36,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,21.12,> rotate <0,-220.48,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,32.73,> rotate <0,-264.71,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-26.65,> rotate <0,-283.12,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,13.05,> rotate <0,-47.84,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,58.83,> rotate <0,-331.79,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,10.78,> rotate <0,-210.88,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,23.07,> rotate <0,-352.28,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-11.13,> rotate <0,-117.73,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-54.48,> rotate <0,-139.43,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,7.90,> rotate <0,-257.94,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-63.23,> rotate <0,-304.76,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,50.35,> rotate <0,-315.86,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-55.73,> rotate <0,-325.14,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,15.87,> rotate <0,-338.2,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-49.62,> rotate <0,-346.79,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-79.47,> rotate <0,-349.78,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,33.83,> rotate <0,-107.05,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,36.45,> rotate <0,-291.03,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-44.88,> rotate <0,-87.39,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-54.38,> rotate <0,-127.87,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-79.57,> rotate <0,-163.12,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-86.15,> rotate <0,-186.41,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,61.50,> rotate <0,-209.38,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-48.22,> rotate <0,-228.13,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,60.17,> rotate <0,-293.29,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,35.23,> rotate <0,-247.76,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,39.32,> rotate <0,-280.82,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,30.95,> rotate <0,-340.38,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.32,0,0> rotate <0,0,-33.18,> rotate <0,-200.79,0>  pigment {rgb <0.21,0.21,0.21>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,31.08,> rotate <0,-345.18,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-57.53,> rotate <0,-130.18,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-19.85,> rotate <0,-272.81,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-35.97,> rotate <0,-113.09,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,4.92,> rotate <0,-16.46,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,6.20,> rotate <0,-63.88,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,50.08,> rotate <0,-288.83,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,13.67,> rotate <0,-235.79,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,30.05,> rotate <0,-324.06,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,41.08,> rotate <0,-168.42,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-19.82,> rotate <0,-287.45,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,14.62,> rotate <0,-332.59,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,20.08,> rotate <0,-307.34,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-75.68,> rotate <0,-204.8,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,48.68,> rotate <0,-329.26,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-14.58,> rotate <0,-337.57,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,10.90,> rotate <0,-278.7,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-73.25,> rotate <0,-119.82,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,54.37,> rotate <0,-145.78,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-70.35,> rotate <0,-342.32,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,7.87,> rotate <0,-312.45,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,74.60,> rotate <0,-211.73,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-9.58,> rotate <0,-283.26,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,16.12,> rotate <0,-193.12,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-42.00,> rotate <0,-253.5,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,3.38,> rotate <0,-294.7,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-59.28,> rotate <0,-212.23,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-0.43,> rotate <0,-287.46,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,68.77,> rotate <0,-14.23,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-35.72,> rotate <0,-76.11,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-34.70,> rotate <0,-80.09,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,41.63,> rotate <0,-315.97,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,86.43,> rotate <0,-184.22,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-8.18,> rotate <0,-104.25,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-19.08,> rotate <0,-285.78,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,10.93,> rotate <0,-340.93,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,11.25,> rotate <0,-309.96,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-20.12,> rotate <0,-86.12,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,6.35,> rotate <0,-206.74,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-25.95,> rotate <0,-34.75,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-15.23,> rotate <0,-55.05,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-20.52,> rotate <0,-61.45,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-0.45,> rotate <0,-271.53,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-24.75,> rotate <0,-359.13,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,71.65,> rotate <0,-138.52,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,54.03,> rotate <0,-338.08,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,2.73,> rotate <0,-128.85,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-17.52,> rotate <0,-85.56,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-31.68,> rotate <0,-126.82,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,22.85,> rotate <0,-192.57,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,57.92,> rotate <0,-234.79,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,59.55,> rotate <0,-276.93,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-42.02,> rotate <0,-290.54,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-68.38,> rotate <0,-291.03,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-7.47,> rotate <0,-293.88,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,-28.22,> rotate <0,-234.57,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,19.30,> rotate <0,-285.27,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,22.67,> rotate <0,-85.52,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,28.25,> rotate <0,-348.94,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,44.92,> rotate <0,-19.27,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,28.17,> rotate <0,-264.03,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.33,0,0> rotate <0,0,26.93,> rotate <0,-335.25,0>  pigment {rgb <0.20875,0.20875,0.20875>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-27.05,> rotate <0,-358.59,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-2.22,> rotate <0,-74.55,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-18.12,> rotate <0,-92.34,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,70.37,> rotate <0,-338.32,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,64.65,> rotate <0,-20.27,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,46.65,> rotate <0,-312.48,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,21.40,> rotate <0,-14.58,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,46.53,> rotate <0,-311.66,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-22.62,> rotate <0,-99.17,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-12.18,> rotate <0,-177.84,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-52.77,> rotate <0,-87.62,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,1.13,> rotate <0,-119.32,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,14.37,> rotate <0,-172.94,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-45.03,> rotate <0,-91.12,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-36.08,> rotate <0,-98.98,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-40.32,> rotate <0,-153.49,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-53.83,> rotate <0,-193.1,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,22.67,> rotate <0,-189.76,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,5.95,> rotate <0,-191.76,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-30.58,> rotate <0,-253.65,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-57.90,> rotate <0,-329.06,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-25.60,> rotate <0,-50.57,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,41.25,> rotate <0,-51.29,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,5.33,> rotate <0,-80.88,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-29.85,> rotate <0,-83.47,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,48.43,> rotate <0,-145.68,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-46.58,> rotate <0,-283.25,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,35.93,> rotate <0,-76.5,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-79.67,> rotate <0,-169.64,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-14.65,> rotate <0,-278.09,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,22.28,> rotate <0,-285.46,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,14.00,> rotate <0,-123.1,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,16.50,> rotate <0,-263.61,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,15.52,> rotate <0,-111.12,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,6.27,> rotate <0,-270.22,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,61.42,> rotate <0,-25.75,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-24.45,> rotate <0,-241.97,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,7.32,> rotate <0,-104.83,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,11.42,> rotate <0,-279.3,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-39.93,> rotate <0,-42.7,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,70.95,> rotate <0,-72.65,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,47.77,> rotate <0,-106.29,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,74.32,> rotate <0,-144.03,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-39.88,> rotate <0,-216.71,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,53.92,> rotate <0,-51.45,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,71.82,> rotate <0,-59.13,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,5.47,> rotate <0,-137.98,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,26.92,> rotate <0,-0.1,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,57.87,> rotate <0,-53.42,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-9.55,> rotate <0,-249.91,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,9.92,> rotate <0,-290.7,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-36.43,> rotate <0,-207.9,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-3.88,> rotate <0,-96.45,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-48.87,> rotate <0,-224.54,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-18.57,> rotate <0,-284.86,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,2.27,> rotate <0,-316.19,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,10.30,> rotate <0,-188.26,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,26.62,> rotate <0,-84.74,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-57.95,> rotate <0,-155.96,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-42.93,> rotate <0,-88.35,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-41.55,> rotate <0,-99.21,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,45.82,> rotate <0,-103.81,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,43.98,> rotate <0,-119.57,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-37.33,> rotate <0,-152.01,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-15.78,> rotate <0,-165.24,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-71.43,> rotate <0,-167.88,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-27.73,> rotate <0,-186.33,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-2.42,> rotate <0,-230.2,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-31.08,> rotate <0,-238.88,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-1.23,> rotate <0,-268,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-17.08,> rotate <0,-346.3,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,29.45,> rotate <0,-348.27,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,77.45,> rotate <0,-247.66,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-10.90,> rotate <0,-268.26,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,24.12,> rotate <0,-348.65,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,45.30,> rotate <0,-25.82,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-78.57,> rotate <0,-183.48,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-81.50,> rotate <0,-271.36,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,42.52,> rotate <0,-254.46,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,47.75,> rotate <0,-48.11,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,17.17,> rotate <0,-94.1,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-57.47,> rotate <0,-161.91,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,62.10,> rotate <0,-232.34,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-22.70,> rotate <0,-285.41,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-57.52,> rotate <0,-335.15,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,40.03,> rotate <0,-114.33,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.34,0,0> rotate <0,0,-37.05,> rotate <0,-120.53,0>  pigment {rgb <0.2075,0.2075,0.2075>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-40.52,> rotate <0,-40.83,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-60.32,> rotate <0,-129.8,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-31.88,> rotate <0,-142.69,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,36.30,> rotate <0,-317.76,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-27.17,> rotate <0,-104.53,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,24.30,> rotate <0,-66,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-33.78,> rotate <0,-183.4,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,13.35,> rotate <0,-152.91,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,15.70,> rotate <0,-223.35,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-16.73,> rotate <0,-336.64,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,46.77,> rotate <0,-154.75,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-10.68,> rotate <0,-348.67,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-20.37,> rotate <0,-122.18,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,84.07,> rotate <0,-124.23,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,62.27,> rotate <0,-231.92,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-5.92,> rotate <0,-278.35,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-66.17,> rotate <0,-216.42,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-3.62,> rotate <0,-273.99,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-27.88,> rotate <0,-265.83,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,31.52,> rotate <0,-309.39,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,15.90,> rotate <0,-97.9,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-3.40,> rotate <0,-321.22,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,28.08,> rotate <0,-291.1,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,34.80,> rotate <0,-23.03,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,69.52,> rotate <0,-284.72,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,48.95,> rotate <0,-34.85,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-29.15,> rotate <0,-88.56,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,38.88,> rotate <0,-262.67,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-21.00,> rotate <0,-281.33,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,10.17,> rotate <0,-321.1,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-55.18,> rotate <0,-127.4,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-16.63,> rotate <0,-186.95,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-33.07,> rotate <0,-29.61,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-10.45,> rotate <0,-54.85,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,49.28,> rotate <0,-294.49,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-15.98,> rotate <0,-116.91,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,10.45,> rotate <0,-256.55,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-3.88,> rotate <0,-257.23,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,70.37,> rotate <0,-299.67,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,24.08,> rotate <0,-128.25,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-64.05,> rotate <0,-87.85,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,39.00,> rotate <0,-110.56,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,52.92,> rotate <0,-129.75,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,23.25,> rotate <0,-219.03,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-26.00,> rotate <0,-310.35,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,-41.85,> rotate <0,-34.85,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,75.22,> rotate <0,-243.13,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.08 translate <10006.35,0,0> rotate <0,0,57.37,> rotate <0,-161.6,0>  pigment {rgb <0.20625,0.20625,0.20625>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-60.95,> rotate <0,-64.09,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-57.52,> rotate <0,-302.78,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,11.25,> rotate <0,-307.8,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,14.38,> rotate <0,-143.97,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-33.92,> rotate <0,-64.76,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,41.10,> rotate <0,-21.86,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-18.52,> rotate <0,-79.82,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-18.50,> rotate <0,-79.83,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-50.75,> rotate <0,-182.02,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-0.68,> rotate <0,-199.38,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,41.95,> rotate <0,-271.25,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,56.58,> rotate <0,-273.67,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,51.37,> rotate <0,-222.39,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,1.17,> rotate <0,-93.98,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-13.38,> rotate <0,-187.52,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-35.73,> rotate <0,-160.22,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-5.92,> rotate <0,-262.45,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,53.50,> rotate <0,-16.79,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-33.70,> rotate <0,-349.93,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,0.63,> rotate <0,-270.07,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-8.15,> rotate <0,-78.39,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,49.08,> rotate <0,-281.92,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-68.22,> rotate <0,-315.37,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-11.72,> rotate <0,-349.42,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,33.03,> rotate <0,-98.43,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-10.77,> rotate <0,-145.08,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,11.03,> rotate <0,-269.36,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,62.68,> rotate <0,-78.38,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-65.63,> rotate <0,-203.4,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-64.65,> rotate <0,-280.59,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-39.82,> rotate <0,-284.39,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,34.82,> rotate <0,-59.51,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-24.00,> rotate <0,-16.8,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,3.87,> rotate <0,-171.46,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-71.40,> rotate <0,-149.04,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-34.62,> rotate <0,-131.7,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-51.35,> rotate <0,-149.09,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,39.90,> rotate <0,-39.33,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,10.97,> rotate <0,-330.51,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-10.42,> rotate <0,-85.19,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,4.33,> rotate <0,-131,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,56.33,> rotate <0,-331.55,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-64.30,> rotate <0,-37.02,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-63.03,> rotate <0,-262.03,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-23.77,> rotate <0,-114.32,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-25.10,> rotate <0,-309.47,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-1.42,> rotate <0,-344.6,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,17.23,> rotate <0,-32.35,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,13.22,> rotate <0,-8.98,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,24.60,> rotate <0,-99.88,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,29.77,> rotate <0,-348.59,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-38.40,> rotate <0,-27.45,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-5.95,> rotate <0,-213.59,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-41.75,> rotate <0,-131.92,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-8.20,> rotate <0,-262.01,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-65.43,> rotate <0,-29.47,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-40.37,> rotate <0,-63.4,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,54.37,> rotate <0,-172.55,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-23.13,> rotate <0,-334.25,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-3.12,> rotate <0,-181.5,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,8.62,> rotate <0,-186.93,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,0.08,> rotate <0,-148.05,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,56.07,> rotate <0,-46.42,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-23.22,> rotate <0,-89.14,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,63.45,> rotate <0,-91.66,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,41.05,> rotate <0,-92.1,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-73.12,> rotate <0,-100.91,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-34.63,> rotate <0,-128.25,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-64.50,> rotate <0,-136.53,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-36.27,> rotate <0,-147.46,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,83.92,> rotate <0,-152.14,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,24.72,> rotate <0,-154.75,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-5.12,> rotate <0,-154.88,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-67.63,> rotate <0,-185.81,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-69.68,> rotate <0,-198.85,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,12.43,> rotate <0,-201.13,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,22.05,> rotate <0,-225.22,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,69.18,> rotate <0,-254.76,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,41.15,> rotate <0,-273.18,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-61.68,> rotate <0,-342.09,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,6.37,> rotate <0,-161.52,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-30.02,> rotate <0,-71.96,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,38.37,> rotate <0,-233.95,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-33.52,> rotate <0,-253.99,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,57.88,> rotate <0,-262.68,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,55.73,> rotate <0,-292.81,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,22.45,> rotate <0,-295.31,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-62.90,> rotate <0,-309.96,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,15.67,> rotate <0,-316.89,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,61.67,> rotate <0,-355.63,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-15.45,> rotate <0,-163.39,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,51.25,> rotate <0,-44.21,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-83.25,> rotate <0,-251.48,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,9.75,> rotate <0,-256.54,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,35.65,> rotate <0,-339.2,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-61.37,> rotate <0,-157.67,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-30.28,> rotate <0,-119.1,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.36,0,0> rotate <0,0,-39.75,> rotate <0,-204.92,0>  pigment {rgb <0.205,0.205,0.205>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-9.75,> rotate <0,-68.81,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-1.65,> rotate <0,-256.72,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,56.65,> rotate <0,-306.6,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,56.92,> rotate <0,-87.64,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,47.27,> rotate <0,-225.16,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,26.47,> rotate <0,-276.67,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,57.63,> rotate <0,-32.87,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,65.03,> rotate <0,-254.1,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-49.90,> rotate <0,-188.5,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,38.58,> rotate <0,-261.01,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,4.87,> rotate <0,-50.91,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-23.78,> rotate <0,-313.25,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,45.25,> rotate <0,-0.18,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-25.62,> rotate <0,-52.98,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,3.05,> rotate <0,-102.91,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-61.28,> rotate <0,-172.81,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,1.02,> rotate <0,-163.05,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,38.78,> rotate <0,-214.95,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,16.82,> rotate <0,-182.63,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,20.97,> rotate <0,-332.63,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-15.02,> rotate <0,-128.14,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,40.87,> rotate <0,-189.7,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,17.42,> rotate <0,-26.65,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,21.87,> rotate <0,-213.67,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,1.30,> rotate <0,-82,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,38.38,> rotate <0,-294.24,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-82.68,> rotate <0,-323.48,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-18.75,> rotate <0,-286.78,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,56.45,> rotate <0,-336.75,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-52.58,> rotate <0,-118.75,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,66.75,> rotate <0,-8.6,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,19.50,> rotate <0,-73.75,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,27.75,> rotate <0,-149.61,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-3.58,> rotate <0,-276.02,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,2.92,> rotate <0,-293.85,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,58.95,> rotate <0,-357.3,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,10.33,> rotate <0,-59.92,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,33.00,> rotate <0,-330.11,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-11.90,> rotate <0,-81.77,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,53.78,> rotate <0,-155.06,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-53.38,> rotate <0,-289.54,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,45.18,> rotate <0,-339.58,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,5.25,> rotate <0,-279.9,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-33.03,> rotate <0,-153.35,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,25.18,> rotate <0,-140.88,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-64.23,> rotate <0,-61.84,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,41.80,> rotate <0,-217.4,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-23.87,> rotate <0,-93.44,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-66.08,> rotate <0,-116.18,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,19.05,> rotate <0,-199.06,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-18.58,> rotate <0,-214.27,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-23.77,> rotate <0,-310.05,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,3.40,> rotate <0,-12.83,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-23.73,> rotate <0,-47.65,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,13.77,> rotate <0,-115.19,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,12.87,> rotate <0,-116.06,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,60.63,> rotate <0,-125.68,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-31.50,> rotate <0,-128.21,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,45.20,> rotate <0,-157.15,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,13.88,> rotate <0,-254.38,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,19.15,> rotate <0,-342.67,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-18.88,> rotate <0,-255.46,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,44.02,> rotate <0,-99.99,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-18.53,> rotate <0,-243.66,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,46.37,> rotate <0,-94.57,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-18.03,> rotate <0,-312.33,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,-49.95,> rotate <0,-320.32,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,75.58,> rotate <0,-91.29,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,0.10,> rotate <0,-318.65,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,12.58,> rotate <0,-339.27,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,36.77,> rotate <0,-178.81,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.37,0,0> rotate <0,0,6.35,> rotate <0,-100.5,0>  pigment {rgb <0.20375,0.20375,0.20375>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,40.48,> rotate <0,-49.3,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-52.13,> rotate <0,-133.17,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-27.45,> rotate <0,-251.07,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-66.70,> rotate <0,-294.97,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-38.52,> rotate <0,-262.36,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,42.25,> rotate <0,-299.48,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,44.05,> rotate <0,-345.69,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-37.80,> rotate <0,-260.66,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-17.85,> rotate <0,-84.82,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-5.37,> rotate <0,-80.11,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-3.30,> rotate <0,-82.23,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,55.23,> rotate <0,-345.68,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,17.67,> rotate <0,-123.55,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,30.35,> rotate <0,-67.22,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-35.55,> rotate <0,-65.78,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,48.57,> rotate <0,-43.34,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-58.90,> rotate <0,-190.87,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-29.88,> rotate <0,-238.05,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,24.52,> rotate <0,-265.27,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,75.53,> rotate <0,-358.1,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,58.32,> rotate <0,-42.94,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-34.37,> rotate <0,-225.49,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-28.90,> rotate <0,-272.99,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,14.98,> rotate <0,-253.02,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,48.85,> rotate <0,-5.02,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-33.40,> rotate <0,-85.36,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,59.93,> rotate <0,-129.8,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-36.35,> rotate <0,-191,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,36.60,> rotate <0,-303.52,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,47.30,> rotate <0,-46.95,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,11.70,> rotate <0,-339.15,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,75.30,> rotate <0,-349.33,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-35.92,> rotate <0,-51.48,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,24.58,> rotate <0,-99.36,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-23.82,> rotate <0,-133.98,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-68.77,> rotate <0,-299.72,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,23.20,> rotate <0,-301.24,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-39.13,> rotate <0,-132.47,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,32.38,> rotate <0,-136.23,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-26.65,> rotate <0,-106.75,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-30.65,> rotate <0,-106.5,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-61.67,> rotate <0,-153.34,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,28.43,> rotate <0,-297.48,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-47.87,> rotate <0,-127.8,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-58.23,> rotate <0,-133.73,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-32.57,> rotate <0,-259.59,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,19.52,> rotate <0,-270.12,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-34.10,> rotate <0,-139.95,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-33.95,> rotate <0,-276.48,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-19.23,> rotate <0,-290.4,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-12.68,> rotate <0,-116.69,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,72.12,> rotate <0,-332.57,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,59.82,> rotate <0,-344.79,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,28.93,> rotate <0,-80.85,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-45.83,> rotate <0,-85.92,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,11.02,> rotate <0,-97.25,0>  pigment {rgb <0.2025,0.2025,0.2025>}}
object {star scale 1.07 translate <10006.38,0,0> rotate <0,0,-51.45,> rotate <0,-119.59,0>  pigment {rgb <0.2025,0.2025,0