// Persistence of Vision Ray Tracer Scene Description File
// File: Interp.pov
// Vers: 3.6
// Desc: Bicubic (and other...) interpolation method 
// Date: Mar 2007
// Auth: Vincent Le Chevalier
// -w800 -h600

#version 3.6;

/*The macro doing the interpolation.
* Inputs:
* - order: the kind of interpolation to perform (0: none, 2: bilinear,
*   6: bicubic)
* - values: a function with two arguments, containing the values to interpolate.
* Think of it as an image_map
* - sizex, sizey: the number of pixels in each direction
* - tiling: indicates whether the image is meant to tile or not. It only
* influences the bicubic interpolation, for the computation of derivatives. If
* set to 0 the resulting function will only exist in (0,1)x(0,1), it is usually
* what is needed for a height field anyway.
*/

#macro Interpolate(order, values, sizex, sizey, tiling)
    
    //The resolution in x and y
    #local dx = 1/sizex;
    #local dy = 1/sizey;
	//Two functions to give the position relative to the nearest grid points
    #local px = function(x){(x*sizex-floor(x*sizex))}
    #local py = function(y){(y*sizey-floor(y*sizey))}
    
    //A function to truncate x between a and b
    #local trunc = function(x,a,b){min(max(x,a),b)}


    #local tvalues = 
     #if(tiling)
		function(x,y){values(x,y)}
	 #else 
		function(x,y){values(trunc(x,0,1),trunc(y,0,1))}
        //if not tiling,
        //restrict the function evaluation to (0,1)
	 #end
 
    #switch(order)
        #case(6)//bicubic interpolation
            //we need an evaluation of the slope at each grid point
		    #local slopex = function(x,y){0.5*(tvalues(x+dx,y)-tvalues(x-dx,y))}
			#local slopey = function(x,y){0.5*(tvalues(x,y+dy)-tvalues(x,y-dy))}
			
            //the interpolation polynomial, a function of the position and of h,
            //the value, sx and sy the slope
            #local Inter = function(x,y,h,sx,sy)
			{
			   x*x*y*y*(h *( 9-6*y-6*x+4*y*x)
					 + sx*(-3+2*y+3*x-2*y*x)
					 + sy*(-3+3*y+2*x-2*y*x)
					  )
			}
            //In theory we should use a second order derivative at each grid
            //point as well. It is choosen to be 0 here, because it makes only a
            //slight difference in the result.
			function(x,y)
            {
            //the truncature is necessary because the values are not
            //restricted to those of the grid points surrounding them, in this
            //method
            trunc(
              Inter(1-px(x),1-py(y),tvalues(x,y)      ,-slopex(x,y)      ,-slopey(x,y))
             +Inter( px(x) ,1-py(y),tvalues(x+dx,y)   , slopex(x+dx,y)   ,-slopey(x+dx,y))
             +Inter( px(x) , py(y) ,tvalues(x+dx,y+dy), slopex(x+dx,y+dy), slopey(x+dx,y+dy))
             +Inter(1-px(x), py(y) ,tvalues(x,y+dy)   ,-slopex(x,y+dy)   , slopey(x,y+dy)),
             0,1)
             
            }
        #break
        #case(2)//the bilinear interpolation
            function(x,y)
            {
              (1-px(x))*(1-py(y)) * tvalues(x,y)
             +  px(x)  *(1-py(y)) * tvalues(x+dx,y) 
             +  px(x)  *  py(y)   * tvalues(x+dx,y+dy)
             +(1-px(x))*  py(y)   * tvalues(x,y+dy)
            }
        #break
        #case(0)
            function(x,y){tvalues(x,y)}
		#break
    #end
#end

#declare hfpig = //the pigment corresponding to the image for the height field
function{pigment{image_map{tga "perlin.tga"}} }

#declare hf = //get the height from the color in the pigment
function(x,y){(hfpig(x,y,0).red*255+hfpig(x,y,0).green)/256}

//and build the function with interpolation
#declare altitudes=Interpolate(6, hf, 50, 50, 0)

#declare Res = 1000; //The resolution of the interpolated height field
height_field
{
	function Res,Res { altitudes(x,y) }
	texture
	{
		pigment{rgb <1,0.9,0.7>}
		finish{specular 0.5}
	}
	scale <1,0.2,1>
}

//The rest of the scene...
background{ rgb 1}
light_source{<-1,3,2>*1000, color rgb 1 parallel point_at 0}
#declare cam_look = <0.6,0,0.6>;
camera{location cam_look+0.8*<1.2,0.8,1.2> angle 45 look_at cam_look}

