// Persistence of Vision Ray Tracer Scene Description File
// File: bsplines.inc
// Vers: 3.5
// Desc: Macro Collection for Bezier-Splines
// Date: 28.03.02 (dd.mm.yy)
// Auth: Tim Nikias Wenclawiak
// Last Update: 23.05.04 (dd.mm.yy)

// -Generated Linear-Interpolations now get the corresponding Nove-Mover of the original BSpline as "t"
// -Added BSpline_Dir
// -Added BSpline2Linear_Ext Macro to name the arrays in the Output-File
// -Added support for 4-component vectors and with it continuancy of 2. Order (tangential continuancy)

/*Macros:
Concerning backward compability:
The macros ending on -3 use the "old" syntax of 3-component-vectors, and don't change anything. When
using the macros ending on -4, these promote the given array to 4-component-vectors. All other
macros that weren't split into 3/4 versions work as expected.

Spline-Modifiers (Spline , 3-component-vector):
BSpline_Translate  => give BSpline and translation vector => modifies BSpline
BSpline_Scale3     => give BSpline and scale vector => modifies BSpline
BSpline_Scale4     => give BSpline and scale vector (using support for 4-comp-vector) => modifies BSpline
BSpline_Rotate3    => give BSplne and rotation vector => modifies BSpline
BSpline_Rotate4    => give BSplne and rotation vector (using support for 4-comp-vector) => modifies BSpline

Spline-Flippers (Spline)
BSpline_Flip3      => reverses the given BSpline => modifies BSpline
BSpline_Flip4      => reverses the given BSpline (using support for 4-comp-vector) => modifies BSpline

CSG-Generators (return bunch of objects):
Show_BSpline       => give BSpline, radius and CSG-Amount => returns cylinders and spheres
 -Show_BSpline(Spline,1,50)
Morph_Show_BSpline => give 2 BSplines, clock, radius and CSG_Amount => returns cylinders and spheres
 -Morph_Show_BSpline(Spline_1,Spline_2,clock,1,50)
Show_BNodes        => give BSpline and radius => returns spheres (placed on nodes)
 -Show_BNodes(Spline,1)
Show_BDirs         => give BSpline and radius => returns cones (indicating direction of bspline)
 -Show_BDirs(Spline,1)

Calculators (Spline , float):
BSpline_Pos        => give BSpline and mover (0 >= float <= 1) => returns position
BSpline_Dir        => give BSpline and mover (0 >= float <= 1) => returns direction
BSpline_Normal     => give BSpline and mover (0 >= float <= 1) => returns curvature-normal
BSpline_Length     => give BSPline and detail-counter (float >=2) => returns length of BSpline

Extra:
BSpline2Linear     => give BSpline, steps, final node-amount, filename => writes file
BSpline2Linear_Ext => give BSpline, steps, final node-amount, filename, Name for Bezier, Name for Linear => writes file
Linear_Pos         => give Linear-Interpolation and mover (0 >= float <= 1) => returns position
Show_Linear        => give Linear-Interpolation and thickness => returns cylinders and spheres

Linear_Pos is just an algorithm to connect an array of vectors. With BSpline2Linear
you can generate and write such an array. Useful to just have a series of points
to be interpolated smoothly (e.g. no change in speed due to different curvatures etc).

The file generated by BSpline2Linear has two arrays and one value:
Bezier_Node : The original BSpline
Linear_Node : A series of points that lie on the BSpline
Length : Length of Bezier-Spline
*/

//Homepage:
// www.nolights.de
//Email:
// timnikias@gmx.net


//My files sometimes build on each other. Checking if following parameter
//was declared/is set to true avoids my other include-files to include
//an already included file again.
#declare _bsplines_inc_tnw=1;

//Just a detail value, no actual need to change it
#declare Zero_Value=.0001;

//Macro required in several places to promote 4-comp.-vector to 3-components
#macro V3(_4comp)
 <_4comp.x,_4comp.y,_4comp.z>
#end

/************* Spline-Modifiers *************/

/************************/
/*****BSpline_Scale******/
/************************/
#macro BSpline_Scale(_bspline_array,Scalement)
 //This one will just flash a note in case the old syntax was used.
 //Note that when using Scale4 on 3-component bezier-splines, the vectors will
 //get promoted to 4-component-vectors.
 #error "BSpline_Scale invalid as of 17.Feb.03, use BSpline_Scale3 for old, -Scale4 for new syntax, see above comment!\n"
#end

/*************************/
/*****BSpline_Scale3******/
/*************************/
#macro BSpline_Scale3(_bspline_array,Scalement)
 #local Saved_Length=dimension_size(_bspline_array,1);
  //Loop that jumps from position-node to position-node (C+2)
  #local C=0; #while (C<Saved_Length)
   #declare _bspline_array[C]=_bspline_array[C]*Scalement;  
  #local C=C+1; #end
#end

/*************************/
/*****BSpline_Scale4******/
/*************************/
#macro BSpline_Scale4(_bspline_array,Scalement)
 #local Saved_Length=dimension_size(_bspline_array,1);
 #local _new_array=array[Saved_Length];
  //Loop that jumps from position-node to position-node (C+2)
  #local C=0; #while (C<Saved_Length)
   #local _new_array[C]=_bspline_array[C]*<Scalement.x,Scalement.y,Scalement.z,1>;
  #local C=C+1; #end
 #declare _bspline_array=_new_array;
#end

/************************/
/*****BSpline_Translate**/
/************************/
#macro BSpline_Translate(_bspline_array,Translation)
 #local Saved_Length=dimension_size(_bspline_array,1);
  //Loop that jumps from position-node to position-node (C+2)
  #local C=0; #while (C<Saved_Length)
   #declare _bspline_array[C]=_bspline_array[C]+V3(Translation);
  #local C=C+2; #end
#end

/************************/
/*****BSpline_Rotate*****/
/************************/
#macro BSpline_Rotate(_bspline_array,Rotation)
 //This one will just flash a note in case the old syntax was used.
 //Note that when using Rotate4 on 3-component bezier-splines, the vectors will
 //get promoted to 4-component-vectors.
 #error "BSpline_Rotate invalid as of 17.Feb.03, use BSpline_Rotate3 for old, -Rotate4 for new syntax, see above comment!\n"
#end

/*************************/
/*****BSpline_Rotate3*****/
/*************************/
#macro BSpline_Rotate3(_bspline_array,Rotation)
 #local Saved_Length=dimension_size(_bspline_array,1);
  //Loop that jumps from position-node to position-node (C+2)
  #local C=0; #while (C<Saved_Length)
   #declare _bspline_array[C]=vrotate(_bspline_array[C],Rotation);
  #local C=C+1; #end
#end

/*************************/
/*****BSpline_Rotate4*****/
/*************************/
#macro BSpline_Rotate4(_bspline_array,Rotation)
 #local Saved_Length=dimension_size(_bspline_array,1);
 #local _new_array=array[Saved_Length];
  //Loop that jumps from position-node to position-node (C+2)
  #local C=0; #while (C<Saved_Length)
   #local _new_array[C]=vrotate(<_bspline_array[C].x,_bspline_array[C].y,_bspline_array[C].z>,Rotation)+_bspline_array[C]*t;
  #local C=C+1; #end
 #declare _bspline_array=_new_array;
#end

/************************/
/*******BSpline_Flip*****/
/************************/
#macro BSpline_Flip(_bspline_array)
 //This one will just flash a note in case the old syntax was used.
 //Note that when using Flip4 on 3-component bezier-splines, the vectors will
 //get promoted to 4-component-vectors.
 #error "BSpline_Flip invalid as of 17.Feb.03, use BSpline_Flip3 for old, -Flip4 for new syntax, see above comment!\n"
#end

/*************************/
/*******BSpline_Flip3*****/
/*************************/
#macro BSpline_Flip3(_bspline_array)
 #local Saved_Length=dimension_size(_bspline_array,1);
 #local Old_Spline=_bspline_array 
  #local C=0; #while (C<Saved_Length)
   #declare _bspline_array[C]=Old_Spline[Saved_Length-2-C];
   #declare _bspline_array[C+1]=-Old_Spline[Saved_Length-1-C];
  #local C=C+2; #end
#end

/*************************/
/*******BSpline_Flip4*****/
/*************************/
#macro BSpline_Flip4(_bspline_array)
 #local Saved_Length=dimension_size(_bspline_array,1);
 #local Old_Spline=_bspline_array 
 #local _new_array=array[Saved_Length]; 
  #local C=0; #while (C<Saved_Length)
   #local _new_array[C]=Old_Spline[Saved_Length-2-C]+t*0;
   #local _new_array[C+1]=
    -V3(Old_Spline[Saved_Length-1-C]) //Vector
     *(1+(Old_Spline[Saved_Length-1-C]*<0,0,0,1>).t) //New direction-length
     +t*((1/(1+(Old_Spline[Saved_Length-1-C]*<0,0,0,1>).t))-1); //New length-modifier
 #local C=C+2; #end
 #declare _bspline_array=_new_array;
#end

/************* CSG-Generators *************/


/***********************/
/******Show_BSpline*****/
/***********************/
#macro Show_BSpline(__bspline_array,_Thickness,_Detail)
  //Loop running through the different nodes
  #local _Slice_Val = 0;
  #while (_Slice_Val <= _Detail)
   #local _Actual_Pos = BSpline_Pos(__bspline_array,(_Slice_Val/_Detail));
   sphere{_Actual_Pos,_Thickness}
   #ifdef (_Former_Pos)
    #if (vlength(_Former_Pos-_Actual_Pos)>=0)
     cylinder{_Former_Pos,_Actual_Pos,_Thickness}
    #end
   #end
  #local _Former_Pos = _Actual_Pos;
  #local _Slice_Val = _Slice_Val+1;
 #end
 #undef _Former_Pos
#end

/*************************/
/****Morph_Show_BSpline***/
/*************************/
#macro Morph_Show_BSpline(_bspline_array_1,_bspline_array_2,Clock,Thickness,Detail)
  //Loop running through the different nodes
  #local Slice_Val = 0;
  #while (Slice_Val <= Detail)
   #local Actual_Pos_1 = BSpline_Pos(_bspline_array_1,Slice_Val/Detail);
   #local Actual_Pos_2 = BSpline_Pos(_bspline_array_2,Slice_Val/Detail);
    #local Actual_Pos=Actual_Pos_1-(Actual_Pos_1-Actual_Pos_2)*Clock;
    sphere{Actual_Pos,Thickness}
    #ifdef (Former_Pos)
     #if (vlength(Former_Pos-Actual_Pos)!=0)
     cylinder{Former_Pos,Actual_Pos,Thickness}
     #end
    #end
   #local Former_Pos = Actual_Pos;
  //End of loop
  #local Slice_Val = Slice_Val+1;
 #end
#end

/***********************/
/******Show_BNodes******/
/***********************/
#macro Show_BNodes(_bspline_array,Thickness)
 #local Nodes = dimension_size(_bspline_array,1);
 #local Node_Val=0;
 #while (Node_Val < Nodes)
  sphere{V3(_bspline_array[Node_Val]),Thickness}
 #local Node_Val=Node_Val+2;
 #end
#end

/***********************/
/*******Show_BDirs******/
/***********************/
#macro Show_BDirs(_bspline_array,Thickness)
 #local Nodes = dimension_size(_bspline_array,1);
 #local Node_Val=0;
 #while (Node_Val < Nodes)
  #if ( (_bspline_array[Node_Val+1]*t).t!=0)
  cone{
   V3(_bspline_array[Node_Val]),Thickness
   V3(_bspline_array[Node_Val]+_bspline_array[Node_Val+1]*(1+(_bspline_array[Node_Val+1]*t).t)),0
   }
  #else
  cone{
   V3(_bspline_array[Node_Val]),Thickness
   V3(_bspline_array[Node_Val]+_bspline_array[Node_Val+1]),0
   }
  #end
 #local Node_Val=Node_Val+2;
 #end
#end

/************* Calculators *************/

/***********************/
/******BSpline_Pos******/
/***********************/
//Parameters:
// 1. BSpline-Array
// 2. Mover 0 -> 1 => Start -> End of BSpline
//Returns:
// Position on BSpline
//Notes:
// Calculates my Bezier-Spline-Algorithm. Values larger than
// 1 or smaller than 0 will return position that are .01 units
// after/before the spline, using the direction at start/end
// of BSpline as indicator.
#macro BSpline_Pos(_b_spl_array,_nm)
 //Amount of interpolations
 #local _nodes = dimension_size(_b_spl_array,1)/2;
 //Calculations method (takes care of special cases)
 #local _max_nm=1;
 #local _a_nm=_nm;
 //Node_Mover-Corrections, keeps them in boundaries
 #if (_nm > 1) #local _a_nm=1; #local _max_nm=2; #end
 #if (_nm < 0) #local _a_nm=0; #local _max_nm=0; #end
 #if (_nm = 1) #local _a_nm=1; #local _max_nm=3; #end
 
 //Index of section:
 #local _i_sec=floor(_a_nm*(_nodes-1))*2;
 //Actual section's interpolation-value (nodemover)
 #local _a_nm=_a_nm*(_nodes-1)-floor(_a_nm*(_nodes-1));

 //Mover < 0
 #if (_max_nm=0)
  #local _Bezier_Pos=_b_spl_array[0]-vnormalize(V3(_b_spl_array[1]))*.0001;
 #end
 //Mover < 1
 #if (_max_nm=1)
   #if (_i_sec!=0)
     #local _Bezier_Pos=
        V3(_b_spl_array[_i_sec])*pow(1-_a_nm,3)+
        3*( V3(_b_spl_array[_i_sec+1])*(1+(_b_spl_array[_i_sec+1]*<0,0,0,1>).t)+V3(_b_spl_array[_i_sec]))*pow(1-_a_nm,2)*_a_nm-
        3*( V3(_b_spl_array[_i_sec+3])-V3(_b_spl_array[_i_sec+2]))*(1-_a_nm)*pow(_a_nm,2)+
        V3(_b_spl_array[_i_sec+2])*pow( _a_nm ,3);
   #else
     #local _Bezier_Pos=
        V3(_b_spl_array[_i_sec])*pow(1-_a_nm,3)+
        3*( V3(_b_spl_array[_i_sec+1])+V3(_b_spl_array[_i_sec]))*pow(1-_a_nm,2)*_a_nm-
        3*( V3(_b_spl_array[_i_sec+3])-V3(_b_spl_array[_i_sec+2]))*(1-_a_nm)*pow(_a_nm,2)+
        V3(_b_spl_array[_i_sec+2])*pow( _a_nm ,3);
   #end
 #end
 //Mover > 1
 #if (_max_nm=2)
  #local _Bezier_Pos=_b_spl_array[_nodes*2-2]+vnormalize(V3(_b_spl_array[_nodes*2-1]))*.01;
 #end
 //Mover = 1
 #if (_max_nm=3)
  #local _Bezier_Pos=_b_spl_array[_nodes*2-2];
 #end
 //Return the Position-Vector
 V3(_Bezier_Pos)
#end

/***********************/
/******BSpline_Dir******/
/***********************/
//Calculates the direction at a given position
#macro BSpline_Dir(_b_spl_array,_nm)
 //Parameters: Bezier-Spline-Array, Node_Mover (0->1)
 //Amount of interpolations
 #local _nodes = dimension_size(_b_spl_array,1)/2;
 #local _max_nm=0;
 #local _a_nm=_nm;
 //Node_Mover-Corrections, keeps them in boundaries and
 //makes notes for other calculations
 #if (_a_nm < 0) #local _a_nm=0; #end
 #if (_a_nm >= 1) #local _a_nm=1; #local _max_nm=1; #end
 #local _i_sec=floor(_a_nm*(_nodes-1))*2;
 //Actual section's interpolation-value (nodemover)
 #local _a_nm=_a_nm*(_nodes-1)-floor(_a_nm*(_nodes-1));
 #if (!_max_nm)
  #if (_i_sec)
   #local Tangent_1=V3(_b_spl_array[_i_sec+1]*(1+_b_spl_array[_i_sec+1]*<0,0,0,1>).t);
   #local Tangent_2=V3((_b_spl_array[_i_sec+2]-_b_spl_array[_i_sec+3])-(_b_spl_array[_i_sec]+_b_spl_array[_i_sec+1]*(1+_b_spl_array[_i_sec+1]*<0,0,0,1>).t));
   #local Tangent_3=V3(_b_spl_array[_i_sec+3]);
  #else
   #local Tangent_1=V3(_b_spl_array[_i_sec+1]);
   #local Tangent_2=V3((_b_spl_array[_i_sec+2]-_b_spl_array[_i_sec+3])-(_b_spl_array[_i_sec]+_b_spl_array[_i_sec+1]));
   #local Tangent_3=V3(_b_spl_array[_i_sec+3]);
  #end
  #local _Bezier_Dir=
    Tangent_1*pow(1-_a_nm,2)+
    Tangent_2*2*_a_nm*(1-_a_nm)+
    Tangent_3*pow(_a_nm,2);
 #end
 #if (_max_nm)
  #local _Bezier_Dir=V3(_b_spl_array[_nodes*2-1]);
 #end
 //Return-Value
 _Bezier_Dir
#end

/***********************/
/*****BSpline_Normal****/
/***********************/
//BSpline_Normal
//Usage: Call Macro with Spline and Node-Mover, returns normal-vector
#macro BSpline_Normal(_spl_array,_nm)

 #local BF_Node_Mover = _nm;
 //Calculating some needed positions on the spline
  #if (BF_Node_Mover >= Zero_Value)
   #local Node_M_1 = BF_Node_Mover-Zero_Value;
   #local Node_M_2 = BF_Node_Mover;
   #local Node_M_3 = BF_Node_Mover+Zero_Value;
  #end
  #if (BF_Node_Mover < Zero_Value)
   #local Node_M_1 = 0;
   #local Node_M_2 = BF_Node_Mover+pow(Zero_Value,2);
   #local Node_M_3 = BF_Node_Mover+Zero_Value;
   #end
  #if (BF_Node_Mover >= 1-Zero_Value)
   #local Node_M_1 = BF_Node_Mover-Zero_Value;
   #local Node_M_2 = BF_Node_Mover-pow(Zero_Value,2);
   #local Node_M_3 = 1;
  #end
 //Use calculated Node_Movers and calculate Positions
  #local Centri_Pos_1 = BSpline_Pos(_spl_array,Node_M_1);
  #local Centri_Pos_2 = BSpline_Pos(_spl_array,Node_M_2);
  #local Centri_Pos_3 = BSpline_Pos(_spl_array,Node_M_3);
   #local _veca=vnormalize(Centri_Pos_2-Centri_Pos_1);
   #local _vecb=vnormalize(Centri_Pos_3-Centri_Pos_2);
   
   //Begin with assumption that we don't need tp do extra calculations
   #local Again=false;
   //Same or inverse? If so, we do some extra calculations
   #if ((_veca.x = _vecb.x & _veca.y = _vecb.y & _veca.z = _vecb.z) | (_veca.x = -_vecb.x & _veca.y = -_vecb.y & _veca.z = -_vecb.z))
    #local Again=true;
   #end
  
  //In Loop, do steps in relation to actual Bezier-Spline
  #local Detail=1/(dimension_size(_spl_array,1)/2)*.05;
  //Counter to limit the loop
  #local C=0;
  #while (Again) 
   //Dependant on where we are, we move different node-movers and re-calculate appropriate positions/vectors
   #if (BF_Node_Mover >= Zero_Value)
    #local Node_M_3 = Node_M_3+Detail;
    #local Centri_Pos_3 = BSpline_Pos(_spl_array,Node_M_3);
    #local _vecb=vnormalize(Centri_Pos_3-Centri_Pos_2);
   #end
   #if (BF_Node_Mover < Zero_Value)
    #local Node_M_3 = Node_M_3+Detail;
    #local Centri_Pos_3 = BSpline_Pos(_spl_array,Node_M_3);
    #local _vecb=vnormalize(Centri_Pos_3-Centri_Pos_2);
   #end
   #if (BF_Node_Mover >= 1-Zero_Value)
    #local Node_M_1 = Node_M_1-Detail;
    #local Centri_Pos_1 = BSpline_Pos(_spl_array,Node_M_1);
    #local _veca=vnormalize(Centri_Pos_2-Centri_Pos_1);
   #end
      
   //Same or inverse again?
   #if ((_veca.x = _vecb.x & _veca.y = _vecb.y & _veca.z = _vecb.z) | (_veca.x = -_vecb.x & _veca.y = -_vecb.y & _veca.z = -_vecb.z))
    #local Again=true; #else
    #local Again=false; #end
   //Increment the counter
   #local C=C+1;
   //Finally, check if we have already looped to much (normally, 25 should be max!)
   #if (C=50) #local Again=false; #end
  #end

 #local _Centri_Vec_=vnormalize(vcross(vcross(Centri_Pos_2-Centri_Pos_1,Centri_Pos_2-Centri_Pos_3),Centri_Pos_2-Centri_Pos_1));
 //Return the normal
 (_Centri_Vec_)
#end


/***********************/
/*****BSpline_Length****/
/***********************/
//Calculates my Bezier-Spline-Algorithm
#macro BSpline_Length(_b_spl_array,_detail)
 #local _the_Spline=_b_spl_array;
 #local _The_Length=0;
 #local _BSPLength_Mover=0;
 #while (_BSPLength_Mover<=1)
  #local _Pos=BSpline_Pos(_the_Spline,_BSPLength_Mover);
  #ifdef (_oldPos)
   #local _The_Length=_The_Length+vlength(_Pos-_oldPos);  
  #end
  #local _oldPos=_Pos;
 #local _BSPLength_Mover=_BSPLength_Mover+1/(_detail-1);
 #end
 //Return
 _The_Length
#end

/***********************/
/*****BSpline_Align****/
/***********************/
//Function:
// Aligns an object (pointing towards -z) to align
// with a given BSpline, including banking and for
// animations.
//Parameters:
// 1. BSpline
// 2. Mover
// 3. Foresight
// 4. Banking Mod
//Returns:
// Transformation
//Notes:
// The higher Foresight is, the smoother the banking will
// take place. Foresight is added to the Mover and returns
// a position slightly ahead of the actual position. Pending
// on complexity of BSpline, values <.1 are sufficient.
#macro BSpline_Align(_Bspl_array,Mover,_Foresight,_Banking)
  //Preps
  #local _this_sky = <0,1,0>;
  //First, get Position on BSpline
  #local _Pos = BSpline_Pos(_Bspl_array,max(min(Mover,1),0));
  //Then, get some information on the curves
  #if (!_Foresight)
   //Foresight = 0 gives very precise results
   #local _this_forward = -vnormalize(BSpline_Dir(_Bspl_array,Mover));
   #local _form_forward = _this_forward;
  #else
   //Foresight != 0 yields smoother results
   #local _next_Pos = BSpline_Pos(_Bspl_array,Mover+_Foresight);
   #local _this_forward = -vnormalize(_next_Pos-_Pos);
   #local _form_Pos = BSpline_Pos(_Bspl_array,Mover-_Foresight);
   #local _form_forward = -vnormalize(_Pos-_form_Pos);
  #end
  #local _this_right = -vnormalize(vcross(_this_sky,_this_forward));
  #local _this_up = vnormalize(vcross(_this_right,_this_forward));
  //Now, right, up and forward provide the matrix for the
  //alignment. Now we just need the banking: difference
  //between the last two forward directions
  #local _bank_angle = degrees(acos(min(vdot(_form_forward,_this_forward),1)));
  //Gets the direction of Banking
  #local _bank_dir=(vdot(_this_sky,vcross(_form_forward,_this_forward))<0?1:-1);
  //Final transformation
  transform{
    rotate z*_bank_angle*_Banking*_bank_dir
    matrix < _this_right.x , _this_right.y , _this_right.z ,
               _this_up.x  ,   _this_up.y  ,   _this_up.z  ,
            _this_forward.x,_this_forward.y,_this_forward.z,
                 _Pos.x    ,     _Pos.y    ,     _Pos.z    >
    }
#end

/***********************/
/*****BSpline_Banking***/
/***********************/
//Function:
// Gets the Z-Rotation that is part of the BSpline_Align-
// Transformation.
//Parameters:
// 1. BSpline
// 2. Mover
// 3. Foresight
// 4. Banking Mod
//Returns:
// Banking-Angle
//Notes:
// Uses the same Algorithm like BSpline_Align
#macro BSpline_Banking(_Bspl_array,Mover,_Foresight,_Banking)
  #local _this_sky = <0,1,0>;
  #local _Pos = BSpline_Pos(_Bspl_array,max(min(Mover,1),0));
  #if (!_Foresight)
   #local _this_forward = -vnormalize(BSpline_Dir(_Bspl_array,Mover));
   #local _form_forward = _this_forward;
  #else
   #local _next_Pos = BSpline_Pos(_Bspl_array,Mover+_Foresight);
   #local _this_forward = -vnormalize(_next_Pos-_Pos);
   #local _form_Pos = BSpline_Pos(_Bspl_array,Mover-_Foresight);
   #local _form_forward = -vnormalize(_Pos-_form_Pos);
  #end
  #local _bank_angle = degrees(acos(min(vdot(_form_forward,_this_forward),1)));
  #local _bank_dir=(vdot(_this_sky,vcross(_form_forward,_this_forward))<0?1:-1);
  //Return
  (_bank_angle*_Banking*_bank_dir)
#end

/************* Extra *************/

/************************/
/*****BSpline2Linear*****/
/************************/
#macro BSpline2Linear(_b_bspline_array,Calc_Detail,Final_Nodes,Filename)

//The created Include-File with saved Spline-Data
#if (!strcmp(Filename,"bsplines.inc"))
 #debug "\n*!* Warning, bsplines.inc reserved! *!*\n*!* Renamed to bspline.spl!\n"
 #declare Filename = "bspline.spl"
#end

//Fragmenting the original Spline and calculating the length...
#local Sm_Array = array[Calc_Detail+1]
#local Bezier_Length = 0;
#local Sm_Array[0]=BSpline_Pos(_b_bspline_array,0)+<0,0,0,0>;

#local Smooth_Check = 1;
#while (Smooth_Check <= Calc_Detail)
 #local Clock = Smooth_Check/Calc_Detail;
 #local Sm_Array[Smooth_Check]=BSpline_Pos(_b_bspline_array,Clock)+<0,0,0,Clock>;
 #local Bezier_Length = Bezier_Length+(vlength(V3(Sm_Array[Smooth_Check]-Sm_Array[Smooth_Check-1])));
#local Smooth_Check = Smooth_Check+1;
#end

//We have the Length and we have an array of vectors. What do we do now?
//Using the Length and the Final_Nodes-Value, a length is calculated
//which defines the distance covered by every section. At those lengths, the
//position is saved. Voila! We have our new pseudo-Spline.

#local Section_Length = Bezier_Length/(Final_Nodes-1);
#local PS_Spline_Node = array[Final_Nodes]

//First point still needs to be first point, as well as last one 
 #local PS_Spline_Node[0]=Sm_Array[0];
 #local PS_Spline_Node[Final_Nodes-1]=Sm_Array[Calc_Detail];

//Length_Check is a integer which keeps track of the fragments passed, in
//order to never use two same nodes. The Length keeps track of the distance
//spanned between two fragments, supplying a method of tracking new pseudo-nodes.

#local Length = 0;
#local Length_Check = 1;
#local Node_Val = 1; 
#while (Length_Check <= Calc_Detail)
 #local Length = Length+(vlength(V3(Sm_Array[Length_Check]-Sm_Array[Length_Check-1])));
 //If-Section which covers the point at which new Node is found
 #if (Length >= Section_Length)
  #local PS_Spline_Node[Node_Val]=Sm_Array[Length_Check];
  #local Node_Val = Node_Val+1;
  #local Length = Length-Section_Length;
 #end
 //Preparing the loop to move on until all Fragments are passed
 #local Length_Check = Length_Check+1;
#end

/* Output */
#fopen Spline Filename write
#write (Spline "//File created by Tim Wenclawiak's Spline2Linear-Macro\n")
#write (Spline "//Original Bezier-Spline\n")
//The original-Spline
#write (Spline "#declare Bezier_Node = array[",str(dimension_size(_b_bspline_array,1),1,0),"]\n{\n")
#local Write = 0;
#while (Write < dimension_size(_b_bspline_array,1))
#write (Spline _b_bspline_array[Write],"\n")
#local Write=Write+1;
#end
#write (Spline "}\n")

//The pseudo-Spline
#write (Spline "\n//Linear-Interpolation-Data\n")

#write (Spline "#declare Linear_Node = array[",str(Final_Nodes,1,0),"]\n{\n")
#local Write = 0;
#while (Write < Final_Nodes)
#write (Spline PS_Spline_Node[Write],"\n")
#local Write=Write+1;
#end
#write (Spline "}\n")

//The Length of the Spline
#write (Spline "\n//Bezier-Length\n")
#write (Spline "#declare Bezier_Length = ",Bezier_Length,";\n")

/* End of Output */
#fclose Spline

/*End of Macro*/

#end

/****************************/
/*****BSpline2Linear_Ext*****/
/****************************/
#macro BSpline2Linear_Ext(_b_bspline_array,Calc_Detail,Final_Nodes,Filename,Spline_Name,Linear_Name)
  //The created Include-File with saved Spline-Data
  #if (strcmp(Filename,"bsplines.inc") = 0)
   #debug "\n*!* Warning, bsplines.inc reserved! *!*\n*!* Renamed to bspline.spl!\n"
   #declare Filename = "bspline.spl"
  #end
  
  //Fragmenting the original Spline and calculating the length...
  #local Sm_Array = array[Calc_Detail+1]
  #local Bezier_Length = 0;
  #local Sm_Array[0]=BSpline_Pos(_b_bspline_array,0)+<0,0,0,0>;
  
  #local Smooth_Check = 1;
  #while (Smooth_Check <= Calc_Detail)
   #local Mover = Smooth_Check/Calc_Detail;
   #local Sm_Array[Smooth_Check]=BSpline_Pos(_b_bspline_array,Mover)+<0,0,0,Mover>;
   #local Bezier_Length = Bezier_Length+(vlength(V3(Sm_Array[Smooth_Check]-Sm_Array[Smooth_Check-1])));
   #local Smooth_Check = Smooth_Check+1;
  #end
  
  //We have the Length and we have an array of vectors. What do we do now?
  //Using the Length and the Final_Nodes-Value, a length is calculated
  //which defines the distance covered by every section. At those lengths, the
  //position is saved. Voila! We have our new pseudo-Spline.
  #local Section_Length = Bezier_Length/(Final_Nodes-1);
  #local PS_Spline_Node = array[Final_Nodes]
  
  //First point still needs to be first point, as well as last one 
  #local PS_Spline_Node[0]=Sm_Array[0];
  #local PS_Spline_Node[Final_Nodes-1]=Sm_Array[Calc_Detail];
  
  //Length_Check is a integer which keeps track of the fragments passed, in
  //order to never use two same nodes. The Length keeps track of the distance
  //spanned between two fragments, supplying a method of tracking new pseudo-nodes.
  #local Length = 0;
  #local Length_Check = 1;
  #local Node_Val = 1; 
  #while (Length_Check <= Calc_Detail)
    #local Length = Length+(vlength(V3(Sm_Array[Length_Check]-Sm_Array[Length_Check-1])));
    //If-Section which covers the point at which new Node is found
    #if (Length >= Section_Length & Node_Val!=Final_Nodes)
      #local PS_Spline_Node[Node_Val]=Sm_Array[Length_Check];
      #local Node_Val = Node_Val+1;
      #local Length = Length-Section_Length;
    #end
    //Preparing the loop to move on until all Fragments are passed
    #local Length_Check = Length_Check+1;
  #end
  
  /* Output */
  #fopen Spline Filename write
    #write (Spline "//File created by Tim Wenclawiak's Spline2Linear-Extended-Macro\n")
    #write (Spline "//Original Bezier-Spline\n")
    
    //The original-Spline
    #write (Spline "#declare ",Spline_Name," = array[",str(dimension_size(_b_bspline_array,1),1,0),"]\n{\n")
    #local Write = 0;
    #while (Write < dimension_size(_b_bspline_array,1))
      #write (Spline _b_bspline_array[Write],"\n")
      #local Write=Write+1;
    #end
    #write (Spline "}\n")
    
    //The pseudo-Spline
    #write (Spline "\n//Linear-Interpolation-Data\n")
    #write (Spline "#declare ",Linear_Name," = array[",str(Final_Nodes,1,0),"]\n{\n")
    #local Write = 0;
    #while (Write < Final_Nodes)
      #write (Spline PS_Spline_Node[Write],"\n")
      #local Write=Write+1;
    #end
    #write (Spline "}\n")
    
    //The Length of the Spline
    #write (Spline "\n//Bezier-Length\n")
    #write (Spline "#declare Bezier_Length = ",Bezier_Length,";\n")
    /* End of Output */
  #fclose Spline
  /*End of Macro*/
#end

/***********************/
/******Linear_Pos*******/
/***********************/
//Usage: Call Macro with Linear-I-Array and Mover between 0 and 1 (beginning and end of spline)
#macro Linear_Pos(_Linear_Array,_LI_Mover)
 //Amount of Interpolations
 #local _Linear_Sections = dimension_size(_Linear_Array,1)-1;
 //Node_Mover-Corrections, keeps them in boundaries
 #if (_LI_Mover > 1) #local _LI_Mover=1; #end
 #if (_LI_Mover < 0) #local _LI_Mover=0; #end
 #if (_LI_Mover = 0)
  #local _Linear_Pos_ = V3(_Linear_Array[0]);
 #end
 #if (_LI_Mover = 1)
  #local _Linear_Pos_ = V3(_Linear_Array[_Linear_Sections]);
 #end
 #if (_LI_Mover > 0 & _LI_Mover < 1)
  #local _Actual = floor(_LI_Mover*_Linear_Sections);
  #local _Next = floor(_LI_Mover*_Linear_Sections)+1;
  #local _Time = _LI_Mover*(_Linear_Sections)-_Actual;
  #local _Linear_Pos_ = V3(_Linear_Array[_Actual]+(_Linear_Array[_Next]-_Linear_Array[_Actual])*_Time);
 #end
_Linear_Pos_
#end

/***********************/
/*****Linear_Mover******/
/***********************/
//Parameters:
// 1. Linear-Array
// 2. Mover-Value
//Returns:
// Mover-Value of original BSpline
//Notes:
// None
#macro Linear_Mover(_Linear_Array,_LI_Mover)
 //Amount of Interpolations
 #local _Linear_Sections = dimension_size(_Linear_Array,1)-1;
 //Node_Mover-Corrections, keeps them in boundaries
 #if (_LI_Mover > 1) #local _LI_Mover=1; #end
 #if (_LI_Mover < 0) #local _LI_Mover=0; #end
 #if (_LI_Mover = 0)
  #local _Linear_Mover_ = _Linear_Array[0].t;
 #end
 #if (_LI_Mover = 1)
  #local _Linear_Mover_ = _Linear_Array[_Linear_Sections].t;
 #end
 #if (_LI_Mover > 0 & _LI_Mover < 1)
  #local _Actual = floor(_LI_Mover*_Linear_Sections);
  #local _Next = floor(_LI_Mover*_Linear_Sections)+1;
  #local _Time = _LI_Mover*(_Linear_Sections)-_Actual;
  #local _Linear_Mover_ = _Linear_Array[_Actual].t+(_Linear_Array[_Next].t-_Linear_Array[_Actual].t)*_Time;
 #end
_Linear_Mover_
#end

/***********************/
/******Show_Linear*****/
/***********************/
#macro Show_Linear(__linear_array,_Thickness)
  #local _Size = dimension_size(__linear_array,1)-1;
  sphere{V3(__linear_array[0]),_Thickness}
  #local _Counter = 0;
  #while (_Counter < _Size)
   cylinder{V3(__linear_array[_Counter]),V3(__linear_array[_Counter+1]),_Thickness}
   sphere{V3(__linear_array[_Counter]),_Thickness}
  #local _Counter=_Counter+1;
  #end
#end
