// Scene description file:  QM_Alphabet.pov
///Alphabetic Quadratic Maps, by Dave Matthews, Minnesota West Community & Technical College   // 
/// May 17, 2006
/* The family of quadratic maps:  x_N+1 = C0 + C1*x_N + C2*x_N*x_N + C3*x_N*y_N + C4*y_N + C5*y_N*y_N
                                  y_N+1 = C6 + C7*x_N + C8*x_N*x_N + C9*x_N*y_N + C10*y_N + C11*y_N*y_N 
                                  
Some examples are given in:  http://mathworld.wolfram.com/StrangeAttractor.html

S. Sprott looked at the family of maps with coefficients in the range { -1.2, 1.2 } coded by letters:
A = -1.2, B = -1.1, ... Y = 1.2 (You can throw in Z = 1.3 if you want, in this program)

According to Sprott, as quoted at MathWorld, the 12-letter strings correspond, in about 1.6 % of the cases, 
to chaotic attractors.  

In many more cases, they will correspond to 1-dimensional loop attractors, or cluster point 
(a few isolated dot) attractors, and, most commonly, they will diverge, and you'll see a blank graph.

So if you try out letter combinations, don't be dismayed if you keep seeing nothing.  Eventually, you'll find 
some that produce pictures.  Alternatively, you can use my handy-dandy random attractor generator, QMA_Random.pov, 
that sorts out all of the divergent ones, and all with 8 or fewer cluster points, and uses "clock" to batch-generate 
lots of examples that you can sort through, to find the "good" ones. 

To zoom in on your attractor, adjust Fac (smaller numbers = closer zooms) and Cen (center of the screen) -- each
square in the background grid represents 4 units, so you can count out and adjust accordingly.

Amazingly, the sequence MATTHEWSDAVE makes a nice attractor ;-)   

(I'm using no light source, and an orthographic projection, so this is a genuine 2-dimensional misuse of POV-Ray.)
 
 */     
 
#default { finish{ambient 1} } // since there's no light



#declare L_A = array[12];        

// Here's where you enter the letters in the string.  Notice that they must be capital letters, and 
// surrounded by quotes.

#declare L_A[0] =  "M";
#declare L_A[1] =  "A";
#declare L_A[2] =  "T";
#declare L_A[3] =  "T";
#declare L_A[4] =  "H";
#declare L_A[5] =  "E";
#declare L_A[6] =  "W";
#declare L_A[7] =  "S";
#declare L_A[8] =  "D";
#declare L_A[9] =  "A";
#declare L_A[10] = "V";
#declare L_A[11] = "E";

#declare Fac = 9;       //ZOOM OUT FACTOR -- see the comments on the camera, below  -- SET THIS LARGE (20-30) FOR EXPLORATIONS
#declare Cen = <0, 6>;  //CENTER
#declare N_Of_Pts = 200000;  // last iteration
#declare Start = 80000;     // First recorded iteration, usually discard a bunch to get nearer the actual attractor
                            // Note that total number of dots = N_Of_Pts - Start 
#declare Rad = 0.003*Fac;  // BASE SIZE OF SPHERE; IF THE ATTRACTOR IS VERY "NARROW," (LOW FRACTAL DIMENSION) YOU MAY NEED TO INCREASE THIS
#declare E_Fac = 0.05;    // sphere expansion factor (increased size of last sphere, compared to first one)
#declare BailOut = 1000000; // BailOut (radius squared) since most maps are unbounded



#declare C_A = array[12];

#local pp = 0;

#while (pp < 12)
#declare C_A[pp] =  0.1*(asc(L_A[pp])-65) - 1.2;    // Here's where the letters get converted into coefficients
#local pp = pp + 1;
#end


#declare X0 = 0;     // Initial value -- this can be changed to observe different orbits/potential attractors
#declare Y0 = 0;


// while loop creating the attractor -- discard first bunch of points as not being in the attractor

#declare I = 0;      // "I" is the counter

#while ((I < N_Of_Pts) & (X0*X0 + Y0*Y0 < BailOut ) )   // "bail-out" radius necessary since most systems are unbounded
#if (I > Start)
sphere { 0, Rad  scale (1 + E_Fac*I/N_Of_Pts) pigment { color rgb <0.9*(I/N_Of_Pts)*(I/N_Of_Pts)*(I/N_Of_Pts),1.2*(1 - I/N_Of_Pts),0.3*(1 - I/N_Of_Pts)> }  translate 10*<X0,0,Y0>}     // The dots, colored and sized by how far along the cycle they occur
#end
#declare XN = C_A[0] + C_A[1]*X0 + C_A[2]*X0*X0 + C_A[3]*X0*Y0 + C_A[4]*Y0 + C_A[5]*Y0*Y0;   // The actual
#declare YN = C_A[6] + C_A[7]*X0 + C_A[8]*X0*X0 + C_A[9]*X0*Y0 + C_A[10]*Y0 + C_A[11]*Y0*Y0; // quadratic maps


#declare X0 = XN;
#declare Y0 = YN;
#declare I = I + 1;
#end  

// end of while loop  

cylinder { <-80*Fac,-1,  0>, <80*Fac,-1,  0>, .005*Fac pigment { color rgb z } } // x-axis
cylinder { <0, -1, -80*Fac>, <0, -1, 80*Fac>, .005*Fac pigment { color rgb z } } // y-axis

plane { y, -1 pigment { checker pigment { color rgb <0.7, 0.9, 1> } pigment { color rgb <0.93, 1, 1> } scale 4}  } //background grid


// Here's the orthographic camera.  The "4" and "3" are for the width to height ratio of the render.  Change
// these if you use other dimensions in your render!
camera {orthographic location <Cen.x,20,Cen.y> look_at <Cen.x,0,Cen.y> right 4*x*Fac up 3*y*Fac }   
  


background { color rgb 0.75 }   

/// Here's the "stamp" with the letter code

union {
text   { ttf "timrom.ttf", concat("C0:  ",L_A[0]),  0.1,  0  pigment { color 0.7*x } 
          scale 2.3 rotate 90*x translate <25,1,-20>}
text   { ttf "timrom.ttf", concat("C1:  ",L_A[1]),  0.1,  0  pigment { color 0.7*x } 
          scale 2.3 rotate 90*x translate <25,1,-22>}
text   { ttf "timrom.ttf", concat("C2:  ",L_A[2]),  0.1,  0  pigment { color 0.7*x } 
          scale 2.3 rotate 90*x translate <25,1,-24>}
text   { ttf "timrom.ttf", concat("C3:  ",L_A[3]),  0.1,  0  pigment { color 0.7*x } 
          scale 2.3 rotate 90*x translate <25,1,-26>}  
text   { ttf "timrom.ttf", concat("C4:  ",L_A[4]),  0.1,  0  pigment { color 0.7*x } 
          scale 2.3 rotate 90*x translate <25,1,-28>}
text   { ttf "timrom.ttf", concat("C5:  ",L_A[5]),  0.1,  0  pigment { color 0.7*x } 
          scale 2.3 rotate 90*x translate <25,1,-30>}
text   { ttf "timrom.ttf", concat("C6:  ",L_A[6]),  0.1,  0  pigment { color 0.7*x } 
          scale 2.3 rotate 90*x translate <25,1,-32>}
text   { ttf "timrom.ttf", concat("C7:  ",L_A[7]),  0.1,  0  pigment { color 0.7*x } 
          scale 2.3 rotate 90*x translate <25,1,-34>}
text   { ttf "timrom.ttf", concat("C8:  ",L_A[8]),  0.1,  0  pigment { color 0.7*x } 
          scale 2.3 rotate 90*x translate <25,1,-36>}
text   { ttf "timrom.ttf", concat("C9:  ",L_A[9]),  0.1,  0  pigment { color 0.7*x } 
          scale 2.3 rotate 90*x translate <25,1,-38>}
text   { ttf "timrom.ttf", concat("C10: ",L_A[10]),  0.1,  0  pigment { color 0.7*x} 
          scale 2.3 rotate 90*x translate <25,1,-40>}
text   { ttf "timrom.ttf", concat("C11: ",L_A[11]),  0.1,  0  pigment { color 0.7*x} 
          scale 2.3 rotate 90*x translate <25,1,-42>} 
  translate <12,0,10>        scale Fac/24
  translate <Cen.x, 0, Cen.y> }
