/*
  Persistence of Vision Ray Tracer Scene Description File

  Tierra v1.0 - A LOTW (landscape of the week) project, "a la Hormann".

  Features: 
  
  + isosurface terrain
  + isosurface water
  + media clouds
  + fog haze
  + CIE_Skylight sky-sphere & sunlight
  + faked isosurface trees
  + basic "rural" mesh houses.

  The generator is somewhat automated, with a few control variables and some 
  random seeds to vary some elements. 

  Notes:
   + aprox. scene scale : 1 unit = 1 km
   + 2:1 aspect ratio

  --   
  Jaime Vives Piqueres, Apr-May-Jun 2004.

*/
#version 3.5;


// ***************
// *** globals ***
// ***************
#declare use_rad=0;  // 0=off, 1=load, 2=save (be prepared to wait!!)
global_settings{
 max_trace_level 32 // raised for the cumulus containers
 assumed_gamma 1.5  // raised a bit for contrast
 #if (use_rad)
 radiosity{ 
  // save settings
  #if (use_rad=2) 
    pretrace_start .05 pretrace_end .01
    count 50 nearest_count 10 error_bound .5
    recursion_limit 1
    brightness 4
    save_file "tierra.rad"
    normal off
    media off
  #else
    // load settings  
    pretrace_start 1 pretrace_end 1 
    load_file "tierra.rad" 
    error_bound .5   
    recursion_limit 1  
    always_sample off  
    brightness 4
    normal off
    media off
  #end
 }
 #end
}
#default{texture{finish{ambient 0}}}


// *************************
// *** standard includes ***
// *************************
#include "colors.inc"
#include "textures.inc"
#include "functions.inc"


// *******************************
// *** CIE & LIGHTSYS includes ***
// *******************************
#include "CIE.inc"
#include "lightsys.inc"          
#declare Lightsys_Brightness=1;
// change white point to typical outdoor white balance
CIE_ColorSystemWhitepoint(CIE_ColSys,Daylight2Whitepoint(5500))
// spectral colors for pigments:
#include "rspd_aster.inc" // from the ASTER library
#include "rspd_jvp.inc"   // sampled by myself with a "EYE-ONE" spectometer


// **********************
// *** TIERRA globals ***
// **********************
#declare Landscape_Size=400; // side of the landscape square


// ***********************
// *** camera settings ***
// ***********************
// + always looks to the center of the terrain
#declare r_cam  =seed(7212); // random camera placement seed
#declare ra=360*rand(r_cam); // random rotation around the center of the terrain
#declare cam_pos=vrotate(<0,0,-Landscape_Size*.5*rand(r_cam)>,ra*y);
// manual tweaking: 
#declare cam_height=0.1017; // camera height over terrain/water
#declare cam_la_height  =0; // camera look_at height
#declare cam_la_dist   =.9; // camera look_at distance to the center (0-1)
#declare cam_offset=-4*z;


// **************************************
// *** elements switches (0=off/1=on) ***
// **************************************
#declare use_terrain   =1; // 2=test/rad, 3=plane
#declare use_water     =1; // 2=test/rad
#declare use_atmosphere=1;
#declare use_forest    =0; // 2= test cylinder
#declare use_buildings =0; 
#declare use_cirrus    =0;
#declare use_cumulus   =1; // 2=test objetc pattern
#declare use_ambient   =0; // quick alternative to radiosity


// ****************************
// *** sun & sky parameters ***
// ****************************
// skylight params:
#declare r_sun=    seed(179); // seed for random sun placement
#declare Current_Turbidity=3; // skylight turbidity (influences haze)
#declare Intensity_Mult   =1; // skylight intensity
#declare Max_Vertices  =1500; // skylight mesh resolution
#declare Sun_Size      =5000; // aparent size of the sun
// sun placement settings (uses rand seed if these are not specified)
//#declare Al=60;    
//#declare Az=90;
//#declare North=z;
// haze params:
#declare Haze_Height    =3; // keep it below the cumulus to avoid container artifacts  
#declare Haze_Distance=80;  // turbidity also affects distance
#declare Haze_Tint     =2;  // 0=Sky color, 1=Sun color, 2=Horizon color


// *****************************
// *** cloudscape parameters ***
// *****************************
#declare r_clouds   =seed(477); // clouds seed
#declare Clouds_Intervals   =1; // clouds media intervals (both types)
#declare Num_Cumulus      =200; // number of cumulus to place
#declare Cumulus_Comps     =64; // # of sph. components of the cloud pattern
#declare Cumulus_Height     =4; // starting height for the cumulus
#declare Cumulus_Type       =1; // 1=spherical, 2=disc-shaped
#declare Cirrus_Height     =24; // Cirrus shell height
#declare Cirrus_Level     =.75; // cloudiness for the cirrus shell
#declare Cirrus_Border    =.05; // softness level for the cirrus borders
#declare Cirrus_Flatness=.0007; // thickness of the cirrus shell
#declare Cirrus_Density=density{  // cirrus shape pattern
//  function{f_granite(x*.1,y*.1,z*.1)*f_agate(x,y,z)} scale 1
  function{f_crackle(x*2,y*2,z*2)*f_agate(x,y,z)} scale 1
//  function{f_bozo(x*4,y*4,z*4)*f_ripples(x,y,z)} scale 1
  warp{turbulence .9 lambda 3} 
  scale .0025
}  


// ************************
// *** water parameters ***
// ************************
#declare r_water =seed(878);  // water surface seed
#declare Water_Waves     =8;  // # of waves averaged on the surface pattern
#declare Water_Height   =.1;  // water level
#declare Water_Surface   =1;  // 1=isosurface, 2=plane with normal
#declare Foam_Level  =.0050;  // level of water foam near the coast


// **************************
// *** terrain parameters ***
// **************************
#declare r_terrf=  seed(310); // terrain formation seed
#declare r_terrl=  seed(498); // terrain location seed
#declare Terrain_Style    =2; // 0-5: arid,semi-arid,green,forest,artic,desert
#declare Terrain_Colors   =0; // 0-2, different sets of terrain colorations
#declare Terrain_Height   =8; // max height
#declare Terrain_MG       =3; // isosurface max gradient
#declare Terrain_Turb  =.333; // turbulence level for the terrain pattern
#declare Sand_Level    =.005; // sand level on the beach for any style
#declare Snow_Level     =2.0; // snow peaks start, over any terrain style
#declare Terrain_Fields   =0; // use fields with any style
#declare Fields_Scale   =.25; // scale of the fields pattern
#declare Fields_Metric    =3; // changes the metric of the crackle fields
#declare Terrain_Detail    =.000; // y depth of the small detail
#declare Terrain_Detail_Size=.02; // scale of the small detail
#declare Terrain_Pattern=pigment{ // terrain pattern (usable with color_map)
// function{f_ridge(x*.1,y*.1,z*.1,2.5,8,.1,0,1,3)*f_dents(x*.1,y*.1,z*.1)}
// function{f_ripples(x*.1,y*.1,z*.1)*f_dents(x*.1,y*.1,z*.1)}
 function{f_leopard(x*.001,y*.001,z*.001)*f_crackle(x*.1,y*.1,z*.1)}
// function{f_bumps(x*1,y*1,z*1)*f_dents(x*.1,y*.1,z*.1)}
// function{f_bumps(x*1,y*.001,z*1)*f_granite(x*.001,y*.001,z*.001)}
// function{f_marble(x*.0001,y*.0001,z*.0001)*f_dents(x*.5,y*.5,z*.5)}
}


// *************************
// *** forest parameters ***
// *************************
#declare r_trees      =seed(29);  // trees seed
#declare Num_Trees       =90000;  // number of trees to place
#declare Trees_Normal      =0.6;  // minimum terrain normal for placement
#declare Trees_MaxAltitude =2.0;  // max altitude for trees
#declare Trees_Area=<cam_pos.x*.97,cam_pos.z*.97,4>; // <Center_X,Center_Z,Rad>


// ****************************
// *** buildings parameters ***
// ****************************
#declare r_houses     =seed(29);  // houses seed
#declare Num_Houses        =500;  // number of houses to place
#declare Houses_Normal      =.9;  // minimum terrain normal for placement
#declare Houses_Area=<cam_pos.x*.98,cam_pos.z*.98,3.1>; //<Center_X,Center_Z,Rad>


// ************************
// *** all the includes ***
// ************************
#include "i_sky.inc"
#include "i_cirrus.inc"
#include "i_cumulus.inc"
#include "i_land.inc"
#include "i_water.inc"        
#include "i_forest.inc"
#include "i_buildings.inc"


// **************
// *** camera ***
// **************
// let it fall at a random location
#declare Norm=<0,0,0>;
#declare Start=cam_pos+<0,Terrain_Height+.1,0>;
#if (use_terrain)
#declare Inter=trace(terrain,Start,-1*y,Norm);
#else
#declare Inter=trace(water,Start,-1*y,Norm);
#end
camera{ 
 location Inter+cam_height*y+cam_offset
 up 0.000018*y right 0.000036*x
 direction 0.000035*z
 look_at
<Inter.x*cam_la_dist,Inter.y+cam_la_height,Inter.z*cam_la_dist>+cam_offset
}
