
/**********************************************************************************/
// Persistence of Vision Ray Tracer Scene Description File
// File name   : Geolandscape_01.pov
// Version     : 3.6
// Description : Makes isosurface landscapes
//               uses heightfields to test same landscape (faster but not accurate).
//               A folded and faulted schist landscape
// Date        : May 2004
// Author      : Mike Williams
// Compiled    : Thomas A. M. de Groot 
 
/**********************************************************************************/

#include "rand.inc"
#include "functions.inc"
#include "colors.inc"
#include "TdG_Textures.inc"

#declare Radiosity = false;

global_settings {
  #if (Radiosity)
  radiosity {
    pretrace_start 0.08           // start pretrace at this size
    pretrace_end   0.04           // end pretrace at this size
    count 35                      // higher -> higher quality (1..1600) [35]
    nearest_count 5               // higher -> higher quality (1..10) [5]
    error_bound 1.8               // higher -> smoother, less accurate [1.8]
    recursion_limit 3             // how much interreflections are calculated (1..5+) [3]
    low_error_factor .5           // reduce error_bound during last pretrace step
    gray_threshold 0.0            // increase for weakening colors (0..1) [0]
    minimum_reuse 0.015           // reuse of old radiosity samples [0.015]
    brightness 0.8                  // brightness of radiosity effects (0..1) [1]

    adc_bailout 0.01/2
    normal on                   // take surface normals into account [off]
    //media on                    // take media into account [off]
    //save_file "file_name"       // save radiosity data
    //load_file "file_name"       // load saved radiosity data
    always_sample off           // turn sampling in final trace off [on]
    //max_sample 1.0              // maximum brightness of samples
  }
  #end
  noise_generator 2
}

/*
Landscape parameters:
- GeoLandscape_01-1: Wrinkling=1.5; Denting=0.05
- Geolandscape_01-2: Wrinkling=1.0; Denting=0.5
- Geolandscape_01-3: Wrinkling=0.1; Denting=0.005
*/

#declare Loc = 4;					// location of the camera and primary scaling
#declare Slope = true;		// true = slope & altitude; false = gradient texture
#declare Wrinkling = 0.1;	// higher values give sharper features (1.5 is excellent)
#declare Denting = 0.005;	  // higher values increases the 'peakiness', a needle-like erosion 
#declare Iso = true;			// true for isosurface; false for heightfield
#declare Xscale = 3;			// increase size of landscape (not scale!) for X
#declare Zscale = 3;			// increase size of landscape (not scale!) for Z
#declare Seed = 92322;		// a random seed
#declare Rmin = 0;
#declare Rmax = 1;
// ----------------------------------------

#declare G =
pigment {
  #if (Slope)
    slope {<0.0, 0.6, 0.0> , 0.0 , 1.0
      altitude <0.0, 0.4, 0.0> , 0.0 , 1.0
    }
  #else
    gradient y
  #end
  colour_map {
    [0.00 rgb <0.1,0.4,0.1>]
    [0.10 rgb <0.5,0.4,0.3>]
    [0.35 rgb 0.50]
    [0.65 rgb 0.75]
    [1.00 rgb 0.90]
  }
  //translate -0.05*y
}

// a folded and faulted landscape:
#declare P1=function {pattern {wrinkles warp {turbulence <0.1, 5, 0.3> octaves 3 omega 0.5 lambda 1} scale 0.5}}
#declare P2=function {pattern {dents warp {turbulence <2, 0.9, 0.9> octaves 5 omega 0.5 lambda 2} scale 0.1}}
#declare P3=function {f_noise3d(x,y*2,z)*1}
#declare P=function {  P3(x, y, z)
										 * P3(x*0.2, 0, z*0.5)
                     - P1(x, y, z)*Wrinkling
                     + P2(x, y, z)*Denting 
                    }

#if (Iso)
isosurface {
  function {y - P(x, 1, 1-z)}
  //open  
  max_gradient 10
  contained_by {box {<0.5-Xscale*0.5, -1, 0.5-Zscale*0.5>, <0.5+Xscale*0.5, 1, 0.5+Zscale*0.5>}}    
  translate <-0.5, 0, -0.5>
  scale <Loc, 1, Loc>
  pigment {G}
  //translate -Loc/2*x
}

#else
height_field{
  function 300, 300 {P((x-0.5)*Xscale+0.5, 0, (y-0.5)*Zscale+0.5)}  
  translate <-0.5, 0, -0.5>
  scale <Loc*Xscale, 1, Loc*Zscale>
  pigment {G}
  //translate Loc/2*x
}

#end


//Water:
box { <-1, -10, -1>, <1, -1, 1>
  scale <Loc*Xscale, 1, Loc*Zscale>
  TdG_NorthSeaWater (
  		      0.6, // filter
                      0.8, // transparency
                      3,   // ScaleX
                      2,   // ScaleY
                      1,   // ScaleZ
                      1,   // fade distance
                      3    // fade power
                    )
}

camera {
  //location  <0, 1, -1.1*Loc> 
  location  <0, 1, -Loc*(Zscale/2)> 
  look_at   <0, 0, 0> 
  right     x*image_width/image_height
  angle     54
}

sky_sphere {
  pigment {
    function {abs(y)}
    color_map {[0.0 color blue 0.9] [1.0 color rgb 1]}
  }
}

light_source {<-1000,2000,-1000> colour rgb 1*1.5}
light_source {<1000,2000,1000> colour rgb 1 shadowless}

fog {
  fog_type 1
  distance 50
  color rgb <0.5, 0.5, 0.9>
}
