
/**********************************************************************************/
// Persistence of Vision Ray Tracer Scene Description File
// File name   : Geolandscape_01.pov
// Version     : 3.6
// Description : Makes isosurface landscapes
//               uses heightfields to test same landscape (faster but not accurate).
//               A folded and faulted schist landscape
// Date        : May 2004
// Author      : Mike Williams
// Compiled    : Thomas A. M. de Groot 
 
/**********************************************************************************/

#include "rand.inc"
#include "functions.inc"
#include "colors.inc"
#include "TdG_Textures.inc"

#declare Radiosity = false;

global_settings {
  #if (Radiosity)
  radiosity {
    pretrace_start 0.08           // start pretrace at this size
    pretrace_end   0.04           // end pretrace at this size
    count 35                      // higher -> higher quality (1..1600) [35]
    nearest_count 5               // higher -> higher quality (1..10) [5]
    error_bound 1.8               // higher -> smoother, less accurate [1.8]
    recursion_limit 3             // how much interreflections are calculated (1..5+) [3]
    low_error_factor .5           // reduce error_bound during last pretrace step
    gray_threshold 0.0            // increase for weakening colors (0..1) [0]
    minimum_reuse 0.015           // reuse of old radiosity samples [0.015]
    brightness 0.8                  // brightness of radiosity effects (0..1) [1]

    adc_bailout 0.01/2
    normal on                   // take surface normals into account [off]
    //media on                    // take media into account [off]
    //save_file "file_name"       // save radiosity data
    //load_file "file_name"       // load saved radiosity data
    always_sample off           // turn sampling in final trace off [on]
    //max_sample 1.0              // maximum brightness of samples
  }
  #end
  noise_generator 2
}

/*
Landscape parameters:
- GeoLandscape_04-1: no P1; no P2; Noise=-1.5; H=0.1; Lacunarity=10; Octaves=3; Offset=0.6; Gain=1.2; Generator=2

*/

#declare Loc = 4;					// location of the camera and primary scaling
#declare Slope = true;		// true = slope & altitude; false = gradient texture
#declare SlopeValue = 0.7;
// P1:
#declare Wrinkling = 5.0;	// higher values give sharper features
#declare Wturb = <0.1, 0.5, 0.3>;			// turbulence in wrinkles warp
#declare Wscale = <0.5, 0.1, 0.1>;	// wrinkling scale
#declare Wrotate = <0, 45, 0>;			// wrinkling rotation
// P2:
#declare Denting = 0.5;	  // amount of denting; higher values increases the 'peakiness', a needle-like erosion 
#declare Dturb = <0.9, 0.9, 0.9>;		// turbulence in dents warp
#declare Dscale = <0.1, 0.1, 0.1>;			// denting scale
#declare Drotate = <0, 0, 0>;				// denting rotation
// P3:
#declare Noise = 0.5;
// P4:
#declare H = 0.1;
#declare Lacunarity = 10;
#declare Octaves = 3;
#declare Offset = 0.6;
#declare Gain = 1.2;
#declare Generator = 2;

#declare Iso = true;			// true for isosurface; false for heightfield
#declare Xscale = 3;			// increase size of landscape (not scale!) for X
#declare Zscale = 3;			// increase size of landscape (not scale!) for Z
#declare Seed = 92322;		// a random seed
#declare Rmin = 0;
#declare Rmax = 1;
// ----------------------------------------

#declare Landtex =
pigment {
  #if (Slope)
    slope {<0.0, SlopeValue, 0.0> , 0.0 , 0.5
      altitude <0.0, 1-SlopeValue, 0.0> , -2 , 1  // bounding box goes from -2 to 1
      // note: put both slope and altitude ranges to 0 and 1, and you get a kind of water edge with algae
    }
  #else
    gradient y
  #end
  #if (SlopeValue < 0.5)
    colour_map {
      [0.00 rgb <0.1,0.4,0.1>]
      [0.25 rgb <0.5,0.4,0.3>]
      [0.50 rgb 0.50]
      [0.75 rgb 0.60]
      [1.00 rgb 0.70]
    }
  #else
    colour_map {
      [0.00 rgb 0.70]
      [0.25 rgb 0.60]
      [0.50 rgb 0.50]
      [0.75 rgb <0.5,0.4,0.3>]
      [1.00 rgb <0.1,0.4,0.1>]
    }
  #end
}

// a folded and faulted landscape:
#declare P1=function {f_wrinkles(x,y,z)}//pattern {wrinkles warp {turbulence Wturb octaves 3 omega 0.5 lambda 1} scale Wscale rotate Wrotate}}
#declare P2=function {f_dents(x,y,z)}//pattern {dents warp {turbulence Dturb octaves 5 omega 0.5 lambda 2} scale Dscale rotate Drotate}}
#declare P3=function {f_noise3d(x,y,z)*Noise}
#declare P4=function {f_ridged_mf(x,y,z, H, Lacunarity, Octaves, Offset, Gain, Generator) }
#declare P=function {  P4(x, y, z)
										 + P3(x, y*2, z)
                     * P1(x, 0, z)*Wrinkling
                     //+ P2(x, y, z)*Denting 
                    }

#if (Iso)
isosurface {
  function {y - P(x, 1, 1-z)}
  //open  
  max_gradient 10
  contained_by {box {<0.5-Xscale*0.5, -2, 0.5-Zscale*0.5>, <0.5+Xscale*0.5, 2, 0.5+Zscale*0.5>}}    
  translate <-0.5, 0, -0.5>
  scale <Loc, 1, Loc>
  pigment {Landtex}
  //translate -Loc/2*x
}

#else
height_field{
  function 800, 800 {P((x-0.5)*Xscale+0.5, 0, (y-0.5)*Zscale+0.5)}  
  translate <-0.5, 0, -0.5>
  scale <Loc*Xscale, 1, Loc*Zscale>
  pigment {Landtex}
  //translate Loc/2*x
}

#end


//Water:
box { <-1, -10, -1>, <1, -1, 1>
  scale <Loc*Xscale, 1, Loc*Zscale>
  TdG_NorthSeaWater (
  		      0.6, // filter
                      0.8, // transparency
                      3,   // ScaleX
                      2,   // ScaleY
                      1,   // ScaleZ
                      1,   // fade distance
                      3    // fade power
                    )
}
box { <-1, -10, -1>, <1, -11, 1>
  scale <Loc*Xscale, 1, Loc*Zscale>
  pigment {DarkTurquoise}
}

camera {
  //location  <0, 1, -1.1*Loc> 
  location  <0, 1, -Loc*(Zscale/2)> 
  look_at   <0, 0, 0> 
  right     x*image_width/image_height
  angle     54
}

sky_sphere {
  pigment {
    function {abs(y)}
    color_map {[0.0 color blue 0.9] [1.0 color rgb 1]}
  }
}

light_source {<-1000,2000,-1000> colour rgb 1*1.5}
light_source {<1000,2000,1000> colour rgb 1 shadowless}

fog {
  fog_type 1
  distance 50
  color rgb <0.5, 0.5, 0.9>
}
