///////////////////////////////////////////////////////////////////////////////
// Glassware 
//
// Copyright (c) 2004 Michael Raiford
//
// Pieces of glassware, from basic to stemware glasses. 
// Each piece is defined as a macro with various parameters changing it's
// overall appearance. Some standard texture macros are also included with 
// these macros. Each of these have an accompanying macro defined with _Liquid
// which creates an object inside the glass for the "liquid"
// Each liquid macro has an amount which is the percentage full (100% being
// all the way at the top)
///////////////////////////////////////////////////////////////////////////////

#include "consts.inc"

#ifndef(epsilon)
    #declare epsilon = 1.0e-10;
#end

///////////////////////////////////////////////////////////////////////////////
// ClearGlassMaterial
//
// A plain material for clear glass. 
//
// c = color of the glass. Filter is changed to 1.0 inside the texture 
// definition to create a transparent glass object tinted with the color given
///////////////////////////////////////////////////////////////////////////////
#macro ClearGlassMaterial(c)

    material
    {
        texture
        {
            pigment {
                color c filter 1.0
                quick_color c
            }
            
            finish {
                ambient 0
                diffuse 0
                conserve_energy
                specular 1.0
                roughness 0.005
                
                reflection
                {
                    0.10, 0.005
                    fresnel on
                }
            }
        }
        
        interior
        {
            ior Quartz_Glass_Ior
        }       
    }
    
    photons
    {
        target
        reflection on
        refraction on
        collect off
    }
    
#end


///////////////////////////////////////////////////////////////////////////////
// CylindricalGlass2
//-----------------------------------------------------------------------------
// Cylindrical Glass / Fancy Base
//
// Parameters
//
// r - Radius of the glass
// th - Thickness of the walls of the glass
// h - Height of the glass
//
// Notes: Glass will be created with the bottom at origin. When using a 
// refracting texture, be sure to set max_trace_level to at least 100 to get
// good texturing.
///////////////////////////////////////////////////////////////////////////////
#macro CylindricalGlass2(r, th, h)

    // This will be differenced from the solid portion of the glass
    #local glass_negative = merge
    {
        cylinder { <0, 0.25*0.5, 0>, <0, h+epsilon, 0>, r - th }
        sphere { <0, 0.5, 0>, 0.5 scale <2*(r-th), 0.25, 2*(r-th)> }
        torus { r, 0.1 translate -0.2*y }    
    }

    // This is the actual glass itself. 
    #local glass_positive = merge
    {
        torus { r - 0.15, 0.15 }
        cylinder { <0, -0.15, 0>, <0, 0, 0>, r-0.15 }

        #local cos_t = vdot(vnormalize(<0.1, 0>), vnormalize(<r-(r-0.15), 0.2>));

        #local bot_r = abs(vrotate(0.1*x, z*degrees(acos(cos_t))).x -r);
        #local bot_t = vrotate(0.1*x, z*degrees(acos(cos_t))).y -0.2;

        // bottom modelling
        cylinder { <0, bot_t-epsilon, 0>, <0, -0.2, 0>, bot_r }
        cylinder { <0, -0.2+epsilon, 0>, <0, -0.2-0.05-epsilon, 0>, r-0.1-0.05}
        torus { r - 0.1 - 0.05, 0.05 translate y*(-0.2) }
    
        // top modelling
        cylinder { <0, 0, 0>, <0, h - th/2, 0>, r }
        torus { r-th/2, th/2 translate <0, h-th/2, 0> }
    
    }

    // Subtract the negative objects from the positive glass
    difference { 
        object {glass_positive}
        object {glass_negative}
        translate y*(0.2+0.05)
    }
    
#end

#macro CylindricalGlass2_Liquid(r, th, h, amt)

    // This will be differenced from the solid portion of the glass
    merge
    {
        cylinder { <0, 0.25*0.5, 0>, <0, h*amt-epsilon, 0>, r - th + epsilon }
        sphere { <0, 0.5, 0>, 0.5 scale <2*(r-th+epsilon), 0.25+epsilon, 2*(r-th+epsilon)> }
        translate y*(0.2+0.05)
    }
    
#end