/*
	Name: 2D-compositing.inc
	Version: 0.5
	Coded by: Hugo Asm
	
	Use this with POV-Ray 3.5 or above. Feel free to modify. If you sort out the bugs or add some new features,
	I'd be very interested in getting some feedback. You can email me at: huas@post3.tele.dk


	This file contain the following macros:

		Blending modes. The variable "Percent" should be a float-value from 0-1):
		
			Normal   ( Image_1, Image_2, Percent ) // Images are averaged
			Add      ( Image_1, Image_2, Percent ) // Image 2 is added
			Subtract ( Image_1, Image_2, Percent ) // Image 2 is subtracted
			Multiply ( Image_1, Image_2, Percent ) // Formula is A*(B*Percent) and will produce a darker image if Percent < 1
			Lighten  ( Image_1, Image_2, Percent ) // Compare the images and choose the lighter value for each color channel 
			Darken   ( Image_1, Image_2, Percent ) // BETA version. Should compare the images and choose the darker value for each color channel
			Screen   ( Image_1, Image_2, Percent ) // Both images are inversed, multiplied with each other, then the result is inversed.
			Overlay  ( Image_1, Image_2, Percent ) // BETA version, but works almost like common paintprograms
			Alpha    ( Image_1, Image_2, Image_3 ) // Image 1 & 2 are blended in "normal mode" with image 3 (which is the alpha map)

	You have access to this useful variable:
	
		One_Pixel    // Vector that gives you the distance in POV-units between each pixel on screen.
		             // It  *only works*  when camera from this file is used. 

	You can specify the screen pixel position where the result will be displayed:
	
		#declare Position =<X,Y>; // This variable can be set by you. X & Y are measured in pixels.


	You MUST specify the resolution of each image that you use (POV-Ray can't determine this based on the image-files)
	otherwise a default resolution will be used (default is 640*480).
*/


#declare One_Pixel = 2/<image_width,image_height,1>;
#ifndef ( Image_1_Resolution )	#declare Image_1_Resolution = <640,480>;	#end
#ifndef ( Image_2_Resolution )	#declare Image_2_Resolution = <640,480>;	#end
#ifndef ( Image_3_Resolution )	#declare Image_3_Resolution = <640,480>;	#end
#ifndef ( Position )			#declare Position = <0,0>;					#end

// =========================
// ==== Blending modes =====
// =========================

// Normal blend
#macro Blend_Normal( Image_1, Image_2, Percent )
	#declare Red_Component=  function { Image_1(x,y,z).red  *(1-Percent)  +  Image_2(x,y,z).red  *Percent };
	#declare Green_Component=function { Image_1(x,y,z).green*(1-Percent)  +  Image_2(x,y,z).green*Percent };
	#declare Blue_Component= function { Image_1(x,y,z).blue *(1-Percent)  +  Image_2(x,y,z).blue *Percent };
	Show_Me()
#end

// Add (image 2 is added by Percent)
#macro Blend_Add( Image_1, Image_2, Percent )
	#declare Red_Component=  function { min( Image_1(x,y,z).red    +  Image_2(x,y,z).red  *Percent ,1) };
	#declare Green_Component=function { min( Image_1(x,y,z).green  +  Image_2(x,y,z).green*Percent ,1) };
	#declare Blue_Component= function { min( Image_1(x,y,z).blue   +  Image_2(x,y,z).blue *Percent ,1) };
	Show_Me()
#end

// Subtract (image 2 is subtracted by Percent)
#macro Blend_Subtract( Image_1, Image_2, Percent )
	#declare Red_Component=  function { max( Image_1(x,y,z).red    -  Image_2(x,y,z).red  *Percent ,0) };
	#declare Green_Component=function { max( Image_1(x,y,z).green  -  Image_2(x,y,z).green*Percent ,0) };
	#declare Blue_Component= function { max( Image_1(x,y,z).blue   -  Image_2(x,y,z).blue *Percent ,0) };
	Show_Me()
#end

// Multiply
#macro Blend_Multiply( Image_1, Image_2, Percent )
	#declare Red_Component=  function { Image_1(x,y,z).red    *  (1-(1-Image_2(x,y,z).red)  *Percent) };
	#declare Green_Component=function { Image_1(x,y,z).green  *  (1-(1-Image_2(x,y,z).green)*Percent) };
	#declare Blue_Component= function { Image_1(x,y,z).blue   *  (1-(1-Image_2(x,y,z).blue) *Percent) };
	Show_Me()
#end

// Lighten (compare images and choose the lighter value for each color channel)
#macro Blend_Lighten( Image_1, Image_2, Percent )
	#declare Red_Component=  function { max( Image_1(x,y,z).red  , Image_2(x,y,z).red  *Percent ) };
	#declare Green_Component=function { max( Image_1(x,y,z).green, Image_2(x,y,z).green*Percent ) };
	#declare Blue_Component= function { max( Image_1(x,y,z).blue , Image_2(x,y,z).blue *Percent ) };
	Show_Me()
#end

// Darken (compare images and choose the darker value for each color channel)
#macro Blend_Darken( Image_1, Image_2, Percent )
	#declare Red_Component=  function { min( Image_1(x,y,z).red,   Image_2(x,y,z).red  *Percent ) };
	#declare Green_Component=function { min( Image_1(x,y,z).green, Image_2(x,y,z).green*Percent ) };
	#declare Blue_Component= function { min( Image_1(x,y,z).blue,  Image_2(x,y,z).blue *Percent ) };
	Show_Me()
#end

// Screen (both images are inversed, multiplied by each other, then the result is inversed)
#macro Blend_Screen( Image_1, Image_2, Percent )
	#declare Red_Component=  function { 1-(1-Image_1(x,y,z).red)    *  (1-Image_2(x,y,z).red  *Percent) };
	#declare Green_Component=function { 1-(1-Image_1(x,y,z).green)  *  (1-Image_2(x,y,z).green*Percent) };
	#declare Blue_Component= function { 1-(1-Image_1(x,y,z).blue)   *  (1-Image_2(x,y,z).blue *Percent) };
	Show_Me()
#end

// Overlay (BETA version; almost identical to PaintShopPRO)
#macro Blend_Overlay( Image_1, Image_2, Percent )

	#local Percent = Percent + Percent*.6; // compensate for difference between my code and PaintShopPRO.. BETA!

	#declare Red_Component=function { max(
		
		(1-Image_1(x,y,z).red) * ( // Underlying layer decides strength of multiply mode. Low value = high strength
			Image_1(x,y,z).red * (1-(1-Image_2(x,y,z).red)*Percent)) // Multiply mode
		+
		
		Image_1(x,y,z).red * ( // Underlying layer decides strength of screen mode. High value = low strength
			1-(1-Image_1(x,y,z).red) * (1-Image_2(x,y,z).red*Percent)) // Screen mode

	,0)};

	// Same goes for green and blue component...
	#declare Green_Component=function { max( 
		
		(1-Image_1(x,y,z).green) * (Image_1(x,y,z).green * (1-(1-Image_2(x,y,z).green)*Percent))
		+
		Image_1(x,y,z).green * (1-(1-Image_1(x,y,z).green) * (1-Image_2(x,y,z).green*Percent))

	,0)};   
    
	#declare Blue_Component=function { max( 
	
		(1-Image_1(x,y,z).blue) * (Image_1(x,y,z).blue * (1-(1-Image_2(x,y,z).blue)*Percent))
		+
		Image_1(x,y,z).blue * (1-(1-Image_1(x,y,z).blue) * (1-Image_2(x,y,z).blue*Percent))

	,0)};

	Show_Me()
#end

// Alpha blend (Alpha map can be in colors)
// Blend mode NORMAL. Formula is (A*(1-alpha) + B*alpha)
#macro Blend_Alpha( Image_1, Image_2, Alpha )
	#declare Red_Component=  function { Image_1(x,y,z).red  *(1-Alpha(x,y,z).red)    +  Image_2(x,y,z).red  *Alpha(x,y,z).red   };
	#declare Green_Component=function { Image_1(x,y,z).green*(1-Alpha(x,y,z).green)  +  Image_2(x,y,z).green*Alpha(x,y,z).green };
	#declare Blue_Component= function { Image_1(x,y,z).blue *(1-Alpha(x,y,z).blue)   +  Image_2(x,y,z).blue *Alpha(x,y,z).blue  };
	Show_Me()
#end


// ======================================================
// ==== Setting up camera and show the result =====
// ======================================================

#macro Show_Me()
	camera { orthographic right 1*x location -1*z look_at 0 angle 90 }

	box { 0, One_Pixel * Displayed_Resolution
		pigment { average
			pigment_map {
				[ function { Red_Component(x,y,z) } color_map { [ 0 rgb 0 ] [ 1 rgb 3*x ] } ]
				[ function { Green_Component(x,y,z) } color_map { [ 0 rgb 0 ] [ 1 rgb 3*y ] } ]
				[ function { Blue_Component(x,y,z) } color_map { [ 0 rgb 0 ] [ 1 rgb 3*z ] } ]
			}
			scale One_Pixel * Displayed_Resolution
		}

		finish { ambient 1 }
		translate <-1,-1,0> + Position*One_Pixel
	}
	#undef Red_Component
	#undef Green_Component
	#undef Blue_Component
#end