//Penrose Tile Include File By Dave Matthews  

//Based on the work of 
//John VanSickle from his excellent IRTC entry of 1999,  "Boxer Solves A Problem"
// http://www.irtc.org/anims/1999-07-15.html   

//(It's his method, I just extracted it into an include file, and modified it so that it
//creates wire frames and rhombs as well as darts).
//Elegant recursion courtesy of John VanSickle.  
//Clumsy recursion courtesy of Dave Matthews. 

//This Include file contains 2 different macros: PenroseRhombs, for creating the 
//"Double Rhomboid" (Type B) Penrose Tiling, and PenroseDarts, for creating the
//"Kites and Darts" (Type A) Penrose Tiling.  Either macro will generate a mesh decagon
//of designated radius, as well as a csg cylinder "wire frame" for the borders of the
//tiles.  If you wish just the mesh part, be aware that it won't look like a classic
//Penrose Tiling, since adjacent congruent polygons will "blend together." 

//The meshes are created with the macros: MeshRhombs and MeshDarts, the wire frames are
//created with the macros:  WireRhombs and WireDarts.  

//User Options:  For the mesh, the user may predefine textures (not pigments, these are 
//triangles, and need pre-defined textures) of the two tiles, under the names:  Tile_Tex_1 and 
//Tile_Tex_2 (defaults are provided).  For the wire frame the user predefine the texture (for 
//simplicity, again, texture, not pigment) of the cylinders used for the frame, under the name,
//Wire_Tex.  The user may also predefine Wire_Width, which is used to determine the thickness
//of the cylinders, relative to their length.  The default is 0.03.



//Summary:

//PenroseDarts(Radius,Depth) 
//MeshDarts(Radius,Depth)
//WireDarts(Radius,Depth)

//PenroseRhombs(Radius,Depth)
//MeshRhombs(Radius,Depth)
//WireRhombs(Radius,Depth)

//Optional:
//Tile_Tex_1 = texture { }
//Tile_Tex_2 = texture { }
//Wire_Tex = texture { }
//Wire_Width = float 





#declare phi = (sqrt(5) - 1)/2;

#ifndef (Tile_Tex_1)
  #declare Tile_Tex_1 = 
       texture { pigment { color rgb <1, 0, 0> } };
#end // #ifndef

#ifndef (Tile_Tex_2)
   #declare Tile_Tex_2 = 
       texture { pigment { color rgb <0, 0, 1> } };
#end // #ifndef

#ifndef (Wire_Tex)
   #declare Wire_Tex = 
        texture { pigment { color rgb <1, 1, 0> } };
#end // #ifndef

#ifndef (Wire_Width)
   #declare Wire_Width = .03;
#end // #ifndef






//Macros for creating the "Kites and Darts" or Type A Penrose Tiling:



//Mesh Triangles

#macro Slim_Dart(A,B,C,D)
  #if (D=0)
    triangle { A,B,C texture { Tile_Tex_1 } }
  #else
    Slim_Dart(B,C,A+(C-A)*phi,D-1)
    Slim_Dart(B,B+(A-B)*phi,A+(C-A)*phi,D-1)
    Wide_Dart(A,A+(C-A)*phi,B+(A-B)*phi,D-1)
  #end
#end  //Slim_Dart

  
#macro Wide_Dart(A,B,C,D)
  #if (D=0)
    triangle { A,B,C texture {Tile_Tex_2 } }
  #else
    Slim_Dart(A,A+(B-A)*phi,C,D-1)
    Wide_Dart(B,C,A+(B-A)*phi,D-1)
  #end
#end    //Wide_Dart


//Wires

#macro SlimWires_Dart(A,B,C,D)
  #if (D=0)
    #local W=vlength(B-C)*Wire_Width; 
    union {
    cylinder {A , B, W }
    //cylinder {A, C, W }
    cylinder {B, C, W }
    texture { Wire_Tex} 
    }
  #else
    SlimWires_Dart(B,C,A+(C-A)*phi,D-1)
    SlimWires_Dart(B,B+(A-B)*phi,A+(C-A)*phi,D-1)
    WideWires_Dart(A,A+(C-A)*phi,B+(A-B)*phi,D-1)
  #end
#end   //SlimWires_Dart

#macro WideWires_Dart(A,B,C,D)
  #if (D=0)
    #local W=vlength(B-C)*Wire_Width; 
    union {
    cylinder {A , B, W }
    //cylinder {A, C, W } 
    cylinder {B, C, W }
    texture { Wire_Tex }
    } 
  #else
    SlimWires_Dart(A,A+(B-A)*phi,C,D-1)
    WideWires_Dart(B,C,A+(B-A)*phi,D-1)
  #end
#end  //WideWires_Dart


//MeshDarts Macro

#macro MeshDarts(_Radius,_Depth)

#local Wedge_Dart=mesh {
  Slim_Dart(<0,0,0>,vrotate(x*_Radius, y*36),vrotate(x*_Radius,y*72),abs(int(_Depth)))
}

#declare Half_Mesh_Dart = 
union {
object { Wedge_Dart }
object { Wedge_Dart rotate y*72 }
object { Wedge_Dart rotate y*144 }
object { Wedge_Dart rotate y*216 }
object { Wedge_Dart rotate y*288 }    
}  

union { object {Half_Mesh_Dart} object {Half_Mesh_Dart scale <-1, 1, 1> rotate y*36} }  

#end // MeshDarts



//WireDarts Macro


#macro WireDarts(_Radius,_Depth)

#local Wire_Wedge_Dart=union {
  SlimWires_Dart(<0,0,0>,vrotate(x*_Radius, -y*36),x*_Radius,abs(int(_Depth)))    }


#declare Half_Wires_Dart = 

union {
object { Wire_Wedge_Dart }
object { Wire_Wedge_Dart rotate y*72 }
object { Wire_Wedge_Dart rotate y*144 }
object { Wire_Wedge_Dart rotate y*216 }
object { Wire_Wedge_Dart rotate y*288 }    
}


union { object {Half_Wires_Dart} object {Half_Wires_Dart scale <-1, 1, 1> rotate y*36} }

#end //WireDarts
                    


//PenroseDarts Macro

#macro PenroseDarts(_R,_D)

union {object{MeshDarts(_R,_D)} object{ WireDarts(_R,_D)} }

#end //PenroseDarts


//Macros for creating the "Rhombs" or Type B Penrose Tiling


//Mesh Triangles

#macro Slim_Rhomb(A,B,C,D) 

#if (mod(D,2) = 0)
  #if (D=0)
    triangle { A,B,C texture { Tile_Tex_1 } }
  #else
     #local N = A + phi*(B - A);
    Slim_Rhomb(C, N, B, D - 2)  
    Wide_Rhomb(A, N, C, D - 1)
  #end
#end   // Slim_Rhomb

#end

#macro Wide_Rhomb(A,B,C,D) 

#if (mod(D,2) = 1 )
  #if (D=1 )
    triangle { A,B,C texture { Tile_Tex_2 } } 
  #else 
    #local N = C - phi*(C - A);
    Slim_Rhomb(C, B, N, D - 1)
    Wide_Rhomb(B, N, A, D - 2)
  #end 
#end
#end   // Wide_Rhomb


//Wires


#macro SlimWires_Rhomb(A,B,C,D) 

#if (mod(D,2) = 0)
  #if (D=0)
    #local W=vlength(A-C)*Wire_Width; 
    union {
    cylinder {A , B, W }
    cylinder {A, C, W }
    texture { Wire_Tex } 
    }
    
  #else
     #local N = A + phi*(B - A);
    SlimWires_Rhomb(C, N, B, D - 2)  
    WideWires_Rhomb(A, N, C, D - 1)
  #end
#end

#end

#macro WideWires_Rhomb(A,B,C,D) 

#if (mod(D,2) = 1 )

  #if (D=1 )
    #local W=vlength(A-B)*Wire_Width; 
    union {
    cylinder {A , B, W }
    cylinder {B, C, W }
    texture { Wire_Tex }
    } 
  #else 
    #local N = C - phi*(C - A);
    SlimWires_Rhomb(C, B, N, D - 1)
    WideWires_Rhomb(B, N, A, D - 2)
  #end 
#end
#end




//MeshRhombs Macro


#macro MeshRhombs(_Radius,_Depth)

#local Wedge_Rhomb=mesh {
  Slim_Rhomb(<0,0,0>,vrotate(x*_Radius, y*36),vrotate(x*_Radius,y*72),2*abs(int(_Depth)))
}

#declare Half_Mesh_Rhomb = 
union {
object { Wedge_Rhomb }
object { Wedge_Rhomb rotate y*72 }
object { Wedge_Rhomb rotate y*144 }
object { Wedge_Rhomb rotate y*216 }
object { Wedge_Rhomb rotate y*288 }    
}  

union { object {Half_Mesh_Rhomb} object {Half_Mesh_Rhomb scale <-1, 1, 1> rotate y*36} }  

#end // MeshRhombs



//WireRhombs Macro


#macro WireRhombs(_Radius,_Depth)

#local Wire_Wedge_Rhomb=union {
  SlimWires_Rhomb(<0,0,0>,vrotate(x*_Radius, -y*36),x*_Radius,2*abs(int(_Depth)))    }


#declare Half_Wires_Rhomb = 

union {
object { Wire_Wedge_Rhomb }
object { Wire_Wedge_Rhomb rotate y*72 }
object { Wire_Wedge_Rhomb rotate y*144 }
object { Wire_Wedge_Rhomb rotate y*216 }
object { Wire_Wedge_Rhomb rotate y*288 }    
}


union { object {Half_Wires_Rhomb} object {Half_Wires_Rhomb scale <-1, 1, 1> rotate y*36} }

#end //WireRhombs
                    


//PenroseRhombs Macro

#macro PenroseRhombs(_R,_D)

union { object {MeshRhombs(_R,_D)} object{ WireRhombs(_R,_D)} }

#end //PenroseDarts