// Copyright (C) 2003
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


// ============================================================================
//
// Notes
//
// Author: Ian Shumsky / ian@outerarm.demon.co.uk
//
// Wireframe type globe. Modified version of model originally created for IRTC
// "Sea" round.
//
// ============================================================================

#version 3.5;

// ============================================================================
//
// Details about the size of the globe.
//
// ============================================================================

#declare land_mass_thickness = 0.004; // 2mm
#declare inner_max = 0.25 - land_mass_thickness; // 25cm - 2mm
#declare vframe_thickness = 0.003; // 3mm
#declare hframe_thickness = 0.003; // 3mm
#declare earth_axis_rotation = 23 + 27/60;
#declare globe_rotation = -127-45;

// ============================================================================
//
// Define the metal textures used on the globe. These are all based on textures
// found in metals.inc
//
// ============================================================================

#declare GoldBase = <1.00, 0.875, 0.575>;
#declare CVect2 = GoldBase - <0.00, 0.15, 0.30>;
#declare P_Gold2 = rgb CVect2;
#declare P_Silver2 = color rgb <0.94, 0.93, 0.85>;
#declare P_Brass3 = color rgb <0.58, 0.42, 0.20>;

#declare globe_gold_t = texture
{
	pigment
	{
		P_Gold2
	}
	finish
	{
		brilliance 4
		diffuse 0.5
		metallic
		specular 0.80
		roughness 1/80

		#if (radiosity_stage = 0)
			ambient 0.25
			reflection 0.25
		#else
			ambient 0
			#if (radiosity_stage = 1)
				reflection 0
			#else
				reflection 0.25
			#end
		#end
	}
}

#declare globe_silver_t = texture
{
	pigment
	{
		P_Silver2
	}
	finish
	{
		brilliance 4
		diffuse 0.5
		metallic
		specular 0.80
		roughness 1/80

		#if (radiosity_stage = 0)
			ambient 0.25
			reflection 0.25
		#else
			ambient 0
			#if (radiosity_stage = 1)
				reflection 0
			#else
				reflection 0.2
			#end
		#end
	}
}

#declare globe_bronze_t = texture
{
	pigment
	{
		P_Brass3
	}
	finish
	{
		brilliance 4
		diffuse 0.5
		metallic
		specular 0.80
		roughness 1/80

		#if (radiosity_stage = 0)
			ambient 0.25
			reflection 0.25
		#else
			ambient 0
			#if (radiosity_stage = 1)
				reflection 0
			#else
				reflection 0.25
			#end
		#end
	}
}

// ============================================================================
//
// First we need to build the frame.
//
// ============================================================================

#declare globe_vframe_element = torus
{
	inner_max - (vframe_thickness / 2), vframe_thickness / 2
	rotate <90, 0, 0>
}

#declare globe_vertical_frame = union
{
	#declare vangle = 18;
	#declare vcount = 0;

	#while (vcount < (180 / vangle))
		object
		{
			globe_vframe_element
			rotate <0, vcount * vangle, 0>

			#if (vcount = 0)
				texture
				{
					globe_silver_t
				}
			#else
				texture
				{
					globe_bronze_t
				}
			#end
		}

		#declare vcount = vcount + 1;
	#end
}

#declare globe_pole_minus = torus
{
	inner_max - (vframe_thickness / 2), (vframe_thickness / 2) + 0.0002
	rotate <90, 0, 0>
}

#declare globe_ice_caps = difference
{
	cylinder
	{
		<0, (inner_max - (vframe_thickness / 2)) + 0.005, 0>, <0, -((inner_max - (vframe_thickness / 2)) + 0.005), 0>, 0.03
	}
	box
	{
		<-1, (inner_max - (vframe_thickness / 2)) -0.005, -1>, <1, -((inner_max - (vframe_thickness / 2)) - 0.005), 1>
	}

	#declare vangle = 18;
	#declare vcount = 0;

	#while (vcount < (180 / vangle))
		object
		{
			globe_pole_minus
			rotate <0, vcount * vangle, 0>
		}

		#declare vcount = vcount + 1;
	#end

	texture
	{
		globe_silver_t
	}
}

#declare globe_horizontal_frame = union
{
	//
	// We are assuming 7 horizontal rings... This gives us angles
	// of 67.5, 45, 22.5, 0, -22.5 -45, -67.5. If we were having 5 rings
	// the angles would be 60, 30, 0, -30, -60.
	//

	#declare base_radius = inner_max - vframe_thickness - (hframe_thickness / 2);

	//
	// equator
	//

	torus
	{
		base_radius, hframe_thickness / 2
		texture
		{
			globe_silver_t
		}
	}

	//
	// The rest...
	//

	union
	{
		#declare new_rad = cos (radians (22.5)) * base_radius;
		#declare new_height = sin (radians (22.5)) * base_radius;

		torus
		{
			new_rad, hframe_thickness / 2
			translate <0, new_height, 0>
		}

		torus
		{
			new_rad, hframe_thickness / 2
			translate <0, -(new_height), 0>
		}

		#declare new_rad = cos (radians (45)) * base_radius;
		#declare new_height = sin (radians (45)) * base_radius;

		torus
		{
			new_rad, hframe_thickness / 2
			translate <0, new_height, 0>
		}

		torus
		{
			new_rad, hframe_thickness / 2
			translate <0, -(new_height), 0>
		}

		#declare new_rad = cos (radians (67.5)) * base_radius;
		#declare new_height = sin (radians (67.5)) * base_radius;

		torus
		{
			new_rad, hframe_thickness / 2
			translate <0, new_height, 0>
		}

		torus
		{
			new_rad, hframe_thickness / 2
			translate <0, -(new_height), 0>
		}

		texture
		{
			globe_bronze_t
		}
	}
}

// ============================================================================
//
// Now define the landmass on the globe. First we declare a function with the
// map_type 1 (spherical) mapping
//
// ============================================================================

#declare land_heightfield = function
{
	pigment
	{
		image_map
		{
			png "land_hf.png"
			map_type 1
			interpolate 2
		}
	}
}

//
// Now create the isosurface.
//

#declare rs = 0.3;

#declare base_land_mass = isosurface
{
	function
	{
		x*x+y*y+z*z + rs - land_heightfield(x,y,z).gray * rs
	}
	contained_by
	{
		sphere
		{
			<0, 0, 0>, 1
		}
	}
	max_gradient 150 
	threshold 1
}
	
#declare scaled_land_mass =  object
{
	base_land_mass
	scale (inner_max + land_mass_thickness)
}

//
// Now diff the iso from a sphere to leave just the land area.
//

#declare globe_land_mass = difference
{
	object
	{
		scaled_land_mass
	}
	sphere
	{
		<0, 0, 0>, inner_max
	}
	bounded_by
	{
		sphere
		{
			<0, 0, 0>, 0.275
		}
	}
	texture
	{
		globe_gold_t
	}
}

// ============================================================================
//
// Add the frame and the land together.
//
// ============================================================================

#declare globe_earth = union
{
	object
	{
		globe_vertical_frame 
	}
	object
	{
		globe_horizontal_frame 
	}
	object
	{
		globe_ice_caps 
	}
	object
	{
		globe_land_mass
	}

	//
	// And rotate it where requested.
	//

	rotate <0, globe_rotation, 0>
}


// ============================================================================
//
// Define the globe... If we are only on the first radiosity pass, just use a
// sphere.
//
// ============================================================================

#if (radiosity_stage = 1)
	#declare globe = object
	{
		sphere
		{
			<0, 0, 0>, 0.25
			texture
			{
				globe_gold_t
			}
		}
	}
#else
	#declare globe = object
	{
		globe_earth 
		rotate <0, 0, earth_axis_rotation>
	}
#end

// ============================================================================
//
// Add the globe to the display box.
//
// ============================================================================

object
{
	globe
	scale 4
	ball_at (2,0)
}
