// Balls
// Copyright (C) 2003 Nicolas P. Rougier (rougier@loria.fr)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.


// ========================================
//  Notes
// ========================================
// Author: Nicolas Rougier
//
// From: povray.binaries.image
// Subject: Tiled sphere (96k + 28k)
// Date: 8 Aug 2003 17:38:24 EDT

// ========================================
//  Spheric to cartesian conversion macro
//   (angle in degrees)
// ========================================
#macro spheric_d (rho,theta,phi)
  <rho*sind(theta)*sind(phi), rho*cosd(theta), rho*sind(theta)*cosd(phi)>
#end

     
// =====================================================
//  Tile a sphere with object with regular spacing
//   "object" dimensions are taken from by bounding box
// =====================================================
//  _size  : Radius of the sphere
//  _object: Object to be used as tiles
//  _lshift: Control placement of object along lateral axis
//  _hdev  : Standard deviation of altitude of object on the sphere
//  _msize : Mortar size as a percentage of object size 
//  theta_start, theta_end, phi_start, phi_end: control sphere "completness"
//  _seed  : any number you like
#macro tiled_sphere (_size, _object, _msize, _lshift, _hdev, theta_start, theta_end, phi_start, phi_end, _seed)
  // Get size of the object
  #local Min = <0,0,0>;
  #local Max = <0,0,0>;
  Extents (_object, Min, Max)

  // Center the object
  #local _tile = object {_object translate -Min-<(Max.x-Min.x), 0, (Max.z-Min.z)>/2}

  // Get size of the new centered object
  #local _tile_size = Max-Min;

  #local st          = seed (_seed);
  #local shift       = 0;
  #local theta       = theta_start;
  #local theta_step  = 360/(2*pi*_size/_tile_size.z);
  
  union {
    #while (theta < theta_end)
      #local ring_radius      = _size*abs(sind(theta));
      #local ring_perimeter   = 2*pi*ring_radius;

      // Test to check if we are at the pole (where ring_perimeter could be null)
      #if (int (ring_perimeter/(_tile_size.x*2)) = 0)
        #local phi_step       = 361;
      #else
        #local phi_step       = 180 / int (ring_perimeter/(_tile_size.x*2));
      #end
      
      #local shift            = shift + _lshift;
      #local shift            = shift - int(shift);
      #local phi              = phi_start + shift*phi_step;      
      #while (phi <= phi_end)
	object {
	  _tile
          scale <1-_msize, _hdev*(.75+.25*rand(st)), 1-_msize>
          translate y*_size
          rotate <theta, phi, 0>
          pigment {rgb .75+.25*rand(st)}
        }
        #local phi = phi+phi_step;
      #end
      #local theta = theta+theta_step;
    #end
  }
#end

// ========================================
//  The ball
// ========================================
#declare tile_size = <0.05, .45, 0.05>;
object {
  tiled_sphere (.75, box {0,1 scale tile_size}, 0.05, .5, 1, 0, 180, 0, 360, 12345)
  rotate -x*30
  rotate -z*20
  ball_at (1,3)
}
