// File written by Hugo Asm, in POV-Ray 3.5
// Version 2, September 24, year 2003. Feel free to use/modify.
//
// Here's what it's doing: It takes any function as input,
// and outputs an orthographical view of it. The catch is,
// the output is now a seamless tile of the function.
//
// This file is also written with animation in mind. If you
// have a function that responds to the clock value, and you
// want a seamless loop, this file will do the trick.
//
// It does degrade the look of the function, that you use as
// input. But it may not be a problem, if you experiment with
// the resolution, and you scale the function to obtain the most
// pleasing results. Besides, I recommend more than 100 frames
// for a nice looping of the function.

// Currently, this file takes Christoph Hormann's wave formula as
// input, but you can change that.

// Please note: For this file to work, it needs a special image_map
// for the seamless tiling. I provide it along with this file.
//
// You also have to use "squared" resolutions, such as 256*256 !
// In other words, image width has to be the same as image height.
// 100*200 is not going to work.


// **** USER PARAMETERS FOR CHRISTOPH HORMANN's WAVE FORMULA ****


#local Amount_Of_Waves = 103;    // Integer value between 1-103 
#local Speed_Of_Wind = 5.2 *.83; // Seems quite sensitive.
#local Scale_Of_Waves = 500;     // This just scales the map
#local Overall_Speed = 4;        // This float changes the overall speed of the animation

// Use these INI settings for a 300 frames animation.
// I don't guarantee that anti-alias will be a good idea. I'd leave it off.
// You may choose any resolution
// +fn16 must be included for a png 16bit grayscale output!

// +w256 +h256 +kff100 +fn16 


// **** END OF USER INPUT! *****


 
global_settings { hf_gray_16 on } // Uncomment for ordinary output.

camera { orthographic right 1*x
	location <1/image_width,1 + 1/image_width,-1/image_height>
	look_at <1/image_width,0,-1/image_height> angle 90
}

// The following 2 functions must have similar settings
// except the time value, to allow a seamless loop over time.

#include "waves.inc"
#declare MAX_WAVES =Amount_Of_Waves;
#local Wind_Direction = 90; // Degrees
#local Wind_Speed = Speed_Of_Wind;
#local Time_1 = (1+clock)* (final_frame/Overall_Speed);

#local Water_1=function {
	min( max(
		Waves(
			radians(Wind_Direction),
			Wind_Speed,
			Time_1,
			Scale_Of_Waves
		)*.035 +.5
	,0),1)
};

// I found that "waves.inc" needs to be included twice for some reason 

#include "waves.inc"
#declare MAX_WAVES =Amount_Of_Waves;
#local Wind_Direction = 90; // Degrees
#local Wind_Speed = Speed_Of_Wind;
#local Time_2 = (2+clock)* (final_frame/Overall_Speed);

#local Water_2=function {
	min( max(
		Waves(
			radians(Wind_Direction),
			Wind_Speed,
			Time_2,
			Scale_Of_Waves
		)*.035 +.5
	,0),1)
};

// Bitmap used to mix copies of each map at the edges
// of the rendering, to allow a seamless tiling

#local Seamless_Tiling = function {
	pigment { image_map { jpeg "Seamless_Tiling.jpg" }
	translate .5 scale 1 + (1/image_width) }
};

#local Time_Map1= pigment { function { Seamless_Tiling(x,z,y).red }
	pigment_map {

		[ 0 function { Seamless_Tiling(x,z,y).green }
			pigment_map {

				[ 0 function { Water_1(x,y,0) } rotate 90*x ]

				[ .5 function {
						min(max(
							(  Water_1(x,y,0) + Water_1(x,y+1,0)  )/1.25 -.27
						,0),1)
					}
					rotate 90*x
				]					

				[ 1 function { Water_1(x,y+1,0) } rotate 90*x ]
			}
		]

		[ 1 function { Seamless_Tiling(x,z,y).green }
			pigment_map { 

				[ 0 function { Water_1(x,y,0) } rotate 90*x translate <1,0,0> ]

				[ .5 function {
						min(max(
							(  Water_1(x-1,y,0) + Water_1(x-1,y+1,0)  )/1.25 -.27
						,0),1)
					}
					rotate 90*x
				]
					
				[ 1 function { Water_1(x-1,y+1,0) } rotate 90*x ]					
			}
		]
	}
	scale 4
};


#local Time_Map2= pigment { function { Seamless_Tiling(x,z,y).red }
	pigment_map {

		[ 0 function { Seamless_Tiling(x,z,y).green }
			pigment_map {

				[ 0 function { Water_2(x,y,0) } rotate 90*x ]

				[ .5 function {
						min(max(
							(  Water_2(x,y,0) + Water_2(x,y+1,0)  )/1.25 -.27
						,0),1)
					}
					rotate 90*x
				]					

				[ 1 function { Water_2(x,y+1,0) } rotate 90*x ]
			}
		]

		[ 1 function { Seamless_Tiling(x,z,y).green }
			pigment_map { 

				[ 0 function { Water_2(x,y,0) } rotate 90*x translate <1,0,0> ]

				[ .5 function {
						min(max(
							(  Water_2(x-1,y,0) + Water_2(x-1,y+1,0)  )/1.25 -.27
						,0),1)
					}
					rotate 90*x
				]
					
				[ 1 function { Water_2(x-1,y+1,0) } rotate 90*x ]					
			}
		]
	}
	scale 4
};


plane { y,0

	// Time Loop
	pigment { gradient y pigment_map {
			[ 0  Time_Map2 ]

			[ .3 bumps noise_generator 3 phase clock*.5 pigment_map {
					[ 0 Time_Map1 ] [ .5 Time_Map2 ] [ 1 Time_Map1 ]
				}
			]

			[ .7 bumps noise_generator 3 phase clock*.5 pigment_map {
					[ 0 Time_Map1 ] [ .5 Time_Map2 ] [ 1 Time_Map1 ]
				}
			]

			[ 1 Time_Map1 ]
		}
		phase clock
		scale .5
	}

	finish { ambient 1 }
}