// Persistence of Vision Ray Tracer Scene Description File
// File: demo2_voronoi.pov v0.9
// POV-Ray Version: 3.5
// Desc: Voronoi include file
// Date: 20/04/2003
// Auth: Jean-Charles Marteau ( exether@free.fr )
// License : This is free for non-commercial use, I'd be glad if you let me and others know you used it.
// 
//
// -------------------------------------------------------------------------------------
// More documentation:
//  - http://www.cs.cornell.edu/Info/People/chew/Delaunay.html
//    features a nice interactive Voronoi applet. Go there before anything else.
//
//  - http://www.ics.uci.edu/~eppstein/gina/voronoi.html
//    has a lot of links on the subject
//
// Known bugs and things to do:
//  - No known bugs in the Voronoi part
//  - There is a bug in hl_geom.inc with some polygons with too sharp edges
//  - I want a macro that cuts polygons to fit them into the area defined at Init
//    so that I can have a perfectly square set of polygons.
//  - I'm thinking about doing the same in 3D, that is polyhedrons of Voronoi
//  - There is a whole set of possible optimisations to do, for now I have been
//    writing the code so that it works and stays as clear as possible.
//  - either translate the comments in English or have you learn French.
//
// -------------------------------------------------------------------------------------
// Voronoi HOWTO:
// The code might be complex, but I tried to make the usage as simple as possible.
//
// What you have to do is follow some steps to actually get a set of computed Voronoi data:
//  - Basic Initialisation
//  - Frame Initialisation
//  - Diagrams setup
//  - Algorithm execution
//  - optional save
//
// ---------------
// Basic Initialisation needs :
//  - The number of points for the diagram (= number of polygons after computation)
//  - Two (x,z) planar points defining the limits in which the points will be.
//  - A initialized random seed
// e.g. -> Init_Voronoi (50, <-40.0,-40.0>, < 40.0, 40.0>, MySeed)
//
// ---------------
// You shouldn't care too much about Frame Init. It is separated from the
// previous init because various Frame initialisation could be choosen.
//  e.g. -> Tamis_initial_rectangulaire ()
//
// ---------------
// Diagrams setup is again another initialisation that will put the points at
// random within the area defined at init. The parameter is the minimum
// distance allowed between two points. You should be very carefull that
// You should be carefull that the total number of points, the surface of
// the area and the minimum distance can't be choosen independantly (it's
// question of fitting disks in a given area).
// Choosing the points at random is not necessary, you could choose them to
// fit a given pattern or even mix random and non random points. Just write
// your own repartition if you feel like it. 
//  e.g. -> Repartition_Aleatoire (5.0)
//  
// ---------------
// This is the main algorithm, it is potentially very long.
//  -> Delaunay_Triangulation ()
//
// ---------------
// The Voronoi library has a great save functionality. It might be pointless with
// 50 points, but will save a lot of time when doing 6000. If you have a look at
// the saved file you will see that it is an include file for POV-Ray containing
// only data. You shall reload it simply by including it.
// Including a Voronoi lib generated file puts exactly in the same situation than
// after the Delaunay_Triangulation call.
//  e.g. -> Save_Voronoi ("vorofield_50.inc")
//
//
// Once your data are ready to use, you may use them.
// Global variables :
//   - 'Points' is an array containing all the points belonging to the polygons.
//   - You should start using the points after index 'Start_index' because the previous
//     points are what we could call Frame points, and they are out of the frame.
//   - 'Point_Polygone' is an array of points that is created when you call the
//     Polygone_Voronoi macro with the index (> Start_index) of the point belonging to the polygon.
//     You should request for the dimension_size(Point_Polygone,1) to know how many points
//     has the polygon.
//
// And don't forget to look at the demo scenes.

// -------------------------------------
//  Initialisation
// -------------------------------------
#macro Init_Voronoi (N, b1, b2, RSeed)
#declare Nb_Points=N;
#declare Borne1=<min(b1.x, b2.x), min(b1.y, b2.y)>;
#declare Borne2=<max(b1.x, b2.x), max(b1.y, b2.y)>;
#declare TheSeed=RSeed;

#declare Nb_Aretes=Nb_Points*3;
#declare Nb_Triangles=Nb_Points*2;
#declare Points=array[Nb_Points];

// Contient les index des points constituants les aretes.
#declare Aretes=array [Nb_Aretes][2];
// Contient les index des aretes des triangles.
#declare Triangles_Aretes=array [Nb_Triangles][3];
// Contient les centres des cercles circonscrits aux triangles.
#declare Triangles_Centres=array [Nb_Triangles];
// Contient les rayons des cercles circonscrits aux triangles.
#declare Triangles_Rayons=array [Nb_Triangles];

// Initialisation
#local index=0;
#while (index<Nb_Points)
  #declare Points[index]=<0.0, 0.0>;
  #local index=index+1;
#end

#local index=0;
#while (index<Nb_Aretes)
  #declare Aretes[index][0]=-1;
  #declare Aretes[index][1]=-1;
  #local index=index+1;
#end

#local index=0;
#while (index<Nb_Triangles)
  #declare Triangles_Rayons[index]=-1;
  #declare Triangles_Aretes[index][0]=-1;
  #declare Triangles_Aretes[index][1]=-1;
  #declare Triangles_Aretes[index][2]=-1;
  #declare Triangles_Centres[index]=<0.0,0.0,0.0>;
  #local index=index+1;
#end

#declare index_Ar=0;
#declare index_Tr=0;

#end

// -------------------------------------
//  Finalisation
// -------------------------------------
#macro Finalisation ()
#undef Nb_Points
#undef Borne1
#undef Borne2
#undef TheSeed

#undef Nb_Aretes
#undef Nb_Triangles
#undef Points

#undef Aretes
#undef Triangles_Aretes
#undef Triangles_Centres
#undef Triangles_Rayons
#undef index_Ar
#undef index_Tr

#ifdef (Triangle_de_l_arete)
  #undef Triangle_de_l_arete
#end
#ifdef (Point_Polygone)
  #undef Point_Polygone
#end
#end

// -------------------------------------
// Methodes de creation des objets
// -------------------------------------
#macro Nouv_Arete (p1, p2)
Debug_Message (concat ("Nouv_Arete ", str(index_Ar,0,0), "  Pts : ", str(p1,0,0), " ", str(p2,0,0), "\n"))
#declare Aretes[index_Ar][0]=p1;
#declare Aretes[index_Ar][1]=p2;
#local Result=index_Ar;
// Optimisable
#declare index_Ar=0;
#while (Aretes[index_Ar][0]!=-1)
  #declare index_Ar=index_Ar+1;
#end
Result
#end

#macro Tue_Arete (index)
Debug_Message (concat ("Tue_Arete ", str(index,0,0), "\n"))
#declare Aretes[index][0]=-1;
#end

#macro Nouv_Triangle (Ar1,Ar2,Ar3,A,B,C)
Debug_Message (concat ("Nouv_Triangle ", str(index_Tr,0,0), " Ar : ", str(Ar1,0,0), " ", str(Ar2,0,0), " ",str(Ar3,0,0), "  Pts : ", str(A,0,0), " ", str(B,0,0), " ", str(C,0,0), " \n"))
#declare Triangles_Aretes[index_Tr][0]=Ar1;
#declare Triangles_Aretes[index_Tr][1]=Ar2;
#declare Triangles_Aretes[index_Tr][2]=Ar3;
#local rayon=-1;                        
#declare Triangles_Centres[index_Tr]=triangle_circle_circ (Points[A], Points[B], Points[C], rayon);
#declare Triangles_Rayons[index_Tr]=rayon;
#local Result=index_Tr;
// Optimisable
#declare index_Tr=0;
#while (Triangles_Rayons[index_Tr]!=-1)
  #declare index_Tr=index_Tr+1;
#end
Result
#end

#macro Tue_Triangle (index)
Debug_Message (concat ("Tue_Triangle ", str(index,0,0), "\n"))
#declare Triangles_Rayons[index]=-1;
#end

// -------------------------------------
//  Methodes de preparation des tamis
// -------------------------------------
#macro Tamis_initial_triangulaire ()
#declare Points[0]=<Borne1.x, (Borne1.y+(Borne2.y-Borne1.y))*2>;
#declare Points[1]=<Borne1.x,-(Borne1.y+(Borne2.y-Borne1.y))*2>;
#declare Points[2]=<Borne1.x+(Borne2.x-Borne1.x)*2, 0.0>;
Debug_Message(concat ("Adding point : ", str(0, 0,0), " <", str(Points[0].x,0,0), ", ", str(Points[0].y,0,0), ">\n"))
Debug_Message(concat ("Adding point : ", str(1, 0,0), " <", str(Points[1].x,0,0), ", ", str(Points[1].y,0,0), ">\n"))
Debug_Message(concat ("Adding point : ", str(2, 0,0), " <", str(Points[2].x,0,0), ", ", str(Points[2].y,0,0), ">\n"))

#local a1=Nouv_Arete (0,1);
#local a2=Nouv_Arete (1,2);
#local a3=Nouv_Arete (2,0);

#local Dummy=Nouv_Triangle(a1,a2,a3,0,1,2);

#declare Start_index=3;
#end

#macro Tamis_initial_rectangulaire ()
#declare Points[0]=<Borne1.x+(Borne2.x-Borne1.x)*2, Borne1.y+(Borne2.y-Borne1.y)*2>;
#declare Points[1]=<Borne1.x+(Borne2.x-Borne1.x)*2, Borne1.y-(Borne2.y-Borne1.y)>;
#declare Points[2]=<Borne1.x-(Borne2.x-Borne1.x),   Borne1.y+(Borne2.y-Borne1.y)*2>;
#declare Points[3]=<Borne1.x-(Borne2.x-Borne1.x),   Borne1.y-(Borne2.y-Borne1.y)>;
Debug_Message(concat ("Adding point : ", str(0, 0,0), " <", str(Points[0].x,0,0), ", ", str(Points[0].y,0,0), ">\n"))
Debug_Message(concat ("Adding point : ", str(1, 0,0), " <", str(Points[1].x,0,0), ", ", str(Points[1].y,0,0), ">\n"))
Debug_Message(concat ("Adding point : ", str(2, 0,0), " <", str(Points[2].x,0,0), ", ", str(Points[2].y,0,0), ">\n"))
Debug_Message(concat ("Adding point : ", str(3, 0,0), " <", str(Points[3].x,0,0), ", ", str(Points[3].y,0,0), ">\n"))

#local a1=Nouv_Arete (0,1);
#local a2=Nouv_Arete (1,2);
#local a3=Nouv_Arete (2,0);
#local a4=Nouv_Arete (2,3);
#local a5=Nouv_Arete (3,1);

#local Dummy=Nouv_Triangle(a1,a2,a3,0,1,2);
#local Dummy=Nouv_Triangle(a2,a4,a5,1,2,3);

#declare Start_index=4;
#end

// -------------------------------------
//  Methodes de repartition des points
// -------------------------------------
#macro Repartition_Aleatoire (min_dist)
#local index=Start_index;
#while (index<Nb_Points)
  // On tire un point au hasard jusqu'a ce qu'il soit assez eloign des autres points.
  #local count_loops=0;
  #local dist_ok=false;
  #while (!dist_ok & (count_loops < 500))
    #declare NPoint=<rand(TheSeed), rand(TheSeed)>*(Borne2-Borne1)+Borne1;
    #local dist_ok=true;
    #local autre_point=0;
    #while ((autre_point<index) & dist_ok)
      #if (vlength (NPoint-Points[autre_point])<min_dist)
	#local dist_ok=false;
      #end
      #local autre_point=autre_point+1;
    #end
    #local count_loops=count_loops + 1;
  #end
  #if (mod(index,100) = 0)
       Debug_Message (concat("Number of points = ", str (index,0,0), "\n"))
  #end
  #if (count_loops = 500)
    #error "Impossible de placer les points, ajuster les parametres de creation aleatoire."
  #end
  #declare Points[index]=NPoint;
  #local index=index+1;
#end
#end

// -------------------------------------
//  Algo principal
// -------------------------------------
#macro Delaunay_Triangulation ()
#declare Aretes_a_conserver=array[Nb_Aretes];

// Les premiers points sont dj triangules, on commence donc au suivant.
// On va ajouter les points a la triangulation un par un.
#local index=Start_index;
#while (index<Nb_Points)
  
  // Voici la liste des aretes qui font partie d'un et un seul triangle contenant le nouveau point.
  // Ces aretes doivent etre conservees pour faire de nouveaux triangles avec le nouveau point.
  // On initialise la structure.
  #local index_ac=0;
  #while (index_ac<Nb_Aretes)
    #declare Aretes_a_conserver[index_ac]=-1;
    #local index_ac=index_ac+1;
  #end
  #local index_ac=0;

  #debug concat ("Adding point : ", str(index, 0,0), " <", str(Points[index].x,0,2), ", ", str(Points[index].y,0,2), ">\n")

  // On parcours la liste des triangles deja constitues pour voir si le cercle circonscrit contient
  // le nouveau point.
  #local Cur_Triangle=0;
  // En principe on parcours tous les triangles, mais on exploite ici le fait qu'on realloue depuis le debut a
  // chaque fois ce qui permet de s'arreter a 2*index qui est le maximum de triangles pour 'index' points.
  #while (Cur_Triangle<2*index)
    
    #if (Triangles_Rayons[Cur_Triangle]!=-1)
      // Si le triangle est valide
      #if (vlength(Points[index]-Triangles_Centres[Cur_Triangle])<Triangles_Rayons[Cur_Triangle])
	// Si le point est dans le cercle circonscrit
	// Alors on ajoute les aretes du triangle dans le tableau des aretes a conserver.
	#local Cur_Arete=0;
	#while (Cur_Arete<3)
	  
	  // Sauf si l'arete est deja presente dans le tableau.
	  #local found=false;
	  #local Cur_Arete_Cons=0;
	  #while (Cur_Arete_Cons<index_ac)
	    // Si cette arete est deja presente, on la supprime du tableau et on efface l'arete.
	    #if (Aretes_a_conserver[Cur_Arete_Cons]=Triangles_Aretes[Cur_Triangle][Cur_Arete])
	      Tue_Arete(Aretes_a_conserver[Cur_Arete_Cons])
	      #declare Aretes_a_conserver[Cur_Arete_Cons]=-1;
	      #local found=true;
	    #end
	    #local Cur_Arete_Cons=Cur_Arete_Cons+1;
	  #end
	  
	  // Si on ne l'a pas deja trouve dans le tableau, alors on l'y ajoute.
	  #if (!found)
	    #declare Aretes_a_conserver[index_ac]=Triangles_Aretes[Cur_Triangle][Cur_Arete];
	    #local index_ac=index_ac+1;
	  #end

	  #local Cur_Arete=Cur_Arete+1;          
	#end
	// On supprime ensuite le triangle.
	Tue_Triangle(Cur_Triangle);
      #end
    #end
    #local Cur_Triangle=Cur_Triangle+1;
  #end
  
  // Arrive ici, on a supprime les triangles qui contenaient le point et on a la liste
  // des aretes avec lesquelles il faut reformer des triangles avec le nouveau point.
  #local Cur_Arete_Cons=0;
  #while (Cur_Arete_Cons<index_ac)
    #if (Aretes_a_conserver[Cur_Arete_Cons]!=-1)
      Debug_Message(concat ("Arete ", str(Cur_Arete_Cons,0,0), " / ", str(Aretes_a_conserver[Cur_Arete_Cons],0,0), "  p1 : ", str(Aretes[Aretes_a_conserver[Cur_Arete_Cons]][0],0,0), "  p2 : ", str(Aretes[Aretes_a_conserver[Cur_Arete_Cons]][1],0,0), "\n"))
    #end
    #local Cur_Arete_Cons=Cur_Arete_Cons+1;
  #end
  
  // A partir de la liste des aretes on va les prendre en compte dans l'ordre, de facon
  // a creer les triangles en tournant.
  
  // On cherche la premiere arete valide
  #local Cur_Arete_Cons=0;
  #while (Aretes_a_conserver[Cur_Arete_Cons]=-1)
    #local Cur_Arete_Cons=Cur_Arete_Cons+1;
  #end

  #local last_point=Aretes[Aretes_a_conserver[Cur_Arete_Cons]][0];
  #local prev_point=Aretes[Aretes_a_conserver[Cur_Arete_Cons]][1];
  #local last_arete=Nouv_Arete(last_point,index);
  #local prev_arete=Nouv_Arete(prev_point,index);
  #local Dummy=Nouv_Triangle
  (last_arete, Aretes_a_conserver[Cur_Arete_Cons], prev_arete,
    Aretes[Aretes_a_conserver[Cur_Arete_Cons]][0], index, Aretes[Aretes_a_conserver[Cur_Arete_Cons]][1]);
  // On peut alors supprimer l'arete qui vient d'etre prise en compte du tableau.
  #declare Aretes_a_conserver[Cur_Arete_Cons]=-1;
  
  // Boucle de prise en compte de toute les aretes, on cherche les aretes qui contiennent
  // le dernier point memorise pour les prendre en compte. A chaque fois qu'on trouve une
  // arete qui est bonne on recommence a zero.
  #local Cur_Arete_Cons=0;
  #while (Cur_Arete_Cons<index_ac)
    #local index_autre_point=-1;

    #if (Aretes_a_conserver[Cur_Arete_Cons]!=-1)
      #if (Aretes[Aretes_a_conserver[Cur_Arete_Cons]][0]=prev_point)
	#local index_prev=0;
	#local index_autre_point=1;
      #else
	#if (Aretes[Aretes_a_conserver[Cur_Arete_Cons]][1]=prev_point)
	  #local index_prev=1;
	  #local index_autre_point=0;
	#end
      #end      
    #end
    
    #if (index_autre_point!=-1)
      #if (Aretes[Aretes_a_conserver[Cur_Arete_Cons]][index_autre_point]=last_point)
	#local next_point=last_point;
	#local next_arete=last_arete;
      #else
	#local next_point=Aretes[Aretes_a_conserver[Cur_Arete_Cons]][index_autre_point];
	#local next_arete=Nouv_Arete(next_point,index);
      #end
      
      #local Dummy=Nouv_Triangle
      (prev_arete, Aretes_a_conserver[Cur_Arete_Cons], next_arete,
	Aretes[Aretes_a_conserver[Cur_Arete_Cons]][index_prev], index, Aretes[Aretes_a_conserver[Cur_Arete_Cons]][index_autre_point]);

      #declare Aretes_a_conserver[Cur_Arete_Cons]=-1;
      #local Cur_Arete_Cons=0;
      #local prev_point=next_point;
      #local prev_arete=next_arete;
    #else 
      #local Cur_Arete_Cons=Cur_Arete_Cons+1;
    #end
  #end
  
  #local index=index+1;
#end
#end

// -----------------------------------------------
//  Traitements des polygones de Voronoi a partir des triangles
// -----------------------------------------------
#macro Polygone_Voronoi (num)

// On cree une base qui donne les triangles appartenant a une arete donnee.
// Cela permet d'optimiser l'algorithme.
#ifndef (Triangle_de_l_arete)
  #declare Triangle_de_l_arete=array[Nb_Aretes][2];
  
  #local Cur_Arete=0;
  #while (Cur_Arete<Nb_Aretes)
    #declare Triangle_de_l_arete[Cur_Arete][0]=-1;
    #declare Triangle_de_l_arete[Cur_Arete][1]=-1;
    #local Cur_Arete=Cur_Arete+1;
  #end

  #local Cur_Triangle=0;
  #while (Cur_Triangle<Nb_Triangles)
    #if (Triangles_Rayons[Cur_Triangle]!=-1)

      #local Cur_Arete=0;
      #while (Cur_Arete<3)
	#if (Triangle_de_l_arete[Triangles_Aretes[Cur_Triangle][Cur_Arete]][0]=-1)
	  #declare Triangle_de_l_arete[Triangles_Aretes[Cur_Triangle][Cur_Arete]][0]=Cur_Triangle;
	#else
	  #declare Triangle_de_l_arete[Triangles_Aretes[Cur_Triangle][Cur_Arete]][1]=Cur_Triangle;
	#end
	#local Cur_Arete=Cur_Arete+1;
      #end

    #end
    #local Cur_Triangle=Cur_Triangle+1;
  #end
  
  #local Cur_Arete=0;
  #while (Cur_Arete<Nb_Aretes)
    Debug_Message (concat("Arete : ", str(Cur_Arete,0,0), " / Pts : ", str(Aretes[Cur_Arete][0],0,0), ", ", str(Aretes[Cur_Arete][1],0,0), " / Tr : ", str(Triangle_de_l_arete[Cur_Arete][0],0,0), ", ", str(Triangle_de_l_arete[Cur_Arete][1],0,0), "\n"))
    #local Cur_Arete=Cur_Arete+1;
  #end    
  
#end    

// On recupere les aretes qui partent du point.  
#declare Aretes_du_point=array[Nb_Aretes];
#local index_AP=0;
#local Cur_Arete=0;
#while (Cur_Arete<Nb_Aretes)
  #if ((Aretes[Cur_Arete][0]=num) | (Aretes[Cur_Arete][1]=num))
    #declare Aretes_du_point[index_AP]=Cur_Arete;
    #local index_AP=index_AP+1;
  #end
  #local Cur_Arete=Cur_Arete+1;
#end 

#ifdef (Point_Polygone)
  #undef Point_Polygone
#end
// Il y a autant de cotes au polygone que d'aretes partant du point.
#declare Point_Polygone=array[index_AP];

// Les cotes du polygone sont les mediatrices des aretes qui partent du point, ce sont en fait les
// segments qui relient les centres des cercles circonscrits aux triangles.
#local prev_triangle=Triangle_de_l_arete[Aretes_du_point[0]][0];
#declare Point_Polygone[0]=V2D(Triangles_Centres[Triangle_de_l_arete[Aretes_du_point[0]][0]]);
#declare Aretes_du_point[0]=-1;

// Il faut relier les centres dans le bon ordre, on suit donc les aretes de proche en proche.
#local index_pp=1;
#local Cur_Arete=1;
#while (Cur_Arete<index_AP)
  #local index=1;          
  // On cherche le triangle qui suit le dernier identifie.
  #while (index<index_AP)
    #if (Aretes_du_point[index]!=-1)
      #local next_triangle=-1;
      #if (Triangle_de_l_arete[Aretes_du_point[index]][0]=prev_triangle)
	#local next_triangle=Triangle_de_l_arete[Aretes_du_point[index]][1];
      #else
	#if (Triangle_de_l_arete[Aretes_du_point[index]][1]=prev_triangle)
	  #local next_triangle=Triangle_de_l_arete[Aretes_du_point[index]][0];
	#end
      #end
      #if (next_triangle!=-1)
	#declare Point_Polygone[index_pp]=V2D(Triangles_Centres[next_triangle]);
	#local index_pp=index_pp+1;
	#declare Aretes_du_point[index]=-1;
	// On repart pour un tour de boucle.
	#local prev_triangle=next_triangle;
	#local Cur_Arete=1;
	#local index=index_AP;
      #end
    #end
    #local index=index+1;
  #end
  #local Cur_Arete=Cur_Arete+1;
#end   

#undef Aretes_du_point
#end

// -----------------------------------------------
// Sauvegarde
// -----------------------------------------------
#macro Save_Voronoi (vfilename)
#fopen voronoi_fd vfilename write

#write (voronoi_fd, concat ("// Fichier de donnees Voronoi.\n\n"))
#write (voronoi_fd, concat ("#declare Nb_Points=", str (Nb_Points,0,0), ";\n"))
#write (voronoi_fd, concat ("#declare Nb_Aretes=", str (Nb_Aretes,0,0), ";\n"))
#write (voronoi_fd, concat ("#declare Nb_Triangles=", str (Nb_Triangles,0,0), ";\n\n"))
#write (voronoi_fd, concat ("#declare Start_index=", str (Start_index,0,0), ";\n"))
#write (voronoi_fd, concat ("#declare Borne1=<", str (Borne1.x,0,2), ",", str (Borne1.y,0,2), ">;\n"))
#write (voronoi_fd, concat ("#declare Borne2=<", str (Borne2.x,0,2), ",", str (Borne2.y,0,2), ">;\n\n"))

// Ecriture des points
#write (voronoi_fd, concat ("#declare Points=array[Nb_Points]\n{"))

#local index=0;
#while (index < Nb_Points)
  #if (index != 0)
    #if (mod (index, 6) = 0)
      #write (voronoi_fd, concat (",\n"))
    #else
      #write (voronoi_fd, concat (", "))	
    #end
  #end
  
  #write (voronoi_fd, V2DStr(Points[index]))
  
  #local index=index+1;
#end

#write (voronoi_fd, concat ("};\n\n"))  

// Ecriture des aretes
#write (voronoi_fd, concat ("#declare Aretes=array[Nb_Aretes][2]\n{"))

#local index=0;
#while (index < Nb_Aretes)
  #if (index != 0)
    #if (mod (index, 10) = 0)
      #write (voronoi_fd, concat (",\n"))
    #else
      #write (voronoi_fd, concat (", "))	
    #end
  #end
  
  #write (voronoi_fd, concat ("{", str(Aretes[index][0], 0, 0), ", ", str(Aretes[index][1], 0, 0), "}"))
  
  #local index=index+1;
#end

#write (voronoi_fd, concat ("};\n\n"))  

// Ecriture des triangles
#write (voronoi_fd, concat ("#declare Triangles_Aretes=array[Nb_Triangles][3]\n{"))

#local index=0;
#while (index < Nb_Triangles)
  #if (index != 0)
    #if (mod (index, 8) = 0)
      #write (voronoi_fd, concat (",\n"))
    #else
      #write (voronoi_fd, concat (", "))	
    #end
  #end
  
  #write (voronoi_fd, concat
    ("{", str(Triangles_Aretes[index][0], 0, 0),
      ", ", str(Triangles_Aretes[index][1], 0, 0)
      ", ", str(Triangles_Aretes[index][2], 0, 0), "}"))
  
  #local index=index+1;
#end

#write (voronoi_fd, concat ("};\n\n"))  

// Ecriture des rayons des triangles
#write (voronoi_fd, concat ("#declare Triangles_Rayons=array[Nb_Triangles]\n{"))

#local index=0;
#while (index < Nb_Triangles)
  #if (index != 0)
    #if (mod (index, 16) = 0)
      #write (voronoi_fd, concat (",\n"))
    #else
      #write (voronoi_fd, concat (", "))	
    #end
  #end
  
  #if (Triangles_Rayons [index] = -1.0)
    #write (voronoi_fd, "-1")
  #else
    #write (voronoi_fd, str(Triangles_Rayons [index], 0, 4))
  #end
  
  #local index=index+1;
#end

#write (voronoi_fd, concat ("};\n\n"))  

// Ecriture des centres des triangles
#write (voronoi_fd, concat ("#declare Triangles_Centres=array[Nb_Triangles]\n{"))

#local index=0;
#while (index < Nb_Triangles)
  #if (index != 0)
    #if (mod (index, 16) = 0)
      #write (voronoi_fd, concat (",\n"))
    #else
      #write (voronoi_fd, concat (", "))	
    #end
  #end
  
  #write (voronoi_fd, V3DStr(Triangles_Centres [index]))
    
  #local index=index+1;
#end

#write (voronoi_fd, concat ("};\n\n"))  

#fclose voronoi_fd
#end

// -----------------------------------------------
// Presentation
// -----------------------------------------------
#macro Dessiner_points ()
union {
  #local index=0;
  #while (index<Nb_Points)
    sphere { Points [index], 1.0 pigment { rgb <0,0,1> } }
    #local index=index+1;
  #end
}
#end

#macro Dessiner_aretes ()
union {
  #local index=0;
  #while (index<Nb_Aretes)
    #if (Aretes[index][0]!=-1)
      cylinder { Points[Aretes[index][0]], Points[Aretes[index][1]], 0.4 pigment { rgb 1 } }
    #end
    #local index=index+1;
  #end
}
#end

#macro Dessiner_cercles ()
#local index=0;
#while (index<Nb_Triangles)
  #if (Triangles_Rayons[index]!=-1)
    disc {
      Triangles_Centres[index],
      z,
      Triangles_Rayons[index]+0.1,
      Triangles_Rayons[index]-0.1
      pigment { rgb 0.5 }
    }
  #end
  #local index=index+1;
#end
#end

#macro Dessiner_polygones ()
#local index=Start_index;
#while (index<Nb_Points)
  Polygone_Voronoi (index)

  #local index_pt=0;
  #while (index_pt < dimension_size(Point_Polygone,1))
    cylinder {
      #if (index_pt=0)
	Point_Polygone[dimension_size(Point_Polygone,1)-1],
      #else
	Point_Polygone[index_pt-1],
      #end
      Point_Polygone[index_pt],
      0.4
      pigment { rgb <0,1,0> }  
    }
    #local index_pt=index_pt+1;
  #end
  #local index=index+1;
#end
#end
