// Persistence of Vision Ray Tracer Scene Description File
// File: math_geom.pov
// POV-Ray Version: 3.5
// Desc: Provides some geometrics features.
//       Like line intersections or triangle surrounding circle computations.
// Date: 20/04/2003
// Auth: Jean-Charles Marteau ( exether@free.fr )

#ifndef (Math_Geom_Inc)
#declare Math_Geom_Inc=version;


#declare epsilon=0.000001;

// ----------------------------------------
#macro V3DStr (V)
  concat ("<", str (V.x, 0, 4), ", ", str (V.y, 0, 4), ", ", str (V.z, 0, 4), ">")
#end

#macro V2DStr (V)
concat ("<", str (V.x, 0, 4), ", ", str (V.y, 0, 4), ">")
#end

#macro V2D (V)
<V.x, V.y>
#end

#macro V3D (V)
<V.x, V.y, 0.0>
#end

// ----------------------------------------
// Les droites sont decrites par un point et un vecteur directeur.
// Calcule le point d'intersection entre les droites (p1, vd1) et (p2, vd2)
// Retourne le point d'intersection, et affecte la variable ok a la validite
// de ce point (si les droites ne se coupent pas, ok = false).
#macro line_intersection (p1, vd1, p2, vd2, ok)
#local AB=vlength (p2-p1);
#local vdn1=vnormalize (vd1);
#local vdn2=vnormalize (vd2);

// Si les vecteurs sont paralleles ou que le probleme n'est pas plan.
#if ( (vlength(vcross(vdn1, vdn2))<=epsilon) |
  (vlength(vcross(vcross(p2-p1, vdn1), vcross(p2-p1, vdn2)))>=epsilon))

  #declare ok=false;
  #local Result=0;
#else
  // Projection de A sur BP
  #local AAp=p2-p1 - vdot(p2-p1, vdn2)*vdn2;
  // Projection de A+u sur BP
  #local AsAt=p2-p1-vdn1 - vdot(p2-p1-vdn1, vdn2)*vdn2;
  
  // On prend un vecteur reference sur les paralleles en prenant en compte
  // Le cas ou A est le point d'intersection
  #if (vlength(AAp)>epsilon)
    #local vdn3=vnormalize(AAp);
  #else
    #local vdn3=vnormalize(AsAt);
  #end
  
  // En valeur algebrique, d'apres le theoreme de Thales :
  // AP=AAs + (AAs x AsAt)/(AAp-AsAt)
  // Or AAs=u=1
  #local AP=1+(1*vdot(AsAt, vdn3))/(vdot(AAp, vdn3)-vdot(AsAt, vdn3));
  #declare ok=true;
  #local Result=p1+AP*vdn1;
#end
Result
#end

// ----------------------------------------
// Calcule le cercle circonscrit au triangle ABC.
// Retourne le centre et affecte le rayon a la variable r.
#macro triangle_circle_circ (A, B, C, r)
// O est le point d'intersection des mediatrices
#local vn=vnormalize (vcross (B-A, C-A));
#local vd1=vcross(B-A, vn);
#local vd2=vcross(C-A, vn);
#local Result_Valid=false;
#declare O=line_intersection ((A+B)/2, vd1, (A+C)/2, vd2, Result_Valid);
// Etant donne le calcul, le resultat doit toujours etre valide.
#declare r=vlength(O-A);
O
#end

// ----------------------------------------
// Retourne la projection du point P sur la droite (O, vd)
#macro proj_point_on_line (P, O, vd)
#local vdn=vnormalize (vd);
O+vdot(P-O, vdn)*vdn
#end


#end // End of package
