// Persistence of Vision Ray Tracer Scene Description File
// File: hl_geom.pov
// POV-Ray Version: 3.5
// Desc: hl stands for High level functionalities.
//       This package provides mainly rounded prisms
// Date: 20/04/2003
// Auth: Jean-Charles Marteau ( exether@free.fr )

#ifndef (High_Level_Geom_Inc)
#declare High_Level_Geom_Inc=version;

#include "math_geom.inc"

// ----------------------------------------
#macro pos_mod(A,B)
  #local Result=mod(A,B);
  #if (Result>=0)
    Result
  #else
    Result+B
  #end
#end

// ----------------------------------------
#macro Unfilled_Array_Prism (Points, n, E)
  prism {
    linear_sweep
    linear_spline
    -E/2,
     E/2,
    n+1,
    #local index=0;
    #while (index < n)
      Points[index],
      #local index=index+1;
    #end
    Points[0]
    rotate <-90.0, 0.0, 0.0>
  }
#end

#macro Array_Prism (Points, E)
  Unfilled_Array_Prism (Points, dimension_size(Points,1), E)
#end

// ----------------------------------------
// Cette macro cree un prisme polygonal de base constituee par les points
// du tableau 'Points', de hauteur E. Ce prisme a ses cotes et ses coins
// arrondis. Le parametre Q permet de modifier la qualite pour accelerer
// le rendu :
//  - 0, prisme normal sans arrondis
//  - 1, prisme arrondi en haut et en bas seulement
//  - 2, prisme arrondi en haut et en bas avec les coins coupes
//  - 3, prisme arrondi complement, les coins sur coupes et remplaces par
//       des cylindres.
//
// Bug connu : quand deux coins sont a une distance inferieure a Bord, il faudrait
// les confondre en un seul.
#macro Round_Prism (Points, E, Bord, Q)
#if (Q=0)
  Array_Prism(Points, E)
#else
  union {
    // Calcul du barycentre
    #local G=<0.0,0.0>;
    #local index=0;
    #while (index < dimension_size(Points,1))
      #local G=G + Points[index];
      #local index=index+1;
    #end
    #local G=G / dimension_size(Points,1);

    #local InsidePoints=array[dimension_size(Points,1)];
    
    #local EdgeCutPoints=array[2*dimension_size(Points,1)];
    #local index_ECP=0;
    
    #local index=0;
    #while (index < dimension_size(Points,1))
      // Pour chaque point, on calcule l'intersection des droites paralleles aux cotes
      // qui sont espacees de 'Bord' des cotes du polygone.
      #local A=Points[pos_mod(index-1, dimension_size(Points,1))];
      #local B=Points[pos_mod(index+1, dimension_size(Points,1))];
      #local O=Points[index];
      
      #local vnA=vnormalize(vcross(O-A, z));
      #local vnB=vnormalize(vcross(O-B,-z));
      #local valid=false;

      #local Ap=A+Bord*vnA;
      #local Bp=B+Bord*vnB;
      #local Op=line_intersection (Ap, O-A, Bp, O-B, valid);
      
      // Si le point interieur se trouve en fait a l'exterieur, on prend le symetrique.
      // Cette erreur resulte d'un probleme de signe. La solution proposee ici suppose
      // une forme a peu pres reguliere du polygone, c'est a dire que les points interieurs
      // sont plus proches du barycentre que les points exterieurs. Pour le cas particulier
      // d'un polygone convexe, la condition est bien sur realisee.
      #if (vdot (O-G, O-Op) < 0)
	#declare Op=O+(O-Op);
      #end
      #declare InsidePoints[index]=<Op.x, Op.y>;
      
      // Si l'angle est convexe, on va calculer les points au bord
      // pour couper les angles afin de pouvoir les arrondir.
      #if (vdot(Op-A, vnormalize(O-A)) < vlength(O-A))
	#declare EdgeCutPoints[index_ECP]=V2D(proj_point_on_line (Op, O, A-O));
	#local index_ECP=index_ECP+1;
	#declare EdgeCutPoints[index_ECP]=V2D(proj_point_on_line (Op, O, B-O));
	#local index_ECP=index_ECP+1;
      #else
	#declare EdgeCutPoints[index_ECP]=V2D(O);
	#local index_ECP=index_ECP+1;      
      #end
      
      #local index=index+1;
    #end
    
    #if (Q=1)
      Array_Prism (Points, E-2*Bord)
    #else
      Unfilled_Array_Prism (EdgeCutPoints, index_ECP, E-2*Bord)
    #end
    
    Array_Prism (InsidePoints, E)
    #local index=0;
    #local prev_point=InsidePoints[dimension_size(Points,1)-1];
    #while (index < dimension_size(Points,1))

      cylinder { prev_point, InsidePoints[index], Bord translate -z*((E/2)-Bord) }
      sphere { InsidePoints[index], Bord translate -z*((E/2)-Bord) }
      cylinder { prev_point, InsidePoints[index], Bord translate z*((E/2)-Bord) }
      sphere { InsidePoints[index], Bord translate z*((E/2)-Bord) }

      #local prev_point=InsidePoints[index];

      #if (Q>2)
	cylinder {
	  InsidePoints[index]-z*((E/2)-Bord),
	  InsidePoints[index]+z*((E/2)-Bord),
	  Bord
	}
      #end

      #local index=index+1;
    #end
  }
#end
#end

// ----------------------------------------
// ----- Etoile

//  #include "shapes.inc"
#macro Etoile (l,d,nb_branches)
merge {
  #local index=0;
  #while (index < nb_branches)
    object { Round_Cone2_Merge(<0.0001,0,0>, 1, <l,0,0>, d) rotate 72*y*index }
    #local index=index+1;
  #end
}
#end

#end // End of package
