////////////////////////////////////////////////////////////////////////////////////
//
// MOSAIC MACROS
//
// written by Douglas C. Eichenberg / www.getinfo.net/douge / douge@nls.net
//
// List of included macros:
//
// 1. GET_AVG_COLOR(w,h,pig): 
//    Determines the average RGB color value of an image.
//
//    w=width of subsampling
//    h=height of subsampling
//    pig=image map to be parsed, in the form of a pigment
//
//    Returns a color vector.  
//    
// 2. WRITE_COLOR_TABLE(bct_num,image_dir,w,h,save_name): 
//    Creates an ASCII text file record of the average
//    RGB color values for a series of images. 
//
//    bct_num=number of images in the directory to be parsed
//    image_dir=image directory string
//    w=width of subsampling
//    h=height of subsampling
//    save_name=directory and name string to save color table under
//
// 3. READ_COLOR_TABLE(read_name): 
//    Reads a stored ASCII RGB color table into memory.
//
//    read_name=directory and name string of ASCII color table.
//
// 4. MONO_BOX_IMAGE(w,h,pig,cutoff): 
//    Creates a 3D likeness of an image by reducing it to
//    2 colors and using rounded boxes to represent one color.
//
//    w=width of subsampling
//    h=height of subsampling
//    pig=image map to be parsed, in the form of a pigment
//    cutoff=float value for determining which colors are dropped
//           from the final image, ranging from 0 to 3.
//    
// 5. MULTI_BOX_IMAGE(w,h,pig): 
//    Creates a 3D likeness of an image by creating rounded
//    boxes with pigments matching sample points in the image.
//    
//    w=width of subsampling
//    h=height of subsampling
//    pig=image map to be parsed, in the form of a pigment
//
// 6. MATCH_PIG(pig,num_entries): 
//    Parses the color table in memory to find which image's
//    average color is the closest match to a particular RGB value.
//  
//    pig=image map to be parsed, in the form of a pigment
//    num_entries=number of entries in the color table
//
// 7. PHOTOMOSAIC_BOX_IMAGE(w,h,pig,dir_string,num_entries): 
//    Creates a photomosaic 3D rounded box representation of an image.
// 
//    w=width of subsampling
//    h=height of subsampling
//    pig=image map to be parsed, in the form of a pigment
//    dir_string=directory and name string of ASCII color table. 
//    num_entries=number of entries in the color table
//
// 8. DISPLAY_COLORTABLE(read_name):
//
//    read_name=directory and name string of ASCII color table.
// 
////////////////////////////////////////////////////////////////////////////////////

#version 3.5;
#include "colors.inc"
#include "stones.inc"
#include "textures.inc" 
#include "functions.inc"
#include "rand.inc" 
#include "math.inc"

#declare Rad_Quality=1;
#declare cam_ortho=0;
#declare room=1;

global_settings {
    assumed_gamma 1
    max_trace_level 256
#switch (Rad_Quality)
 #case (1)
  radiosity {             // --- Settings 1 (fast) ---
    pretrace_start 0.08
    pretrace_end   0.02
    count 100
    nearest_count 20 
    low_error_factor 0.01
    error_bound 0.5
    recursion_limit 1 
    gray_threshold 0.1
    brightness 0.85
    
  }
 #break
 #case (2)
  radiosity {             // --- Settings 2 (medium quality) ---
    pretrace_start 0.08
    pretrace_end   0.01
    count 200    
    nearest_count 20 
    low_error_factor 0.01
    error_bound 0.25
    recursion_limit 1  
    gray_threshold 0.5
    brightness 0.75
  }
 #break
 #case (3)
  radiosity {             // --- Settings 3 (high quality) ---
    pretrace_start 0.08
    pretrace_end   0.005
    count 800
    nearest_count 20
    error_bound 0.1
    low_error_factor 0.01 
    gray_threshold 0.5
    recursion_limit 1
    brightness 0.75
  }
 #break
 #case (4)
  radiosity {             // --- Settings 4 (medium quality, recursion_limit 2) ---
    pretrace_start 0.08
    pretrace_end   0.005
    count 350
    error_bound 0.15
    recursion_limit 2
  }
 #break
 #end

}


/////////////////////////////////////////////////
//
// pigment definitions
//
/////////////////////////////////////////////////

#declare Pig1 = pigment {
 image_map{sys "marilyn.bmp"}
}
#declare Pig2 = pigment {
 image_map{jpeg "mona.jpg"}
} 

#declare Pig3 = pigment {
 image_map{jpeg "lincoln.jpg"}
}


////////////////////////////////////////////////////////////////////////
//   
// MACRO: MULTI_BOX_IMAGE 
// By default, the image exactly fills the area from <0,0,0> to <1,1,0>
// so it will be parsed through that area into w*h "pixels" Each parsed
// "pixel" becomes a 3d POVRay rounded box with the same color as the
// parsed pixel.
//
// Parses at 92.59 pps on my laptop
//
//////////////////////////////////////////////////////////////////////// 
#declare rs=seed(466);
#macro multi_box_image(w,h,pig)         
   #declare num_pixels=w*h;
   #declare h_step=1/h;
   #declare w_step=1/w;
   #declare X=0; #declare Y=0;
   #declare cnt=1;
   #while (cnt<num_pixels)      
      #declare pig_val=eval_pigment(pig,<X,Y,0>);
      #debug concat("<", str(X,5,5), ", ", str(Y,5,5), "> = ", "<", str(pig_val.x,5,5), ", ", str(pig_val.y,5,5),", ", str(pig_val.z,5,5), "> \n")
      isosurface{function{f_rounded_box(x,y,z,0.2,0.5,0.5,0.5)} translate<0.5,0.5,0.5> scale <h_step,h_step,2*h_step> translate<X,Y,0.01*rand(rs)> texture{pigment{color red pig_val.x green pig_val.y blue pig_val.z} finish{ambient 0 diffuse 1 specular .25 roughness 0.0005 reflection 0.1}}}
         
      #if (X < 1)
         #declare X=X+w_step;
      #else
         #declare X=0;
         #declare Y=Y+h_step;
      #end
      #declare cnt=cnt+1;
   #end   
#end

//union{multi_box_image(100,100,Pig1) scale 1 rotate<0,45,0> translate<-0.75,-0.5,-1.5>}

//union{multi_box_image(20,20,Pig2) scale 0.75 rotate<0,25,0> translate<0,-0.5,-0.5>}  

////////////////////////////////////////////////////////////////////////
//   
// MACRO: MONO_BOX_IMAGE
// Makes a simple one-color version of the image from boxes.
// Cutoff ranges from 0 to 3 and determines how much of the image
// is shown, versus how much is cut out (higher cutoff = more pixels)
//
////////////////////////////////////////////////////////////////////////
#macro mono_box_image(w,h,pig,cutoff)         
   #declare num_pixels=w*h;
   #declare h_step=1/h;
   #declare w_step=1/w;
   #declare X=0; #declare Y=0;
   #declare cnt=1;
   #while (cnt<num_pixels)      
      #declare pig_val=eval_pigment(pig,<X,Y,0>);
      #debug concat("<", str(X,5,5), ", ", str(Y,5,5), "> = ", "<", str(pig_val.x,5,5), ", ", str(pig_val.y,5,5),", ", str(pig_val.z,5,5), "> \n")
      
      #if ((pig_val.x+pig_val.y+pig_val.z) < cutoff)
           isosurface{function{f_rounded_box(x,y,z,0.2,0.5,0.5,0.5)} translate<0.5,0.5,0.5> scale <h_step,h_step,2*h_step> translate<X,Y,0.01*rand(rs)> texture{pigment {White} finish{ambient 0 diffuse 1 specular .25 roughness 0.0005 reflection 0.1}}}
      #end
      #if (X < 1)
         #declare X=X+w_step;
      #else
         #declare X=0;
         #declare Y=Y+h_step;
      #end
      #declare cnt=cnt+1;
   #end   
#end


//union{mono_box_image(70,70,Pig3,0.75) scale <0.8028,1,1>*2 rotate<0,180+30,0> translate<0.75,-0.5,-2.5>} 
//union{mono_box_image(50,50,Pig2,1) scale <0.666,1,1>*1.5 rotate<0,180-30,0> translate<1.25,-0.5,-2.25>}  
//union{mono_box_image(50,50,Pig1,1.9) scale <0.7732,1,1>*2 rotate<0,180+15,0> translate<0.75,-0.5,-2.75>}


////////////////////////////////////////////////////////////////////////
//   
// MACRO: GET_AVG_COLOR
// Calculates and returns the average color of an image.
//
//////////////////////////////////////////////////////////////////////// 
#macro get_avg_color(w,h,pig)         
   #declare num_pixels=w*h;
   #declare h_step=1/h;
   #declare w_step=1/w;
   #declare X=0; #declare Y=0; 
   #declare x_total=0; #declare y_total=0; #declare z_total=0;
   #declare gac_cnt=1;
   #while (gac_cnt<num_pixels)      
      #declare pig_val=eval_pigment(pig,<X,Y,0>);
      #declare x_total=x_total+pig_val.x;
      #declare y_total=y_total+pig_val.y;
      #declare z_total=z_total+pig_val.z; 
      #if (X < 1)
         #declare X=X+w_step;
      #else
         #declare X=0;
         #declare Y=Y+h_step;
      #end
      #declare gac_cnt=gac_cnt+1;
   #end 
   <(x_total/num_pixels),(y_total/num_pixels),(z_total/num_pixels)>  
#end 
////////////////////////////////
//
// Call get_avg_color and make a 
// box of that color
//
//////////////////////////////// 
/*
#declare avg_color=get_avg_color(100,100,Pig1);
#debug concat("<", str(avg_color.x,5,5), ", ", str(avg_color.y,5,5), ", ", str(avg_color.z,5,5), "> \n")
box{<0,0,0> <1,1,1> 
     scale 0.5
     rotate<0,25,0> 
     translate<0,-.5,-1.5> 
     texture{pigment{color rgb avg_color} 
             finish{ambient 0 diffuse 1 specular .25 roughness 0.0005 reflection 0}
            }
   } 
*/            
////////////////////////////////////////////////////////////////////////
//   
// MACRO: WRITE_COLOR_TABLE
// Parses a directory of images, creates a color table, and saves the
// color table as a text file.  The images must 
// be named as consecutive numbers in jpg format (ie. 1.jpg, 2.jpg, 3.jpg,
// ... etc.)
// 
// bct_num=number of images in the directory to be parsed
// image_dir=image directory
// w,h=same as in other functions
// save_name=directory and name to save color table as
//
// COLOR TABLE FORMAT:
// 
// num_elements
// color vector.x, color vector.y, color vector.z,
// color vector.x, color vector.y, color vector.z,
//     ....
// color vector.x, color vector.y, color vector.z
//
// for example: 1
//              0.34591, 0.77539, 0.21190
// 
// write_color_table(36,"C:\\Documents and Settings\\Owner\\Desktop\\eval_images\\",10,10,"C:\\Documents and Settings\\Owner\\Desktop\\eval_images\\color_table.txt")
//
//////////////////////////////////////////////////////////////////////// 
#macro write_color_table(bct_num,image_dir,w,h,save_name)
   #debug "Creating color table..." 
   #local bct_cnt=1;
   #fopen COLOR_TABLE save_name append
   #write(COLOR_TABLE, bct_num, ", \n")
   #while (defined(COLOR_TABLE))      
      #while (bct_cnt <= bct_num)
         #declare pig=pigment{image_map {jpeg concat(image_dir, str(bct_cnt,0,0), ".jpg")}};   
         #declare avg_color=get_avg_color(w,h,pig);
         #write(COLOR_TABLE, avg_color.x, ", ", avg_color.y, ", ", avg_color.z, ", \n") 
         #declare bct_cnt=bct_cnt+1;
      #end                  
      #fclose COLOR_TABLE
   #end
   #debug "done! \n" 
#end

//write_color_table(143,"C:\\Documents and Settings\\Owner\\Desktop\\eval_images\\",100,100,"C:\\Documents and Settings\\Owner\\Desktop\\eval_images\\color_table.txt")

////////////////////////////////////////////////////////////////////////
//   
// MACRO: READ_COLOR_TABLE
// 
// Opens a color table and reads it into an array.  If successful,
// the global array can be accessed as "color_table_array".
//
// read_name=directory and name of color table made with build_color_table
// 
//////////////////////////////////////////////////////////////////////// 
#macro read_color_table(read_name)
   #declare rct_cnt=0;
   #fopen COLOR_TABLE read_name read 
   #read (COLOR_TABLE,num_entries)
   #declare color_table_array=array[num_entries][3]         
   #while (rct_cnt < num_entries)  
      #read (COLOR_TABLE, ct_color_x, ct_color_y, ct_color_z)
      #declare color_table_array[rct_cnt][0]=ct_color_x;
      #declare color_table_array[rct_cnt][1]=ct_color_y;
      #declare color_table_array[rct_cnt][2]=ct_color_z;
      #debug concat(str(color_table_array[rct_cnt][0],5,5), ", ", str(color_table_array[rct_cnt][1],5,5), ", ", str(color_table_array[rct_cnt][2],5,5), " \n")
      #declare rct_cnt=rct_cnt+1;
   #end   
   #debug concat("Created color table array with ", str(rct_cnt,0,0), " entries. \n") 
   num_entries
#end
    
//read_color_table("C:\\Documents and Settings\\Owner\\Desktop\\eval_images\\color_table.txt")

////////////////////////////////////////////////////////////////////////
//   
// MACRO: MATCH_PIG
// 
// Matches the supplied pigment to an image in color_table_array
// and returns the array index number.
//
////////////////////////////////////////////////////////////////////////
#macro match_pig(pig,num_entries)
   #declare min_dif=3;
   #declare mp_cnt=0;
   #while (mp_cnt < num_entries)
      #local x_dif=abs(color_table_array[mp_cnt][0]-pig.x);
      #local y_dif=abs(color_table_array[mp_cnt][1]-pig.y);
      #local z_dif=abs(color_table_array[mp_cnt][2]-pig.z);
      #local total_dif=x_dif+y_dif+z_dif;
      #if (total_dif < min_dif)
         #declare min_dif=total_dif; 
         #declare dif_index=mp_cnt;
      #end 
      #declare mp_cnt=mp_cnt+1;
   #end                        
   dif_index
#end


////////////////////////////////////////////////////////////////////////
//   
// MACRO: DISPLAY_COLOR_TABLE 
//
//////////////////////////////////////////////////////////////////////// 
#macro display_color_table(read_name)
   #declare num_pixels=read_color_table(read_name);
   #declare w=int(sqrt(num_pixels));
   #if (mod(num_pixels,w)=0)
      #declare h=num_pixels/w;
   #else
      #declare h=(num_pixels/w)+1;
   #end         
   #declare h_step=1/h;
   #declare w_step=1/w;
   #declare X=0; #declare Y=0;
   #declare dct_cnt=1;
   #while (dct_cnt<num_pixels)      
      #local x_pig_val=color_table_array[dct_cnt][0];
      #local y_pig_val=color_table_array[dct_cnt][1];
      #local z_pig_val=color_table_array[dct_cnt][2];
     
      isosurface{function{f_rounded_box(x,y,z,0.1,0.5,0.5,0.5)} translate<0.5,0.5,0.5> scale <h_step,h_step,2*h_step> translate<X,Y,0> texture{pigment{color red x_pig_val green y_pig_val blue z_pig_val} finish{ambient 1 diffuse 0}}}
         
      #if (X < (1-w_step))
         #declare X=X+w_step;
      #else
         #declare X=0;
         #declare Y=Y+h_step;
      #end
      #declare dct_cnt=dct_cnt+1;
   #end   
#end

//union{display_color_table("C:\\Documents and Settings\\Owner\\Desktop\\eval_images\\color_table.txt") scale 1 translate <-0.5,-0.5,-1> }

////////////////////////////////////////////////////////////////////////
//   
// MACRO: AVERAGE_COLOR
// Calculates and returns the average of two colors
//
//////////////////////////////////////////////////////////////////////// 
#macro average_color(C1,C2)
   <(C1.x+C2.x)/2,(C1.y+C2.y)/2,(C1.z+C2.z)/2>
#end

////////////////////////////////////////////////////////////////////////
//   
//  MACRO: AVERAGE_IMAGES
//  Averages two images together
//
////////////////////////////////////////////////////////////////////////
#macro average_images(w,h,pig1,pig2)         
   #declare num_pixels=w*h;
   #declare h_step=1/h;
   #declare w_step=1/w;
   #declare X=0; #declare Y=0;
   #declare X1=0; #declare Y1=0; 
   #declare X2=0; #declare Y2=0; 
   #declare pig_val=<0,0,0>; #declare pig_val1=<0,0,0>; #declare pig_val2=<0,0,0>;
   #declare cnt=1;
   #while (cnt<num_pixels)      
      #declare pig_val1=eval_pigment(pig1,<X,Y,0>);
      #declare pig_val2=eval_pigment(pig2,<X,Y,0>);
      #declare pig_val=average_color(pig_val1,pig_val2);
      
      box{<0,0,0> <1,1,1> scale h_step translate<X,Y,0> texture{pigment{color red pig_val.x green pig_val.y blue pig_val.z} finish{ambient 1 diffuse 0}}}
         
      #if (X < 1)
         #declare X=X+w_step;
      #else
         #declare X=0;
         #declare Y=Y+h_step;
      #end
      #declare cnt=cnt+1;
   #end   
#end

//union{average_images(200,200,Pig1,Pig3) scale 1 translate<-0.5,-0.5,0>}      

////////////////////////////////////////////////////////////////////////
//   
//  MACRO: MIX_IMAGES
//  Mixes two images together by alternating pixels.
//
////////////////////////////////////////////////////////////////////////
#macro mix_images(w,h,pig1,pig2)         
   #declare num_pixels=w*h;
   #declare h_step=1/h;
   #declare w_step=1/w;
   #declare X=0; #declare Y=0;
   #declare X1=0; #declare Y1=0; 
   #declare X2=0; #declare Y2=0; 
   #declare pig_val=<0,0,0>; #declare pig_val1=<0,0,0>; #declare pig_val2=<0,0,0>;
   #declare cnt=1;
   #while (cnt<num_pixels)      
      #declare pig_val1=eval_pigment(pig1,<X,Y,0>);
      #declare pig_val2=eval_pigment(pig2,<X,Y,0>);
      #if (even(cnt)=1)
         #declare pig_val=pig_val1;
      #else
         #declare pig_val=pig_val2;
      #end
     
      box{<0,0,0> <1,1,1> scale h_step translate<X,Y,0> texture{pigment{color red pig_val.x green pig_val.y blue pig_val.z} finish{ambient 1 diffuse 0}}}
         
      #if (X < 1)
         #declare X=X+w_step;
      #else
         #declare X=0;
         #declare Y=Y+h_step;
      #end
      #declare cnt=cnt+1;
   #end   
#end

//union{mix_images(100,100,Pig1,Pig3) scale 1 translate<-0.5,-0.5,0>} 

////////////////////////////////////////////////////////////////////////
//   
// MACRO: PHOTOMOSAIC_BOX_IMAGE
//
////////////////////////////////////////////////////////////////////////
#macro photomosaic_box_image(w,h,pig,dir_string,num_entries)         
   #declare num_pixels=w*h;
   #declare h_step=1/h;
   #declare w_step=1/w;
   #declare X=0; #declare Y=0;
   #declare pbi_cnt=1;
   #while (pbi_cnt<num_pixels)      
      #declare pig_val=eval_pigment(pig,<X,Y,0>);
      #declare image_index=match_pig(pig_val,num_entries); 
      
      isosurface{function{f_rounded_box(x,y,z,0.1,0.5,0.5,0.5)} translate<0.5,0.5,0.5> texture{pigment{image_map{ jpeg concat(dir_string, str(image_index+1,0,0), ".jpg") }} finish{ambient 0 diffuse 1 specular .15 roughness 0.005 reflection 0.1}} scale <h_step,h_step,2*h_step> translate<X,Y,0.01*rand(rs)> }
         
      #if (X < (1-w_step))
         #declare X=X+w_step;
      #else
         #declare X=0;
         #declare Y=Y+h_step;
      #end
      #declare pbi_cnt=pbi_cnt+1;
   #end   
   #debug str(pbi_cnt,0,0)
#end  


//union{photomosaic_box_image(100,100,Pig2,"C:\\Documents and Settings\\Owner\\Desktop\\eval_images\\",143) scale <0.666,1,1>*2 rotate<0,180+25,0> translate<0,-0.5,-2.75>}

//////////////////// room ////////////////// 

#if (room=1)
box { // floor
  <-5, -0.1, 0>
  < 5,  0,  -10> 
  texture{pigment{White*1} finish{ambient 0.1 diffuse 1 specular .25 roughness 0.0005 reflection 0.25}}
  translate<0,-0.5,0>
}
    
box { //ceiling
  <-5, -0.1, 0>
  < 5,  0,  -10> 
  texture{pigment{White} finish{ambient 0.1 diffuse 1 specular .25 roughness 0.0005 reflection 0}}
  translate<0,3,0>
}   
box { //rear wall
  <-5, 0, -5>
  < 5,  10,  -5.1> 
  texture{pigment{White*1.25} finish{ambient 0.1 diffuse 1 specular .25 roughness 0.0005 reflection 0}}
  translate<0,-1,0>
}      
box { //camera wall
  <-5, 0, -5>
  < 5,  10,  -5.1> 
//  texture{pigment{gradient x pigment_map{[0 Blue*2][0.5 Blue*0.5][0.51 Orange*2][1 Orange*0.5]} scale 4} finish{ambient 1 diffuse 0 specular .25 roughness 0.0005 reflection 0}}
  texture{pigment{White*1.75} finish{ambient 1 diffuse 0 specular .25 roughness 0.0005 reflection 0}}
  translate<0,-1,8>
}    
box { //left wall
  <0, 0, 0>
  < 0.1,  10,  -10> 
  texture{pigment{White*1.5} finish{ambient 0.1 diffuse 1 specular .25 roughness 0.0005 reflection 0}}
  translate<2,-0.5,0>
}   
box { //right wall
  <0, 0, 0>
  < 0.1,  10,  -10> 
  texture{pigment{White*1.1} finish{ambient 0.1 diffuse 1 specular .25 roughness 0.0005 reflection 0}}
  translate<-2,-0.5,0>
}
#end

camera {
  #if (cam_ortho=1)
    orthographic
  #end
  location <0,0,1>     // position & direction of view
  look_at  <0,0,0>
  right 1*x            // horizontal size of view  \___ to be rendered at square size
  up 1*y               // vertical size of view    /
}